///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// 
// This code presents "vec" expressions
// also print matlab code for non-regression test purpose
//
// usage:
//    blas1 | octave -q
//
//  machine-precision independent computation:
//    blas1 -ndigits 12 | octave -q
//
# include "rheolef/skit.h"
using namespace rheolef;
using namespace std;

bool do_trunc = false;

void check (const vec<Float>& y, const char* expr)
{
    static int i = 0;
    cout << "y=" << expr    << ";\n";
    cout << "z=" << ml << y << ";\n";
    
    cout << "error" << ++i << "=";
    if (!do_trunc)
	cout << "norm(y-z)\n\n";
    else
	cout << "eps1*round(norm(y-z)/eps1)\n\n";
}
int main (int argc, char* argv[])
{
    warning_macro ("blas1_tst[0]");
    int digits10 = numeric_limits<Float>::digits10;

    warning_macro ("blas1_tst[1]");
    cout.setf(ios::scientific, ios::floatfield);
    
    warning_macro ("blas1_tst[2]");
    // avoid machine-dependent output digits in non-regression mode:
    if (argc == 3) {
 	do_trunc = true;
        digits10 = atoi(argv[2]);
	cout << "eps1=sqrt(10^(-" << digits10 << "));\n";
    }
    cout << setprecision(digits10);

    const unsigned int n = 10;
    // initialize with a constant: FIRST allocation
        vec<Float> x(n);

        x = 1;
        cout << "x=" << ml << x << ";" << endl;

    // shallow copy: y points to x, really
        vec<Float> y = x;
        check (y, "x");

    // set vec to a constant; SECOND (and last) allocation, for the result in y
        const int c = 17;
        cout << "c=" << c << ";" << endl;

        y = c;
        check (y, "c");

    // unary operator

        y = -x;
        cout << "x=" << ml << x << ";" << endl;
        check (y, "-x");

    // scalar-vec mixtes expressions: loop enrolled...
        y = 2*(x + c);
        check (y, "2*(x + c)");

	y = 2*x - 1/(x*x);
        check (y, "2*x - 1./(x.*x)");

    // computed assignment
        y += c;
        check (y, "y+c");
        
	y *= 2;
        check (y, "2*y");
        
	y += 2*x + c;
        check (y, "y+2*x+c");
    
    // function call
	y = log(x);
        check (y, "log(x)");
	
	y = log(x+c) + x;
        check (y, "log(x+c) + x");
    
    // vec appears in left and right hand
        y = -y;
        check (y, "-y");
}
