% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rvar-.R
\name{rvar}
\alias{rvar}
\title{Random variables of arbitrary dimension}
\usage{
rvar(
  x = double(),
  dim = NULL,
  dimnames = NULL,
  nchains = 1L,
  with_chains = FALSE
)
}
\arguments{
\item{x}{(multiple options) The object to convert to an \code{rvar}:
\itemize{
\item A vector of draws from a distribution.
\item An array where the first dimension represents draws from a distribution.
The resulting \code{\link{rvar}} will have dimension \code{dim(x)[-1]}; that is,
everything except the first dimension is used for the shape of the
variable, and the first dimension is used to index draws from the
distribution (see \strong{Examples}). Optionally,
if \code{with_chains == TRUE}, the first dimension indexes the iteration and the
second dimension indexes the chain (see \code{with_chains}).
}}

\item{dim}{(integer vector) One or more integers giving the maximal indices
in each dimension to override the dimensions of the \code{\link{rvar}} to be created
(see \code{\link[=dim]{dim()}}). If \code{NULL} (the default), \code{dim} is determined by the input.
\strong{NOTE:} This argument controls the dimensions of the \code{\link{rvar}}, not the
underlying array, so you cannot change the number of draws using this
argument.}

\item{dimnames}{(list) Character vectors giving the names in each dimension
to override the names of the dimensions of the \code{\link{rvar}} to be created (see
\code{\link[=dimnames]{dimnames()}}). If \code{NULL} (the default), this is determined by the input.
\strong{NOTE:} This argument controls the names of the dimensions of the
\code{\link{rvar}}, not the underlying array.}

\item{nchains}{(positive integer) The number of chains. The default is \code{1}.}

\item{with_chains}{(logical) Does \code{x} include a dimension for chains?
If \code{FALSE} (the default), chains are not included, the first dimension of
the input array should index draws, and the \code{nchains} argument can be
used to determine the number of chains. If \code{TRUE}, the \code{nchains} argument
is ignored and the second dimension of \code{x} is used to index chains.
Internally, the array will be converted to a format without the chain index.}
}
\value{
An object of class \code{"rvar"} representing a random variable.
}
\description{
Random variables backed by arrays of arbitrary dimension
}
\details{
The \code{"rvar"} class internally represents random variables as arrays of arbitrary
dimension, where the first dimension is used to index draws from the distribution.
Most mathematical operators and functions are supported, including efficient matrix
multiplication and vector and array-style indexing. The intent is that an \code{rvar}
works as closely as possible to how a base vector/matrix/array does, with a few
differences:
\itemize{
\item The default behavior when subsetting is not to drop extra dimensions (i.e.
the default \code{drop} argument for \code{[} is \code{FALSE}, not \code{TRUE}).
\item Rather than base R-style recycling, \code{rvar}s use a limited form of broadcasting:
if an operation is being performed on two vectors with different size of the same
dimension, the smaller vector will be recycled up to the size of the larger one
along that dimension so long as it has size 1.
}

For functions that expect base numeric arrays and for which \code{rvar}s cannot be
used directly as arguments, you can use \code{\link[=rfun]{rfun()}} or \code{\link[=rdo]{rdo()}} to translate your
code into code that executes across draws from one or more random variables
and returns a random variable as output. Typically \code{\link[=rdo]{rdo()}} offers the most
straightforward translation.

As \code{\link[=rfun]{rfun()}} and \code{\link[=rdo]{rdo()}} incur some performance cost, you can also operate directly
on the underlying array using the \code{\link[=draws_of]{draws_of()}} function. To re-use existing
random number generator functions to efficiently create \code{rvar}s, use \code{\link[=rvar_rng]{rvar_rng()}}.
}
\examples{

set.seed(1234)

# To create a "scalar" `rvar`, pass a one-dimensional array or a vector
# whose length (here `4000`) is the desired number of draws:
x <- rvar(rnorm(4000, mean = 1, sd = 1))
x

# Create random vectors by adding an additional dimension:
n <- 4   # length of output vector
x <- rvar(array(rnorm(4000 * n, mean = rep(1:n, each = 4000), sd = 1), dim = c(4000, n)))
x

# Create a random matrix:
rows <- 4
cols <- 3
x <- rvar(array(rnorm(4000 * rows * cols, mean = 1, sd = 1), dim = c(4000, rows, cols)))
x

# If the input sample comes from multiple chains, we can indicate that using the
# nchains argument (here, 1000 draws each from 4 chains):
x <- rvar(rnorm(4000, mean = 1, sd = 1), nchains = 4)
x

# Or if the input sample has chain information as its second dimension, we can
# use with_chains to create the rvar
x <- rvar(array(rnorm(4000, mean = 1, sd = 1), dim = c(1000, 4)), with_chains = TRUE)
x

}
\seealso{
\code{\link[=as_rvar]{as_rvar()}} to convert objects to \code{rvar}s. See \code{\link[=rdo]{rdo()}}, \code{\link[=rfun]{rfun()}}, and
\code{\link[=rvar_rng]{rvar_rng()}} for higher-level interfaces for creating \code{rvar}s.
}
