/**
 * @file config_file.c 
 * @brief Handle configuratio file 
 * @created 2005-12-12 
 * @date 2007-09-02
 * @author Bruno Ethvignot
 */
/*
 * copyright (c) 1998-2007 TLK Games all rights reserved
 * $Id: config_file.c,v 1.9 2007/09/02 14:55:29 gurumeditation Exp $
 *
 * Powermanga is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Powermanga is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
#include <stdbool.h>
#include "powermanga.h"
#include "tools.h"
#include "config_file.h"
#include "lispreader.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef _WIN32
#include <direct.h>
#define MKDIR(d, p) _mkdir(d)
#else
#include <dirent.h>
#define MKDIR mkdir
#endif
#ifdef _WIN32
#define CONFIG_DIR_NAME "tlkgames"
#else
#define CONFIG_DIR_NAME ".tlkgames"
#endif
#define CONFIG_FILE_NAME "powermanga.conf"

config_file *power_conf = NULL;
static char *lang_to_text[MAX_OF_LANGUAGES] = { "en", "fr" };
static char *config_dir = NULL;
static char config_filename[] = CONFIG_DIR_NAME;
static char config_file_name[] = CONFIG_FILE_NAME;
static char *configname;
static bool configfile_check_dir ();
static void configfile_reset_values ();
static FILE *fopen_data (const char *rel_filename, const char *mode);

/** 
 * Reset all configuration values
 */
void
configfile_reset_values ()
{
  power_conf->fullscreen = TRUE;
  power_conf->nosound = FALSE;
  power_conf->resolution = 640;
  power_conf->verbose = 0;
  power_conf->difficulty = 1;
  power_conf->scale_x = 2;
  if (getenv ("LANG") != NULL && strncmp(getenv ("LANG"), "fr", 2) == 0) 
   {
     power_conf->lang = FR_LANG;
   }
   else
   {
     power_conf->lang = EN_LANG;
   }
  
}

/** 
 * Display values of the configuration file
 */
void
configfile_print (void)
{
  fprintf (stdout, "  <config info>\n"
           "- fullscreen: %i\n- nosound: %i\n- resolution:%i\n"
           "- verbose: %i\n - difficulty: %i\n - lang:%s\n - scale_x:%i\n",
           power_conf->fullscreen, power_conf->nosound,
           power_conf->resolution, power_conf->verbose,
           power_conf->difficulty, lang_to_text[power_conf->lang],
           power_conf->scale_x);
}

/** 
 * Check if config directory exists; if not create it and set config_dir
 * @return TRUE if it completed successfully or FALSE otherwise
 */
bool
configfile_check_dir (void)
{
  Uint32 length;
  if (config_dir == NULL)
    {
      length = strlen (config_filename) + 3;
#ifdef _WIN32
      length++;
#else
      if (getenv ("HOME") != NULL)
        {
          length += strlen (getenv ("HOME"));
        }
      else
        {
          length++;
        }
#endif
      config_dir = memory_allocation (length);
      if (config_dir == NULL)
        {
          fprintf (stderr, "config_file.c/configfile_load() "
                   "not enough memory to allocate %i bytes!\n", length);
          return FALSE;
        }
    }
#ifdef _WIN32
  _snprintf (config_dir, strlen (config_dir) - 1, "./%s", config_filename);
  /* create directory if not exist */
  MKDIR (config_dir, S_IRWXU);
  fprintf (stderr, "ok\n");
#else
  snprintf (config_dir, strlen (config_dir) - 1, "%s/%s",
            (getenv ("HOME") ? getenv ("HOME") : "."), config_filename);
  /* test and create .lgames */
  if (!opendir (config_dir))
    {
      fprintf (stderr, "couldn't find/open config directory '%s'\n",
               config_dir);
      fprintf (stderr, "attempting to create it... ");
      MKDIR (config_dir, S_IRWXU);
      if (!opendir (config_dir))
        {
          fprintf (stderr, "failed\n");
          return FALSE;
        }
      else
        {
          fprintf (stderr, "ok\n");
        }
    }
#endif
  return TRUE;
}

/**
 * Load configuration file from "~/.tlkgames/powermanga.conf"
 * @return TRUE if it completed successfully or FALSE otherwise
 */
bool
configfile_load (void)
{
  Uint32 length;
  lisp_object_t *root_obj, *lst;
  char *str;
  /* allocate config structure */
  if (power_conf == NULL)
    {
      power_conf = (config_file *) memory_allocation (sizeof (config_file));
      if (power_conf == NULL)
        {
          fprintf (stderr, "(!)config_file.c/configfile_load() "
                   "not enough memory to allocate 'power_conf'!\n");
          return FALSE;
        }
    }
  configfile_reset_values ();
  if (!configfile_check_dir ())
    {
      return TRUE;
    }

  if (configname == NULL)
    {
      length = strlen (config_dir) + strlen (config_file_name) + 2;
      configname = memory_allocation (length);
      if (configname == NULL)
        {
          fprintf (stderr, "(!)config_file.c/configfile_load() "
                   "not enough memory to allocate %i bytes!\n", length);
          return FALSE;
        }
    }
  sprintf (configname, "%s/%s", config_dir, config_file_name);
  root_obj = lisp_read_file (configname);
  if (root_obj == NULL)
    {
      fprintf (stderr, "(!)config_file.c/configfile_load() "
               "lisp_read_file(%s) failed!\n", configname);
      return TRUE;
    }
  if (root_obj->type == LISP_TYPE_EOF
      || root_obj->type == LISP_TYPE_PARSE_ERROR)
    {
      fprintf (stderr, "(!)config_file.c/configfile_load(): "
               "lisp_read_file(%s) failed!\n", configname);
      lisp_free (root_obj);
      return TRUE;
    }
  if (strcmp (lisp_symbol (lisp_car (root_obj)), "powermanga-config") != 0)
    {
      fprintf (stderr, "(!)config_file.c/configfile_load(): "
               "lisp_read_file(%s) failed!\n", configname);
      lisp_free (root_obj);
      return TRUE;
    }
  lst = lisp_cdr (root_obj);
  if (!lisp_read_string (lst, "lang", &str))
    {
      power_conf->lang = EN_LANG;
    }
  else
    {
      if (strcmp (str, "fr") == 0)
        {
          power_conf->lang = FR_LANG;
        }
      else
        {
          power_conf->lang = EN_LANG;
        }
    }
  if (!lisp_read_bool (lst, "fullscreen", &power_conf->fullscreen))
    {
      power_conf->fullscreen = TRUE;
    }
  if (!lisp_read_bool (lst, "nosound", &power_conf->nosound))
    {
      power_conf->nosound = FALSE;
    }
  if (!lisp_read_bool (lst, "nosync", &power_conf->nosync))
    {
      power_conf->nosync = FALSE;
    }
  if (!lisp_read_int (lst, "verbose", &power_conf->verbose))
    {
      power_conf->verbose = 0;
    }
  if (!lisp_read_int (lst, "scale_x", &power_conf->scale_x))
    {
      power_conf->scale_x = 2;
    }
  if (power_conf->scale_x < 1 || power_conf->scale_x > 4)
    {
      power_conf->scale_x = 2;
    }
  if (!lisp_read_int (lst, "resolution", &power_conf->resolution))
    {
      power_conf->resolution = 640;
    }
  if (power_conf->resolution != 320 && power_conf->resolution != 640)
    {
      power_conf->resolution = 640;
    }
  if (power_conf->scale_x > 1)
    {
      power_conf->resolution = 640;
    }
  lisp_free (root_obj);
  return TRUE;
}

/** 
 * Save config file "~/.tlkgames/powermanga.conf"
 */
void
configfile_save (void)
{
  FILE *config = fopen_data (configname, "w");
  if (config == NULL)
    {
      return;
    }
  fprintf (config, "(powermanga-config\n");
  fprintf (config, "\t;; the following options can be set to #t or #f:\n");
  fprintf (config, "\t(fullscreen %s)\n",
           power_conf->fullscreen ? "#t" : "#f");
  fprintf (config, "\t(nosound %s)\n", power_conf->nosound ? "#t" : "#f");
  fprintf (config, "\t(nosync %s)\n", power_conf->nosync ? "#t" : "#f");

  fprintf (config, "\n\t;; window size (320 or 640):\n");
  fprintf (config, "\t(resolution  %d)\n", power_conf->resolution);

  fprintf (config, "\n\t;; scale_x (1, 2, 3 or 4):\n");
  fprintf (config, "\t(scale_x   %d)\n", power_conf->scale_x);

  fprintf (config,
           "\n\t;; verbose mode 0 (disabled), 1 (enable) or 2 (more messages)\n");
  fprintf (config, "\t(verbose   %d)\n", power_conf->verbose);

  fprintf (config, "\n\t;; difficulty 0 (easy), 1 (normal) or 2 (hard)\n");
  fprintf (config, "\t(difficulty   %d)\n", power_conf->difficulty);

  fprintf (config, "\n\t;; langage en or fr\n");
  fprintf (config, "\t(lang      ");
  switch (power_conf->lang)
    {
    case FR_LANG:
      fprintf (config, "\"fr\")\n");
      break;
    default:
      fprintf (config, "\"en\")\n");
      break;
    }
  fprintf (config, ")\n");

  fclose (config);
}

/**
 * Release the configuration data structure and filenames strings
 */
void
configfile_free (void)
{
  if (power_conf != NULL)
    {
      free_memory ((char *) power_conf);
      power_conf = NULL;
    }
  if (config_dir != NULL)
    {
      free_memory (config_dir);
      config_dir = NULL;
    }
  if (configname != NULL)
    {
      free_memory (configname);
      configname = NULL;
    }
}

/** 
 * Open a file and return an I/O stream
 * @param fname The filename specified by path 
 * @param fmode Mode parameter
 * @return A pointer to a FILE structure 
 */
FILE *
fopen_data (const char *fname, const char *fmode)
{
  FILE *fi;
  fi = fopen (fname, fmode);
  if (fi == NULL)
    {
      fprintf (stderr, "(!)config_file.c/fopen_data(): "
          "Warning: Unable to open the file \"%s\" ", fname);
      if (strcmp (fmode, "r") == 0)
        {
          fprintf (stderr, "for read!!!\n");
        }
      else if (strcmp (fmode, "w") == 0)
        {
          fprintf (stderr, "for write!!!\n");
        }
    }
  return fi;
}

/** 
 * Scan command line arguments
 * @param arg_count the number of arguments
 * @param arg_values he command line arguments
 * @return FALSE if exit, TRUE otherwise 
 */
bool
configfile_scan_arguments (Sint32 arg_count, char **arg_values)
{
  Sint32 i;
  for (i = 1; i < arg_count; i++)
    {
      if (*arg_values[i] != '-')
        {
          continue;
        }

      /* display help */
      if (!strcmp (arg_values[i], "-h") || !strcmp (arg_values[i], "--help"))
        {
          fprintf (stdout, "\noptions:\n"
                   "-h, --help     print Help (this message) and exit\n"
                   "--version      print version information and exit\n"
                   "--320          game run in a 320*200 window (slow machine)\n"
                   "--2x           scale2x\n"
                   "--3x           scale3x\n" "--4x           scale4x\n");
#ifdef POWERMANGA_SDL
          fprintf (stdout,
                   "--window       windowed mode (full screen by default) \n");
#endif
          fprintf (stdout,
                   "-v             verbose mode\n"
                   "--verbose      verbose mode (more messages)\n"
                   "--nosound      force no sound\n"
                   "--sound        force sound\n"
                   "--nosync       disable timer\n"
                   "--easy         easy bonuses\n"
                   "--hard         hard bonuses\n"
                   "--------------------------------------------------------------\n"
                   "keys recognized during the game:\n"
                   "[Ctrl] + [S]   enable/disable the music\n"
                   "[Ctrl] + [Q]   finish the play current\n"
                   "[Ctrl] + [A]   about Powermanga\n"
                   "[F10]          quit Powermanga\n"
                   "[P]            enable/disable pause\n"
                   "[Page Down]    volume down\n"
                   "[Page Up]      volume up\n");
#ifdef POWERMANGA_SDL
          fprintf (stdout,
                   "F              switch between full screen and windowed mode\n");
#endif
          return FALSE;

        }

      /* print version information and exit */
      if (!strcmp (arg_values[i], "--version"))
        {
          printf (POWERMANGA_VERSION);
          printf ("\n");
          printf ("copyright (c) 1998-2007 TLK Games\n");
          printf ("website: http://linux.tlk.fr/\n");
          return FALSE;
        }
      /* force window mode */
      if (!strcmp (arg_values[i], "--window"))
        {
          power_conf->fullscreen = FALSE;
          continue;
        }

      /* force fullscreen mode */
      if (!strcmp (arg_values[i], "--fullscreen"))
        {
          power_conf->fullscreen = TRUE;
          continue;
        }

      /* resolution, low-res or high-res */
      if (!strcmp (arg_values[i], "--320"))
        {
          power_conf->resolution = 320;
          power_conf->scale_x = 1;
          continue;
        }
      if (!strcmp (arg_values[i], "--640"))
        {
          power_conf->resolution = 640;
          power_conf->scale_x = 1;
          continue;
        }
      if (!strcmp (arg_values[i], "--2x"))
        {
          power_conf->scale_x = 2;
          power_conf->resolution = 640;
          continue;
        }
      if (!strcmp (arg_values[i], "--3x"))
        {
          power_conf->scale_x = 3;
          power_conf->resolution = 640;
          continue;
        }
      if (!strcmp (arg_values[i], "--4x"))
        {
          power_conf->scale_x = 4;
          power_conf->resolution = 640;
          continue;
        }

      /* enable verbose mode */
      if (!strcmp (arg_values[i], "-v"))
        {
          power_conf->verbose = 1;
          continue;
        }
      if (!strcmp (arg_values[i], "--verbose"))
        {
          power_conf->verbose = 2;
          continue;
        }
      if (!strcmp (arg_values[i], "--noverbose"))
        {
          power_conf->verbose = 0;
          continue;
        }

      /* disable sound */
      if (!strcmp (arg_values[i], "--nosound"))
        {
          power_conf->nosound = TRUE;
          continue;
        }

      /* disable sound */
      if (!strcmp (arg_values[i], "--sound"))
        {
          power_conf->nosound = FALSE;
          continue;
        }

      /* disable timer */
      if (!strcmp (arg_values[i], "--nosync"))
        {
          power_conf->nosync = TRUE;
          continue;
        }

      /* difficulty: easy or hard (normal bu default) */
      if (!strcmp (arg_values[i], "--easy"))
        {
          power_conf->difficulty = 0;
          continue;
        }
      if (!strcmp (arg_values[i], "--hard"))
        {
          power_conf->difficulty = 2;
          continue;
        }
    }
  return TRUE;
}

/**
 * Return current language
 * @return current language 'en' or 'fr'
 */
char *
configfile_get_lang (void)
{
  return lang_to_text[power_conf->lang];
}
