/*
 * ----------------------------------------------------------------------------
 * pmud - PMU daemon for Linux/PPC on Powerbooks.
 * At present this supports the PB2400/3400/3500,
 * the 1999 G3 Powerbooks (aka "Lombard"), and the 
 * previous generation of G3 Powerbooks (aka "Wallstreet").
 *
 * Copyright 1999 Paul Mackerras.
 * Heavily modified by Stephan Leemburg.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------
 * $Log: pmud.c,v $
 * Revision 1.1.1.1  2001/12/07 11:31:46  sleemburg
 * Initial CVS import of the unreleased pmud-0.8 to apmud (new project name
 * because of a name clash at sourceforge.net).
 *
 * Revision 1.12  2001/11/09 10:55:29  stephan
 * use "pmu.h"
 *
 * Revision 1.11  2001/11/09 10:47:30  stephan
 * added -k option for my Titanium
 *
 * Revision 1.10  2000/11/21 20:58:32  stephan
 * bugfix for transposed mouse/trackpad settings. Thanks Michael Schmitz!
 *
 * Revision 1.9  2000/10/09 14:17:24  stephan
 * reset trackpad settings
 * calculate percentage power correct with two batteries
 * fixed compile warning on sigaction initialising
 * powerbook 3400 APM support
 * on unsupported new machines, shutdown the machine in stead of sleep
 * facility commandline set now works
 *
 * Revision 1.8  2000/05/11 14:53:40  stephan
 * intro pseudo apm
 *
 * Revision 1.7  2000/03/25 21:25:13  stephan
 * wakeup argument to pwrctl
 * only debugging syslog messages when debugging is enabled
 * warning argument to pwrctl
 *
 * Revision 1.6  2000/03/09 13:25:33  stephan
 * some security checking in run_program
 *
 * Revision 1.5  2000/03/09 12:57:54  stephan
 * pmud-0.4 modifications:
 * o avoid SIGPIPE faults
 * o pmu type is now correctly printed
 * o powerlevel is now set on detection of change to avoid constant
 *   calling of pwrctl on the first change
 * o arguments to pwrctl extended
 * o various other little things
 *
 * Revision 1.4  2000/01/11 09:01:30  stephan
 * display of machine type was wrong
 *
 * Revision 1.3  2000/01/06 22:52:31  stephan
 * changed -l and -m argument from minutes to seconds
 *
 * Revision 1.2  2000/01/06 13:48:19  stephan
 * part of pmud package
 * ----------------------------------------------------------------------------
 */
static char *rcsid = "@(#)$Id: pmud.c,v 1.1.1.1 2001/12/07 11:31:46 sleemburg Exp $";

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/poll.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <sys/kd.h>
#include <getopt.h>
#include <signal.h>
#include <linux/fb.h>
#include <linux/adb.h>
#include "pmu.h"
#include "pmud.h"
#include "apm.h"

#define chk(x, mess)	if (x) { syslog(LOG_ERR, mess); bye(EXIT_FAILURE,mess);}
#define novm(p, what)	if ((p) == 0) { \
			syslog(LOG_ERR, "Couldn't allocate memory for "what); \
			bye(EXIT_FAILURE, what); }

#define INBSZ	128
#define OUTBSZ	128

#define STAT_OUTSTANDING	2

#define FLG_DEBUG		(1<<0)
#define FLG_SIGPWR		(1<<1)
#define FLG_SHUTDOWN		(1<<2)
#define FLG_NODETACH		(1<<3)
#define FLG_APM			(1<<4)
#define FLG_WARN		(1<<5)
#define FLG_NOOFFB		(1<<6)
#define FLG_NOLID		(1<<7)
#define FLG_NOACLID		(1<<8)
#define FLG_NOTCP		(1<<9)

#define TPS_TAP			(1<<0)
#define TPS_DRAG		(1<<1)
#define TPS_LOCK		(1<<2)

struct sockstate {
	int	fd;
	int	nstat;		/* number of statuses we can send */
	int	inb_cnt;
	char	inbuf[INBSZ];
	int	out_pending;
	char	*outptr;
	char	outbuf[OUTBSZ];
	struct sockstate *next;
	struct sockstate **prevp;
};

struct fd_info {
	void	(*proc)(int, void *, int);
	void	*arg;
} *polli;


#define ADBlisten	2
#define ADBtalk		3
#define ADB_DATA_LEN	32

typedef struct adbpacket_t
{
	char preamble;
	char control;
	char data[ADB_DATA_LEN];
} adbpacket_t;

struct my_apm_info apm;

int lid_closed = 0;
int pmu_fd;
int adb_fd;
int apm_fd;
int fb0_fd;
int pmu_version;
int powermode;				/* 0 for battery, 1 for AC */
int powerlevel = -1;			/* current power level setting */
int powerlevels[2] = { 1, 3 };		/* settings for battery & AC */
int chloralhydrate;			/* makes you sleepy */
int critical_time;              	/* # seconds of critically low power */
int critical_left = CRITICAL_TIMELEFT;	/* # seconds of power left */
int critical_margin = CRITICAL_TIME;	
struct sockstate *all_sockets;
struct timeval pmu_poll_time;
char *apmfile = APM_FILE;
char sock_welcome[64];
struct pollfd *polls;
int ipoll_current;
int nall_polls;
int npolls;
int flags = 0;
int ntick;

static int do_poll(unsigned);
static int do_sigpower(char);
static int get_pmu_version(void);
static int open_apm(char *);
static int pmu_op(unsigned char, unsigned char*, int, void*, int , const char*);
static int send_socket(struct sockstate *, char *, int);
static int sock_write(struct sockstate *, char *, int);
static int usage(char *, int);
static int get_trackpad_id(int);
static int adb_program(int, int, int);
static int get_trackpad_settings(int, int);
static int set_trackpad_settings(int, int, int);
static int adbread(int, char, char *, int);
static int adbwrite(int, char, char, char, char *, int);
static int validate_program(char *);
static void add_fd(int, int, void (*proc)(int, void *, int), void *);
static void apm_data(int, struct my_apm_info *);
static void bye(int, const char*);
static void conn_sock(int, void *, int);
static void delete_socket(struct sockstate *);
static void do_cmd(struct sockstate *);
static void do_power(struct sockstate *, char *);
static void do_signal(int);
static void pmu_intr(int, void *, int);
static void poll_init(void);
static void poll_pmu(void);
static void read_sock(int, void *, int);
static void run_program(char *, char *, char *);
static void send_more(int, void *, int);
static void set_power(int, int);
static void snooze(int);
static void fbon(int);
static void beep(unsigned int ms, unsigned int freq);


int main(int ac, char **av)
{
	struct timeval now, tleft;
	unsigned tms;
	int s, len, fl;
	struct sockaddr *sa;
	struct sockaddr_in sin;
	struct sockaddr_un sun;
	FILE *f;

	struct option options[] = 
	{
		{"apm", 2, 0, 'a'},
		{"debug", 0, 0, 'd'},
		{"facility", 1, 0, 'f'},
		{"help", 0, 0, 'h'},
		{"critical-left", 1, 0, 'l'},
		{"critical-margin", 1, 0, 'm'},
		{"nolid", 0, 0, 'k'},
		{"noaclid", 0, 0, 'k'},
		{"nodetach", 0, 0, 'n'},
		{"port", 1, 0, 'p'},
		{"sigpower", 0, 0, 's'},
		{"shutdown", 0, 0, 'S'},
		{"af_unix", 0, 0, 'u'},
		{"version", 0, 0, 'v'},
		{0, 0, 0, 0}
	};
	int opt;
	unsigned short port = PORT;
	int syslog_options = PMU_SYSLOG_OPTIONS;
	int facility = PMU_FACILITY;
	struct sigaction sigpipe = { {SIG_IGN}, {{0}}, SA_RESTART, 0 };
	struct sigaction sigterm = {{do_signal}, {{0}}, SA_RESTART , 0 };

	// there is a bug in basename() 

	char *p = strrchr(*av, '/');

	p = p ? p + 1 : *av;

	while((opt=getopt_long(ac,av, "a::df:hKkl:m:nop:sSuv", options,0))!=EOF)
		switch(opt)
		{
		case 'a':
			flags |= FLG_APM;
			if(optarg && *optarg)
				apmfile = optarg;
			break;

		case 'd':
			flags |= (FLG_DEBUG|FLG_NODETACH);
			syslog_options |= LOG_PERROR;
			break;
		case 'f':
			if(!strcasecmp(optarg, "daemon"))
				facility = LOG_DAEMON;
			else if(!strcasecmp(optarg, "user"))
				facility = LOG_USER;
			else if(!strcasecmp(optarg, "local0"))
				facility = LOG_LOCAL0;
			else if(!strcasecmp(optarg, "local1"))
				facility = LOG_LOCAL1;
			else if(!strcasecmp(optarg, "local2"))
				facility = LOG_LOCAL2;
			else if(!strcasecmp(optarg, "local3"))
				facility = LOG_LOCAL3;
			else if(!strcasecmp(optarg, "local4"))
				facility = LOG_LOCAL4;
			else if(!strcasecmp(optarg, "local5"))
				facility = LOG_LOCAL5;
			else if(!strcasecmp(optarg, "local6"))
				facility = LOG_LOCAL6;
			else if(!strcasecmp(optarg, "local7"))
				facility = LOG_LOCAL7;
			else
				return usage(p, EXIT_FAILURE);
			break;
		case 'K':
			flags |= FLG_NOACLID;
		case 'k':
			flags |= FLG_NOLID;
			break;
		case 'l':
			critical_left = atoi(optarg);
			break;
		case 'm':
			critical_margin = atoi(optarg);
			break;
		case 'n':
			flags |= FLG_NODETACH;
			break;
		case 'o':
			flags |= FLG_NOOFFB;
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 's':
			flags |= FLG_SIGPWR;
			break;
		case 'S':
			flags |= FLG_SHUTDOWN|FLG_SIGPWR;
			break;
		case 'u':
			flags |= FLG_NOTCP;
			break;
		case 'v':
			printf("%s\n", rcsid);
			return 0;

		case 'h':
		default :
			return usage(p, opt == 'h' ? EXIT_SUCCESS:EXIT_FAILURE);
		}
	openlog(p, syslog_options, facility);

	if(sigaction(SIGTERM, &sigterm, 0))
		bye(EXIT_FAILURE, "could not install SIGTERM handler");

	if(sigaction(SIGPIPE, &sigpipe, 0))
		bye(EXIT_FAILURE, "could not install SIGPIPE handler");

	syslog(LOG_INFO, "%s [treshold = %d, margin = %d] started",
		p, 
		critical_left, 
		critical_margin
		);

	if(!validate_program(POWERLEVEL_SCRIPT))
		bye(EXIT_FAILURE, 
			POWERLEVEL_SCRIPT " is insecure, cannot start");

	poll_init();

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "opening " ADB_FILE);

	adb_fd = open(ADB_FILE, O_RDWR);
	chk(adb_fd < 0, "Couldn't open " ADB_FILE);

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "opening " PMU_FILE);

	pmu_fd = open(PMU_FILE, O_RDWR);
	if(pmu_fd < 0)
		pmu_fd = open(PMU_DEVFS_FILE, O_RDWR);

	chk(pmu_fd < 0, "Couldn't open " PMU_FILE " or " PMU_DEVFS_FILE);

	chk(fcntl(pmu_fd, F_GETFL, &fl) < 0, "fcntl(F_GETFL)");

	fl |= O_NONBLOCK;
	chk(fcntl(pmu_fd, F_SETFL, &fl) < 0, "fcntl(F_SETFL)");

	add_fd(pmu_fd, POLLIN, pmu_intr, NULL);

	fb0_fd = open(FB0_FILE, O_RDONLY);
	chk(fb0_fd < 0, "Couldn't open " FB0_FILE);


	if(!(flags&FLG_NOTCP))
	{
		s = socket(PF_INET, SOCK_STREAM, 0);
		chk(s < 0, "Couldn't create socket");

		sin.sin_family = AF_INET;
		sin.sin_port = htons(port);
		sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);

		len = sizeof(sin);
		sa = (struct sockaddr*)&sin;
	}
	else
	{
		s = socket(PF_UNIX, SOCK_STREAM, 0);
		chk(s < 0, "Couldn't create socket");

		sun.sun_family = AF_UNIX;
		strcpy(sun.sun_path, SOCKET_FILE);

		chk(fchmod(s, 0600) < 0, "Couldn't change socket mode");

		len = sizeof(sun);
		sa = (struct sockaddr*)&sun;
	}

	chk(bind(s, sa, len) < 0, "bind socket");

	chk(listen(s, 5) < 0, "socket listen");

	add_fd(s, POLLIN|POLLPRI, conn_sock, NULL);

	chk(getsockname(s, sa, &len) < 0, "getsockname");

	if(!get_pmu_version())
		bye(EXIT_FAILURE, "incorrect PMU version");

	if(flags&FLG_APM)
	{
#if 0
		if(pmu_version == PMU_VERSION_3400) 
		{
			flags &= ~FLG_APM;
			syslog(LOG_WARNING, "no apm support for 3400 present");
		}
		else
#endif
			apm_fd = open_apm(apmfile);
	}

	sprintf(sock_welcome, "pmud %s %d\n", PMUD_VERSION, pmu_version);

	if(!(flags&FLG_NODETACH))
	{
		int pid = fork();

		chk(pid < 0, "Couldn't fork");
		if (pid > 0)
			bye(0, "parent exiting [note PID]");

		chdir("/");
		setsid();
		close(0);
		close(1);
		close(2);
	}

	if ((f = fopen(POWER_FILE, "r")) != 0) 
	{
		fscanf(f, "%d %d", &powerlevels[0], &powerlevels[1]);
		fclose(f);

		if(flags&FLG_DEBUG)
			syslog(LOG_DEBUG, "powerlevels from %s: %d %d",
				POWER_FILE,
				powerlevels[0],
				powerlevels[0]
				);

		if(powerlevels[0] < 1 || powerlevels[0] > 3)
		{
			syslog(LOG_ERR, "invalid powerlevel (%d)"
					" for battery: setting to minimum",
					powerlevels[0]
				);
				powerlevels[0] = 1;
		}
		if(powerlevels[1] < 1 || powerlevels[1] > 3)
		{
			syslog(LOG_ERR, "invalid powerlevel (%d)"
					" for AC: setting to maximum",
					powerlevels[1]
				);
				powerlevels[1] = 2;
		}
	}

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "starting monitoring loop");

	gettimeofday(&pmu_poll_time, NULL);

	for(chloralhydrate = 0; ; )
	{
		gettimeofday(&now, NULL);

		if (!timercmp(&pmu_poll_time, &now, >))
			poll_pmu();

		if (chloralhydrate) 
		{
			if(flags&FLG_SHUTDOWN)
			{
				if(do_sigpower('F'))
					sleep(120);
			}
			else	snooze(powermode);

			chloralhydrate = 0;
			critical_time = 0;
			continue;
		}
		gettimeofday(&now, NULL);
		timersub(&pmu_poll_time, &now, &tleft);
		tms = tleft.tv_sec * 1000 + tleft.tv_usec / 1000 + 1;

		if(lid_closed)
		{
			unsigned char switches;

			if(pmu_op(PMU_GET_COVER,0,0,&switches,1,"switch req")<0)
				switches = 1;

			lid_closed = switches & 1;
			if(!lid_closed)
			{
				fbon(1);
				 run_program(POWERLEVEL_SCRIPT, 
						 "lid-opened",
						ac ? "ac" : "battery"
					);
			}
		}
		if(!do_poll(tms))
			bye(EXIT_FAILURE, "could not poll PMU");
	}
}

static int validate_program(char *p)
{
	struct stat sb;

	if(p && *p && !stat(p, &sb))
	{
		if(!sb.st_uid		// owned by root
		&& !(sb.st_mode & 022))	// only writeable by root
			return 1;
	}
	return 0;
}

static void poll_init(void)
{
	nall_polls = 10;
	polls = malloc(nall_polls * sizeof(struct pollfd));
	novm(polls, "poll table");
	polli = malloc(nall_polls * sizeof(struct fd_info));
	novm(polli, "poll info table");
	npolls = 0;
}

static void add_fd(
		int fd, 
		int events, 
		void (*proc)(int, void *, int), 
		void *arg
		)
{
	if (npolls >= nall_polls) 
	{
		nall_polls = npolls + 10;
		polls = realloc(polls, nall_polls * sizeof(struct pollfd));
		polli = realloc(polli, nall_polls * sizeof(struct fd_info));
	}
	polls[npolls].fd = fd;
	polls[npolls].events = events;
	polls[npolls].revents = 0;
	polli[npolls].proc = proc;
	polli[npolls].arg = arg;
	++npolls;
}

static void rem_fd(int fd, int events)
{
	int i;

	for (i = 0; i < npolls; ++i)
		if (polls[i].fd == fd && polls[i].events == events)
			break;

	if (i >= npolls)
		return;

	for (--npolls; i < npolls; ++i) 
	{
		if (i == ipoll_current)
			--ipoll_current;

		polls[i] = polls[i+1];
		polli[i] = polli[i+1];
		++i;
	}
}

static int do_poll(unsigned to)
{
	int i, nr;

	nr = poll(polls, npolls, to);
	if (nr < 0) 
	{
		if (errno == EINTR)
			return 1;

		syslog(LOG_ERR, "poll (%m)");
		return 0;
	}

	for (i = 0; i < npolls && nr > 0; ++i) 
	{
		ipoll_current = i;
		if (polls[i].revents) 
		{
			if (polls[i].revents & POLLNVAL)
				rem_fd(polls[i].fd, polls[i].events);
			else
				polli[i].proc(polls[i].fd, polli[i].arg,
					      polls[i].revents);
			--nr;
		}
		i = ipoll_current;
	}
	return 1;
}

static int usage(char *name, int ret)
{
	printf("usage: %s [args]\n\n"
		"\twhere [args] is:\n\n"
		"\t-a [<fifo>]  : emulate an apm file in <fifo>\n"
		"\t-d           : enable debugging text; this flag implicitly\n"
		"\t               sets the -n flag to prevent backgrounding.\n"
		"\t-f <facility>: use <facility> for syslogd(8) messages.\n"
		"\t-h           : this text.\n"
		"\t-k           : do not detect lid close\n"
		"\t-K           : do not detect lid close when on AC\n"
		"\t-l <seconds> : the numbers of <seconds> power left which\n"
		"\t               is considered to be a critical low level.\n"
		"\t-m <seconds> : the number of <seconds> a critical low (see\n"
		"\t               the -l flag) power level has to endure in\n"
		"\t               order to enable sleeping|shutdown.\n"
		"\t-n           : do not detach. With this option the daemon\n"
		"\t               will not become a background process.\n"
		"\t-p <port>    : listen on <port> for requestes.\n"
		"\t-s           : send SIGPWR to init(8) on low power level;\n"
		"\t               this does not put the machine to sleep as\n"
		"\t               another SIGPWR needs to be send to init(8)\n"
		"\t               when power is restored.\n"
		"\t-S           : same as -s, but also send SIGPWR on every\n"
		"\t               sleep request or closing of the lid\n"
		"\t-u           : communicate through a AF_UNIX socket;\n"
		"\t-v           : print the version string of %s.\n\n"
		"\tfor the following flags, you can also use long-flags\n\n"
		"\t-a [<fifo>]  : --apm [<fifo>]\n"
		"\t-d           : --debug\n"
		"\t-f <facility>: --facility <facility>\n"
		"\t-h           : --help\n"
		"\t-k           : --nolid\n"
		"\t-K           : --noaclid\n"
		"\t-l <seconds> : --critical-left <seconds>\n"
		"\t-m <seconds> : --critical-margin <seconds>\n"
		"\t-p <port>    : --port <port>\n"
		"\t-s           : --sigpower\n"
		"\t-v           : --version\n",
		name,
		name
		);

	return ret;
}

static int pmu_op(
		unsigned char cmd, 
		unsigned char *par, 
		int npar,
		void *res, 
		int minres, 
		const char *desc
		)
{
	unsigned char data[32];
	int i, n;

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "pmu_op(0x%02x, ..., %s)\n", 	
			cmd, 
			desc ? desc : ""
			);

	data[0] = 6;		/* packet for PMU */
	data[1] = cmd;

	for (i = 0; i < npar; ++i)
		data[i+2] = par[i];

	chk(write(adb_fd, data, npar+2) < 0, desc);

	n = read(adb_fd, data, sizeof(data));
	chk(n < 0, desc);

	if (n < minres + 1 || data[0] != 0) 
	{
		syslog(LOG_ERR, "huh? %d bytes from PMU:", n);

		for (i = 0; i < n; ++i)
			syslog(LOG_ERR, " %.2x", data[i]);

		return -1;
	}

	if (minres > 0)
		memcpy(res, data+1, minres);

	return n - 1;
}

static int get_pmu_version(void)
{
	unsigned char vers;
	char *version[] = {
		"unknown (and unsupported)",		// 0
		"unknown (and unsupported)",		// 1
		"unknown (and unsupported)",		// 2
		"unknown (and unsupported)",		// 3
		"unknown (and unsupported)",		// 4
		"unknown (and unsupported)",		// 5
		"unknown (and unsupported)",		// 6
		"unknown (and unsupported)",		// 7
		"unknown (and unsupported)",		// 8
		"2400/3400/3500", 			// 9
		"G3 Wallstreet",			// 10
		"1999 G3 Lombard",			// 11
		"iBook/G3 Pismo/G4 Titanium",		// 12
		"Wow, later than an Titanium! ;-)" 	// > 12
	};
#define max_version (sizeof(version)/sizeof(char*))
	static char env[16];

	if (pmu_op(PMU_GET_VERSION, 0, 0, &vers, 1, "version req") < 0)
		bye(1, "could not determine PMU version");

	pmu_version = abs(vers) > max_version ? max_version - 1 : vers;

	printf("PMU version %d: %s\n", vers, version[pmu_version]);

	syslog(LOG_INFO, "PMU version %d: %s", vers, version[pmu_version]);

	switch(vers) 
	{
	case PMU_VERSION_3400:
	case PMU_VERSION_WALLSTREET:
	case PMU_VERSION_LOMBARD:
	case PMU_VERSION_KEYLARGO:
		break;

#ifdef PMU_VERSION_UNSUPPORTED
	case PMU_VERSION_UNSUPPORTED:
		/* HACK: shut the machine down when lid is closed */
		syslog(LOG_WARNING, "Sleep for this PMU unsupported: "
			"will shutdown the machine on sleep request");

		flags |= FLG_SIGPWR;

		break;
#endif

	default:
		return 0;
	}
	/* set an environment variable for pwrctl script to use */

	sprintf(env, "PMUVERSION=%d", vers);
	putenv(env);

	return 1;
}

static void poll_pmu(void)
{
	struct sockstate *ss, *ssnext;
	int n, pwr, vb, i;
	char status[64];
	unsigned char switches;
	struct timeval now;
	int timeleft = -1;
	static int state = 0;
	int charging = 0;

	if (pmu_op(PMU_GET_COVER, 0, 0, &switches, 1, "switch req") < 0)
		return;

	if (pmu_version == PMU_VERSION_3400) 
	{
		/* use extended battery request (6b) */

		float j;
		unsigned char reply[8];
		int pcharge, charge=0;
		int current=0;
		float lrange[] = { 0.0, 275.0, 850.0, 1680.0, 2325.0, 
				2765.0, 3160.0, 3500.0, 3830.0, 4115.0, 
				4360.0, 4585.0, 4795.0, 4990.0, 5170.0, 
				5340.0, 5510.0, 5710.0, 5930.0, 6150.0, 
				6370.0, 6500.0
				};

		if (pmu_op(PMU_BATTERY_STATE, 0, 0, reply, 8, "batt state") < 0)
			return;

		pwr = reply[0] & 1;			/* AC indicator */
		charging = reply[0] & 2;
		vb = (reply[1] << 8) + reply[2];	/* battery voltage */

		for (i = 0; i < 8; ++i)
			status[i] = ((reply[0] >> (7 - i)) & 1) + '0';

		sprintf(status+8, " %d %d %d %d %d %x\n", vb, reply[3],
			reply[4], reply[5], (reply[6] << 8) + reply[7],
			switches);

		if (!pwr) 
		{
			if (reply[5] > 200)
		    		vb += (int)((reply[5] - 200) * 0.15);
		} 
		else if (charging)
			vb -= 10;

		j = (330 - vb) / 10.0;
		i = (int) (j);

		if (i <= 0)
			charge = 0;
		else if (i >= 21)
			charge = 6500;
		else
			charge = (lrange[i+1]-lrange[i])*(j-(float)i)+lrange[i];

		charge = (1000 - charge / 6.5) / 10;
		if (reply[0]&0x40) 
		{
			pcharge = (reply[6] << 8) + reply[7];//pcharge

			if (pcharge > 6500)
				pcharge = 6500;

			pcharge = (1000 - pcharge / 6.5) / 10;

			if (pcharge < charge)
				charge = pcharge;
		}
		current=reply[5];

		if (!pwr && (current > 0))
			timeleft = (int)((charge * 274) / current) * 60;

		apm.battery_percentage = charge;

		if (apm.battery_percentage > 100)
			apm.battery_percentage = 100;
	} 
	else 
	{
		/* wallstreet/lombard, use smart battery request (6f) */

		signed short batt[5], batt1, batt2;
		unsigned char par;
		int i, j;
		int charge, current;
		char *p = status;

		*p++ = 'S';
		pwr = 0;
		charge = current = 0;
		batt1 = batt2 = 0;

		for (i = 0; i < 2; ++i) 
		{
			par = i + 1;

			if (pmu_op(PMU_SMART_BATT, &par, 1, batt, sizeof(batt),
				   "smart battery req") < 0)
				return;

			*p++ = ' ';
			*p++ = '{';

			for (j = 0; j < 3; ++j)
				*p++ = ((batt[0] >> j) & 1) + '0';

			pwr |= batt[0] & 1;

			if (batt[0] & 4) 
			{
				/* have a battery */

				charge += batt[1];
				current += batt[3];

				for (j = 1; j < 5; ++j) 
				{
					sprintf(p, " %d", batt[j]);
					p += strlen(p);
				}

				if(!i)
				{
					batt1 = batt[1];
					batt2 = batt[2];

					/* 
					 * charging flag valid and 
					 * charging battery? 
					 */
					charging = batt[0] & 0x02;
				}
				else
				{
					batt1 += batt[1];
					batt2 += batt[2];

					/* 
					 * charging flag valid and 
					 * charging battery? 
					 */
					charging |= batt[0] & 0x02;
				}
				apm.battery_percentage  = batt1 * 100;
				apm.battery_percentage /= batt2 ? batt2 : 1;
			}
			*p++ = '}';
		}
		*p++ = '\n';
		*p = 0;

		if (current < 0)
			timeleft = charge * 3552 / -current;

	}

	/* apm pseudo info */

	if(flags&FLG_APM)
	{
		apm.apm_flags = 0;
		apm.ac_line_status = pwr;
		apm.battery_time = pwr ? 0 : timeleft;
		apm.using_minutes = 0;


		if(apm.battery_percentage <= APM_CRITICAL)
		{
			apm.battery_status = 0x02;
			apm.battery_flags = 0x04;
		}
		else if(apm.battery_percentage <= APM_LOW)
		{
			apm.battery_status = 0x01;
			apm.battery_flags = 0x02;
		}
		else
		{
			apm.battery_status = 0x00;
			apm.battery_flags = 0x01;
		}

		if(charging)
		{
			apm.battery_status = 0x03;
			apm.battery_flags |= 0x08;
		}
	}

	n = strlen(status);
	for (ss = all_sockets; ss != 0; ss = ssnext) 
	{
		ssnext = ss->next;

		if (ss->nstat >= STAT_OUTSTANDING)
			continue;

		switch (send_socket(ss, status, n)) 
		{
		case 1:
			++ss->nstat;
			break;
		case -1:
			delete_socket(ss);
			break;
		}
	}

	if ((switches & 1) != 0) 
	{
		if(!lid_closed)
		{
			lid_closed=1;

			fbon(0);

			run_program(POWERLEVEL_SCRIPT, 
					"lid-closed",
					pwr ? "ac" : "battery"
				);
		}

		if(!(flags&FLG_NOLID) || (!pwr && (flags&FLG_NOACLID)))
		{
			syslog(LOG_INFO, "lid closed: request sleep");
			chloralhydrate = 1;	/* lid is closed, go to sleep */
		}
	}		

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "timeleft = %d", timeleft);

	if (!pwr && timeleft >= 0 && timeleft < critical_left) 
	{
		apm.battery_status = 2; /* critical */

		beep(BEEP_TIME, BEEP_WARN);

		if(!(state&FLG_SIGPWR))
		{
			if((critical_time == 5 
			|| critical_time >= critical_margin)
			&& !(state&FLG_WARN))
			{

				run_program(POWERLEVEL_SCRIPT, 
						"warning", 
						"battery"
					);
				state |= FLG_WARN;
			}

			if (++critical_time >= critical_margin) 
			{
				beep(BEEP_TIME, BEEP_WARN);

				syslog(LOG_CRIT, "battery critically low: %s\n",
					flags&FLG_SIGPWR ?
					"sending SIG_PWR to init" :
					"request system sleep"
					);

				if((flags&FLG_SIGPWR) && do_sigpower('F'))
					state |= FLG_SIGPWR;
				else 
					chloralhydrate = 1;
			}
			else if(flags&FLG_DEBUG)
				syslog(LOG_DEBUG, "critical margin not "
						"reached %d : %d",
						critical_time,
						critical_margin
						);
		} 
	} 
	else
	{
		critical_time = 0;
		if(state&FLG_SIGPWR)
		{
			state &= ~FLG_SIGPWR;
			do_sigpower('O');
		}
		state &= ~FLG_WARN;
	}

	powermode = pwr;
	if (powerlevels[pwr] != powerlevel)
	{
		set_power(powerlevels[pwr], powermode);
		powerlevel = powerlevels[pwr];
	}

	gettimeofday(&now, NULL);
	++pmu_poll_time.tv_sec;

	if (timercmp(&pmu_poll_time, &now, <)) 
	{
		pmu_poll_time = now;
		++pmu_poll_time.tv_sec;
	}

	if(flags&FLG_APM)
	{
		if(apm_fd < 0)
			apm_fd = open_apm(apmfile);
		if(apm_fd >= 0)
			apm_data(apm_fd, &apm);
	}

}

static void pmu_intr(int fd, void *arg, int ev)
{
	unsigned char intr[16];
	int i, n;
	char str[8];

	n = read(fd, intr, sizeof(intr));
	if (n <= 0) 
	{
		if (n < 0)
			syslog(LOG_ERR, "read pmu (%m)");

		rem_fd(fd, POLLIN);
		return;
	}
	switch (intr[0]) 
	{
	case 0x80:		/* tick interrupt */
		if (n != 1)
			break;

		++ntick;
		return;
	case 4:			/* PC card eject button */
		if (n < 3 || intr[2] < 1 || intr[2] > 2)
			break;

		if(flags&FLG_DEBUG)
			syslog(LOG_DEBUG, "eject of PC card noticed");

		sprintf(str, "%d", intr[2] - 1);
		run_program("/sbin/cardctl", "eject", str);
		return;
	}

	if(flags&FLG_DEBUG)
	{
		syslog(LOG_INFO, "PMU interrupt:");
		for (i = 0; i < n; ++i)
			syslog(LOG_INFO, " %.2x", intr[i]);
	}
}

static int send_socket(struct sockstate *ss, char *p, int nb)
{
	int n;
	
	if (ss->out_pending != 0)
		return 0;

	n = sock_write(ss, p, nb);
	if (n < 0)
		return -1;

	if ((nb -= n) > 0) 
	{
		memcpy(ss->outbuf, p + n, nb);
		ss->outptr = ss->outbuf;
		ss->out_pending = nb;
		add_fd(ss->fd, POLLOUT, send_more, ss);
	}
	return 1;
}

static void send_more(int fd, void *arg, int ev)
{
	struct sockstate *ss = arg;
	int n;

	if (ss->out_pending == 0)
		return;

	n = sock_write(ss, ss->outptr, ss->out_pending);
	if (n < 0) 
	{
		/* oops, looks bad */
		delete_socket(ss);
		return;
	}
	ss->outptr += n;
	ss->out_pending -= n;

	if (ss->out_pending == 0)
		rem_fd(fd, POLLOUT);
}

static int sock_write(struct sockstate *ss, char *ptr, int nb)
{
	int n;

	n = write(ss->fd, ptr, nb);
	if (n < 0) 
	{
		if (errno == EWOULDBLOCK || errno == EAGAIN)
			n = 0;
		else
			syslog(LOG_ERR, "write socket (%m)");
	}
	return n;
}

static void conn_sock(int fd, void *arg, int ev)
{
	int fe, fl;
	struct sockstate *ss;

	fe = accept(fd, NULL, 0);
	chk(fcntl(fe, F_GETFL, &fl) < 0, "fcntl(F_GETFL)");

	fl |= O_NONBLOCK;
	chk(fcntl(fe, F_SETFL, &fl) < 0, "fcntl(F_SETFL)");

	ss = malloc(sizeof(*ss));
	if (ss == 0) 
	{
		syslog(LOG_ERR, "warning: no space for socket state\n");
		close(fe);
		return;
	}
	memset(ss, 0, sizeof(*ss));
	ss->fd = fe;
	ss->next = all_sockets;
	ss->prevp = &all_sockets;

	if (all_sockets != 0)
		all_sockets->prevp = &ss->next;

	all_sockets = ss;
	add_fd(fe, POLLIN, read_sock, ss);
	send_socket(ss, sock_welcome, strlen(sock_welcome));
}

static void delete_socket(struct sockstate *ss)
{
	rem_fd(ss->fd, POLLIN);
	rem_fd(ss->fd, POLLOUT);
	close(ss->fd);
	*ss->prevp = ss->next;
	free(ss);
}

static void read_sock(int fd, void *arg, int ev)
{
	struct sockstate *ss = arg;
	int n;
	int c = ss->inb_cnt;
	char *p;

	n = read(fd, ss->inbuf + c, sizeof(ss->inbuf) - c - 1);
	if (n <= 0) 
	{
		if (n < 0) 
		{
			if (errno == EWOULDBLOCK || errno == EAGAIN)
				return;

			if (errno != ECONNRESET)
				syslog(LOG_ERR, "read socket (%m)");
		}
		delete_socket(ss);
		return;
	}
	ss->inbuf[c + n] = 0;
	p = strchr(&ss->inbuf[c], '\n');
	c += n;

	if (p == 0) 
	{
		if (c < sizeof(ss->inbuf) - 1) 
		{
			ss->inb_cnt = c;
			return;
		}
	} 
	else 
	{
		c = p - ss->inbuf;
		*p++ = 0;
	}

	if (c > 0 && ss->inbuf[c-1] == '\r')
		ss->inbuf[c-1] = 0;

	do_cmd(ss);

	c = 0;
	if (p != 0 && *p != 0) 
	{
		c = strlen(p);
		memmove(ss->inbuf, p, c);
	}
	ss->inb_cnt = c;
}

static void do_cmd(struct sockstate *ss)
{
	char *p, *q;

	for (p = ss->inbuf; isspace(*p); ++p)
		;

	if (*p == 0) 
	{
		/* acknowledgement of status */
		if (ss->nstat)
			--ss->nstat;

		return;
	}

	for (q = p; *q != 0 && !isspace(*q); ++q)
		;

	if (*q != 0) 
		for (*q++ = 0; isspace(*q); ++q)
			;

	if (strcmp(p, "sleep") == 0)
	{
		syslog(LOG_INFO, "initiating user requested sleep");
		chloralhydrate = 1; 	/* go to sleep */
	}
	else if (strcmp(p, "power") == 0)
		do_power(ss, q);
}

static void do_power(struct sockstate *ss, char *args)
{
	int p1, p2;
	char str[32];
	FILE *f;

	switch (sscanf(args, "%d %d", &p1, &p2)) 
	{
	case -1:
	case 0:
		sprintf(str, "%d %d\n", powerlevels[0], powerlevels[1]);
		send_socket(ss, str, strlen(str));
		return;
	case 1:
		powerlevels[powermode] = p1;
		break;
	case 2:
		powerlevels[0] = p1;
		powerlevels[1] = p2;
		break;
	}

	f = fopen(POWER_FILE, "w");
	if (f != 0) 
	{
		if(flags&FLG_DEBUG)
			syslog(LOG_DEBUG, "writing %d %d to %s",
				powerlevels[0],
				powerlevels[1],
				POWER_FILE
				);

		fprintf(f, "%d %d\n", powerlevels[0], powerlevels[1]);
		fclose(f);
	} 
	else
		syslog(LOG_ERR, "Couldn't store power settings in " POWER_FILE);

	if (powerlevels[powermode] != powerlevel)
	{
		set_power(powerlevels[powermode], powermode);
		powerlevel = powerlevels[powermode];
	}
}

static void set_power(int level, int ac)
{
	run_program(POWERLEVEL_SCRIPT, 
			level == 1 ? "minimum" :
			level == 2 ? "medium"  :
			level == 3 ? "maximum" :
			"error",
			ac ? "ac" : "battery"
			);
}

static void snooze(int ac)
{
	int ts = 0; /* initialized to stop gcc from complaining */
	int level;
	int id;

	if(!(flags&FLG_NOOFFB))
		(void) ioctl(pmu_fd, PMU_IOC_GET_BACKLIGHT, &level);

	fbon(0);
	lid_closed=1;

	id = get_trackpad_id(adb_fd);
	if(id >= 0)
		ts = get_trackpad_settings(adb_fd, id);
	
	beep(BEEP_TIME, BEEP_OK);

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "calling sync()");
	sync();

	run_program(POWERLEVEL_SCRIPT, "sleep", ac ? "ac" : "battery");

	syslog(LOG_INFO, "going to sleep");
	
	if (ioctl(pmu_fd, PMU_IOC_SLEEP, 0) < 0)
		syslog(LOG_ALERT, "cannot put system to sleep! (%m)");
	else
		syslog(LOG_INFO, "system awake again");

	beep(BEEP_TIME, BEEP_OK);

	if(id >= 0)
		set_trackpad_settings(adb_fd, id, ts);

	gettimeofday(&pmu_poll_time, NULL);

	run_program(POWERLEVEL_SCRIPT, "wakeup", ac ? "ac" : "battery");

	if(!(flags&FLG_NOOFFB))
		(void) ioctl(pmu_fd, PMU_IOC_SET_BACKLIGHT, &level);

}

/*
 * This is very insecure....
 */

static void run_program(char *prog, char *a1, char *a2)
{
	pid_t pid;
	int status = 0;
	static int maxfiles = 0;

	if(!maxfiles)
	{
		struct rlimit l;

		if(!getrlimit(RLIMIT_NOFILE, &l))
			maxfiles = l.rlim_cur;
	}

	syslog(LOG_INFO, "running %s %s %s", 
			prog, 
			a1 ? a1 : "",
			a1 && a2 ? a2 : "" 
			);

	if(!validate_program(prog))
	{
		syslog(LOG_ALERT, "refuse to run insecure %s", prog);
		return;
	}

	pid = fork();
	if (pid == 0) 
	{
		char *argv[4];
		register int i;

		for(i=0; i < maxfiles; i++)
			close(i);

		/* stdin, stdout and stderr to /dev/null */

		open("/dev/null", O_RDONLY);
		open("/dev/null", O_WRONLY);
		open("/dev/null", O_WRONLY);

		argv[0] = prog;
		argv[1] = a1;
		argv[2] = a2;
		argv[3] = 0;
		execv(prog, argv);
		exit(99);
	}
	waitpid(pid, &status, 0);

	if (WIFEXITED(status)) 
	{
		if (WEXITSTATUS(status) != 0)
			syslog(LOG_ERR, "%s exited with code %d\n", prog,
			       WEXITSTATUS(status));
	} 
	else if (WIFSIGNALED(status)) 
		syslog(LOG_ERR, "%s got signal %d\n", prog, WTERMSIG(status));
}

static void do_signal(int signum)
{
	char msg[32];
	sprintf(msg, "catched signal %d", signum);
	bye(signum == SIGTERM ? EXIT_SUCCESS : EXIT_FAILURE, msg);
}

static int do_sigpower(char level)
{
	int fd = open(POWERSTATUS_FILE, O_RDWR|O_TRUNC|O_CREAT);
	if(fd < 0)
	{
		syslog(LOG_ALERT, "open of %s failed (%m) reverting to sleep",
			POWERSTATUS_FILE
			);
		flags &= ~FLG_SIGPWR;

		return 0;
	}
	write(fd, &level, 1);

	close(fd);

	return !kill(1, SIGPWR);
}

static void bye(int ret, const char *msg)
{
	if(ret)
		beep(BEEP_TIME, BEEP_ERR);

	syslog(ret == EXIT_SUCCESS ? LOG_INFO : LOG_ERR, 
		"daemon stopped (%s)",
		msg && *msg ? msg : "no additional information"
		);
	exit(ret);
}

static int open_apm(char *file)
{
	int fd;
	struct stat sb;

	if(!file || !*file)
	{
		syslog(LOG_ERR, "open_apm: no apmfile specified");
		return -1;
	}

	errno = 0;
	if(!stat(file, &sb)
	&& !S_ISFIFO(sb.st_mode))
	{
		syslog(LOG_ERR, "apmfile '%s' exists, but is not a FIFO", file);
		return -1;
	}

	if(errno == ENOENT && mkfifo(file, 0644))
	{
		syslog(LOG_ERR, "error creating '%s': %m", file);
		return -1;
	}

	fd = open(file, O_WRONLY|O_NONBLOCK);

	return fd;
}

static void apm_data(int fd, struct my_apm_info *apm)
{
	char data[128];

	if(fd < 0 || !apm)
		return;
	
	if(apm->battery_time >= APM_THRESHOLD)
	{
		apm->battery_time /= 60;
		apm->using_minutes = 1;
	}
	snprintf(data, sizeof(data), "%s %s %x %x %x %x %d%% %d %s\n",
		PMUD_SERVICE,
		PMUD_VERSION,
		apm->apm_flags,
		apm->ac_line_status,
		apm->battery_status,
		apm->battery_flags,
		apm->battery_percentage,
		apm->battery_time,
		apm->using_minutes ? "min" : "sec"
	);
	(void) write(fd, data, strlen(data));
}

static int get_trackpad_id(int fd)
{
	int n;
	int id = -1;
	register int i;

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "polling adb for trackpad");

	/* always poll all devices, someone might have attached a mouse
	 * while sleeping and thus maybe address relocation occured
	 */

        for (i=1; i<0x10; i++)
        {
               	unsigned char buf[9];

		if(adbwrite(fd, i, 1, ADBtalk, 0, 0) <= 0)
			continue;

		n = adbread(fd, i, buf, sizeof(buf));
			
		/* must read tpad */
               	if(n >= 5 
		&& buf[1] == 't'
		&& buf[2] == 'p'
		&& buf[3] == 'a'
		&& buf[4] == 'd')
               	{
			id = i;
			break;
               	}
        }
	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "found %s trackpad (id = %d)", 
				id < 0 ? "no" : "a",
				id
			);
	return id;
}

static int get_trackpad_settings(int fd, int id)
{
	int nr;
	int settings = 0;
	unsigned char buf[9];

	if(adb_program(fd, id, 1) < 0)
		return 0;

	if(adbwrite(fd, id, 2, ADBtalk, 0, 0) <= 0)
	{
		syslog(LOG_ERR, "get.get adbwrite: %m");
		return -1;
	}
	nr = adbread(fd, id, buf, sizeof(buf));
	if(nr >= 5)
	{
		settings |= buf[1] == 0x99 ? TPS_TAP : 0;
		settings |= buf[2] == 0x94 ? TPS_DRAG : 0;
		settings |= buf[4] == 0xff ? TPS_LOCK : 0;
	}
	adb_program(fd, id, 0);

	if(flags&FLG_DEBUG)
		syslog(LOG_DEBUG, "trackpad settings %stap %sdrag %slock",
			settings & TPS_TAP ? "" : "no",
			settings & TPS_DRAG ? "" : "no",
			settings & TPS_LOCK ? "" : "no"
		);

	return settings;
}

static int set_trackpad_settings(int fd, int id, int settings)
{
	int nr;
	int tries = 5;
	unsigned char buf[9];

	if(adb_program(fd, id, 1) < 0)
		return 0;

	while(tries--)
	{
		if(adbwrite(fd, id, 2, ADBtalk, 0, 0) <= 0)
		{
			syslog(LOG_ERR, "set.get adbwrite: %m");
			continue;
		}
		nr = adbread(fd, id, buf, sizeof(buf));
		if(nr < 5)
			continue;

		buf[1] = settings & TPS_TAP ? 0x99 : 0x19;
		buf[2] = settings & TPS_DRAG ? 0x94 : 0x14;
		buf[4] = settings & TPS_LOCK ? 0xff : 0xb2;

		/*
		 * Fix iBook2 jittery trackpad after wake from sleep
		 * This was taken from drivers/macintosh/adbhid.c
		 * and is part of how the trackpad is initialized.
		 */

		if(nr>6)
			buf[6] = 0x0d;

		if(adbwrite(fd, id, 2, ADBlisten, buf+1, nr) <= 0)
		{
			syslog(LOG_ERR, "set.set adbwrite: %m");
			continue;
		}
		nr = adbread(fd, id, buf, sizeof(buf));
	}
	adb_program(fd, id, 0);

	if(flags&FLG_DEBUG)
	{
		syslog(LOG_DEBUG, "requested settings %stap %sdrag %slock",
			settings & TPS_TAP ? "" : "no",
			settings & TPS_DRAG ? "" : "no",
			settings & TPS_LOCK ? "" : "no"
		);
		settings = get_trackpad_settings(fd, id);
	}

	return 0;
}

static int adb_program(int fd, int id, int on)
{
	int nr;
	int tries = 5;
	unsigned char buf[16];

	while(tries--)
	{
		if(adbwrite(fd, id, 1, ADBtalk, 0, 0) <= 0)
		{
			syslog(LOG_ERR, "program.get adbwrite: %m");
			continue;
		}

		nr = adbread(fd, id, buf, sizeof(buf));
		if(nr < 8)
			continue;

		buf[7] = on ? 0x0d : 0x03;
	
		errno=0;
		if((nr=adbwrite(fd, id, 1, ADBlisten, buf+1, nr-1)) < 0)
		{
			syslog(LOG_ERR, "program.set adbwrite: %m (%d)", nr);
			continue;
		}
		return adbread(fd, id, buf, sizeof(buf));
	}
	return -1;
}

static int adbwrite(int fd, char id, char reg, char op, char *data, int size)
{
	adbpacket_t packet;

	packet.preamble = ADB_PACKET;

	switch(op)
	{
	case ADBlisten:
		packet.control = ADB_WRITEREG(id, reg);
		break;

	case ADBtalk:
		packet.control = ADB_READREG(id, reg);
		break;

	default:
		syslog(LOG_ERR, "adbwrite: unknown op = (%d)", op);
		return -1;
	}
	if(size < 0 || size > ADB_DATA_LEN)
	{
		syslog(LOG_ERR, "adbwrite: invalid data lenght = (%d)", size);
		return -1;
	}

	if(size)
		memcpy(packet.data, data, size);

	return write(fd, &packet, size + 2);
}

static int adbread(int fd, char id, char *data, int size)
{
	int nr;
	adbpacket_t packet;
	char *raw = ((char*)&packet)+1;

	packet.preamble = (id << 4);

	do
	{
		nr = read(fd, raw, sizeof(packet) - 1);
		if(nr <= 0)
			return nr;

	} while((packet.control & 0xf0) != (id<<4));

	if(nr > 0)
		memcpy(data, raw, size >= nr ? size : nr);

	return nr;
}

static void fbon(int on)
{
	ioctl(fb0_fd, FBIOBLANK, !on);
}

static void beep(unsigned int ms, unsigned int freq)
{
	int arg;
	static int fd = -1;

	if (fd < 0)
	{
		fd = open(CONSOLE_FILE, O_RDWR);
		if (fd < 0)
			return;
	}

	arg = (ms << 16) | freq;
	ioctl(fd, KDMKTONE, arg);
	usleep(ms*1000);
}

