#!/usr/bin/python
# -*- coding: UTF-8 -*-

# Phatch - Photo Batch Processor
# Copyright (C) 2007-2009 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for python editing.

# Does not follow PEP 8 because of dumped data

try:
    _
except NameError:
    import gettext
    gettext.install('test')

import datetime
import types

from reverse_translation import _t

MONTHS = (_t('January'), _t('February'), _t('March'), _t('April'),
    _t('May'), _t('June'), _t('July'), _t('August'), _t('September'),
    _t('October'), _t('November'), _t('December'))
WEEKDAYS = (_t('Monday'), _t('Tuesday'), _t('Wednesday'), _t('Thursday'),
    _t('Friday'), _t('Saturday'), _t('Sunday'))


def is_string(x):
    """Checks whether ``x`` is a string not.

    :param x: variable
    :returns: True if string, False otherwise
    :rtype: bool
    """
    return type(x) in types.StringTypes


class DateTime:
    format = '%Y:%m:%d %H:%M:%S'
    attrs = ('year', 'month', 'monthname', 'day', 'weekday',
        'weekdayname', 'hour', 'minute', 'second', 'microsecond')

    def __init__(self, source):
        """Extends :class:`datetime.datetime` so it has also as attributes
        ``weekday``, ``weekdayname`` and ``monthname``. If the source is
        given as a string it should be of the format ``%Y:%m:%d %H:%M:%S``.

        :param source: source from which to construct the DateTime
        :type source: :class:`datetime.datetime`/string
        """
        if is_string(source):
            self.from_exif_string(source)
        else:
            self.datetime = source

    def __getattr__(self, attr):
        """Gets the attribute. If the attr is not ``weekday``,
        ``weekdayname`` or ``monthname``, delegate to attributes
        of the underlying :class:`datetime.datetime` instance.

        :param attr: attribute
        :type attr: string
        """
        if attr == 'weekday':
            return self.datetime.weekday()
        elif attr == 'weekdayname':
            return WEEKDAYS[self.datetime.weekday()]
        elif attr == 'monthname':
            return MONTHS[self.datetime.month-1]
        else:
            return getattr(self.datetime, attr)

    def from_exif_string(self, exif_string):
        """Creates the DateTime from an exif string (as returned by
        PIL exif).

        :param exif_string: datetime string with format ``%Y:%m:%d %H:%M:%S``
        :type exif_string: string
        """
        d = datetime.datetime(1, 1, 1)
        self.datetime = d.strptime(exif_string, self.format)

    def __str__(self):
        """Represents DateTime as ``%Y:%m:%d %H:%M:%S`` string."""
        return self.datetime.strftime(self.format)

    def __repr__(self):
        """Represents DateTime."""
        return "DateTime('%s')"%self


def now():
    return DateTime(datetime.datetime.now())


class Rational:
    denominator = 1
    numerator = 1

    def __repr__(self):
        return 'Rational()'

    def __str__(self):
        return 'Rational()'

def main(path):
    import glob
    import metadata
    import os
    import pprint
    result = {}
    filenames = glob.glob(os.path.join(path, '*'))
    for filename in filenames:
        print('Scanning file "%s" ...\n'%filename)
        info = metadata.InfoExtract(filename)
        info.extract_all()
        result.update(info.dump(expand=False))
    result.update(INFO_TEST)
    pprint.pprint(result)

INFO_TEST = \
{'Exif_CanonCs_0x0000': 92,
 'Exif_CanonCs_0x0006': 0,
 'Exif_CanonCs_0x0008': 65535,
 'Exif_CanonCs_0x0009': 1,
 'Exif_CanonCs_0x0015': 32767,
 'Exif_CanonCs_0x001e': 0,
 'Exif_CanonCs_0x001f': 0,
 'Exif_CanonCs_0x0026': 0,
 'Exif_CanonCs_0x0027': 0,
 'Exif_CanonCs_0x0029': 0,
 'Exif_CanonCs_0x002b': 32767,
 'Exif_CanonCs_0x002c': 0,
 'Exif_CanonCs_0x002d': 0,
 'Exif_CanonCs_0x002e': 65535,
 'Exif_CanonCs_0x002f': 100,
 'Exif_CanonCs_AESetting': 0,
 'Exif_CanonCs_AFPoint': 16390,
 'Exif_CanonCs_ColorTone': 32767,
 'Exif_CanonCs_Contrast': 0,
 'Exif_CanonCs_DigitalZoom': 0,
 'Exif_CanonCs_DisplayAperture': 0,
 'Exif_CanonCs_DriveMode': 0,
 'Exif_CanonCs_EasyMode': 0,
 'Exif_CanonCs_ExposureProgram': 0,
 'Exif_CanonCs_FlashActivity': 0,
 'Exif_CanonCs_FlashDetails': 0,
 'Exif_CanonCs_FlashMode': 5,
 'Exif_CanonCs_FocusContinuous': 0,
 'Exif_CanonCs_FocusMode': 4,
 'Exif_CanonCs_FocusType': 1,
 'Exif_CanonCs_ISOSpeed': 15,
 'Exif_CanonCs_ImageSize': 0,
 'Exif_CanonCs_ImageStabilization': 1,
 'Exif_CanonCs_Lens': (44400, 7400, 1000),
 'Exif_CanonCs_LensType': 65535,
 'Exif_CanonCs_Macro': 2,
 'Exif_CanonCs_MaxAperture': 95,
 'Exif_CanonCs_MeteringMode': 3,
 'Exif_CanonCs_MinAperture': 221,
 'Exif_CanonCs_PhotoEffect': 65535,
 'Exif_CanonCs_Quality': 5,
 'Exif_CanonCs_Saturation': 0,
 'Exif_CanonCs_Selftimer': 0,
 'Exif_CanonCs_Sharpness': 0,
 'Exif_CanonCs_ZoomSourceWidth': 3648,
 'Exif_CanonCs_ZoomTargetWidth': 3648,
 'Exif_CanonSi_0x0000': 68,
 'Exif_CanonSi_0x0001': 65523,
 'Exif_CanonSi_0x0003': 224,
 'Exif_CanonSi_0x0006': 0,
 'Exif_CanonSi_0x0008': 0,
 'Exif_CanonSi_0x000a': 0,
 'Exif_CanonSi_0x000b': 0,
 'Exif_CanonSi_0x000c': 0,
 'Exif_CanonSi_0x000d': 0,
 'Exif_CanonSi_0x0010': 0,
 'Exif_CanonSi_0x0011': 0,
 'Exif_CanonSi_0x0012': 1,
 'Exif_CanonSi_0x0014': 0,
 'Exif_CanonSi_0x0017': 0,
 'Exif_CanonSi_0x0018': 0,
 'Exif_CanonSi_0x0019': 1,
 'Exif_CanonSi_0x001a': 250,
 'Exif_CanonSi_0x001b': 3,
 'Exif_CanonSi_0x001c': 0,
 'Exif_CanonSi_0x001d': 0,
 'Exif_CanonSi_0x001e': 0,
 'Exif_CanonSi_0x001f': 0,
 'Exif_CanonSi_0x0020': 0,
 'Exif_CanonSi_0x0021': 0,
 'Exif_CanonSi_AFPointUsed': 0,
 'Exif_CanonSi_ApertureValue': 128,
 'Exif_CanonSi_FlashBias': 0,
 'Exif_CanonSi_ISOSpeed': 160,
 'Exif_CanonSi_Sequence': 0,
 'Exif_CanonSi_ShutterSpeedValue': 219,
 'Exif_CanonSi_SubjectDistance': 6553,
 'Exif_CanonSi_TargetAperture': 128,
 'Exif_CanonSi_TargetShutterSpeed': 223,
 'Exif_CanonSi_WhiteBalance': 0,
 'Exif_Canon_0x0000': (0, 0, 0, 0, 0, 16388),
 'Exif_Canon_0x0003': (0, 0, 0, 0),
 'Exif_Canon_0x000d': (370L,
                       411L,
                       0L,
                       0L,
                       0L,
                       384L,
                       657L,
                       4294967255L,
                       0L,
                       0L,
                       0L,
                       0L,
                       576L,
                       675L,
                       4294967204L,
                       0L,
                       0L,
                       4294967283L,
                       3L,
                       0L,
                       12L,
                       3L,
                       0L,
                       270L,
                       21L,
                       10L,
                       674L,
                       671L,
                       673L,
                       384L,
                       854L,
                       4294967204L,
                       0L,
                       0L,
                       671L,
                       673L,
                       0L,
                       0L,
                       5L,
                       3072L,
                       3072L,
                       3072L,
                       3072L,
                       3072L,
                       4294964224L,
                       4294964224L,
                       4294964224L,
                       4294964224L,
                       4294964224L,
                       4294967263L,
                       4294964224L,
                       4294967283L,
                       6L,
                       0L,
                       0L,
                       0L,
                       0L,
                       0L,
                       0L,
                       0L,
                       0L,
                       0L,
                       102L,
                       1024L,
                       1024L,
                       4294966843L,
                       198L,
                       0L,
                       0L,
                       0L,
                       0L,
                       0L,
                       0L,
                       407L,
                       0L,
                       4294966844L,
                       196L,
                       0L,
                       0L,
                       342144L,
                       342140L,
                       0L,
                       0L,
                       1054L,
                       1034L,
                       1043L,
                       1303L,
                       0L,
                       4294966844L,
                       195L,
                       4294967287L,
                       898L,
                       2066L,
                       1346L,
                       898L,
                       1L,
                       976L,
                       288L,
                       674L,
                       659L,
                       4294967204L,
                       3L,
                       0L,
                       511L,
                       0L,
                       0L,
                       0L,
                       0L,
                       401L,
                       5L,
                       0L,
                       0L,
                       0L,
                       0L,
                       1L,
                       0L,
                       456L,
                       0L,
                       0L,
                       0L,
                       511L,
                       0L,
                       4294946160L,
                       4L,
                       9L,
                       372L,
                       362L,
                       360L,
                       370L,
                       363L,
                       360L,
                       369L,
                       363L,
                       360L,
                       65535L,
                       1216L,
                       228L,
                       279L,
                       52L,
                       219L,
                       41L,
                       0L,
                       0L,
                       3L,
                       3L,
                       26L,
                       9L,
                       643159549L),
 'Exif_Canon_0x0013': (0, 0, 0, 0),
 'Exif_Canon_0x0018': '0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ',
 'Exif_Canon_0x0019': 1,
 'Exif_Canon_0x001c': 0,
 'Exif_Canon_0x001d': (32, 1, 0, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0),
 'Exif_Canon_0x001e': 16779008L,
 'Exif_Canon_0x001f': (138,
                       1,
                       0,
                       4,
                       8,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0),
 'Exif_Canon_0x0022': (416,
                       0,
                       0,
                       16,
                       8,
                       1,
                       1,
                       640,
                       480,
                       0,
                       0,
                       0,
                       0,
                       0,
                       8,
                       384,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0),
 'Exif_Canon_0x0023': (8L, 0L),
 'Exif_Canon_0x0024': (156,
                       35,
                       0,
                       3648,
                       2736,
                       1,
                       1,
                       16,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0,
                       0),
 'Exif_Canon_0x0025': '14 35 0 0 0 0 0 0 0 0 0 0 0 0 ',
 'Exif_Canon_0x0026': (96,
                       4,
                       9,
                       9,
                       3648,
                       2736,
                       1216,
                       228,
                       219,
                       219,
                       219,
                       219,
                       219,
                       219,
                       219,
                       219,
                       219,
                       41,
                       41,
                       41,
                       41,
                       41,
                       41,
                       41,
                       41,
                       41,
                       65317,
                       0,
                       219,
                       65317,
                       0,
                       219,
                       65317,
                       0,
                       219,
                       65495,
                       65495,
                       65495,
                       0,
                       0,
                       0,
                       41,
                       41,
                       41,
                       144,
                       0,
                       0,
                       4),
 'Exif_Canon_0x0027': (4, 0),
 'Exif_Canon_0x0028': '106 88 154 161 142 223 21 20 202 181 48 165 254 199 68 9 ',
 'Exif_Canon_0x002d': 0L,
 'Exif_Canon_0x00d0': 0L,
 'Exif_Canon_FirmwareVersion': 'Firmware Version 1.00',
 'Exif_Canon_FocalLength': (2, 7400, 294, 221),
 'Exif_Canon_ImageNumber': 1060010L,
 'Exif_Canon_ImageType': 'IMG:PowerShot G7 JPEG',
 'Exif_Canon_ModelID': 26738688L,
 'Exif_Canon_OwnerName': '',
 'Exif_GPSInfo_GPSAltitude': Rational(),
 'Exif_GPSInfo_GPSAltitudeRef': '0 ',
 'Exif_GPSInfo_GPSDateStamp': DateTime('2009:06:21 23:06:40'),
 'Exif_GPSInfo_GPSLatitude': (Rational(), Rational(), Rational()),
 'Exif_GPSInfo_GPSLatitudeRef': 'N',
 'Exif_GPSInfo_GPSLongitude': (Rational(), Rational(), Rational()),
 'Exif_GPSInfo_GPSLongitudeRef': 'E',
 'Exif_GPSInfo_GPSMapDatum': 'WGS-84',
 'Exif_GPSInfo_GPSProcessingMethod': 'ASCII\x00\x00\x00HYBRID-FIX',
 'Exif_GPSInfo_GPSTimeStamp': (Rational(),
                               Rational(),
                               Rational()),
 'Exif_GPSInfo_GPSVersionID': '2 2 0 0 ',
 'Exif_Image_0xa401': 0,
 'Exif_Image_0xa402': 0,
 'Exif_Image_0xa403': 0,
 'Exif_Image_0xa404': Rational(),
 'Exif_Image_0xa405': 38,
 'Exif_Image_0xa406': 0,
 'Exif_Image_0xa407': 0,
 'Exif_Image_0xa408': 0,
 'Exif_Image_0xa409': 0,
 'Exif_Image_0xa40a': 0,
 'Exif_Image_0xa40c': 0,
 'Exif_Image_0xea1c': '\x1c\xea\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Image_Artist': 'e2interactive',
 'Exif_Image_Copyright': 'Copyright 2006, e2interactive',
 'Exif_Image_DateTime': DateTime('2007:04:15 04:00:38'),
 'Exif_Image_ExifTag': 178L,
 'Exif_Image_GPSTag': 6112L,
 'Exif_Image_ImageDescription': '                               ',
 'Exif_Image_Make': 'Panasonic',
 'Exif_Image_Model': 'NV-GS400 ',
 'Exif_Image_Orientation': 1,
 'Exif_Image_PrintImageMatching': 'PrintIM\x000300\x00\x00\x02\x00\x02\x00\x01\x00\x00\x00\x01\x01\x01\x00\x00\x00',
 'Exif_Image_ResolutionUnit': 2,
 'Exif_Image_Software': 'Ver.1.0  ',
 'Exif_Image_XResolution': Rational(),
 'Exif_Image_YCbCrPositioning': 2,
 'Exif_Image_YResolution': Rational(),
 'Exif_Iop_InteroperabilityIndex': 'R98',
 'Exif_Iop_InteroperabilityVersion': '0100',
 'Exif_Iop_RelatedImageLength': 2736,
 'Exif_Iop_RelatedImageWidth': 3648,
 'Exif_MakerNote_ByteOrder': 'II',
 'Exif_MakerNote_Offset': 712L,
 'Exif_Nikon3_0x000a': Rational(),
 'Exif_Nikon3_0x0017': '\x00\x01\x06\x00',
 'Exif_Nikon3_0x009b': '0 0 ',
 'Exif_Nikon3_0x009d': 0,
 'Exif_Nikon3_0x00a3': '0 ',
 'Exif_Nikon3_0x00ae': '\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Nikon3_0x00b6': '\x07\xd9\x06\x1c\x10\x1a3\x00',
 'Exif_Nikon3_0x00bb': '0100\xff\x00',
 'Exif_Nikon3_AFFocusPos': '\x01\x00\x00\x01',
 'Exif_Nikon3_AFResponse': 'STANDARD ',
 'Exif_Nikon3_ActiveDLighting': 0,
 'Exif_Nikon3_AutoBracketRelease': 2,
 'Exif_Nikon3_AuxiliaryLens': 'OFF         ',
 'Exif_Nikon3_ColorBalance1': (Rational(), Rational(), Rational(), Rational()),
 'Exif_Nikon3_ColorBalance2': '0210\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xa0W%\n\x06\x19Br\xddK\xd0n#\xac\xd0\xd7\xd8\xdb=\xd2\xfe\x80\x1b\x8c\x95vi\x9c\x99\x97"\xf6\x10\x83QR\xf4v\xf7CH\x88\x01X\x13P\x7f\x15H\xd9m\xbf\xecf\xc9\xd8\x04x\x99\t\x88\xcd\xd5\x07\xc1\xb6\xc1k\xa4\xf8\xe7\xe5\xbeo\xceq\x04\x8b)\xee\xb9\x80\x9d0\xdf!z\xecu\x13\xc8\x96{u\x87C\xf0\x99\xb5\xea\xd7\x89S\xc9,\xc8au\xf3]\xdex)\xef\xcc\xc2\xce\xf1+|\xe4c\xf9\xa6jE7Ob\xdb\xe7L\xc6Y\x01\xeb\x9e\x8c\x93\xa7\xe2-\x8f\x08\x98?\xfd\xd2\xbe\xc1\xdb\x0cT\xb3)\xb6Z\x15\xe7\xdc\xd2\xce\x17\x1d\xb6)\xb1U\x0c\xd8\xc3\x93\xd2\xfd?\x98\x08\x8f-\xe2\xaf\x91\x8a\x9c\xc5\x03X\xc6K\xe5\x97\x8a@\xd2E\x98\xa7\xf9c\x15|\xd3\xf1"\xc3\xcd\xee(y\xdf\\\xf2\x9f\x97:,5S\x88\x97:\x07V\xf0\xb3\x84v\x85\x96\xdf\x10)\xedM\x08!\xe7\x9b\x97\xb1\xac/\x85\x0b\x92>\xef\xb3\xa0\xa2\xb9\xe0#\xd6\xf7\x83 \xd1\xa5\x89\x86f\xc2\xe6]\xcbP\xee\xa3\xc0OX]fB\xed\x81\xfed\xb3\xeb\x0c\x16\t\xe5\xaaX\xef',
 'Exif_Nikon3_ColorMode': 'COLOR',
 'Exif_Nikon3_ColorSpace': 1,
 'Exif_Nikon3_CropHiSpeed': (0, 3904, 2616, 3904, 2616, 0, 0),
 'Exif_Nikon3_DataDump': '\x05\x02\x00\x00\x00\x00\x00\x00\x00\x00\xff\x01\x00\x00\x19a\x121\x00\x00@\xb7\x00\x00\x1a\x8d\x00\x00QR\x00\x01EH\x00\x01EH\x00\x00Q\x83\x00\x00@\xe8\x00d\x00\x9a\x00@\x00\x1c\x00\x00\x1e\x01\x00\x00\x0c\xa3\r*!u\x00\x00\x00\x00\r\r\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00(\n\x00\x00<\x00\x10K\x00\xd2\r*\x00\x00\n\x8f\x00<\x02\xfa\x00}\'\x10$\x00\x1f\x05""""\x00\x19\x00\n\x00\x0b\x00\x0e\n\xaa\x04\x00\x00p\x00\x00\x11\x11\x11\x11\x01\x84\x03\xa1\x03\xeb\x01.\x02\x00\x01\xd2\x01\xc8\x01\xfe\x01\x03\x90]\x00\x00\x03\xff\x00\x14\x00\r\x00\x14\x00\r\x00\n\x00\x04\x00\x00\x00\x00\x00\x00\x00\x10\x00\x00\x00\x1c\x00\x01\x01\x00\x00\x9a\x01B\x01\xea\x02\x92\x03:\x0f\x0f\x02\x1b\x01\xfe\x01\xb1\x00\x00\t\xd5\n\xaa\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x91\x00\x04\x00\x00\x00\x14\x00\x00\x00\x00\x00\x00p\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00P\x04>\x03\xe8\x04>\x00\x12\x01\xea\x01\x0f\x01\xda\x01\xe7\x01\xf2\x01\xef\x01\xe9\x04\x00\x11\x00\x00\x81\x01g\x01\xbf\x01\xea\x01\x9e\x01:\x01\x0f\x00\x00\x00\x00\x03\xdc\x05*\x05T\x03n\x01\xb6\x01!\x00\x00\x00\x00\x01\xda\x01\xe7\x01\xf2\x01\xef\x01\xe9\x00\x00\x00\x00\x00\x00\x05\xc8\x05\xe4\x05\xe7\x05\xb6\x05]\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcf\x01\xa2\x00\xd4\x01\x80\x00\xe8\x01mwwww\x15.\x06\x1b\x15\x1f\x10\x1c\x0c\x14\x08\x00L\x00h\n\x00\x0e+\x01\x00\x00\x08\x00U\x02\x89\x90\x1d+\x00d\x1e!d<$$&#\x01\xf9\x01e\x01o\x02\xa4d\x00\x00A\x00\x0c\x00\x0f\x01\xd2\x01\xc8\x1e!\x10\x1d\x061\n#\x00T\x01\x08\x195\x00\x00',
 'Exif_Nikon3_DigitalZoom': Rational(),
 'Exif_Nikon3_ExposureDiff': '\xfb\x01\x0c\x00',
 'Exif_Nikon3_FileInfo': '0100\x00\x00\x00d\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Nikon3_FlashComp': '\x00\x01\x06\x00',
 'Exif_Nikon3_FlashDevice': '                   ',
 'Exif_Nikon3_FlashInfo': '0102\x00\x00\x00\x00\x00\x00\x00\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Nikon3_FlashMode': '0 ',
 'Exif_Nikon3_FlashSetting': 'NORMAL      ',
 'Exif_Nikon3_Focus': 'AF-A  ',
 'Exif_Nikon3_FocusDistance': Rational(),
 'Exif_Nikon3_HighISONoiseReduction': 0,
 'Exif_Nikon3_HueAdjustment': '0',
 'Exif_Nikon3_ISOInfo': 'T\x01\x0c\x00\x00\x00T\x01\x0c\x00\x00\x00\x00\x00',
 'Exif_Nikon3_ISOSelection': 'AUTO  ',
 'Exif_Nikon3_ISOSettings': (0, 400),
 'Exif_Nikon3_ISOSpeed': (0, 400),
 'Exif_Nikon3_ImageAdjustment': 'AUTO         ',
 'Exif_Nikon3_ImageBoundary': (0, 0, 3872, 2592),
 'Exif_Nikon3_ImageDataSize': 5008376L,
 'Exif_Nikon3_ImageOptimization': 'VIVID          ',
 'Exif_Nikon3_ImageProcessing': '                                        ',
 'Exif_Nikon3_ImageStabilization': 'VR-OFF     ',
 'Exif_Nikon3_Lens': (Rational(), Rational(), Rational(), Rational()),
 'Exif_Nikon3_LensData': '0202\xb5{&\x08\x00|n\x1e\x9cf\x82B\x1eq\xe3\xc5\x81\xddd\x97\xfe\x81\x1a\xce\x90shv',
 'Exif_Nikon3_LensFStops': '@\x01\x0c\x00',
 'Exif_Nikon3_LensType': '14 ',
 'Exif_Nikon3_LightSource': 'NATURAL    ',
 'Exif_Nikon3_MultiExposure': '0100\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Nikon3_NoiseReduction': 'OFF ',
 'Exif_Nikon3_Preview': 3006L,
 'Exif_Nikon3_ProgramShift': '\x00\x01\x06\x00',
 'Exif_Nikon3_Quality': 'FINE   ',
 'Exif_Nikon3_RetouchHistory': (0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
 'Exif_Nikon3_Saturation': 'ENHANCED       ',
 'Exif_Nikon3_SceneAssist': '                   ',
 'Exif_Nikon3_SceneMode': '               ',
 'Exif_Nikon3_SensorPixelSize': (Rational(), Rational()),
 'Exif_Nikon3_SerialNumber': '7460067',
 'Exif_Nikon3_Sharpening': 'MED.H ',
 'Exif_Nikon3_ShootingMode': 1,
 'Exif_Nikon3_ShotInfo': '0211\x8a3E\xf6\x06\x19\xadj2\xa5?\x80"\x8c\x96\xc6\xd8\xff=\x92\xa7\x8f\x7f\xed\x94siv\x9a\xd5\'\x90\x10\xa7U\x1a\xf6\xe9\xf3\x14L\x9b\x01~\x12\xbd\x7fXHOm\xe9\xeeQ\xcbu\x04\xc3\xc4\xb0\x8a\\\xd2;\x805\x85\xea\xe6;\xa3(\xbc\xc51\xb6\x02\x88/\xe8\xb8\x9f\x9d\xb2\xde!{\xedt\x15\xc9\x82zu\xda\x86\xf5B\xae\r\xe4\xd3^%!\xd5\x8fp\xf2]\xdf\x85\xb5\xef#,\xce\xfc~\xbc\xe4q\xfa\xd3jITh`\x87\xe6r\xc6~\x83\xe8\x9c\x96_[\xe2\x0cPl\x98b\xf7\x13\xbe\xff\xb6\x9aT\xf0+ZZ)Z.\xd0\xe8\xabh\xb6500\x0c\x98\xee\xec\xd2\xc5\x10O\x08\x81\x11\xa0\xae\x94E\x16\xc4\x08X\xd1J\xc1\x9f\xde@\x13%=\xa6\xff\x89C|&[\x88\xc2\xc4W$x\xd11\x80\x9ex>\xdf\x07b\xb0\x9e\x02\x86s\xc2\x85\xb2CO\xa1\xf0,@\xdeO\x14\xee\x8d\xd9\xdb\xfd\xac\x19\xb91\xbe\x1e\xa9\x86\xe4\x9d\xfd\xd0\x16\xbd\xc1\xb7\x15\xe0\x90\xb0\xbc\xdd\xfa7j\xf8t\xca\x89Q`Vp_\x991X-\xb4|$\xd5\xcf\xd16q\xe5A\xdfi\x9a\xb4\\CT\xb3\x89*\xb4\xcd\xab\x15\xcb\xf4\xee\xea\xc5\nA\xfba\x93\xb4l3#\xe4\xcb\xb9N\x00w\x07Sf9 \n<k\xb3\xc3\x82\r\xad\xd58\x1f\x1f\x9dO{\xc1T\xe5Is.\xeb/=\x08a\x1d\x0f\x8dG\xd3\xe9\x83\xf1\xe6\xbcCD\xebi\xce\xaa\xc7O\xeaL\xd9\xa9T\xfa}\xe7\xb8\x8a\xa5\xd7\xa0\xe0[e\x86F\x95\x83\x08\x19\x0e\x14+g\xa8\n\x8d\x03\xa4fI>A[\x8c\xc5\xe9\x93\x0e\xcbO]hA\xbd\xaf\xe2\'s\t\xec\x9d\x81{\x07R\x9b\xbe3\x89=\xacX/X\nVE\xaf\xd8w\xcb]\x16\xa0\xc1\xf4\xc4\xd4\'8\xe2!\xfd\xe6p\x1eB\xa8e\xce\xf8_\xddr2\x14GR\x9d\xd3\x08V\xba58`0\x07\xf5\xfa\x16I\x93\xf4l\xfb\xa1^2\x1d\x1f8h\xaf\r\x82\x0e\xb1k<$#9f\xaa\x05w\x00\xa0W%\n\x06\x19C\x84\xdcK\xd1n"\xed\xcf\xc8\xd8\xff=\x92\xfe\x81\x1b\xfc\xb6;\x8bT\xac\xc3\'\xc4\xebYU\xc8\xe6\xe9\xf3\x14(\x9b\x01*0\x9f]\x8aX\xcfm\xb34k\xf2]\x13\xc3\x99\x86\x8a\xa51!\x80\xf7\xa7*\xe6\x9b\xa7\xa4\xbc\xeb1\x81\xd7\x85/\xe8\xb8\x9f\x9d\xb2\xbb\x0cx\xect\x13\xc9\xd4Zu@\xbbr\x87\x8f:0v\xac\xcb\x16\x10a\x9e\xce\xbb\xdfx(\x102=\xce\xf1+|\xcfI\xf9\xa6*\x7f7\x88\x9f\xe8\x1a5\xc6Y\x0b;ct\x91\xae\xe2-\x8f\x08\xef\x0e\x02\xadA\xbe$soe\x10fZ\x15\xe7\xd0\xc7\xe7\x1c\xdav(\xb3\xab\xf3\xe1\xec\xbf\xd2\xf9.\x8c\t\xbf\x0f\xaaL\xb3\xbd\x8a\xc4W\xa28J7\x87`@7!j\xa5\xf87\xa1\\+\xf1\xce\xc2\xb1uhU\x8cq\xce\x02a;-6W\x89\xd79\xb5u\xf0\xb0\x87az\x96\xc9\x13\x8c\xfe{ \xde\xb6\x9d\x9e\xb8\xe8/\x8d\x02\x8e1\xeb\xbc\xb4\xa3\xb9\xe2.\x85\xf7\x80 \xd7\xa5\x82\x863i\xa3\xfdi\xf4H\x07\xc4\xe6\xee\xfd\xdb\x15\xba\xd6\xa83\xe8\xb2WA^\xb2\xfc\x00\xb44\x85q?\xd0\xc0\xd46i\xbd%[7\x94YzliH\x87\xcay\xed}%\xe7\xba\xa5\xab\x8d\xf1\x02\xa2\xd2\x9a\n\xc6\x91\x86\x86\x98\xcd\x10\xad!\xa8\x0c\xc9\x9e\xbc\x80\x96\xf8\x02X\xccQ1\xd0\xb6^V\xa3\x92\xd6f\xac%\xf4\x89X4*<d\x99\xf3Y\xdf\x86)\xe6\xca\xc7\xce\xf7.{\xe6i\xf3\xa7kC0Be\x9d\xef\xb0\xc6_\x01\xcf\x9e\x83\x9dP\x1f/v\x1c\x986\xf6\xd3@\xcb\xdf\x06R\xb6!\xb5\\\x1c\xeb\xdf\xd8\xe2\x10\\\xbe \xbcR\x04\xd3\xc8\xb5\xd4\xf74\x9d\x00\x87 \xf2\xa2\x97\x85\x9d\xc2\x0bX\xd1N\xec\x86dG3\xba~\xafh9^\xc5\xabqNBMo\xa8\xf8_\xddg2\xe2\xb9\xaa\xd1\xfe%[\xe3\x17\x9eu0\x07\xf5\xfa\xe9I\x93\xf4l\xe9\x95\x08J\x1d\x1f7\xe8\xaf\r\x8d\x8c\xb1k,\x01#9v\x8f\x05w\x10\x19W$\xf5\\\x19C\xa5UK\xd1N\xef\xed\xcf\xc8\xd8\xff=\x92\xfe\x81\x1b\xcc\x94siv\x9a\xd5l1\x87\xa7\x19t\xb3\xe9\xf3\x14L\x9b\x01~\x12\xbd\x7fXHOm\xa2\xeeQ\xcb\\\r\xc1\x9av\x83\xaf\xd9%\x86\xf0\x83-\xe9\xbd\\Z\xbd\xeb1\x81\x1d\x8d/\xe7\x9c\x9f\x9d\xbd\xfa!{\xe3S\x13\xc9\x99,u\x87\xbf\xacG\xb55V\x89S;\xac;a\x91s]\xdfr\xaf\xef\xcd\xc8I\xf1+w\xd5c\xf9\xad\xaaE7K\xa9\x97\xe5E\xd7Y\x03\xcb\x82',
 'Exif_Nikon3_ShutterCount': 4053L,
 'Exif_Nikon3_ToneComp': 'NORMAL  ',
 'Exif_Nikon3_VRInfo': '0100\x01\x00\x00\x00',
 'Exif_Nikon3_VariProgram': '               ',
 'Exif_Nikon3_Version': '0210',
 'Exif_Nikon3_WhiteBalance': 'CLOUDY      ',
 'Exif_Nikon3_WhiteBalanceBias': ('0', '0'),
 'Exif_Nikon3_WorldTime': '\x00<\x01\x00',
 'Exif_NikonPreview_Compression': 6,
 'Exif_NikonPreview_JPEGInterchangeFormat': 3114L,
 'Exif_NikonPreview_JPEGInterchangeFormatLength': 24364L,
 'Exif_NikonPreview_ResolutionUnit': 2,
 'Exif_NikonPreview_XResolution': Rational(),
 'Exif_NikonPreview_YCbCrPositioning': 2,
 'Exif_NikonPreview_YResolution': Rational(),
 'Exif_Panasonic_0x0022': 0,
 'Exif_Panasonic_0x0027': 0,
 'Exif_Panasonic_0x002f': 1,
 'Exif_Panasonic_0x0031': 3,
 'Exif_Panasonic_0x0037': 257,
 'Exif_Panasonic_0x0038': 3,
 'Exif_Panasonic_0x8002': 1,
 'Exif_Panasonic_0x8003': 2,
 'Exif_Panasonic_0x8007': 1,
 'Exif_Panasonic_AFMode': '16 16 ',
 'Exif_Panasonic_Audio': 2,
 'Exif_Panasonic_BabyAge': '9999:99:99 00:00:00',
 'Exif_Panasonic_BurstMode': 0,
 'Exif_Panasonic_ColorEffect': 1,
 'Exif_Panasonic_ColorMode': 0,
 'Exif_Panasonic_Contrast': 0,
 'Exif_Panasonic_ConversionLens': 1,
 'Exif_Panasonic_DataDump': 'DV\x01\x02EP\x00\x00\xf0\xffDB\x18\x04\xf0\xffAF\xaa\x00\xa0\xaf\x10\x84\xa2\xaf\x00\x86\xa4\xaf @\xa6\xaf\x00\x00\xca\xaf@\x00\xb2\xaf\x03\x00\xb4\xaf\x02\x00\xba\xaf\x00\x00\xbc\xaf\x00\x00\xbe\xaf\xe5\xff\xf8\xaf\x00\x00\xfa\xaf\x00\x00\xa8\xaff\x10\xaa\xaf\x14\x0b\xc8\xaf;\x01\xd8\xaf3\x01\xb6\xaf0\x00\xb8\xaf\x00\x00\xce\xafh\t\xd2\xaf6\x00\xd0\xafz\x00\xac\xaf\xc3\x0c\xb0\xafy\x00\xae\xaf\x97\x01\xd4\xaf\xe8\x03\xf4\xafL\x00\xf2\xaf\xf4\x00\xe2\xaf\x01\x00\xe4\xaf\x01\x00\xe0\xaf/\x0b\xe6\xaf\x02\x00\xe8\xaf\x02\x00\xea\xaf\x00\x00\xec\xaf\x00\x00\xee\xaf\xcd\x00\xf0\xaf\x00\x00\xf6\xaf\x00\x00\xda\xaf\x00\x00\xd6\xaf\x00\x00\x04\x06\xcd\x00\n\x06\x14\x0b\xf0\xffSTJ\x00\xa4\x06\x00\x00\xa6\x06\x00\x00\xa8\x06\x00\x00\xaa\x06\x00\x00\xac\x06\x00\x00\xfa\x07\x00\x00\xae\x06\x00\x00\xb0\x06\x00\x00\xb6\x06\x00\x00\xb8\x06\x00\x00\xba\x06\x00\x00\xf4\x07\x00\x00\xf6\x07\x00\x00\xb2\x06m\x02\xb4\x06\x03\x00\xb0\x04\x00\x00\xb2\x04\x00\x00\xf0\xffAE\xde\x00<\x05\x07\x00\x14\x05\x17\x01\xea\x06#\x01(\x05\xc2\x01,\x05\xc2\x01F\x05\xc2\x01$\x05\x00\x02\x10\x05\x88\x00\x12\x05\xea\x00 \x05\xfc\x006\x05\xaa\x01"\x050\x02:\x05\x01\x00;\x05\x01\x00\xc8\x06\x01\x01&\x05\x0b\x00\xc0\x06\x00\x00*\x05n\x00\xc8\x19\x80\x01\xc6\x19\x87\x01R\x05\xab\x01P\x05\x87\x01.\x05\x00\x000\x05\x00\x002\x05\x00\x00@\x05\x00\x00A\x05\x00\x00B\x05\x00\x00\xce\x19\x00\x00\xd0\x19\x00\x00\xd2\x19\x00\x00C\x05\x00\x00>\x05\x04\x00\x1c\x05\xbd\x03D\x05\xf2\x02"\n+\x01\xc0\x19\x00\x00\xc1\x19\x01\x00\xe8\x06K\x14\xcc\x06\x00\x00\xd4\x06\x00\x01\xce\x06\x00\x00\xd0\x06\x00\x00\xd6\x06\x00\x00P\x18\x00\x00\xda\x06\x00\x00\xc5\x19\x02\x00\xef\x06\x01\x00\xd4\x19\x00\x00\xd6\x19\n\x18\xd8\x19\xa1\x1b\xda\x19\x9a\x08\xee\x06\x00\x00\x00\x05\x04\x00\xf0\xffWB\xf2\x00\x00\x04\xa9\t\x02\x046\x06\\\x04\x1e\x04\x04\x04U\x01\x06\x04\x15\x01`\x04\x88\x00\x1a\x04\xff\xff^\x04\x00\x00_\x04\x00\x00\x12\x04\xff\xff\x14\x04\xff\xff\x16\x04\xff\xff\x18\x04\xff\xff\xcc\x04\x9b\x05\xce\x04L\t\xd0\x04\xcc\x07\xd2\x04%\x06\xdc\x049\t\xde\x04\x02\x06\xb4\x04\x00\x00\xb6\x04\x10\x000\x04`\x008\x04~\x002\x04`\x00:\x04\x98\x004\x04\xc8\xff<\x04\x18\x006\x04\x0c\x00>\x04\x80\x00L\x04\xff\xffN\x04\xff\xff\xc0\x04\xff\xff\xc2\x04\xff\xff\xea\x04\x00\x00\x80\x05\x00\x00\x82\x05\x9d\x00\x83\x05\x83\x00\x88\x05\x00\x00\x8a\x05\x00\x00\x8b\x05\x00\x00\x8c\x05\x00\x00\x8e\x05\x9d\x00\x8f\x05\x83\x00R\x04j\x00T\x04\xf6\x00V\x04\xf6\x00X\x04\xf6\x00\xd4\x04\x8f\x00\xd6\x04v\x00\xd8\x04\x8f\x00\xda\x04v\x00\xf0\x04\x00\x00\xf2\x04\x00\x00\xf4\x04\x00\x00\xf6\x04\x00\x00\xf8\x04\x00\x00\xfa\x04\x00\x00\xfc\x04\x00\x00\xfe\x04\x00\x00\xf0\xffYC\x96\x00\xa0\x05\x05\x03\xa2\x05<P\xa4\x05\x00\x05\xa6\x05\x87\x9b\xa8\x05\xf1\xf1\xaa\x05\x00\x00\xac\x05\x00\x00\xae\x05\xf1\xf1\xb0\x05\x00\x00\xb2\x05\x00\x00\xb4\x05\x00\x00\xb6\x05\x05\x00\xb8\x05<P\xba\x05\x00\x05\xbc\x05\x87\x9b\xbe\x05\xf3\xf3\xc0\x05\x0e\xf2\xc2\x05\x0e\xf2\xc4\x05\xf3\xf3\xc6\x05\x00\x00\xc8\x05\xe5 \xca\x05 \xdf\xcc\x05\x00\x00X\xaa\x01\x00Z\xaa\x01\x00\\\xaa\x01\x00^\xaa\x08\x00`\xaa\x80\x80b\xaa\x80\x80d\xaa\xa0\xa0f\xaa\xa0\xa0h\xaa``j\xaa\x04\x04l\xaa\x00\x00n\xaa\x03\x00\xce\x05\x00\x00\xf0\xffCM\n\x00\xfc\x05\x000\xf0\xffIS\xae\x00\x80\xae\x16\x04\x82\xae\xea\x03\x84\xae\xc7\x03\x86\xae\xe4\x03@\xae\xee\x01`\xae\xec\x01\x88\xae\x00\x00\x00\xae\xf8\x03\x02\xae\x08\x04\x04\xae$\x04\x06\xae8\x04\x08\xaeL\x04\n\xaeX\x04\x0c\xaej\x04\x0e\xaer\x04\x10\xaep\x04\x12\xaen\x04\x14\xaeb\x04\x16\xae`\x04\x18\xaeT\x04\x1a\xaeL\x04\x1c\xae:\x04\x1e\xae,\x04 \xae\xe4\x03"\xae\xca\x03$\xae\xb6\x03&\xae\xa8\x03(\xae\x92\x03*\xae\x82\x03,\xae\x80\x03.\xaex\x030\xae|\x032\xae\x84\x034\xae\x8e\x036\xae\x9a\x038\xae\xaa\x03:\xae\xbc\x03<\xae\xce\x03>\xae\xde\x03\x8c\xae\xe7\x01\x8e\xae\xfe\x01\x90\xae\x0c\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00AEBM\xf1\x00\xd0\x00\xcb\x00\xda\x00\xb6\x00\xc0\x00\xb5\x00\xee\x00\x97\x01\xc5\x01\xa0\x01<\x010\x01\xe9\x00\xc9\x00\xd5\x00\xb0\x00\xcd\x00\xe5\x00\x1e\x01\xf7\x01\xf8\x01\xea\x01\x91\x01\x8f\x01\xed\x00\xca\x00\xc2\x00\xe9\x00\r\x01l\x01\x9c\x01\xe8\x01\xa0\x01K\x01x\x00\xe4\x01\xc9\x00\x94\x00\xab\x00\x86\x01\x87\x01\xb4\x00\n\x01S\x01\xe9\x00\xde\x00\xe1\x00\xd2\x00\x85\x00{\x00\xe3\x00\xef\x00:\x01\x81\x01`\x01\xf1\x00\xa5\x01\t\x01\xf0\x00n\x01\x8f\x00\xcb\x00\xdf\x00\x85\x01H\x01\x93\x01_\x01*\x01\xb1\x01!\x01\xca\x00^\x01g\x00\xb9\x00\xbb\x009\x01d\x01l\x01I\x01\x83\x01\xc0\x01(\x01\xf5\x00P\x01a\x00\xb9\x00\xb5\x00v\x00\x00\x01R\x00\x9c\x00\xeb\x00\x19\x01 \x01\x0f\x01\x1b\x01\x93\x00\xbc\x00\xb9\x00\x8b\x00n\x00c\x00\x8c\x00\xc5\x00\x17\x01*\x01"\x01\xc8\x00\xb0\x00\xc0\x00\xbd\x00\xae\x00\xa4\x00\xa2\x00\xab\x00\xcd\x00\x0f\x01/\x01\r\x01\xb6\x00\xb0\x00\xc1\x00\xbd\x00\xbc\x00\xb8\x00\xb8\x00\xc0\x00\xd5\x00\xf8\x00\x0e\x01\xf9\x00\xaa\x00\x9f\x00\xb8\x00\xbc\x00\xc4\x00\xc4\x00\xc5\x00\xc4\x00\xd3\x00\x0e\x01r\x01"\x01PRST\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00FCCV\n\x00\x02\x00\x14\x0b\x00\x00/\x0b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00S\x0ca\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00?\x0c#\x04\xc0\x0ct\x03\xcb\x03`\x03\x93\x03\xa6\x03v\x03\x00\x00n\x08\x96\x04\x07\x0e\xd8\x02\x02\x03\xb0\x02\xfa\x02,\x04\xb3\x02\x00\x00\x94\t\xe0\x02\xb7\x01\x02\x01\x00\x00\x00\x00\x00\x00\x00\x00\x0f\x0c\n\x04\xdc\x0c\x98\x03\xdd\x03Y\x03\xb2\x03\xb3\x03m\x03\x00\x00\x8c\x08\xdc\x04\xed\x0e\xeb\x02\x15\x03\xc1\x02\xef\x02m\x04\xd8\x02\x00\x00+\n\r\x03\xd7\x01\xfd\x00\x00\x00\x00\x00\x00\x00\x00\x00\xdf\x0bs\x04<\x0e\xb1\x03\xe6\x03\x81\x03\xab\x03\xf7\x03}\x03\x00\x00l\tH\x05\x07\x10\x18\x03\x17\x03\xd3\x02\xfb\x02\xd9\x04\x0c\x03\x00\x00\t\x0bA\x03\x03\x02\x06\x01\x00\x00\x00\x00\x00\x00\x00\x00\xaf\x0b\x8f\x04\x8c\x0e\xbe\x03\xee\x03\x84\x03\xbf\x03\xf9\x03\x86\x03\x00\x00\xac\tM\x05w\x10\x03\x03#\x03\xea\x02\r\x03"\x05\x0b\x03\x00\x00R\x0b]\x03\xfb\x01\x0b\x01\x00\x00\x00\x00\x00\x00\x00\x00\x7f\x0b\xc4\x04}\x0f\xe9\x03\x0b\x04\x92\x03\xf4\x03\x1a\x04\xa3\x03\x00\x00!\n\x9e\x05g\x11\x1f\x03]\x03\x14\x03@\x03<\x05)\x03\x00\x00\xc2\x0b\x8a\x03\x19\x02\x0c\x01\x00\x00\x00\x00\x00\x00\x00\x00O\x0b\xe2\x04\x02\x10\x08\x04 \x04\xc5\x031\x04y\x04\xd0\x03\x00\x00y\n\xd9\x05\xf0\x11J\x03y\x03\x1b\x03B\x03\xac\x05v\x03\x00\x00E\x0c\x8f\x03\x17\x02\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x1f\x0b\xf0\x04f\x10\xfe\x03U\x04\xbd\x03<\x04\xbc\x04\xde\x03\x00\x00\x88\n\xc7\x05\x1f\x12K\x03\x93\x03/\x03a\x03$\x06{\x03\x00\x00&\x0c~\x035\x02\x1c\x01\x00\x00\x00\x00\x00\x00\x00\x00\xef\n\xd6\x04#\x100\x04s\x04\xdc\x03D\x04\xc4\x04\xf6\x03\x00\x00A\n\x93\x05\x9b\x11\x85\x03\xb4\x034\x03\x7f\x039\x06\x8b\x03\x00\x00\xb2\x0bq\x03\x1d\x02\x1f\x01\x00\x00\x00\x00\x00\x00\x00\x00\xbf\n{\x049\x0f\r\x04\x89\x04\xdc\x03_\x04\xa6\x04\xdb\x03\x00\x00\x9f\t(\x05\x04\x11y\x03\xa6\x03$\x03{\x03\xf4\x05\x7f\x03\x00\x00%\x0b5\x03\x10\x02\x15\x01\x00\x00\x00\x00\x00\x00\x00\x00\x8f\n>\x04\xbf\x0e(\x04z\x04\xce\x03A\x04\x9e\x04\xce\x03\x00\x00@\t\x1b\x05U\x10~\x03\xb2\x035\x03x\x03\xf3\x05r\x03\x00\x00\xda\n\x11\x03\x01\x02\x1b\x01\x00\x00\x00\x00\x00\x00\x00\x00\xa2\x0b\xe7\x04d\x10o\x03\xeb\x03\xad\x03\x00\x04\xee\x03\xbf\x03\x00\x00\xc0\n\xbc\x05\x80\x12\xec\x02,\x03\n\x03N\x03\xed\x04p\x03\x00\x00\x88\x0c\x84\x03\x0f\x02\x13\x01\x00\x00\x00\x00\x00\x00\x00\x00\xe2\x0b\x95\x04\xe3\x0eP\x03\xbf\x03s\x03\xb8\x03\xbd\x03\x86\x03\x00\x00\xd8\tk\x05\x9d\x11\xd2\x02\x08\x03\xd7\x02+\x03~\x04\x10\x03\x00\x00\xe4\x0b_\x03\xf8\x01\x13\x01\x00\x00\x00\x00\x00\x00\x00\x00"\x0cl\x04\xf7\rS\x03\xba\x03j\x03\xa8\x03\xab\x03y\x03\x00\x009\t\x1b\x05~\x10\xbe\x02\xfe\x02\xc3\x02\x0f\x03V\x04\xe6\x02\x00\x00.\x0b/\x03\xda\x01\x0b\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00WBCZ\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\n\n\x04\x04\x00\x04\x04\x04\x04\x04\x04\x04\x04\n\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x00\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x04\x07\x00\x00\x00\x1c-\x00T\xff\xff\x00\x00\x1c9\x00T\t\t\x13\x0e\x11\x1f\x01\x00\x00\x00\x03\x00\x84 \x00\x00\xa8\xce\x01T\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00<\x91\x04T\x01\x05\x00\x00!\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00 \'\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xb3\xb2\x02T\x0f\x00\x00\x00\x03\x01\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0.\x00\x00\xf4\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Panasonic_ExifVersion': '0200',
 'Exif_Panasonic_FirmwareVersion': '\x00\x01\x00\t',
 'Exif_Panasonic_FlashBias': 0,
 'Exif_Panasonic_FocusMode': 1,
 'Exif_Panasonic_ImageStabilization': 4,
 'Exif_Panasonic_InternalSerialNumber': 'F510611290076\x00\x00\x00',
 'Exif_Panasonic_Macro': 2,
 'Exif_Panasonic_MakerNoteVersion': '0100',
 'Exif_Panasonic_NoiseReduction': 0,
 'Exif_Panasonic_OpticalZoomMode': 1,
 'Exif_Panasonic_Quality': 2,
 'Exif_Panasonic_Rotation': 1,
 'Exif_Panasonic_SceneMode': 0,
 'Exif_Panasonic_SelfTimer': 1,
 'Exif_Panasonic_SequenceNumber': 0L,
 'Exif_Panasonic_ShootingMode': 1,
 'Exif_Panasonic_TimeSincePowerOn': 6503L,
 'Exif_Panasonic_TravelDay': 65535,
 'Exif_Panasonic_WBBlueLevel': 1590,
 'Exif_Panasonic_WBGreenLevel': 1054,
 'Exif_Panasonic_WBRedLevel': 2473,
 'Exif_Panasonic_WhiteBalance': 3,
 'Exif_Panasonic_WhiteBalanceBias': 0,
 'Exif_Photo_0xea1c': '\x1c\xea\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Photo_0xea1d': 4156L,
 'Exif_Photo_ApertureValue': Rational(),
 'Exif_Photo_CFAPattern': '\x00\x02\x00\x02\x01\x02\x00\x01',
 'Exif_Photo_ColorSpace': 1,
 'Exif_Photo_ComponentsConfiguration': '\x01\x02\x03\x00',
 'Exif_Photo_CompressedBitsPerPixel': Rational(),
 'Exif_Photo_Contrast': 0,
 'Exif_Photo_CustomRendered': 0,
 'Exif_Photo_DateTimeDigitized': DateTime('2007:04:15 04:00:38'),
 'Exif_Photo_DateTimeOriginal': DateTime('2007:04:15 04:00:38'),
 'Exif_Photo_DigitalZoomRatio': Rational(),
 'Exif_Photo_ExifVersion': '0220',
 'Exif_Photo_ExposureBiasValue': Rational(),
 'Exif_Photo_ExposureIndex': Rational(),
 'Exif_Photo_ExposureMode': 1,
 'Exif_Photo_ExposureProgram': 2,
 'Exif_Photo_ExposureTime': Rational(),
 'Exif_Photo_FNumber': Rational(),
 'Exif_Photo_FileSource': '\x03',
 'Exif_Photo_Flash': 16,
 'Exif_Photo_FlashpixVersion': '0100',
 'Exif_Photo_FocalLength': Rational(),
 'Exif_Photo_FocalLengthIn35mmFilm': 35,
 'Exif_Photo_FocalPlaneResolutionUnit': 2,
 'Exif_Photo_FocalPlaneXResolution': Rational(),
 'Exif_Photo_FocalPlaneYResolution': Rational(),
 'Exif_Photo_GainControl': 1,
 'Exif_Photo_ISOSpeedRatings': 100,
 'Exif_Photo_InteroperabilityTag': 440L,
 'Exif_Photo_LightSource': 0,
 'Exif_Photo_MakerNote': 'SONY DSC \x00\x00\x00%\x00\x00\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x02\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00 \x07\x00\x01\x00\x00\x00\x00\x00\x00\x00 \xb0\x02\x00\t\x00\x00\x00\x92\x04\x00\x00@\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00A\xb0\x03\x00\x01\x00\x00\x00\x06\x00\x00\x00B\xb0\x03\x00\x01\x00\x00\x00\x01\x00\x00\x00C\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00D\xb0\x03\x00\x01\x00\x00\x00\x01\x00\x00\x00E\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00F\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00G\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00H\xb0\x08\x00\x01\x00\x00\x00\x00\x00\x00\x00I\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00J\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00K\xb0\x03\x00\x01\x00\x00\x00\x02\x00\x00\x00L\xb0\x05\x00\x01\x00\x00\x00\x9c\x04\x00\x00M\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00N\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x90\x07\x00(\x01\x00\x00\xa4\x04\x00\x00\x02\x90\x07\x00\x00\x01\x00\x00\xcc\x05\x00\x00\x03\x90\x07\x00\x90\x01\x00\x00\xcc\x06\x00\x00\x04\x90\x07\x00\xb8\x01\x00\x00\\\x08\x00\x00\x05\x90\x07\x00h\x01\x00\x00\x14\n\x00\x00\x06\x90\x07\x00D\x04\x00\x00|\x0b\x00\x00\x07\x90\x07\x00\x90\x01\x00\x00\xc0\x0f\x00\x00\x08\x90\x07\x00\x90\x01\x00\x00P\x11\x00\x00\t\x90\x07\x00\x00\x01\x00\x00\xe0\x12\x00\x00\n\x90\x07\x00\xc8\x00\x00\x00\xe0\x13\x00\x00\x0b\x90\x07\x00\xc8\x00\x00\x00\xa8\x14\x00\x00\x0c\x90\x07\x00\xf4\x01\x00\x00p\x15\x00\x00\x00\xa0\x04\x00\x01\x00\x00\x00\x04\x00\x00\x80\x01\xa0\x01\x00\x01\x00\x00\x00\x07\x00\x00\x00\x00\xa1\x07\x00\x84\x00\x00\x00d\x17\x00\x00\x01\xa1\x07\x00\x94\x01\x00\x00\xe8\x17\x00\x00\x00\xa2\x07\x00T\n\x00\x00|\x19\x00\x00Standard\x00\x00\n\x00\x00\x00\n\x00\x00\x00\xe7\x10\x00p\x86p\x01\x00\x07\x00h\x00!\x05|\xea\x00yp\x00\x00 \xea\x00R\x00\x00\x00\x00\x00\x00\x00R\x00\x00\x00P\x14(\x00\x08\x00\xd8\x00\xe63\x00\x00]Q\xd8\x00\xc4\xcb.\xea\xcf\xb6\x81\x00\\\x00\x87\x00:\x00o\xd7\x97\x88\xe6\x8ao\xd7\x97\x88\xe6\x8a0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xa9[l\x01\xf6\x00\xa9\x00\xa9\x00@\x7f\xd3$G\x7f8\x96$ J\x08J\x00\x0e\x00\x0e\x00\x00\x00\x00\x00YZ\x00\x00w/\x00\x00\xe8\xea\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x80\xc2\x00\x00\x8d\xd8\xdd\x01$\'\x00J\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1b\x00\xfb\x9b\x00\x00\x00\x00\x00\x00\x00\x00\xd8V\x8a\x00\x08\x00\x08\x00\xa5\x8a\x00\x00i\x00\xeap\xeap\xeap\xcdp\xcdpi\x00\x02p\x9b\x8a\x00\x00\x00\x00\x00\x00\x00\x00\xacp"\x8a\xff\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00i\x00Q\x00\x00\xd8}\x00\x00}\xec\x01\x01\x00}\x00\x8a\x00\x01\x00\x00\x00\x00\x00\xe7\x00\x00\x00J\x00\xff\x01\xd3\x01\x00\x00\xd1\x00\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd1\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x01\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xbd\x00\x01&\x01\xf4\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\xa5}ilO\x00\x95\xa0\xac}\x0b\xa0\xd8\x00olp}\xf1\x8d!\x17TE\x13S\x00\xe0\x078\x00\xe0\x078\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xc51\xef2\xf0O\x1aRp\xd7\x05\xe0\xd5/\x1b\xd8\xbb#Z\x8a\x1b^\xeduB\x88\x08\xd8B\xb7\x98j\xddKG\x00\x8d\xf0\x8d\x13\xaf\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\x00\x00\xd0\xbbW\xe7\xfd\xd7\xf9\xe7\x82pS\x8a_\x8aRp)\xff~\xff\x00\x00\xfc\xfe\xcc<p\xcdPs\x83\x05!\x88}\x01\x93\x00o\x00\x13\x00\xbd\xbd\xbd\xbd\xbd\xbd\xf2\x07\xfe\xe8\x17\xef\xe7D\x9eL\xdc\x18B\x1a\x9e\x10\xad\xef\xf3\x102V\x9b*]m\xb4\xd4\xcf\x84(\xf0R\xe8\xc2$\x80\'=\x1a\xaf10\xf0u\xbc$\x04\xf9z\x11\x9f!\x06\x8f\x00\xdc\x1a\x91s\x0c\x1a\x8ds\x02\xe2wF:y\x07\xa9\xdc\x1a\x91sD\xc3J\x01\x17\xef\xe7D\x00\x01\x17\xef\xe7D\x00\x00\x00\x00\x01\x00\x08\x01+\xb6\x00\x00\x00\x00\x95V$\xcd@\xe2\xbcG\x8a\xf3I\xc4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00)\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xdbJ\x00\xbd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x95\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd}\x01\xc4\x1b\xc4\x1b\xc4\x1b\xc4\x1b3\x1b3\x1b3\x1b3\x1b\xc4\x1bo\xea\xfd\xea\x92\x00d\x01\x8c\xd4(\x00\x10\xe5(\x00\x18\x15(\x00I\x15(\x00I\xff\xff\xffI\xff\xff\xffI\xff\xff\xffI\xff\xff\xff\x8a\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x01\x01\x00\x00\x01\x01\x00\x00\x01\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x08\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\xd1\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1b\xe2yyy\x87jjy`u`\x1a\x079`b\xa9by\x87\x87u\xa9J\xa9G\x87\x87`b\xa9b\x02L\x87y`u`\x1a\x079\xe2yyy\x07jjV\xcf^\xb1\x83\xcf5W\xba\x15\xcd\x9aa\xa1u\x13\x8a\xb1\xebAT\x835_4%\x7f\xa1\xd9\x13Y2]a\xebA\x9a\xd4\x82\xa1\xfc\xc8p\xd0\xa42\xee\xee\x03\xcf\x122\xb1\xe4q\xc8\x1b\xd0\xbd2\x94e2Ep\xb1.m3\xc8N\x13\x9a2oe8E\x07\x1f\x1f\xa1L\xa1\xe2tH2#eNE\xeb\xda\xe1\xa1;\xc8\xfdl\x9c\xa0\xe3\xd3\x82\xd3&\x95.\xd3&(4l\xb2\xa0\x18\x95\x94/5/\x1d\xd3\x00\x00\x12l\x94\xa0\xf1Q\xb9/5]\x8a]\x00\x00^\xa0\xc7\xa0/Q\xdf\\\xe0L\xb1Q\x00\x00\xcalz\xa0\xdc\xd3$\xe2\x90\xf0xj@\xeccl\xc7\xa02/\n\x02\xe8LW\x87\x00\x00\xbcl\xb0\xa0\xa3\x95\xf6]7\xa0\x00\x00\x00\x00\xbc\x95\xd7\x95\xda\x0c\xe0\xbbl\xec\xf9\x95\'\x870\x95\x82\xa0\xd0\xec\xfb\xbb5(\xe5Q\x00\x00\xfe\xa0\x13\xa00\x0cv\x81\x9b(X9\x00\x00V\x95w\xa0u\xec1\x0cGl9j\x00\x00\x83\x95\x08\xa0\xd9\xec\xdb\xec<\xa0\x89]K\x87\x88Q\xf7lXl\x8d\x0c(Q\x1e\xa0\x00\x00\xa6Qyl<l\xf4\x92$l\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00G\x00\xbe\x1b\xea}\x00\x00\x00\x00\x02\x00p\x00\x82\x00~@\xea}\x00\x00\x00\x00\x00\x00\x00\x00`\x00"@\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00?\x1b\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x08\xea}\x00\x00\x00\x00\x00\x00\x00\x00=\x01\xea}\xea}\x00\x00\x00\x00\x00\x00\x00\x00\r\x01\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}@}\xea}\xea}\xea}\x00\x00\x00\x00\x00\x00\x1b\x00\xef\x007\x00\xd2\x01G\x089\x1b\x1d\x1b2@\xe2}|\xd8\xd9^\xf6\xe7H\x04\xe6\x04"\x04x\xe7\xe5\x0e0\x0eV\xe7\x8f\x04GV\xf7\x0e\x8a@e\x08.\x01w\x01\xd2\x01\x86\x01\xdd\x01\xcc\x01\xa4\x01U\x01\x98\x01_\x01t\x01\xa1\x01\x9f\x01\xc4\x01\xa9\x01\x14\x01\xdb\x01A\x01\x9c\x00\x9e\x00I\x00=\x00\xbd\x00\xc3\x00_\x00D\x00t\x00\x96\x00E\x00N\x00\x13\x00\xc2\x00\xdb\x00\xe9\x00\x9c\x00O\x01\x91\x7f\xaci@\x00\x88\x00\x8e\x00\x00\x00\x95\x00$\x00p$\xff\xff\x00\x00\x01\xf3\x00\x00\x00\x83\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x0c\x00\xea\x00\x01\x00\x00\x00\x01\x00\x00\x00\xe7\x00\xd3\x000\x00\x00\x00\x00\x00\x01\x00\x01\x00\x8a\x00l\x000\x00\x00\x00\x00\x00\x00\x00\x01\x00\x1b\x00V\x00p\x00\x00\x00\x00\x00\x01\x00\x01\x00\x01\x00^\x00p\x00V\x00\x00\x00\x01\x00\x1b\x00\x01\x00V\x00\xb6\x00\x00\x00\x00\x00\x08\x00\x1b\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x1b\x006\xd8\x8a\x0ew\xe7\xb4\xe7G\x0e\x9c\xd8\xcb\x0e\xc0\xe7\xfa\x04\xbaV|\xe7\x8f\x0e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe1@\xd2\xd8\xd2\x0eE\x0e\xb1\xd84@\x94}\x8b^\xbe\xe7\r\x0e\xae^ \xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x810I(\xcd/\xc1//l\xb7\xd7\xacp/pip\x08p\x9b\x8a"\x8a\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00}\x00\x00\x00\xc7\xae\xf2\x00\xad6f\x01%Mf\x01q\x87\xfb\x00%1\x08\x00u\'(\x00\x10eG\x00I\x93\xba\x00~\xedQ\x00\x04\xca\xc2\x00\x9fR-\x00\xa2p-\x00\x16\xcc+\x00\x96\xfb\xe2\x00\xaa=\x93\x00\xae\x0e\xd0\x00\xa1&\xe2\x00\xaco\x1a\x00\xee*K\x00\x87>K\x00\xbe1\xc5\x00,Q\x00\x00\xca\x01\x08\x00\x9bZ\x01\x00\n\xe3\x01\x00\xb4\xca\x00\x00\'\xf8\x00\x00\xb4\xca\x00\x00\xb4\xca\x00\x00\xe5\x1b\xe5\x1b\xe5\x1b\xe5\x1b\xa6pr\xb6|pk\x8aq\xb6\xa7\x8a"\x88\x82\xcd\x9fp\xf7p2\x05\xba0\xa7\x05Q0fp\xa5\xb6\xd7\xb6\xfb\x8a~ \xda\x8az\x92#\x05\xef\xd7\xdb\x8a\x99\x8a\xe0\x88U\x05\xe20&p1\xb6\xc5\xb6}p\x13\x88X\x8aI\x81\xea\x8a\xf4\xbe\xa6\x8a\xaf\x8a\xe4 N\x05\x8604pt\xb6\xd8\xb6ap\x95i\xf7\x05\xc100\x05\x06 \r\x8aT\x8a\xce\x881\x05\xcf0gp\xe0\xb6y\xb6\xacp=i\xc9\xcd\xc5\x883\xcdh\x88\xdc\x8aN\x8a\t\xd7Z\x05\xd00\xf2pL\xb6\xe0\xb6Xp\xd9\xb67\x05?\x88\xe8\xcd\x91i\xcf\x8a\x9f\xb6\xf3\xb6\x98\x05\xa90\np\xf6\xb6\x04\xb6\xc1p\xcbi \x05\xb3i\xe7\xcd\xe5\xb6\x94\x05\x07\xb6\xee\xb6\xf1\x05W0\xe7V\xd8\x0e\x92H\xa1\x04V^^iU\xb8\x04V\xd8^\x05\xe0=\x04V\x0e\x04\x8a\x96\xa2\x04V\xb6\xb6\x0c\x9a\x1c\xe7V0\xd7\x071\xe1\x0eVV\xb6\xa0bI\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x08\xdc8\x01\x00\x00\x00\x00\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00$y$y\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x97f\x00\x00y\x00\x95\x0ep\xb60^\xbe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x97f\x00\x00\\\x01\x00p\x00\x00\x00\x12\x03f\xc4\x00\x00\x00tX\xf3\x95\x02`\x95\x02$$+\x00\x00\x00\x00\x00y\xba\x91\xac\xe0\x91\xe0\xba\x00\x00\x00\x00\xc3)\x00p\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\xdb)\x00\x95\xe4)\x00Q\xa0\x8b\x00\xd3\xa0\x8b\x00\xb6\x94\x8b\x00/\x94\x8b\x00$\xc2\x9e\x00\xc4b\x9e\x00\xe3b\x9e\x00y\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcd\x99\x8c\x01\x10a\x1b\x01\x13\xb4\\1\xa7w\xda\x1a\xaag\xcd\x11\xd0\xaa\x0b\r\xff\x8c\x8c\x1ft\xcc\x8f\x0f@gJ3$\x01\xc9\x0e\x8c\x82\xd6\x16\xdf\xf9jm\xec\xc93\x03\xc9"\xca*\t\x98=\x15:N\x06;\x83\x07\x91"\n\xbe\xd4:,\xff\xdb%Vy\xe87\xb3i1,\x83\nT\x18\x13\xac\xb5\'2\xe5a\t\x82qjp@\x17\x9eMx\x11\xca_\x9d\xb6(wV\x07\x04\x08\x05\x08\x1a\t\x88N\x96&\x93\xea\xdf\x8d\xaf\x92y\x00\x1dU\x1e\x08\n\xa7\x1a\x01\x13t\xce8\r\xbd\xdb\x1a\xaa\xf7_\x18\nl\n\r\xff\x1c\x1e\x16\x0e\n\x8e\x0f f\xca;\x05\x10\xc9\x0e\xec\x82v\x1f_\xf8jm\x8c\xc9\xa3\nM#\xca*c\x98\xad\x1c\xbeE\x06;\xe2\x07\x01+\xf7\xbc\xd4:L\xffK<qz\xe87\xd3i\xa1%\xe3\tT\x18s\xac%.\xf2\xe6a\t\xe2q\x98\x18?M\xc2\x17AI\r\x03H\xe6z%gV\x94\x01yL\x1a\x00,N\x03 \xc3\xce\xef$\xcf\x92\xe9\x01\xf1\xab\\/R\x9b\xe8.\\f\xdf\xee[\xf48\x19\xafWl\x01SG\xec>\xaa<>\x01=xn)\x87D\xc8\x1f\x1e`n\xe1\xe4\xa4!\x12\xb0I|-\xa6\xaa\xed\x0cjY\x8c\x18\x9f\xf6B\x01\x96\xe6[3\xef6\xc8(\xdb\x83\x19\x0f/\xfd\xaf"\xdc\xb0\xd5-x\t3\'\x14\xc2\xd8\x06\x9aQ9\x18k\xe3\xab=\x92\xcf\x01\x02\xbb\xebm\'\xee\x9d\xf0"OBl>0\xc0!\x1e\xfb\x8d<\x1a\x96 \x1c1\xa8b\xf5 E\x03\xb92\x01c\x97\xc9\x88\xfc\xac,\xbd\xd8M\x0c\xdd\xe6O\x08\xa4\xeey \x13\x04N!q\xd7\x98\t3\xf6\xb1\xf5\xf9\xb8\xc6\xcf\x15v\xe7r\xe7r\xbcV\xb5\xdd\x9c\xfb/\xadcQUB\xce\xfd\x95\xe5\xb2=\x97x(\xd5a\x03_\x82\xb9\x04Q\xc5\x99J\xa8\x865\x86\xe0\xf3e\xac0\x9f\x81 \xec\x9e\xb9!9\xc3\x7f\xb8\xe7\x04\x95\xc4\xb4\xe8\x95\xdd\xc9\xe5\x018\xcc\xf1Q\x82l>a\x87\xd4\xe7\x91!a\x88\xcb"\xe0\ty04\xe1\x99\x07M\xcfs\xd2\xc5\x95\x95\x13X\x8a\xfdT7\x08\x8a\xab\x00\x01\xc3x\xce8\r\xbd\xdb\x1a\n\xfb_\x18jl\n\r_\x10\x1e\x16N\x08n\x0e@fZ2\xa5\x13)\x0fW.\x08\x0e\x08\xf5pm\xf5\xc9\xa3\n\x8d"Z#\x03\x98=\x15\xdeG\x962\x87\x07\x9d"\xce\xb7\xd4:K\xf9\xdb5qy\xe87\x93i\xa1%\x89\nD\x18\x13\xac\xb5\'2\xe5a\t2}\x08\x11\x0fIR\x1e\x11I\x87\n\xa2\xe2\xea,wV\x94\x01\x99H\x1a\x00<N\x03 \x83\xcb\xe5$\xaf\x92y\x08Q\xae\\/2\x9bx\',`O\xe7;\xf4\xb8\x10\xcfQl\x013G|7j:>\x01=xn) C\xd2\x1f\x11`n\xe1\x14\xa3\xb1\x1b\xd0I\xec$6\xad}\x05\x0fY\x16\x11\x13\xf2B\x01\xf6\xe6\xcb:O>\xc8(\xcb\x83\x19\x0f%\xf4\xbf"\xbc\xb0E$\x18\x003\'\x14\xc2H\x0f:X\xa9\x11\xe8\xe3!4\xc3\xc5\x91\x0b[\xebm\'n\x97\xf0"\xafBl>\xd5\xca;\x1e\x02\x84\xac\x13\xd6+\x1c1\xc8be)e\x08);oc\'\xc0\x88\xf0\xac,\xdd\xd8\xdd\x05}\xeaO\x08\x94\xeey \xea\tT!\x0e\xd7\x98\t\xad\xc0\x83\x04\xb6\xf5^1\xf4u\x11\x00\xc2#\xa3/b\xa2\x82\x00\xbe\xec\xb35(\xe3i4d0\xb1\x16g\x83\x8b\x08\xed\xc0\x172\x99\x88\xae J\xd1\xb8\x03\x1c\x83\xe1\x02M<Y\x11\xb60h\x1b\xf9Kg\x0cM\\Z0\x8b\xfcG\'\xb6\xcd\x9c\x1f\x94D)7v\\,/v\x15\xef\x12\xca9\xdb2Jl\xaf\x1f\xa1i\x85\x17\xaew]\x1c\xcc\xd5j\x1b\x1fSO\x11\x04\xaf\x83\x10N\xdc\xb9\x0e\xbe\x03\xf2\x0f;\x0e\xa7\t<\xca\x180G\xfd8,E.\x97\x03\xae/z#\xf8>\xd7\x03\x02\x1af\x0f4\xb2S!)D\xa57\x8eY>:\x93e\xb3/\x1ct\xf5\x11\xb9S\x89"J\xfb\x8a)\x10\x16\xbd\x17\x0e\xf3\xb4)\xef\x9ci\x1b\xc5\xc3\x16\xe4\xe9{O\x10\x91\xce\r&>a\x0b+\x95A\xc0\x10\x88,4\x1a\x93\x9dA5\xa2\xb2l\x03*\xf7p3"By\x11g\xf9\x0b,<F3\x11F\xadL8\xb6\xc8s-bG\xe80C\t\x8b\x0c_1t3&\xee\x95,\xee%\xee&\x1e\xfb\xe8-\xcd\x0c\xe3,y\xe4e0{"\xdd\x17\x0e\xbd`\x1d[9+C\xf0W]\x0f\x0ft\x8b\x18H\xfb\xb0\r\x91\x9d95H\r\xb7(\xb6\x98\x12&\xff\xb6\x18\x14DL\x87$\n\x92\x81\x02\xab!\xf8!\xca0\x813[\xf8\xe7\x10\x18\x05\x89\x14Y\x93\xc9\x0bf\xc1g%g\x96C4\x8d\x03\x0c\x1b\xb0\xd3\xb61Z\r\xf9\x1a\xdf\xd7]\x1c\xbd\x97r,b-\xc8\x1f\xe0\x99\xdd\x1c8\xf5\x92 \xd3\x1f\x8e\x00q\xddg\x11\xed\xff\xf9\x1b\xcfB\x8e$\xb6\xe6\x8d\x08\x8c\xcd\xeb!+\xd5\x84\x0b\xf9\xf6\xf2.\xf7<\x1c*c?\xa3\x02S\xdb\xc9(#46\t\xb7\x94\xb8 I\xec=3d4y\x06/v\xba)\xa1\xf7e\x19\x8d<\x17\x1cl\xa0\xe3\x16\xc7\xf2\xa5\x17I\xb0o\x05\xde\xb5y(\x10\x97\x06\x1f\x9e\xf8\x92\x05K\x06\xd0\x1dQ\x1e\x91\x1a\xd2\x0c\xc3\x103\xa8\xd7\x0b\xa5<O\x0f\xc2\xbb\x7f0\nb\x8c6I(o#\xda\x01*)\xd5\xbb}\x01N\xc6\xdf\x13q\xd9z\x1e\x9a\x04z\x05\xc4\x94\xe5\x15`\x17\x9e\x17\xab\xe5\xf1\x1d\xb4*b\x071\xf4c\x17\x89\'\x9f5\x11\xde}80\xaad:\xee\x0f\x04\rj\xfe\xe2\x1d\x0f\x0b\x99\x06u\xcdV\x1a\xb9\xeb\xf7\x07\xb9\xca\x94\x0c\x07\xf3\xba f\xe5\xe8\x19n\xa8h.\xbe\\\x11\x1d\x10\xc1\xa9(\xa7\xa5\xe2#~B|\x1aa\x80\xae\x10\xc5\xd6=,\xc4\xcd\x8b*\x03\x0c\xd4*\xb6\xdeT\t\xdcV\x9a\x05\xcf\xdbK/\x8e\xe6O\x00\\\xd9-\x0f\x1a\x84E*i\x0c\xe4=\xf6$\xe5\x01\x8bw\xf7\x0b\x9c\xec\xd6\x12+l\xc0\x013\xb1\xeb\x0e.&\xa1\x12j\x84i.\x83\xcc\xfa\x17a\xb98*0f\x8c*\xab\x0b18\x1eB##t\x97\xfd1&\xebS?\x90/<\x17\x0e\xeec\r\x1fV\xc8\x1d\x89I \x1a\xcd\xa9\x8f#\xa6\xa6\xcc\r\x80\xdb\xff\t\xa9\xfc"\x01\xc8E5\x1d\x9dR\x9c#\xea\xf2(\x1c\x10h\x8a\x1b\xfc\x07 >;\x83\xa3\x15\xd2p#/\x93\xb7L,\x16&k\x12\xaf\x0c\xd4!\x07\x0f//\xdfn\xa7\x17\x8a\x88\xc0\x1d\xbbv\xdf\x18k\x02(\x07\xcc\xd4g%)\x7fg-\x10\xba\x94\x1aO\x00\xb9\x08\xb3w\x9f\x0e)\xfcB\x00\x1c\x9d?\x05\x91\xa4>\x0f\xec\xd9\x8a#\x96\x87f\x0cj\xb9-\x07\xcc\xb5\xc6(,\xb3\xaa%\x93\xa163\xa1+Q&x5\x0f$\x87\x96\x939\x03\xe6\x08"w\xe4;\x15\x982M\x07^A\xca6\xe1\xb0\n\x16\x08\x0478\x13$\xe9\x03$\xaf\x11\x0bc\x17\x14&\x9d\xf3\xa3\x0f\xe1_\xd3!\x18\x97!1\xa9G\xca#F\xf5\'%hF\x150\xf4\x1f\xbc6\xe8`m\x07a\x9d\xa0\x04\x82Z\xe4,\xa3\xc1\xef0\x93\xcf\xa0\x1f\xafy\xd9\x15\xdd\x91\xbe)By6\x1f\x03e\x93\x01\xd3\x10\xed\x06\xb8:\xad\x0cc\xbfO\t\x81\x9e\x991D6\xa70\xd0\x93\xf3!\xe2q\xe9\x0f\x19\xf5\xb5\x0b\x1b\x08\x9b\x1f\x91\xd8\xfe8\x02\xbb\xa9!\xe2\xf8\x1f*\xbf\xd6\x8a,[X\xf70\xac F2\xe8S\'\x13N3#7b\xe4;\x1c\xdb\x02d\x02_\x98\x1922\xf5o,\x0fO\\\x03\xff\xd8\xf4+\xd1D\x8e\x05\x96\xa4I(\x04xm\x11\xb0\xe1V22W20=\xc9\x18/\xd1\x04\xa0\x17Ao\x1f\x12\x9e\xec\x15\x136\xedf#\xb7nq\x05\x1c\xbf\xfb2\x96\x0b\x87\x07\xa4\x10i\x1a\x96 \x8f\x04Z\xb0h/\x06u\x871\xe3\xf6\xda/\x06\xb3u\x1bvEv.\xe8&\xb3.PW\xa5\x03e\xd7_3\x89W\xa3 ]P\x0c\x0e\xce\xb84\x0b^I\x82 c\xb7t\x1f-\xe4\xd1\x17E\x92&%W~\x96\x076\xc7\x9e"E\xee\xe1&r\x17Q\x1dX\x1b\x89>v\xa9R:\x9d) )\x8c|\x8a\x07\xccU\xca\x028\x07\xde\x07Cd\x14#k\xe2#\x0bl\x08\xc4\x0c|\xbb9\x02\x82yl8\x0b\x0eB-\xa1\x1b\x113\xdf\xc6\x8e\'!\xe3\x86/\xca+\x17\x03p\x83\x96\x1d\xfc\xe0\xa4\x12\xc5\x08\x9e\x14j\x80@\x12\x8fbH\x13uO\x00\x16\xc4\x80\x02.\x02+G7\x88\x99\xd86\xafwt3\x03\x97\xe8\x05E\x86\x85\x19\x05\x19\xc5\x06V\xe5\xc7\x04\x8d\x88b\x0eUV\xd25\xb0\xb67&\xbaWM\x05bg\x07$\xbe\xe3\xf1\x12if\x9c0\n\xaf\xf8\tr\xdd^2\xdc\x8e\xee\x10\xf1\xc1\xc9\x03\x15L\x0e\x1c\x01\x122#\x8bV\x8f(z\xf5\xca\x05\xc2\xca\x86$\x00\x02\xb7\x18\x90\xe4\xd5$\xdb\xf06$\x9e>\x88\x18\xeb\xf3S/i\xba\x013\xe4\xad\xb9\x1b\xdf\x08\xaa*\xb5\x87^\x15\xb8\xc4\xd0\x08\x89\xf8:\x14\x9b\xd2B*Z\xdf\x8d\x16\xfe;\x03#\x97j\xcc\x10;\xb5\x01\x195\xc2_\x14\xb0Jp7K\xbcz3\x813x\x1c\xb1\xf4\xa5\x02\xd5\xc4a\x19\xe5\xd0S(O\x12p\x02&\xf3\xf4$\xbfoK4N\xba\x98#W\x94 >\xcfV\x17\'\xdaq\xf1*\x93k\x0b6|~X\x03\x8d\x9c\xe8!\xb0\xcb\x975|\x85S/\x9c\x95\xaa\x1d\x01q\x92,w\xa7\xec\'\xefN\x06\'\xde\x9d\x11#U\x8cb!r6\xf2#\xefe\xfd\x04q3\x1f\x0e\x97\xef;\x11\x88\xa10\x17C\xae?\x1d\x03\xc5\xee/v^$\x1f\xf8&3\x18\xdeU\x86-JH\x8c.,\xca\xa250G\x04\x1e{) ;w\x81~\x15\xa0K}\x0b[\xef\xa9#\x92\x83_-u^\x99\x00`Iu\x06\x0b\xd6g/p\xc4\xa3\x03\xb7\xbaw&\xf1\x07\xf3\x12\xcb\x86I%\x85me\x16\xb9\xab\xfe\x10\xa7GW\n;\xc9\xa1\x0b\xde\xdc|7\x86\xd3\x9b)J\xec\xdb\x11\xd3\x95\xd7!Q\xa8L*iv\xce\x0f\x9a\xc7Y\x1f\x15\x83O\nT\xb9Z\x0bWi\x90\x11F\x19t1\xf6\x94i;\x95\x10Q\nj\xf3\r\x1e&(r\x15\x00\x97\x83\x06\x1a}\xdf2\xa6\xab\x9b>/\xc44\x1a9\xfe\xf9\x08\x15(_\x07{\xc6\x93\x1b\x90Z\x95\x1d\x14{\xd4$\xd4\x90%\'g\xb3|\x01\xc6I\x16\x12\x9c\x990\'PY\xdd%\xcaxw\x01e\xaaq3N%`\x1b\xcf\x0b\x04-O\xf5#\x07N\xd7I\x12\xde\x8a\xd8-\xb57;$\xfb\xd8\x94\x17\x8a%"\x01\x91\x8d\x86%\xa0\xd6\xbf\x15\xf9i|\x07\xfc*\x05\x10)\x89\xf5+\xa45$\x08\xcbB\x9c&\xc4uB\x18|\x13\xfa\x13\x13\xf4\\3\x00Tf\x10\xf8<O\x13p\x94\xcf*\x0e&\xbc&=\xd4o\x01X\xec\xf0\x03\x00\xb2f$G\x92\x127\xeb\xf6\'\x11\x15\xb2\xfd\x05W\xe7\x0f8_\xd16\xc9\xb5\x83,\xc6\xa5\x02\xb2\xfcs\x94\xe3\xd7\xe9\xf3\xe9\xe9\x81\xf3\xa2\xcb\tzx\xeeE7\xa0\xc7\xf9\x99V\xcc\xb1\xf8p\xd6W\xf4P\xc3\x7fW\xdc\xfb%\x89d\xfc\x97D\x1d\xcf\xc5\xf4\x8f\xd5e#q\xef\xd7\xfa\xab\xddc\xdc/\xfe/\xd0V\xdb/\xc0z\xe8GV\xf1\xe3\xe5\x11\xd9\xd8\xa7\xd4\xb9\xc2+*\xb1\xdd\x01\xc7\xa9\xf8\x03\t\xc4\xf7_\xcc?\xcc\xa3\xd4\xc5\xd3\xf3dS\xc6\xa9K\xb1\xdau\xe9\xde\xe2\xa7\xc3f\xddsL\xc9\xdf\xebl\x04\xd0/\x87V\xd6\xff\xbds\xf4\x81\xab\xf3\xcf\xe1\xb7\xc3\xd9i4\xf5\xe1W\x12\x1c\xcdW7\xe3\xdd3U\xd1\xfa5\xcd$\xefgE\x90\xda\x99\x04\xe3\xc63\x9dA\xf73\x17\x1d\xe7\x15\x0b1\xfb\'q,\xcd\xe5\xbc\xc0\xd9\xf32<\xc3\xe7n\x93\xfc9=)\xcf\xf5\x8b$\xf3\x05p\xbc\xceWuk\xd2\x054\xce\xc7\xf5\xe4\xd4\xe8k\x1f\t\xcc\x85\xe0L\xc4\xbf\xb2M\xee\x97]\xe3\xeb\xab\xf4\x82\xe9I\x92\x8c\xe1\t\xba\xb6\xc5\xf3\x94\x9e\xf0\x19\xb6\xed\xeb\xd9-\xb3\xf8Q\xe0\xcf\xe7yu3\xd8y\xf5\xe1\xd0\xe5E\x91\xd5\x0b$5\xc9e6@\xea\xe3\x08\t\xf0/\xf6`\xc5/G\xd2\xd2\xc5\xf1\x94\xca\x016\x90\xcf\xd5\xd0\xc3\xc5\xf1\xf2\xf2\xda\tkm\xed\xffq\xd2\xd3!\xc5;\xdey\xe6*\xf9E\xe5\\\xe3\x83\xd2\x7f\xd39d\x0e\xf65\xe0[\xefk&\x9b\xef\xa5j\x13\xe8#\x94\xd5\xf2\xf3:m\xd5\x1f\xd8\x14\xef\x01\x1f\r\xf6y#\xb0\xf1_\xe9\xd4\xd7+[\xaf\xd33\x0cu\xfbE9{\xf3k\xfd\x07\xd9\xe5*\x8d\xd5\x7f\x9aB\xfa\x95X\x98\xfd_\xf5\x05\xcae\x1b@\xddWY\x99\xfd\x17\x12$\xca\xe7r\xf9\xd8\xa3Fr\xd5',
 'Exif_Photo_MaxApertureValue': Rational(),
 'Exif_Photo_MeteringMode': 5,
 'Exif_Photo_PixelXDimension': 2288L,
 'Exif_Photo_PixelYDimension': 1728L,
 'Exif_Photo_Saturation': 0,
 'Exif_Photo_SceneCaptureType': 0,
 'Exif_Photo_SceneType': '\x01',
 'Exif_Photo_SensingMethod': 2,
 'Exif_Photo_Sharpness': 0,
 'Exif_Photo_ShutterSpeedValue': Rational(),
 'Exif_Photo_SubSecTime': '10',
 'Exif_Photo_SubSecTimeDigitized': '10',
 'Exif_Photo_SubSecTimeOriginal': '10',
 'Exif_Photo_SubjectDistanceRange': 0,
 'Exif_Photo_UserComment': '\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Photo_WhiteBalance': 0,
 'Exif_Sony_0x1000': 0L,
 'Exif_Sony_0x1001': 0L,
 'Exif_Sony_0x1002': 0L,
 'Exif_Sony_0x2000': '\x00',
 'Exif_Sony_0x9001': "\xe7\x10\x00p\x86p\x01\x00\x07\x00h\x00!\x05|\xea\x00yp\x00\x00 \xea\x00R\x00\x00\x00\x00\x00\x00\x00R\x00\x00\x00P\x14(\x00\x08\x00\xd8\x00\xe63\x00\x00]Q\xd8\x00\xc4\xcb.\xea\xcf\xb6\x81\x00\\\x00\x87\x00:\x00o\xd7\x97\x88\xe6\x8ao\xd7\x97\x88\xe6\x8a0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xa9[l\x01\xf6\x00\xa9\x00\xa9\x00@\x7f\xd3$G\x7f8\x96$ J\x08J\x00\x0e\x00\x0e\x00\x00\x00\x00\x00YZ\x00\x00w/\x00\x00\xe8\xea\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x80\xc2\x00\x00\x8d\xd8\xdd\x01$'\x00J\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1b\x00\xfb\x9b\x00\x00\x00\x00\x00\x00\x00\x00",
 'Exif_Sony_0x9002': '\xd8V\x8a\x00\x08\x00\x08\x00\xa5\x8a\x00\x00i\x00\xeap\xeap\xeap\xcdp\xcdpi\x00\x02p\x9b\x8a\x00\x00\x00\x00\x00\x00\x00\x00\xacp"\x8a\xff\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00i\x00Q\x00\x00\xd8}\x00\x00}\xec\x01\x01\x00}\x00\x8a\x00\x01\x00\x00\x00\x00\x00\xe7\x00\x00\x00J\x00\xff\x01\xd3\x01\x00\x00\xd1\x00\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd1\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x01\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x9003': "\xd8\xbd\x00\x01&\x01\xf4\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\xa5}ilO\x00\x95\xa0\xac}\x0b\xa0\xd8\x00olp}\xf1\x8d!\x17TE\x13S\x00\xe0\x078\x00\xe0\x078\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xc51\xef2\xf0O\x1aRp\xd7\x05\xe0\xd5/\x1b\xd8\xbb#Z\x8a\x1b^\xeduB\x88\x08\xd8B\xb7\x98j\xddKG\x00\x8d\xf0\x8d\x13\xaf\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\x00\x00\xd0\xbbW\xe7\xfd\xd7\xf9\xe7\x82pS\x8a_\x8aRp)\xff~\xff\x00\x00\xfc\xfe\xcc<p\xcdPs\x83\x05!\x88}\x01\x93\x00o\x00\x13\x00\xbd\xbd\xbd\xbd\xbd\xbd\xf2\x07\xfe\xe8\x17\xef\xe7D\x9eL\xdc\x18B\x1a\x9e\x10\xad\xef\xf3\x102V\x9b*]m\xb4\xd4\xcf\x84(\xf0R\xe8\xc2$\x80'=\x1a\xaf10\xf0u\xbc$\x04\xf9z\x11\x9f!\x06\x8f\x00\xdc\x1a\x91s\x0c\x1a\x8ds\x02\xe2wF:y\x07\xa9\xdc\x1a\x91sD\xc3J\x01\x17\xef\xe7D\x00\x01\x17\xef\xe7D\x00\x00\x00\x00\x01\x00\x08\x01+\xb6\x00\x00\x00\x00\x95V$\xcd@\xe2\xbcG\x8a\xf3I\xc4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00)\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xdbJ\x00\xbd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x95\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd",
 'Exif_Sony_0x9004': '}\x01\xc4\x1b\xc4\x1b\xc4\x1b\xc4\x1b3\x1b3\x1b3\x1b3\x1b\xc4\x1bo\xea\xfd\xea\x92\x00d\x01\x8c\xd4(\x00\x10\xe5(\x00\x18\x15(\x00I\x15(\x00I\xff\xff\xffI\xff\xff\xffI\xff\xff\xffI\xff\xff\xff\x8a\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x9005': '@\x01\x01\x00\x00\x01\x01\x00\x00\x01\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x08\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\xd1\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x9006': '\x1b\xe2yyy\x87jjy`u`\x1a\x079`b\xa9by\x87\x87u\xa9J\xa9G\x87\x87`b\xa9b\x02L\x87y`u`\x1a\x079\xe2yyy\x07jjV\xcf^\xb1\x83\xcf5W\xba\x15\xcd\x9aa\xa1u\x13\x8a\xb1\xebAT\x835_4%\x7f\xa1\xd9\x13Y2]a\xebA\x9a\xd4\x82\xa1\xfc\xc8p\xd0\xa42\xee\xee\x03\xcf\x122\xb1\xe4q\xc8\x1b\xd0\xbd2\x94e2Ep\xb1.m3\xc8N\x13\x9a2oe8E\x07\x1f\x1f\xa1L\xa1\xe2tH2#eNE\xeb\xda\xe1\xa1;\xc8\xfdl\x9c\xa0\xe3\xd3\x82\xd3&\x95.\xd3&(4l\xb2\xa0\x18\x95\x94/5/\x1d\xd3\x00\x00\x12l\x94\xa0\xf1Q\xb9/5]\x8a]\x00\x00^\xa0\xc7\xa0/Q\xdf\\\xe0L\xb1Q\x00\x00\xcalz\xa0\xdc\xd3$\xe2\x90\xf0xj@\xeccl\xc7\xa02/\n\x02\xe8LW\x87\x00\x00\xbcl\xb0\xa0\xa3\x95\xf6]7\xa0\x00\x00\x00\x00\xbc\x95\xd7\x95\xda\x0c\xe0\xbbl\xec\xf9\x95\'\x870\x95\x82\xa0\xd0\xec\xfb\xbb5(\xe5Q\x00\x00\xfe\xa0\x13\xa00\x0cv\x81\x9b(X9\x00\x00V\x95w\xa0u\xec1\x0cGl9j\x00\x00\x83\x95\x08\xa0\xd9\xec\xdb\xec<\xa0\x89]K\x87\x88Q\xf7lXl\x8d\x0c(Q\x1e\xa0\x00\x00\xa6Qyl<l\xf4\x92$l\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00G\x00\xbe\x1b\xea}\x00\x00\x00\x00\x02\x00p\x00\x82\x00~@\xea}\x00\x00\x00\x00\x00\x00\x00\x00`\x00"@\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00?\x1b\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x08\xea}\x00\x00\x00\x00\x00\x00\x00\x00=\x01\xea}\xea}\x00\x00\x00\x00\x00\x00\x00\x00\r\x01\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}@}\xea}\xea}\xea}\x00\x00\x00\x00\x00\x00\x1b\x00\xef\x007\x00\xd2\x01G\x089\x1b\x1d\x1b2@\xe2}|\xd8\xd9^\xf6\xe7H\x04\xe6\x04"\x04x\xe7\xe5\x0e0\x0eV\xe7\x8f\x04GV\xf7\x0e\x8a@e\x08.\x01w\x01\xd2\x01\x86\x01\xdd\x01\xcc\x01\xa4\x01U\x01\x98\x01_\x01t\x01\xa1\x01\x9f\x01\xc4\x01\xa9\x01\x14\x01\xdb\x01A\x01\x9c\x00\x9e\x00I\x00=\x00\xbd\x00\xc3\x00_\x00D\x00t\x00\x96\x00E\x00N\x00\x13\x00\xc2\x00\xdb\x00\xe9\x00\x9c\x00O\x01\x91\x7f\xaci@\x00\x88\x00\x8e\x00\x00\x00\x95\x00$\x00p$\xff\xff\x00\x00\x01\xf3\x00\x00\x00\x83\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x0c\x00\xea\x00\x01\x00\x00\x00\x01\x00\x00\x00\xe7\x00\xd3\x000\x00\x00\x00\x00\x00\x01\x00\x01\x00\x8a\x00l\x000\x00\x00\x00\x00\x00\x00\x00\x01\x00\x1b\x00V\x00p\x00\x00\x00\x00\x00\x01\x00\x01\x00\x01\x00^\x00p\x00V\x00\x00\x00\x01\x00\x1b\x00\x01\x00V\x00\xb6\x00\x00\x00\x00\x00\x08\x00\x1b\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x9007': '\x1b\x006\xd8\x8a\x0ew\xe7\xb4\xe7G\x0e\x9c\xd8\xcb\x0e\xc0\xe7\xfa\x04\xbaV|\xe7\x8f\x0e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe1@\xd2\xd8\xd2\x0eE\x0e\xb1\xd84@\x94}\x8b^\xbe\xe7\r\x0e\xae^ \xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x810I(\xcd/\xc1//l\xb7\xd7\xacp/pip\x08p\x9b\x8a"\x8a\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x9008': '}\x00\x00\x00\xc7\xae\xf2\x00\xad6f\x01%Mf\x01q\x87\xfb\x00%1\x08\x00u\'(\x00\x10eG\x00I\x93\xba\x00~\xedQ\x00\x04\xca\xc2\x00\x9fR-\x00\xa2p-\x00\x16\xcc+\x00\x96\xfb\xe2\x00\xaa=\x93\x00\xae\x0e\xd0\x00\xa1&\xe2\x00\xaco\x1a\x00\xee*K\x00\x87>K\x00\xbe1\xc5\x00,Q\x00\x00\xca\x01\x08\x00\x9bZ\x01\x00\n\xe3\x01\x00\xb4\xca\x00\x00\'\xf8\x00\x00\xb4\xca\x00\x00\xb4\xca\x00\x00\xe5\x1b\xe5\x1b\xe5\x1b\xe5\x1b\xa6pr\xb6|pk\x8aq\xb6\xa7\x8a"\x88\x82\xcd\x9fp\xf7p2\x05\xba0\xa7\x05Q0fp\xa5\xb6\xd7\xb6\xfb\x8a~ \xda\x8az\x92#\x05\xef\xd7\xdb\x8a\x99\x8a\xe0\x88U\x05\xe20&p1\xb6\xc5\xb6}p\x13\x88X\x8aI\x81\xea\x8a\xf4\xbe\xa6\x8a\xaf\x8a\xe4 N\x05\x8604pt\xb6\xd8\xb6ap\x95i\xf7\x05\xc100\x05\x06 \r\x8aT\x8a\xce\x881\x05\xcf0gp\xe0\xb6y\xb6\xacp=i\xc9\xcd\xc5\x883\xcdh\x88\xdc\x8aN\x8a\t\xd7Z\x05\xd00\xf2pL\xb6\xe0\xb6Xp\xd9\xb67\x05?\x88\xe8\xcd\x91i\xcf\x8a\x9f\xb6\xf3\xb6\x98\x05\xa90\np\xf6\xb6\x04\xb6\xc1p\xcbi \x05\xb3i\xe7\xcd\xe5\xb6\x94\x05\x07\xb6\xee\xb6\xf1\x05W0\xe7V\xd8\x0e\x92H\xa1\x04V^^iU\xb8\x04V\xd8^\x05\xe0=\x04V\x0e\x04\x8a\x96\xa2\x04V\xb6\xb6\x0c\x9a\x1c\xe7V0\xd7\x071\xe1\x0eVV\xb6\xa0bI\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x9009': '\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x900a': '\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x900b': '\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0x900c': '\x08\x08\xdc8\x01\x00\x00\x00\x00\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00$y$y\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x97f\x00\x00y\x00\x95\x0ep\xb60^\xbe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x97f\x00\x00\\\x01\x00p\x00\x00\x00\x12\x03f\xc4\x00\x00\x00tX\xf3\x95\x02`\x95\x02$$+\x00\x00\x00\x00\x00y\xba\x91\xac\xe0\x91\xe0\xba\x00\x00\x00\x00\xc3)\x00p\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\xdb)\x00\x95\xe4)\x00Q\xa0\x8b\x00\xd3\xa0\x8b\x00\xb6\x94\x8b\x00/\x94\x8b\x00$\xc2\x9e\x00\xc4b\x9e\x00\xe3b\x9e\x00y\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Exif_Sony_0xa000': 2147483652L,
 'Exif_Sony_0xa001': '7 ',
 'Exif_Sony_0xa100': '\xcd\x99\x8c\x01\x10a\x1b\x01\x13\xb4\\1\xa7w\xda\x1a\xaag\xcd\x11\xd0\xaa\x0b\r\xff\x8c\x8c\x1ft\xcc\x8f\x0f@gJ3$\x01\xc9\x0e\x8c\x82\xd6\x16\xdf\xf9jm\xec\xc93\x03\xc9"\xca*\t\x98=\x15:N\x06;\x83\x07\x91"\n\xbe\xd4:,\xff\xdb%Vy\xe87\xb3i1,\x83\nT\x18\x13\xac\xb5\'2\xe5a\t\x82qjp@\x17\x9eMx\x11\xca_\x9d\xb6(wV\x07\x04\x08\x05\x08\x1a\t\x88N\x96&\x93\xea\xdf\x8d\xaf\x92y\x00',
 'Exif_Sony_0xa101': '\x1dU\x1e\x08\n\xa7\x1a\x01\x13t\xce8\r\xbd\xdb\x1a\xaa\xf7_\x18\nl\n\r\xff\x1c\x1e\x16\x0e\n\x8e\x0f f\xca;\x05\x10\xc9\x0e\xec\x82v\x1f_\xf8jm\x8c\xc9\xa3\nM#\xca*c\x98\xad\x1c\xbeE\x06;\xe2\x07\x01+\xf7\xbc\xd4:L\xffK<qz\xe87\xd3i\xa1%\xe3\tT\x18s\xac%.\xf2\xe6a\t\xe2q\x98\x18?M\xc2\x17AI\r\x03H\xe6z%gV\x94\x01yL\x1a\x00,N\x03 \xc3\xce\xef$\xcf\x92\xe9\x01\xf1\xab\\/R\x9b\xe8.\\f\xdf\xee[\xf48\x19\xafWl\x01SG\xec>\xaa<>\x01=xn)\x87D\xc8\x1f\x1e`n\xe1\xe4\xa4!\x12\xb0I|-\xa6\xaa\xed\x0cjY\x8c\x18\x9f\xf6B\x01\x96\xe6[3\xef6\xc8(\xdb\x83\x19\x0f/\xfd\xaf"\xdc\xb0\xd5-x\t3\'\x14\xc2\xd8\x06\x9aQ9\x18k\xe3\xab=\x92\xcf\x01\x02\xbb\xebm\'\xee\x9d\xf0"OBl>0\xc0!\x1e\xfb\x8d<\x1a\x96 \x1c1\xa8b\xf5 E\x03\xb92\x01c\x97\xc9\x88\xfc\xac,\xbd\xd8M\x0c\xdd\xe6O\x08\xa4\xeey \x13\x04N!q\xd7\x98\t3\xf6\xb1\xf5\xf9\xb8\xc6\xcf\x15v\xe7r\xe7r\xbcV\xb5\xdd\x9c\xfb/\xadcQUB\xce\xfd\x95\xe5\xb2=\x97x(\xd5a\x03_\x82\xb9\x04Q\xc5\x99J\xa8\x865\x86\xe0\xf3e\xac0\x9f\x81 \xec\x9e\xb9!9\xc3\x7f\xb8\xe7\x04\x95\xc4\xb4\xe8\x95\xdd\xc9\xe5\x018\xcc\xf1Q\x82l>a\x87\xd4\xe7\x91!a\x88\xcb"\xe0\ty04\xe1\x99\x07M\xcfs\xd2\xc5\x95\x95\x13X\x8a',
 'Exif_Sony_0xa200': '\xfdT7\x08\x8a\xab\x00\x01\xc3x\xce8\r\xbd\xdb\x1a\n\xfb_\x18jl\n\r_\x10\x1e\x16N\x08n\x0e@fZ2\xa5\x13)\x0fW.\x08\x0e\x08\xf5pm\xf5\xc9\xa3\n\x8d"Z#\x03\x98=\x15\xdeG\x962\x87\x07\x9d"\xce\xb7\xd4:K\xf9\xdb5qy\xe87\x93i\xa1%\x89\nD\x18\x13\xac\xb5\'2\xe5a\t2}\x08\x11\x0fIR\x1e\x11I\x87\n\xa2\xe2\xea,wV\x94\x01\x99H\x1a\x00<N\x03 \x83\xcb\xe5$\xaf\x92y\x08Q\xae\\/2\x9bx\',`O\xe7;\xf4\xb8\x10\xcfQl\x013G|7j:>\x01=xn) C\xd2\x1f\x11`n\xe1\x14\xa3\xb1\x1b\xd0I\xec$6\xad}\x05\x0fY\x16\x11\x13\xf2B\x01\xf6\xe6\xcb:O>\xc8(\xcb\x83\x19\x0f%\xf4\xbf"\xbc\xb0E$\x18\x003\'\x14\xc2H\x0f:X\xa9\x11\xe8\xe3!4\xc3\xc5\x91\x0b[\xebm\'n\x97\xf0"\xafBl>\xd5\xca;\x1e\x02\x84\xac\x13\xd6+\x1c1\xc8be)e\x08);oc\'\xc0\x88\xf0\xac,\xdd\xd8\xdd\x05}\xeaO\x08\x94\xeey \xea\tT!\x0e\xd7\x98\t\xad\xc0\x83\x04\xb6\xf5^1\xf4u\x11\x00\xc2#\xa3/b\xa2\x82\x00\xbe\xec\xb35(\xe3i4d0\xb1\x16g\x83\x8b\x08\xed\xc0\x172\x99\x88\xae J\xd1\xb8\x03\x1c\x83\xe1\x02M<Y\x11\xb60h\x1b\xf9Kg\x0cM\\Z0\x8b\xfcG\'\xb6\xcd\x9c\x1f\x94D)7v\\,/v\x15\xef\x12\xca9\xdb2Jl\xaf\x1f\xa1i\x85\x17\xaew]\x1c\xcc\xd5j\x1b\x1fSO\x11\x04\xaf\x83\x10N\xdc\xb9\x0e\xbe\x03\xf2\x0f;\x0e\xa7\t<\xca\x180G\xfd8,E.\x97\x03\xae/z#\xf8>\xd7\x03\x02\x1af\x0f4\xb2S!)D\xa57\x8eY>:\x93e\xb3/\x1ct\xf5\x11\xb9S\x89"J\xfb\x8a)\x10\x16\xbd\x17\x0e\xf3\xb4)\xef\x9ci\x1b\xc5\xc3\x16\xe4\xe9{O\x10\x91\xce\r&>a\x0b+\x95A\xc0\x10\x88,4\x1a\x93\x9dA5\xa2\xb2l\x03*\xf7p3"By\x11g\xf9\x0b,<F3\x11F\xadL8\xb6\xc8s-bG\xe80C\t\x8b\x0c_1t3&\xee\x95,\xee%\xee&\x1e\xfb\xe8-\xcd\x0c\xe3,y\xe4e0{"\xdd\x17\x0e\xbd`\x1d[9+C\xf0W]\x0f\x0ft\x8b\x18H\xfb\xb0\r\x91\x9d95H\r\xb7(\xb6\x98\x12&\xff\xb6\x18\x14DL\x87$\n\x92\x81\x02\xab!\xf8!\xca0\x813[\xf8\xe7\x10\x18\x05\x89\x14Y\x93\xc9\x0bf\xc1g%g\x96C4\x8d\x03\x0c\x1b\xb0\xd3\xb61Z\r\xf9\x1a\xdf\xd7]\x1c\xbd\x97r,b-\xc8\x1f\xe0\x99\xdd\x1c8\xf5\x92 \xd3\x1f\x8e\x00q\xddg\x11\xed\xff\xf9\x1b\xcfB\x8e$\xb6\xe6\x8d\x08\x8c\xcd\xeb!+\xd5\x84\x0b\xf9\xf6\xf2.\xf7<\x1c*c?\xa3\x02S\xdb\xc9(#46\t\xb7\x94\xb8 I\xec=3d4y\x06/v\xba)\xa1\xf7e\x19\x8d<\x17\x1cl\xa0\xe3\x16\xc7\xf2\xa5\x17I\xb0o\x05\xde\xb5y(\x10\x97\x06\x1f\x9e\xf8\x92\x05K\x06\xd0\x1dQ\x1e\x91\x1a\xd2\x0c\xc3\x103\xa8\xd7\x0b\xa5<O\x0f\xc2\xbb\x7f0\nb\x8c6I(o#\xda\x01*)\xd5\xbb}\x01N\xc6\xdf\x13q\xd9z\x1e\x9a\x04z\x05\xc4\x94\xe5\x15`\x17\x9e\x17\xab\xe5\xf1\x1d\xb4*b\x071\xf4c\x17\x89\'\x9f5\x11\xde}80\xaad:\xee\x0f\x04\rj\xfe\xe2\x1d\x0f\x0b\x99\x06u\xcdV\x1a\xb9\xeb\xf7\x07\xb9\xca\x94\x0c\x07\xf3\xba f\xe5\xe8\x19n\xa8h.\xbe\\\x11\x1d\x10\xc1\xa9(\xa7\xa5\xe2#~B|\x1aa\x80\xae\x10\xc5\xd6=,\xc4\xcd\x8b*\x03\x0c\xd4*\xb6\xdeT\t\xdcV\x9a\x05\xcf\xdbK/\x8e\xe6O\x00\\\xd9-\x0f\x1a\x84E*i\x0c\xe4=\xf6$\xe5\x01\x8bw\xf7\x0b\x9c\xec\xd6\x12+l\xc0\x013\xb1\xeb\x0e.&\xa1\x12j\x84i.\x83\xcc\xfa\x17a\xb98*0f\x8c*\xab\x0b18\x1eB##t\x97\xfd1&\xebS?\x90/<\x17\x0e\xeec\r\x1fV\xc8\x1d\x89I \x1a\xcd\xa9\x8f#\xa6\xa6\xcc\r\x80\xdb\xff\t\xa9\xfc"\x01\xc8E5\x1d\x9dR\x9c#\xea\xf2(\x1c\x10h\x8a\x1b\xfc\x07 >;\x83\xa3\x15\xd2p#/\x93\xb7L,\x16&k\x12\xaf\x0c\xd4!\x07\x0f//\xdfn\xa7\x17\x8a\x88\xc0\x1d\xbbv\xdf\x18k\x02(\x07\xcc\xd4g%)\x7fg-\x10\xba\x94\x1aO\x00\xb9\x08\xb3w\x9f\x0e)\xfcB\x00\x1c\x9d?\x05\x91\xa4>\x0f\xec\xd9\x8a#\x96\x87f\x0cj\xb9-\x07\xcc\xb5\xc6(,\xb3\xaa%\x93\xa163\xa1+Q&x5\x0f$\x87\x96\x939\x03\xe6\x08"w\xe4;\x15\x982M\x07^A\xca6\xe1\xb0\n\x16\x08\x0478\x13$\xe9\x03$\xaf\x11\x0bc\x17\x14&\x9d\xf3\xa3\x0f\xe1_\xd3!\x18\x97!1\xa9G\xca#F\xf5\'%hF\x150\xf4\x1f\xbc6\xe8`m\x07a\x9d\xa0\x04\x82Z\xe4,\xa3\xc1\xef0\x93\xcf\xa0\x1f\xafy\xd9\x15\xdd\x91\xbe)By6\x1f\x03e\x93\x01\xd3\x10\xed\x06\xb8:\xad\x0cc\xbfO\t\x81\x9e\x991D6\xa70\xd0\x93\xf3!\xe2q\xe9\x0f\x19\xf5\xb5\x0b\x1b\x08\x9b\x1f\x91\xd8\xfe8\x02\xbb\xa9!\xe2\xf8\x1f*\xbf\xd6\x8a,[X\xf70\xac F2\xe8S\'\x13N3#7b\xe4;\x1c\xdb\x02d\x02_\x98\x1922\xf5o,\x0fO\\\x03\xff\xd8\xf4+\xd1D\x8e\x05\x96\xa4I(\x04xm\x11\xb0\xe1V22W20=\xc9\x18/\xd1\x04\xa0\x17Ao\x1f\x12\x9e\xec\x15\x136\xedf#\xb7nq\x05\x1c\xbf\xfb2\x96\x0b\x87\x07\xa4\x10i\x1a\x96 \x8f\x04Z\xb0h/\x06u\x871\xe3\xf6\xda/\x06\xb3u\x1bvEv.\xe8&\xb3.PW\xa5\x03e\xd7_3\x89W\xa3 ]P\x0c\x0e\xce\xb84\x0b^I\x82 c\xb7t\x1f-\xe4\xd1\x17E\x92&%W~\x96\x076\xc7\x9e"E\xee\xe1&r\x17Q\x1dX\x1b\x89>v\xa9R:\x9d) )\x8c|\x8a\x07\xccU\xca\x028\x07\xde\x07Cd\x14#k\xe2#\x0bl\x08\xc4\x0c|\xbb9\x02\x82yl8\x0b\x0eB-\xa1\x1b\x113\xdf\xc6\x8e\'!\xe3\x86/\xca+\x17\x03p\x83\x96\x1d\xfc\xe0\xa4\x12\xc5\x08\x9e\x14j\x80@\x12\x8fbH\x13uO\x00\x16\xc4\x80\x02.\x02+G7\x88\x99\xd86\xafwt3\x03\x97\xe8\x05E\x86\x85\x19\x05\x19\xc5\x06V\xe5\xc7\x04\x8d\x88b\x0eUV\xd25\xb0\xb67&\xbaWM\x05bg\x07$\xbe\xe3\xf1\x12if\x9c0\n\xaf\xf8\tr\xdd^2\xdc\x8e\xee\x10\xf1\xc1\xc9\x03\x15L\x0e\x1c\x01\x122#\x8bV\x8f(z\xf5\xca\x05\xc2\xca\x86$\x00\x02\xb7\x18\x90\xe4\xd5$\xdb\xf06$\x9e>\x88\x18\xeb\xf3S/i\xba\x013\xe4\xad\xb9\x1b\xdf\x08\xaa*\xb5\x87^\x15\xb8\xc4\xd0\x08\x89\xf8:\x14\x9b\xd2B*Z\xdf\x8d\x16\xfe;\x03#\x97j\xcc\x10;\xb5\x01\x195\xc2_\x14\xb0Jp7K\xbcz3\x813x\x1c\xb1\xf4\xa5\x02\xd5\xc4a\x19\xe5\xd0S(O\x12p\x02&\xf3\xf4$\xbfoK4N\xba\x98#W\x94 >\xcfV\x17\'\xdaq\xf1*\x93k\x0b6|~X\x03\x8d\x9c\xe8!\xb0\xcb\x975|\x85S/\x9c\x95\xaa\x1d\x01q\x92,w\xa7\xec\'\xefN\x06\'\xde\x9d\x11#U\x8cb!r6\xf2#\xefe\xfd\x04q3\x1f\x0e\x97\xef;\x11\x88\xa10\x17C\xae?\x1d\x03\xc5\xee/v^$\x1f\xf8&3\x18\xdeU\x86-JH\x8c.,\xca\xa250G\x04\x1e{) ;w\x81~\x15\xa0K}\x0b[\xef\xa9#\x92\x83_-u^\x99\x00`Iu\x06\x0b\xd6g/p\xc4\xa3\x03\xb7\xbaw&\xf1\x07\xf3\x12\xcb\x86I%\x85me\x16\xb9\xab\xfe\x10\xa7GW\n;\xc9\xa1\x0b\xde\xdc|7\x86\xd3\x9b)J\xec\xdb\x11\xd3\x95\xd7!Q\xa8L*iv\xce\x0f\x9a\xc7Y\x1f\x15\x83O\nT\xb9Z\x0bWi\x90\x11F\x19t1\xf6\x94i;\x95\x10Q\nj\xf3\r\x1e&(r\x15\x00\x97\x83\x06\x1a}\xdf2\xa6\xab\x9b>/\xc44\x1a9\xfe\xf9\x08\x15(_\x07{\xc6\x93\x1b\x90Z\x95\x1d\x14{\xd4$\xd4\x90%\'g\xb3|\x01\xc6I\x16\x12\x9c\x990\'PY\xdd%\xcaxw\x01e\xaaq3N%`\x1b\xcf\x0b\x04-O\xf5#\x07N\xd7I\x12\xde\x8a\xd8-\xb57;$\xfb\xd8\x94\x17\x8a%"\x01\x91\x8d\x86%\xa0\xd6\xbf\x15\xf9i|\x07\xfc*\x05\x10)\x89\xf5+\xa45$\x08\xcbB\x9c&\xc4uB\x18|\x13\xfa\x13\x13\xf4\\3\x00Tf\x10\xf8<O\x13p\x94\xcf*\x0e&\xbc&=\xd4o\x01X\xec\xf0\x03\x00\xb2f$G\x92\x127\xeb\xf6\'\x11\x15\xb2\xfd\x05W\xe7\x0f8_\xd16\xc9\xb5\x83,\xc6\xa5\x02\xb2\xfcs\x94\xe3\xd7\xe9\xf3\xe9\xe9\x81\xf3\xa2\xcb\tzx\xeeE7\xa0\xc7\xf9\x99V\xcc\xb1\xf8p\xd6W\xf4P\xc3\x7fW\xdc\xfb%\x89d\xfc\x97D\x1d\xcf\xc5\xf4\x8f\xd5e#q\xef\xd7\xfa\xab\xddc\xdc/\xfe/\xd0V\xdb/\xc0z\xe8GV\xf1\xe3\xe5\x11\xd9\xd8\xa7\xd4\xb9\xc2+*\xb1\xdd\x01\xc7\xa9\xf8\x03\t\xc4\xf7_\xcc?\xcc\xa3\xd4\xc5\xd3\xf3dS\xc6\xa9K\xb1\xdau\xe9\xde\xe2\xa7\xc3f\xddsL\xc9\xdf\xebl\x04\xd0/\x87V\xd6\xff\xbds\xf4\x81\xab\xf3\xcf\xe1\xb7\xc3\xd9i4\xf5\xe1W\x12\x1c\xcdW7\xe3\xdd3U\xd1\xfa5\xcd$\xefgE\x90\xda\x99\x04\xe3\xc63\x9dA\xf73\x17\x1d\xe7\x15\x0b1\xfb\'q,\xcd\xe5\xbc\xc0\xd9\xf32<\xc3\xe7n\x93\xfc9=)\xcf\xf5\x8b$\xf3\x05p\xbc\xceWuk\xd2\x054\xce\xc7\xf5\xe4\xd4\xe8k\x1f\t\xcc\x85\xe0L\xc4\xbf\xb2M\xee\x97]\xe3\xeb\xab\xf4\x82\xe9I\x92\x8c\xe1\t\xba\xb6\xc5\xf3\x94\x9e\xf0\x19\xb6\xed\xeb\xd9-\xb3\xf8Q\xe0\xcf\xe7yu3\xd8y\xf5\xe1\xd0\xe5E\x91\xd5\x0b$5\xc9e6@\xea\xe3\x08\t\xf0/\xf6`\xc5/G\xd2\xd2\xc5\xf1\x94\xca\x016\x90\xcf\xd5\xd0\xc3\xc5\xf1\xf2\xf2\xda\tkm\xed\xffq\xd2\xd3!\xc5;\xdey\xe6*\xf9E\xe5\\\xe3\x83\xd2\x7f\xd39d\x0e\xf65\xe0[\xefk&\x9b\xef\xa5j\x13\xe8#\x94\xd5\xf2\xf3:m\xd5\x1f\xd8\x14\xef\x01\x1f\r\xf6y#\xb0\xf1_\xe9\xd4\xd7+[\xaf\xd33\x0cu\xfbE9{\xf3k\xfd\x07\xd9\xe5*\x8d\xd5\x7f\x9aB\xfa\x95X\x98\xfd_\xf5\x05\xcae\x1b@\xddWY\x99\xfd\x17\x12$\xca\xe7r\xf9\xd8\xa3Fr\xd5',
 'Exif_Sony_0xb020': 'Standard',
 'Exif_Sony_0xb040': 0,
 'Exif_Sony_0xb041': 6,
 'Exif_Sony_0xb042': 1,
 'Exif_Sony_0xb043': 0,
 'Exif_Sony_0xb044': 1,
 'Exif_Sony_0xb045': 0,
 'Exif_Sony_0xb046': 0,
 'Exif_Sony_0xb047': 0,
 'Exif_Sony_0xb048': '0',
 'Exif_Sony_0xb049': 0,
 'Exif_Sony_0xb04a': 0,
 'Exif_Sony_0xb04b': 2,
 'Exif_Sony_0xb04c': Rational(),
 'Exif_Sony_0xb04d': 0,
 'Exif_Sony_0xb04e': 0,
 'Exif_Thumbnail_Compression': 6,
 'Exif_Thumbnail_DateTime': DateTime('2007:04:15 04:00:38'),
 'Exif_Thumbnail_JPEGInterchangeFormat': 664L,
 'Exif_Thumbnail_JPEGInterchangeFormatLength': 5906L,
 'Exif_Thumbnail_Make': 'Panasonic',
 'Exif_Thumbnail_Model': 'NV-GS400 ',
 'Exif_Thumbnail_Orientation': 1,
 'Exif_Thumbnail_ResolutionUnit': 2,
 'Exif_Thumbnail_XResolution': Rational(),
 'Exif_Thumbnail_YCbCrPositioning': 2,
 'Exif_Thumbnail_YResolution': Rational(),
 'Iptc_Application2_Byline': 'e2interactive',
 'Iptc_Application2_Caption': 'Taken Just after Sunset.',
 'Iptc_Application2_Copyright': 'Copyright 2006, e2interactive',
 'Iptc_Application2_ObjectName': 'Moon Sliver',
 'Iptc_Application2_RecordVersion': 2,
 'Pexif_ApertureValue': (297, 100),
 'Pexif_CFAPattern': '\x00\x02\x00\x02\x01\x02\x00\x01',
 'Pexif_ColorSpace': 1,
 'Pexif_ComponentsConfiguration': '\x01\x02\x03\x00',
 'Pexif_CompressedBitsPerPixel': (34, 10),
 'Pexif_DateTime': DateTime('2007:04:15 04:00:38'),
 'Pexif_DateTimeDigitized': DateTime('2007:04:15 04:00:38'),
 'Pexif_DateTimeOriginal': DateTime('2007:04:15 04:00:38'),
 'Pexif_ExifImageHeight': 1728,
 'Pexif_ExifImageWidth': 2288,
 'Pexif_ExifInteroperabilityOffset': 440,
 'Pexif_ExifOffset': 178,
 'Pexif_ExifVersion': '0220',
 'Pexif_ExposureIndex': (100, 1),
 'Pexif_ExposureProgram': 2,
 'Pexif_ExposureTime': (1, 2000),
 'Pexif_FNumber': (17, 10),
 'Pexif_FileSource': '\x03',
 'Pexif_Flash': 16,
 'Pexif_FlashPixVersion': '0100',
 'Pexif_FocalLength': (580, 100),
 'Pexif_FocalPlaneResolutionUnit': 2,
 'Pexif_FocalPlaneXResolution': (3648000, 287),
 'Pexif_FocalPlaneYResolution': (2736000, 215),
 'Pexif_GPSInfo': {0: (2, 2, 0, 0),
                   1: 'N',
                   2: ((62, 1), (18, 1), (34, 1)),
                   3: 'E',
                   4: ((25, 1), (44, 1), (4, 1)),
                   5: 0,
                   6: (13600, 100)},
 'Pexif_ISOSpeedRatings': 100,
 'Pexif_ImageDescription': '                               ',
 'Pexif_LightSource': 0,
 'Pexif_Make': 'Panasonic',
 'Pexif_MakerNote': 'SONY DSC \x00\x00\x00%\x00\x00\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x02\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00 \x07\x00\x01\x00\x00\x00\x00\x00\x00\x00 \xb0\x02\x00\t\x00\x00\x00\x92\x04\x00\x00@\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00A\xb0\x03\x00\x01\x00\x00\x00\x06\x00\x00\x00B\xb0\x03\x00\x01\x00\x00\x00\x01\x00\x00\x00C\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00D\xb0\x03\x00\x01\x00\x00\x00\x01\x00\x00\x00E\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00F\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00G\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00H\xb0\x08\x00\x01\x00\x00\x00\x00\x00\x00\x00I\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00J\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00K\xb0\x03\x00\x01\x00\x00\x00\x02\x00\x00\x00L\xb0\x05\x00\x01\x00\x00\x00\x9c\x04\x00\x00M\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00N\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x90\x07\x00(\x01\x00\x00\xa4\x04\x00\x00\x02\x90\x07\x00\x00\x01\x00\x00\xcc\x05\x00\x00\x03\x90\x07\x00\x90\x01\x00\x00\xcc\x06\x00\x00\x04\x90\x07\x00\xb8\x01\x00\x00\\\x08\x00\x00\x05\x90\x07\x00h\x01\x00\x00\x14\n\x00\x00\x06\x90\x07\x00D\x04\x00\x00|\x0b\x00\x00\x07\x90\x07\x00\x90\x01\x00\x00\xc0\x0f\x00\x00\x08\x90\x07\x00\x90\x01\x00\x00P\x11\x00\x00\t\x90\x07\x00\x00\x01\x00\x00\xe0\x12\x00\x00\n\x90\x07\x00\xc8\x00\x00\x00\xe0\x13\x00\x00\x0b\x90\x07\x00\xc8\x00\x00\x00\xa8\x14\x00\x00\x0c\x90\x07\x00\xf4\x01\x00\x00p\x15\x00\x00\x00\xa0\x04\x00\x01\x00\x00\x00\x04\x00\x00\x80\x01\xa0\x01\x00\x01\x00\x00\x00\x07\x00\x00\x00\x00\xa1\x07\x00\x84\x00\x00\x00d\x17\x00\x00\x01\xa1\x07\x00\x94\x01\x00\x00\xe8\x17\x00\x00\x00\xa2\x07\x00T\n\x00\x00|\x19\x00\x00Standard\x00\x00\n\x00\x00\x00\n\x00\x00\x00\xe7\x10\x00p\x86p\x01\x00\x07\x00h\x00!\x05|\xea\x00yp\x00\x00 \xea\x00R\x00\x00\x00\x00\x00\x00\x00R\x00\x00\x00P\x14(\x00\x08\x00\xd8\x00\xe63\x00\x00]Q\xd8\x00\xc4\xcb.\xea\xcf\xb6\x81\x00\\\x00\x87\x00:\x00o\xd7\x97\x88\xe6\x8ao\xd7\x97\x88\xe6\x8a0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xa9[l\x01\xf6\x00\xa9\x00\xa9\x00@\x7f\xd3$G\x7f8\x96$ J\x08J\x00\x0e\x00\x0e\x00\x00\x00\x00\x00YZ\x00\x00w/\x00\x00\xe8\xea\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x80\xc2\x00\x00\x8d\xd8\xdd\x01$\'\x00J\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1b\x00\xfb\x9b\x00\x00\x00\x00\x00\x00\x00\x00\xd8V\x8a\x00\x08\x00\x08\x00\xa5\x8a\x00\x00i\x00\xeap\xeap\xeap\xcdp\xcdpi\x00\x02p\x9b\x8a\x00\x00\x00\x00\x00\x00\x00\x00\xacp"\x8a\xff\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00i\x00Q\x00\x00\xd8}\x00\x00}\xec\x01\x01\x00}\x00\x8a\x00\x01\x00\x00\x00\x00\x00\xe7\x00\x00\x00J\x00\xff\x01\xd3\x01\x00\x00\xd1\x00\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd1\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x01\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xbd\x00\x01&\x01\xf4\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\xa5}ilO\x00\x95\xa0\xac}\x0b\xa0\xd8\x00olp}\xf1\x8d!\x17TE\x13S\x00\xe0\x078\x00\xe0\x078\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xc51\xef2\xf0O\x1aRp\xd7\x05\xe0\xd5/\x1b\xd8\xbb#Z\x8a\x1b^\xeduB\x88\x08\xd8B\xb7\x98j\xddKG\x00\x8d\xf0\x8d\x13\xaf\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\x00\x00\xd0\xbbW\xe7\xfd\xd7\xf9\xe7\x82pS\x8a_\x8aRp)\xff~\xff\x00\x00\xfc\xfe\xcc<p\xcdPs\x83\x05!\x88}\x01\x93\x00o\x00\x13\x00\xbd\xbd\xbd\xbd\xbd\xbd\xf2\x07\xfe\xe8\x17\xef\xe7D\x9eL\xdc\x18B\x1a\x9e\x10\xad\xef\xf3\x102V\x9b*]m\xb4\xd4\xcf\x84(\xf0R\xe8\xc2$\x80\'=\x1a\xaf10\xf0u\xbc$\x04\xf9z\x11\x9f!\x06\x8f\x00\xdc\x1a\x91s\x0c\x1a\x8ds\x02\xe2wF:y\x07\xa9\xdc\x1a\x91sD\xc3J\x01\x17\xef\xe7D\x00\x01\x17\xef\xe7D\x00\x00\x00\x00\x01\x00\x08\x01+\xb6\x00\x00\x00\x00\x95V$\xcd@\xe2\xbcG\x8a\xf3I\xc4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00)\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xdbJ\x00\xbd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x95\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd}\x01\xc4\x1b\xc4\x1b\xc4\x1b\xc4\x1b3\x1b3\x1b3\x1b3\x1b\xc4\x1bo\xea\xfd\xea\x92\x00d\x01\x8c\xd4(\x00\x10\xe5(\x00\x18\x15(\x00I\x15(\x00I\xff\xff\xffI\xff\xff\xffI\xff\xff\xffI\xff\xff\xff\x8a\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x01\x01\x00\x00\x01\x01\x00\x00\x01\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x08\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\xd1\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1b\xe2yyy\x87jjy`u`\x1a\x079`b\xa9by\x87\x87u\xa9J\xa9G\x87\x87`b\xa9b\x02L\x87y`u`\x1a\x079\xe2yyy\x07jjV\xcf^\xb1\x83\xcf5W\xba\x15\xcd\x9aa\xa1u\x13\x8a\xb1\xebAT\x835_4%\x7f\xa1\xd9\x13Y2]a\xebA\x9a\xd4\x82\xa1\xfc\xc8p\xd0\xa42\xee\xee\x03\xcf\x122\xb1\xe4q\xc8\x1b\xd0\xbd2\x94e2Ep\xb1.m3\xc8N\x13\x9a2oe8E\x07\x1f\x1f\xa1L\xa1\xe2tH2#eNE\xeb\xda\xe1\xa1;\xc8\xfdl\x9c\xa0\xe3\xd3\x82\xd3&\x95.\xd3&(4l\xb2\xa0\x18\x95\x94/5/\x1d\xd3\x00\x00\x12l\x94\xa0\xf1Q\xb9/5]\x8a]\x00\x00^\xa0\xc7\xa0/Q\xdf\\\xe0L\xb1Q\x00\x00\xcalz\xa0\xdc\xd3$\xe2\x90\xf0xj@\xeccl\xc7\xa02/\n\x02\xe8LW\x87\x00\x00\xbcl\xb0\xa0\xa3\x95\xf6]7\xa0\x00\x00\x00\x00\xbc\x95\xd7\x95\xda\x0c\xe0\xbbl\xec\xf9\x95\'\x870\x95\x82\xa0\xd0\xec\xfb\xbb5(\xe5Q\x00\x00\xfe\xa0\x13\xa00\x0cv\x81\x9b(X9\x00\x00V\x95w\xa0u\xec1\x0cGl9j\x00\x00\x83\x95\x08\xa0\xd9\xec\xdb\xec<\xa0\x89]K\x87\x88Q\xf7lXl\x8d\x0c(Q\x1e\xa0\x00\x00\xa6Qyl<l\xf4\x92$l\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00G\x00\xbe\x1b\xea}\x00\x00\x00\x00\x02\x00p\x00\x82\x00~@\xea}\x00\x00\x00\x00\x00\x00\x00\x00`\x00"@\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00?\x1b\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x08\xea}\x00\x00\x00\x00\x00\x00\x00\x00=\x01\xea}\xea}\x00\x00\x00\x00\x00\x00\x00\x00\r\x01\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}@}\xea}\xea}\xea}\x00\x00\x00\x00\x00\x00\x1b\x00\xef\x007\x00\xd2\x01G\x089\x1b\x1d\x1b2@\xe2}|\xd8\xd9^\xf6\xe7H\x04\xe6\x04"\x04x\xe7\xe5\x0e0\x0eV\xe7\x8f\x04GV\xf7\x0e\x8a@e\x08.\x01w\x01\xd2\x01\x86\x01\xdd\x01\xcc\x01\xa4\x01U\x01\x98\x01_\x01t\x01\xa1\x01\x9f\x01\xc4\x01\xa9\x01\x14\x01\xdb\x01A\x01\x9c\x00\x9e\x00I\x00=\x00\xbd\x00\xc3\x00_\x00D\x00t\x00\x96\x00E\x00N\x00\x13\x00\xc2\x00\xdb\x00\xe9\x00\x9c\x00O\x01\x91\x7f\xaci@\x00\x88\x00\x8e\x00\x00\x00\x95\x00$\x00p$\xff\xff\x00\x00\x01\xf3\x00\x00\x00\x83\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x0c\x00\xea\x00\x01\x00\x00\x00\x01\x00\x00\x00\xe7\x00\xd3\x000\x00\x00\x00\x00\x00\x01\x00\x01\x00\x8a\x00l\x000\x00\x00\x00\x00\x00\x00\x00\x01\x00\x1b\x00V\x00p\x00\x00\x00\x00\x00\x01\x00\x01\x00\x01\x00^\x00p\x00V\x00\x00\x00\x01\x00\x1b\x00\x01\x00V\x00\xb6\x00\x00\x00\x00\x00\x08\x00\x1b\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x1b\x006\xd8\x8a\x0ew\xe7\xb4\xe7G\x0e\x9c\xd8\xcb\x0e\xc0\xe7\xfa\x04\xbaV|\xe7\x8f\x0e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe1@\xd2\xd8\xd2\x0eE\x0e\xb1\xd84@\x94}\x8b^\xbe\xe7\r\x0e\xae^ \xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x810I(\xcd/\xc1//l\xb7\xd7\xacp/pip\x08p\x9b\x8a"\x8a\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00}\x00\x00\x00\xc7\xae\xf2\x00\xad6f\x01%Mf\x01q\x87\xfb\x00%1\x08\x00u\'(\x00\x10eG\x00I\x93\xba\x00~\xedQ\x00\x04\xca\xc2\x00\x9fR-\x00\xa2p-\x00\x16\xcc+\x00\x96\xfb\xe2\x00\xaa=\x93\x00\xae\x0e\xd0\x00\xa1&\xe2\x00\xaco\x1a\x00\xee*K\x00\x87>K\x00\xbe1\xc5\x00,Q\x00\x00\xca\x01\x08\x00\x9bZ\x01\x00\n\xe3\x01\x00\xb4\xca\x00\x00\'\xf8\x00\x00\xb4\xca\x00\x00\xb4\xca\x00\x00\xe5\x1b\xe5\x1b\xe5\x1b\xe5\x1b\xa6pr\xb6|pk\x8aq\xb6\xa7\x8a"\x88\x82\xcd\x9fp\xf7p2\x05\xba0\xa7\x05Q0fp\xa5\xb6\xd7\xb6\xfb\x8a~ \xda\x8az\x92#\x05\xef\xd7\xdb\x8a\x99\x8a\xe0\x88U\x05\xe20&p1\xb6\xc5\xb6}p\x13\x88X\x8aI\x81\xea\x8a\xf4\xbe\xa6\x8a\xaf\x8a\xe4 N\x05\x8604pt\xb6\xd8\xb6ap\x95i\xf7\x05\xc100\x05\x06 \r\x8aT\x8a\xce\x881\x05\xcf0gp\xe0\xb6y\xb6\xacp=i\xc9\xcd\xc5\x883\xcdh\x88\xdc\x8aN\x8a\t\xd7Z\x05\xd00\xf2pL\xb6\xe0\xb6Xp\xd9\xb67\x05?\x88\xe8\xcd\x91i\xcf\x8a\x9f\xb6\xf3\xb6\x98\x05\xa90\np\xf6\xb6\x04\xb6\xc1p\xcbi \x05\xb3i\xe7\xcd\xe5\xb6\x94\x05\x07\xb6\xee\xb6\xf1\x05W0\xe7V\xd8\x0e\x92H\xa1\x04V^^iU\xb8\x04V\xd8^\x05\xe0=\x04V\x0e\x04\x8a\x96\xa2\x04V\xb6\xb6\x0c\x9a\x1c\xe7V0\xd7\x071\xe1\x0eVV\xb6\xa0bI\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x08\xdc8\x01\x00\x00\x00\x00\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00$y$y\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x97f\x00\x00y\x00\x95\x0ep\xb60^\xbe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x97f\x00\x00\\\x01\x00p\x00\x00\x00\x12\x03f\xc4\x00\x00\x00tX\xf3\x95\x02`\x95\x02$$+\x00\x00\x00\x00\x00y\xba\x91\xac\xe0\x91\xe0\xba\x00\x00\x00\x00\xc3)\x00p\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\xdb)\x00\x95\xe4)\x00Q\xa0\x8b\x00\xd3\xa0\x8b\x00\xb6\x94\x8b\x00/\x94\x8b\x00$\xc2\x9e\x00\xc4b\x9e\x00\xe3b\x9e\x00y\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcd\x99\x8c\x01\x10a\x1b\x01\x13\xb4\\1\xa7w\xda\x1a\xaag\xcd\x11\xd0\xaa\x0b\r\xff\x8c\x8c\x1ft\xcc\x8f\x0f@gJ3$\x01\xc9\x0e\x8c\x82\xd6\x16\xdf\xf9jm\xec\xc93\x03\xc9"\xca*\t\x98=\x15:N\x06;\x83\x07\x91"\n\xbe\xd4:,\xff\xdb%Vy\xe87\xb3i1,\x83\nT\x18\x13\xac\xb5\'2\xe5a\t\x82qjp@\x17\x9eMx\x11\xca_\x9d\xb6(wV\x07\x04\x08\x05\x08\x1a\t\x88N\x96&\x93\xea\xdf\x8d\xaf\x92y\x00\x1dU\x1e\x08\n\xa7\x1a\x01\x13t\xce8\r\xbd\xdb\x1a\xaa\xf7_\x18\nl\n\r\xff\x1c\x1e\x16\x0e\n\x8e\x0f f\xca;\x05\x10\xc9\x0e\xec\x82v\x1f_\xf8jm\x8c\xc9\xa3\nM#\xca*c\x98\xad\x1c\xbeE\x06;\xe2\x07\x01+\xf7\xbc\xd4:L\xffK<qz\xe87\xd3i\xa1%\xe3\tT\x18s\xac%.\xf2\xe6a\t\xe2q\x98\x18?M\xc2\x17AI\r\x03H\xe6z%gV\x94\x01yL\x1a\x00,N\x03 \xc3\xce\xef$\xcf\x92\xe9\x01\xf1\xab\\/R\x9b\xe8.\\f\xdf\xee[\xf48\x19\xafWl\x01SG\xec>\xaa<>\x01=xn)\x87D\xc8\x1f\x1e`n\xe1\xe4\xa4!\x12\xb0I|-\xa6\xaa\xed\x0cjY\x8c\x18\x9f\xf6B\x01\x96\xe6[3\xef6\xc8(\xdb\x83\x19\x0f/\xfd\xaf"\xdc\xb0\xd5-x\t3\'\x14\xc2\xd8\x06\x9aQ9\x18k\xe3\xab=\x92\xcf\x01\x02\xbb\xebm\'\xee\x9d\xf0"OBl>0\xc0!\x1e\xfb\x8d<\x1a\x96 \x1c1\xa8b\xf5 E\x03\xb92\x01c\x97\xc9\x88\xfc\xac,\xbd\xd8M\x0c\xdd\xe6O\x08\xa4\xeey \x13\x04N!q\xd7\x98\t3\xf6\xb1\xf5\xf9\xb8\xc6\xcf\x15v\xe7r\xe7r\xbcV\xb5\xdd\x9c\xfb/\xadcQUB\xce\xfd\x95\xe5\xb2=\x97x(\xd5a\x03_\x82\xb9\x04Q\xc5\x99J\xa8\x865\x86\xe0\xf3e\xac0\x9f\x81 \xec\x9e\xb9!9\xc3\x7f\xb8\xe7\x04\x95\xc4\xb4\xe8\x95\xdd\xc9\xe5\x018\xcc\xf1Q\x82l>a\x87\xd4\xe7\x91!a\x88\xcb"\xe0\ty04\xe1\x99\x07M\xcfs\xd2\xc5\x95\x95\x13X\x8a\xfdT7\x08\x8a\xab\x00\x01\xc3x\xce8\r\xbd\xdb\x1a\n\xfb_\x18jl\n\r_\x10\x1e\x16N\x08n\x0e@fZ2\xa5\x13)\x0fW.\x08\x0e\x08\xf5pm\xf5\xc9\xa3\n\x8d"Z#\x03\x98=\x15\xdeG\x962\x87\x07\x9d"\xce\xb7\xd4:K\xf9\xdb5qy\xe87\x93i\xa1%\x89\nD\x18\x13\xac\xb5\'2\xe5a\t2}\x08\x11\x0fIR\x1e\x11I\x87\n\xa2\xe2\xea,wV\x94\x01\x99H\x1a\x00<N\x03 \x83\xcb\xe5$\xaf\x92y\x08Q\xae\\/2\x9bx\',`O\xe7;\xf4\xb8\x10\xcfQl\x013G|7j:>\x01=xn) C\xd2\x1f\x11`n\xe1\x14\xa3\xb1\x1b\xd0I\xec$6\xad}\x05\x0fY\x16\x11\x13\xf2B\x01\xf6\xe6\xcb:O>\xc8(\xcb\x83\x19\x0f%\xf4\xbf"\xbc\xb0E$\x18\x003\'\x14\xc2H\x0f:X\xa9\x11\xe8\xe3!4\xc3\xc5\x91\x0b[\xebm\'n\x97\xf0"\xafBl>\xd5\xca;\x1e\x02\x84\xac\x13\xd6+\x1c1\xc8be)e\x08);oc\'\xc0\x88\xf0\xac,\xdd\xd8\xdd\x05}\xeaO\x08\x94\xeey \xea\tT!\x0e\xd7\x98\t\xad\xc0\x83\x04\xb6\xf5^1\xf4u\x11\x00\xc2#\xa3/b\xa2\x82\x00\xbe\xec\xb35(\xe3i4d0\xb1\x16g\x83\x8b\x08\xed\xc0\x172\x99\x88\xae J\xd1\xb8\x03\x1c\x83\xe1\x02M<Y\x11\xb60h\x1b\xf9Kg\x0cM\\Z0\x8b\xfcG\'\xb6\xcd\x9c\x1f\x94D)7v\\,/v\x15\xef\x12\xca9\xdb2Jl\xaf\x1f\xa1i\x85\x17\xaew]\x1c\xcc\xd5j\x1b\x1fSO\x11\x04\xaf\x83\x10N\xdc\xb9\x0e\xbe\x03\xf2\x0f;\x0e\xa7\t<\xca\x180G\xfd8,E.\x97\x03\xae/z#\xf8>\xd7\x03\x02\x1af\x0f4\xb2S!)D\xa57\x8eY>:\x93e\xb3/\x1ct\xf5\x11\xb9S\x89"J\xfb\x8a)\x10\x16\xbd\x17\x0e\xf3\xb4)\xef\x9ci\x1b\xc5\xc3\x16\xe4\xe9{O\x10\x91\xce\r&>a\x0b+\x95A\xc0\x10\x88,4\x1a\x93\x9dA5\xa2\xb2l\x03*\xf7p3"By\x11g\xf9\x0b,<F3\x11F\xadL8\xb6\xc8s-bG\xe80C\t\x8b\x0c_1t3&\xee\x95,\xee%\xee&\x1e\xfb\xe8-\xcd\x0c\xe3,y\xe4e0{"\xdd\x17\x0e\xbd`\x1d[9+C\xf0W]\x0f\x0ft\x8b\x18H\xfb\xb0\r\x91\x9d95H\r\xb7(\xb6\x98\x12&\xff\xb6\x18\x14DL\x87$\n\x92\x81\x02\xab!\xf8!\xca0\x813[\xf8\xe7\x10\x18\x05\x89\x14Y\x93\xc9\x0bf\xc1g%g\x96C4\x8d\x03\x0c\x1b\xb0\xd3\xb61Z\r\xf9\x1a\xdf\xd7]\x1c\xbd\x97r,b-\xc8\x1f\xe0\x99\xdd\x1c8\xf5\x92 \xd3\x1f\x8e\x00q\xddg\x11\xed\xff\xf9\x1b\xcfB\x8e$\xb6\xe6\x8d\x08\x8c\xcd\xeb!+\xd5\x84\x0b\xf9\xf6\xf2.\xf7<\x1c*c?\xa3\x02S\xdb\xc9(#46\t\xb7\x94\xb8 I\xec=3d4y\x06/v\xba)\xa1\xf7e\x19\x8d<\x17\x1cl\xa0\xe3\x16\xc7\xf2\xa5\x17I\xb0o\x05\xde\xb5y(\x10\x97\x06\x1f\x9e\xf8\x92\x05K\x06\xd0\x1dQ\x1e\x91\x1a\xd2\x0c\xc3\x103\xa8\xd7\x0b\xa5<O\x0f\xc2\xbb\x7f0\nb\x8c6I(o#\xda\x01*)\xd5\xbb}\x01N\xc6\xdf\x13q\xd9z\x1e\x9a\x04z\x05\xc4\x94\xe5\x15`\x17\x9e\x17\xab\xe5\xf1\x1d\xb4*b\x071\xf4c\x17\x89\'\x9f5\x11\xde}80\xaad:\xee\x0f\x04\rj\xfe\xe2\x1d\x0f\x0b\x99\x06u\xcdV\x1a\xb9\xeb\xf7\x07\xb9\xca\x94\x0c\x07\xf3\xba f\xe5\xe8\x19n\xa8h.\xbe\\\x11\x1d\x10\xc1\xa9(\xa7\xa5\xe2#~B|\x1aa\x80\xae\x10\xc5\xd6=,\xc4\xcd\x8b*\x03\x0c\xd4*\xb6\xdeT\t\xdcV\x9a\x05\xcf\xdbK/\x8e\xe6O\x00\\\xd9-\x0f\x1a\x84E*i\x0c\xe4=\xf6$\xe5\x01\x8bw\xf7\x0b\x9c\xec\xd6\x12+l\xc0\x013\xb1\xeb\x0e.&\xa1\x12j\x84i.\x83\xcc\xfa\x17a\xb98*0f\x8c*\xab\x0b18\x1eB##t\x97\xfd1&\xebS?\x90/<\x17\x0e\xeec\r\x1fV\xc8\x1d\x89I \x1a\xcd\xa9\x8f#\xa6\xa6\xcc\r\x80\xdb\xff\t\xa9\xfc"\x01\xc8E5\x1d\x9dR\x9c#\xea\xf2(\x1c\x10h\x8a\x1b\xfc\x07 >;\x83\xa3\x15\xd2p#/\x93\xb7L,\x16&k\x12\xaf\x0c\xd4!\x07\x0f//\xdfn\xa7\x17\x8a\x88\xc0\x1d\xbbv\xdf\x18k\x02(\x07\xcc\xd4g%)\x7fg-\x10\xba\x94\x1aO\x00\xb9\x08\xb3w\x9f\x0e)\xfcB\x00\x1c\x9d?\x05\x91\xa4>\x0f\xec\xd9\x8a#\x96\x87f\x0cj\xb9-\x07\xcc\xb5\xc6(,\xb3\xaa%\x93\xa163\xa1+Q&x5\x0f$\x87\x96\x939\x03\xe6\x08"w\xe4;\x15\x982M\x07^A\xca6\xe1\xb0\n\x16\x08\x0478\x13$\xe9\x03$\xaf\x11\x0bc\x17\x14&\x9d\xf3\xa3\x0f\xe1_\xd3!\x18\x97!1\xa9G\xca#F\xf5\'%hF\x150\xf4\x1f\xbc6\xe8`m\x07a\x9d\xa0\x04\x82Z\xe4,\xa3\xc1\xef0\x93\xcf\xa0\x1f\xafy\xd9\x15\xdd\x91\xbe)By6\x1f\x03e\x93\x01\xd3\x10\xed\x06\xb8:\xad\x0cc\xbfO\t\x81\x9e\x991D6\xa70\xd0\x93\xf3!\xe2q\xe9\x0f\x19\xf5\xb5\x0b\x1b\x08\x9b\x1f\x91\xd8\xfe8\x02\xbb\xa9!\xe2\xf8\x1f*\xbf\xd6\x8a,[X\xf70\xac F2\xe8S\'\x13N3#7b\xe4;\x1c\xdb\x02d\x02_\x98\x1922\xf5o,\x0fO\\\x03\xff\xd8\xf4+\xd1D\x8e\x05\x96\xa4I(\x04xm\x11\xb0\xe1V22W20=\xc9\x18/\xd1\x04\xa0\x17Ao\x1f\x12\x9e\xec\x15\x136\xedf#\xb7nq\x05\x1c\xbf\xfb2\x96\x0b\x87\x07\xa4\x10i\x1a\x96 \x8f\x04Z\xb0h/\x06u\x871\xe3\xf6\xda/\x06\xb3u\x1bvEv.\xe8&\xb3.PW\xa5\x03e\xd7_3\x89W\xa3 ]P\x0c\x0e\xce\xb84\x0b^I\x82 c\xb7t\x1f-\xe4\xd1\x17E\x92&%W~\x96\x076\xc7\x9e"E\xee\xe1&r\x17Q\x1dX\x1b\x89>v\xa9R:\x9d) )\x8c|\x8a\x07\xccU\xca\x028\x07\xde\x07Cd\x14#k\xe2#\x0bl\x08\xc4\x0c|\xbb9\x02\x82yl8\x0b\x0eB-\xa1\x1b\x113\xdf\xc6\x8e\'!\xe3\x86/\xca+\x17\x03p\x83\x96\x1d\xfc\xe0\xa4\x12\xc5\x08\x9e\x14j\x80@\x12\x8fbH\x13uO\x00\x16\xc4\x80\x02.\x02+G7\x88\x99\xd86\xafwt3\x03\x97\xe8\x05E\x86\x85\x19\x05\x19\xc5\x06V\xe5\xc7\x04\x8d\x88b\x0eUV\xd25\xb0\xb67&\xbaWM\x05bg\x07$\xbe\xe3\xf1\x12if\x9c0\n\xaf\xf8\tr\xdd^2\xdc\x8e\xee\x10\xf1\xc1\xc9\x03\x15L\x0e\x1c\x01\x122#\x8bV\x8f(z\xf5\xca\x05\xc2\xca\x86$\x00\x02\xb7\x18\x90\xe4\xd5$\xdb\xf06$\x9e>\x88\x18\xeb\xf3S/i\xba\x013\xe4\xad\xb9\x1b\xdf\x08\xaa*\xb5\x87^\x15\xb8\xc4\xd0\x08\x89\xf8:\x14\x9b\xd2B*Z\xdf\x8d\x16\xfe;\x03#\x97j\xcc\x10;\xb5\x01\x195\xc2_\x14\xb0Jp7K\xbcz3\x813x\x1c\xb1\xf4\xa5\x02\xd5\xc4a\x19\xe5\xd0S(O\x12p\x02&\xf3\xf4$\xbfoK4N\xba\x98#W\x94 >\xcfV\x17\'\xdaq\xf1*\x93k\x0b6|~X\x03\x8d\x9c\xe8!\xb0\xcb\x975|\x85S/\x9c\x95\xaa\x1d\x01q\x92,w\xa7\xec\'\xefN\x06\'\xde\x9d\x11#U\x8cb!r6\xf2#\xefe\xfd\x04q3\x1f\x0e\x97\xef;\x11\x88\xa10\x17C\xae?\x1d\x03\xc5\xee/v^$\x1f\xf8&3\x18\xdeU\x86-JH\x8c.,\xca\xa250G\x04\x1e{) ;w\x81~\x15\xa0K}\x0b[\xef\xa9#\x92\x83_-u^\x99\x00`Iu\x06\x0b\xd6g/p\xc4\xa3\x03\xb7\xbaw&\xf1\x07\xf3\x12\xcb\x86I%\x85me\x16\xb9\xab\xfe\x10\xa7GW\n;\xc9\xa1\x0b\xde\xdc|7\x86\xd3\x9b)J\xec\xdb\x11\xd3\x95\xd7!Q\xa8L*iv\xce\x0f\x9a\xc7Y\x1f\x15\x83O\nT\xb9Z\x0bWi\x90\x11F\x19t1\xf6\x94i;\x95\x10Q\nj\xf3\r\x1e&(r\x15\x00\x97\x83\x06\x1a}\xdf2\xa6\xab\x9b>/\xc44\x1a9\xfe\xf9\x08\x15(_\x07{\xc6\x93\x1b\x90Z\x95\x1d\x14{\xd4$\xd4\x90%\'g\xb3|\x01\xc6I\x16\x12\x9c\x990\'PY\xdd%\xcaxw\x01e\xaaq3N%`\x1b\xcf\x0b\x04-O\xf5#\x07N\xd7I\x12\xde\x8a\xd8-\xb57;$\xfb\xd8\x94\x17\x8a%"\x01\x91\x8d\x86%\xa0\xd6\xbf\x15\xf9i|\x07\xfc*\x05\x10)\x89\xf5+\xa45$\x08\xcbB\x9c&\xc4uB\x18|\x13\xfa\x13\x13\xf4\\3\x00Tf\x10\xf8<O\x13p\x94\xcf*\x0e&\xbc&=\xd4o\x01X\xec\xf0\x03\x00\xb2f$G\x92\x127\xeb\xf6\'\x11\x15\xb2\xfd\x05W\xe7\x0f8_\xd16\xc9\xb5\x83,\xc6\xa5\x02\xb2\xfcs\x94\xe3\xd7\xe9\xf3\xe9\xe9\x81\xf3\xa2\xcb\tzx\xeeE7\xa0\xc7\xf9\x99V\xcc\xb1\xf8p\xd6W\xf4P\xc3\x7fW\xdc\xfb%\x89d\xfc\x97D\x1d\xcf\xc5\xf4\x8f\xd5e#q\xef\xd7\xfa\xab\xddc\xdc/\xfe/\xd0V\xdb/\xc0z\xe8GV\xf1\xe3\xe5\x11\xd9\xd8\xa7\xd4\xb9\xc2+*\xb1\xdd\x01\xc7\xa9\xf8\x03\t\xc4\xf7_\xcc?\xcc\xa3\xd4\xc5\xd3\xf3dS\xc6\xa9K\xb1\xdau\xe9\xde\xe2\xa7\xc3f\xddsL\xc9\xdf\xebl\x04\xd0/\x87V\xd6\xff\xbds\xf4\x81\xab\xf3\xcf\xe1\xb7\xc3\xd9i4\xf5\xe1W\x12\x1c\xcdW7\xe3\xdd3U\xd1\xfa5\xcd$\xefgE\x90\xda\x99\x04\xe3\xc63\x9dA\xf73\x17\x1d\xe7\x15\x0b1\xfb\'q,\xcd\xe5\xbc\xc0\xd9\xf32<\xc3\xe7n\x93\xfc9=)\xcf\xf5\x8b$\xf3\x05p\xbc\xceWuk\xd2\x054\xce\xc7\xf5\xe4\xd4\xe8k\x1f\t\xcc\x85\xe0L\xc4\xbf\xb2M\xee\x97]\xe3\xeb\xab\xf4\x82\xe9I\x92\x8c\xe1\t\xba\xb6\xc5\xf3\x94\x9e\xf0\x19\xb6\xed\xeb\xd9-\xb3\xf8Q\xe0\xcf\xe7yu3\xd8y\xf5\xe1\xd0\xe5E\x91\xd5\x0b$5\xc9e6@\xea\xe3\x08\t\xf0/\xf6`\xc5/G\xd2\xd2\xc5\xf1\x94\xca\x016\x90\xcf\xd5\xd0\xc3\xc5\xf1\xf2\xf2\xda\tkm\xed\xffq\xd2\xd3!\xc5;\xdey\xe6*\xf9E\xe5\\\xe3\x83\xd2\x7f\xd39d\x0e\xf65\xe0[\xefk&\x9b\xef\xa5j\x13\xe8#\x94\xd5\xf2\xf3:m\xd5\x1f\xd8\x14\xef\x01\x1f\r\xf6y#\xb0\xf1_\xe9\xd4\xd7+[\xaf\xd33\x0cu\xfbE9{\xf3k\xfd\x07\xd9\xe5*\x8d\xd5\x7f\x9aB\xfa\x95X\x98\xfd_\xf5\x05\xcae\x1b@\xddWY\x99\xfd\x17\x12$\xca\xe7r\xf9\xd8\xa3Fr\xd5',
 'Pexif_MaxApertureValue': (58, 16),
 'Pexif_MeteringMode': 5,
 'Pexif_Model': 'NV-GS400 ',
 'Pexif_Orientation': 1,
 'Pexif_ResolutionUnit': 2,
 'Pexif_SceneType': '\x01',
 'Pexif_SensingMethod': 2,
 'Pexif_Software': 'Ver.1.0  ',
 'Pexif_SubsecTime': '10',
 'Pexif_SubsecTimeDigitized': '10',
 'Pexif_SubsecTimeOriginal': '10',
 'Pexif_UserComment': '\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Pexif_XResolution': (72, 1),
 'Pexif_YCbCrPositioning': 2,
 'Pexif_YResolution': (72, 1),
 'Pil_jfif': 257,
 'Pil_jfif_density': (72, 72),
 'Pil_jfif_unit': 1,
 'Pil_jfif_version': (1, 1),
 'Pil_progression': 1,
 'Zexif_0x010e': '                               ',
 'Zexif_0x010f': 'Panasonic',
 'Zexif_0x0110': 'NV-GS400 ',
 'Zexif_0x0112': 1,
 'Zexif_0x011a': (72, 1),
 'Zexif_0x011b': (72, 1),
 'Zexif_0x0128': 2,
 'Zexif_0x0131': 'Ver.1.0  ',
 'Zexif_0x0132': DateTime('2007:04:15 04:00:38'),
 'Zexif_0x0213': 2,
 'Zexif_0x829a': (1, 2000),
 'Zexif_0x829d': (17, 10),
 'Zexif_0x8769': 178,
 'Zexif_0x8822': 2,
 'Zexif_0x8825': {0: (2, 2, 0, 0),
                  1: 'N',
                  2: ((62, 1), (18, 1), (34, 1)),
                  3: 'E',
                  4: ((25, 1), (44, 1), (4, 1)),
                  5: 0,
                  6: (13600, 100)},
 'Zexif_0x8827': 100,
 'Zexif_0x9000': '0220',
 'Zexif_0x9003': DateTime('2007:04:15 04:00:38'),
 'Zexif_0x9004': DateTime('2007:04:15 04:00:38'),
 'Zexif_0x9101': '\x01\x02\x03\x00',
 'Zexif_0x9102': (34, 10),
 'Zexif_0x9202': (297, 100),
 'Zexif_0x9205': (58, 16),
 'Zexif_0x9207': 5,
 'Zexif_0x9208': 0,
 'Zexif_0x9209': 16,
 'Zexif_0x920a': (580, 100),
 'Zexif_0x927c': 'SONY DSC \x00\x00\x00%\x00\x00\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x02\x10\x04\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00 \x07\x00\x01\x00\x00\x00\x00\x00\x00\x00 \xb0\x02\x00\t\x00\x00\x00\x92\x04\x00\x00@\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00A\xb0\x03\x00\x01\x00\x00\x00\x06\x00\x00\x00B\xb0\x03\x00\x01\x00\x00\x00\x01\x00\x00\x00C\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00D\xb0\x03\x00\x01\x00\x00\x00\x01\x00\x00\x00E\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00F\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00G\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00H\xb0\x08\x00\x01\x00\x00\x00\x00\x00\x00\x00I\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00J\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00K\xb0\x03\x00\x01\x00\x00\x00\x02\x00\x00\x00L\xb0\x05\x00\x01\x00\x00\x00\x9c\x04\x00\x00M\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00N\xb0\x03\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x90\x07\x00(\x01\x00\x00\xa4\x04\x00\x00\x02\x90\x07\x00\x00\x01\x00\x00\xcc\x05\x00\x00\x03\x90\x07\x00\x90\x01\x00\x00\xcc\x06\x00\x00\x04\x90\x07\x00\xb8\x01\x00\x00\\\x08\x00\x00\x05\x90\x07\x00h\x01\x00\x00\x14\n\x00\x00\x06\x90\x07\x00D\x04\x00\x00|\x0b\x00\x00\x07\x90\x07\x00\x90\x01\x00\x00\xc0\x0f\x00\x00\x08\x90\x07\x00\x90\x01\x00\x00P\x11\x00\x00\t\x90\x07\x00\x00\x01\x00\x00\xe0\x12\x00\x00\n\x90\x07\x00\xc8\x00\x00\x00\xe0\x13\x00\x00\x0b\x90\x07\x00\xc8\x00\x00\x00\xa8\x14\x00\x00\x0c\x90\x07\x00\xf4\x01\x00\x00p\x15\x00\x00\x00\xa0\x04\x00\x01\x00\x00\x00\x04\x00\x00\x80\x01\xa0\x01\x00\x01\x00\x00\x00\x07\x00\x00\x00\x00\xa1\x07\x00\x84\x00\x00\x00d\x17\x00\x00\x01\xa1\x07\x00\x94\x01\x00\x00\xe8\x17\x00\x00\x00\xa2\x07\x00T\n\x00\x00|\x19\x00\x00Standard\x00\x00\n\x00\x00\x00\n\x00\x00\x00\xe7\x10\x00p\x86p\x01\x00\x07\x00h\x00!\x05|\xea\x00yp\x00\x00 \xea\x00R\x00\x00\x00\x00\x00\x00\x00R\x00\x00\x00P\x14(\x00\x08\x00\xd8\x00\xe63\x00\x00]Q\xd8\x00\xc4\xcb.\xea\xcf\xb6\x81\x00\\\x00\x87\x00:\x00o\xd7\x97\x88\xe6\x8ao\xd7\x97\x88\xe6\x8a0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xa9[l\x01\xf6\x00\xa9\x00\xa9\x00@\x7f\xd3$G\x7f8\x96$ J\x08J\x00\x0e\x00\x0e\x00\x00\x00\x00\x00YZ\x00\x00w/\x00\x00\xe8\xea\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x80\xc2\x00\x00\x8d\xd8\xdd\x01$\'\x00J\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1b\x00\xfb\x9b\x00\x00\x00\x00\x00\x00\x00\x00\xd8V\x8a\x00\x08\x00\x08\x00\xa5\x8a\x00\x00i\x00\xeap\xeap\xeap\xcdp\xcdpi\x00\x02p\x9b\x8a\x00\x00\x00\x00\x00\x00\x00\x00\xacp"\x8a\xff\x00\x00\x00\x00\x00\x08\x00\x00\x00\x00\x00i\x00Q\x00\x00\xd8}\x00\x00}\xec\x01\x01\x00}\x00\x8a\x00\x01\x00\x00\x00\x00\x00\xe7\x00\x00\x00J\x00\xff\x01\xd3\x01\x00\x00\xd1\x00\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd1\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x01\xd3\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xbd\x00\x01&\x01\xf4\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\x00\x01\xa5}ilO\x00\x95\xa0\xac}\x0b\xa0\xd8\x00olp}\xf1\x8d!\x17TE\x13S\x00\xe0\x078\x00\xe0\x078\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xc51\xef2\xf0O\x1aRp\xd7\x05\xe0\xd5/\x1b\xd8\xbb#Z\x8a\x1b^\xeduB\x88\x08\xd8B\xb7\x98j\xddKG\x00\x8d\xf0\x8d\x13\xaf\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\x00\x00\xd0\xbbW\xe7\xfd\xd7\xf9\xe7\x82pS\x8a_\x8aRp)\xff~\xff\x00\x00\xfc\xfe\xcc<p\xcdPs\x83\x05!\x88}\x01\x93\x00o\x00\x13\x00\xbd\xbd\xbd\xbd\xbd\xbd\xf2\x07\xfe\xe8\x17\xef\xe7D\x9eL\xdc\x18B\x1a\x9e\x10\xad\xef\xf3\x102V\x9b*]m\xb4\xd4\xcf\x84(\xf0R\xe8\xc2$\x80\'=\x1a\xaf10\xf0u\xbc$\x04\xf9z\x11\x9f!\x06\x8f\x00\xdc\x1a\x91s\x0c\x1a\x8ds\x02\xe2wF:y\x07\xa9\xdc\x1a\x91sD\xc3J\x01\x17\xef\xe7D\x00\x01\x17\xef\xe7D\x00\x00\x00\x00\x01\x00\x08\x01+\xb6\x00\x00\x00\x00\x95V$\xcd@\xe2\xbcG\x8a\xf3I\xc4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00)\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xdbJ\x00\xbd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x95\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd\xbd}\x01\xc4\x1b\xc4\x1b\xc4\x1b\xc4\x1b3\x1b3\x1b3\x1b3\x1b\xc4\x1bo\xea\xfd\xea\x92\x00d\x01\x8c\xd4(\x00\x10\xe5(\x00\x18\x15(\x00I\x15(\x00I\xff\xff\xffI\xff\xff\xffI\xff\xff\xffI\xff\xff\xff\x8a\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x01\x01\x00\x00\x01\x01\x00\x00\x01\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x08\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\xd1\x00\x00\x00\x00\x00\x1b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1b\xe2yyy\x87jjy`u`\x1a\x079`b\xa9by\x87\x87u\xa9J\xa9G\x87\x87`b\xa9b\x02L\x87y`u`\x1a\x079\xe2yyy\x07jjV\xcf^\xb1\x83\xcf5W\xba\x15\xcd\x9aa\xa1u\x13\x8a\xb1\xebAT\x835_4%\x7f\xa1\xd9\x13Y2]a\xebA\x9a\xd4\x82\xa1\xfc\xc8p\xd0\xa42\xee\xee\x03\xcf\x122\xb1\xe4q\xc8\x1b\xd0\xbd2\x94e2Ep\xb1.m3\xc8N\x13\x9a2oe8E\x07\x1f\x1f\xa1L\xa1\xe2tH2#eNE\xeb\xda\xe1\xa1;\xc8\xfdl\x9c\xa0\xe3\xd3\x82\xd3&\x95.\xd3&(4l\xb2\xa0\x18\x95\x94/5/\x1d\xd3\x00\x00\x12l\x94\xa0\xf1Q\xb9/5]\x8a]\x00\x00^\xa0\xc7\xa0/Q\xdf\\\xe0L\xb1Q\x00\x00\xcalz\xa0\xdc\xd3$\xe2\x90\xf0xj@\xeccl\xc7\xa02/\n\x02\xe8LW\x87\x00\x00\xbcl\xb0\xa0\xa3\x95\xf6]7\xa0\x00\x00\x00\x00\xbc\x95\xd7\x95\xda\x0c\xe0\xbbl\xec\xf9\x95\'\x870\x95\x82\xa0\xd0\xec\xfb\xbb5(\xe5Q\x00\x00\xfe\xa0\x13\xa00\x0cv\x81\x9b(X9\x00\x00V\x95w\xa0u\xec1\x0cGl9j\x00\x00\x83\x95\x08\xa0\xd9\xec\xdb\xec<\xa0\x89]K\x87\x88Q\xf7lXl\x8d\x0c(Q\x1e\xa0\x00\x00\xa6Qyl<l\xf4\x92$l\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00G\x00\xbe\x1b\xea}\x00\x00\x00\x00\x02\x00p\x00\x82\x00~@\xea}\x00\x00\x00\x00\x00\x00\x00\x00`\x00"@\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00?\x1b\xea}\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd3\x08\xea}\x00\x00\x00\x00\x00\x00\x00\x00=\x01\xea}\xea}\x00\x00\x00\x00\x00\x00\x00\x00\r\x01\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}\xea}@}\xea}\xea}\xea}\x00\x00\x00\x00\x00\x00\x1b\x00\xef\x007\x00\xd2\x01G\x089\x1b\x1d\x1b2@\xe2}|\xd8\xd9^\xf6\xe7H\x04\xe6\x04"\x04x\xe7\xe5\x0e0\x0eV\xe7\x8f\x04GV\xf7\x0e\x8a@e\x08.\x01w\x01\xd2\x01\x86\x01\xdd\x01\xcc\x01\xa4\x01U\x01\x98\x01_\x01t\x01\xa1\x01\x9f\x01\xc4\x01\xa9\x01\x14\x01\xdb\x01A\x01\x9c\x00\x9e\x00I\x00=\x00\xbd\x00\xc3\x00_\x00D\x00t\x00\x96\x00E\x00N\x00\x13\x00\xc2\x00\xdb\x00\xe9\x00\x9c\x00O\x01\x91\x7f\xaci@\x00\x88\x00\x8e\x00\x00\x00\x95\x00$\x00p$\xff\xff\x00\x00\x01\xf3\x00\x00\x00\x83\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x0c\x00\xea\x00\x01\x00\x00\x00\x01\x00\x00\x00\xe7\x00\xd3\x000\x00\x00\x00\x00\x00\x01\x00\x01\x00\x8a\x00l\x000\x00\x00\x00\x00\x00\x00\x00\x01\x00\x1b\x00V\x00p\x00\x00\x00\x00\x00\x01\x00\x01\x00\x01\x00^\x00p\x00V\x00\x00\x00\x01\x00\x1b\x00\x01\x00V\x00\xb6\x00\x00\x00\x00\x00\x08\x00\x1b\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x1b\x006\xd8\x8a\x0ew\xe7\xb4\xe7G\x0e\x9c\xd8\xcb\x0e\xc0\xe7\xfa\x04\xbaV|\xe7\x8f\x0e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe1@\xd2\xd8\xd2\x0eE\x0e\xb1\xd84@\x94}\x8b^\xbe\xe7\r\x0e\xae^ \xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x810I(\xcd/\xc1//l\xb7\xd7\xacp/pip\x08p\x9b\x8a"\x8a\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00}\x00\x00\x00\xc7\xae\xf2\x00\xad6f\x01%Mf\x01q\x87\xfb\x00%1\x08\x00u\'(\x00\x10eG\x00I\x93\xba\x00~\xedQ\x00\x04\xca\xc2\x00\x9fR-\x00\xa2p-\x00\x16\xcc+\x00\x96\xfb\xe2\x00\xaa=\x93\x00\xae\x0e\xd0\x00\xa1&\xe2\x00\xaco\x1a\x00\xee*K\x00\x87>K\x00\xbe1\xc5\x00,Q\x00\x00\xca\x01\x08\x00\x9bZ\x01\x00\n\xe3\x01\x00\xb4\xca\x00\x00\'\xf8\x00\x00\xb4\xca\x00\x00\xb4\xca\x00\x00\xe5\x1b\xe5\x1b\xe5\x1b\xe5\x1b\xa6pr\xb6|pk\x8aq\xb6\xa7\x8a"\x88\x82\xcd\x9fp\xf7p2\x05\xba0\xa7\x05Q0fp\xa5\xb6\xd7\xb6\xfb\x8a~ \xda\x8az\x92#\x05\xef\xd7\xdb\x8a\x99\x8a\xe0\x88U\x05\xe20&p1\xb6\xc5\xb6}p\x13\x88X\x8aI\x81\xea\x8a\xf4\xbe\xa6\x8a\xaf\x8a\xe4 N\x05\x8604pt\xb6\xd8\xb6ap\x95i\xf7\x05\xc100\x05\x06 \r\x8aT\x8a\xce\x881\x05\xcf0gp\xe0\xb6y\xb6\xacp=i\xc9\xcd\xc5\x883\xcdh\x88\xdc\x8aN\x8a\t\xd7Z\x05\xd00\xf2pL\xb6\xe0\xb6Xp\xd9\xb67\x05?\x88\xe8\xcd\x91i\xcf\x8a\x9f\xb6\xf3\xb6\x98\x05\xa90\np\xf6\xb6\x04\xb6\xc1p\xcbi \x05\xb3i\xe7\xcd\xe5\xb6\x94\x05\x07\xb6\xee\xb6\xf1\x05W0\xe7V\xd8\x0e\x92H\xa1\x04V^^iU\xb8\x04V\xd8^\x05\xe0=\x04V\x0e\x04\x8a\x96\xa2\x04V\xb6\xb6\x0c\x9a\x1c\xe7V0\xd7\x071\xe1\x0eVV\xb6\xa0bI\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x08\xdc8\x01\x00\x00\x00\x00\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00$y$y\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x97f\x00\x00y\x00\x95\x0ep\xb60^\xbe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x97f\x00\x00\\\x01\x00p\x00\x00\x00\x12\x03f\xc4\x00\x00\x00tX\xf3\x95\x02`\x95\x02$$+\x00\x00\x00\x00\x00y\xba\x91\xac\xe0\x91\xe0\xba\x00\x00\x00\x00\xc3)\x00p\x00\'\xda)\x00\x00\x00\x00J\x01"\x8b\x85\xc3\xb7\x95\xdb)\x00\x95\xe4)\x00Q\xa0\x8b\x00\xd3\xa0\x8b\x00\xb6\x94\x8b\x00/\x94\x8b\x00$\xc2\x9e\x00\xc4b\x9e\x00\xe3b\x9e\x00y\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcd\x99\x8c\x01\x10a\x1b\x01\x13\xb4\\1\xa7w\xda\x1a\xaag\xcd\x11\xd0\xaa\x0b\r\xff\x8c\x8c\x1ft\xcc\x8f\x0f@gJ3$\x01\xc9\x0e\x8c\x82\xd6\x16\xdf\xf9jm\xec\xc93\x03\xc9"\xca*\t\x98=\x15:N\x06;\x83\x07\x91"\n\xbe\xd4:,\xff\xdb%Vy\xe87\xb3i1,\x83\nT\x18\x13\xac\xb5\'2\xe5a\t\x82qjp@\x17\x9eMx\x11\xca_\x9d\xb6(wV\x07\x04\x08\x05\x08\x1a\t\x88N\x96&\x93\xea\xdf\x8d\xaf\x92y\x00\x1dU\x1e\x08\n\xa7\x1a\x01\x13t\xce8\r\xbd\xdb\x1a\xaa\xf7_\x18\nl\n\r\xff\x1c\x1e\x16\x0e\n\x8e\x0f f\xca;\x05\x10\xc9\x0e\xec\x82v\x1f_\xf8jm\x8c\xc9\xa3\nM#\xca*c\x98\xad\x1c\xbeE\x06;\xe2\x07\x01+\xf7\xbc\xd4:L\xffK<qz\xe87\xd3i\xa1%\xe3\tT\x18s\xac%.\xf2\xe6a\t\xe2q\x98\x18?M\xc2\x17AI\r\x03H\xe6z%gV\x94\x01yL\x1a\x00,N\x03 \xc3\xce\xef$\xcf\x92\xe9\x01\xf1\xab\\/R\x9b\xe8.\\f\xdf\xee[\xf48\x19\xafWl\x01SG\xec>\xaa<>\x01=xn)\x87D\xc8\x1f\x1e`n\xe1\xe4\xa4!\x12\xb0I|-\xa6\xaa\xed\x0cjY\x8c\x18\x9f\xf6B\x01\x96\xe6[3\xef6\xc8(\xdb\x83\x19\x0f/\xfd\xaf"\xdc\xb0\xd5-x\t3\'\x14\xc2\xd8\x06\x9aQ9\x18k\xe3\xab=\x92\xcf\x01\x02\xbb\xebm\'\xee\x9d\xf0"OBl>0\xc0!\x1e\xfb\x8d<\x1a\x96 \x1c1\xa8b\xf5 E\x03\xb92\x01c\x97\xc9\x88\xfc\xac,\xbd\xd8M\x0c\xdd\xe6O\x08\xa4\xeey \x13\x04N!q\xd7\x98\t3\xf6\xb1\xf5\xf9\xb8\xc6\xcf\x15v\xe7r\xe7r\xbcV\xb5\xdd\x9c\xfb/\xadcQUB\xce\xfd\x95\xe5\xb2=\x97x(\xd5a\x03_\x82\xb9\x04Q\xc5\x99J\xa8\x865\x86\xe0\xf3e\xac0\x9f\x81 \xec\x9e\xb9!9\xc3\x7f\xb8\xe7\x04\x95\xc4\xb4\xe8\x95\xdd\xc9\xe5\x018\xcc\xf1Q\x82l>a\x87\xd4\xe7\x91!a\x88\xcb"\xe0\ty04\xe1\x99\x07M\xcfs\xd2\xc5\x95\x95\x13X\x8a\xfdT7\x08\x8a\xab\x00\x01\xc3x\xce8\r\xbd\xdb\x1a\n\xfb_\x18jl\n\r_\x10\x1e\x16N\x08n\x0e@fZ2\xa5\x13)\x0fW.\x08\x0e\x08\xf5pm\xf5\xc9\xa3\n\x8d"Z#\x03\x98=\x15\xdeG\x962\x87\x07\x9d"\xce\xb7\xd4:K\xf9\xdb5qy\xe87\x93i\xa1%\x89\nD\x18\x13\xac\xb5\'2\xe5a\t2}\x08\x11\x0fIR\x1e\x11I\x87\n\xa2\xe2\xea,wV\x94\x01\x99H\x1a\x00<N\x03 \x83\xcb\xe5$\xaf\x92y\x08Q\xae\\/2\x9bx\',`O\xe7;\xf4\xb8\x10\xcfQl\x013G|7j:>\x01=xn) C\xd2\x1f\x11`n\xe1\x14\xa3\xb1\x1b\xd0I\xec$6\xad}\x05\x0fY\x16\x11\x13\xf2B\x01\xf6\xe6\xcb:O>\xc8(\xcb\x83\x19\x0f%\xf4\xbf"\xbc\xb0E$\x18\x003\'\x14\xc2H\x0f:X\xa9\x11\xe8\xe3!4\xc3\xc5\x91\x0b[\xebm\'n\x97\xf0"\xafBl>\xd5\xca;\x1e\x02\x84\xac\x13\xd6+\x1c1\xc8be)e\x08);oc\'\xc0\x88\xf0\xac,\xdd\xd8\xdd\x05}\xeaO\x08\x94\xeey \xea\tT!\x0e\xd7\x98\t\xad\xc0\x83\x04\xb6\xf5^1\xf4u\x11\x00\xc2#\xa3/b\xa2\x82\x00\xbe\xec\xb35(\xe3i4d0\xb1\x16g\x83\x8b\x08\xed\xc0\x172\x99\x88\xae J\xd1\xb8\x03\x1c\x83\xe1\x02M<Y\x11\xb60h\x1b\xf9Kg\x0cM\\Z0\x8b\xfcG\'\xb6\xcd\x9c\x1f\x94D)7v\\,/v\x15\xef\x12\xca9\xdb2Jl\xaf\x1f\xa1i\x85\x17\xaew]\x1c\xcc\xd5j\x1b\x1fSO\x11\x04\xaf\x83\x10N\xdc\xb9\x0e\xbe\x03\xf2\x0f;\x0e\xa7\t<\xca\x180G\xfd8,E.\x97\x03\xae/z#\xf8>\xd7\x03\x02\x1af\x0f4\xb2S!)D\xa57\x8eY>:\x93e\xb3/\x1ct\xf5\x11\xb9S\x89"J\xfb\x8a)\x10\x16\xbd\x17\x0e\xf3\xb4)\xef\x9ci\x1b\xc5\xc3\x16\xe4\xe9{O\x10\x91\xce\r&>a\x0b+\x95A\xc0\x10\x88,4\x1a\x93\x9dA5\xa2\xb2l\x03*\xf7p3"By\x11g\xf9\x0b,<F3\x11F\xadL8\xb6\xc8s-bG\xe80C\t\x8b\x0c_1t3&\xee\x95,\xee%\xee&\x1e\xfb\xe8-\xcd\x0c\xe3,y\xe4e0{"\xdd\x17\x0e\xbd`\x1d[9+C\xf0W]\x0f\x0ft\x8b\x18H\xfb\xb0\r\x91\x9d95H\r\xb7(\xb6\x98\x12&\xff\xb6\x18\x14DL\x87$\n\x92\x81\x02\xab!\xf8!\xca0\x813[\xf8\xe7\x10\x18\x05\x89\x14Y\x93\xc9\x0bf\xc1g%g\x96C4\x8d\x03\x0c\x1b\xb0\xd3\xb61Z\r\xf9\x1a\xdf\xd7]\x1c\xbd\x97r,b-\xc8\x1f\xe0\x99\xdd\x1c8\xf5\x92 \xd3\x1f\x8e\x00q\xddg\x11\xed\xff\xf9\x1b\xcfB\x8e$\xb6\xe6\x8d\x08\x8c\xcd\xeb!+\xd5\x84\x0b\xf9\xf6\xf2.\xf7<\x1c*c?\xa3\x02S\xdb\xc9(#46\t\xb7\x94\xb8 I\xec=3d4y\x06/v\xba)\xa1\xf7e\x19\x8d<\x17\x1cl\xa0\xe3\x16\xc7\xf2\xa5\x17I\xb0o\x05\xde\xb5y(\x10\x97\x06\x1f\x9e\xf8\x92\x05K\x06\xd0\x1dQ\x1e\x91\x1a\xd2\x0c\xc3\x103\xa8\xd7\x0b\xa5<O\x0f\xc2\xbb\x7f0\nb\x8c6I(o#\xda\x01*)\xd5\xbb}\x01N\xc6\xdf\x13q\xd9z\x1e\x9a\x04z\x05\xc4\x94\xe5\x15`\x17\x9e\x17\xab\xe5\xf1\x1d\xb4*b\x071\xf4c\x17\x89\'\x9f5\x11\xde}80\xaad:\xee\x0f\x04\rj\xfe\xe2\x1d\x0f\x0b\x99\x06u\xcdV\x1a\xb9\xeb\xf7\x07\xb9\xca\x94\x0c\x07\xf3\xba f\xe5\xe8\x19n\xa8h.\xbe\\\x11\x1d\x10\xc1\xa9(\xa7\xa5\xe2#~B|\x1aa\x80\xae\x10\xc5\xd6=,\xc4\xcd\x8b*\x03\x0c\xd4*\xb6\xdeT\t\xdcV\x9a\x05\xcf\xdbK/\x8e\xe6O\x00\\\xd9-\x0f\x1a\x84E*i\x0c\xe4=\xf6$\xe5\x01\x8bw\xf7\x0b\x9c\xec\xd6\x12+l\xc0\x013\xb1\xeb\x0e.&\xa1\x12j\x84i.\x83\xcc\xfa\x17a\xb98*0f\x8c*\xab\x0b18\x1eB##t\x97\xfd1&\xebS?\x90/<\x17\x0e\xeec\r\x1fV\xc8\x1d\x89I \x1a\xcd\xa9\x8f#\xa6\xa6\xcc\r\x80\xdb\xff\t\xa9\xfc"\x01\xc8E5\x1d\x9dR\x9c#\xea\xf2(\x1c\x10h\x8a\x1b\xfc\x07 >;\x83\xa3\x15\xd2p#/\x93\xb7L,\x16&k\x12\xaf\x0c\xd4!\x07\x0f//\xdfn\xa7\x17\x8a\x88\xc0\x1d\xbbv\xdf\x18k\x02(\x07\xcc\xd4g%)\x7fg-\x10\xba\x94\x1aO\x00\xb9\x08\xb3w\x9f\x0e)\xfcB\x00\x1c\x9d?\x05\x91\xa4>\x0f\xec\xd9\x8a#\x96\x87f\x0cj\xb9-\x07\xcc\xb5\xc6(,\xb3\xaa%\x93\xa163\xa1+Q&x5\x0f$\x87\x96\x939\x03\xe6\x08"w\xe4;\x15\x982M\x07^A\xca6\xe1\xb0\n\x16\x08\x0478\x13$\xe9\x03$\xaf\x11\x0bc\x17\x14&\x9d\xf3\xa3\x0f\xe1_\xd3!\x18\x97!1\xa9G\xca#F\xf5\'%hF\x150\xf4\x1f\xbc6\xe8`m\x07a\x9d\xa0\x04\x82Z\xe4,\xa3\xc1\xef0\x93\xcf\xa0\x1f\xafy\xd9\x15\xdd\x91\xbe)By6\x1f\x03e\x93\x01\xd3\x10\xed\x06\xb8:\xad\x0cc\xbfO\t\x81\x9e\x991D6\xa70\xd0\x93\xf3!\xe2q\xe9\x0f\x19\xf5\xb5\x0b\x1b\x08\x9b\x1f\x91\xd8\xfe8\x02\xbb\xa9!\xe2\xf8\x1f*\xbf\xd6\x8a,[X\xf70\xac F2\xe8S\'\x13N3#7b\xe4;\x1c\xdb\x02d\x02_\x98\x1922\xf5o,\x0fO\\\x03\xff\xd8\xf4+\xd1D\x8e\x05\x96\xa4I(\x04xm\x11\xb0\xe1V22W20=\xc9\x18/\xd1\x04\xa0\x17Ao\x1f\x12\x9e\xec\x15\x136\xedf#\xb7nq\x05\x1c\xbf\xfb2\x96\x0b\x87\x07\xa4\x10i\x1a\x96 \x8f\x04Z\xb0h/\x06u\x871\xe3\xf6\xda/\x06\xb3u\x1bvEv.\xe8&\xb3.PW\xa5\x03e\xd7_3\x89W\xa3 ]P\x0c\x0e\xce\xb84\x0b^I\x82 c\xb7t\x1f-\xe4\xd1\x17E\x92&%W~\x96\x076\xc7\x9e"E\xee\xe1&r\x17Q\x1dX\x1b\x89>v\xa9R:\x9d) )\x8c|\x8a\x07\xccU\xca\x028\x07\xde\x07Cd\x14#k\xe2#\x0bl\x08\xc4\x0c|\xbb9\x02\x82yl8\x0b\x0eB-\xa1\x1b\x113\xdf\xc6\x8e\'!\xe3\x86/\xca+\x17\x03p\x83\x96\x1d\xfc\xe0\xa4\x12\xc5\x08\x9e\x14j\x80@\x12\x8fbH\x13uO\x00\x16\xc4\x80\x02.\x02+G7\x88\x99\xd86\xafwt3\x03\x97\xe8\x05E\x86\x85\x19\x05\x19\xc5\x06V\xe5\xc7\x04\x8d\x88b\x0eUV\xd25\xb0\xb67&\xbaWM\x05bg\x07$\xbe\xe3\xf1\x12if\x9c0\n\xaf\xf8\tr\xdd^2\xdc\x8e\xee\x10\xf1\xc1\xc9\x03\x15L\x0e\x1c\x01\x122#\x8bV\x8f(z\xf5\xca\x05\xc2\xca\x86$\x00\x02\xb7\x18\x90\xe4\xd5$\xdb\xf06$\x9e>\x88\x18\xeb\xf3S/i\xba\x013\xe4\xad\xb9\x1b\xdf\x08\xaa*\xb5\x87^\x15\xb8\xc4\xd0\x08\x89\xf8:\x14\x9b\xd2B*Z\xdf\x8d\x16\xfe;\x03#\x97j\xcc\x10;\xb5\x01\x195\xc2_\x14\xb0Jp7K\xbcz3\x813x\x1c\xb1\xf4\xa5\x02\xd5\xc4a\x19\xe5\xd0S(O\x12p\x02&\xf3\xf4$\xbfoK4N\xba\x98#W\x94 >\xcfV\x17\'\xdaq\xf1*\x93k\x0b6|~X\x03\x8d\x9c\xe8!\xb0\xcb\x975|\x85S/\x9c\x95\xaa\x1d\x01q\x92,w\xa7\xec\'\xefN\x06\'\xde\x9d\x11#U\x8cb!r6\xf2#\xefe\xfd\x04q3\x1f\x0e\x97\xef;\x11\x88\xa10\x17C\xae?\x1d\x03\xc5\xee/v^$\x1f\xf8&3\x18\xdeU\x86-JH\x8c.,\xca\xa250G\x04\x1e{) ;w\x81~\x15\xa0K}\x0b[\xef\xa9#\x92\x83_-u^\x99\x00`Iu\x06\x0b\xd6g/p\xc4\xa3\x03\xb7\xbaw&\xf1\x07\xf3\x12\xcb\x86I%\x85me\x16\xb9\xab\xfe\x10\xa7GW\n;\xc9\xa1\x0b\xde\xdc|7\x86\xd3\x9b)J\xec\xdb\x11\xd3\x95\xd7!Q\xa8L*iv\xce\x0f\x9a\xc7Y\x1f\x15\x83O\nT\xb9Z\x0bWi\x90\x11F\x19t1\xf6\x94i;\x95\x10Q\nj\xf3\r\x1e&(r\x15\x00\x97\x83\x06\x1a}\xdf2\xa6\xab\x9b>/\xc44\x1a9\xfe\xf9\x08\x15(_\x07{\xc6\x93\x1b\x90Z\x95\x1d\x14{\xd4$\xd4\x90%\'g\xb3|\x01\xc6I\x16\x12\x9c\x990\'PY\xdd%\xcaxw\x01e\xaaq3N%`\x1b\xcf\x0b\x04-O\xf5#\x07N\xd7I\x12\xde\x8a\xd8-\xb57;$\xfb\xd8\x94\x17\x8a%"\x01\x91\x8d\x86%\xa0\xd6\xbf\x15\xf9i|\x07\xfc*\x05\x10)\x89\xf5+\xa45$\x08\xcbB\x9c&\xc4uB\x18|\x13\xfa\x13\x13\xf4\\3\x00Tf\x10\xf8<O\x13p\x94\xcf*\x0e&\xbc&=\xd4o\x01X\xec\xf0\x03\x00\xb2f$G\x92\x127\xeb\xf6\'\x11\x15\xb2\xfd\x05W\xe7\x0f8_\xd16\xc9\xb5\x83,\xc6\xa5\x02\xb2\xfcs\x94\xe3\xd7\xe9\xf3\xe9\xe9\x81\xf3\xa2\xcb\tzx\xeeE7\xa0\xc7\xf9\x99V\xcc\xb1\xf8p\xd6W\xf4P\xc3\x7fW\xdc\xfb%\x89d\xfc\x97D\x1d\xcf\xc5\xf4\x8f\xd5e#q\xef\xd7\xfa\xab\xddc\xdc/\xfe/\xd0V\xdb/\xc0z\xe8GV\xf1\xe3\xe5\x11\xd9\xd8\xa7\xd4\xb9\xc2+*\xb1\xdd\x01\xc7\xa9\xf8\x03\t\xc4\xf7_\xcc?\xcc\xa3\xd4\xc5\xd3\xf3dS\xc6\xa9K\xb1\xdau\xe9\xde\xe2\xa7\xc3f\xddsL\xc9\xdf\xebl\x04\xd0/\x87V\xd6\xff\xbds\xf4\x81\xab\xf3\xcf\xe1\xb7\xc3\xd9i4\xf5\xe1W\x12\x1c\xcdW7\xe3\xdd3U\xd1\xfa5\xcd$\xefgE\x90\xda\x99\x04\xe3\xc63\x9dA\xf73\x17\x1d\xe7\x15\x0b1\xfb\'q,\xcd\xe5\xbc\xc0\xd9\xf32<\xc3\xe7n\x93\xfc9=)\xcf\xf5\x8b$\xf3\x05p\xbc\xceWuk\xd2\x054\xce\xc7\xf5\xe4\xd4\xe8k\x1f\t\xcc\x85\xe0L\xc4\xbf\xb2M\xee\x97]\xe3\xeb\xab\xf4\x82\xe9I\x92\x8c\xe1\t\xba\xb6\xc5\xf3\x94\x9e\xf0\x19\xb6\xed\xeb\xd9-\xb3\xf8Q\xe0\xcf\xe7yu3\xd8y\xf5\xe1\xd0\xe5E\x91\xd5\x0b$5\xc9e6@\xea\xe3\x08\t\xf0/\xf6`\xc5/G\xd2\xd2\xc5\xf1\x94\xca\x016\x90\xcf\xd5\xd0\xc3\xc5\xf1\xf2\xf2\xda\tkm\xed\xffq\xd2\xd3!\xc5;\xdey\xe6*\xf9E\xe5\\\xe3\x83\xd2\x7f\xd39d\x0e\xf65\xe0[\xefk&\x9b\xef\xa5j\x13\xe8#\x94\xd5\xf2\xf3:m\xd5\x1f\xd8\x14\xef\x01\x1f\r\xf6y#\xb0\xf1_\xe9\xd4\xd7+[\xaf\xd33\x0cu\xfbE9{\xf3k\xfd\x07\xd9\xe5*\x8d\xd5\x7f\x9aB\xfa\x95X\x98\xfd_\xf5\x05\xcae\x1b@\xddWY\x99\xfd\x17\x12$\xca\xe7r\xf9\xd8\xa3Fr\xd5',
 'Zexif_0x9286': '\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'Zexif_0x9290': '10',
 'Zexif_0x9291': '10',
 'Zexif_0x9292': '10',
 'Zexif_0xa000': '0100',
 'Zexif_0xa001': 1,
 'Zexif_0xa002': 2288,
 'Zexif_0xa003': 1728,
 'Zexif_0xa005': 440,
 'Zexif_0xa20e': (3648000, 287),
 'Zexif_0xa20f': (2736000, 215),
 'Zexif_0xa210': 2,
 'Zexif_0xa215': (100, 1),
 'Zexif_0xa217': 2,
 'Zexif_0xa300': '\x03',
 'Zexif_0xa301': '\x01',
 'Zexif_0xa302': '\x00\x02\x00\x02\x01\x02\x00\x01',
 'Zexif_0xa401': 0,
 'Zexif_0xa402': 1,
 'Zexif_0xa403': 0,
 'Zexif_0xa404': (0, 10),
 'Zexif_0xa405': 35,
 'Zexif_0xa406': 0,
 'Zexif_0xa407': 1,
 'Zexif_0xa408': 0,
 'Zexif_0xa409': 0,
 'Zexif_0xa40a': 0,
 'Zexif_0xa40c': 0,
 'Zexif_0xc4a5': 'PrintIM\x000300\x00\x00\x02\x00\x02\x00\x01\x00\x00\x00\x01\x01\x01\x00\x00\x00',
 'Zexif_0xea1c': '\x1c\xea\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00',
 'aspect': None,
 'compression': 'none',
 'day': 1,
 'desktop': '/home/stani/Bureaublad',
 'dpi': 72,
 'filename': u'IMGA3012',
 'filesize': 1548851,
 'folder': u'/home/stani/sync/Afbeeldingen/20070414 new orleans',
 'folderindex': 0,
 'foldername': u'20070414 new orleans',
 'format': 'JPEG',
 'formatdescription': 'JPEG (ISO 10918)',
 'gamma': None,
 'height': 1728,
 'hour': 16,
 'index': 0,
 'interlace': None,
 'minute': 29,
 'mode': 'RGB',
 'month': 5,
 'monthname': 'May',
 'orientation': 1,
 'path': u'/home/stani/sync/Afbeeldingen/20070414 new orleans/IMGA3012.JPG',
 'root': u'/home/stani/sync/Afbeeldingen',
 'second': 27,
 'size': (2288, 1728),
 'subfolder': u'',
 'transparency': None,
 'type': u'JPG',
 'weekday': 3,
 'weekdayname': 'Thursday',
 'width': 2288,
 'year': 2008}

if __name__ == '__main__':
    main('/home/stani/sync/python/phatch/test images/test info')
