/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <string.h>
#include <errno.h>
#include <vm/vm_param.h>
#include <sys/stat.h>
#include <sys/conf.h>
#include <limits.h>
#include "utils.h"

#define MIB_SIZE 3

__thread struct xswdev* xswbuf = NULL;
__thread size_t xswbuf_len = 0;
__thread uint16_t number_swaps;
extern long pagesize;

int sa_close_swap() {
	if (xswbuf != NULL) {
		free(xswbuf);
		xswbuf = NULL;
		xswbuf_len = 0;
	}

	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	int ret = sa_reset_swaps();
	if (ret != 0)
		return ret;

	*number = number_swaps;
	return 0;
}

int sa_reset_swaps() {
	size_t len;
	do {
		// See http://gitweb.dragonflybsd.org/dragonfly.git/blobdiff/79634a6643a5f76dd3cf8995a5c054ba6ad27192..a8a3cea3c7ac1ff2cbb8bb7aeee54a784e044128:/lib/libkvm/kvm_getswapinfo.c
		if (sysctlbyname("vm.swap_info_array", NULL, &len, NULL, 0) == -1 || len == 0)
			return ENOSYS;
		if (len > xswbuf_len) {
			safe_realloc((void*) &xswbuf, &len);
			xswbuf_len = len;
			if (xswbuf == NULL)
				return ENOMEM;
		}

		errno = 0;
		if ((sysctlbyname("vm.swap_info_array", xswbuf, &len, NULL, 0) == -1 || len == 0) && errno != ENOMEM)
			return ENOSYS;
	} while (errno == ENOMEM);

	number_swaps = (uint16_t) (len / xswbuf->xsw_size);
	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	int i, j;
	i = j = 0;
	while (i < number_swaps) {
		if (xswbuf->xsw_dev != NODEV && (xswbuf->xsw_flags & SW_FREED) != 0) {
			if (index == j) {
				strlcpy(dst->name, devname(xswbuf->xsw_dev, S_IFCHR), SA_SWAP_NAME);
				dst->total = xswbuf->xsw_nblks * pagesize;
				dst->free = (xswbuf->xsw_nblks - xswbuf->xsw_used) * pagesize;
				return 0;
			}
			++j;
		}
		++i;
		xswbuf += xswbuf->xsw_size;
	}

	return ENODEV;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	int i, j;
	*written = i = j = 0;
	while (i < number_swaps) {
		if (xswbuf->xsw_dev != NODEV && (xswbuf->xsw_flags & SW_FREED) != 0) {
			if (j == dst_size)
				return ENOMEM;

			strlcpy(dst[j].name, devname(xswbuf->xsw_dev, S_IFCHR), SA_SWAP_NAME);
			dst[j].total = xswbuf->xsw_nblks * pagesize;
			dst[j].free = (xswbuf->xsw_nblks - xswbuf->xsw_used) * pagesize;
			++j;
			++(*written);
		}
		++i;
		xswbuf += xswbuf->xsw_size;
	}

	return 0;
}
