// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.psi.impl.source;

import com.intellij.lang.ASTNode;
import com.intellij.lang.Language;
import com.intellij.lang.java.JavaLanguage;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.NotNullLazyKey;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.pom.java.LanguageLevel;
import com.intellij.psi.*;
import com.intellij.psi.codeStyle.JavaCodeStyleManager;
import com.intellij.psi.impl.JavaPsiImplementationHelper;
import com.intellij.psi.impl.PsiFileEx;
import com.intellij.psi.impl.PsiImplUtil;
import com.intellij.psi.impl.java.stubs.JavaStubElementTypes;
import com.intellij.psi.impl.java.stubs.PsiJavaFileStub;
import com.intellij.psi.impl.source.resolve.ClassResolverProcessor;
import com.intellij.psi.impl.source.resolve.SymbolCollectingProcessor;
import com.intellij.psi.impl.source.tree.JavaElementType;
import com.intellij.psi.scope.*;
import com.intellij.psi.stubs.StubElement;
import com.intellij.psi.tree.IElementType;
import com.intellij.psi.util.*;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.Processor;
import com.intellij.util.containers.MostlySingularMultiMap;
import com.intellij.util.indexing.IndexingDataKeys;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

public abstract class PsiJavaFileBaseImpl extends PsiFileImpl implements PsiJavaFile {
  private static final Logger LOG = Logger.getInstance("#com.intellij.psi.impl.source.PsiJavaFileBaseImpl");
  private static final String[] IMPLICIT_IMPORTS = { CommonClassNames.DEFAULT_PACKAGE };

  private final CachedValue<MostlySingularMultiMap<String, SymbolCollectingProcessor.ResultWithContext>> myResolveCache;
  private volatile String myPackageName;

  protected PsiJavaFileBaseImpl(IElementType elementType, IElementType contentElementType, FileViewProvider viewProvider) {
    super(elementType, contentElementType, viewProvider);
    myResolveCache = CachedValuesManager.getManager(myManager.getProject()).createCachedValue(new MyCacheBuilder(this), false);
  }

  @Override
  public void subtreeChanged() {
    super.subtreeChanged();
    myPackageName = null;
  }

  @Override
  @NotNull
  public PsiClass[] getClasses() {
    final StubElement<?> stub = getGreenStub();
    if (stub != null) {
      return stub.getChildrenByType(JavaStubElementTypes.CLASS, PsiClass.ARRAY_FACTORY);
    }

    return calcTreeElement().getChildrenAsPsiElements(Constants.CLASS_BIT_SET, PsiClass.ARRAY_FACTORY);
  }

  @Override
  public PsiPackageStatement getPackageStatement() {
    ASTNode node = calcTreeElement().findChildByType(JavaElementType.PACKAGE_STATEMENT);
    return node != null ? (PsiPackageStatement)node.getPsi() : null;
  }

  @Override
  @NotNull
  public String getPackageName() {
    PsiJavaFileStub stub = (PsiJavaFileStub)getGreenStub();
    if (stub != null) {
      return stub.getPackageName();
    }

    String name = myPackageName;
    if (name == null) {
      PsiPackageStatement statement = getPackageStatement();
      myPackageName = name = statement == null ? "" : statement.getPackageName();
    }
    return name;
  }

  @Override
  public void setPackageName(final String packageName) throws IncorrectOperationException {
    if (PsiUtil.isModuleFile(this)) {
      throw new IncorrectOperationException("Cannot set package name for module declarations");
    }

    final PsiPackageStatement packageStatement = getPackageStatement();
    final PsiElementFactory factory = JavaPsiFacade.getElementFactory(getProject());
    if (packageStatement != null) {
      if (!packageName.isEmpty()) {
        final PsiJavaCodeReferenceElement reference = packageStatement.getPackageReference();
        reference.replace(factory.createReferenceFromText(packageName, packageStatement));
      }
      else {
        packageStatement.delete();
      }
    }
    else if (!packageName.isEmpty()) {
      cleanupBrokenPackageKeyword();
      PsiElement anchor = getFirstChild();
      if (PsiPackage.PACKAGE_INFO_FILE.equals(getName())) {
        // If javadoc is already present in a package-info.java file, try to position the new package statement after it,
        // so the package becomes documented.
        anchor = getImportList();
        assert anchor != null; // import list always available inside package-info.java
        final PsiElement prev = anchor.getPrevSibling();
        if (prev instanceof PsiComment) {
          final String text = prev.getText().trim();
          if (text.startsWith("/*") && !text.endsWith("*/")) {
            // close any open javadoc/comments before import list
            prev.replace(factory.createCommentFromText(text + (StringUtil.containsLineBreak(text) ? "\n*/" : " */"), prev));
          }
        }
      }
      addBefore(factory.createPackageStatement(packageName), anchor);
    }
  }

  private void cleanupBrokenPackageKeyword() {
    PsiElement child = getFirstChild();
    while (child instanceof PsiWhiteSpace || child instanceof PsiComment || child instanceof PsiErrorElement) {
      if (child instanceof PsiErrorElement && child.getFirstChild() != null && child.getFirstChild().textMatches(PsiKeyword.PACKAGE)) {
        child.delete();
        break;
      }
      child = child.getNextSibling();
    }
  }

  @Override
  public PsiImportList getImportList() {
    StubElement<?> stub = getGreenStub();
    if (stub != null) {
      PsiImportList[] nodes = stub.getChildrenByType(JavaStubElementTypes.IMPORT_LIST, PsiImportList.ARRAY_FACTORY);
      if (nodes.length == 1) return nodes[0];
      assert nodes.length == 0;
      return null;
    }

    ASTNode node = calcTreeElement().findChildByType(JavaElementType.IMPORT_LIST);
    return (PsiImportList)SourceTreeToPsiMap.treeElementToPsi(node);
  }

  @Override
  @NotNull
  public PsiElement[] getOnDemandImports(boolean includeImplicit, boolean checkIncludes) {
    PsiImportList importList = getImportList();
    if (importList == null) return EMPTY_ARRAY;

    List<PsiElement> array = new ArrayList<>();

    PsiImportStatement[] statements = importList.getImportStatements();
    for (PsiImportStatement statement : statements) {
      if (statement.isOnDemand()) {
        PsiElement resolved = statement.resolve();
        if (resolved != null) {
          array.add(resolved);
        }
      }
    }

    if (includeImplicit){
      PsiJavaCodeReferenceElement[] implicitRefs = getImplicitlyImportedPackageReferences();
      for (PsiJavaCodeReferenceElement implicitRef : implicitRefs) {
        final PsiElement resolved = implicitRef.resolve();
        if (resolved != null) {
          array.add(resolved);
        }
      }
    }

    return PsiUtilCore.toPsiElementArray(array);
  }

  @Override
  @NotNull
  public PsiClass[] getSingleClassImports(boolean checkIncludes) {
    PsiImportList importList = getImportList();
    if (importList == null) return PsiClass.EMPTY_ARRAY;

    List<PsiClass> array = new ArrayList<>();
    PsiImportStatement[] statements = importList.getImportStatements();
    for (PsiImportStatement statement : statements) {
      if (!statement.isOnDemand()) {
        PsiElement ref = statement.resolve();
        if (ref instanceof PsiClass) {
          array.add((PsiClass)ref);
        }
      }
    }
    return array.toArray(PsiClass.EMPTY_ARRAY);
  }

  @Override
  public PsiJavaCodeReferenceElement findImportReferenceTo(PsiClass aClass) {
    PsiImportList importList = getImportList();
    if (importList != null) {
      PsiImportStatement[] statements = importList.getImportStatements();
      for (PsiImportStatement statement : statements) {
        if (!statement.isOnDemand()) {
          PsiElement ref = statement.resolve();
          if (ref != null && getManager().areElementsEquivalent(ref, aClass)) {
            return statement.getImportReference();
          }
        }
      }
    }
    return null;
  }

  @Override
  @NotNull
  public String[] getImplicitlyImportedPackages() {
    return IMPLICIT_IMPORTS;
  }

  @Override
  @NotNull
  public PsiJavaCodeReferenceElement[] getImplicitlyImportedPackageReferences() {
    return PsiImplUtil.namesToPackageReferences(myManager, IMPLICIT_IMPORTS);
  }

  private static class StaticImportFilteringProcessor implements PsiScopeProcessor {
    private final PsiScopeProcessor myDelegate;
    private boolean myIsProcessingOnDemand;
    private final Collection<String> myHiddenFieldNames = new HashSet<>();
    private final Collection<String> myHiddenMethodNames = new HashSet<>();
    private final Collection<String> myHiddenTypeNames = new HashSet<>();
    private final Collection<PsiElement> myCollectedElements = new HashSet<>();

    StaticImportFilteringProcessor(final PsiScopeProcessor delegate) {
      myDelegate = delegate;
    }

    @Override
    public <T> T getHint(@NotNull final Key<T> hintKey) {
      return myDelegate.getHint(hintKey);
    }

    @Override
    public void handleEvent(@NotNull final Event event, final Object associated) {
      if (JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT.equals(event) && associated instanceof PsiImportStaticStatement) {
        final PsiImportStaticStatement importStaticStatement = (PsiImportStaticStatement)associated;
        myIsProcessingOnDemand = importStaticStatement.isOnDemand();
      }
      myDelegate.handleEvent(event, associated);
    }

    /**
     * JLS 6.4 Shadowing and Obscuring
     * A single-static-import declaration d in a compilation unit c of package p that imports a field named n shadows the declaration of any
     * static field named n imported by a static-import-on-demand declaration in c, throughout c.
     *
     * A single-static-import declaration d in a compilation unit c of package p that imports a method named n with signature s shadows the
     * declaration of any static method named n with signature s imported by a static-import-on-demand declaration in c, throughout c.
     *
     * A single-static-import declaration d in a compilation unit c of package p that imports a type named n shadows, throughout c, the declarations of:
     * - any static type named n imported by a static-import-on-demand declaration in c;
     * - any top level type (p7.6) named n declared in another compilation unit (p7.3) of p;
     * - any type named n imported by a type-import-on-demand declaration (p7.5.2) in c.
     */
    private void registerSingleStaticImportHiding(JavaResolveResult result, String referenceName) {
      getHiddenMembers(result.getElement()).add(referenceName);
    }

    private Collection<String> getHiddenMembers(PsiElement element) {
      if (element instanceof PsiField) {
        return myHiddenFieldNames;
      }
      else {
        return element instanceof PsiClass ? myHiddenTypeNames
                                           : myHiddenMethodNames;
      }
    }

    @Override
    public boolean execute(@NotNull final PsiElement element, @NotNull final ResolveState state) {
      if (element instanceof PsiModifierListOwner && ((PsiModifierListOwner)element).hasModifierProperty(PsiModifier.STATIC)) {
        if (element instanceof PsiNamedElement && myIsProcessingOnDemand) {
          final String name = ((PsiNamedElement)element).getName();
          if (getHiddenMembers(element).contains(name)) return true;
        }
        if (myCollectedElements.add(element)) {
          return myDelegate.execute(element, state);
        }
      }
      return true;
    }
  }

  @Override
  public boolean processDeclarations(@NotNull final PsiScopeProcessor processor,
                                     @NotNull final ResolveState state,
                                     PsiElement lastParent,
                                     @NotNull PsiElement place) {
    assert isValid();

    if (processor instanceof ClassResolverProcessor &&
        isPhysical() &&
        (getUserData(PsiFileEx.BATCH_REFERENCE_PROCESSING) == Boolean.TRUE || myResolveCache.hasUpToDateValue()) &&
        !PsiUtil.isInsideJavadocComment(place)) {
      final ClassResolverProcessor hint = (ClassResolverProcessor)processor;
      String name = hint.getName(state);
      MostlySingularMultiMap<String, SymbolCollectingProcessor.ResultWithContext> cache = myResolveCache.getValue();
      MyResolveCacheProcessor cacheProcessor = new MyResolveCacheProcessor(processor, state);
      return name != null ? cache.processForKey(name, cacheProcessor) : cache.processAllValues(cacheProcessor);
    }

    return processDeclarationsNoGuess(processor, state, lastParent, place);
  }

  private boolean processDeclarationsNoGuess(PsiScopeProcessor processor, @NotNull ResolveState state, PsiElement lastParent, PsiElement place) {
    PsiImportList importList = getImportList();
    PsiImportStatement[] importStatements = importList != null ? importList.getImportStatements() : PsiImportStatement.EMPTY_ARRAY;
    PsiImportStaticStatement[] importStaticStatements = importList != null ? importList.getImportStaticStatements() : PsiImportStaticStatement.EMPTY_ARRAY;
    StaticImportFilteringProcessor staticImportProcessor = new StaticImportFilteringProcessor(processor);

    boolean shouldProcessClasses = shouldProcessClasses(processor);
    if (shouldProcessClasses && !processOwnClasses(processor, state)) return false;
    
    if (!processNamedImports(state, importStatements, importStaticStatements, staticImportProcessor)) return false;
    
    if (shouldProcessClasses) {
      if (!processCurrentPackage(processor, state, place)) return false;
      if (!processOnDemandTypeImports(processor, state, place, importStatements)) return false;
    }

    if (!processOnDemandStaticImports(state, lastParent, place, importStaticStatements, staticImportProcessor)) return false;

    return !shouldProcessClasses || processImplicitImports(processor, state, place);
  }

  private boolean processOwnClasses(PsiScopeProcessor processor, @NotNull ResolveState state) {
    processor.handleEvent(PsiScopeProcessor.Event.SET_DECLARATION_HOLDER, this);
    for (PsiClass aClass : getClasses()) {
      if (!processor.execute(aClass, state)) return false;
    }
    return true;
  }

  private static boolean shouldProcessClasses(PsiScopeProcessor processor) {
    final ElementClassHint classHint = processor.getHint(ElementClassHint.KEY);
    return classHint == null || classHint.shouldProcess(ElementClassHint.DeclarationKind.CLASS);
  }

  private static boolean processNamedImports(ResolveState state,
                                             PsiImportStatement[] importStatements,
                                             PsiImportStaticStatement[] importStaticStatements,
                                             StaticImportFilteringProcessor staticImportProcessor) {
    PsiScopeProcessor processor = staticImportProcessor.myDelegate;

    NameHint nameHint = processor.getHint(NameHint.KEY);
    String name = nameHint != null ? nameHint.getName(state) : null;

    if (shouldProcessClasses(processor) && !processNamedTypeImports(processor, state, name, importStatements)) {
      return false;
    }

    return processNamedStaticImports(state, name, importStaticStatements, staticImportProcessor);
  }

  private static boolean processNamedTypeImports(PsiScopeProcessor processor, @NotNull ResolveState state, @Nullable String nameHint, PsiImportStatement[] imports) {
    for (PsiImportStatement statement : imports) {
      if (!statement.isOnDemand()) {
        if (nameHint != null) {
          final String refText = statement.getQualifiedName();
          if (refText == null || !nameHint.equals(StringUtil.getShortName(refText))) continue;
        }

        final PsiElement resolved = statement.resolve();
        if (resolved instanceof PsiClass) {
          processor.handleEvent(JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT, statement);
          PsiClass containingClass = ((PsiClass)resolved).getContainingClass();
          ResolveState classState = containingClass != null && containingClass.hasTypeParameters() 
                                    ? state.put(PsiSubstitutor.KEY, createRawSubstitutor(containingClass)) 
                                    : state;

          if (!processor.execute(resolved, classState)) return false;
        }
      }
    }
    return true;
  }

  private boolean processCurrentPackage(PsiScopeProcessor processor, ResolveState state, PsiElement place) {
    processor.handleEvent(JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT, null);
    PsiPackage aPackage = JavaPsiFacade.getInstance(myManager.getProject()).findPackage(getPackageName());
    return aPackage == null || processPackageDeclarations(processor, state, place, aPackage);
  }

  private static boolean processOnDemandTypeImports(PsiScopeProcessor processor, ResolveState state, PsiElement place, PsiImportStatement[] imports) {
    for (PsiImportStatement statement : imports) {
      if (statement.isOnDemand()) {
        final PsiElement resolved = statement.resolve();
        if (resolved != null) {
          processor.handleEvent(JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT, statement);
          if (!processOnDemandTarget(resolved, processor, state, place)) return false;
        }
      }
    }
    return true;
  }

  private static boolean processNamedStaticImports(@NotNull ResolveState state,
                                                   @Nullable String nameHint,
                                                   @NotNull PsiImportStaticStatement[] importStaticStatements,
                                                   StaticImportFilteringProcessor staticImportProcessor) {
    for (PsiImportStaticStatement importStaticStatement : importStaticStatements) {
      if (importStaticStatement.isOnDemand()) continue;

      String referenceName = importStaticStatement.getReferenceName();
      if (nameHint != null && !nameHint.equals(referenceName)) continue;

      final PsiJavaCodeReferenceElement reference = importStaticStatement.getImportReference();
      if (reference != null) {
        final JavaResolveResult[] results = reference.multiResolve(false);
        if (results.length > 0) {
          staticImportProcessor.handleEvent(JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT, importStaticStatement);
          for (JavaResolveResult result : results) {
            staticImportProcessor.registerSingleStaticImportHiding(result, referenceName);
            PsiElement element = result.getElement();
            if (element != null && !staticImportProcessor.execute(element, state)) return false;
          }
        }
      }
    }
    return true;
  }

  private static boolean processOnDemandStaticImports(@NotNull ResolveState state,
                                                      PsiElement lastParent,
                                                      PsiElement place,
                                                      PsiImportStaticStatement[] importStaticStatements,
                                                      StaticImportFilteringProcessor staticImportProcessor) {
    for (PsiImportStaticStatement importStaticStatement : importStaticStatements) {
      if (!importStaticStatement.isOnDemand()) continue;
      final PsiClass targetElement = importStaticStatement.resolveTargetClass();
      if (targetElement != null) {
        staticImportProcessor.handleEvent(JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT, importStaticStatement);
        if (!targetElement.processDeclarations(staticImportProcessor, state, lastParent, place)) return false;
      }
    }
    return true;
  }

  private boolean processImplicitImports(PsiScopeProcessor processor, ResolveState state, PsiElement place) {
    processor.handleEvent(JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT, null);
    for (PsiJavaCodeReferenceElement aImplicitlyImported : getImplicitlyImportedPackageReferences()) {
      final PsiElement resolved = aImplicitlyImported.resolve();
      if (resolved != null) {
        if (!processOnDemandTarget(resolved, processor, state, place)) return false;
      }
    }
    return true;
  }

  private static boolean processPackageDeclarations(PsiScopeProcessor processor,
                                                    @NotNull ResolveState state,
                                                    PsiElement place,
                                                    @NotNull PsiPackage aPackage) {
    if (!aPackage.getQualifiedName().isEmpty()) {
      processor = new DelegatingScopeProcessor(processor) {
        @Nullable
        @Override
        public <T> T getHint(@NotNull Key<T> hintKey) {
          if (hintKey == ElementClassHint.KEY) {
            //noinspection unchecked
            return (T)(ElementClassHint)kind -> kind == ElementClassHint.DeclarationKind.CLASS;
          }
          return super.getHint(hintKey);
        }
      };
    }
    return aPackage.processDeclarations(processor, state, null, place);
  }

  @NotNull
  private static PsiSubstitutor createRawSubstitutor(PsiClass containingClass) {
    return JavaPsiFacade.getElementFactory(containingClass.getProject()).createRawSubstitutor(containingClass);
  }

  private static boolean processOnDemandTarget(PsiElement target, PsiScopeProcessor processor, ResolveState substitutor, PsiElement place) {
    if (target instanceof PsiPackage) {
      if (!processPackageDeclarations(processor, substitutor, place, (PsiPackage)target)) {
        return false;
      }
    }
    else if (target instanceof PsiClass) {
      PsiClass[] inners = ((PsiClass)target).getInnerClasses();
      if (((PsiClass)target).hasTypeParameters()) {
        substitutor = substitutor.put(PsiSubstitutor.KEY, createRawSubstitutor((PsiClass)target));
      }

      for (PsiClass inner : inners) {
        if (!processor.execute(inner, substitutor)) return false;
      }
    }
    else {
      LOG.error(target);
    }
    return true;
  }

  @Override
  public void accept(@NotNull PsiElementVisitor visitor){
    if (visitor instanceof JavaElementVisitor) {
      ((JavaElementVisitor)visitor).visitJavaFile(this);
    }
    else {
      visitor.visitFile(this);
    }
  }

  @Override
  @NotNull
  public Language getLanguage() {
    return JavaLanguage.INSTANCE;
  }

  @Override
  public boolean importClass(@NotNull PsiClass aClass) {
    return JavaCodeStyleManager.getInstance(getProject()).addImport(this, aClass);
  }

  private static final NotNullLazyKey<LanguageLevel, PsiJavaFileBaseImpl> LANGUAGE_LEVEL_KEY = NotNullLazyKey.create("LANGUAGE_LEVEL",
                                                                                                                     file -> file.getLanguageLevelInner());

  @Override
  @NotNull
  public LanguageLevel getLanguageLevel() {
    return LANGUAGE_LEVEL_KEY.getValue(this);
  }

  @Nullable
  @Override
  public PsiJavaModule getModuleDeclaration() {
    return null;
  }

  @Override
  public void clearCaches() {
    super.clearCaches();
    putUserData(LANGUAGE_LEVEL_KEY, null);
  }

  @Override
  public void setOriginalFile(@NotNull PsiFile originalFile) {
    super.setOriginalFile(originalFile);
    clearCaches();
  }

  private LanguageLevel getLanguageLevelInner() {
    if (myOriginalFile instanceof PsiJavaFile) {
      return ((PsiJavaFile)myOriginalFile).getLanguageLevel();
    }

    LanguageLevel forcedLanguageLevel = getUserData(PsiUtil.FILE_LANGUAGE_LEVEL_KEY);
    if (forcedLanguageLevel != null) return forcedLanguageLevel;

    VirtualFile virtualFile = getVirtualFile();
    if (virtualFile == null) virtualFile = getUserData(IndexingDataKeys.VIRTUAL_FILE);

    final Project project = getProject();
    if (virtualFile == null) {
      final PsiFile originalFile = getOriginalFile();
      if (originalFile instanceof PsiJavaFile && originalFile != this) {
        return ((PsiJavaFile)originalFile).getLanguageLevel();
      }
      return LanguageLevel.HIGHEST;
    }

    return JavaPsiImplementationHelper.getInstance(project).getEffectiveLanguageLevel(virtualFile);
  }

  private static class MyCacheBuilder implements CachedValueProvider<MostlySingularMultiMap<String, SymbolCollectingProcessor.ResultWithContext>> {
    private final PsiJavaFileBaseImpl myFile;

    MyCacheBuilder(PsiJavaFileBaseImpl file) {
      myFile = file;
    }

    @Override
    public Result<MostlySingularMultiMap<String, SymbolCollectingProcessor.ResultWithContext>> compute() {
      SymbolCollectingProcessor p = new SymbolCollectingProcessor();
      myFile.processDeclarationsNoGuess(p, ResolveState.initial(), myFile, myFile);
      MostlySingularMultiMap<String, SymbolCollectingProcessor.ResultWithContext> results = p.getResults();
      return Result.create(results, PsiModificationTracker.MODIFICATION_COUNT, myFile);
    }
  }

  private static class MyResolveCacheProcessor implements Processor<SymbolCollectingProcessor.ResultWithContext> {
    private final PsiScopeProcessor myProcessor;
    private final ResolveState myState;

    MyResolveCacheProcessor(PsiScopeProcessor processor, ResolveState state) {
      myProcessor = processor;
      myState = state;
    }

    @Override
    public boolean process(SymbolCollectingProcessor.ResultWithContext result) {
      final PsiElement context = result.getFileContext();
      myProcessor.handleEvent(JavaScopeProcessorEvent.SET_CURRENT_FILE_CONTEXT, context);
      final PsiNamedElement element = result.getElement();

      if (element instanceof PsiClass && context instanceof PsiImportStatement) {
        final PsiClass containingClass = ((PsiClass)element).getContainingClass();
        if (containingClass != null && containingClass.hasTypeParameters()) {
          return myProcessor.execute(element, myState.put(PsiSubstitutor.KEY, createRawSubstitutor(containingClass)));
        }
      }

      return myProcessor.execute(element, myState);
    }
  }
}