/*
 * Copyright (c) 2008-2009, Paul Mattes.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the names of Paul Mattes nor the names of his contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PAUL MATTES "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL PAUL MATTES BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *	unicode.c
 *		A Windows console-based 3270 Terminal Emulator
 *		EBCDIC/Unicode translation functions
 */
#include "globals.h"
#include <stdio.h>
#if !defined(_MSC_VER) /*[*/
#include <strings.h>
#endif /*]*/
#include <errno.h>
#include "3270ds.h"
#if !defined(PR3287) /*[*/
#include "appres.h"
#endif /*]*/
#include "unicodec.h"
#include "unicode_dbcsc.h"
#include "utf8c.h"
#if !defined(PR3287) /*[*/
#include "utilc.h"
#endif /*]*/

#if defined(_WIN32) /*[*/
#include <windows.h>
#endif /*]*/

#if defined(USE_ICONV) /*[*/
iconv_t i_u2mb = (iconv_t)-1;
iconv_t i_mb2u = (iconv_t)-1;
#if !defined(_LIBICONV_VERSION) || _LIBICONV_VERSION < 0x0108 /*[*/
typedef char *ici_t;		/* old iconv */
#else /*][*/
typedef const char *ici_t;	/* new iconv */
#endif /*]*/
#endif /*]*/

#define DEFAULT_CSNAME	"us"

#if defined(_WIN32) /*[*/
# if defined(WS3270) /*[*/
#  define LOCAL_CODEPAGE	appres.local_cp
# else /*[*/
#  define LOCAL_CODEPAGE	CP_ACP
# endif /*]*/
#endif /*]*/

/*
 * EBCDIC-to-Unicode translation tables.
 * Each table maps EBCDIC codes X'41' through X'FE' to UCS-2.
 * Other codes are mapped programmatically.
 */
#define UT_SIZE		190
#define UT_OFFSET	0x41

typedef struct {
    char *name;
    unsigned short code[UT_SIZE];
    const char *host_codepage;
    const char *cgcsgid;
    const char *display_charset;
} uni_t;

static uni_t uni[] = {
{ "cp037", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x005e, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005b, 0x005d, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da },  "37", "37", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp273", { 0x00a0, 0x00e2, 0x007b, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00c4, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x007e, 0x00dc, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x005b, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00f6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x00a7, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x00df, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x0040, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e4, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00a6, 0x00f2, 0x00f3, 0x00f5, 0x00fc, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x007d, 0x00f9, 0x00fa, 0x00ff, 0x00d6, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x005c, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x005d, 0x00d9, 0x00da }, "273", "273", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp275", { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00c9, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0024, 0x00c7, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00e7, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00e3, 0x003a, 0x00d5, 0x00c3, 0x0027, 0x003d, 0x0022, 0x0000, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00f5, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00e9, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005c, 0x0000, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 }, "275", "275", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1"  },
{ "cp277", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x007d, 0x00e7, 0x00f1, 0x0023, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x00a4, 0x00c5, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x0024, 0x00c7, 0x00d1, 0x00f8, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00a6, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x00c6, 0x00d8, 0x0027, 0x003d, 0x0022, 0x0040, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x007b, 0x00b8, 0x005b, 0x005d, 0x00b5, 0x00fc, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e6, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x00e5, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x007e, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "277", "277", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1"  },
{ "cp278", { 0x00a0, 0x00e2, 0x007b, 0x00e0, 0x00e1, 0x00e3, 0x007d, 0x00e7, 0x00f1, 0x00a7, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x0060, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x00a4, 0x00c5, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x0023, 0x00c0, 0x00c1, 0x00c3, 0x0024, 0x00c7, 0x00d1, 0x00f6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00e9, 0x003a, 0x00c4, 0x00d6, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x005d, 0x00b5, 0x00fc, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x005b, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e4, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00a6, 0x00f2, 0x00f3, 0x00f5, 0x00e5, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x007e, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x0040, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "278", "278", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp280", { 0x00a0, 0x00e2, 0x00e4, 0x007b, 0x00e1, 0x00e3, 0x00e5, 0x005c, 0x00f1, 0x00b0, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x005d, 0x00ea, 0x00eb, 0x007d, 0x00ed, 0x00ee, 0x00ef, 0x007e, 0x00df, 0x00e9, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00f2, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00f9, 0x003a, 0x00a3, 0x00a7, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x005b, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x00ec, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x0023, 0x00a5, 0x00b7, 0x00a9, 0x0040, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e0, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00a6, 0x00f3, 0x00f5, 0x00e8, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x0060, 0x00fa, 0x00ff, 0x00e7, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "280", "280", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp284", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00a6, 0x005b, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x005d, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x0023, 0x00f1, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x00d1, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x00a8, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005e, 0x0021, 0x00af, 0x007e, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "284", "284", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp285", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x0024, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x0021, 0x00a3, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x00af, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x005b, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005e, 0x005d, 0x007e, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "285", "285", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp297", { 0x00a0, 0x00e2, 0x00e4, 0x0040, 0x00e1, 0x00e3, 0x00e5, 0x005c, 0x00f1, 0x00b0, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x007b, 0x00ea, 0x00eb, 0x007d, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x00a7, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00f9, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00b5, 0x003a, 0x00a3, 0x00e0, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x005b, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x0060, 0x00a8, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x0023, 0x00a5, 0x00b7, 0x00a9, 0x005d, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x007e, 0x00b4, 0x00d7, 0x00e9, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x00e8, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00a6, 0x00fa, 0x00ff, 0x00e7, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "297", "297", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp424", { 0x5d0, 0x05d1, 0x05d2, 0x05d3, 0x05d4, 0x05d5, 0x05d6, 0x05d7, 0x05d8, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x05d9, 0x05da, 0x05db, 0x05dc, 0x05dd, 0x05de, 0x05df, 0x05e0, 0x05e1, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x05e2, 0x05e3, 0x05e4, 0x05e5, 0x05e6, 0x05e7, 0x05e8, 0x05e9, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x0000, 0x05ea, 0x0000, 0x0000, 0x00a0, 0x0000, 0x0000, 0x0000, 0x21d4, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x0000, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x0000, 0x0000, 0x0000, 0x0000, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0000, 0x0000, 0x0000, 0x00b8, 0x0000, 0x00a4, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00ae, 0x005e, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005b, 0x005d, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x0000, 0x0000, 0x0000, 0x0000 }, "424", "0x03ad01a8", "3270cg-8,iso10646-1,iso8859-8" },
{ "cp500", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x005b, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x005d, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "500", "500", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp870", { 0x00a0, 0x00e2, 0x00e4, 0x0163, 0x00e1, 0x0103, 0x010d, 0x00e7, 0x0107, 0x005b, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x0119, 0x00eb, 0x016f, 0x00ed, 0x00ee, 0x013e, 0x013a, 0x00df, 0x005d, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x02dd, 0x00c1, 0x0102, 0x010c, 0x00c7, 0x0106, 0x007c, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x02c7, 0x00c9, 0x0118, 0x00cb, 0x016e, 0x00cd, 0x00ce, 0x013d, 0x0139, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x02d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x015b, 0x0148, 0x0111, 0x00fd, 0x0159, 0x015f, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0142, 0x0144, 0x0161, 0x00b8, 0x02db, 0x00a4, 0x0105, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x015a, 0x0147, 0x0110, 0x00dd, 0x0158, 0x015e, 0x00b7, 0x0104, 0x017c, 0x0162, 0x017b, 0x00a7, 0x017e, 0x017a, 0x017d, 0x0179, 0x0141, 0x0143, 0x0160, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x0155, 0x00f3, 0x0151, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x011a, 0x0171, 0x00fc, 0x0165, 0x00fa, 0x011b, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x010f, 0x00d4, 0x00d6, 0x0154, 0x00d3, 0x0150, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x010e, 0x0170, 0x00dc, 0x0164, 0x00da }, "870", "0x03bf0366", "iso10646-1,iso8859-2" },
{ "cp871", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00fe, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x00c6, 0x0024, 0x002a, 0x0029, 0x003b, 0x00d6, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00f0, 0x003a, 0x0023, 0x00d0, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x0060, 0x00fd, 0x007b, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x007d, 0x00b8, 0x005d, 0x00a4, 0x00b5, 0x00f6, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x0040, 0x00dd, 0x005b, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x005c, 0x00d7, 0x00de, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x007e, 0x00f2, 0x00f3, 0x00f5, 0x00e6, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x00b4, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x005e, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "871", "871", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp875", { 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 0x0398, 0x0399, 0x005b, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f, 0x03a0, 0x03a1, 0x03a3, 0x005d, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x03a4, 0x03a5, 0x03a6, 0x03a7, 0x03a8, 0x03a9, 0x03aa, 0x03ab, 0x0000, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00a8, 0x0386, 0x0388, 0x0389, 0x2207, 0x038a, 0x038c, 0x038e, 0x038f, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x0385, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x03b7, 0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x00b4, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x03bd, 0x03be, 0x03bf, 0x03c0, 0x03c1, 0x03c3, 0x00a3, 0x03ac, 0x03ad, 0x03ae, 0x0390, 0x03af, 0x03cc, 0x03cd, 0x03b0, 0x03ce, 0x03c2, 0x03c4, 0x03c5, 0x03c6, 0x03c7, 0x03c8, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x03c9, 0x03ca, 0x03cb, 0x2018, 0x2015, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b1, 0x00bd, 0x0000, 0x00b7, 0x2019, 0x00a6, 0x005c, 0x0000, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00a7, 0x0000, 0x0000, 0x00ab, 0x00ac, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00a9, 0x0000, 0x0000, 0x00bb }, "875", "0x0464036b", "3270cg-7,iso10646-1,iso8859-7" },
{ "cp880", { 0x0000, 0x0452, 0x0453, 0x0451, 0x0000, 0x0455, 0x0456, 0x0457, 0x0458, 0x005b, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x0459, 0x045a, 0x045b, 0x045c, 0x0000, 0x045f, 0x042a, 0x2116, 0x0402, 0x005d, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x0403, 0x0401, 0x0000, 0x0405, 0x0406, 0x0407, 0x0408, 0x0409, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x040a, 0x040b, 0x040c, 0x0000, 0x0000, 0x040f, 0x044e, 0x0430, 0x0431, 0x0000, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x0446, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x0434, 0x0435, 0x0444, 0x0433, 0x0445, 0x0438, 0x0439, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f, 0x044f, 0x0000, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432, 0x044c, 0x044b, 0x0437, 0x0448, 0x044d, 0x0449, 0x0447, 0x044a, 0x042e, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413, 0x0000, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x0425, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x0000, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x041d, 0x041e, 0x041f, 0x042f, 0x0420, 0x0421, 0x005c, 0x00a4, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x0422, 0x0423, 0x0416, 0x0412, 0x042c, 0x042b, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x0417, 0x0428, 0x042d, 0x0429, 0x0427 }, "880", "0x03bf0370", "iso10646-1,koi8-r" },
#if defined(X3270_DBCS) /*[*/
{ "cp930", {
 /* 0x40 */         0xff61, 0xff62, 0xff63, 0xff64, 0xff65, 0xff66, 0xff67,
 /* 0x48 */ 0xff68, 0xff69, 0x00a3, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c,
 /* 0x50 */ 0x0026, 0xff6a, 0xff6b, 0xff6c, 0xff6d, 0xff6e, 0xff6f, 0x0000,
 /* 0x58 */ 0xff70, 0x0000, 0x0021, 0x00a5, 0x002a, 0x0029, 0x003b, 0x00ac,
 /* 0x60 */ 0x002d, 0x002f, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066,
 /* 0x68 */ 0x0067, 0x0068, 0x0000, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f,
 /* 0x70 */ 0x005b, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
 /* 0x78 */ 0x0070, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022,
 /* 0x80 */ 0x005d, 0xff67, 0xff68, 0xff73, 0xff74, 0xff75, 0xff76, 0xff77,
 /* 0x88 */ 0xff78, 0xff79, 0xff7a, 0x0071, 0xff7b, 0xff7c, 0xff7d, 0xff7e,
 /* 0x90 */ 0xff7f, 0xff80, 0xff81, 0xff82, 0xff83, 0xff84, 0xff85, 0xff86,
 /* 0x98 */ 0xff87, 0xff88, 0xff89, 0x0072, 0x0000, 0xff8a, 0xff8b, 0xff8c,
 /* 0xa0 */ 0x007e, 0x00af, 0xff8d, 0xff8e, 0xff8f, 0xff90, 0xff91, 0xff92,
 /* 0xa8 */ 0xff93, 0xff94, 0xff95, 0x0073, 0xff96, 0xff97, 0xff98, 0xff99,
 /* 0xb0 */ 0x005e, 0x00a2, 0x005c, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078,
 /* 0xb8 */ 0x0079, 0x007a, 0xff9a, 0xff9b, 0xff9c, 0xff9d, 0xff9e, 0xff9f,
 /* 0xc0 */ 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
 /* 0xc8 */ 0x0048, 0x0049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 /* 0xd0 */ 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050,
 /* 0xd8 */ 0x0051, 0x0052, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 /* 0xe0 */ 0x0024, 0x20ac, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058,
 /* 0xe8 */ 0x0059, 0x005a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
 /* 0xf0 */ 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
 /* 0xf8 */ 0x0038, 0x0039, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000
   } , "930", "0x04940122" /* 1172, 0290 */, "iso10646-1,jisx0201.1976-0" },
{ "cp935", { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00a3, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0021, 0x00a5, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x0000, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007e, 0x00af, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005e, 0x0000, 0x005c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005b, 0x005d, 0x0000, 0x0000, 0x0000, 0x0000, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0024, 0x0000, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },  "935", "0x04960344" /* 1174, 836 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp937", { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x0000, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005b, 0x005d, 0x0000, 0x0000, 0x0000, 0x0000, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005c, 0x0000, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 }, "937", "0x04970025" /* 1175, 037 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ "cp939", {
    /* 40 */         0x0000, 0xff61, 0xff62, 0xff63, 0xff64, 0xff65, 0xff66,
    /* 48 */ 0xff67, 0xff68, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c,
    /* 50 */ 0x0026, 0xff69, 0xff6a, 0xff6b, 0xff6c, 0xff6d, 0xff6e, 0xff6f,
    /* 58 */ 0xff70, 0xff71, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac,
    /* 60 */ 0x002d, 0x002f, 0xff72, 0xff73, 0xff74, 0xff75, 0xff76, 0xff77,
    /* 68 */ 0xff78, 0xff79, 0x0000, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f,
    /* 70 */ 0xff7a, 0xff7b, 0xff7c, 0xff7d, 0xff7e, 0xff7f, 0xff80, 0xff81,
    /* 78 */ 0xff82, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022,
    /* 80 */ 0x0000, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
    /* 88 */ 0x0068, 0x0069, 0xff83, 0xff84, 0xff85, 0xff86, 0xff87, 0xff88,
    /* 90 */ 0x0000, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070,
    /* 98 */ 0x0071, 0x0072, 0xff89, 0xff8a, 0xff8b, 0xff8c, 0xff8d, 0xff8e,
    /* a0 */ 0x00af, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078,
    /* a8 */ 0x0079, 0x007a, 0xff8f, 0xff90, 0xff91, 0x005b, 0xff92, 0xff93,
    /* b0 */ 0x005e, 0x00a3, 0x00a5, 0xff94, 0xff95, 0xff96, 0xff97, 0xff98,
    /* b8 */ 0xff99, 0xff9a, 0xff9b, 0xff9c, 0xff9d, 0x005d, 0xff9e, 0xff9f,
    /* c0 */ 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
    /* c8 */ 0x0048, 0x0049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* d0 */ 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050,
    /* d8 */ 0x0051, 0x0052, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* e0 */ 0x005c, 0x20ac, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058,
    /* e8 */ 0x0059, 0x005a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* f0 */ 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
    /* f8 */ 0x0038, 0x0039, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	"939", "0x04940403" /* 1172, 1027 */, "iso10646-1,jisx0201.1976-0" },
#endif /*]*/
{ "cp1026", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x007b, 0x00f1, 0x00c7, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x011e, 0x0130, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x005b, 0x00d1, 0x015f, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0131, 0x003a, 0x00d6, 0x015e, 0x0027, 0x003d, 0x00dc, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x007d, 0x0060, 0x00a6, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x02db, 0x00c6, 0x00a4, 0x00b5, 0x00f6, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x005d, 0x0024, 0x0040, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x2014, 0x00a8, 0x00b4, 0x00d7, 0x00e7, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x007e, 0x00f2, 0x00f3, 0x00f5, 0x011f, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x005c, 0x00f9, 0x00fa, 0x00ff, 0x00fc, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x0023, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x0022, 0x00d9, 0x00da }, "1026", "0x04800402", "iso10646-1,iso8859-9" },
{ "cp1047", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x005b, 0x00de, 0x00ae, 0x00ac, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00dd, 0x00a8, 0x00af, 0x005d, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1047", "1047", "iso10646-1,iso8859-1" },
{ "cp1140", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x20ac, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x005e, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005b, 0x005d, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1140", "0x02b70474" /* 695, 1140 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1141", { 0x00a0, 0x00e2, 0x007b, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00c4, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x007e, 0x00dc, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x005b, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00f6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x00a7, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x20ac, 0x00b5, 0x00df, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x0040, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e4, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00a6, 0x00f2, 0x00f3, 0x00f5, 0x00fc, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x007d, 0x00f9, 0x00fa, 0x00ff, 0x00d6, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x005c, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x005d, 0x00d9, 0x00da }, "1141", "0x02b70475" /* 695, 1141 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1142", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x007d, 0x00e7, 0x00f1, 0x0023, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x20ac, 0x00c5, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x0024, 0x00c7, 0x00d1, 0x00f8, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00a6, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x00c6, 0x00d8, 0x0027, 0x003d, 0x0022, 0x0040, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x007b, 0x00b8, 0x005b, 0x005d, 0x00b5, 0x00fc, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e6, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x00e5, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x007e, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1142", "0x02b70476" /* 695, 1142 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1143", { 0x00a0, 0x00e2, 0x007b, 0x00e0, 0x00e1, 0x00e3, 0x007d, 0x00e7, 0x00f1, 0x00a7, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x0060, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x20ac, 0x00c5, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x0023, 0x00c0, 0x00c1, 0x00c3, 0x0024, 0x00c7, 0x00d1, 0x00f6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x005c, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00e9, 0x003a, 0x00c4, 0x00d6, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x005d, 0x00b5, 0x00fc, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x005b, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e4, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00a6, 0x00f2, 0x00f3, 0x00f5, 0x00e5, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x007e, 0x00f9, 0x00fa, 0x00ff, 0x00c9, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x0040, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1143", "0x02b70477" /* 695, 1143 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1144", { 0x00a0, 0x00e2, 0x00e4, 0x007b, 0x00e1, 0x00e3, 0x00e5, 0x005c, 0x00f1, 0x00b0, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x005d, 0x00ea, 0x00eb, 0x007d, 0x00ed, 0x00ee, 0x00ef, 0x007e, 0x00df, 0x00e9, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00f2, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00f9, 0x003a, 0x00a3, 0x00a7, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x005b, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x20ac, 0x00b5, 0x00ec, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x0023, 0x00a5, 0x00b7, 0x00a9, 0x0040, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x00e0, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00a6, 0x00f3, 0x00f5, 0x00e8, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x0060, 0x00fa, 0x00ff, 0x00e7, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1144", "0x02b70478" /* 695, 1144 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1145", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00a6, 0x005b, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x005d, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x0023, 0x00f1, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x00d1, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x20ac, 0x00b5, 0x00a8, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005e, 0x0021, 0x00af, 0x007e, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1145", "0x02b70478" /* 695, 1145 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1146", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x0024, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x0021, 0x00a3, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x20ac, 0x00b5, 0x00af, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x005b, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005e, 0x005d, 0x007e, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1146", "0x02b7047a" /* 695, 1146 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1147", { 0x00a0, 0x00e2, 0x00e4, 0x0040, 0x00e1, 0x00e3, 0x00e5, 0x005c, 0x00f1, 0x00b0, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x007b, 0x00ea, 0x00eb, 0x007d, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x00a7, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00f9, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00b5, 0x003a, 0x00a3, 0x00e0, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x005b, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x20ac, 0x0060, 0x00a8, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x0023, 0x00a5, 0x00b7, 0x00a9, 0x005d, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x007e, 0x00b4, 0x00d7, 0x00e9, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x00e8, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00a6, 0x00fa, 0x00ff, 0x00e7, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1147", "0x02b7047a" /* 695, 1147 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1148", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x005b, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x005d, 0x0024, 0x002a, 0x0029, 0x003b, 0x005e, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x20ac, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1148", "0x02b7047c" /* 695, 1148 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1149", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00de, 0x002e, 0x003c, 0x0028, 0x002b, 0x0021, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x00c6, 0x0024, 0x002a, 0x0029, 0x003b, 0x00d6, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00f0, 0x003a, 0x0023, 0x00d0, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x0060, 0x00fd, 0x007b, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x007d, 0x00b8, 0x005d, 0x20ac, 0x00b5, 0x00f6, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x0040, 0x00dd, 0x005b, 0x00ae, 0x00a2, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00ac, 0x007c, 0x00af, 0x00a8, 0x005c, 0x00d7, 0x00fe, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x007e, 0x00f2, 0x00f3, 0x00f5, 0x00e6, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x00b4, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x005e, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "1149", "0x02b7047d" /* 695, 1149 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-15" },
{ "cp1160", { 0x00a0, 0x0e01, 0x0e02, 0x0e03, 0x0e04, 0x0e05, 0x0e06, 0x0e07, 0x005b, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x0e48, 0x0e08, 0x0e09, 0x0e0a, 0x0e0b, 0x0e0c, 0x0e0d, 0x0e0e, 0x005d, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x0e0f, 0x0e10, 0x0e11, 0x0e12, 0x0e13, 0x0e14, 0x0e15, 0x005e, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x0e3f, 0x0e4e, 0x0e16, 0x0e17, 0x0e18, 0x0e19, 0x0e1a, 0x0e1b, 0x0e1c, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x0e4f, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x0e1d, 0x0e1e, 0x0e1f, 0x0e20, 0x0e21, 0x0e22, 0x0e5a, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0e23, 0x0e24, 0x0e25, 0x0e26, 0x0e27, 0x0e28, 0x0e5b, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x0e29, 0x0e2a, 0x0e2b, 0x0e2c, 0x0e2d, 0x0e2e, 0x0e50, 0x0e51, 0x0e52, 0x0e53, 0x0e54, 0x0e55, 0x0e56, 0x0e57, 0x0e58, 0x0e59, 0x0e2f, 0x0e30, 0x0e31, 0x0e32, 0x0e33, 0x0e34, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x0e49, 0x0e35, 0x0e36, 0x0e37, 0x0e38, 0x0e39, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x0e3a, 0x0e40, 0x0e41, 0x0e42, 0x0e43, 0x0e44, 0x005c, 0x0e4a, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x0e45, 0x0e46, 0x0e47, 0x0e48, 0x0e49, 0x0e4a, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x0e4b, 0x0e4c, 0x0e4d, 0x0e4b, 0x20ac }, "1160", "0x05730488" /* 1395, 1160 */, "iso10646-1,iso8859-11" },
#if defined(X3270_DBCS) /*[*/
{ "cp1388", { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00a3, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0021, 0x00a5, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x0000, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007e, 0x00af, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005e, 0x0000, 0x005c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005b, 0x005d, 0x0000, 0x0000, 0x0000, 0x0000, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0024, 0x0000, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 }, "1388", "0x04960344" /* 1174, 836 */, "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
#endif /*]*/
{ "apl", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x00dd, 0x00de, 0x00ae, 0x005e, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x005b, 0x005d, 0x00af, 0x00a8, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "37", "37", "3270cg-1a,iso10646-1"  },
{ "bracket", { 0x00a0, 0x00e2, 0x00e4, 0x00e0, 0x00e1, 0x00e3, 0x00e5, 0x00e7, 0x00f1, 0x00a2, 0x002e, 0x003c, 0x0028, 0x002b, 0x007c, 0x0026, 0x00e9, 0x00ea, 0x00eb, 0x00e8, 0x00ed, 0x00ee, 0x00ef, 0x00ec, 0x00df, 0x0021, 0x0024, 0x002a, 0x0029, 0x003b, 0x00ac, 0x002d, 0x002f, 0x00c2, 0x00c4, 0x00c0, 0x00c1, 0x00c3, 0x00c5, 0x00c7, 0x00d1, 0x00a6, 0x002c, 0x0025, 0x005f, 0x003e, 0x003f, 0x00f8, 0x00c9, 0x00ca, 0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x0060, 0x003a, 0x0023, 0x0040, 0x0027, 0x003d, 0x0022, 0x00d8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x00ab, 0x00bb, 0x00f0, 0x00fd, 0x00fe, 0x00b1, 0x00b0, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x00aa, 0x00ba, 0x00e6, 0x00b8, 0x00c6, 0x00a4, 0x00b5, 0x007e, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00a1, 0x00bf, 0x00d0, 0x005b, 0x00de, 0x00ae, 0x005e, 0x00a3, 0x00a5, 0x00b7, 0x00a9, 0x00a7, 0x00b6, 0x00bc, 0x00bd, 0x00be, 0x00dd, 0x00a8, 0x00af, 0x005d, 0x00b4, 0x00d7, 0x007b, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x00ad, 0x00f4, 0x00f6, 0x00f2, 0x00f3, 0x00f5, 0x007d, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x00b9, 0x00fb, 0x00fc, 0x00f9, 0x00fa, 0x00ff, 0x005c, 0x00f7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x00b2, 0x00d4, 0x00d6, 0x00d2, 0x00d3, 0x00d5, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x00b3, 0x00db, 0x00dc, 0x00d9, 0x00da }, "37+", "37", "3270cg-1a,3270cg-1,iso10646-1,iso8859-1" },
{ NULL }
};

/* Code page aliases. */

typedef struct {
    char *alias;
    char *canon;
} cpalias_t;

cpalias_t cpaliases[] = {
    { "belgian",	"cp500" },
    { "belgian-euro",	"cp1148" },
    { "brazilian",	"cp275" },
#if defined(X3270_DBCS) /*[*/
    { "chinese-gb18030","cp1388" },
    { "cp1027",         "cp939" },	/* historical error */
    { "cp290",          "cp930" },	/* historical error */
#endif /*]*/
    { "cp37",           "cp037" },
#if defined(X3270_DBCS) /*[*/
    { "cp836",          "cp935" },	/* historical error */
#endif /*]*/
    { "finnish",	"cp278" },
    { "finnish-euro",	"cp1143" },
    { "french",		"cp297" },
    { "french-euro",	"cp1147" },
    { "german",		"cp273" },
    { "german-euro",	"cp1141" },
    { "greek",		"cp875" },
    { "hebrew",		"cp424" },
    { "icelandic",	"cp871" },
    { "icelandic-euro",	"cp1149" },
    { "italian",	"cp280" },
    { "italian-euro",	"cp1144" },
#if defined(X3270_DBCS) /*[*/
    { "japanese-1027",	"cp939" },	/* historical error */
    { "japanese-290",	"cp930" },	/* historical error */
    { "japanese-kana",  "cp930" },
    { "japanese-latin", "cp939" },
#endif /*]*/
    { "norwegian",	"cp277" },
    { "norwegian-euro",	"cp1142" },
    { "oldibm",		"bracket" },
    { "bracket437",	"bracket" },
    { "polish",		"cp870" },
    { "russian",	"cp880" },
#if defined(X3270_DBCS) /*[*/
    { "simplified-chinese","cp935" },
#endif /*]*/
    { "slovenian",	"cp870" },
    { "spanish",	"cp284" },
    { "spanish-euro",	"cp1145" },
    { "thai",		"cp1160" },
#if defined(X3270_DBCS) /*[*/
    { "traditional-chinese", "cp937" },
#endif /*]*/
    { "turkish",	"cp1026" },
    { "uk",		"cp285" },
    { "uk-euro",	"cp1146" },
    { DEFAULT_CSNAME,	"cp037" },
    { "us-euro",	"cp1140" },
    { "us-intl",	"cp037" },
    { NULL,		NULL }
};

static uni_t *cur_uni = NULL;

void
charset_list(void)
{
    	int i;
	int j;
	char *sep = "";

	printf("SBCS host code pages (with aliases):\n");
	for (i = 0; uni[i].name != NULL; i++) {
		Boolean any = False;
		char *asep = " (";

	    	printf("%s%s", sep, uni[i].name);
		for (j = 0; cpaliases[j].alias != NULL; j++) {

		    	if (!strcmp(cpaliases[j].canon, uni[i].name)) {
			    	printf("%s%s", asep, cpaliases[j].alias);
				asep = ", ";
				any = True;
			}
		}
		if (any)
		    	printf(")");
		sep = ", ";
	}
	printf("\n");

#if defined(X3270_DBCS) /*[*/
	charset_list_dbcs();
#endif /*]*/
}

/*
 * Translate a single EBCDIC character in an arbitrary character set to
 * Unicode.  Note that CS_DBCS is never used -- use CS_BASE and pass an
 * EBCDIC character > 0xff.
 *
 * Returns 0 for no translation.
 */
ucs4_t
ebcdic_to_unicode(ebc_t c, unsigned char cs, unsigned flags)
{
	int iuc;
	ucs4_t uc;

#if 0	/* I'm not sure why this was put in, but it breaks display of DUP
	   and FM.
	   Hopefully I'll figure out why it was put in in the first place
	   and I can put it back under the right conditions. */
	/* Control characters become blanks. */
    	if (c <= 0x41 || c == 0xff)
	    	uc = 0;
#endif

	/*
	 * We do not pay attention to BLANK_UNDEF -- we always return 0
	 * for undefined characters.
	 */
	flags &= ~EUO_BLANK_UNDEF;

	/* Dispatch on the character set. */
	if ((cs & CS_GE) || ((cs & CS_MASK) == CS_APL)) {
		iuc = apl_to_unicode(c, flags);
		if (iuc != -1)
		    	uc = iuc;
		else
		    	uc = 0;
	} else if (cs == CS_LINEDRAW) {
	    	iuc = linedraw_to_unicode(c /* XXX: flags */);
		if (iuc != -1)
		    	uc = iuc;
		else
		    	uc = 0;
	} else if (cs != CS_BASE)
	    	uc = 0;
	else
	    	uc = ebcdic_base_to_unicode(c, flags);

	return uc;
}

/*
 * Translate a single EBCDIC character in the base or DBCS character sets to
 *  Unicode.
 *
 * EBCDIC 'FM' and 'DUP' characters are treated specially.  If EUO_UPRIV
 *  is set, they are returned as U+f8fe and U+feff (private-use) respectively
 *  so they can be displayed with overscores in the special 3270 font;
 *  otherwise they are returned as '*' and ';'.
 * EBCDIC 'EO' and 'SUB' are special-cased to U+25cf and U+25a0, respectively.
 *
 * If EUO_BLANK_UNDEF is set, other undisplayable characters are returned as
 *  spaces; otherwise they are returned as 0.
 */
ucs4_t
ebcdic_base_to_unicode(ebc_t c, unsigned flags)
{
#if defined(X3270_DBCS) /*[*/
    if (c & 0xff00)
	return ebcdic_dbcs_to_unicode(c, flags);
#endif /*]*/

    if (c == 0x40)
	return 0x0020;

    if (c >= UT_OFFSET && c < 0xff) {
	ebc_t uc = cur_uni->code[c - UT_OFFSET];

	return uc? uc: ((flags & EUO_BLANK_UNDEF)? ' ': 0);

    } else switch (c) {

	case EBC_fm:
	    return (flags & EUO_UPRIV)? UPRIV_fm: ';';
	case EBC_dup:
	    return (flags & EUO_UPRIV)? UPRIV_dup: '*';
	case EBC_eo:
#if defined(C3270) /*[*/
	    if (flags & EUO_ASCII_BOX)
		    return (flags & EUO_BLANK_UNDEF)? ' ': 0;
#endif /*]*/
	    return (flags & EUO_UPRIV)? UPRIV_eo: 0x25cf; /* solid circle */
	case EBC_sub:
#if defined(C3270) /*[*/
	    if (flags & EUO_ASCII_BOX)
		    return (flags & EUO_BLANK_UNDEF)? ' ': 0;
#endif /*]*/
	    return (flags & EUO_UPRIV)? UPRIV_sub: 0x25a0; /* solid block */
	default:
	    if (flags & EUO_BLANK_UNDEF)
		return ' ';
	    else
		return 0;
    }
}

/*
 * Map a UCS-4 character to an EBCDIC character.
 * Returns 0 for failure, nonzero for success.
 */
ebc_t
unicode_to_ebcdic(ucs4_t u)
{
    int i;
#if defined(X3270_DBCS) /*[*/
    ebc_t d;
#endif /*]*/

    if (!u)
	return 0;
    if (u == 0x0020)
	return 0x40;

    for (i = 0; i < UT_SIZE; i++) {
	if (cur_uni->code[i] == u) {
	    return UT_OFFSET + i;
	}
    }
#if defined(X3270_DBCS) /*[*/
    /* See if it's DBCS. */
    d = unicode_to_ebcdic_dbcs(u);
    if (d)
	return d;
#endif /*]*/

    return 0;
}

/*
 * Map a UCS-4 character to an EBCDIC character, possibly including APL (GE)
 * characters.
 * Returns 0 for failure, nonzero for success.
 */
ebc_t
unicode_to_ebcdic_ge(ucs4_t u, Boolean *ge)
{
    ebc_t e;

    *ge = False;
    e = unicode_to_ebcdic(u);
    if (e)
	return e;

    /* Handle GEs.  Yes, this is slow, but I'm lazy. */
    for (e = 0x70; e <= 0xfe; e++) {
	if ((ucs4_t)apl_to_unicode(e, EUO_NONE) == u) {
	    *ge = True;
	    return e;
	}
    }

    return 0;
}

/*
 * Set the SBCS EBCDIC-to-Unicode translation table.
 * Returns 0 for success, -1 for failure.
 */
int
set_uni(const char *csname, const char **host_codepage,
	const char **cgcsgid, const char **display_charsets)
{
	int i;
	const char *realname;
	int rc = -1;
	Boolean cannot_fail = False;

	/*
	 * If the csname is NULL, this is a fallback to the default
	 * and the iconv lookup cannot fail.
	 */
	if (csname == NULL) {
	    	csname = DEFAULT_CSNAME;
		cannot_fail = True;
	}
	realname = csname;

	/* Search for an alias. */
	for (i = 0; cpaliases[i].alias != NULL; i++) {
		if (!strcasecmp(csname, cpaliases[i].alias)) {
			realname = cpaliases[i].canon;
			break;
		}
	}

	/* Search for a match. */
	for (i = 0; uni[i].name != NULL; i++) {
		if (!strcasecmp(realname, uni[i].name)) {
			cur_uni = &uni[i];
			*host_codepage = uni[i].host_codepage;
			*cgcsgid = uni[i].cgcsgid;
			*display_charsets = uni[i].display_charset;
			rc = 0;
			break;
		}
	}

	if (cannot_fail && rc == -1)
		Error("Cannot find default charset definition");

#if defined(USE_ICONV) /*[*/
	/*
	 * wchar_t's are not Unicode, so getting to/from Unicode is only half
	 * the battle.  We need to use iconv() to get between Unicode to the
	 * local multi-byte representation.  We'll explicitly use UTF-8, which
	 * appears to be the most broadly-supported translation.
	 */
	if (rc == 0) {
		if (!is_utf8) {
			/*
			 * If iconv doesn't support the locale's codeset, then
			 * this box is hosed.
			 */
			i_u2mb = iconv_open(locale_codeset, "UTF-8");
			if (i_u2mb == (iconv_t)(-1))
				rc = -1;
			else {
				i_mb2u = iconv_open("UTF-8", locale_codeset);
				if (i_mb2u == (iconv_t)(-1)) {
					iconv_close(i_u2mb);
					rc = -1;
				}
			}
		}

		if (rc == -1 && cannot_fail) {
		    	/* Try again with plain-old ASCII. */
#if defined(PR3287) /*[*/
		    	Warning("Cannot find iconv translation from locale "
				"codeset to UTF-8, using ASCII");
#else /*][*/
		    	xs_warning("Cannot find iconv translation from locale "
				"codeset '%s' to UTF-8, using ASCII",
				locale_codeset);
#endif /*]*/
			i_u2mb = iconv_open("ASCII", "UTF-8");
			if (i_u2mb == (iconv_t)-1)
			    	Error("No iconv UTF-8 to ASCII translation");
			i_mb2u = iconv_open("UTF-8", "ASCII");
			if (i_mb2u == (iconv_t)-1)
			    	Error("No iconv ASCII to UTF-8 translation");
			rc = 0;
		}
	}
#endif /*]*/

	return rc;
}

/*
 * Translate an x3270 font line-drawing character (the first two rows of a
 * standard X11 fixed-width font) to Unicode.
 *
 * Returns -1 if there is no translation.
 */
int
linedraw_to_unicode(ebc_t c)
{
	static ebc_t ld2uc[32] = {
    /* 00 */	0x2588, 0x25c6, 0x2592, 0x0000, 0x0000, 0x0000, 0x0000, 0x00b0,
    /* 08 */	0x00b1, 0x0000, 0x0000, 0x2518, 0x2510, 0x250c, 0x2514, 0x253c,
    /* 10 */	0x002d, 0x002d, 0x2500, 0x002d, 0x005f, 0x251c, 0x2524, 0x2534,
    /* 18 */	0x252c, 0x2502, 0x2264, 0x2265, 0x03c0, 0x2260, 0x00a3, 0x2022
	};

	if (c < 32 && ld2uc[c] != 0x0000)
	    	return ld2uc[c];
	else
	    	return -1;
}

int
apl_to_unicode(ebc_t c, unsigned flags)
{
	static ebc_t apl2uc[256] = {
    /* 00 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 08 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 10 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 18 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 20 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 28 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 30 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 38 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 40 */	0x0020, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 48 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 50 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 58 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 60 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 68 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 70 */	0x25c6, 0x22c0, 0x00a8, 0x223b, 0x2378, 0x2377, 0x22a2, 0x22a3,
    /* 78 */	0x2228, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 80 */	0x007e, 0x0000, 0x0000, 0x23b8, 0x23b9, 0x2502, 0x0000, 0x0000,
    /* 88 */	0x0000, 0x0000, 0x2191, 0x2193, 0x2264, 0x2308, 0x230a, 0x2192,
    /* 90 */	0x2395, 0x258c, 0x2590, 0x2580, 0x2584, 0x25a0, 0x0000, 0x0000,
    /* 98 */	0x0000, 0x0000, 0x2283, 0x2282, 0x00a4, 0x25cb, 0x00b1, 0x2190,
    /* a0 */	0x00af, 0x00b0, 0x2500, 0x2022, 0x0000, 0x0000, 0x0000, 0x0000,
    /* a8 */	0x0000, 0x0000, 0x22c2, 0x22c3, 0x22a5, 0x005b, 0x2265, 0x2218,
    /* b0 */	0x03b1, 0x03b5, 0x03b9, 0x03c1, 0x03c9, 0x0000, 0x00d7, 0x005c,
    /* b8 */	0x00f7, 0x0000, 0x2207, 0x2206, 0x22a4, 0x005d, 0x2260, 0x2502,
    /* c0 */	0x007b, 0x207c, 0x002b, 0x220e, 0x2514, 0x250c, 0x251c, 0x2534,
    /* c8 */	0x00a7, 0x0000, 0x2372, 0x2371, 0x2337, 0x233d, 0x2342, 0x2349,
    /* d0 */	0x007d, 0x207e, 0x002d, 0x253c, 0x2518, 0x2510, 0x2524, 0x252c,
    /* d8 */	0x00b6, 0x0000, 0x2336, 0x0021, 0x2352, 0x234b, 0x235e, 0x235d,
    /* e0 */	0x2261, 0x2081, 0x0282, 0x0283, 0x2364, 0x2365, 0x236a, 0x20ac,
    /* e8 */	0x0000, 0x0000, 0x233f, 0x2240, 0x2235, 0x2296, 0x2339, 0x2355,
    /* f0 */	0x2070, 0x00b9, 0x00b2, 0x00b3, 0x2074, 0x2075, 0x2076, 0x2077,
    /* f8 */	0x2078, 0x2079, 0x0000, 0x236b, 0x2359, 0x235f, 0x234e, 0x0000
	};
#if defined(C3270) /*[*/
	static ebc_t apla2uc[256] = {
    /* 00 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 08 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 10 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 18 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 20 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 28 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 30 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 38 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 40 */	0x0020, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 48 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 50 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 58 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 60 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 68 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 70 */	0x0000, 0x0000, 0x00a8, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 78 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 80 */	0x007e, 0x0000, 0x0000, 0x0000, 0x0000, 0x007c, 0x0000, 0x0000,
    /* 88 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 90 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* 98 */	0x0000, 0x0000, 0x0000, 0x0000, 0x00a4, 0x0000, 0x00b1, 0x0000,
    /* a0 */	0x00af, 0x00b0, 0x002d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* a8 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x005b, 0x0000, 0x0000,
    /* b0 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00d7, 0x005c,
    /* b8 */	0x00f7, 0x0000, 0x0000, 0x0000, 0x0000, 0x005d, 0x0000, 0x007c,
    /* c0 */	0x007b, 0x0000, 0x002b, 0x0000, 0x002b, 0x002b, 0x002b, 0x002b,
    /* c8 */	0x00a7, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* d0 */	0x007d, 0x0000, 0x002d, 0x002b, 0x002b, 0x002b, 0x002b, 0x002b,
    /* d8 */	0x00b6, 0x0000, 0x0000, 0x0021, 0x0000, 0x0000, 0x0000, 0x0000,
    /* e0 */	0x0000, 0x0000, 0x0282, 0x0283, 0x0000, 0x0000, 0x0000, 0x0000,
    /* e8 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    /* f0 */	0x0000, 0x00b9, 0x00b2, 0x00b3, 0x0000, 0x0000, 0x0000, 0x0000,
    /* f8 */	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000
	};
#endif /*]*/

#if defined(C3270) /*[*/
	if (flags & EUO_ASCII_BOX) {
		if (c < 256 && apla2uc[c] != 0x0000) {
#if defined(_WIN32) /*[*/
		    	/* Windows DBCS fonts make U+0080..U+00ff wide, too. */
		    	if (apla2uc[c] > 0x7f)
			    	return -1;
#endif /*]*/
			return apla2uc[c];
		} else
			return -1;
	}
#endif /*]*/

	if (c < 256 && apl2uc[c] != 0x0000)
	    	return apl2uc[c];
	else
	    	return -1;
}

/*
 * Translate an EBCDIC character to the current locale's multi-byte
 * representation.
 *
 * Returns the number of bytes in the multi-byte representation, including
 * the terminating NULL.  mb[] should be big enough to include the NULL
 * in the result.
 *
 * Also returns in 'ucp' the UCS-4 Unicode value of the EBCDIC character.
 *
 * Note that 'ebc' is an ebc_t (uint16_t), not an unsigned char.  This is
 * so that DBCS values can be passed in as 16 bits (with the first byte
 * in the high-order bits).  There is no ambiguity because all valid EBCDIC
 * DBCS characters have a nonzero first byte.
 *
 * Returns 0 if EUO_BLANK_UNDEF is clear and there is no printable EBCDIC
 * translation for 'ebc'.
 *
 * Returns '?' in mb[] if there is no local multi-byte representation of
 * the EBCDIC character.
 */
int
ebcdic_to_multibyte_x(ebc_t ebc, unsigned char cs, char mb[],
	int mb_len, unsigned flags, ucs4_t *ucp)
{
	ucs4_t uc;

#if defined(_WIN32) /*[*/
	int nc;
	BOOL udc;
	wchar_t wuc;
#elif defined(UNICODE_WCHAR) /*][*/
	int nc;
	wchar_t wuc;
#else /*][*/
	char u8b[7];
	size_t nu8;
	ici_t inbuf;
	char *outbuf;
	size_t inbytesleft, outbytesleft;
	size_t nc;
#endif /*]*/

	/* Translate from EBCDIC to Unicode. */
	uc = ebcdic_to_unicode(ebc, cs, flags);
	if (ucp != NULL)
		*ucp = uc;
	if (uc == 0) {
		if (flags & EUO_BLANK_UNDEF) {
			mb[0] = ' ';
			mb[1] = '\0';
			return 2;
		} else {
			return 0;
		}
	}

	/* Translate from Unicode to local multibyte. */

#if defined(_WIN32) /*[*/
	/*
	 * wchar_t's are Unicode.
	 */
	wuc = uc;
	nc = WideCharToMultiByte(LOCAL_CODEPAGE, 0, &wuc, 1, mb, mb_len,
		"?", &udc);
	if (nc != 0) {
		mb[nc++] = '\0';
		return nc;
	} else {
		mb[0] = '?';
		mb[1] = '\0';
		return 2;
	}

#elif defined(UNICODE_WCHAR) /*][*/
	/*
	 * wchar_t's are Unicode.
	 * If 'is_utf8' is set, use unicode_to_utf8().  This allows us to set
	 *  'is_utf8' directly, ignoring the locale, for Tcl.
	 * Otherwise, use wctomb().
	 */
	if (is_utf8) {
		nc = unicode_to_utf8(uc, mb);
		if (nc < 0)
			return 0;
		mb[nc++] = '\0';
		return nc;
	}

	wuc = uc;
	nc = wctomb(mb, uc);
	if (nc > 0) {
		/* Return to the initial shift state and null-terminate. */
		nc += wctomb(mb + nc, 0);
		return nc;
	} else {
		mb[0] = '?';
		mb[1] = '\0';
		return 2;
	}
#else /*][*/
	/*
	 * Use iconv.
	 */

	/* Translate the wchar_t we got from UCS-4 to UTF-8. */
	nu8 = unicode_to_utf8(uc, u8b);
	if (nu8 < 0)
		return 0;

	/* Local multi-byte might be UTF-8, in which case, we're done. */
	if (is_utf8) {
	    memcpy(mb, u8b, nu8);
	    mb[nu8++] = '\0';
	    return nu8;
	}

	/* Let iconv translate from UTF-8 to local multi-byte. */
	inbuf = u8b;
	inbytesleft = nu8;
	outbuf = mb;
	outbytesleft = mb_len;
	nc = iconv(i_u2mb, &inbuf, &inbytesleft, &outbuf, &outbytesleft);
	if (nc < 0 || inbytesleft == nu8) {
		mb[0] = '?';
		mb[1] = '\0';
		return 2;
	}

	/* Return to the initial shift state. */
	nc = iconv(i_u2mb, NULL, NULL, &outbuf, &outbytesleft);
	if (nc < 0) {
		mb[0] = '?';
		mb[1] = '\0';
		return 0;
	}

	/* Null-terminate the return the length. */
	mb[mb_len - outbytesleft--] = '\0';
	return mb_len - outbytesleft;

#endif /*]*/
}

/* Commonest version of ebcdic_to_multibyte_x:
 *  cs is CS_BASE
 *  EUO_BLANK_UNDEF is set
 *  ucp is ignored
 */
int
ebcdic_to_multibyte(ebc_t ebc, char mb[], int mb_len)
{
    	return ebcdic_to_multibyte_x(ebc, CS_BASE, mb, mb_len, EUO_BLANK_UNDEF,
		NULL);
}

/*
 * Convert an EBCDIC string to a multibyte string.
 * Makes lots of assumptions: standard character set, EUO_BLANK_UNDEF.
 * Returns the length of the multibyte string.
 */
int
ebcdic_to_multibyte_string(unsigned char *ebc, size_t ebc_len, char mb[],
	size_t mb_len)
{
    	int nmb = 0;

    	while (ebc_len && mb_len) {
	    	int xlen;

		xlen = ebcdic_to_multibyte(*ebc, mb, mb_len);
		if (xlen) {
		    	mb += xlen - 1;
			mb_len -= (xlen - 1);
			nmb += xlen - 1;
		}
		ebc++;
		ebc_len--;
	}
	return nmb;
}

/*
 * Return the maximum buffer length needed to translate 'len' EBCDIC characters
 * in the current locale.
 */
int
mb_max_len(int len)
{
#if defined(_WIN32) /*[*/
    /*
     * On Windows, it's 1:1 (we don't do DBCS, and we don't support locales
     * like UTF-8).
     */
    return len + 1;
#elif defined(UNICODE_WCHAR) /*][*/
    /* Allocate enough space for shift-state transitions. */
    return (MB_CUR_MAX * (len * 2)) + 1;
#else /*]*/
    if (is_utf8)
	return (len * 6) + 1;
    else
	/*
	 * We don't actually know.  Guess that MB_CUR_MAX is 16, and compute
	 * as for UNICODE_WCHAR.
	 */
	return (16 * (len * 2)) + 1;
#endif /*]*/
}

/*
 * Translate a multi-byte character in the current locale to UCS-4.
 *
 * Returns a UCS-4 character or 0, indicating an error in translation.
 * Also returns the number of characters consumed.
 */
ucs4_t
multibyte_to_unicode(const char *mb, size_t mb_len, int *consumedp,
	enum me_fail *errorp)
{
    size_t nw;
    ucs4_t ucs4;
#if defined(_WIN32) /*[*/
    wchar_t wc[3];
    int i;

    /* Use MultiByteToWideChar() to get from the ANSI codepage to UTF-16. */
    for (i = 1; i <= mb_len; i++) {
	nw = MultiByteToWideChar(LOCAL_CODEPAGE, MB_ERR_INVALID_CHARS,
		mb, i, wc, 3);
	if (nw != 0)
	    break;
    }
    if (i > mb_len) {
	*errorp = ME_INVALID;
	return 0;
    }
    *consumedp = i;
    ucs4 = wc[0];
#elif defined(UNICODE_WCHAR) /*][*/
    wchar_t wc[3];
    /* wchar_t's are Unicode. */

    if (is_utf8) {
	int nc;

	/*
	 * Use utf8_to_unicode() instead of mbtowc(), so we can set is_utf8
	 * directly and ignore the locale for Tcl.
	 */
	nc = utf8_to_unicode(mb, mb_len, &ucs4);
	if (nc > 0) {
	    *errorp = ME_NONE;
	    *consumedp = nc;
	    return ucs4;
	} else if (nc == 0) {
	    *errorp = ME_SHORT;
	    return 0;
	} else {
	    *errorp = ME_INVALID;
	    return 0;
	}
    }

    /* mbtowc() will translate to Unicode. */
    nw = mbtowc(wc, mb, mb_len);
    if (nw == (size_t)-1) {
	if (errno == EILSEQ)
	    *errorp = ME_INVALID;
	else
	    *errorp = ME_SHORT;
	nw = mbtowc(NULL, NULL, 0);
	return 0;
    }

    /*
     * Reset the shift state.
     * XXX: Doing this will ruin the shift state if this function is called
     * repeatedly to process a string.  There should probably be a parameter
     * passed in to control whether or not to reset the shift state, or
     * perhaps there should be a function to translate a string.
     */
    *consumedp = nw;
    nw = mbtowc(NULL, NULL, 0);

    ucs4 = wc[0];
#else /*][*/
    /* wchar_t's have unknown encoding. */
    if (!is_utf8) {
	ici_t inbuf;
	char *outbuf;
	size_t inbytesleft, outbytesleft;
	char utf8buf[16];
	size_t ibl;

	/* Translate from local MB to UTF-8 using iconv(). */
	for (ibl = 1; ibl <= mb_len; ibl++) {
	    inbuf = (ici_t)mb;
	    outbuf = utf8buf;
	    inbytesleft = ibl;
	    outbytesleft = sizeof(utf8buf);
	    nw = iconv(i_mb2u, &inbuf, &inbytesleft, &outbuf, &outbytesleft);
	    if (nw < 0) {
		if (errno == EILSEQ) {
		    *errorp = ME_INVALID;
		    (void) iconv(i_mb2u, NULL, NULL, NULL, NULL);
		    return 0;
		} else {
		    if (ibl == mb_len) {
			*errorp = ME_SHORT;
			(void) iconv(i_mb2u, NULL, NULL, NULL, NULL);
			return 0;
		    }
		}
	    } else
	    	break;
	}
	*consumedp = ibl - inbytesleft;

	/* Translate from UTF-8 to UCS-4. */
	(void) utf8_to_unicode(utf8buf, sizeof(utf8buf) - outbytesleft, &ucs4);
    } else {
	/* Translate from UTF-8 to UCS-4. */
	nw = utf8_to_unicode(mb, mb_len, &ucs4);
	if (nw < 0) {
	    *errorp = ME_INVALID;
	    return 0;
	}
	if (nw == 0) {
	    *errorp = ME_SHORT;
	    return 0;
	}
	*consumedp = nw;
    }
#endif /*]*/

    /* Translate from UCS4 to EBCDIC. */
    return ucs4;
}

/*
 * Convert a multi-byte string to a UCS-4 string.
 * Does not NULL-terminate the result.
 * Returns the number of UCS-4 characters stored.
 */
int
multibyte_to_unicode_string(char *mb, size_t mb_len, ucs4_t *ucs4,
	size_t u_len)
{
    int consumed;
    enum me_fail error;
    int nr = 0;

    error = ME_NONE;

    while (u_len && mb_len &&
	    (*ucs4++ = multibyte_to_unicode(mb, mb_len, &consumed,
					    &error)) != 0) {
	u_len--;
	mb += consumed;
	mb_len -= consumed;
	nr++;
    }

    if (error != ME_NONE)
	return -1;
    else
	return nr;
}

/*
 * Translate a multi-byte character in the current locale to an EBCDIC
 * character.
 *
 * Returns an 8-bit (SBCS) or 16-bit (DBCS) EBCDIC character, or 0, indicating
 * an error in translation.  Also returns the number of characters consumed.
 */
ebc_t
multibyte_to_ebcdic(const char *mb, size_t mb_len, int *consumedp,
	enum me_fail *errorp)
{
    ucs4_t ucs4;

    ucs4 = multibyte_to_unicode(mb, mb_len, consumedp, errorp);
    if (ucs4 == 0)
	return 0;
    return unicode_to_ebcdic(ucs4);
}

/*
 * Convert a local multi-byte string to an EBCDIC string.
 * Returns the length of the resulting EBCDIC string, or -1 if there is a
 * conversion error.
 */
int
multibyte_to_ebcdic_string(char *mb, size_t mb_len, unsigned char *ebc,
	size_t ebc_len, enum me_fail *errorp)
{
    int ne = 0;
    Boolean in_dbcs = False;

    while (mb_len > 0 && ebc_len > 0) {
	ebc_t e;
	int consumed;

	e = multibyte_to_ebcdic(mb, mb_len, &consumed, errorp);
	if (e == 0)
	    return -1;
	if (e & 0xff00) {
	    /* DBCS. */
	    if (!in_dbcs) {
		/* Make sure there's room for SO, b1, b2, SI. */
		if (ebc_len < 4)
		    return ne;
		*ebc++ = EBC_so;
		ebc_len++;
		ne++;
		in_dbcs = True;
	    }
	    /* Make sure there's room for b1, b2, SI. */
	    if (ebc_len < 3) {
		*ebc++ = EBC_si;
		ne++;
		return ne;
	    }
	    *ebc++ = (e >> 8) & 0xff;
	    *ebc++ = e & 0xff;
	    ebc_len -= 2;
	    ne += 2;
	} else {
	    /* SBCS. */
	    if (in_dbcs) {
		*ebc++ = EBC_si;
		ne++;
		if (!--ebc_len)
		    return ne;
		in_dbcs = False;
	    }
	    *ebc++ = e & 0xff;
	    ebc_len--;
	    ne++;
	}
	mb += consumed;
	mb_len -= consumed;
    }

    /*
     * Terminate the DBCS string, if we end inside it.
     * We're guaranteed to have space for the SI; we checked before adding
     * the last DBCS character.
     */
    if (in_dbcs) {
	*ebc++ = EBC_si;
	ne++;
    }

    return ne;
}

/*
 * Translate a UCS-4 character to a local multi-byte string.
 */
int
unicode_to_multibyte(ucs4_t ucs4, char *mb, size_t mb_len)
{
#if defined(_WIN32) /*[*/
    wchar_t wuc = ucs4;
    BOOL udc;
    int nc;

    nc = WideCharToMultiByte(LOCAL_CODEPAGE, 0, &wuc, 1, mb, mb_len,
	    "?", &udc);
    if (nc > 0)
	mb[nc++] = '\0';
    return nc;
#elif defined(UNICODE_WCHAR) /*][*/
    int nc;

    if (is_utf8) {
	nc = unicode_to_utf8(ucs4, mb);
	if (nc < 0)
	    return 0;
	mb[nc++] = '\0';
	return nc;
    }

    nc = wctomb(mb, ucs4);
    if (nc > 0) {
	/* Return to the initial shift state and null-terminate. */
	nc += wctomb(mb + nc, 0);
	return nc;
    } else {
	mb[0] = '?';
	mb[1] = '\0';
	return 2;
    }
#else /*][*/
    int nu8;
    char u8b[16];
    ici_t inbuf;
    char *outbuf;
    size_t inbytesleft, outbytesleft;
    size_t nc;

    /* Use iconv. */

    /* Translate the wchar_t we got from UCS-4 to UTF-8. */
    nu8 = unicode_to_utf8(ucs4, u8b);
    if (nu8 < 0)
	return 0;

    /* Local multi-byte might be UTF-8, in which case, we're done. */
    if (is_utf8) {
	memcpy(mb, u8b, nu8);
	mb[nu8++] = '\0';
	return nu8;
    }

    /* Let iconv translate from UTF-8 to local multi-byte. */
    inbuf = u8b;
    inbytesleft = nu8;
    outbuf = mb;
    outbytesleft = mb_len;
    nc = iconv(i_u2mb, &inbuf, &inbytesleft, &outbuf, &outbytesleft);
    if (nc == (size_t)-1) {
	mb[0] = '?';
	mb[1] = '\0';
	return 2;
    }

    /* Return to the initial shift state. */
    nc = iconv(i_u2mb, NULL, NULL, &outbuf, &outbytesleft);
    if (nc == (size_t)-1) {
	mb[0] = '?';
	mb[1] = '\0';
	return 0;
    }

    /* Null-terminate the return the length. */
    mb[mb_len - outbytesleft--] = '\0';
    return mb_len - outbytesleft;
#endif /*]*/
}
