-------------------------------------------------------------------------------
--									                                                  --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/sqlite3/gnu-db-sqlite3.ads,v $
--  Description     : SQLite Base Package                                    --
--  Author          : Ching Bon Lam                                          --
--  Created         : 26.7.2003                                              --
--  Last Modified By: $Author: persan $				                             --
--  Last Modified On: $Date: 2007/09/07 21:09:51 $		                       --
--  Status          : $State: Exp $					                             --
--									                                                  --
--  Copyright (C) 2003 Ching Bon Lam                                         --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  Thick binding to sqlite (http://www.sqlite.org).                         --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  References                                                               --
--  ==========                                                               --
--  Sqlite homepage : http://www.sqlite.org                                  --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--    Ching Bon Lam  <cblam@gmx.net>                                         --
--                                                                           --
-------------------------------------------------------------------------------


with GNU.DB.Support.Wide_String_Tables;
with GNU.DB.Support.Wide_String_Vectors;

with GNU.DB.Support.String_Vectors;
with GNU.DB.Support.String_Tables;

package GNU.DB.SQLite3 is
   use GNU.DB.Support;
   type Object is tagged private;
   type Handle is access all Object;
   -- sqlite3.h:47
   -- typedef struct sqlite3 sqlite3;

   ----------------------------------------------------------------------------
   -- return values -----------------------------------------------------------
   ----------------------------------------------------------------------------

   type Return_Value is (-- sqlite3.h:133-161
                         SQLITE_OK,         -- Successful result
                         SQLITE_ERROR,      -- SQL error or missing database
                         SQLITE_INTERNAL,   -- An internal logic error in SQLite
                         SQLITE_PERM,       -- Access permission denied
                         SQLITE_ABORT,      -- Callback routine requested an abort
                         SQLITE_BUSY,       -- The database file is locked
                         SQLITE_LOCKED,     -- A table in the database is locked
                         SQLITE_NOMEM,      -- A malloc() failed
                         SQLITE_READONLY,   -- Attempt to write a readonly database
                         SQLITE_INTERRUPT,  -- Operation terminated by sqlite3_interrupt()
                         SQLITE_IOERR,      -- Some kind of disk I/O error occurred
                         SQLITE_CORRUPT,    -- The database disk image is malformed
                         SQLITE_NOTFOUND,   -- (Internal Only) Table or record not found
                         SQLITE_FULL,       -- Insertion failed because database is full
                         SQLITE_CANTOPEN,   -- Unable to open the database file
                         SQLITE_PROTOCOL,   -- Database lock protocol error
                         SQLITE_EMPTY,      -- Database is empty
                         SQLITE_SCHEMA,     -- The database schema changed
                         SQLITE_TOOBIG,     -- Too much data for one row of a table
                         SQLITE_CONSTRAINT, -- Abort due to contraint violation
                         SQLITE_MISMATCH,   -- Data type mismatch
                         SQLITE_MISUSE,     -- Library used incorrectly
                         SQLITE_NOLFS,      -- Uses OS features not supported on host
                         SQLITE_AUTH,       -- Authorization denied
                         SQLITE_FORMAT,     -- Auxiliary database format error
                         SQLITE_RANGE,      -- 2nd parameter to sqlite3_bind out of range
                         SQLITE_NOTADB,     -- File opened that is not a database file
                         SQLITE_ROW,        -- sqlite3_step() has another row ready
                         SQLITE_DONE,       -- sqlite3_step() has finished executing
                         UNDEFINED_ERROR);

   ----------------------------------------------------------------------------
   -- Other important types ---------------------------------------------------
   ----------------------------------------------------------------------------

   type Statement is tagged private;
   type Statement_Reference is access all Statement;
   -- sqlite3.h:554
   -- typedef struct sqlite3_stmt sqlite3_stmt;

   type Uint64 is mod 2 ** 64;
   type Int64 is
   range -(2 ** (64 - 1)) ..
     +(2 ** (64 - 1) - 1);
   -- sqlite3.h:55





   type Callback is access function
     (Argc        : Integer;
      Argv        : String_Vectors.Vector;
      ColumnNames : String_Vectors.Vector)
      return Return_Value;

   type Callback16 is access function
     (Argc        : Integer;
      Argv        : Wide_String_Vectors.Vector;
      ColumnNames : Wide_String_Vectors.Vector)
      return Return_Value;


   type Table_Reference is access all String_Tables.Vector;
   type Table16_Reference is access all Wide_String_Tables.Vector;

   ----------------------------------------------------------------------------
   -- Basic operations --------------------------------------------------------
   ----------------------------------------------------------------------------

   function Close
     (Self : access Object)
      return Return_Value;
   procedure Close (Self : access Object);

   -- sqlite3.h:75
   -- int sqlite3_close(sqlite3 *);
   function Last_Insert_Rowid
     (Self : access Object)
      return Uint64;
   -- sqlite3.h:172
   -- sqlite_int64 sqlite3_last_insert_rowid(sqlite3*);

   function Changes
     (Self : access Object)
      return Integer;
   -- sqlite3.h:194
   -- int sqlite3_changes(sqlite3*);

   function Total_Changes
     (Self : access Object)
      return Integer;
   -- sqlite3.h:212
   -- int sqlite3_total_changes(sqlite3*);

   procedure Interrupt
     (Self : access Object);
   -- sqlite3.h:220
   -- void sqlite3_interrupt(sqlite3*);

   function Complete
     (Sql : String)
      return Boolean;
   -- sqlite3.h:233
   -- int sqlite3_complete(const char *sql);

   function Complete16
     (Sql : Wide_String)
      return Boolean;
   -- sqlite3.h:234
   -- int sqlite3_complete16(const void *sql);

   function Open
     (Self     : access Object;
      Filename : String)
      return Return_Value;
   procedure Open
     (Self     : access Object;
      Filename : String);
   -- sqlite3.h:504
   -- int sqlite3_open(
   --   const char *filename,   /* Database filename (UTF-8) */
   --   sqlite3 **ppDb          /* OUT: SQLite db access Object */
   -- );

   function Open16
     (Self     : access Object;
      Filename : Wide_String)
      return Return_Value;
   -- sqlite3.h:508
   -- int sqlite3_open16(
   --   const void *filename,   /* Database filename (UTF-16) */
   --   sqlite3 **ppDb          /* OUT: SQLite db access Object */
   -- );

   ----------------------------------------------------------------------------
   -- Exec --------------------------------------------------------------------
   ----------------------------------------------------------------------------

   generic
      type Data_Type is private;
      type Data_Handle is access all Data_Type;
   package Generic_Exec is

      type Callback is access function
        (PArg        : Data_Handle;
         Argc        : Integer;
         Argv        : String_Vectors.Vector;
         ColumnNames : String_Vectors.Vector)
         return Return_Value;
      -- sqlite3.h:80
      -- typedef int (*sqlite3_callback)(void*,int,char**, char**);

      function Exec
        (Self    : access Object;
         Sql     : String;
         Cb      : Callback;
         Arg     : Data_Handle)
         return Return_Value;
      -- sqlite3.h:122
      -- int sqlite3_exec(
      --   sqlite3*,         /* An open database */
      --   const char *sql,  /* SQL to be executed */
      --   sqlite3_callback, /* Callback function */
      --   void *,           /* 1st argument to callback function */
      --   char **errmsg     /* Error msg written here */
      -- );
   end Generic_Exec;

   generic
      type Data_Type is private;
      type Data_Handle is access all Data_Type;
   package Generic_Exec16 is

      type Callback16 is access function
        (PArg        : Data_Handle;
         Argc        : Integer;
         Argv        : Wide_String_Vectors.Vector;
         ColumnNames : Wide_String_Vectors.Vector)
         return Return_Value;

      function Exec16
        (Self    : access Object;
         Sql     : Wide_String;
         Cb      : Callback16;
         Arg     : Data_Handle)
         return Return_Value;
   end Generic_Exec16;

   function Exec
     (Self : access Object;
      Sql  : String;
      Cb   : Callback)
      return Return_Value;
   procedure Exec
     (Self : access Object;
      Sql  : String;
      Cb   : Callback);

   function Exec16
     (Self : access Object;
      Sql  : Wide_String;
      Cb   : Callback16)
      return Return_Value;

   function Exec
     (Self : access Object;
      Sql  : String)
      return Return_Value;
   procedure Exec
     (Self : access Object;
      Sql  : String);

   function Exec16
     (Self : access Object;
      Sql  : Wide_String)
      return Return_Value;

   ----------------------------------------------------------------------------
   -- Tables ------------------------------------------------------------------
   ----------------------------------------------------------------------------

   function Get_Table
     (Self  : access Object;
      Sql   : String;
      Table : Table_Reference)
      return Return_Value;

   function Get_Table16
     (Self  : access Object;
      Sql   : Wide_String;
      Table : Table16_Reference)
      return Return_Value;

   function Row_Count
     (Self : Table_Reference)
      return Natural;

   function Column_Count
     (Self : Table_Reference;
      Row  : Natural)
      return Natural;

   function Element
     (Self : Table_Reference;
      Row  : Natural;
      Col  : Natural)
      return String;

   function Element16
     (Self : Table16_Reference;
      Row  : Natural;
      Col  : Natural)
      return Wide_String;


   ----------------------------------------------------------------------------
   -- Error stuff -------------------------------------------------------------
   ----------------------------------------------------------------------------

   function Errcode
     (Self : access Object)
      return Integer;
   -- sqlite3.h:528
   -- int sqlite3_errcode(sqlite3 *db);

   function Errmsg
     (Self : access Object)
      return String;
   -- sqlite3.h:538
   -- const char *sqlite3_errmsg(sqlite3*);

   function Errmsg16
     (Self : access Object)
      return Wide_String;
   -- sqlite3.h:548
   -- const void *sqlite3_errmsg16(sqlite3*);

   ----------------------------------------------------------------------------
   -- Stepping ----------------------------------------------------------------
   ----------------------------------------------------------------------------

   SQLITE_INTEGER : constant Integer := 1;
   SQLITE_FLOAT   : constant Integer := 2;
   SQLITE_TEXT    : constant Integer := 3;
   SQLITE_BLOB    : constant Integer := 4;
   SQLITE_NULL    : constant Integer := 5;

   function Prepare
     (Self : access Object;
      Sql  : String;
      Stmt : Statement_Reference)
      return Return_Value;
   procedure Prepare
     (Self : access Object;
      Sql  : String;
      Stmt : Statement_Reference);

   -- sqlite3.h:581
   -- int sqlite3_prepare(
   --   sqlite3 *db,            /* Database handle */
   --   const char *zSql,       /* SQL statement, UTF-8 encoded */
   --   int nBytes,             /* Length of zSql in bytes. */
   --   sqlite3_stmt **ppStmt,  /* OUT: Statement handle */
   --   const char **pzTail     /* OUT: Pointer to unused portion of zSql */
   -- );

   function Prepare16
     (Self : access Object;
      Sql  : Wide_String;
      Stmt : Statement_Reference)
      return Return_Value;
   -- sqlite3.h:588
   -- int sqlite3_prepare16(
   --   sqlite3 *db,            /* Database handle */
   --   const void *zSql,       /* SQL statement, UTF-16 encoded */
   --   int nBytes,             /* Length of zSql in bytes. */
   --   sqlite3_stmt **ppStmt,  /* OUT: Statement handle */
   --   const void **pzTail     /* OUT: Pointer to unused portion of zSql */
   -- );


--     generic
--        type Data_Type is private;
--     package Blob_Generic is
--        function Bind_Blob
--          (Stmt : Statement_Reference;
--           N    : Integer;
--           B    : Data_Type)
--           return Return_Value;
--        procedure Bind_Blob
--          (Stmt : Statement_Reference;
--           N    : Integer;
--           B    : Data_Type);
--
--     end Blob_Generic;
   -- function bind_blob
   --   (stmt : Statement_Reference;
   --    N    : integer;
   --    B    : something)
   --    return Return_Value;
   -- sqlite3.h:626
   -- int sqlite3_bind_blob(sqlite3_stmt*, int, const void*, int n, void(*)(void*));

   function Bind_Double
     (Stmt : access Statement;
      N    : Integer;
      D    : Long_Float)
      return Return_Value;
   -- sqlite3.h:627
   -- int sqlite3_bind_double(sqlite3_stmt*, int, double);

   function Bind_Int
     (Stmt : access Statement;
      N    : Integer;
      I    : Integer)
      return Return_Value;
   -- sqlite3.h:628
   -- int sqlite3_bind_int(sqlite3_stmt*, int, int);

   function Bind_Int64
     (Stmt : access Statement;
      N    : Integer;
      I    : Int64)
      return Return_Value;
   -- sqlite3.h:629
   -- int sqlite3_bind_int64(sqlite3_stmt*, int, sqlite_int64);

   function Bind_Null
     (Stmt : access Statement;
      N    : Integer)
      return Return_Value;
   -- sqlite3.h:630
   -- int sqlite3_bind_null(sqlite3_stmt*, int);

   function Bind_Text
     (Stmt : access Statement;
      N    : Integer;
      T    : String)
      return Return_Value;
   -- sqlite3.h:631
   -- int sqlite3_bind_text(sqlite3_stmt*, int, const char*, int n, void(*)(void*));

   function Bind_Text16
     (Stmt : access Statement;
      N    : Integer;
      T    : Wide_String)
      return Return_Value;
   -- sqlite3.h:632
   -- int sqlite3_bind_text16(sqlite3_stmt*, int, const void*, int, void(*)(void*));

   -- function bind_value
   --   (stmt : Statement_Reference;
   --    N    : integer;
   --    V    : Value)
   --    return Return_Value;
   -- sqlite3.h:633
   -- int sqlite3_bind_value(sqlite3_stmt*, int, const sqlite3_value*);

   function Column_Count
     (Stmt : access Statement)
      return Integer;
   -- sqlite3.h:661
   -- int sqlite3_column_count(sqlite3_stmt *pStmt);

   function Column_Name
     (Stmt : access Statement;
      ICol : Integer)
      return String;
   -- sqlite3.h:669
   -- const char *sqlite3_column_name(sqlite3_stmt*,int);

   function Column_Name16
     (Stmt : access Statement;
      ICol : Integer)
      return Wide_String;
   -- sqlite3.h:670
   -- const void *sqlite3_column_name16(sqlite3_stmt*,int);

   function Column_Decltype
     (Stmt : access Statement;
      ICol : Integer)
      return String;
   -- sqlite3.h:690
   -- const char *sqlite3_column_decltype(sqlite3_stmt *, int i);

   function Column_Decltype16
     (Stmt : access Statement;
      ICol : Integer)
      return Wide_String;
   -- sqlite3.h:710
   -- const void *sqlite3_column_decltype16(sqlite3_stmt*,int);

   function Step
     (Stmt : access Statement)
      return Return_Value;

   function Step
     (Stmt : access Statement)
      return Boolean;

   procedure Step
     (Stmt : access Statement);

   -- sqlite3.h:744
   -- int sqlite3_step(sqlite3_stmt*);

   function Data_Count
     (Stmt : access Statement)
      return Integer;
   -- sqlite3.h:755
   -- int sqlite3_data_count(sqlite3_stmt *pStmt);

   function Column_Int
     (Stmt : access Statement;
      ICol : Integer)
      return Integer;
   -- sqlite3.h:828
   -- int sqlite3_column_int(sqlite3_stmt*, int iCol);

   function Column_Text
     (Stmt : access Statement;
      ICol : Integer)
      return String;
   -- sqlite3.h:830
   --const unsigned char *sqlite3_column_text(sqlite3_stmt*, int iCol);

   function Column_Text16
     (Stmt : access Statement;
      ICol : Integer)
      return Wide_String;
   -- sqlite3.h:831
   --const void *sqlite3_column_text16(sqlite3_stmt*, int iCol);

   function Column_Type
     (Stmt : access Statement;
      ICol : Integer)
      return Integer;
   -- sqlite3.h:832
   -- int sqlite3_column_type(sqlite3_stmt*, int iCol);

   function Finalize
     (Stmt : access Statement)
      return Return_Value;
   procedure Finalize
     (Stmt : access Statement);
   -- sqlite3.h:848
   -- int sqlite3_finalize(sqlite3_stmt *pStmt);

   function Reset
     (Stmt : access Statement)
      return Return_Value;
   -- sqlite3.h:857
   -- int sqlite3_reset(sqlite3_stmt *pStmt);

   ----------------------------------------------------------------------------
   -- PRIVATE -----------------------------------------------------------------
   ----------------------------------------------------------------------------
   SQLITE_Exception : exception;
private

   type DB_Type is limited null record;
   type DB_Access is access all DB_Type;
   for DB_Access'Storage_Size use 0;
   pragma Convention (C, DB_Access);

   type Object is tagged record
      Db : aliased DB_Access;
   end record;
   -- sqlite3.h:47
   -- typedef struct sqlite3 sqlite3;

   type Stmt_Type is limited null record;
   type Stmt_Access is access all Stmt_Type;
   for Stmt_Access'Storage_Size use 0;
   pragma Convention (C, Stmt_Access);

   type Statement is tagged record
      Stmt : aliased Stmt_Access := null;
   end record;

   pragma Linker_Options ("-lsqlite3");
end GNU.DB.SQLite3;
