//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/CompoundForm.cpp
//! @brief     Implements class CompoundForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleDesigner/CompoundForm.h"
#include "GUI/Model/Sample/CompoundItem.h"
#include "GUI/Support/Util/ActionFactory.h"
#include "GUI/View/SampleDesigner/FormLayouter.h"
#include "GUI/View/SampleDesigner/LayerEditorUtil.h"
#include "GUI/View/Tool/GroupBoxCollapser.h"

#include <QAction>
#include <QPushButton>

CompoundForm::CompoundForm(QWidget* parent, CompoundItem* compoundItem, SampleEditorController* ec,
                           bool allowRemove)
    : QGroupBox(parent)
    , m_compositionItem(compoundItem)
    , m_ec(ec)
{
    setTitle("Composition of particles");

    FormLayouter layouter(this, ec);
    layouter.setContentsMargins(30, 6, 0, 0);
    layouter.addVector(compoundItem->position(), false);
    layouter.addSelection(compoundItem->rotationSelection());
    layouter.addValue(compoundItem->abundance());

    for (auto* particle : compoundItem->itemsWithParticles())
        layouter.addRow(
            LayerEditorUtil::createWidgetForItemWithParticles(this, particle, false, ec));

    auto* btn = LayerEditorUtil::createAddParticleButton(
        this, [=](FormFactorItemCatalog::Type type) { ec->addCompoundItem(compoundItem, type); },
        [=](ItemWithParticlesCatalog::Type type) { ec->addCompoundItem(compoundItem, type); });

    m_structureEditingWidgets << btn;
    layouter.addStructureEditingRow(btn);

    auto* collapser = GroupBoxCollapser::installIntoGroupBox(this);
    collapser->setExpanded(compoundItem->isExpandCompound());
    connect(collapser, &GroupBoxCollapser::toggled, this,
            [compoundItem](bool b) { compoundItem->setExpandCompound(b); });

    // top right corner actions
    // show in real space
    {
        auto* showInRealspaceAction = ActionFactory::createShowInRealspaceAction(
            this, "particle composition",
            [ec, compoundItem] { ec->requestViewInRealspace(compoundItem); });
        collapser->addAction(showInRealspaceAction);
    }
    // duplicate
    {
        m_duplicateAction =
            ActionFactory::createDuplicateAction(this, "particle composition", [ec, compoundItem] {
                ec->duplicateItemWithParticles(compoundItem);
            });
        collapser->addAction(m_duplicateAction);
    }
    // remove
    {
        m_removeAction = ActionFactory::createRemoveAction(
            this, "particle composition", [ec, compoundItem] { ec->removeParticle(compoundItem); });
        if (allowRemove)
            collapser->addAction(m_removeAction);
    }

    m_layout = layouter.layout();
}

CompoundForm::~CompoundForm() = default;

void CompoundForm::enableStructureEditing(bool b)
{
    m_removeAction->setVisible(b);
    m_duplicateAction->setVisible(b);

    for (auto* w : m_structureEditingWidgets)
        w->setVisible(b);
}

CompoundItem* CompoundForm::compositionItem() const
{
    return m_compositionItem;
}

void CompoundForm::onParticleAdded(ItemWithParticles* p)
{
    int index = m_compositionItem->itemsWithParticles().indexOf(p);
    const int rowInLayout = m_layout->rowCount() - 1
                            - (m_compositionItem->itemsWithParticles().size() - 1)
                            + index; // -1: btn

    m_layout->insertRow(rowInLayout,
                        LayerEditorUtil::createWidgetForItemWithParticles(this, p, false, m_ec));
}

void CompoundForm::onAboutToRemoveParticle(ItemWithParticles* item)
{
    int index = m_compositionItem->itemsWithParticles().indexOf(item);
    const int rowInLayout = m_layout->rowCount() - m_compositionItem->itemsWithParticles().size()
                            - 1 + index; // -1: btn

    m_layout->removeRow(rowInLayout);
}
