/*
 * Copyright (c) 2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include "../../bfloat.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void a64_hybrid_fp32bf16fp32_mmla_4x24 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const bfloat16 *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const bfloat16 *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 130f\n"
      "cmp %x[M], #0x2\n"
      "bgt 87f\n"
      "beq 44f\n"
      "mov x10, %x[bias]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "cbz x10, 3f\n"
      "ldr q8, [x10, #0x0]\n"
      "ldr q9, [x10, #0x10]\n"
      "zip2 v14.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "zip2 v15.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "ldr q12, [x10, #0x40]\n"
      "ldr q13, [x10, #0x50]\n"
      "zip2 v16.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v17.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "add x10, x10, #0x60\n"
      "zip2 v18.2d, v12.2d, v12.2d\n"
      "zip1 v12.2d, v12.2d, v12.2d\n"
      "zip2 v19.2d, v13.2d, v13.2d\n"
      "zip1 v13.2d, v13.2d, v13.2d\n"
      "b 19f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 18f\n"
      "cmp x9, #0x18\n"
      "bge 16f\n"
      "tbz x9, #4, 7f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "ld1 { v12.4s }, [x27], #0x10\n"
      "tbz x9, #2, 5f\n"
      "ld1 { v13.4s }, [x27], #0x10\n"
      "tbz x9, #1, 4f\n"
      "ldr d20, [x27], #0x8\n"
      "mov x20, #0x58\n"
      "tbz x9, #0, 15f\n"
      "ld1 { v20.s }[2], [x27]\n"
      "b 15f\n"
      "4:"  // Height 1: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x9, #0, 15f\n"
      "ldr s20, [x27, #0x0]\n"
      "b 15f\n"
      "5:"  // Height 1: Partial accumulate: partial_2_16
      "tbz x9, #1, 6f\n"
      "ldr d13, [x27], #0x8\n"
      "mov x20, #0x48\n"
      "tbz x9, #0, 15f\n"
      "ld1 { v13.s }[2], [x27]\n"
      "b 15f\n"
      "6:"  // Height 1: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x9, #0, 15f\n"
      "ldr s13, [x27, #0x0]\n"
      "b 15f\n"
      "7:"  // Height 1: Partial accumulate: partial_8_0
      "tbz x9, #3, 11f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "tbz x9, #2, 9f\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "tbz x9, #1, 8f\n"
      "ldr d12, [x27], #0x8\n"
      "mov x20, #0x38\n"
      "tbz x9, #0, 15f\n"
      "ld1 { v12.s }[2], [x27]\n"
      "b 15f\n"
      "8:"  // Height 1: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x9, #0, 15f\n"
      "ldr s12, [x27, #0x0]\n"
      "b 15f\n"
      "9:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x9, #1, 10f\n"
      "ldr d11, [x27], #0x8\n"
      "mov x20, #0x28\n"
      "tbz x9, #0, 15f\n"
      "ld1 { v11.s }[2], [x27]\n"
      "b 15f\n"
      "10:"  // Height 1: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x9, #0, 15f\n"
      "ldr s11, [x27, #0x0]\n"
      "b 15f\n"
      "11:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x9, #2, 13f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "tbz x9, #1, 12f\n"
      "ldr d10, [x27], #0x8\n"
      "mov x20, #0x18\n"
      "tbz x9, #0, 15f\n"
      "ld1 { v10.s }[2], [x27]\n"
      "b 15f\n"
      "12:"  // Height 1: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x9, #0, 15f\n"
      "ldr s10, [x27, #0x0]\n"
      "b 15f\n"
      "13:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x9, #1, 14f\n"
      "ldr d9, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x9, #0, 15f\n"
      "ld1 { v9.s }[2], [x27]\n"
      "b 15f\n"
      "14:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s9, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "15:"  // Height 1: Partial accumulate: Done
      "sub x27, x27, x20\n"
      "b 17f\n"
      "16:"  // Height 1: full accumulate
      "ldr q9, [x27, #0x0]\n"
      "ldr q10, [x27, #0x10]\n"
      "ldr q11, [x27, #0x20]\n"
      "ldr q12, [x27, #0x30]\n"
      "ldr q13, [x27, #0x40]\n"
      "ldr q20, [x27, #0x50]\n"
      "17:"  // Height 1: MMLA fixup
      "zip1 v8.2d, v9.2d, v14.2d\n"
      "zip2 v14.2d, v9.2d, v14.2d\n"
      "zip1 v9.2d, v10.2d, v15.2d\n"
      "zip2 v15.2d, v10.2d, v15.2d\n"
      "zip1 v10.2d, v11.2d, v16.2d\n"
      "zip2 v16.2d, v11.2d, v16.2d\n"
      "zip1 v11.2d, v12.2d, v17.2d\n"
      "zip2 v17.2d, v12.2d, v17.2d\n"
      "zip1 v12.2d, v13.2d, v18.2d\n"
      "zip2 v18.2d, v13.2d, v18.2d\n"
      "zip1 v13.2d, v20.2d, v19.2d\n"
      "zip2 v19.2d, v20.2d, v19.2d\n"
      "b 19f\n"
      "18:"  // Height 1: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "19:"  // Height 1: setup done
      "mov x26, #0x0\n"
      "20:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 21f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "cbnz x26, 22f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "b 22f\n"
      "21:"  // Height 1: setup direct input
      "mov x24, %x[input_ptr]\n"
      "22:"  // Height 1: input setup done
      "cmp x25, #0x4\n"
      "blt 25f\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      "ldr q4, [x28, #0x0]\n"
      "cmp x25, #0x8\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "blt 24f\n"
      "23:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "ldr q24, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      "ldr q23, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      "ldr q22, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "ldr q21, [x28, #0x70]\n"
      ".inst 0x6e58ec0a  // bfmmla v10.4s, v0.8h, v24.8h\n"
      "ldr q24, [x28, #0x80]\n"
      ".inst 0x6e57ec10  // bfmmla v16.4s, v0.8h, v23.8h\n"
      "ldr q23, [x28, #0x90]\n"
      ".inst 0x6e56ec0b  // bfmmla v11.4s, v0.8h, v22.8h\n"
      "ldr q22, [x28, #0xa0]\n"
      ".inst 0x6e55ec11  // bfmmla v17.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0xb0]\n"
      "sub x25, x25, #0x4\n"
      "cmp x25, #0x8\n"
      ".inst 0x6e58ec0c  // bfmmla v12.4s, v0.8h, v24.8h\n"
      "add x28, x28, #0xc0\n"
      "ldr q4, [x28, #0x0]\n"
      ".inst 0x6e57ec12  // bfmmla v18.4s, v0.8h, v23.8h\n"
      "ldr q5, [x28, #0x10]\n"
      ".inst 0x6e56ec0d  // bfmmla v13.4s, v0.8h, v22.8h\n"
      "ldr q6, [x28, #0x20]\n"
      ".inst 0x6e55ec13  // bfmmla v19.4s, v0.8h, v21.8h\n"
      "ldr q7, [x28, #0x30]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      "bge 23b\n"
      "24:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "ldr q23, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      "ldr q25, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      "ldr q21, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "ldr q24, [x28, #0x70]\n"
      ".inst 0x6e57ec0a  // bfmmla v10.4s, v0.8h, v23.8h\n"
      "ldr q23, [x28, #0x80]\n"
      ".inst 0x6e59ec10  // bfmmla v16.4s, v0.8h, v25.8h\n"
      "ldr q22, [x28, #0x90]\n"
      ".inst 0x6e55ec0b  // bfmmla v11.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0xa0]\n"
      ".inst 0x6e58ec11  // bfmmla v17.4s, v0.8h, v24.8h\n"
      "ldr q5, [x28, #0xb0]\n"
      "sub x25, x25, #0x4\n"
      ".inst 0x6e57ec0c  // bfmmla v12.4s, v0.8h, v23.8h\n"
      ".inst 0x6e56ec12  // bfmmla v18.4s, v0.8h, v22.8h\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e55ec0d  // bfmmla v13.4s, v0.8h, v21.8h\n"
      ".inst 0x6e45ec13  // bfmmla v19.4s, v0.8h, v5.8h\n"
      "25:"  // Height 1: Multiply loop: Main loop skip
      "cbz x25, 28f\n"
      "cbz x25, 28f\n"
      "tbz x25, #1, 26f\n"
      "ldr d0, [x24], #0x8\n"
      "tbz x25, #0, 27f\n"
      "ld1 { v0.s }[2], [x24]\n"
      "b 27f\n"
      "26:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x24, #0x0]\n"
      "27:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q21, [x28, #0x0]\n"
      "ldr q1, [x28, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x6e55ec08  // bfmmla v8.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0x20]\n"
      "ldr q22, [x28, #0x30]\n"
      ".inst 0x6e41ec0e  // bfmmla v14.4s, v0.8h, v1.8h\n"
      ".inst 0x6e55ec09  // bfmmla v9.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0x40]\n"
      "ldr q23, [x28, #0x50]\n"
      ".inst 0x6e56ec0f  // bfmmla v15.4s, v0.8h, v22.8h\n"
      ".inst 0x6e55ec0a  // bfmmla v10.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0x60]\n"
      "ldr q22, [x28, #0x70]\n"
      ".inst 0x6e57ec10  // bfmmla v16.4s, v0.8h, v23.8h\n"
      ".inst 0x6e55ec0b  // bfmmla v11.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0x80]\n"
      "ldr q23, [x28, #0x90]\n"
      ".inst 0x6e56ec11  // bfmmla v17.4s, v0.8h, v22.8h\n"
      ".inst 0x6e55ec0c  // bfmmla v12.4s, v0.8h, v21.8h\n"
      "ldr q22, [x28, #0xa0]\n"
      "ldr q21, [x28, #0xb0]\n"
      ".inst 0x6e57ec12  // bfmmla v18.4s, v0.8h, v23.8h\n"
      ".inst 0x6e56ec0d  // bfmmla v13.4s, v0.8h, v22.8h\n"
      ".inst 0x6e55ec13  // bfmmla v19.4s, v0.8h, v21.8h\n"
      "add x28, x28, #0xc0\n"
      "28:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 20b\n"
      "uzp1 v8.2d, v8.2d, v14.2d\n"
      "uzp1 v9.2d, v9.2d, v15.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v10.2d, v10.2d, v16.2d\n"
      "uzp1 v11.2d, v11.2d, v17.2d\n"
      "uzp1 v12.2d, v12.2d, v18.2d\n"
      "uzp1 v13.2d, v13.2d, v19.2d\n"
      "tbz %x[flags], #1, 29f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v22.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v21.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v22.4s\n"
      "fmin v9.4s, v9.4s, v22.4s\n"
      "fmin v10.4s, v10.4s, v22.4s\n"
      "fmin v11.4s, v11.4s, v22.4s\n"
      "fmin v12.4s, v12.4s, v22.4s\n"
      "fmin v13.4s, v13.4s, v22.4s\n"
      "fmax v8.4s, v8.4s, v21.4s\n"
      "fmax v9.4s, v9.4s, v21.4s\n"
      "fmax v10.4s, v10.4s, v21.4s\n"
      "fmax v11.4s, v11.4s, v21.4s\n"
      "fmax v12.4s, v12.4s, v21.4s\n"
      "fmax v13.4s, v13.4s, v21.4s\n"
      "29:"  // Height 1: No activation
      "cmp x9, #0x18\n"
      "bge 42f\n"
      "tbz x9, #4, 33f\n"
      "st1 { v8.4s }, [x27], #0x10\n"
      "st1 { v9.4s }, [x27], #0x10\n"
      "st1 { v10.4s }, [x27], #0x10\n"
      "st1 { v11.4s }, [x27], #0x10\n"
      "tbz x9, #2, 31f\n"
      "st1 { v12.4s }, [x27], #0x10\n"
      "tbz x9, #1, 30f\n"
      "str d13, [x27], #0x8\n"
      "tbz x9, #0, 41f\n"
      "st1 { v13.s }[2], [x27]\n"
      "b 41f\n"
      "30:"  // Height 1: Partial direct writeback: partial_1_20
      "tbz x9, #0, 41f\n"
      "str s13, [x27, #0x0]\n"
      "b 41f\n"
      "31:"  // Height 1: Partial direct writeback: partial_2_16
      "tbz x9, #1, 32f\n"
      "str d12, [x27], #0x8\n"
      "tbz x9, #0, 41f\n"
      "st1 { v12.s }[2], [x27]\n"
      "b 41f\n"
      "32:"  // Height 1: Partial direct writeback: partial_1_16
      "tbz x9, #0, 41f\n"
      "str s12, [x27, #0x0]\n"
      "b 41f\n"
      "33:"  // Height 1: Partial direct writeback: partial_8_0
      "tbz x9, #3, 37f\n"
      "st1 { v8.4s }, [x27], #0x10\n"
      "st1 { v9.4s }, [x27], #0x10\n"
      "tbz x9, #2, 35f\n"
      "st1 { v10.4s }, [x27], #0x10\n"
      "tbz x9, #1, 34f\n"
      "str d11, [x27], #0x8\n"
      "tbz x9, #0, 41f\n"
      "st1 { v11.s }[2], [x27]\n"
      "b 41f\n"
      "34:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x9, #0, 41f\n"
      "str s11, [x27, #0x0]\n"
      "b 41f\n"
      "35:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x9, #1, 36f\n"
      "str d10, [x27], #0x8\n"
      "tbz x9, #0, 41f\n"
      "st1 { v10.s }[2], [x27]\n"
      "b 41f\n"
      "36:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x9, #0, 41f\n"
      "str s10, [x27, #0x0]\n"
      "b 41f\n"
      "37:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x9, #2, 39f\n"
      "st1 { v8.4s }, [x27], #0x10\n"
      "tbz x9, #1, 38f\n"
      "str d9, [x27], #0x8\n"
      "tbz x9, #0, 41f\n"
      "st1 { v9.s }[2], [x27]\n"
      "b 41f\n"
      "38:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x9, #0, 41f\n"
      "str s9, [x27, #0x0]\n"
      "b 41f\n"
      "39:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x9, #1, 40f\n"
      "str d8, [x27], #0x8\n"
      "tbz x9, #0, 41f\n"
      "st1 { v8.s }[2], [x27]\n"
      "b 41f\n"
      "40:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x27, #0x0]\n"
      "41:"  // Height 1: Partial direct writeback: Done
      "b 43f\n"
      "42:"  // Height 1: Full writeback
      "str q8, [x27, #0x0]\n"
      "str q9, [x27, #0x10]\n"
      "str q10, [x27, #0x20]\n"
      "str q11, [x27, #0x30]\n"
      "str q12, [x27, #0x40]\n"
      "str q13, [x27, #0x50]\n"
      "add x27, x27, #0x60\n"
      "43:"  // Height 1: Writeback done
      "subs x9, x9, #0x18\n"
      "bgt 2b\n"
      "b 174f\n"
      "44:"  // Height 2
      "mov x10, %x[bias]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "45:"  // Height 2: Column loop
      "cbz x10, 46f\n"
      "ldr q8, [x10, #0x0]\n"
      "ldr q9, [x10, #0x10]\n"
      "zip2 v14.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "zip2 v15.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "ldr q12, [x10, #0x40]\n"
      "ldr q13, [x10, #0x50]\n"
      "zip2 v16.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v17.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "add x10, x10, #0x60\n"
      "zip2 v18.2d, v12.2d, v12.2d\n"
      "zip1 v12.2d, v12.2d, v12.2d\n"
      "zip2 v19.2d, v13.2d, v13.2d\n"
      "zip1 v13.2d, v13.2d, v13.2d\n"
      "b 62f\n"
      "46:"  // Height 2: no bias
      "tbz %x[flags], #0, 61f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x9, #0x18\n"
      "add x23, x27, x20, LSL #2\n"
      "bge 59f\n"
      "tbz x9, #4, 50f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v12.4s }, [x27], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "tbz x9, #2, 48f\n"
      "ld1 { v13.4s }, [x27], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "tbz x9, #1, 47f\n"
      "ldr d20, [x27], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "mov x20, #0x58\n"
      "tbz x9, #0, 58f\n"
      "ld1 { v20.s }[2], [x27]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "b 58f\n"
      "47:"  // Height 2: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x9, #0, 58f\n"
      "ldr s20, [x27, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "b 58f\n"
      "48:"  // Height 2: Partial accumulate: partial_2_16
      "tbz x9, #1, 49f\n"
      "ldr d13, [x27], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "mov x20, #0x48\n"
      "tbz x9, #0, 58f\n"
      "ld1 { v13.s }[2], [x27]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "b 58f\n"
      "49:"  // Height 2: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x9, #0, 58f\n"
      "ldr s13, [x27, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "b 58f\n"
      "50:"  // Height 2: Partial accumulate: partial_8_0
      "tbz x9, #3, 54f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "tbz x9, #2, 52f\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "tbz x9, #1, 51f\n"
      "ldr d12, [x27], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "mov x20, #0x38\n"
      "tbz x9, #0, 58f\n"
      "ld1 { v12.s }[2], [x27]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "b 58f\n"
      "51:"  // Height 2: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x9, #0, 58f\n"
      "ldr s12, [x27, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "b 58f\n"
      "52:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x9, #1, 53f\n"
      "ldr d11, [x27], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "mov x20, #0x28\n"
      "tbz x9, #0, 58f\n"
      "ld1 { v11.s }[2], [x27]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "b 58f\n"
      "53:"  // Height 2: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x9, #0, 58f\n"
      "ldr s11, [x27, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "b 58f\n"
      "54:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x9, #2, 56f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "tbz x9, #1, 55f\n"
      "ldr d10, [x27], #0x8\n"
      "ldr d15, [x23], #0x8\n"
      "mov x20, #0x18\n"
      "tbz x9, #0, 58f\n"
      "ld1 { v10.s }[2], [x27]\n"
      "ld1 { v15.s }[2], [x23]\n"
      "b 58f\n"
      "55:"  // Height 2: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x9, #0, 58f\n"
      "ldr s10, [x27, #0x0]\n"
      "ldr s15, [x23, #0x0]\n"
      "b 58f\n"
      "56:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x9, #1, 57f\n"
      "ldr d9, [x27], #0x8\n"
      "ldr d14, [x23], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x9, #0, 58f\n"
      "ld1 { v9.s }[2], [x27]\n"
      "ld1 { v14.s }[2], [x23]\n"
      "b 58f\n"
      "57:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s9, [x27, #0x0]\n"
      "ldr s14, [x23, #0x0]\n"
      "mov x20, #0x0\n"
      "58:"  // Height 2: Partial accumulate: Done
      "sub x27, x27, x20\n"
      "b 60f\n"
      "59:"  // Height 2: full accumulate
      "ldr q9, [x27, #0x0]\n"
      "ldr q10, [x27, #0x10]\n"
      "ldr q11, [x27, #0x20]\n"
      "ldr q12, [x27, #0x30]\n"
      "ldr q13, [x27, #0x40]\n"
      "ldr q20, [x27, #0x50]\n"
      "ldr q14, [x23, #0x0]\n"
      "ldr q15, [x23, #0x10]\n"
      "ldr q16, [x23, #0x20]\n"
      "ldr q17, [x23, #0x30]\n"
      "ldr q18, [x23, #0x40]\n"
      "ldr q19, [x23, #0x50]\n"
      "60:"  // Height 2: MMLA fixup
      "zip1 v8.2d, v9.2d, v14.2d\n"
      "zip2 v14.2d, v9.2d, v14.2d\n"
      "zip1 v9.2d, v10.2d, v15.2d\n"
      "zip2 v15.2d, v10.2d, v15.2d\n"
      "zip1 v10.2d, v11.2d, v16.2d\n"
      "zip2 v16.2d, v11.2d, v16.2d\n"
      "zip1 v11.2d, v12.2d, v17.2d\n"
      "zip2 v17.2d, v12.2d, v17.2d\n"
      "zip1 v12.2d, v13.2d, v18.2d\n"
      "zip2 v18.2d, v13.2d, v18.2d\n"
      "zip1 v13.2d, v20.2d, v19.2d\n"
      "zip2 v19.2d, v20.2d, v19.2d\n"
      "b 62f\n"
      "61:"  // Height 2: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "62:"  // Height 2: setup done
      "mov x26, #0x0\n"
      "63:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 64f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "cbnz x26, 65f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "b 65f\n"
      "64:"  // Height 2: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21, LSL #2\n"
      "65:"  // Height 2: input setup done
      "cmp x25, #0x4\n"
      "blt 68f\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      "ld1 { v1.4s }, [x23], #0x10\n"
      "cmp x25, #0x8\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "blt 67f\n"
      "66:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "ldr q3, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      "ldr q23, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      "ldr q22, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "ldr q21, [x28, #0x70]\n"
      ".inst 0x6e43ec0a  // bfmmla v10.4s, v0.8h, v3.8h\n"
      "ldr q1, [x28, #0x80]\n"
      ".inst 0x6e57ec10  // bfmmla v16.4s, v0.8h, v23.8h\n"
      "ldr q23, [x28, #0x90]\n"
      ".inst 0x6e56ec0b  // bfmmla v11.4s, v0.8h, v22.8h\n"
      "ldr q22, [x28, #0xa0]\n"
      ".inst 0x6e55ec11  // bfmmla v17.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0xb0]\n"
      "sub x25, x25, #0x4\n"
      "cmp x25, #0x8\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e41ec0c  // bfmmla v12.4s, v0.8h, v1.8h\n"
      "ldr q4, [x28, #0x0]\n"
      ".inst 0x6e57ec12  // bfmmla v18.4s, v0.8h, v23.8h\n"
      "ldr q5, [x28, #0x10]\n"
      ".inst 0x6e56ec0d  // bfmmla v13.4s, v0.8h, v22.8h\n"
      "ldr q6, [x28, #0x20]\n"
      ".inst 0x6e55ec13  // bfmmla v19.4s, v0.8h, v21.8h\n"
      "ldr q7, [x28, #0x30]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "ld1 { v1.4s }, [x23], #0x10\n"
      "bge 66b\n"
      "67:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "ldr q24, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      "ldr q23, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      "ldr q22, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "ldr q21, [x28, #0x70]\n"
      ".inst 0x6e58ec0a  // bfmmla v10.4s, v0.8h, v24.8h\n"
      "ldr q24, [x28, #0x80]\n"
      ".inst 0x6e57ec10  // bfmmla v16.4s, v0.8h, v23.8h\n"
      "ldr q23, [x28, #0x90]\n"
      ".inst 0x6e56ec0b  // bfmmla v11.4s, v0.8h, v22.8h\n"
      "ldr q22, [x28, #0xa0]\n"
      ".inst 0x6e55ec11  // bfmmla v17.4s, v0.8h, v21.8h\n"
      "ldr q21, [x28, #0xb0]\n"
      "sub x25, x25, #0x4\n"
      ".inst 0x6e58ec0c  // bfmmla v12.4s, v0.8h, v24.8h\n"
      ".inst 0x6e57ec12  // bfmmla v18.4s, v0.8h, v23.8h\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6e56ec0d  // bfmmla v13.4s, v0.8h, v22.8h\n"
      ".inst 0x6e55ec13  // bfmmla v19.4s, v0.8h, v21.8h\n"
      "add x28, x28, #0xc0\n"
      "68:"  // Height 2: Multiply loop: Main loop skip
      "cbz x25, 71f\n"
      "cbz x25, 71f\n"
      "tbz x25, #1, 69f\n"
      "ldr d0, [x24], #0x8\n"
      "ldr d1, [x23], #0x8\n"
      "tbz x25, #0, 70f\n"
      "ld1 { v0.s }[2], [x24]\n"
      "ld1 { v1.s }[2], [x23]\n"
      "b 70f\n"
      "69:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x24, #0x0]\n"
      "ldr s1, [x23, #0x0]\n"
      "70:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q24, [x28, #0x0]\n"
      "ldr q23, [x28, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      "ldr q22, [x28, #0x20]\n"
      "ldr q21, [x28, #0x30]\n"
      ".inst 0x6e58ec08  // bfmmla v8.4s, v0.8h, v24.8h\n"
      ".inst 0x6e57ec0e  // bfmmla v14.4s, v0.8h, v23.8h\n"
      "ldr q24, [x28, #0x40]\n"
      "ldr q23, [x28, #0x50]\n"
      ".inst 0x6e56ec09  // bfmmla v9.4s, v0.8h, v22.8h\n"
      ".inst 0x6e55ec0f  // bfmmla v15.4s, v0.8h, v21.8h\n"
      "ldr q22, [x28, #0x60]\n"
      "ldr q21, [x28, #0x70]\n"
      ".inst 0x6e58ec0a  // bfmmla v10.4s, v0.8h, v24.8h\n"
      ".inst 0x6e57ec10  // bfmmla v16.4s, v0.8h, v23.8h\n"
      "ldr q24, [x28, #0x80]\n"
      "ldr q23, [x28, #0x90]\n"
      ".inst 0x6e56ec0b  // bfmmla v11.4s, v0.8h, v22.8h\n"
      ".inst 0x6e55ec11  // bfmmla v17.4s, v0.8h, v21.8h\n"
      "ldr q22, [x28, #0xa0]\n"
      "ldr q21, [x28, #0xb0]\n"
      ".inst 0x6e58ec0c  // bfmmla v12.4s, v0.8h, v24.8h\n"
      ".inst 0x6e57ec12  // bfmmla v18.4s, v0.8h, v23.8h\n"
      ".inst 0x6e56ec0d  // bfmmla v13.4s, v0.8h, v22.8h\n"
      ".inst 0x6e55ec13  // bfmmla v19.4s, v0.8h, v21.8h\n"
      "add x28, x28, #0xc0\n"
      "71:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 63b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20, LSL #2\n"
      "uzp1 v4.2d, v8.2d, v14.2d\n"
      "uzp2 v8.2d, v8.2d, v14.2d\n"
      "uzp1 v14.2d, v9.2d, v15.2d\n"
      "uzp2 v9.2d, v9.2d, v15.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "uzp1 v15.2d, v10.2d, v16.2d\n"
      "uzp2 v10.2d, v10.2d, v16.2d\n"
      "uzp1 v16.2d, v11.2d, v17.2d\n"
      "uzp2 v11.2d, v11.2d, v17.2d\n"
      "uzp1 v17.2d, v12.2d, v18.2d\n"
      "uzp2 v12.2d, v12.2d, v18.2d\n"
      "uzp1 v18.2d, v13.2d, v19.2d\n"
      "uzp2 v13.2d, v13.2d, v19.2d\n"
      "tbz %x[flags], #1, 72f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v22.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v21.4s }, [x20]\n"
      "fmin v4.4s, v4.4s, v22.4s\n"
      "fmin v14.4s, v14.4s, v22.4s\n"
      "fmin v15.4s, v15.4s, v22.4s\n"
      "fmin v16.4s, v16.4s, v22.4s\n"
      "fmin v17.4s, v17.4s, v22.4s\n"
      "fmin v18.4s, v18.4s, v22.4s\n"
      "fmin v8.4s, v8.4s, v22.4s\n"
      "fmin v9.4s, v9.4s, v22.4s\n"
      "fmin v10.4s, v10.4s, v22.4s\n"
      "fmin v11.4s, v11.4s, v22.4s\n"
      "fmin v12.4s, v12.4s, v22.4s\n"
      "fmin v13.4s, v13.4s, v22.4s\n"
      "fmax v4.4s, v4.4s, v21.4s\n"
      "fmax v14.4s, v14.4s, v21.4s\n"
      "fmax v15.4s, v15.4s, v21.4s\n"
      "fmax v16.4s, v16.4s, v21.4s\n"
      "fmax v17.4s, v17.4s, v21.4s\n"
      "fmax v18.4s, v18.4s, v21.4s\n"
      "fmax v8.4s, v8.4s, v21.4s\n"
      "fmax v9.4s, v9.4s, v21.4s\n"
      "fmax v10.4s, v10.4s, v21.4s\n"
      "fmax v11.4s, v11.4s, v21.4s\n"
      "fmax v12.4s, v12.4s, v21.4s\n"
      "fmax v13.4s, v13.4s, v21.4s\n"
      "72:"  // Height 2: No activation
      "cmp x9, #0x18\n"
      "bge 85f\n"
      "tbz x9, #4, 76f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v14.4s }, [x27], #0x10\n"
      "st1 { v15.4s }, [x27], #0x10\n"
      "st1 { v16.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v9.4s }, [x23], #0x10\n"
      "st1 { v10.4s }, [x23], #0x10\n"
      "st1 { v11.4s }, [x23], #0x10\n"
      "tbz x9, #2, 74f\n"
      "st1 { v17.4s }, [x27], #0x10\n"
      "st1 { v12.4s }, [x23], #0x10\n"
      "tbz x9, #1, 73f\n"
      "str d18, [x27], #0x8\n"
      "str d13, [x23], #0x8\n"
      "tbz x9, #0, 84f\n"
      "st1 { v18.s }[2], [x27]\n"
      "st1 { v13.s }[2], [x23]\n"
      "b 84f\n"
      "73:"  // Height 2: Partial direct writeback: partial_1_20
      "tbz x9, #0, 84f\n"
      "str s18, [x27, #0x0]\n"
      "str s13, [x23, #0x0]\n"
      "b 84f\n"
      "74:"  // Height 2: Partial direct writeback: partial_2_16
      "tbz x9, #1, 75f\n"
      "str d17, [x27], #0x8\n"
      "str d12, [x23], #0x8\n"
      "tbz x9, #0, 84f\n"
      "st1 { v17.s }[2], [x27]\n"
      "st1 { v12.s }[2], [x23]\n"
      "b 84f\n"
      "75:"  // Height 2: Partial direct writeback: partial_1_16
      "tbz x9, #0, 84f\n"
      "str s17, [x27, #0x0]\n"
      "str s12, [x23, #0x0]\n"
      "b 84f\n"
      "76:"  // Height 2: Partial direct writeback: partial_8_0
      "tbz x9, #3, 80f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v14.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v9.4s }, [x23], #0x10\n"
      "tbz x9, #2, 78f\n"
      "st1 { v15.4s }, [x27], #0x10\n"
      "st1 { v10.4s }, [x23], #0x10\n"
      "tbz x9, #1, 77f\n"
      "str d16, [x27], #0x8\n"
      "str d11, [x23], #0x8\n"
      "tbz x9, #0, 84f\n"
      "st1 { v16.s }[2], [x27]\n"
      "st1 { v11.s }[2], [x23]\n"
      "b 84f\n"
      "77:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x9, #0, 84f\n"
      "str s16, [x27, #0x0]\n"
      "str s11, [x23, #0x0]\n"
      "b 84f\n"
      "78:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x9, #1, 79f\n"
      "str d15, [x27], #0x8\n"
      "str d10, [x23], #0x8\n"
      "tbz x9, #0, 84f\n"
      "st1 { v15.s }[2], [x27]\n"
      "st1 { v10.s }[2], [x23]\n"
      "b 84f\n"
      "79:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x9, #0, 84f\n"
      "str s15, [x27, #0x0]\n"
      "str s10, [x23, #0x0]\n"
      "b 84f\n"
      "80:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x9, #2, 82f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "tbz x9, #1, 81f\n"
      "str d14, [x27], #0x8\n"
      "str d9, [x23], #0x8\n"
      "tbz x9, #0, 84f\n"
      "st1 { v14.s }[2], [x27]\n"
      "st1 { v9.s }[2], [x23]\n"
      "b 84f\n"
      "81:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x9, #0, 84f\n"
      "str s14, [x27, #0x0]\n"
      "str s9, [x23, #0x0]\n"
      "b 84f\n"
      "82:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x9, #1, 83f\n"
      "str d4, [x27], #0x8\n"
      "str d8, [x23], #0x8\n"
      "tbz x9, #0, 84f\n"
      "st1 { v4.s }[2], [x27]\n"
      "st1 { v8.s }[2], [x23]\n"
      "b 84f\n"
      "83:"  // Height 2: Partial direct writeback: partial_1_0
      "str s4, [x27, #0x0]\n"
      "str s8, [x23, #0x0]\n"
      "84:"  // Height 2: Partial direct writeback: Done
      "b 86f\n"
      "85:"  // Height 2: Full writeback
      "str q4, [x27, #0x0]\n"
      "str q14, [x27, #0x10]\n"
      "str q15, [x27, #0x20]\n"
      "str q16, [x27, #0x30]\n"
      "str q17, [x27, #0x40]\n"
      "str q18, [x27, #0x50]\n"
      "add x27, x27, #0x60\n"
      "str q8, [x23, #0x0]\n"
      "str q9, [x23, #0x10]\n"
      "str q10, [x23, #0x20]\n"
      "str q11, [x23, #0x30]\n"
      "str q12, [x23, #0x40]\n"
      "str q13, [x23, #0x50]\n"
      "86:"  // Height 2: Writeback done
      "subs x9, x9, #0x18\n"
      "bgt 45b\n"
      "b 174f\n"
      "87:"  // Height 3
      "mov x10, %x[bias]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "88:"  // Height 3: Column loop
      "cbz x10, 89f\n"
      "ldr q8, [x10, #0x0]\n"
      "ldr q9, [x10, #0x10]\n"
      "zip2 v14.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "zip2 v15.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "ldr q12, [x10, #0x40]\n"
      "ldr q13, [x10, #0x50]\n"
      "zip2 v16.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v17.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "add x10, x10, #0x60\n"
      "zip2 v18.2d, v12.2d, v12.2d\n"
      "zip1 v12.2d, v12.2d, v12.2d\n"
      "zip2 v19.2d, v13.2d, v13.2d\n"
      "zip1 v13.2d, v13.2d, v13.2d\n"
      "mov v20.16b, v8.16b\n"
      "mov v26.16b, v14.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v27.16b, v15.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v28.16b, v16.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v29.16b, v17.16b\n"
      "mov v24.16b, v12.16b\n"
      "mov v30.16b, v18.16b\n"
      "mov v25.16b, v13.16b\n"
      "mov v31.16b, v19.16b\n"
      "b 105f\n"
      "89:"  // Height 3: no bias
      "tbz %x[flags], #0, 104f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20, LSL #2\n"
      "cmp x9, #0x18\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 102f\n"
      "tbz x9, #4, 93f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v23.4s }, [x22], #0x10\n"
      "ld1 { v12.4s }, [x27], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v24.4s }, [x22], #0x10\n"
      "tbz x9, #2, 91f\n"
      "ld1 { v13.4s }, [x27], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v25.4s }, [x22], #0x10\n"
      "tbz x9, #1, 90f\n"
      "ldr d20, [x27], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "mov x20, #0x58\n"
      "ldr d4, [x22], #0x8\n"
      "tbz x9, #0, 101f\n"
      "ld1 { v20.s }[2], [x27]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v4.s }[2], [x22]\n"
      "b 101f\n"
      "90:"  // Height 3: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x9, #0, 101f\n"
      "ldr s20, [x27, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s4, [x22, #0x0]\n"
      "b 101f\n"
      "91:"  // Height 3: Partial accumulate: partial_2_16
      "tbz x9, #1, 92f\n"
      "ldr d13, [x27], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "mov x20, #0x48\n"
      "ldr d25, [x22], #0x8\n"
      "tbz x9, #0, 101f\n"
      "ld1 { v13.s }[2], [x27]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v25.s }[2], [x22]\n"
      "b 101f\n"
      "92:"  // Height 3: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x9, #0, 101f\n"
      "ldr s13, [x27, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s25, [x22, #0x0]\n"
      "b 101f\n"
      "93:"  // Height 3: Partial accumulate: partial_8_0
      "tbz x9, #3, 97f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "tbz x9, #2, 95f\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v23.4s }, [x22], #0x10\n"
      "tbz x9, #1, 94f\n"
      "ldr d12, [x27], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d24, [x22], #0x8\n"
      "tbz x9, #0, 101f\n"
      "ld1 { v12.s }[2], [x27]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v24.s }[2], [x22]\n"
      "b 101f\n"
      "94:"  // Height 3: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x9, #0, 101f\n"
      "ldr s12, [x27, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s24, [x22, #0x0]\n"
      "b 101f\n"
      "95:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x9, #1, 96f\n"
      "ldr d11, [x27], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d23, [x22], #0x8\n"
      "tbz x9, #0, 101f\n"
      "ld1 { v11.s }[2], [x27]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "b 101f\n"
      "96:"  // Height 3: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x9, #0, 101f\n"
      "ldr s11, [x27, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "b 101f\n"
      "97:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x9, #2, 99f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "tbz x9, #1, 98f\n"
      "ldr d10, [x27], #0x8\n"
      "ldr d15, [x23], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d22, [x22], #0x8\n"
      "tbz x9, #0, 101f\n"
      "ld1 { v10.s }[2], [x27]\n"
      "ld1 { v15.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "b 101f\n"
      "98:"  // Height 3: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x9, #0, 101f\n"
      "ldr s10, [x27, #0x0]\n"
      "ldr s15, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "b 101f\n"
      "99:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x9, #1, 100f\n"
      "ldr d9, [x27], #0x8\n"
      "ldr d14, [x23], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "tbz x9, #0, 101f\n"
      "ld1 { v9.s }[2], [x27]\n"
      "ld1 { v14.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "b 101f\n"
      "100:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s9, [x27, #0x0]\n"
      "ldr s14, [x23, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s21, [x22, #0x0]\n"
      "101:"  // Height 3: Partial accumulate: Done
      "sub x27, x27, x20\n"
      "b 103f\n"
      "102:"  // Height 3: full accumulate
      "ldr q9, [x27, #0x0]\n"
      "ldr q10, [x27, #0x10]\n"
      "ldr q11, [x27, #0x20]\n"
      "ldr q12, [x27, #0x30]\n"
      "ldr q13, [x27, #0x40]\n"
      "ldr q20, [x27, #0x50]\n"
      "ldr q14, [x23, #0x0]\n"
      "ldr q15, [x23, #0x10]\n"
      "ldr q16, [x23, #0x20]\n"
      "ldr q17, [x23, #0x30]\n"
      "ldr q18, [x23, #0x40]\n"
      "ldr q19, [x23, #0x50]\n"
      "ldr q21, [x22, #0x0]\n"
      "ldr q22, [x22, #0x10]\n"
      "ldr q23, [x22, #0x20]\n"
      "ldr q24, [x22, #0x30]\n"
      "ldr q25, [x22, #0x40]\n"
      "ldr q4, [x22, #0x50]\n"
      "103:"  // Height 3: MMLA fixup
      "zip1 v8.2d, v9.2d, v14.2d\n"
      "zip2 v14.2d, v9.2d, v14.2d\n"
      "zip1 v9.2d, v10.2d, v15.2d\n"
      "zip2 v15.2d, v10.2d, v15.2d\n"
      "zip1 v10.2d, v11.2d, v16.2d\n"
      "zip2 v16.2d, v11.2d, v16.2d\n"
      "zip1 v11.2d, v12.2d, v17.2d\n"
      "zip2 v17.2d, v12.2d, v17.2d\n"
      "zip1 v12.2d, v13.2d, v18.2d\n"
      "zip2 v18.2d, v13.2d, v18.2d\n"
      "zip1 v13.2d, v20.2d, v19.2d\n"
      "zip2 v19.2d, v20.2d, v19.2d\n"
      "zip1 v20.2d, v21.2d, v26.2d\n"
      "zip2 v26.2d, v21.2d, v26.2d\n"
      "zip1 v21.2d, v22.2d, v27.2d\n"
      "zip2 v27.2d, v22.2d, v27.2d\n"
      "zip1 v22.2d, v23.2d, v28.2d\n"
      "zip2 v28.2d, v23.2d, v28.2d\n"
      "zip1 v23.2d, v24.2d, v29.2d\n"
      "zip2 v29.2d, v24.2d, v29.2d\n"
      "zip1 v24.2d, v25.2d, v30.2d\n"
      "zip2 v30.2d, v25.2d, v30.2d\n"
      "zip1 v25.2d, v4.2d, v31.2d\n"
      "zip2 v31.2d, v4.2d, v31.2d\n"
      "b 105f\n"
      "104:"  // Height 3: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "movi v31.16b, #0x0\n"
      "105:"  // Height 3: setup done
      "mov x26, #0x0\n"
      "106:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 107f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "cbnz x26, 108f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "b 108f\n"
      "107:"  // Height 3: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "108:"  // Height 3: input setup done
      "cmp x25, #0x4\n"
      "blt 111f\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      "ld1 { v1.4s }, [x23], #0x10\n"
      "cmp x25, #0x8\n"
      "ld1 { v2.4s }, [x22], #0x10\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "blt 110f\n"
      "109:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "sub x25, x25, #0x4\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      ".inst 0x6e44ec54  // bfmmla v20.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec5a  // bfmmla v26.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      "cmp x25, #0x8\n"
      ".inst 0x6e46ec55  // bfmmla v21.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x6e47ec5b  // bfmmla v27.4s, v2.8h, v7.8h\n"
      "ldr q3, [x28, #0x70]\n"
      ".inst 0x6e44ec0a  // bfmmla v10.4s, v0.8h, v4.8h\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "ld1 { v1.4s }, [x23], #0x10\n"
      ".inst 0x6e44ec56  // bfmmla v22.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x80]\n"
      ".inst 0x6e45ec10  // bfmmla v16.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec5c  // bfmmla v28.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x90]\n"
      ".inst 0x6e46ec0b  // bfmmla v11.4s, v0.8h, v6.8h\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x6e46ec57  // bfmmla v23.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0xa0]\n"
      ".inst 0x6e43ec11  // bfmmla v17.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec5d  // bfmmla v29.4s, v2.8h, v3.8h\n"
      "ldr q3, [x28, #0xb0]\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e44ec0c  // bfmmla v12.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec58  // bfmmla v24.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x0]\n"
      ".inst 0x6e45ec12  // bfmmla v18.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec5e  // bfmmla v30.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x10]\n"
      ".inst 0x6e46ec0d  // bfmmla v13.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec59  // bfmmla v25.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0x20]\n"
      ".inst 0x6e43ec13  // bfmmla v19.4s, v0.8h, v3.8h\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      ".inst 0x6e43ec5f  // bfmmla v31.4s, v2.8h, v3.8h\n"
      "ld1 { v2.4s }, [x22], #0x10\n"
      "ldr q7, [x28, #0x30]\n"
      "bge 109b\n"
      "110:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "sub x25, x25, #0x4\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      ".inst 0x6e44ec54  // bfmmla v20.4s, v2.8h, v4.8h\n"
      "ldr q3, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec5a  // bfmmla v26.4s, v2.8h, v5.8h\n"
      "ldr q4, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x6e46ec55  // bfmmla v21.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6e47ec5b  // bfmmla v27.4s, v2.8h, v7.8h\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x6e43ec0a  // bfmmla v10.4s, v0.8h, v3.8h\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x6e43ec56  // bfmmla v22.4s, v2.8h, v3.8h\n"
      "ldr q5, [x28, #0x80]\n"
      ".inst 0x6e44ec10  // bfmmla v16.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5c  // bfmmla v28.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x90]\n"
      ".inst 0x6e46ec0b  // bfmmla v11.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec57  // bfmmla v23.4s, v2.8h, v6.8h\n"
      "ldr q3, [x28, #0xa0]\n"
      ".inst 0x6e41ec11  // bfmmla v17.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5d  // bfmmla v29.4s, v2.8h, v1.8h\n"
      "ldr q1, [x28, #0xb0]\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e45ec0c  // bfmmla v12.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec58  // bfmmla v24.4s, v2.8h, v5.8h\n"
      ".inst 0x6e44ec12  // bfmmla v18.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5e  // bfmmla v30.4s, v2.8h, v4.8h\n"
      ".inst 0x6e43ec0d  // bfmmla v13.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec59  // bfmmla v25.4s, v2.8h, v3.8h\n"
      ".inst 0x6e41ec13  // bfmmla v19.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5f  // bfmmla v31.4s, v2.8h, v1.8h\n"
      "111:"  // Height 3: Multiply loop: Main loop skip
      "cbz x25, 114f\n"
      "cbz x25, 114f\n"
      "tbz x25, #1, 112f\n"
      "ldr d0, [x24], #0x8\n"
      "ldr d1, [x23], #0x8\n"
      "ldr d2, [x22], #0x8\n"
      "tbz x25, #0, 113f\n"
      "ld1 { v0.s }[2], [x24]\n"
      "ld1 { v1.s }[2], [x23]\n"
      "ld1 { v2.s }[2], [x22]\n"
      "b 113f\n"
      "112:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x24, #0x0]\n"
      "ldr s1, [x23, #0x0]\n"
      "ldr s2, [x22, #0x0]\n"
      "113:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q5, [x28, #0x0]\n"
      "ldr q4, [x28, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      "ldr q3, [x28, #0x20]\n"
      "ldr q1, [x28, #0x30]\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      ".inst 0x6e45ec08  // bfmmla v8.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec54  // bfmmla v20.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x40]\n"
      ".inst 0x6e44ec0e  // bfmmla v14.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5a  // bfmmla v26.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x50]\n"
      ".inst 0x6e43ec09  // bfmmla v9.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec55  // bfmmla v21.4s, v2.8h, v3.8h\n"
      "ldr q3, [x28, #0x60]\n"
      ".inst 0x6e41ec0f  // bfmmla v15.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5b  // bfmmla v27.4s, v2.8h, v1.8h\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x6e45ec0a  // bfmmla v10.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec56  // bfmmla v22.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x80]\n"
      ".inst 0x6e44ec10  // bfmmla v16.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5c  // bfmmla v28.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x90]\n"
      ".inst 0x6e43ec0b  // bfmmla v11.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec57  // bfmmla v23.4s, v2.8h, v3.8h\n"
      "ldr q3, [x28, #0xa0]\n"
      ".inst 0x6e41ec11  // bfmmla v17.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5d  // bfmmla v29.4s, v2.8h, v1.8h\n"
      "ldr q1, [x28, #0xb0]\n"
      ".inst 0x6e45ec0c  // bfmmla v12.4s, v0.8h, v5.8h\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e45ec58  // bfmmla v24.4s, v2.8h, v5.8h\n"
      ".inst 0x6e44ec12  // bfmmla v18.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5e  // bfmmla v30.4s, v2.8h, v4.8h\n"
      ".inst 0x6e43ec0d  // bfmmla v13.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec59  // bfmmla v25.4s, v2.8h, v3.8h\n"
      ".inst 0x6e41ec13  // bfmmla v19.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5f  // bfmmla v31.4s, v2.8h, v1.8h\n"
      "114:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 106b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 v4.2d, v8.2d, v14.2d\n"
      "uzp2 v8.2d, v8.2d, v14.2d\n"
      "uzp1 v14.2d, v9.2d, v15.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "uzp2 v9.2d, v9.2d, v15.2d\n"
      "uzp1 v15.2d, v10.2d, v16.2d\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp2 v10.2d, v10.2d, v16.2d\n"
      "uzp1 v16.2d, v11.2d, v17.2d\n"
      "uzp2 v11.2d, v11.2d, v17.2d\n"
      "uzp1 v17.2d, v12.2d, v18.2d\n"
      "uzp2 v12.2d, v12.2d, v18.2d\n"
      "uzp1 v18.2d, v13.2d, v19.2d\n"
      "uzp2 v13.2d, v13.2d, v19.2d\n"
      "uzp1 v20.2d, v20.2d, v26.2d\n"
      "uzp1 v21.2d, v21.2d, v27.2d\n"
      "uzp1 v22.2d, v22.2d, v28.2d\n"
      "uzp1 v23.2d, v23.2d, v29.2d\n"
      "uzp1 v24.2d, v24.2d, v30.2d\n"
      "uzp1 v25.2d, v25.2d, v31.2d\n"
      "tbz %x[flags], #1, 115f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v1.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmin v4.4s, v4.4s, v1.4s\n"
      "fmin v14.4s, v14.4s, v1.4s\n"
      "fmin v15.4s, v15.4s, v1.4s\n"
      "fmin v16.4s, v16.4s, v1.4s\n"
      "fmin v17.4s, v17.4s, v1.4s\n"
      "fmin v18.4s, v18.4s, v1.4s\n"
      "fmin v8.4s, v8.4s, v1.4s\n"
      "fmin v9.4s, v9.4s, v1.4s\n"
      "fmin v10.4s, v10.4s, v1.4s\n"
      "fmin v11.4s, v11.4s, v1.4s\n"
      "fmin v12.4s, v12.4s, v1.4s\n"
      "fmin v13.4s, v13.4s, v1.4s\n"
      "fmin v20.4s, v20.4s, v1.4s\n"
      "fmin v21.4s, v21.4s, v1.4s\n"
      "fmin v22.4s, v22.4s, v1.4s\n"
      "fmin v23.4s, v23.4s, v1.4s\n"
      "fmin v24.4s, v24.4s, v1.4s\n"
      "fmin v25.4s, v25.4s, v1.4s\n"
      "fmax v4.4s, v4.4s, v0.4s\n"
      "fmax v14.4s, v14.4s, v0.4s\n"
      "fmax v15.4s, v15.4s, v0.4s\n"
      "fmax v16.4s, v16.4s, v0.4s\n"
      "fmax v17.4s, v17.4s, v0.4s\n"
      "fmax v18.4s, v18.4s, v0.4s\n"
      "fmax v8.4s, v8.4s, v0.4s\n"
      "fmax v9.4s, v9.4s, v0.4s\n"
      "fmax v10.4s, v10.4s, v0.4s\n"
      "fmax v11.4s, v11.4s, v0.4s\n"
      "fmax v12.4s, v12.4s, v0.4s\n"
      "fmax v13.4s, v13.4s, v0.4s\n"
      "fmax v20.4s, v20.4s, v0.4s\n"
      "fmax v21.4s, v21.4s, v0.4s\n"
      "fmax v22.4s, v22.4s, v0.4s\n"
      "fmax v23.4s, v23.4s, v0.4s\n"
      "fmax v24.4s, v24.4s, v0.4s\n"
      "fmax v25.4s, v25.4s, v0.4s\n"
      "115:"  // Height 3: No activation
      "cmp x9, #0x18\n"
      "bge 128f\n"
      "tbz x9, #4, 119f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v14.4s }, [x27], #0x10\n"
      "st1 { v15.4s }, [x27], #0x10\n"
      "st1 { v16.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v9.4s }, [x23], #0x10\n"
      "st1 { v10.4s }, [x23], #0x10\n"
      "st1 { v11.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v23.4s }, [x22], #0x10\n"
      "tbz x9, #2, 117f\n"
      "st1 { v17.4s }, [x27], #0x10\n"
      "st1 { v12.4s }, [x23], #0x10\n"
      "st1 { v24.4s }, [x22], #0x10\n"
      "tbz x9, #1, 116f\n"
      "str d18, [x27], #0x8\n"
      "str d13, [x23], #0x8\n"
      "str d25, [x22], #0x8\n"
      "tbz x9, #0, 127f\n"
      "st1 { v18.s }[2], [x27]\n"
      "st1 { v13.s }[2], [x23]\n"
      "st1 { v25.s }[2], [x22]\n"
      "b 127f\n"
      "116:"  // Height 3: Partial direct writeback: partial_1_20
      "tbz x9, #0, 127f\n"
      "str s18, [x27, #0x0]\n"
      "str s13, [x23, #0x0]\n"
      "str s25, [x22, #0x0]\n"
      "b 127f\n"
      "117:"  // Height 3: Partial direct writeback: partial_2_16
      "tbz x9, #1, 118f\n"
      "str d17, [x27], #0x8\n"
      "str d12, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "tbz x9, #0, 127f\n"
      "st1 { v17.s }[2], [x27]\n"
      "st1 { v12.s }[2], [x23]\n"
      "st1 { v24.s }[2], [x22]\n"
      "b 127f\n"
      "118:"  // Height 3: Partial direct writeback: partial_1_16
      "tbz x9, #0, 127f\n"
      "str s17, [x27, #0x0]\n"
      "str s12, [x23, #0x0]\n"
      "str s24, [x22, #0x0]\n"
      "b 127f\n"
      "119:"  // Height 3: Partial direct writeback: partial_8_0
      "tbz x9, #3, 123f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v14.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v9.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "tbz x9, #2, 121f\n"
      "st1 { v15.4s }, [x27], #0x10\n"
      "st1 { v10.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "tbz x9, #1, 120f\n"
      "str d16, [x27], #0x8\n"
      "str d11, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "tbz x9, #0, 127f\n"
      "st1 { v16.s }[2], [x27]\n"
      "st1 { v11.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "b 127f\n"
      "120:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x9, #0, 127f\n"
      "str s16, [x27, #0x0]\n"
      "str s11, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "b 127f\n"
      "121:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x9, #1, 122f\n"
      "str d15, [x27], #0x8\n"
      "str d10, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "tbz x9, #0, 127f\n"
      "st1 { v15.s }[2], [x27]\n"
      "st1 { v10.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "b 127f\n"
      "122:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x9, #0, 127f\n"
      "str s15, [x27, #0x0]\n"
      "str s10, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "b 127f\n"
      "123:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x9, #2, 125f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "tbz x9, #1, 124f\n"
      "str d14, [x27], #0x8\n"
      "str d9, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "tbz x9, #0, 127f\n"
      "st1 { v14.s }[2], [x27]\n"
      "st1 { v9.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "b 127f\n"
      "124:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x9, #0, 127f\n"
      "str s14, [x27, #0x0]\n"
      "str s9, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "b 127f\n"
      "125:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x9, #1, 126f\n"
      "str d4, [x27], #0x8\n"
      "str d8, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "tbz x9, #0, 127f\n"
      "st1 { v4.s }[2], [x27]\n"
      "st1 { v8.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "b 127f\n"
      "126:"  // Height 3: Partial direct writeback: partial_1_0
      "str s4, [x27, #0x0]\n"
      "str s8, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "127:"  // Height 3: Partial direct writeback: Done
      "b 129f\n"
      "128:"  // Height 3: Full writeback
      "str q4, [x27, #0x0]\n"
      "str q14, [x27, #0x10]\n"
      "str q15, [x27, #0x20]\n"
      "str q16, [x27, #0x30]\n"
      "str q17, [x27, #0x40]\n"
      "str q18, [x27, #0x50]\n"
      "add x27, x27, #0x60\n"
      "str q8, [x23, #0x0]\n"
      "str q9, [x23, #0x10]\n"
      "str q10, [x23, #0x20]\n"
      "str q11, [x23, #0x30]\n"
      "str q12, [x23, #0x40]\n"
      "str q13, [x23, #0x50]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x22, #0x40]\n"
      "str q25, [x22, #0x50]\n"
      "129:"  // Height 3: Writeback done
      "subs x9, x9, #0x18\n"
      "bgt 88b\n"
      "b 174f\n"
      "130:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x10\n"
      "mov x10, %x[bias]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "131:"  // Height 4: Column loop
      "cbz x10, 132f\n"
      "ldr q8, [x10, #0x0]\n"
      "ldr q9, [x10, #0x10]\n"
      "zip2 v14.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "zip2 v15.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "ldr q12, [x10, #0x40]\n"
      "ldr q13, [x10, #0x50]\n"
      "zip2 v16.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v17.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "add x10, x10, #0x60\n"
      "zip2 v18.2d, v12.2d, v12.2d\n"
      "zip1 v12.2d, v12.2d, v12.2d\n"
      "zip2 v19.2d, v13.2d, v13.2d\n"
      "zip1 v13.2d, v13.2d, v13.2d\n"
      "mov v20.16b, v8.16b\n"
      "mov v26.16b, v14.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v27.16b, v15.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v28.16b, v16.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v29.16b, v17.16b\n"
      "mov v24.16b, v12.16b\n"
      "mov v30.16b, v18.16b\n"
      "mov v25.16b, v13.16b\n"
      "mov v31.16b, v19.16b\n"
      "b 148f\n"
      "132:"  // Height 4: no bias
      "tbz %x[flags], #0, 147f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "cmp x9, #0x18\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 145f\n"
      "tbz x9, #4, 136f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v23.4s }, [x22], #0x10\n"
      "ld1 { v28.4s }, [x21], #0x10\n"
      "ld1 { v12.4s }, [x27], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v24.4s }, [x22], #0x10\n"
      "ld1 { v29.4s }, [x21], #0x10\n"
      "tbz x9, #2, 134f\n"
      "ld1 { v13.4s }, [x27], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v25.4s }, [x22], #0x10\n"
      "ld1 { v30.4s }, [x21], #0x10\n"
      "tbz x9, #1, 133f\n"
      "ldr d20, [x27], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "mov x20, #0x58\n"
      "ldr d4, [x22], #0x8\n"
      "ldr d31, [x21], #0x8\n"
      "tbz x9, #0, 144f\n"
      "ld1 { v20.s }[2], [x27]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v4.s }[2], [x22]\n"
      "ld1 { v31.s }[2], [x21]\n"
      "b 144f\n"
      "133:"  // Height 4: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x9, #0, 144f\n"
      "ldr s20, [x27, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s4, [x22, #0x0]\n"
      "ldr s31, [x21, #0x0]\n"
      "b 144f\n"
      "134:"  // Height 4: Partial accumulate: partial_2_16
      "tbz x9, #1, 135f\n"
      "ldr d13, [x27], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "mov x20, #0x48\n"
      "ldr d25, [x22], #0x8\n"
      "ldr d30, [x21], #0x8\n"
      "tbz x9, #0, 144f\n"
      "ld1 { v13.s }[2], [x27]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v25.s }[2], [x22]\n"
      "ld1 { v30.s }[2], [x21]\n"
      "b 144f\n"
      "135:"  // Height 4: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x9, #0, 144f\n"
      "ldr s13, [x27, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s25, [x22, #0x0]\n"
      "ldr s30, [x21, #0x0]\n"
      "b 144f\n"
      "136:"  // Height 4: Partial accumulate: partial_8_0
      "tbz x9, #3, 140f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v10.4s }, [x27], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "tbz x9, #2, 138f\n"
      "ld1 { v11.4s }, [x27], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v23.4s }, [x22], #0x10\n"
      "ld1 { v28.4s }, [x21], #0x10\n"
      "tbz x9, #1, 137f\n"
      "ldr d12, [x27], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d24, [x22], #0x8\n"
      "ldr d29, [x21], #0x8\n"
      "tbz x9, #0, 144f\n"
      "ld1 { v12.s }[2], [x27]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v24.s }[2], [x22]\n"
      "ld1 { v29.s }[2], [x21]\n"
      "b 144f\n"
      "137:"  // Height 4: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x9, #0, 144f\n"
      "ldr s12, [x27, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s24, [x22, #0x0]\n"
      "ldr s29, [x21, #0x0]\n"
      "b 144f\n"
      "138:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x9, #1, 139f\n"
      "ldr d11, [x27], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d28, [x21], #0x8\n"
      "tbz x9, #0, 144f\n"
      "ld1 { v11.s }[2], [x27]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v28.s }[2], [x21]\n"
      "b 144f\n"
      "139:"  // Height 4: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x9, #0, 144f\n"
      "ldr s11, [x27, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s28, [x21, #0x0]\n"
      "b 144f\n"
      "140:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x9, #2, 142f\n"
      "ld1 { v9.4s }, [x27], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "tbz x9, #1, 141f\n"
      "ldr d10, [x27], #0x8\n"
      "ldr d15, [x23], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "tbz x9, #0, 144f\n"
      "ld1 { v10.s }[2], [x27]\n"
      "ld1 { v15.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "b 144f\n"
      "141:"  // Height 4: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x9, #0, 144f\n"
      "ldr s10, [x27, #0x0]\n"
      "ldr s15, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "b 144f\n"
      "142:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x9, #1, 143f\n"
      "ldr d9, [x27], #0x8\n"
      "ldr d14, [x23], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "tbz x9, #0, 144f\n"
      "ld1 { v9.s }[2], [x27]\n"
      "ld1 { v14.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "b 144f\n"
      "143:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s9, [x27, #0x0]\n"
      "ldr s14, [x23, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "144:"  // Height 4: Partial accumulate: Done
      "sub x27, x27, x20\n"
      "b 146f\n"
      "145:"  // Height 4: full accumulate
      "ldr q9, [x27, #0x0]\n"
      "ldr q10, [x27, #0x10]\n"
      "ldr q11, [x27, #0x20]\n"
      "ldr q12, [x27, #0x30]\n"
      "ldr q13, [x27, #0x40]\n"
      "ldr q20, [x27, #0x50]\n"
      "ldr q14, [x23, #0x0]\n"
      "ldr q15, [x23, #0x10]\n"
      "ldr q16, [x23, #0x20]\n"
      "ldr q17, [x23, #0x30]\n"
      "ldr q18, [x23, #0x40]\n"
      "ldr q19, [x23, #0x50]\n"
      "ldr q21, [x22, #0x0]\n"
      "ldr q22, [x22, #0x10]\n"
      "ldr q23, [x22, #0x20]\n"
      "ldr q24, [x22, #0x30]\n"
      "ldr q25, [x22, #0x40]\n"
      "ldr q4, [x22, #0x50]\n"
      "ldr q26, [x21, #0x0]\n"
      "ldr q27, [x21, #0x10]\n"
      "ldr q28, [x21, #0x20]\n"
      "ldr q29, [x21, #0x30]\n"
      "ldr q30, [x21, #0x40]\n"
      "ldr q31, [x21, #0x50]\n"
      "146:"  // Height 4: MMLA fixup
      "zip1 v8.2d, v9.2d, v14.2d\n"
      "zip2 v14.2d, v9.2d, v14.2d\n"
      "zip1 v9.2d, v10.2d, v15.2d\n"
      "zip2 v15.2d, v10.2d, v15.2d\n"
      "zip1 v10.2d, v11.2d, v16.2d\n"
      "zip2 v16.2d, v11.2d, v16.2d\n"
      "zip1 v11.2d, v12.2d, v17.2d\n"
      "zip2 v17.2d, v12.2d, v17.2d\n"
      "zip1 v12.2d, v13.2d, v18.2d\n"
      "zip2 v18.2d, v13.2d, v18.2d\n"
      "zip1 v13.2d, v20.2d, v19.2d\n"
      "zip2 v19.2d, v20.2d, v19.2d\n"
      "zip1 v20.2d, v21.2d, v26.2d\n"
      "zip2 v26.2d, v21.2d, v26.2d\n"
      "zip1 v21.2d, v22.2d, v27.2d\n"
      "zip2 v27.2d, v22.2d, v27.2d\n"
      "zip1 v22.2d, v23.2d, v28.2d\n"
      "zip2 v28.2d, v23.2d, v28.2d\n"
      "zip1 v23.2d, v24.2d, v29.2d\n"
      "zip2 v29.2d, v24.2d, v29.2d\n"
      "zip1 v24.2d, v25.2d, v30.2d\n"
      "zip2 v30.2d, v25.2d, v30.2d\n"
      "zip1 v25.2d, v4.2d, v31.2d\n"
      "zip2 v31.2d, v4.2d, v31.2d\n"
      "b 148f\n"
      "147:"  // Height 4: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "movi v31.16b, #0x0\n"
      "148:"  // Height 4: setup done
      "mov x26, #0x0\n"
      "149:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 150f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "ldr x21, [x20, #0x18]\n"
      "cbnz x26, 151f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "add x21, x21, x20, LSL #2\n"
      "b 151f\n"
      "150:"  // Height 4: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "add x21, x22, x21, LSL #2\n"
      "151:"  // Height 4: input setup done
      "cmp x25, #0x4\n"
      "blt 154f\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      "ld1 { v2.4s }, [x22], #0x10\n"
      "cmp x25, #0x8\n"
      "ld1 { v1.4s }, [x23], #0x10\n"
      "ld1 { v3.4s }, [x21], #0x10\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "blt 153f\n"
      "152:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x25, x25, #0x4\n"
      "cmp x25, #0x8\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x6e44ec54  // bfmmla v20.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "ld1 { v1.4s }, [x23], #0x10\n"
      ".inst 0x6e45ec5a  // bfmmla v26.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec55  // bfmmla v21.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x6e47ec5b  // bfmmla v27.4s, v2.8h, v7.8h\n"
      "ldr q7, [x28, #0x70]\n"
      ".inst 0x6e44ec0a  // bfmmla v10.4s, v0.8h, v4.8h\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "ld1 { v3.4s }, [x21], #0x10\n"
      ".inst 0x6e44ec56  // bfmmla v22.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x80]\n"
      ".inst 0x6e45ec10  // bfmmla v16.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec5c  // bfmmla v28.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x90]\n"
      ".inst 0x6e46ec0b  // bfmmla v11.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec57  // bfmmla v23.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0xa0]\n"
      ".inst 0x6e47ec11  // bfmmla v17.4s, v0.8h, v7.8h\n"
      ".inst 0x6e47ec5d  // bfmmla v29.4s, v2.8h, v7.8h\n"
      "ldr q7, [x28, #0xb0]\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e44ec0c  // bfmmla v12.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec58  // bfmmla v24.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x0]\n"
      ".inst 0x6e45ec12  // bfmmla v18.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec5e  // bfmmla v30.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x10]\n"
      ".inst 0x6e46ec0d  // bfmmla v13.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec59  // bfmmla v25.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0x20]\n"
      ".inst 0x6e47ec13  // bfmmla v19.4s, v0.8h, v7.8h\n"
      "ld1 { v0.4s }, [x24], #0x10\n"
      ".inst 0x6e47ec5f  // bfmmla v31.4s, v2.8h, v7.8h\n"
      "ld1 { v2.4s }, [x22], #0x10\n"
      "ldr q7, [x28, #0x30]\n"
      "bge 152b\n"
      "153:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x25, x25, #0x4\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x6e44ec08  // bfmmla v8.4s, v0.8h, v4.8h\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6e44ec54  // bfmmla v20.4s, v2.8h, v4.8h\n"
      "ldr q3, [x28, #0x40]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x6e45ec5a  // bfmmla v26.4s, v2.8h, v5.8h\n"
      "ldr q4, [x28, #0x50]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x6e46ec55  // bfmmla v21.4s, v2.8h, v6.8h\n"
      "ldr q6, [x28, #0x60]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      ".inst 0x6e47ec5b  // bfmmla v27.4s, v2.8h, v7.8h\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x6e43ec0a  // bfmmla v10.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec56  // bfmmla v22.4s, v2.8h, v3.8h\n"
      "ldr q5, [x28, #0x80]\n"
      ".inst 0x6e44ec10  // bfmmla v16.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5c  // bfmmla v28.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x90]\n"
      ".inst 0x6e46ec0b  // bfmmla v11.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec57  // bfmmla v23.4s, v2.8h, v6.8h\n"
      "ldr q3, [x28, #0xa0]\n"
      ".inst 0x6e41ec11  // bfmmla v17.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5d  // bfmmla v29.4s, v2.8h, v1.8h\n"
      "ldr q1, [x28, #0xb0]\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e45ec0c  // bfmmla v12.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec58  // bfmmla v24.4s, v2.8h, v5.8h\n"
      ".inst 0x6e44ec12  // bfmmla v18.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5e  // bfmmla v30.4s, v2.8h, v4.8h\n"
      ".inst 0x6e43ec0d  // bfmmla v13.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec59  // bfmmla v25.4s, v2.8h, v3.8h\n"
      ".inst 0x6e41ec13  // bfmmla v19.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5f  // bfmmla v31.4s, v2.8h, v1.8h\n"
      "154:"  // Height 4: Multiply loop: Main loop skip
      "cbz x25, 157f\n"
      "cbz x25, 157f\n"
      "tbz x25, #1, 155f\n"
      "ldr d0, [x24], #0x8\n"
      "ldr d1, [x23], #0x8\n"
      "ldr d2, [x22], #0x8\n"
      "ldr d3, [x21], #0x8\n"
      "tbz x25, #0, 156f\n"
      "ld1 { v0.s }[2], [x24]\n"
      "ld1 { v1.s }[2], [x23]\n"
      "ld1 { v2.s }[2], [x22]\n"
      "ld1 { v3.s }[2], [x21]\n"
      "b 156f\n"
      "155:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x24, #0x0]\n"
      "ldr s1, [x23, #0x0]\n"
      "ldr s2, [x22, #0x0]\n"
      "ldr s3, [x21, #0x0]\n"
      "156:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q5, [x28, #0x0]\n"
      "ldr q4, [x28, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q6, [x28, #0x30]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x6e45ec08  // bfmmla v8.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec54  // bfmmla v20.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x40]\n"
      ".inst 0x6e44ec0e  // bfmmla v14.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5a  // bfmmla v26.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x50]\n"
      ".inst 0x6e47ec09  // bfmmla v9.4s, v0.8h, v7.8h\n"
      ".inst 0x6e47ec55  // bfmmla v21.4s, v2.8h, v7.8h\n"
      "ldr q3, [x28, #0x60]\n"
      ".inst 0x6e46ec0f  // bfmmla v15.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec5b  // bfmmla v27.4s, v2.8h, v6.8h\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x6e45ec0a  // bfmmla v10.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec56  // bfmmla v22.4s, v2.8h, v5.8h\n"
      "ldr q5, [x28, #0x80]\n"
      ".inst 0x6e44ec10  // bfmmla v16.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5c  // bfmmla v28.4s, v2.8h, v4.8h\n"
      "ldr q4, [x28, #0x90]\n"
      ".inst 0x6e43ec0b  // bfmmla v11.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec57  // bfmmla v23.4s, v2.8h, v3.8h\n"
      "ldr q3, [x28, #0xa0]\n"
      ".inst 0x6e41ec11  // bfmmla v17.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5d  // bfmmla v29.4s, v2.8h, v1.8h\n"
      "ldr q1, [x28, #0xb0]\n"
      "add x28, x28, #0xc0\n"
      ".inst 0x6e45ec0c  // bfmmla v12.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec58  // bfmmla v24.4s, v2.8h, v5.8h\n"
      ".inst 0x6e44ec12  // bfmmla v18.4s, v0.8h, v4.8h\n"
      ".inst 0x6e44ec5e  // bfmmla v30.4s, v2.8h, v4.8h\n"
      ".inst 0x6e43ec0d  // bfmmla v13.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec59  // bfmmla v25.4s, v2.8h, v3.8h\n"
      ".inst 0x6e41ec13  // bfmmla v19.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec5f  // bfmmla v31.4s, v2.8h, v1.8h\n"
      "157:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 149b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 v4.2d, v8.2d, v14.2d\n"
      "add x21, x22, x20, LSL #2\n"
      "uzp2 v8.2d, v8.2d, v14.2d\n"
      "uzp1 v14.2d, v9.2d, v15.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp2 v9.2d, v9.2d, v15.2d\n"
      "uzp1 v15.2d, v10.2d, v16.2d\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp2 v10.2d, v10.2d, v16.2d\n"
      "uzp1 v16.2d, v11.2d, v17.2d\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "uzp2 v11.2d, v11.2d, v17.2d\n"
      "uzp1 v17.2d, v12.2d, v18.2d\n"
      "uzp2 v12.2d, v12.2d, v18.2d\n"
      "uzp1 v18.2d, v13.2d, v19.2d\n"
      "uzp2 v13.2d, v13.2d, v19.2d\n"
      "uzp1 v19.2d, v20.2d, v26.2d\n"
      "uzp2 v20.2d, v20.2d, v26.2d\n"
      "uzp1 v26.2d, v21.2d, v27.2d\n"
      "uzp2 v21.2d, v21.2d, v27.2d\n"
      "uzp1 v27.2d, v22.2d, v28.2d\n"
      "uzp2 v22.2d, v22.2d, v28.2d\n"
      "uzp1 v28.2d, v23.2d, v29.2d\n"
      "uzp2 v23.2d, v23.2d, v29.2d\n"
      "uzp1 v29.2d, v24.2d, v30.2d\n"
      "uzp2 v24.2d, v24.2d, v30.2d\n"
      "uzp1 v30.2d, v25.2d, v31.2d\n"
      "uzp2 v25.2d, v25.2d, v31.2d\n"
      "tbz %x[flags], #1, 158f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v1.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmin v4.4s, v4.4s, v1.4s\n"
      "fmin v14.4s, v14.4s, v1.4s\n"
      "fmin v15.4s, v15.4s, v1.4s\n"
      "fmin v16.4s, v16.4s, v1.4s\n"
      "fmin v17.4s, v17.4s, v1.4s\n"
      "fmin v18.4s, v18.4s, v1.4s\n"
      "fmin v8.4s, v8.4s, v1.4s\n"
      "fmin v9.4s, v9.4s, v1.4s\n"
      "fmin v10.4s, v10.4s, v1.4s\n"
      "fmin v11.4s, v11.4s, v1.4s\n"
      "fmin v12.4s, v12.4s, v1.4s\n"
      "fmin v13.4s, v13.4s, v1.4s\n"
      "fmin v19.4s, v19.4s, v1.4s\n"
      "fmin v26.4s, v26.4s, v1.4s\n"
      "fmin v27.4s, v27.4s, v1.4s\n"
      "fmin v28.4s, v28.4s, v1.4s\n"
      "fmin v29.4s, v29.4s, v1.4s\n"
      "fmin v30.4s, v30.4s, v1.4s\n"
      "fmin v20.4s, v20.4s, v1.4s\n"
      "fmin v21.4s, v21.4s, v1.4s\n"
      "fmin v22.4s, v22.4s, v1.4s\n"
      "fmin v23.4s, v23.4s, v1.4s\n"
      "fmin v24.4s, v24.4s, v1.4s\n"
      "fmin v25.4s, v25.4s, v1.4s\n"
      "fmax v4.4s, v4.4s, v0.4s\n"
      "fmax v14.4s, v14.4s, v0.4s\n"
      "fmax v15.4s, v15.4s, v0.4s\n"
      "fmax v16.4s, v16.4s, v0.4s\n"
      "fmax v17.4s, v17.4s, v0.4s\n"
      "fmax v18.4s, v18.4s, v0.4s\n"
      "fmax v8.4s, v8.4s, v0.4s\n"
      "fmax v9.4s, v9.4s, v0.4s\n"
      "fmax v10.4s, v10.4s, v0.4s\n"
      "fmax v11.4s, v11.4s, v0.4s\n"
      "fmax v12.4s, v12.4s, v0.4s\n"
      "fmax v13.4s, v13.4s, v0.4s\n"
      "fmax v19.4s, v19.4s, v0.4s\n"
      "fmax v26.4s, v26.4s, v0.4s\n"
      "fmax v27.4s, v27.4s, v0.4s\n"
      "fmax v28.4s, v28.4s, v0.4s\n"
      "fmax v29.4s, v29.4s, v0.4s\n"
      "fmax v30.4s, v30.4s, v0.4s\n"
      "fmax v20.4s, v20.4s, v0.4s\n"
      "fmax v21.4s, v21.4s, v0.4s\n"
      "fmax v22.4s, v22.4s, v0.4s\n"
      "fmax v23.4s, v23.4s, v0.4s\n"
      "fmax v24.4s, v24.4s, v0.4s\n"
      "fmax v25.4s, v25.4s, v0.4s\n"
      "158:"  // Height 4: No activation
      "cmp x9, #0x18\n"
      "bge 171f\n"
      "tbz x9, #4, 162f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v14.4s }, [x27], #0x10\n"
      "st1 { v15.4s }, [x27], #0x10\n"
      "st1 { v16.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v9.4s }, [x23], #0x10\n"
      "st1 { v10.4s }, [x23], #0x10\n"
      "st1 { v11.4s }, [x23], #0x10\n"
      "st1 { v19.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x22], #0x10\n"
      "st1 { v27.4s }, [x22], #0x10\n"
      "st1 { v28.4s }, [x22], #0x10\n"
      "st1 { v20.4s }, [x21], #0x10\n"
      "st1 { v21.4s }, [x21], #0x10\n"
      "st1 { v22.4s }, [x21], #0x10\n"
      "st1 { v23.4s }, [x21], #0x10\n"
      "tbz x9, #2, 160f\n"
      "st1 { v17.4s }, [x27], #0x10\n"
      "st1 { v12.4s }, [x23], #0x10\n"
      "st1 { v29.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "tbz x9, #1, 159f\n"
      "str d18, [x27], #0x8\n"
      "str d13, [x23], #0x8\n"
      "str d30, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "tbz x9, #0, 170f\n"
      "st1 { v18.s }[2], [x27]\n"
      "st1 { v13.s }[2], [x23]\n"
      "st1 { v30.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "b 170f\n"
      "159:"  // Height 4: Partial direct writeback: partial_1_20
      "tbz x9, #0, 170f\n"
      "str s18, [x27, #0x0]\n"
      "str s13, [x23, #0x0]\n"
      "str s30, [x22, #0x0]\n"
      "str s25, [x21, #0x0]\n"
      "b 170f\n"
      "160:"  // Height 4: Partial direct writeback: partial_2_16
      "tbz x9, #1, 161f\n"
      "str d17, [x27], #0x8\n"
      "str d12, [x23], #0x8\n"
      "str d29, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "tbz x9, #0, 170f\n"
      "st1 { v17.s }[2], [x27]\n"
      "st1 { v12.s }[2], [x23]\n"
      "st1 { v29.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "b 170f\n"
      "161:"  // Height 4: Partial direct writeback: partial_1_16
      "tbz x9, #0, 170f\n"
      "str s17, [x27, #0x0]\n"
      "str s12, [x23, #0x0]\n"
      "str s29, [x22, #0x0]\n"
      "str s24, [x21, #0x0]\n"
      "b 170f\n"
      "162:"  // Height 4: Partial direct writeback: partial_8_0
      "tbz x9, #3, 166f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v14.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v9.4s }, [x23], #0x10\n"
      "st1 { v19.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x22], #0x10\n"
      "st1 { v20.4s }, [x21], #0x10\n"
      "st1 { v21.4s }, [x21], #0x10\n"
      "tbz x9, #2, 164f\n"
      "st1 { v15.4s }, [x27], #0x10\n"
      "st1 { v10.4s }, [x23], #0x10\n"
      "st1 { v27.4s }, [x22], #0x10\n"
      "st1 { v22.4s }, [x21], #0x10\n"
      "tbz x9, #1, 163f\n"
      "str d16, [x27], #0x8\n"
      "str d11, [x23], #0x8\n"
      "str d28, [x22], #0x8\n"
      "str d23, [x21], #0x8\n"
      "tbz x9, #0, 170f\n"
      "st1 { v16.s }[2], [x27]\n"
      "st1 { v11.s }[2], [x23]\n"
      "st1 { v28.s }[2], [x22]\n"
      "st1 { v23.s }[2], [x21]\n"
      "b 170f\n"
      "163:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x9, #0, 170f\n"
      "str s16, [x27, #0x0]\n"
      "str s11, [x23, #0x0]\n"
      "str s28, [x22, #0x0]\n"
      "str s23, [x21, #0x0]\n"
      "b 170f\n"
      "164:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x9, #1, 165f\n"
      "str d15, [x27], #0x8\n"
      "str d10, [x23], #0x8\n"
      "str d27, [x22], #0x8\n"
      "str d22, [x21], #0x8\n"
      "tbz x9, #0, 170f\n"
      "st1 { v15.s }[2], [x27]\n"
      "st1 { v10.s }[2], [x23]\n"
      "st1 { v27.s }[2], [x22]\n"
      "st1 { v22.s }[2], [x21]\n"
      "b 170f\n"
      "165:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x9, #0, 170f\n"
      "str s15, [x27, #0x0]\n"
      "str s10, [x23, #0x0]\n"
      "str s27, [x22, #0x0]\n"
      "str s22, [x21, #0x0]\n"
      "b 170f\n"
      "166:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x9, #2, 168f\n"
      "st1 { v4.4s }, [x27], #0x10\n"
      "st1 { v8.4s }, [x23], #0x10\n"
      "st1 { v19.4s }, [x22], #0x10\n"
      "st1 { v20.4s }, [x21], #0x10\n"
      "tbz x9, #1, 167f\n"
      "str d14, [x27], #0x8\n"
      "str d9, [x23], #0x8\n"
      "str d26, [x22], #0x8\n"
      "str d21, [x21], #0x8\n"
      "tbz x9, #0, 170f\n"
      "st1 { v14.s }[2], [x27]\n"
      "st1 { v9.s }[2], [x23]\n"
      "st1 { v26.s }[2], [x22]\n"
      "st1 { v21.s }[2], [x21]\n"
      "b 170f\n"
      "167:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x9, #0, 170f\n"
      "str s14, [x27, #0x0]\n"
      "str s9, [x23, #0x0]\n"
      "str s26, [x22, #0x0]\n"
      "str s21, [x21, #0x0]\n"
      "b 170f\n"
      "168:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x9, #1, 169f\n"
      "str d4, [x27], #0x8\n"
      "str d8, [x23], #0x8\n"
      "str d19, [x22], #0x8\n"
      "str d20, [x21], #0x8\n"
      "tbz x9, #0, 170f\n"
      "st1 { v4.s }[2], [x27]\n"
      "st1 { v8.s }[2], [x23]\n"
      "st1 { v19.s }[2], [x22]\n"
      "st1 { v20.s }[2], [x21]\n"
      "b 170f\n"
      "169:"  // Height 4: Partial direct writeback: partial_1_0
      "str s4, [x27, #0x0]\n"
      "str s8, [x23, #0x0]\n"
      "str s19, [x22, #0x0]\n"
      "str s20, [x21, #0x0]\n"
      "170:"  // Height 4: Partial direct writeback: Done
      "b 172f\n"
      "171:"  // Height 4: Full writeback
      "str q4, [x27, #0x0]\n"
      "str q14, [x27, #0x10]\n"
      "str q15, [x27, #0x20]\n"
      "str q16, [x27, #0x30]\n"
      "str q17, [x27, #0x40]\n"
      "str q18, [x27, #0x50]\n"
      "add x27, x27, #0x60\n"
      "str q8, [x23, #0x0]\n"
      "str q9, [x23, #0x10]\n"
      "str q10, [x23, #0x20]\n"
      "str q11, [x23, #0x30]\n"
      "str q12, [x23, #0x40]\n"
      "str q13, [x23, #0x50]\n"
      "str q19, [x22, #0x0]\n"
      "str q26, [x22, #0x10]\n"
      "str q27, [x22, #0x20]\n"
      "str q28, [x22, #0x30]\n"
      "str q29, [x22, #0x40]\n"
      "str q30, [x22, #0x50]\n"
      "str q20, [x21, #0x0]\n"
      "str q21, [x21, #0x10]\n"
      "str q22, [x21, #0x20]\n"
      "str q23, [x21, #0x30]\n"
      "str q24, [x21, #0x40]\n"
      "str q25, [x21, #0x50]\n"
      "172:"  // Height 4: Writeback done
      "subs x9, x9, #0x18\n"
      "bgt 131b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 174f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 173f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "173:"  // Update direct input
      "mov x20, #0x10\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "174:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
