/*
** Copyright (C) 2001-2010 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/

#ifndef _BUFFER_H_
#define _BUFFER_H_

#include <stdlib.h>

struct _Buffer;
typedef struct _Buffer Buffer;

typedef unsigned int BufferIterator;

/*
 * buffer_new:
 * create a new buffer
 *
 * returns: a new buffer or NULL if something went wrong
 */
Buffer*     buffer_new          (void);


/*
 * buffer_destroy:
 * destroy an existing buffer
 *
 * buffer: an existing buffer
 *
 * precondition: buffer != NULL
 */
void     buffer_destroy      (Buffer* buffer);



/*
 * buffer_manage_data:
 * manage an existing char array. The data must be freeable
 * (replaces existing data).
 *
 * return new length of buffer or -1 if mem allocation failed
 *
 * buffer: a Buffer ptr
 * data  : a char ptr
 * len   : length of the char buffer
 *
 * preconditions: buffer != NULL && 
 *                data == NULL -> len == 0 
 *                len >= 0
 */
int        buffer_manage_data  (Buffer* buffer, char* data, unsigned int len);


/*
 * buffer_data:
 * get a ptr to the data inside the buffer
 *
 * buffer_data_to_string:
 * copy buffer data to zero-terminated string
 *
 * buffer: a Buffer ptr
 *
 * returns: a ptr to the buffer data
 * 
 * precondition: buffer != NULL
 */
const char* buffer_data (const Buffer* buffer);
char *buffer_data_to_string (const Buffer* buffer);


/*
 * buffer_data:
 * get a ptr to the data inside the buffer at position pos
 *
 * buffer: a Buffer ptr
 * pos: a valid buffer iterator
 *
 * returns: a ptr to the buffer data at pos
 * 
 * precondition: buffer != NULL
 */
const char* buffer_data_pos (const Buffer* buffer, BufferIterator pos);




/*
 * buffer_length:
 * get the length of the buffer
 *
 * buffer: a Buffer ptr
 *
 * precondition: buffer != NULL
 */
unsigned int buffer_length (const Buffer *buffer);



/*
 * buffer_append, buffer_prepend:
 * append or prepend data to the buffer
 *
 * return length of buffer or -1 if mem allocation failed
 *
 * buffer: a buffer ptr
 * data  : a char* buffer
 * len   : length of the buffer
 *
 * preconditions: buffer != NULL && 
 *                data==NULL -> len == 0
 *                len >= 0
 */
int       buffer_append 	(Buffer* buffer, const char* data, unsigned int len);


/*
 * buffer_erase:
 * erase a part of the buffer
 * returns the new length of the buffer
 *
 * buffer: a Buffer ptr
 * pos   : start position of the part to erase
 * len   : length of the part to erase
 *
 * preconditions: buffer != NULL &&
 *                pos < buffer_length (buffer) &&
 *                pos + len < buffer_length (buffer)
 *                len >= 0
 */
unsigned int        buffer_erase   	(Buffer* buffer, BufferIterator pos, 
					 unsigned int len);


/*
 * buffer_find:
 * find a string in the buffer
 *
 * buffer   : a Buffer ptr
 * offset   : offset to start looking
 * pattern  : the pattern to search
 * len      : length of the pattern
 *
 * returns: BufferIterator pointing to the found position, or to buffer_end 
 * if nothing was found
 *
 * preconditions: buffer != NULL &&
 *                pattern == NULL -> len == 0
 *                len >= 0
 *                0 <= offset < buffer length
 */
BufferIterator  buffer_find_offset         (const Buffer* buffer, unsigned int offset, 
					    const char* pattern, unsigned int len);
#define buffer_find(b,p,l)   (buffer_find_offset(b,0,p,l))
/*
 * buffer_clear:
 * clear the buffer
 *
 * buffer  : a buffer ptr
 *
 * preconditions: buffer != NULL
 */
void	    buffer_clear   	(Buffer* buffer);


/*
 * buffer_begin, buffer_end:
 * macros for c++-stl like iterators
 */
#define buffer_begin(b) ((BufferIterator)0)
#define buffer_end(b)   ((BufferIterator)buffer_length(b))


/*
 * buffer_at:
 * get a char from the buffer
 *
 * buffer: a buffer ptr
 * pos   : a buffer iterator
 *
 * returns: a character
 *
 * preconditions: buffer != NULL 
 *                && pos < buffer_end(buffer)
 */
char 	   buffer_at (const Buffer* buffer, BufferIterator pos);

#endif /*_BUFFER_H_*/
