/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "APITest.h"
#include "OpenGLContext.h"
#include "PNGHelper.h"

#include <libgltf.h>
#include "TimeFunction.h"

#include <iostream>
#include <fstream>
#include <sstream>
#include <cmath>

namespace libgltf { namespace test
{

void APITest::setUp()
{
}

void APITest::tearDown()
{
}

// I) Testing gltf_renderer_init() method
void APITest::test_gltf_renderer_init()
{
    // 1) Test the usual case, valid *.json and empty vInputFiles
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = gltf_renderer_init(DUCK_FILE_NAME, vInputFiles);
        CPPUNIT_ASSERT(pHandle);
        CPPUNIT_ASSERT_EQUAL(size_t(4), vInputFiles.size());

        CPPUNIT_ASSERT_EQUAL(std::string("duckCM.png"), vInputFiles[0].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_IMAGE, vInputFiles[0].type);

        CPPUNIT_ASSERT_EQUAL(std::string("duck.bin"), vInputFiles[1].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_BINARY, vInputFiles[1].type);

        CPPUNIT_ASSERT_EQUAL(std::string("duck0FS.glsl"), vInputFiles[2].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_GLSL, vInputFiles[2].type);

        CPPUNIT_ASSERT_EQUAL(std::string("duck0VS.glsl"), vInputFiles[3].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_GLSL, vInputFiles[3].type);

        gltf_renderer_release(pHandle);
    }

    // 2) When *.json file name is empty
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = gltf_renderer_init("", vInputFiles);
        CPPUNIT_ASSERT(!pHandle);
    }

    // 3) When *.json file does not exist
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = gltf_renderer_init("../duck_model/duck.json", vInputFiles);
        CPPUNIT_ASSERT(!pHandle);
    }

    // 4) When *.json file is broken (invalid content)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = gltf_renderer_init("../data/duck_model/duck_broken.json", vInputFiles);
        CPPUNIT_ASSERT(!pHandle);
    }

    // 5) When the output vector is not empty, check whether the method overwrites the vector
    {
        std::vector<glTFFile> vInputFiles;
        vInputFiles.resize(10);
        vInputFiles[0].filename = "dummy";
        vInputFiles[0].size = 2000;
        glTFHandle* pHandle = gltf_renderer_init(DUCK_FILE_NAME, vInputFiles);
        CPPUNIT_ASSERT(pHandle);
        CPPUNIT_ASSERT_EQUAL(size_t(4), vInputFiles.size());

        CPPUNIT_ASSERT_EQUAL(std::string("duckCM.png"), vInputFiles[0].filename);
        CPPUNIT_ASSERT_EQUAL(size_t(0), vInputFiles[0].size);

        gltf_renderer_release(pHandle);
    }

    // 6) When *.json file contains not only file name of image/binary files but also a path
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = gltf_renderer_init("../data/duck_model/duck_with_relative_path.json", vInputFiles);
        CPPUNIT_ASSERT(pHandle);
        CPPUNIT_ASSERT_EQUAL(size_t(4), vInputFiles.size());

        CPPUNIT_ASSERT_EQUAL(std::string("textures/duckCM.png"), vInputFiles[0].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_IMAGE, vInputFiles[0].type);

        CPPUNIT_ASSERT_EQUAL(std::string("buffers/duck.bin"), vInputFiles[1].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_BINARY, vInputFiles[1].type);

        CPPUNIT_ASSERT_EQUAL(std::string("shaders/duck0FS.glsl"), vInputFiles[2].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_GLSL, vInputFiles[2].type);

        CPPUNIT_ASSERT_EQUAL(std::string("shaders/duck0VS.glsl"), vInputFiles[3].filename);
        CPPUNIT_ASSERT_EQUAL(GLTF_GLSL, vInputFiles[3].type);

        gltf_renderer_release(pHandle);
    }
}

// II) Testing gltf_renderer_set_content() method
void APITest::test_gltf_renderer_set_content()
{
    OpenGLContext aContext;
    aContext.init();

    // 1) Check when all files are available
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_SUCCESS, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 2) Check when the texture file is missing entirely
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[0].buffer;
        vInputFiles.erase(vInputFiles.begin());

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 3) Check when the texture was not loaded actually
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[0].buffer;
        vInputFiles[0].buffer = 0;
        vInputFiles[0].imagewidth = 0;

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 4) Check when the *.bin file is missing entirely
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[1].buffer;
        vInputFiles.erase(++vInputFiles.begin());

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 5) Check when the *.bin file was not loaded actually
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[1].buffer;
        vInputFiles[1].buffer = 0;
        vInputFiles[1].size = 0;

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 6) Check when the *.bin file size is not equal with the size read from the *.json file
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        vInputFiles[1].size = 10;

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 7) Check when fragment shader is missing entirely
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[2].buffer;
        vInputFiles.erase(++(++vInputFiles.begin()));

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 8) Check when fragment shader file was not loaded actually
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[2].buffer;
        vInputFiles[2].buffer = 0;
        vInputFiles[2].size = 0;

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 9) Check when fragment shader is invalid (GLSL error)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles,"../data/duck_model/duck_with_invalid_FS.json" );

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_SHADER_ERROR, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 10) Check when vertex shader is missing entirely
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[3].buffer;
        vInputFiles.erase(++(++(++vInputFiles.begin())));

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 11) Check when vertex shader file was not loaded actually
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        delete vInputFiles[3].buffer;
        vInputFiles[3].buffer = 0;
        vInputFiles[3].size = 0;

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 12) Check when vertex shader is invalid (GLSL error)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles,"../data/duck_model/duck_with_invalid_VS.json" );

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_SHADER_ERROR, nRet);
        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 13) Check when non of the files are available
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        releaseInputFiles(vInputFiles);

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_FILE_NOT_LOAD, nRet);
        gltf_renderer_release(pHandle);
    }

    // 14) Check when input files are given with relative path
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles, "../data/duck_model/duck_with_relative_path.json");

        int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
        CPPUNIT_ASSERT_EQUAL(LIBGLTF_SUCCESS, nRet);
        gltf_renderer_release(pHandle);
    }
}

// III) Testing gltf_get_camera_pos() method
void APITest::test_gltf_get_camera_pos()
{
    // 1) When gltf_renderer_set_content() was not called yet (crash test)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vEye.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vEye.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vEye.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vView.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vView.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vView.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vUp.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vUp.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vUp.z, 0.0001);

        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 2) When one of the parameters is null pointer (crash test)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, 0, &vView, &vUp);
        gltf_get_camera_pos(pHandle, &vEye, 0, &vUp);
        gltf_get_camera_pos(pHandle, &vEye, &vView, 0);

        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 3) Call with invalid handle
    {
        glm::vec3 vEye(1.0f), vView(1.0f), vUp(1.0f);
        gltf_get_camera_pos(0, &vEye, &vView, &vUp);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vEye.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vEye.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vEye.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vView.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vView.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vView.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vUp.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vUp.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vUp.z, 0.0001);
    }

    OpenGLContext aContext;
    aContext.init();

    // 4) When everything is ok
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(400.1130371, vEye.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(463.2640075, vEye.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(-431.0780029, vEye.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(399.5765686, vView.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(462.6428528, vView.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(-430.5067138, vView.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(-0.4252052, vUp.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.7836933, vUp.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.4527971, vUp.z, 0.0001);
        gltf_renderer_release(pHandle);
    }

    // 5) Test whether moving camera in walkthrough mode has any effect on camera position
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        // Get initial position
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        // Move camera
        gltf_renderer_move_camera(pHandle, 2.0, 3.0, 0.0, 0.0);

        // Get changed position
        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Affect on eye and view vector
        CPPUNIT_ASSERT(std::abs(vEye.x - vEye2.x) > 0.01 ||
                         std::abs(vEye.y - vEye2.y) > 0.01 ||
                         std::abs(vEye.z - vEye2.z) > 0.01);

        CPPUNIT_ASSERT(std::abs(vView.x - vView2.x) > 0.01 ||
                         std::abs(vView.y - vView2.y) > 0.01 ||
                         std::abs(vView.z - vView2.z) > 0.01);

        // No affect on up vector
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.x, vUp2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.y, vUp2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.z, vUp2.z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 6) Test whether rotating camera in walkthrough mode has any effect on camera position
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        // a) horizontal rotation should affect view vector
        // Get initial position
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        // Rotate camera
        gltf_renderer_rotate_camera(pHandle, 3.0, 0.0, 0.0);

        // Get changed position
        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Affect on view vector
        CPPUNIT_ASSERT(std::abs(vView.x - vView2.x) > 0.001 ||
                       std::abs(vView.y - vView2.y) > 0.001 ||
                       std::abs(vView.z - vView2.z) > 0.001);

        // No affect on eye and up vector
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.x, vEye2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.y, vEye2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.z, vEye2.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.x, vUp2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.y, vUp2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.z, vUp2.z, 0.0001);

        // b) vertical rotation should affect both view and up vector
        // Get initial position
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        // Rotate camera
        gltf_renderer_rotate_camera(pHandle, 0.0, 3.0, 0.0);

        // Get changed position
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Affect on view and up vector
        CPPUNIT_ASSERT(std::abs(vView.x - vView2.x) > 0.001 ||
                       std::abs(vView.y - vView2.y) > 0.001 ||
                       std::abs(vView.z - vView2.z) > 0.001);

        CPPUNIT_ASSERT(std::abs(vUp.x - vUp2.x) < 0.01 ||
                       std::abs(vUp.y - vUp2.y) < 0.01 ||
                       std::abs(vUp.z - vUp2.z) < 0.01);

        // No affect on eye vector
        // TODO: small changes in eye vector (be more precise)?
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.x, vEye2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.y, vEye2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.z, vEye2.z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 7) Test whether rotating camera in orbit mode has any effect on camera position
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        // a) horizontal rotation should affect eye vector
        gltf_orbit_mode_start(pHandle);
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        gltf_renderer_rotate_model(pHandle, 2.0, 0.0, 0.0);

        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Affect on eye vector
        CPPUNIT_ASSERT(std::abs(vEye.x - vEye2.x) > 0.01 ||
                       std::abs(vEye.y - vEye2.y) > 0.01 ||
                       std::abs(vEye.z - vEye2.z) > 0.01);

        // No affect on view and up vector
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.x, vView2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.y, vView2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.z, vView2.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.x, vUp.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.y, vUp.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.z, vUp.z, 0.0001);

        // b) vertical rotation should affect up and eye vectors
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);
        gltf_orbit_mode_start(pHandle);

        gltf_renderer_rotate_model(pHandle, 0.0, 2.0, 0.0);

        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Affect on eye and up vector
        CPPUNIT_ASSERT(std::abs(vEye.x - vEye2.x) > 0.01 ||
                       std::abs(vEye.y - vEye2.y) > 0.01 ||
                       std::abs(vEye.z - vEye2.z) > 0.01);

        CPPUNIT_ASSERT(std::abs(vUp.x - vUp2.x) > 0.001 ||
                       std::abs(vUp.y - vUp2.y) > 0.001 ||
                       std::abs(vUp.z - vUp2.z) > 0.001);

        // No affect on view vector
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.x, vView2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.y, vView2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.z, vView2.z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 8) Test whether moving camera in orbit mode has any effect on camera position
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_orbit_mode_start(pHandle);
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        gltf_renderer_move_camera(pHandle, 2.0, 3.0, 0.0, 0.0);

        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Affect on eye vector
        CPPUNIT_ASSERT(std::abs(vEye.x - vEye2.x) > 0.01 ||
                       std::abs(vEye.y - vEye2.y) > 0.01 ||
                       std::abs(vEye.z - vEye2.z) > 0.01);

        // No affect on view and up vector
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.x, vView2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.y, vView2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.z, vView2.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.x, vUp.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.y, vUp.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.z, vUp.z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 9) Test whether camera position loaded from *.json file has any effect on the returned values
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        // Get camera position in walkthrough mode
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);
        gltf_orbit_mode_start(pHandle);

        // Compare it with the camera postion of the orbit mode (orbit mode ignores the default camera position)
        // All the three vectors should change
        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        CPPUNIT_ASSERT(std::abs(vEye.x - vEye2.x) > 0.01 ||
                       std::abs(vEye.y - vEye2.y) > 0.01 ||
                       std::abs(vEye.z - vEye2.z) > 0.01);

        CPPUNIT_ASSERT(std::abs(vView.x - vView2.x) > 0.01 ||
                       std::abs(vView.y - vView2.y) > 0.01 ||
                       std::abs(vView.z - vView2.z) > 0.01);

        CPPUNIT_ASSERT(std::abs(vUp.x - vUp2.x) > 0.01 ||
                       std::abs(vUp.y - vUp2.y) > 0.01 ||
                       std::abs(vUp.z - vUp2.z) > 0.01);

        gltf_renderer_release(pHandle);
    }
}

// IV) Testing gltf_get_model_center_pos() method
void APITest::test_gltf_get_model_center_pos()
{
    // 1) When gltf_renderer_set_content() was not called yet (initial value: 0,0,0)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        glm::vec3* vCenterPos = gltf_get_model_center_pos(pHandle);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vCenterPos->x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vCenterPos->y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, vCenterPos->z, 0.0001);

        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 2) Call with invalid handle
    {
        glm::vec3* vCenterPos = gltf_get_model_center_pos(0);

        CPPUNIT_ASSERT_EQUAL((glm::vec3*)(0), vCenterPos);
    }

    OpenGLContext aContext;
    aContext.init();

    // 3) Default view in walkthrough mode
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        glm::vec3* vCenterPos = gltf_get_model_center_pos(pHandle);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(13.4406967, vCenterPos->x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(86.9496841, vCenterPos->y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(-3.70149993, vCenterPos->z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 4) Test whether switching to orbit mode makes any difference
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_orbit_mode_start(pHandle);

        glm::vec3* vCenterPos = gltf_get_model_center_pos(pHandle);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(13.4406967, vCenterPos->x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(86.9496841, vCenterPos->y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(-3.70149993, vCenterPos->z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 5) Check whether camera look at the center position in orbit mode
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_orbit_mode_start(pHandle);

        glm::vec3* vCenterPos = gltf_get_model_center_pos(pHandle);

        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.x, vCenterPos->x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.y, vCenterPos->y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.z, vCenterPos->z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 6) Change the position via the pointer and check whether it has effect
    //    on the internal variables of the scene
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_orbit_mode_start(pHandle);

        glm::vec3* vCenterPos = gltf_get_model_center_pos(pHandle);
        vCenterPos->x = 6.0f;

        glm::vec3* vCenterPos2 = gltf_get_model_center_pos(pHandle);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vCenterPos->x, vCenterPos2->x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vCenterPos->y, vCenterPos2->y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vCenterPos->z, vCenterPos2->z, 0.0001);

        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vCenterPos->x, vView.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vCenterPos->y, vView.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vCenterPos->z, vView.z, 0.0001);

        gltf_renderer_release(pHandle);
    }
}

// VI) Testing gltf_get_model_size() method
void APITest::test_gltf_get_model_size()
{
    // 1) When gltf_renderer_set_content() was not called yet (initial value: 0.0)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        double dSize = gltf_get_model_size(pHandle);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, dSize, 0.0001);

        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    // 2) Call with invalid handle
    {
        double dSize = gltf_get_model_size(0);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, dSize, 0.0001);
    }

    OpenGLContext aContext;
    aContext.init();

    // 3) When everything is ok
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        double dSize = gltf_get_model_size(0);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, dSize, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 4) Test whether view affect the size (view independent size)
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        double dSize = gltf_get_model_size(0);

        gltf_orbit_mode_start(pHandle);

        double dSize2 = gltf_get_model_size(0);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(dSize, dSize2, 0.0001);

        gltf_renderer_release(pHandle);
    }
}

// VI) Testing gltf_{enable\disable}_rotation() method
void APITest::test_gltf_enable_disable_rotation()
{
    // 1) When gltf_renderer_set_content() was not called yet (crash test)
    {
        std::vector<glTFFile> vInputFiles;
        glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles);

        gltf_enable_rotation(pHandle);
        gltf_disable_rotation(pHandle);

        gltf_renderer_release(pHandle);
        releaseInputFiles(vInputFiles);
    }

    OpenGLContext aContext;
    aContext.init();

    // 2) By default camera rotation is enabled -> rotation works
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        // Get initial position
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        // Rotate camera
        gltf_renderer_rotate_camera(pHandle, 3.0, 0.0, 0.0);

        // Get changed position
        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Rotation has affect on view vector
        CPPUNIT_ASSERT(std::abs(vView.x - vView2.x) > 0.001 ||
                       std::abs(vView.y - vView2.y) > 0.001 ||
                       std::abs(vView.z - vView2.z) > 0.001);

        gltf_renderer_release(pHandle);
    }

    // 3) disable rotation in walkthrough mode -> no rotation
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_disable_rotation(pHandle);

        // Get initial position
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        // Rotate camera
        gltf_renderer_rotate_camera(pHandle, 3.0, 0.0, 0.0);

        // Get changed position
        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Rotation was not applied on the camera position
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.x, vView2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.y, vView2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.z, vView2.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.x, vEye2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.y, vEye2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.z, vEye2.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.x, vUp2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.y, vUp2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.z, vUp2.z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 4) disable and reenable rotation in walkthrough mode -> rotation applied on the position
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_disable_rotation(pHandle);
        gltf_enable_rotation(pHandle);

        // Get initial position
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        // Rotate camera
        gltf_renderer_rotate_camera(pHandle, 3.0, 0.0, 0.0);

        // Get changed position
        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Rotation has affect on view vector
        CPPUNIT_ASSERT(std::abs(vView.x - vView2.x) > 0.001 ||
                       std::abs(vView.y - vView2.y) > 0.001 ||
                       std::abs(vView.z - vView2.z) > 0.001);

        gltf_renderer_release(pHandle);
    }

    // 5) disable rotation in orbit mode -> no rotation
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_disable_rotation(pHandle);

        gltf_orbit_mode_start(pHandle);
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        gltf_renderer_rotate_model(pHandle, 2.0, 0.0, 0.0);

        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Rotation was not applied on the camera position
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.x, vEye2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.y, vEye2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vEye.z, vEye2.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.x, vView2.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.y, vView2.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vView.z, vView2.z, 0.0001);

        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.x, vUp.x, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.y, vUp.y, 0.0001);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(vUp.z, vUp.z, 0.0001);

        gltf_renderer_release(pHandle);
    }

    // 6) disable and reenable rotation in orbit mode -> rotation applied on the position
    {
        glTFHandle* pHandle = initSceneAndSetContent();

        gltf_disable_rotation(pHandle);
        gltf_enable_rotation(pHandle);

        gltf_orbit_mode_start(pHandle);
        glm::vec3 vEye, vView, vUp;
        gltf_get_camera_pos(pHandle, &vEye, &vView, &vUp);

        gltf_renderer_rotate_model(pHandle, 2.0, 0.0, 0.0);

        glm::vec3 vEye2, vView2, vUp2;
        gltf_get_camera_pos(pHandle, &vEye2, &vView2, &vUp2);

        // Affect on eye vector
        CPPUNIT_ASSERT(std::abs(vEye.x - vEye2.x) > 0.01 ||
                       std::abs(vEye.y - vEye2.y) > 0.01 ||
                       std::abs(vEye.z - vEye2.z) > 0.01);

        gltf_renderer_release(pHandle);
    }
}

glTFHandle* APITest::parseJsonAndLoadInputFiles(std::vector<glTFFile>& vInputFiles, const std::string& sFilename)
{
    glTFHandle* pHandle = gltf_renderer_init(sFilename.c_str(), vInputFiles);
    CPPUNIT_ASSERT(pHandle);
    CPPUNIT_ASSERT_EQUAL(size_t(4), vInputFiles.size());

    readInputFiles(vInputFiles, sFilename);

    return pHandle;
}

glTFHandle* APITest::initSceneAndSetContent(const std::string& sFilename)
{
    std::vector<glTFFile> vInputFiles;
    glTFHandle* pHandle = parseJsonAndLoadInputFiles(vInputFiles, sFilename);

    int nRet = gltf_renderer_set_content(pHandle, vInputFiles);
    CPPUNIT_ASSERT_EQUAL(LIBGLTF_SUCCESS, nRet);
    releaseInputFiles(vInputFiles);
    return pHandle;
}

void APITest::readInputFiles(std::vector<glTFFile>& vInputFiles, const std::string& sJsonName)
{
    for( size_t i = 0; i < vInputFiles.size(); ++i )
    {
        glTFFile& rFile = vInputFiles[i];
        const std::string sFileName(sJsonName.substr(0,sJsonName.find_last_of('/')+1) + rFile.filename);
        if( rFile.type == GLTF_IMAGE )
        {
            bool bRet = pnghelper::ReadPNGFromFile(sFileName, &rFile.buffer, rFile.imagewidth, rFile.imageheight );
            rFile.size = 4 * rFile.imagewidth * rFile.imageheight;
            CPPUNIT_ASSERT_EQUAL(true, bRet);
        }
        else
        {
            std::ifstream ifs(sFileName.c_str());
            CPPUNIT_ASSERT_EQUAL(true, ifs.is_open());

            ifs.seekg (0, ifs.end);
            rFile.size = ifs.tellg();
            ifs.seekg (0, ifs.beg);
            rFile.buffer = new char[rFile.size];
            ifs.read (rFile.buffer,rFile.size);
            ifs.close();
        }
    }
}

void APITest::releaseInputFiles(std::vector<glTFFile>& vInputFiles)
{
    for( size_t i = 0; i < vInputFiles.size(); ++i )
    {
        glTFFile& rFile = vInputFiles[i];
        delete rFile.buffer;
        rFile.buffer = 0;
        rFile.size = 0;
        rFile.imagewidth = 0;
        rFile.imageheight = 0;
    }
    vInputFiles.clear();
}

CPPUNIT_TEST_SUITE_REGISTRATION(APITest);

} // test
} // libgltf


/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
