/* This file is part of the TDE libraries
   Copyright (C) 1996 Martynas Kunigelis

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
/**
 * KGameProgress -- a progress indicator widget for TDE.
 */

#include <tqpainter.h>
#include <tqpixmap.h>
#include <tqstring.h>
#include <tqregexp.h>
#include <tqstyle.h>

#include "kgameprogress.h"

#include <tdeapplication.h>

KGameProgress::KGameProgress(TQWidget *parent, const char *name)
	: TQFrame(parent, name),
	TQRangeControl(0, 100, 1, 10, 0),
	orient(TQt::Horizontal)
{
	initialize();
}

KGameProgress::KGameProgress(TQt::Orientation orientation, TQWidget *parent, const char *name)
	: TQFrame(parent, name),
	TQRangeControl(0, 100, 1, 10, 0),
	orient(orientation)
{
	initialize();
}

KGameProgress::KGameProgress(int minValue, int maxValue, int value,
                     TQt::Orientation orientation, TQWidget *parent, const char *name)
	: TQFrame(parent, name),
	TQRangeControl(minValue, maxValue, 1, 10, value),
	orient(orientation)
{
	initialize();
}

KGameProgress::~KGameProgress()
{
	delete bar_pixmap;
}

void KGameProgress::advance(int offset)
{
	setValue(value() + offset);
}

void KGameProgress::initialize()
{
	format_ = "%p%";
	use_supplied_bar_color = false;
	bar_pixmap = 0;
	bar_style = Solid;
	text_enabled = TRUE;
	setBackgroundMode( PaletteBackground );
	connect(kapp, TQ_SIGNAL(appearanceChanged()), this, TQ_SLOT(paletteChange()));
	paletteChange();
}

void KGameProgress::paletteChange()
{
	TQPalette p = kapp->palette();
	const TQColorGroup &colorGroup = p.active();
	if (!use_supplied_bar_color)
		bar_color = colorGroup.highlight();
	bar_text_color = colorGroup.highlightedText();
	text_color = colorGroup.text();
	setPalette(p);

	adjustStyle();
}


void KGameProgress::setBarPixmap(const TQPixmap &pixmap)
{
	if (pixmap.isNull())
		return;
	if (bar_pixmap)
		delete bar_pixmap;

	bar_pixmap = new TQPixmap(pixmap);
}

void KGameProgress::setBarColor(const TQColor &color)
{
	bar_color = color;
	use_supplied_bar_color = true;
	if (bar_pixmap) {
		delete bar_pixmap;
		bar_pixmap = 0;
	}
}

void KGameProgress::setBarStyle(BarStyle style)
{
	if (bar_style != style) {
		bar_style = style;
		update();
	}
}

void KGameProgress::setOrientation(TQt::Orientation orientation)
{
	if (orient != orientation) {
		orient = orientation;
		update();
	}
}

void KGameProgress::setValue(int value)
{
	TQRangeControl::setValue(value);
}

void KGameProgress::setTextEnabled(bool enable)
{
	text_enabled = enable;
}

const TQColor & KGameProgress::barColor() const
{
	return bar_color;
}

const TQPixmap * KGameProgress::barPixmap() const
{
	return bar_pixmap;
}

bool KGameProgress::textEnabled() const
{
	return text_enabled;
}

TQSize KGameProgress::sizeHint() const
{
	TQSize s( size() );

	if(orientation() == TQt::Vertical) {
		s.setWidth(24);
	} else {
		s.setHeight(24);
	}

	return s;
}

TQSize KGameProgress::minimumSizeHint() const
{
	return sizeHint();
}

TQSizePolicy KGameProgress::sizePolicy() const
{
	if ( orientation()==TQt::Vertical )
		return TQSizePolicy( TQSizePolicy::Fixed, TQSizePolicy::Expanding );
	else
		return TQSizePolicy( TQSizePolicy::Expanding, TQSizePolicy::Fixed );
}

KGameProgress::Orientation KGameProgress::orientation() const
{
	return orient;
}

KGameProgress::BarStyle KGameProgress::barStyle() const
{
	return bar_style;
}

int KGameProgress::recalcValue(int range)
{
	int abs_value = value() - minValue();
	int abs_range = maxValue() - minValue();
	return abs_range ? range * abs_value / abs_range : 0;
}

void KGameProgress::valueChange()
{
	repaint(contentsRect(), FALSE);
	emit percentageChanged(recalcValue(100));
}

void KGameProgress::rangeChange()
{
	repaint(contentsRect(), FALSE);
	emit percentageChanged(recalcValue(100));
}

void KGameProgress::styleChange(TQStyle&)
{
	adjustStyle();
}

void KGameProgress::adjustStyle()
{
	switch (style().styleHint(TQStyle::SH_GUIStyle)) {
		case WindowsStyle:
			setFrameStyle(TQFrame::WinPanel | TQFrame::Sunken);
			break;
		case MotifStyle:
		default:
			setFrameStyle(TQFrame::Panel | TQFrame::Sunken);
			setLineWidth( 2 );
			break;
	}
	update();
}

void KGameProgress::paletteChange( const TQPalette &p )
{
	// This never gets called for global color changes 
	// because we call setPalette() ourselves.
	TQFrame::paletteChange(p);
}

void KGameProgress::drawText(TQPainter *p)
{
	TQRect r(contentsRect());
	//TQColor c(bar_color.rgb() ^ backgroundColor().rgb());

	// Rik: Replace the tags '%p', '%v' and '%m' with the current percentage,
	// the current value and the maximum value respectively.
	TQString s(format_);

	s.replace(TQRegExp(TQString::fromLatin1("%p")), TQString::number(recalcValue(100)));
	s.replace(TQRegExp(TQString::fromLatin1("%v")), TQString::number(value()));
	s.replace(TQRegExp(TQString::fromLatin1("%m")), TQString::number(maxValue()));

	p->setPen(text_color);
	TQFont font = p->font();
	font.setBold(true);
	p->setFont(font);
	//p->setRasterOp(XorROP);
	p->drawText(r, AlignCenter, s);
	p->setClipRegion( fr );
	p->setPen(bar_text_color);
	p->drawText(r, AlignCenter, s);
}

void KGameProgress::drawContents(TQPainter *p)
{
	TQRect cr = contentsRect(), er = cr;
	fr = cr;
	TQBrush fb(bar_color), eb(backgroundColor());

	if (bar_pixmap)
		fb.setPixmap(*bar_pixmap);

	if (backgroundPixmap())
		eb.setPixmap(*backgroundPixmap());

	switch (bar_style) {
		case Solid:
			if (orient ==TQt::Horizontal) {
				fr.setWidth(recalcValue(cr.width()));
				er.setLeft(fr.right() + 1);
			} else {
				fr.setTop(cr.bottom() - recalcValue(cr.height()));
				er.setBottom(fr.top() - 1);
			}

			p->setBrushOrigin(cr.topLeft());
			p->fillRect(fr, fb);

			p->fillRect(er, eb);

			break;

		case Blocked:
			const int margin = 2;
			int max, num, dx, dy;
			if (orient ==TQt::Horizontal) {
				fr.setHeight(cr.height() - 2 * margin);
				fr.setWidth((int)(0.67 * fr.height()));
				fr.moveTopLeft(TQPoint(cr.left() + margin, cr.top() + margin));
				dx = fr.width() + margin;
				dy = 0;
				max = (cr.width() - margin) / (fr.width() + margin) + 1;
				num = recalcValue(max);
			} else {
				fr.setWidth(cr.width() - 2 * margin);
				fr.setHeight((int)(0.67 * fr.width()));
				fr.moveBottomLeft(TQPoint(cr.left() + margin, cr.bottom() - margin));
				dx = 0;
				dy = - (fr.height() + margin);
				max = (cr.height() - margin) / (fr.height() + margin) + 1;
				num = recalcValue(max);
			}
			p->setClipRect(cr.x() + margin, cr.y() + margin,
			               cr.width() - margin, cr.height() - margin);
			for (int i = 0; i < num; i++) {
				p->setBrushOrigin(fr.topLeft());
				p->fillRect(fr, fb);
				fr.moveBy(dx, dy);
			}
			
			if (num != max) {
				if (orient ==TQt::Horizontal)
					er.setLeft(fr.right() + 1);
				else
					er.setBottom(fr.bottom() + 1);
				if (!er.isNull()) {
					p->setBrushOrigin(cr.topLeft());
					p->fillRect(er, eb);
				}
			}

			break;
	}

	if (text_enabled && bar_style != Blocked)
		drawText(p);
}

void KGameProgress::setFormat(const TQString & format)
{
	format_ = format;
}

TQString KGameProgress::format() const
{
	return format_;
}

#include "kgameprogress.moc"
