/* This file is part of the KDE project
   Copyright (C) 2001 Carsten Pfeiffer <pfeiffer@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <dcopclient.h>

#include <tdeapplication.h>
#include <tdeaction.h>
#include <tdeconfig.h>
#include <tdeglobal.h>
#include <tdehtml_part.h>
#include <kinstance.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <tdepopupmenu.h>
#include <tdeprotocolmanager.h>
#include <kgenericfactory.h>
#include <tdeaboutdata.h>

#include "settingsplugin.h"

typedef KGenericFactory<SettingsPlugin> SettingsPluginFactory;
static const TDEAboutData aboutdata("tdehtmlsettingsplugin", I18N_NOOP("HTML Settings") , "1.0" );
K_EXPORT_COMPONENT_FACTORY( libtdehtmlsettingsplugin,
                            SettingsPluginFactory( &aboutdata ) )

SettingsPlugin::SettingsPlugin( TQObject* parent, const char* name,
                                const TQStringList & )
    : KParts::Plugin( parent, name ), mConfig(0)
{

    setInstance(SettingsPluginFactory::instance());

    if ( !kapp->dcopClient()->isAttached() )
	kapp->dcopClient()->attach();

    TDEActionMenu *menu = new TDEActionMenu( i18n("HTML Settings"), "configure",
					 actionCollection(), "action menu" );
    menu->setDelayed( false );

    TDEToggleAction *action;


    action = new TDEToggleAction( i18n("Java&Script"), 0,
				this, TQT_SLOT(toggleJavascript()),
				actionCollection(), "javascript" );
    menu->insert( action );

    action = new TDEToggleAction( i18n("&Java"), 0,
				this, TQT_SLOT(toggleJava()),
				actionCollection(), "java" );
    menu->insert( action );

    action = new TDEToggleAction( i18n("&Cookies"), 0,
				this, TQT_SLOT(toggleCookies()),
				actionCollection(), "cookies" );
    menu->insert( action );

    action = new TDEToggleAction( i18n("&Plugins"), 0,
				this, TQT_SLOT(togglePlugins()),
				actionCollection(), "plugins" );
    menu->insert( action );

    action = new TDEToggleAction( i18n("Autoload &Images"), 0,
				this, TQT_SLOT(toggleImageLoading()),
				actionCollection(), "imageloading" );
    menu->insert( action );

    menu->insert( new TDEActionSeparator(actionCollection()) );

    action = new TDEToggleAction( i18n("Enable Pro&xy"), 0,
				this, TQT_SLOT(toggleProxy()),
				actionCollection(), "useproxy" );
    action->setCheckedState(i18n("Disable Pro&xy"));
    menu->insert( action );

    action = new TDEToggleAction( i18n("Enable Cac&he"), 0,
				this, TQT_SLOT(toggleCache()),
				actionCollection(), "usecache" );
    action->setCheckedState(i18n("Disable Cac&he"));
    menu->insert( action );


    TDESelectAction *sAction = new TDESelectAction( i18n("Cache Po&licy"), 0,
						0, 0, actionCollection(),
                                                "cachepolicy" );
    TQStringList policies;
    policies += i18n( "&Keep Cache in Sync" );
    policies += i18n( "&Use Cache if Possible" );
    policies += i18n( "&Offline Browsing Mode" );
    sAction->setItems( policies );
    connect( sAction, TQT_SIGNAL( activated( int ) ), TQT_SLOT( cachePolicyChanged(int) ) );

    menu->insert( sAction );

    connect( menu->popupMenu(), TQT_SIGNAL( aboutToShow() ), TQT_SLOT( showPopup() ));
}

SettingsPlugin::~SettingsPlugin()
{
  delete mConfig;
}

void SettingsPlugin::showPopup()
{
  if( !parent() || !parent()->inherits("TDEHTMLPart"))
    return;

  if (!mConfig)
    mConfig = new TDEConfig("settingspluginrc", false, false);
    
  TDEHTMLPart *part = static_cast<TDEHTMLPart *>( parent() );

  KProtocolManager::reparseConfiguration();
  bool cookies = cookiesEnabled( part->url().url() );

  ((TDEToggleAction*)actionCollection()->action("useproxy"))->setChecked(KProtocolManager::useProxy());
  ((TDEToggleAction*)actionCollection()->action("java"))->setChecked( part->javaEnabled() );
  ((TDEToggleAction*)actionCollection()->action("javascript"))->setChecked( part->jScriptEnabled() );
  ((TDEToggleAction*)actionCollection()->action("cookies"))->setChecked( cookies );
  ((TDEToggleAction*)actionCollection()->action("plugins"))->setChecked( part->pluginsEnabled() );
  ((TDEToggleAction*)actionCollection()->action("imageloading"))->setChecked( part->autoloadImages() );
  ((TDEToggleAction*)actionCollection()->action("usecache"))->setChecked(KProtocolManager::useCache());

  TDEIO::CacheControl cc = KProtocolManager::cacheControl();
  switch ( cc ) 
  {
      case TDEIO::CC_Verify:
          ((TDESelectAction*)actionCollection()->action("cachepolicy"))->setCurrentItem( 0 );
          break;
      case TDEIO::CC_CacheOnly:
          ((TDESelectAction*)actionCollection()->action("cachepolicy"))->setCurrentItem( 2 );
          break;
      case TDEIO::CC_Cache:
          ((TDESelectAction*)actionCollection()->action("cachepolicy"))->setCurrentItem( 1 );
          break;
      case TDEIO::CC_Reload: // nothing for now
      case TDEIO::CC_Refresh:
      default:
          break;
      
  }
}

void SettingsPlugin::toggleJava()
{
  if( parent() && parent()->inherits("TDEHTMLPart"))
  {
      TDEHTMLPart *part = static_cast<TDEHTMLPart *>(parent());
      part->setJavaEnabled( ((TDEToggleAction*)actionCollection()->action("java"))->isChecked() );
  }
}

void SettingsPlugin::toggleJavascript()
{
  if( parent() && parent()->inherits("TDEHTMLPart"))
  {
      TDEHTMLPart *part = static_cast<TDEHTMLPart *>(parent());
      part->setJScriptEnabled( ((TDEToggleAction*)actionCollection()->action("javascript"))->isChecked() );
  }
}

void SettingsPlugin::toggleCookies()
{
  if( !parent() || !parent()->inherits("TDEHTMLPart"))
    return;
    
  TDEHTMLPart *part = static_cast<TDEHTMLPart *>( parent() );

  TQString advice;
  bool enable = ((TDEToggleAction*)actionCollection()->action("cookies"))->isChecked();
  advice = enable ? "Accept" : "Reject";

  TQCString replyType;
  TQByteArray data, replyData;
  TQDataStream stream( data, IO_WriteOnly );
  stream << part->url().url() << advice;
  bool ok = kapp->dcopClient()->call( "kded", "kcookiejar",
        "setDomainAdvice(TQString,TQString)",
        data, replyType, replyData, true );

  if ( !ok )
     KMessageBox::sorry( part->widget(),
          i18n("I can't enable cookies, because the "
                "cookie daemon could not be started."),
          i18n("Cookies Disabled"));
}

void SettingsPlugin::togglePlugins()
{
  if( parent() && parent()->inherits("TDEHTMLPart"))
  {
      TDEHTMLPart *part = static_cast<TDEHTMLPart *>(parent());
      part->setPluginsEnabled( ((TDEToggleAction*)actionCollection()->action("plugins"))->isChecked() );
  }
}

void SettingsPlugin::toggleImageLoading()
{
  if( parent() && parent()->inherits("TDEHTMLPart"))
  {
      TDEHTMLPart *part = static_cast<TDEHTMLPart *>(parent());
      part->setAutoloadImages( ((TDEToggleAction*)actionCollection()->action("imageloading"))->isChecked() );
  }
}

bool SettingsPlugin::cookiesEnabled( const TQString& url )
{
    TQByteArray data, reply;
    TQCString replyType;
    TQDataStream stream( data, IO_WriteOnly );
    stream << url;
    kapp->dcopClient()->call( "kcookiejar", "kcookiejar", "getDomainAdvice(TQString)", data, replyType, reply, true );

    bool enabled = false;

    if ( replyType == "TQString" ) 
    {
        TQString advice;
        TQDataStream s( reply, IO_ReadOnly );
        s >> advice;
        enabled = ( advice == "Accept" );
        if ( !enabled && advice == "Dunno" ) {
            // TODO, check the global setting via dcop
            TDEConfig kc( "kcookiejarrc", true, false );
            kc.setGroup( "Cookie Policy" );
            enabled =
          (kc.readEntry( "CookieGlobalAdvice", "Reject" ) == "Accept");
        }
    }

    return enabled;
}


//
// sync with kcontrol/tdeio/ksaveioconfig.* !
//

void SettingsPlugin::toggleProxy()
{
    bool enable = ((TDEToggleAction*)actionCollection()->action("useproxy"))->isChecked();   
    
    int type;
    
    if( enable )
      type = mConfig->readNumEntry( "SavedProxyType", KProtocolManager::ManualProxy );
    else
    {
      mConfig->writeEntry( "SavedProxyType", KProtocolManager::proxyType() );
      type = KProtocolManager::NoProxy;
    }
    
    TDEConfig config("tdeioslaverc", false, false);
    config.setGroup( "Proxy Settings" );    
    config.writeEntry( "ProxyType", type );

    ((TDEToggleAction*)actionCollection()->action("useproxy"))->setChecked(enable);
    updateIOSlaves();
}


void SettingsPlugin::toggleCache()
{
    bool usesCache = KProtocolManager::useCache();
    TDEConfig config( "tdeio_httprc", false, false );
    config.writeEntry( "UseCache", !usesCache );

    ((TDEToggleAction*)actionCollection()->action("usecache"))->setChecked( !usesCache );

    updateIOSlaves();
}

void SettingsPlugin::cachePolicyChanged( int p )
{
    TQString policy;

    switch ( p ) {
        case 0:
            policy = TDEIO::getCacheControlString( TDEIO::CC_Verify );
            break;
        case 1:
            policy = TDEIO::getCacheControlString( TDEIO::CC_Cache );
            break;
        case 2:
            policy = TDEIO::getCacheControlString( TDEIO::CC_CacheOnly );
            break;
    };

    if ( !policy.isEmpty() ) {
        TDEConfig config("tdeio_httprc", false, false);
        config.writeEntry("cache", policy);

        updateIOSlaves();
    }
}

void SettingsPlugin::updateIOSlaves()
{
    TQByteArray data;
    TQDataStream stream( data, IO_WriteOnly );

    DCOPClient* client = kapp->dcopClient();
    if ( !client->isAttached() )
        client->attach();

    TQString protocol; // null -> all of them
    stream << protocol;
    client->send( "*", "TDEIO::Scheduler",
                  "reparseSlaveConfiguration(TQString)", data );
}

#include "settingsplugin.moc"
