/* This file is part of the KDE project
   Copyright (C) 2002 Harald Fernengel <harry@kdevelop.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "execcommand.h"

#include <tdeprocess.h>
#include <kprogress.h>
#include <tdelocale.h>
#include <tdemessagebox.h>

ExecCommand::ExecCommand( const TQString& executable, const TQStringList& args, 
                          const TQString& workingDir, const TQStringList& env,
                          TQObject* parent, const char* name ):
	TQObject( parent, name ), out( "" ) /* make sure out is not TQString() since that would mean "error" */

{
  progressDlg = 0;

  proc = new TDEProcess();
  proc->setWorkingDirectory( workingDir );
  for ( TQStringList::ConstIterator it = env.begin(); it != env.end(); ++it )
    proc->setEnvironment( (*it).section( '=', 0, 0 ), (*it).section( '=', 1, 1 ) );
  *proc << executable;
  *proc << args;

  connect( proc, TQ_SIGNAL(processExited(TDEProcess*)),
           this, TQ_SLOT(processExited()) );
  connect( proc, TQ_SIGNAL(receivedStdout(TDEProcess*,char*,int)),
           this, TQ_SLOT(receivedStdout(TDEProcess*,char*,int)) );
  connect( proc, TQ_SIGNAL(receivedStderr(TDEProcess*,char*,int)),
           this, TQ_SLOT(receivedStderr(TDEProcess*,char*,int)) );

  bool ok = proc->start( TDEProcess::NotifyOnExit, TDEProcess::AllOutput );

  if ( !ok ) {
    KMessageBox::error( 0, i18n("Could not invoke \"%1\". Please make sure it is installed correctly").arg( executable ),
                        i18n("Error Invoking Command") );

    emit finished( TQString(), TQString() );
    deleteLater();

  } else {
    progressDlg = new KProgressDialog( 0, 0, i18n("Command running..."),
                      i18n("Please wait until the \"%1\" command finishes.").arg( executable ), false );
    connect( progressDlg, TQ_SIGNAL(cancelClicked()),
             this, TQ_SLOT(cancelClicked()) );
  }
}

void ExecCommand::receivedStdout (TDEProcess*, char *buffer, int buflen)
{
  out += TQString::fromUtf8( buffer, buflen );  
}

void ExecCommand::receivedStderr (TDEProcess*, char *buffer, int buflen)
{
  err += TQString::fromUtf8( buffer, buflen );
}

void ExecCommand::processExited()
{
  delete progressDlg;
  progressDlg = 0;

  emit finished( out, err );
  deleteLater();
}

void ExecCommand::cancelClicked()
{
  delete progressDlg;
  progressDlg = 0;
  proc->kill();

  emit finished( TQString(), TQString() );
  deleteLater();
}

ExecCommand::~ExecCommand()
{
  delete proc;
  delete progressDlg;
}

#include "execcommand.moc"
