/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-09-19
 * Description : Autodetect gpsbabel binary program and version
 *
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// KDE includes

#include <tdeapplication.h>
#include <kprocess.h>
#include <tdemessagebox.h>
#include <tdelocale.h>
#include <tdeglobal.h>
#include <kdebug.h>

// Local includes

#include "gpsbabelbinary.h"
#include "gpsbabelbinary.moc"

namespace KIPIGPSSyncPlugin
{

class GPSBabelBinaryPriv
{
public:

    GPSBabelBinaryPriv()
    {
        available = false;
        version   = TQString();
    }

    bool    available;

    TQString version;
};

GPSBabelBinary::GPSBabelBinary()
              : TQObject()
{
    d = new GPSBabelBinaryPriv;
    checkSystem();
}

GPSBabelBinary::~GPSBabelBinary()
{
    delete d;
}

void GPSBabelBinary::checkSystem()
{
    TDEProcess process;
    process.clearArguments();
    process << path() << "-V";    

    connect(&process, TQ_SIGNAL(receivedStdout(TDEProcess *, char*, int)),
            this, TQ_SLOT(slotReadStdoutFromGPSBabel(TDEProcess*, char*, int)));

    d->available = process.start(TDEProcess::Block, TDEProcess::Stdout);
}

void GPSBabelBinary::slotReadStdoutFromGPSBabel(TDEProcess*, char* buffer, int buflen)
{
    // The gpsbabel output look like this : GPSBabel Version 1.2.5
    TQString headerStarts("GPSBabel Version ");

    TQString stdOut    = TQString::fromLocal8Bit(buffer, buflen);
    TQString firstLine = stdOut.section('\n', 1, 1);    

    if (firstLine.startsWith(headerStarts))
    {
        d->version = firstLine.remove(0, headerStarts.length());   
        kdDebug( 51001 ) << "Found gpsbabel version: " << version() << endl;    
    }
}

const char *GPSBabelBinary::path()
{
    return "gpsbabel";
}

bool GPSBabelBinary::isAvailable() const
{
    return d->available;
}

TQString GPSBabelBinary::version() const
{
    return d->version;
}

bool GPSBabelBinary::versionIsRight() const
{
    if (d->version.isNull() || !isAvailable())
        return false;

    if (d->version.toFloat() >= minimalVersion().toFloat())
        return true;

    return false;
}

TQString GPSBabelBinary::minimalVersion() const
{
    return TQString("1.2.5");
}

}  // namespace KIPIGPSSyncPlugin
