// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "device/fido/enclave/attestation.h"

#include "testing/gtest/include/gtest/gtest.h"

namespace device::enclave {

namespace {

// This is a sample handshake hash and attestation from the production enclave.

const uint8_t kHandshakeHash[32] = {
    0x18, 0x2a, 0x9e, 0xe9, 0x9d, 0x04, 0x44, 0x9c, 0xc5, 0xf0, 0xcd,
    0x6e, 0xf2, 0xff, 0xe1, 0xac, 0x4d, 0xde, 0xae, 0x01, 0x6d, 0x26,
    0x5f, 0xe2, 0x9b, 0xc8, 0xad, 0x92, 0xb9, 0x16, 0x10, 0xf6};

const uint8_t kAttestation[] = {
    0xa5, 0x00, 0x58, 0x48, 0x30, 0x46, 0x02, 0x21, 0x00, 0xc8, 0x13, 0x6a,
    0xf8, 0x52, 0x69, 0x78, 0xa1, 0x1e, 0x11, 0xc0, 0x69, 0x6b, 0x3b, 0x20,
    0xf5, 0x84, 0xa0, 0xc7, 0x3b, 0x10, 0xa9, 0x52, 0xa9, 0x65, 0x39, 0x50,
    0x6d, 0xcf, 0x2c, 0x9c, 0x14, 0x02, 0x21, 0x00, 0xc2, 0x88, 0xdb, 0x6c,
    0x37, 0xb2, 0xf7, 0xcc, 0x66, 0x3e, 0x75, 0x60, 0xd6, 0x76, 0x23, 0x2e,
    0x10, 0xe4, 0xf6, 0x28, 0x69, 0xd3, 0xf4, 0xab, 0xc4, 0xa7, 0x18, 0x13,
    0x3f, 0x36, 0xdf, 0x73, 0x01, 0x58, 0x41, 0x04, 0xd8, 0x52, 0x16, 0x32,
    0x86, 0x9d, 0xcc, 0x72, 0x63, 0x8f, 0x51, 0x04, 0x1d, 0x7b, 0x29, 0xe7,
    0x5f, 0xfe, 0x88, 0xf8, 0xed, 0x28, 0xd0, 0x1c, 0x00, 0xac, 0x8a, 0x05,
    0x6c, 0xd4, 0xe8, 0x17, 0x9c, 0x15, 0x8e, 0x3d, 0x3a, 0x9f, 0x26, 0x21,
    0xe8, 0x5e, 0x9f, 0xa7, 0x3a, 0xf7, 0xb5, 0xea, 0x2d, 0xe3, 0x49, 0x74,
    0x2d, 0x6c, 0x19, 0x6b, 0xc7, 0x9e, 0x21, 0x04, 0xf7, 0xa3, 0x55, 0xa0,
    0x02, 0x58, 0x40, 0xf1, 0x0f, 0x4b, 0x11, 0x62, 0x8b, 0xf1, 0xdf, 0xb0,
    0x91, 0x7f, 0xce, 0x18, 0x85, 0x5f, 0x91, 0x44, 0xce, 0x07, 0xcb, 0xec,
    0xa9, 0x19, 0x20, 0x6e, 0xd7, 0xc9, 0x7b, 0x5b, 0x8a, 0xe2, 0x3b, 0x7d,
    0x63, 0xf8, 0xef, 0x66, 0xb2, 0x0e, 0xb4, 0x39, 0xb7, 0x0f, 0x31, 0x8f,
    0xf0, 0x5d, 0x4a, 0x85, 0x00, 0x05, 0xb4, 0x43, 0xda, 0x79, 0x86, 0x9c,
    0x87, 0x1e, 0x41, 0x79, 0x0d, 0x9a, 0xdc, 0x03, 0x59, 0x0a, 0x99, 0x0a,
    0x8d, 0x0a, 0x08, 0x01, 0x12, 0xa0, 0x09, 0x03, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xdb, 0x25,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x02, 0xb4, 0x3b, 0x26, 0x01, 0xf3, 0xfc, 0x5c, 0xc8,
    0xe3, 0xe2, 0xf6, 0xfe, 0x97, 0x2a, 0x1e, 0x47, 0xba, 0x50, 0x21, 0x3d,
    0x70, 0x46, 0x67, 0x81, 0x26, 0x83, 0x1b, 0x8a, 0x5e, 0x99, 0xca, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc4, 0x5c, 0x93, 0xd6, 0x88,
    0x64, 0x05, 0xce, 0x9f, 0x54, 0xed, 0x35, 0xe9, 0xbd, 0x1d, 0x0e, 0xf1,
    0x30, 0xdd, 0x40, 0x5d, 0x79, 0x2f, 0xfd, 0xb7, 0x2e, 0xc6, 0xe2, 0x2d,
    0x6b, 0x30, 0x45, 0x26, 0xbb, 0x01, 0xa7, 0x03, 0xaf, 0x6a, 0xf2, 0xa3,
    0x8a, 0x6e, 0x6a, 0x12, 0xbd, 0x65, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x59, 0x5c, 0x61, 0x2e, 0x40, 0x29, 0xf4, 0x9f, 0xb9,
    0x61, 0x4c, 0x85, 0x7a, 0x1f, 0x43, 0x30, 0x65, 0xc7, 0x70, 0x78, 0x9e,
    0x67, 0xf6, 0x72, 0x96, 0xb5, 0xe9, 0x97, 0x50, 0x62, 0x6f, 0xd4, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x16, 0xd1, 0x19, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xde, 0xa9, 0xbc, 0x16, 0x04, 0x62, 0xcd, 0x72, 0xc3,
    0xb7, 0x77, 0xde, 0x0c, 0x6f, 0x08, 0x6c, 0x9f, 0x6d, 0x17, 0x86, 0x8d,
    0x8a, 0x1c, 0xcb, 0x18, 0x9c, 0x71, 0xbc, 0x58, 0x36, 0x91, 0xa3, 0xdc,
    0x4d, 0x69, 0x22, 0x98, 0xee, 0xa6, 0x18, 0x9d, 0x41, 0x12, 0xe7, 0x67,
    0xf8, 0x0e, 0x2c, 0x6c, 0x71, 0x95, 0x23, 0x8a, 0x03, 0x9a, 0x90, 0x8d,
    0x03, 0xb6, 0x7a, 0xc6, 0x05, 0x1c, 0xa0, 0x04, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x18, 0xdb, 0x1d, 0x37, 0x01, 0x00, 0x1d, 0x37, 0x01, 0x00, 0x04,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xdb, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0xe0, 0xbf, 0x65, 0xa0,
    0x57, 0x2e, 0x03, 0x8d, 0x0b, 0x38, 0x83, 0xeb, 0x83, 0xc2, 0xa3, 0x0a,
    0xaf, 0x75, 0x5f, 0x0d, 0xa0, 0x70, 0x2c, 0xd4, 0x32, 0x0e, 0x17, 0x4a,
    0x91, 0x52, 0x05, 0xf1, 0x52, 0x0c, 0x38, 0x1d, 0xad, 0xe7, 0xd5, 0x6f,
    0xf2, 0xe2, 0xcd, 0xd0, 0x73, 0x0d, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0xff, 0x6a, 0x69, 0x15,
    0x15, 0x53, 0x83, 0x9e, 0x96, 0x25, 0x12, 0x82, 0x69, 0x73, 0x8e, 0x7e,
    0x0d, 0xf5, 0x75, 0x93, 0x70, 0xa4, 0x0c, 0xb0, 0x97, 0xae, 0xa8, 0x70,
    0xb7, 0x8e, 0xd0, 0xfd, 0x42, 0xdb, 0x98, 0x56, 0x39, 0x32, 0x0a, 0xb6,
    0xdb, 0x6a, 0x32, 0x4e, 0xed, 0xd6, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x1a, 0x66, 0xa7, 0x01, 0x02, 0x02, 0x54, 0x87, 0xae,
    0x47, 0xad, 0xb8, 0x31, 0x7e, 0xec, 0x81, 0x16, 0x92, 0x67, 0xb0, 0x9c,
    0xac, 0xbc, 0x39, 0x6f, 0x38, 0x4c, 0x03, 0x26, 0x04, 0x81, 0x02, 0x20,
    0x01, 0x21, 0x58, 0x20, 0x27, 0x6d, 0xe9, 0x4b, 0x02, 0x4c, 0x39, 0x56,
    0x02, 0x98, 0xaf, 0x0a, 0x59, 0xe6, 0x94, 0x73, 0xc4, 0x81, 0xdc, 0x64,
    0x23, 0xf4, 0x7c, 0x95, 0x18, 0xa3, 0xdb, 0x81, 0x71, 0xbb, 0x67, 0x8e,
    0x22, 0x58, 0x20, 0x33, 0x78, 0x8e, 0xc9, 0x3a, 0x07, 0x7e, 0x6c, 0x84,
    0x78, 0x81, 0x1a, 0xdc, 0x0e, 0x4d, 0x65, 0x39, 0x3b, 0x5e, 0xfa, 0xef,
    0x11, 0x8f, 0xe7, 0x15, 0x4c, 0x90, 0xb7, 0xe7, 0xd4, 0x13, 0xb6, 0x12,
    0xda, 0x02, 0x0a, 0xd7, 0x02, 0x84, 0x43, 0xa1, 0x01, 0x26, 0xa1, 0x04,
    0x52, 0x41, 0x73, 0x79, 0x6d, 0x6d, 0x65, 0x74, 0x72, 0x69, 0x63, 0x45,
    0x43, 0x44, 0x53, 0x41, 0x32, 0x35, 0x36, 0x58, 0xf9, 0xa5, 0x01, 0x78,
    0x28, 0x38, 0x37, 0x61, 0x65, 0x34, 0x37, 0x61, 0x64, 0x62, 0x38, 0x33,
    0x31, 0x37, 0x65, 0x65, 0x63, 0x38, 0x31, 0x31, 0x36, 0x39, 0x32, 0x36,
    0x37, 0x62, 0x30, 0x39, 0x63, 0x61, 0x63, 0x62, 0x63, 0x33, 0x39, 0x36,
    0x66, 0x33, 0x38, 0x34, 0x63, 0x02, 0x78, 0x28, 0x63, 0x62, 0x66, 0x62,
    0x30, 0x34, 0x34, 0x64, 0x33, 0x65, 0x37, 0x65, 0x66, 0x32, 0x39, 0x32,
    0x35, 0x61, 0x61, 0x33, 0x36, 0x30, 0x31, 0x61, 0x30, 0x62, 0x36, 0x34,
    0x30, 0x31, 0x39, 0x34, 0x35, 0x38, 0x34, 0x31, 0x39, 0x33, 0x61, 0x31,
    0x3a, 0x00, 0x47, 0x44, 0x57, 0x58, 0x66, 0xa7, 0x01, 0x02, 0x02, 0x54,
    0xcb, 0xfb, 0x04, 0x4d, 0x3e, 0x7e, 0xf2, 0x92, 0x5a, 0xa3, 0x60, 0x1a,
    0x0b, 0x64, 0x01, 0x94, 0x58, 0x41, 0x93, 0xa1, 0x03, 0x26, 0x04, 0x81,
    0x02, 0x20, 0x01, 0x21, 0x58, 0x20, 0x7a, 0x46, 0x72, 0xa9, 0x53, 0xb1,
    0xde, 0xd0, 0x90, 0x47, 0x3c, 0x85, 0xd9, 0xf7, 0x3e, 0xc0, 0x19, 0x99,
    0x35, 0xdb, 0xbe, 0xf8, 0x7e, 0xe7, 0xb4, 0x94, 0x10, 0x44, 0x0c, 0x55,
    0x1f, 0xe7, 0x22, 0x58, 0x20, 0x9b, 0xd3, 0xd1, 0x9b, 0x60, 0x02, 0x11,
    0x71, 0xa8, 0xd7, 0x7b, 0x07, 0x61, 0x86, 0xc1, 0x9b, 0x28, 0x5c, 0x25,
    0x13, 0x37, 0xbb, 0x93, 0x82, 0xb2, 0xb5, 0x41, 0xe4, 0x59, 0x4c, 0xad,
    0x8b, 0x3a, 0x00, 0x47, 0x44, 0x58, 0x42, 0x20, 0x00, 0x3a, 0x00, 0x47,
    0x44, 0x6c, 0xa1, 0x3a, 0x00, 0x47, 0x44, 0x6b, 0x58, 0x20, 0x81, 0x5c,
    0x99, 0xd0, 0x5d, 0x7f, 0x4a, 0xae, 0x48, 0x60, 0xb5, 0x48, 0xe9, 0x4c,
    0x16, 0x61, 0x85, 0x89, 0x57, 0xac, 0x0c, 0x60, 0xe5, 0xf5, 0xd3, 0x4b,
    0x8b, 0x34, 0x89, 0x17, 0x59, 0xc4, 0x58, 0x40, 0xd4, 0x55, 0xde, 0x70,
    0x4e, 0x90, 0x8d, 0x00, 0x24, 0x3c, 0xfd, 0x64, 0x7b, 0x6c, 0x00, 0xbf,
    0x1d, 0x8f, 0x3a, 0x35, 0x9d, 0xa0, 0xb6, 0x44, 0xd0, 0xaf, 0x01, 0xa0,
    0x03, 0x88, 0xa2, 0x46, 0xde, 0x3f, 0xb5, 0x43, 0x18, 0x0f, 0xcf, 0x6a,
    0x37, 0xf6, 0x91, 0x57, 0x45, 0x35, 0xb3, 0x76, 0x4e, 0x4a, 0x51, 0xd1,
    0xe1, 0xb6, 0x71, 0x77, 0xfd, 0x9f, 0xc0, 0x6a, 0x78, 0x63, 0xc5, 0x39,
    0x1a, 0x92, 0x05, 0x0a, 0xb5, 0x02, 0x84, 0x43, 0xa1, 0x01, 0x26, 0xa1,
    0x04, 0x52, 0x41, 0x73, 0x79, 0x6d, 0x6d, 0x65, 0x74, 0x72, 0x69, 0x63,
    0x45, 0x43, 0x44, 0x53, 0x41, 0x32, 0x35, 0x36, 0x58, 0xd7, 0xa5, 0x01,
    0x78, 0x28, 0x63, 0x62, 0x66, 0x62, 0x30, 0x34, 0x34, 0x64, 0x33, 0x65,
    0x37, 0x65, 0x66, 0x32, 0x39, 0x32, 0x35, 0x61, 0x61, 0x33, 0x36, 0x30,
    0x31, 0x61, 0x30, 0x62, 0x36, 0x34, 0x30, 0x31, 0x39, 0x34, 0x35, 0x38,
    0x34, 0x31, 0x39, 0x33, 0x61, 0x31, 0x02, 0x78, 0x28, 0x37, 0x38, 0x34,
    0x64, 0x35, 0x39, 0x61, 0x61, 0x38, 0x65, 0x36, 0x38, 0x32, 0x35, 0x65,
    0x66, 0x64, 0x38, 0x30, 0x31, 0x34, 0x39, 0x64, 0x36, 0x36, 0x64, 0x62,
    0x61, 0x61, 0x66, 0x36, 0x32, 0x64, 0x39, 0x65, 0x36, 0x38, 0x39, 0x34,
    0x34, 0x3a, 0x00, 0x47, 0x44, 0x57, 0x58, 0x44, 0xa6, 0x01, 0x01, 0x02,
    0x54, 0x78, 0x4d, 0x59, 0xaa, 0x8e, 0x68, 0x25, 0xef, 0xd8, 0x01, 0x49,
    0xd6, 0x6d, 0xba, 0xaf, 0x62, 0xd9, 0xe6, 0x89, 0x44, 0x03, 0x38, 0x1e,
    0x04, 0x81, 0x05, 0x20, 0x04, 0x21, 0x58, 0x20, 0x5a, 0xbb, 0xca, 0x2b,
    0x6e, 0xef, 0xe0, 0x9e, 0xb2, 0xf9, 0x7c, 0x61, 0xdd, 0x87, 0xaa, 0xc3,
    0xa5, 0xfe, 0x88, 0x9a, 0xa8, 0x96, 0x02, 0x80, 0x8e, 0x65, 0x47, 0x2d,
    0xc2, 0xf1, 0x85, 0x15, 0x3a, 0x00, 0x47, 0x44, 0x58, 0x42, 0x20, 0x00,
    0x3a, 0x00, 0x47, 0x44, 0x6c, 0xa1, 0x3a, 0x00, 0x47, 0x44, 0x6b, 0x58,
    0x20, 0x5c, 0x32, 0x11, 0x82, 0x0e, 0x8c, 0xaa, 0xde, 0xfc, 0xf6, 0x67,
    0x02, 0x58, 0x6c, 0x03, 0x56, 0xd8, 0x56, 0xea, 0x44, 0xb8, 0x37, 0x2e,
    0x69, 0xf4, 0x50, 0x77, 0x22, 0x9f, 0x76, 0x94, 0x66, 0x58, 0x40, 0x3d,
    0x0b, 0x9c, 0x7f, 0x4a, 0xaf, 0xea, 0x35, 0x3a, 0x5f, 0x26, 0x81, 0xe4,
    0x91, 0xa2, 0x1c, 0xae, 0xe1, 0x36, 0xd5, 0xcd, 0xe3, 0xf4, 0x3e, 0x90,
    0xfd, 0x99, 0x67, 0x61, 0xdb, 0x68, 0xa7, 0x95, 0xe8, 0xa6, 0x9d, 0x86,
    0xbc, 0x5d, 0x84, 0xa9, 0x2b, 0xa1, 0xe3, 0xfd, 0xee, 0x3a, 0x78, 0x8a,
    0x03, 0x73, 0x75, 0xa0, 0x87, 0x13, 0x3f, 0xbd, 0xd6, 0x1a, 0xef, 0xf0,
    0x90, 0xf0, 0x8b, 0x12, 0xd7, 0x02, 0x84, 0x43, 0xa1, 0x01, 0x26, 0xa1,
    0x04, 0x52, 0x41, 0x73, 0x79, 0x6d, 0x6d, 0x65, 0x74, 0x72, 0x69, 0x63,
    0x45, 0x43, 0x44, 0x53, 0x41, 0x32, 0x35, 0x36, 0x58, 0xf9, 0xa5, 0x01,
    0x78, 0x28, 0x63, 0x62, 0x66, 0x62, 0x30, 0x34, 0x34, 0x64, 0x33, 0x65,
    0x37, 0x65, 0x66, 0x32, 0x39, 0x32, 0x35, 0x61, 0x61, 0x33, 0x36, 0x30,
    0x31, 0x61, 0x30, 0x62, 0x36, 0x34, 0x30, 0x31, 0x39, 0x34, 0x35, 0x38,
    0x34, 0x31, 0x39, 0x33, 0x61, 0x31, 0x02, 0x78, 0x28, 0x63, 0x36, 0x30,
    0x32, 0x64, 0x61, 0x38, 0x63, 0x37, 0x30, 0x63, 0x62, 0x39, 0x35, 0x36,
    0x61, 0x34, 0x34, 0x38, 0x30, 0x31, 0x35, 0x31, 0x37, 0x61, 0x62, 0x31,
    0x64, 0x61, 0x65, 0x30, 0x39, 0x64, 0x61, 0x62, 0x64, 0x66, 0x62, 0x61,
    0x62, 0x3a, 0x00, 0x47, 0x44, 0x57, 0x58, 0x66, 0xa7, 0x01, 0x02, 0x02,
    0x54, 0xc6, 0x02, 0xda, 0x8c, 0x70, 0xcb, 0x95, 0x6a, 0x44, 0x80, 0x15,
    0x17, 0xab, 0x1d, 0xae, 0x09, 0xda, 0xbd, 0xfb, 0xab, 0x03, 0x26, 0x04,
    0x81, 0x02, 0x20, 0x01, 0x21, 0x58, 0x20, 0x59, 0xa3, 0x66, 0x8b, 0x0d,
    0x26, 0xcc, 0xd4, 0x9c, 0x7f, 0xb1, 0xa6, 0xb0, 0xa3, 0xf1, 0x82, 0xcd,
    0xcd, 0xca, 0xe9, 0xc6, 0x47, 0xb4, 0x47, 0x2c, 0x39, 0xcd, 0xe6, 0xbe,
    0xc8, 0x1c, 0x6b, 0x22, 0x58, 0x20, 0x5c, 0x76, 0xe6, 0xfe, 0x78, 0x9c,
    0xf7, 0xf3, 0xa1, 0x0c, 0x02, 0xe2, 0x52, 0xa3, 0x8f, 0xb2, 0x36, 0x8a,
    0x2e, 0xe6, 0x4c, 0x24, 0x81, 0x43, 0x21, 0xc0, 0x49, 0x21, 0xc7, 0x01,
    0x23, 0x3f, 0x3a, 0x00, 0x47, 0x44, 0x58, 0x42, 0x20, 0x00, 0x3a, 0x00,
    0x47, 0x44, 0x6c, 0xa1, 0x3a, 0x00, 0x47, 0x44, 0x6b, 0x58, 0x20, 0x5c,
    0x32, 0x11, 0x82, 0x0e, 0x8c, 0xaa, 0xde, 0xfc, 0xf6, 0x67, 0x02, 0x58,
    0x6c, 0x03, 0x56, 0xd8, 0x56, 0xea, 0x44, 0xb8, 0x37, 0x2e, 0x69, 0xf4,
    0x50, 0x77, 0x22, 0x9f, 0x76, 0x94, 0x66, 0x58, 0x40, 0x38, 0x05, 0xee,
    0x2e, 0x38, 0x94, 0x02, 0xf8, 0x36, 0x90, 0x05, 0x95, 0xcb, 0xaa, 0x43,
    0x76, 0x28, 0x08, 0xca, 0xe0, 0x67, 0x03, 0xc4, 0x8e, 0x0c, 0xb1, 0xb8,
    0x76, 0xed, 0xea, 0xf5, 0x4f, 0x91, 0x20, 0x33, 0xd0, 0x04, 0xee, 0x19,
    0xda, 0xdf, 0xb2, 0x49, 0xf1, 0x44, 0x1e, 0xeb, 0x8c, 0x11, 0x1d, 0x1e,
    0x45, 0x42, 0x03, 0xfa, 0x7a, 0x06, 0x28, 0x73, 0x9c, 0x06, 0xad, 0x28,
    0xab, 0x22, 0x94, 0x03, 0x12, 0x82, 0x02, 0x0a, 0x06, 0x53, 0x74, 0x61,
    0x67, 0x65, 0x30, 0x12, 0xf7, 0x01, 0x0a, 0x39, 0x74, 0x79, 0x70, 0x65,
    0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x61, 0x70, 0x69, 0x73, 0x2e,
    0x63, 0x6f, 0x6d, 0x2f, 0x6f, 0x61, 0x6b, 0x2e, 0x61, 0x74, 0x74, 0x65,
    0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x76, 0x31, 0x2e, 0x53,
    0x74, 0x61, 0x67, 0x65, 0x30, 0x4d, 0x65, 0x61, 0x73, 0x75, 0x72, 0x65,
    0x6d, 0x65, 0x6e, 0x74, 0x73, 0x12, 0xb9, 0x01, 0x0a, 0x20, 0x4c, 0xd0,
    0x20, 0x82, 0x0d, 0xa6, 0x63, 0x06, 0x3f, 0x41, 0x85, 0xca, 0x14, 0xa7,
    0xe8, 0x03, 0xcd, 0x7c, 0x9c, 0xa1, 0x48, 0x3c, 0x64, 0xe8, 0x36, 0xdb,
    0x84, 0x06, 0x04, 0xb6, 0xfa, 0xc1, 0x12, 0x20, 0xa2, 0x5a, 0x7e, 0x2a,
    0xb3, 0xba, 0xe8, 0x1f, 0xde, 0xf8, 0xb3, 0x19, 0x74, 0x59, 0x61, 0x67,
    0xef, 0x31, 0xaf, 0x59, 0x12, 0x8b, 0xa7, 0xb6, 0xe0, 0x5b, 0x5e, 0xe4,
    0x73, 0x22, 0x2b, 0x02, 0x1a, 0x20, 0x8c, 0x33, 0x5c, 0x19, 0x1d, 0x37,
    0x48, 0xd4, 0x80, 0x3a, 0x17, 0x67, 0x8f, 0xad, 0x63, 0xde, 0xa9, 0x75,
    0x7e, 0x74, 0xcc, 0x1a, 0x1d, 0x42, 0x42, 0xbd, 0x8e, 0xc2, 0x1f, 0xfe,
    0x6d, 0x0f, 0x22, 0x20, 0x80, 0x76, 0x25, 0xf9, 0x63, 0xce, 0x93, 0xab,
    0xe1, 0xf2, 0xd9, 0x96, 0x5b, 0x5d, 0xa0, 0x3c, 0x66, 0xd4, 0x76, 0xae,
    0xc9, 0x75, 0x6f, 0xde, 0xd4, 0x05, 0xce, 0x5f, 0x41, 0x18, 0x2b, 0x6f,
    0x2a, 0x20, 0xdb, 0xac, 0xca, 0xe7, 0xbf, 0xbf, 0x00, 0x6e, 0x2b, 0x86,
    0x23, 0xa8, 0x2f, 0x1a, 0x5f, 0xcd, 0xa2, 0xea, 0x03, 0x92, 0x23, 0x3c,
    0x26, 0xb1, 0x83, 0x56, 0xb3, 0xbc, 0xfa, 0xc2, 0x31, 0xeb, 0x32, 0x0d,
    0x63, 0x6f, 0x6e, 0x73, 0x6f, 0x6c, 0x65, 0x3d, 0x74, 0x74, 0x79, 0x53,
    0x30, 0x12, 0x8c, 0x01, 0x0a, 0x22, 0x6f, 0x61, 0x6b, 0x5f, 0x72, 0x65,
    0x73, 0x74, 0x72, 0x69, 0x63, 0x74, 0x65, 0x64, 0x5f, 0x6b, 0x65, 0x72,
    0x6e, 0x65, 0x6c, 0x5f, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74, 0x72,
    0x61, 0x74, 0x6f, 0x72, 0x12, 0x66, 0x0a, 0x3b, 0x74, 0x79, 0x70, 0x65,
    0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x61, 0x70, 0x69, 0x73, 0x2e,
    0x63, 0x6f, 0x6d, 0x2f, 0x6f, 0x61, 0x6b, 0x2e, 0x61, 0x74, 0x74, 0x65,
    0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2e, 0x76, 0x31, 0x2e, 0x41,
    0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x4c, 0x61,
    0x79, 0x65, 0x72, 0x44, 0x61, 0x74, 0x61, 0x12, 0x27, 0x0a, 0x23, 0x92,
    0x01, 0x20, 0x41, 0x92, 0x4e, 0x83, 0xc9, 0x81, 0xda, 0x37, 0xf5, 0xc8,
    0xe4, 0xa3, 0x6c, 0xc8, 0x23, 0xfe, 0xe7, 0x6e, 0xa5, 0x75, 0x5a, 0x55,
    0x0a, 0x6e, 0x69, 0x4f, 0x50, 0xaf, 0x23, 0x04, 0x3a, 0x61, 0x12, 0x00,
    0x04, 0x59, 0x05, 0x47, 0x30, 0x82, 0x05, 0x43, 0x30, 0x82, 0x02, 0xf7,
    0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30, 0x41, 0x06, 0x09,
    0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x34, 0xa0,
    0x0f, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04,
    0x02, 0x02, 0x05, 0x00, 0xa1, 0x1c, 0x30, 0x1a, 0x06, 0x09, 0x2a, 0x86,
    0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0d, 0x06, 0x09, 0x60,
    0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x02, 0x05, 0x00, 0xa2, 0x03,
    0x02, 0x01, 0x30, 0x30, 0x7b, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
    0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72,
    0x69, 0x6e, 0x67, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
    0x13, 0x02, 0x55, 0x53, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04,
    0x07, 0x0c, 0x0b, 0x53, 0x61, 0x6e, 0x74, 0x61, 0x20, 0x43, 0x6c, 0x61,
    0x72, 0x61, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c,
    0x02, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x04, 0x0a,
    0x0c, 0x16, 0x41, 0x64, 0x76, 0x61, 0x6e, 0x63, 0x65, 0x64, 0x20, 0x4d,
    0x69, 0x63, 0x72, 0x6f, 0x20, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x73,
    0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x09, 0x53,
    0x45, 0x56, 0x2d, 0x4d, 0x69, 0x6c, 0x61, 0x6e, 0x30, 0x1e, 0x17, 0x0d,
    0x32, 0x35, 0x30, 0x31, 0x31, 0x32, 0x32, 0x30, 0x32, 0x35, 0x35, 0x36,
    0x5a, 0x17, 0x0d, 0x33, 0x32, 0x30, 0x31, 0x31, 0x32, 0x32, 0x30, 0x32,
    0x35, 0x35, 0x36, 0x5a, 0x30, 0x7a, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03,
    0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65,
    0x72, 0x69, 0x6e, 0x67, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
    0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
    0x04, 0x07, 0x0c, 0x0b, 0x53, 0x61, 0x6e, 0x74, 0x61, 0x20, 0x43, 0x6c,
    0x61, 0x72, 0x61, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x08,
    0x0c, 0x02, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x04,
    0x0a, 0x0c, 0x16, 0x41, 0x64, 0x76, 0x61, 0x6e, 0x63, 0x65, 0x64, 0x20,
    0x4d, 0x69, 0x63, 0x72, 0x6f, 0x20, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65,
    0x73, 0x31, 0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x08,
    0x53, 0x45, 0x56, 0x2d, 0x56, 0x43, 0x45, 0x4b, 0x30, 0x76, 0x30, 0x10,
    0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x05, 0x2b,
    0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00, 0x04, 0xcd, 0x9a, 0x4f, 0x8a,
    0x45, 0xb4, 0xd4, 0x56, 0x06, 0x11, 0x71, 0x6a, 0x2b, 0x8a, 0xa7, 0x9f,
    0xbe, 0x1d, 0x00, 0xf2, 0x5d, 0xa4, 0x2d, 0x56, 0x94, 0xce, 0x17, 0xb3,
    0x0c, 0x65, 0x5c, 0xa2, 0x5d, 0x32, 0x42, 0xa0, 0xab, 0x1f, 0x4a, 0x64,
    0x45, 0x67, 0xdb, 0x1a, 0xc6, 0x09, 0x35, 0x23, 0xe1, 0x9b, 0xf1, 0xc1,
    0x0b, 0x14, 0xdb, 0x13, 0x8b, 0xfc, 0xe4, 0x05, 0x90, 0xfd, 0x73, 0xc0,
    0xae, 0xda, 0xaa, 0x25, 0x1c, 0x42, 0x32, 0xf5, 0x7d, 0x10, 0x02, 0x20,
    0xf3, 0x36, 0x04, 0xa3, 0xaa, 0xa4, 0x18, 0x8d, 0x16, 0xfb, 0xd4, 0x90,
    0xfe, 0xcb, 0x94, 0x89, 0xfb, 0xd4, 0x33, 0xfb, 0xa3, 0x82, 0x01, 0x17,
    0x30, 0x82, 0x01, 0x13, 0x30, 0x10, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x04,
    0x01, 0x9c, 0x78, 0x01, 0x01, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x17,
    0x06, 0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x9c, 0x78, 0x01, 0x02, 0x04,
    0x0a, 0x16, 0x08, 0x4d, 0x69, 0x6c, 0x61, 0x6e, 0x2d, 0x42, 0x30, 0x30,
    0x11, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x9c, 0x78, 0x01, 0x03,
    0x01, 0x04, 0x03, 0x02, 0x01, 0x03, 0x30, 0x11, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x9c, 0x78, 0x01, 0x03, 0x02, 0x04, 0x03, 0x02, 0x01,
    0x00, 0x30, 0x11, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x9c, 0x78,
    0x01, 0x03, 0x04, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x11, 0x06, 0x0a,
    0x2b, 0x06, 0x01, 0x04, 0x01, 0x9c, 0x78, 0x01, 0x03, 0x05, 0x04, 0x03,
    0x02, 0x01, 0x00, 0x30, 0x11, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01,
    0x9c, 0x78, 0x01, 0x03, 0x06, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x11,
    0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x9c, 0x78, 0x01, 0x03, 0x07,
    0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x11, 0x06, 0x0a, 0x2b, 0x06, 0x01,
    0x04, 0x01, 0x9c, 0x78, 0x01, 0x03, 0x03, 0x04, 0x03, 0x02, 0x01, 0x16,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x9c, 0x78, 0x01,
    0x03, 0x08, 0x04, 0x04, 0x02, 0x02, 0x00, 0xd1, 0x30, 0x4d, 0x06, 0x09,
    0x2b, 0x06, 0x01, 0x04, 0x01, 0x9c, 0x78, 0x01, 0x04, 0x04, 0x40, 0xde,
    0xa9, 0xbc, 0x16, 0x04, 0x62, 0xcd, 0x72, 0xc3, 0xb7, 0x77, 0xde, 0x0c,
    0x6f, 0x08, 0x6c, 0x9f, 0x6d, 0x17, 0x86, 0x8d, 0x8a, 0x1c, 0xcb, 0x18,
    0x9c, 0x71, 0xbc, 0x58, 0x36, 0x91, 0xa3, 0xdc, 0x4d, 0x69, 0x22, 0x98,
    0xee, 0xa6, 0x18, 0x9d, 0x41, 0x12, 0xe7, 0x67, 0xf8, 0x0e, 0x2c, 0x6c,
    0x71, 0x95, 0x23, 0x8a, 0x03, 0x9a, 0x90, 0x8d, 0x03, 0xb6, 0x7a, 0xc6,
    0x05, 0x1c, 0xa0, 0x30, 0x41, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x0a, 0x30, 0x34, 0xa0, 0x0f, 0x30, 0x0d, 0x06, 0x09,
    0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x02, 0x05, 0x00, 0xa1,
    0x1c, 0x30, 0x1a, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
    0x01, 0x08, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03,
    0x04, 0x02, 0x02, 0x05, 0x00, 0xa2, 0x03, 0x02, 0x01, 0x30, 0x03, 0x82,
    0x02, 0x01, 0x00, 0x66, 0x42, 0x83, 0x79, 0x97, 0xfd, 0x5a, 0xf0, 0x45,
    0xd5, 0x8b, 0xa4, 0x29, 0x20, 0x66, 0xc3, 0xc1, 0x41, 0x0b, 0x32, 0xb1,
    0x79, 0xa8, 0x80, 0xcd, 0x44, 0x3e, 0x8b, 0xb4, 0xe9, 0xbd, 0x43, 0x7c,
    0x15, 0x49, 0x60, 0x2b, 0x87, 0xa9, 0xf4, 0xa7, 0xb2, 0xab, 0xd3, 0xf0,
    0x03, 0x0a, 0x0a, 0x39, 0xa6, 0x8d, 0x2e, 0x4d, 0x33, 0xf0, 0xe7, 0xc1,
    0x43, 0xc2, 0x44, 0x7a, 0x1e, 0xbd, 0x5c, 0x1a, 0x13, 0xc1, 0xa9, 0xda,
    0xc6, 0xa5, 0x12, 0x20, 0xda, 0x4e, 0x71, 0x4e, 0x09, 0x0c, 0xc2, 0x39,
    0x07, 0xeb, 0x91, 0x4e, 0xa5, 0x48, 0xec, 0xd0, 0xf8, 0xce, 0x57, 0x48,
    0x0a, 0xac, 0xc7, 0x30, 0xe8, 0x05, 0x3c, 0x2f, 0x56, 0x73, 0xbc, 0x43,
    0xcb, 0xfd, 0x77, 0x45, 0xd3, 0xfe, 0x76, 0x47, 0xba, 0x9c, 0x37, 0x61,
    0x53, 0x45, 0x78, 0x38, 0x20, 0x86, 0xc7, 0x95, 0xb9, 0x54, 0x3b, 0x12,
    0x03, 0x55, 0x0f, 0x33, 0x89, 0x41, 0xea, 0x08, 0xd5, 0x36, 0x58, 0xdd,
    0xde, 0x10, 0x34, 0x52, 0x09, 0x83, 0xa1, 0xba, 0x81, 0x95, 0xe8, 0x8c,
    0x4e, 0xbb, 0x90, 0x09, 0x2d, 0x60, 0x20, 0x59, 0x0d, 0x4c, 0x6b, 0xd1,
    0x23, 0xc4, 0xa1, 0xcd, 0x4c, 0x4e, 0x85, 0x31, 0xb0, 0x43, 0x7c, 0x08,
    0x63, 0x5d, 0x5d, 0x1d, 0x2e, 0x72, 0x6c, 0xe1, 0xc7, 0x27, 0x8a, 0x45,
    0xac, 0x47, 0xaf, 0xac, 0x40, 0x04, 0x1f, 0x0c, 0x9a, 0x5b, 0xe6, 0x86,
    0xd6, 0x76, 0xa4, 0x7b, 0x0f, 0xd2, 0xa5, 0x6b, 0xdf, 0xbf, 0x58, 0x5a,
    0x55, 0xe0, 0x6d, 0x88, 0xa0, 0x6e, 0xb2, 0x84, 0x77, 0xb0, 0x77, 0x0b,
    0x24, 0xcc, 0xd7, 0x53, 0x22, 0xb3, 0xb9, 0xb5, 0x41, 0x0b, 0x1a, 0x45,
    0x87, 0xe6, 0xe3, 0x62, 0x06, 0xad, 0xb6, 0x23, 0xb9, 0xaa, 0x03, 0x96,
    0xbb, 0xbc, 0x4b, 0x43, 0x9c, 0xc5, 0x69, 0x99, 0xbc, 0x8b, 0x70, 0x21,
    0x67, 0x4d, 0x33, 0x05, 0xb1, 0xc3, 0xa5, 0x35, 0xbb, 0x2e, 0x81, 0x35,
    0x43, 0x7a, 0x96, 0x67, 0x7f, 0x1c, 0x39, 0xab, 0x9f, 0x50, 0xf6, 0xbf,
    0x57, 0xa8, 0x51, 0x7f, 0x07, 0xbf, 0x04, 0x2a, 0x94, 0x84, 0xe6, 0x4e,
    0xbc, 0x70, 0xab, 0xdd, 0xff, 0xf5, 0xb5, 0x73, 0xbf, 0xe6, 0xa9, 0xb5,
    0x85, 0xd5, 0xda, 0x7d, 0x5a, 0xc0, 0xf2, 0xe7, 0x86, 0xae, 0xbc, 0xd8,
    0xc7, 0x59, 0xaf, 0xd7, 0x8a, 0x8a, 0x76, 0x5e, 0xff, 0x9c, 0x0d, 0x4c,
    0x57, 0x83, 0x0f, 0x6c, 0x4f, 0x6d, 0xc9, 0x6a, 0x73, 0xfa, 0x51, 0x09,
    0x8d, 0xad, 0x95, 0xa8, 0x1c, 0x15, 0xa6, 0xc8, 0x1c, 0xa7, 0xf8, 0x57,
    0x64, 0xff, 0xc8, 0x32, 0xe1, 0xd1, 0xde, 0x1e, 0x4d, 0xb9, 0xc8, 0x88,
    0xa9, 0x52, 0x90, 0xc5, 0x60, 0x8c, 0x3b, 0x95, 0x77, 0x1a, 0x37, 0x32,
    0xbe, 0x1a, 0x14, 0x10, 0x0d, 0xbd, 0xfd, 0x24, 0x1c, 0x60, 0xf2, 0x1f,
    0x6b, 0xc8, 0x93, 0x23, 0x95, 0xa9, 0xe5, 0x8c, 0x04, 0xdd, 0x98, 0xf8,
    0x22, 0xde, 0x38, 0xd7, 0x76, 0xbb, 0x84, 0x80, 0x75, 0x9d, 0x13, 0x36,
    0x53, 0xb0, 0xd2, 0x65, 0xc2, 0xc8, 0xe0, 0x12, 0x56, 0xd2, 0x77, 0x55,
    0xe2, 0xee, 0x69, 0xf1, 0x11, 0x7b, 0x4b, 0x0c, 0x4f, 0x73, 0x4f, 0x86,
    0x32, 0x8d, 0xe5, 0x73, 0x78, 0x09, 0x8f, 0xed, 0x7a, 0xb5, 0x17, 0xe8,
    0x24, 0x52, 0x0d, 0x9e, 0xfd, 0x07, 0x58, 0x50, 0xef, 0xf2, 0x3f, 0x4a,
    0x1b, 0x1c, 0xc9, 0x06, 0x6e, 0xf9, 0x87, 0x78, 0x00, 0x21, 0x88, 0x02,
    0xed, 0xa9, 0xbc, 0x0b, 0x8d, 0x8a, 0xe5, 0x9c, 0x7b, 0x53, 0x34, 0x92,
    0x71, 0xb7, 0x56, 0x0a, 0x31, 0x6f, 0xdb, 0x94, 0xa4, 0x7b, 0x69, 0x37,
    0x5b, 0xaa, 0xa2, 0xee, 0x56, 0xc4, 0x55, 0x67, 0xbe, 0x49, 0x40};

TEST(EnclaveAttestationTest, Basic) {
  base::expected<AttestationResult, const char*> result =
      ProcessAttestation(kAttestation, kHandshakeHash);
  ASSERT_TRUE(result.has_value()) << result.error();
  EXPECT_EQ(result->boot_loader, 3);
  EXPECT_EQ(result->tee, 0);
  EXPECT_EQ(result->snp, 22);
  EXPECT_EQ(result->microcode, 209);
}

TEST(EnclaveAttestationTest, DontCrash) {
  std::vector<uint8_t> attestation(std::begin(kAttestation),
                                   std::end(kAttestation));
  const uint8_t kBitFlip = 0x40;
  for (size_t i = 0; i < sizeof(kAttestation); i++) {
    if (i > 0) {
      attestation[i - 1] ^= kBitFlip;
    }
    attestation[i] ^= kBitFlip;
    (void)ProcessAttestation(attestation, kHandshakeHash);
  }
}

}  // namespace

}  // namespace device::enclave
