/****************************************************************************
**
** Copyright (C) 2016 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the tools applications of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef UTILS_H
#define UTILS_H

#include "ui4.h"
#include <qstring.h>
#include <qlist.h>
#include <qhash.h>

QT_BEGIN_NAMESPACE

inline bool toBool(const QString &str)
{ return str.toLower() == QLatin1String("true"); }

inline QString toString(const DomString *str)
{ return str ? str->text() : QString(); }

enum StringFlags {
    Utf8String = 0x1,
    MultiLineString = 0x2
};

inline QString fixString(const QString &str, const QString &indent,
                         unsigned *stringFlags = 0)
{
    QString cursegment;
    QStringList result;
    const QByteArray utf8 = str.toUtf8();
    const int utf8Length = utf8.length();

    unsigned flags = 0;

    for (int i = 0; i < utf8Length; ++i) {
        const uchar cbyte = utf8.at(i);
        if (cbyte >= 0x80) {
            cursegment += QLatin1Char('\\');
            cursegment += QString::number(cbyte, 8);
            flags |= Utf8String;
        } else {
            switch(cbyte) {
            case '\\':
                cursegment += QLatin1String("\\\\"); break;
            case '\"':
                cursegment += QLatin1String("\\\""); break;
            case '\r':
                break;
            case '\n':
                flags |= MultiLineString;
                cursegment += QLatin1String("\\n\"\n\""); break;
            default:
                cursegment += QLatin1Char(cbyte);
            }
        }

        if (cursegment.length() > 1024) {
            result << cursegment;
            cursegment.clear();
        }
    }

    if (!cursegment.isEmpty())
        result << cursegment;


    QString joinstr = QLatin1String("\"\n");
    joinstr += indent;
    joinstr += indent;
    joinstr += QLatin1Char('"');

    QString rc(QLatin1Char('"'));
    rc += result.join(joinstr);
    rc += QLatin1Char('"');

    if (result.size() > 1)
        flags |= MultiLineString;

    if (stringFlags)
        *stringFlags = flags;

    return rc;
}

inline QHash<QString, DomProperty *> propertyMap(const QList<DomProperty *> &properties)
{
    QHash<QString, DomProperty *> map;

    for (int i=0; i<properties.size(); ++i) {
        DomProperty *p = properties.at(i);
        map.insert(p->attributeName(), p);
    }

    return map;
}

QT_END_NAMESPACE

#endif // UTILS_H
