# Copyright (c) 1996, 1997    The TERENA Association
# Copyright (c) 1998, 1999    RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Filename          :   TemplateAux.pm
# Purpose           :   Provide miscellaneous functions for related to templates.
# Author            :   Lee Wilmot
# Date              :   971212
# Language Version  :   Perl5, version 5.003_07
# OSs Tested        :   BSD
# Command Line      :   Nothing executable from command line.
# Input Files       :   NONE
# Output Files      :   NONE
# External Programs :   NONE

# Problems          :
# To Do             :
# Comments          :
#------------------------------------------------------------------------------

######################## PACKAGE INTERFACE #############################
#
# Please see the package concerned for descriptions of imported symbols.

package TemplateAux;

use strict;

BEGIN {

    use vars qw( @ISA @EXPORT_OK );

    use Exporter ();

    @ISA = qw( Exporter );
    
    @EXPORT_OK = qw(
        &get_template_key 
        &template_name &template_type
        &template_max_number &template_min_number
        &is_template_name &is_db_template &is_template_keyfield
        &rough_show_template &correct_template_name_misspellings 
        &looks_like_possible_corrupt_template_header
    );
}

use Misc qw( 
    &key_of_field
);

use RobotConfig qw( 
    :DEBUG_FLAGS
    %TEMPLATE_NAMES %TEMPLATE_DATA %TEMPLATE_MISSPELLINGS %TEMPLATE_KEYFIELDS
    $TEMPLATE_START_REG $TEMPLATE_END_REG $TEMPLATE_NAME_EXTENSION
    %F 
);

######################## PACKAGE INTERFACE #############################

#   Purpose  :  Return the key associated with a template name.
#               e.g. given 'PERSON TEMPLATE', return 'T_PERSON'.
#   In       :  $:  The name of a template.
#   Out      :  $:  The key of the template with name $name, or 'undef' if a template 
#                   with name $name doesn't exist.
#
sub get_template_key {

    my $key;
    my $name = shift @_;

    foreach $key (keys %TEMPLATE_NAMES) {
        return $key if ( $TEMPLATE_NAMES{$key} eq $name );
    }

    return undef;
}

#   Purpose  :  Return the name associated with a template key.
#               e.g. given T_PERSON return 'PERSON TEMPLATE'.
#   In       :  $:  The key of a template.
#   Out      :  $:  The name
#
sub template_name {

    return $TEMPLATE_NAMES{$_[0]};
}

#   Purpose  :      Return the type associated with a template key.
#                   e.g. given T_PERSON return 'FIELD'.
#   In       :      $:  The key of a template.
#   Out      :      $:  The template type.
#
sub template_type {

    my @data = split / /, $TEMPLATE_DATA{ $_[0] };

    return $data[0];

}

#   Purpose  :  Return the minimum number of templates with a given key
#               which may appear on a request form.
#               e.g. given T_PERSON return 0.
#   In       :  $:  The key of a template.
#   Out      :  $:  numeric: the minimum number of templates of the type given
#
sub template_min_number {

    my @data = split / /, $TEMPLATE_DATA{ $_[0] };

    return $data[1];
}

#   Purpose  :  Return the maximum number of templates with a given key
#               which may appear on a request form.
#               e.g. given T_PERSON return 0.
#   In       :  $:  The key of a template.
#   Out      :  $:  numeric: the maximum number of templates of the type given
#
sub template_max_number {

    my @data = split / /, $TEMPLATE_DATA{ $_[0] };

    return $data[2];
}

#   Purpose  :  Signify whether a given string is the same as an existing template name
#   In       :  $: The string to check
#   Out      :  $: boolean: yes/no
#
sub is_template_name {

    return ( scalar ( grep { $TEMPLATE_NAMES{$_} eq $_[0]} keys %TEMPLATE_NAMES ) )
}

#   Purpose  :  Signify whether the template corresponding to a particular key
#               is a database template (will be used in the database).
#   In       :  $:  A template key
#   Out      :  $:  boolean: yes/no
#
sub is_db_template {

    my @data = split / /, $TEMPLATE_DATA{ $_[0] };

    return ( $data[3] eq 'YES' );
}

#   Purpose  :  Look for misspellings in template names
#   In       :  $:  A template name
#   Out      :  $:  The original OR the corrected name
#
sub correct_template_name_misspellings {

    my $name_to_correct = shift @_;

    my $misspelling;
    
    # Go through the hash of possible misspellings.

    foreach $misspelling ( keys %TEMPLATE_MISSPELLINGS ) {

        # If our string matches a known misspelling, correct it to
        # the proper value

        $name_to_correct = $TEMPLATE_MISSPELLINGS{$misspelling}
            if ( $name_to_correct =~ $misspelling );
    }

    return $name_to_correct;
}

#   Purpose  :  Given a field name, decide whether it is the keyfield
#               of a template. Yes -> return the key of the template
#               type. No -> undef
#   In       :  $:  a field name
#   Out      :  $:  a template key, or undef if the field name isn't
#					the key field of any template.
#   
sub is_template_keyfield {

    return $TEMPLATE_KEYFIELDS{ &key_of_field( $_[0] ) }
        if ( scalar ( grep { $F{$_} eq $_[0] } keys %TEMPLATE_KEYFIELDS  ) > 0);

    return undef;
}

# Purpose   :   Show a template for debugging purposes. We want to see all
#               of the fields and fields separators etc.
# In        :   $%: pointer to template to show
# Out       :   VOID
#
sub rough_show_template {

    my $template_ref = shift @_;

    print "--------- *";
    print  ( $template_ref->{$F{TNAME}} || "TEMPLATE NAME UNDEFINED" );
    print "* ---------\n";

    my $key;

    foreach $key (keys %$template_ref) {
	print "--->", $key, ":\t";
        print $template_ref->{$key} || "<Undefined>";
        print "\n";
    }
}

# Purpose   :   Check for possible missed template headers because they
#               avoided the correct format. Basically, if they missed
#				out a marker, or the $TEMPLATE_NAME_EXTENSION
# In        :   $: line to check
# Out       :   $: boolean
#
sub looks_like_possible_corrupt_template_header {
    
    my $possible_header = shift @_;

    my ( $key, $this_name );

    foreach $key ( keys %TEMPLATE_NAMES ) {

        $this_name = $TEMPLATE_NAMES{$key};

        $this_name =~ s/\s*$TEMPLATE_NAME_EXTENSION\s*$//;

        return 1
            if ( $possible_header =~ /$this_name\s*$TEMPLATE_END_REG/i 
				|| $possible_header =~ /$TEMPLATE_START_REG\s*$this_name/i
			); 
    }

	return 0;
}


1;
