# Copyright (c) 1998                            RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Filename          :   ReportAux.pm
# Purpose           :   Provide routines to produce an email report about
#                       a set of templates held in @TEMPLATES.  
# Author            :   Lee Wilmot
# Date              :   19971118
# Language Version  :   Perl 5.003_07
# OSs Tested        :   BSD
# Input Files       :   $EMAILTEXT_FILE     :   texts used in generating a report.
#                       $PROBLEMTEXT_FILE   :   a set of texts describing all the
#                                               problems which request can generate.
# Output Files      :   NONE
# External Programs :   NONE
# Problems          :   
# To Do             :
# Comments          :
# Description       :

############################ PACKAGE INTERFACE ###############################
#
# Please see the package concerned for descriptions of imported symbols.
#
# ### FIELD NAMES ADDED TO TEMPLATES BY THIS PACKAGE ###
# 
# $F{TOTALERROR} $F{TOTALWARN} $F{TOTALINFORM}
#
# Added only to meta-template. Set to zero, therefore defined even if no
# errors occur, for instance. Counts the total number of each problem
# type in all of the templates.
#
# $F{WARNCOUNT} $F{ERRORCOUNT} $F{INFORMCOUNT}

# Added to all templates. Set to zero initially, therefore defined even
# if there are no problems with a template.

package ReportAux;

use strict;

BEGIN {

    use vars qw( @EXPORT_OK @ISA );

    use Exporter ();
    
    @ISA = qw(Exporter);
    
    @EXPORT_OK = qw( 
            &get_problem_texts &get_email_texts &get_web_texts
            &count_errors_and_warns &substitute_variables_in_text
    );
}

use Misc qw(
        &dprint &fatal
);

use Check qw(
        &prob
);

use RobotConfig qw(
        $TEXTS_IGNORE_LINE_REG  $TEXTEND_REG

        $PROBLEMTEXT_FILE $WEBTEXT_FILE  $EMAILTEXT_FILE
        %TEXT_VARIABLES

        %F $FORM_FS $PROB_FS

        $MAX_NO_WARNS_IN_TEMPLATE_THRESHOLD
        $MAX_NO_ERRORS_IN_TEMPLATE_THRESHOLD
);

#   Purpose :   Load problem texts from a file into three hashes.
#   In      :   $:  Flag: whether to convert to text format.
#               $%: Pointer to hash which will hold the problem types.
#               $%: Pointer to hash which will hold the short problem texts.
#               $%: Pointer to hash which will hold the long problem texts.
#   Out     :   VOID
#   Comments:   The problem type, short description and long description are
#               each read into a separate hash, with the key being the problem
#               code (see $RobotConfig:PROBLEMTEXTS_FILE for more details of
#               'problem code' etc.)
#
sub get_problem_texts {

    my ( $email_report_flag, $prob_type, $prob_short, $prob_long ) = @_;

    my $problem_type;

    &dprint("READING: $PROBLEMTEXT_FILE");

    open TEXTS, $PROBLEMTEXT_FILE or
        &fatal("$! while opening problem text file $PROBLEMTEXT_FILE.");

    my ( $code, $short_text, $long_text );

    while (<TEXTS>) {

        next if /$TEXTS_IGNORE_LINE_REG/;             # ignore comments/blank lines

        ########## GET PROBLEM CODE ###########

        chop;

        $code = $_;
    
        &fatal("Format error in $PROBLEMTEXT_FILE: code = $code.")
            if ( $code =~ /$TEXTEND_REG/ || 
                 $code =~ /^\s*$/ || 
                 $code !~ /^[A-Z_]+\s*$/ ); 
    
        ########## GET PROBLEM TYPE (ERROR/WARN) ###########
    
        $_ = <TEXTS>;
        $_ = <TEXTS>    
            while /$TEXTS_IGNORE_LINE_REG/;         # ignore comments/blank lines
    
        chop;

        $problem_type = $_;
    
        &fatal("Format error in $PROBLEMTEXT_FILE: code = $code: problem type $problem_type bad format.")
            if ( $problem_type !~ /^WARNING$/i && 
                 $problem_type !~ /^ERROR$/i && 
                 $problem_type !~ /^INFORM$/i ); 
    
        ########## GET SHORT PROBLEM DESCRIPTION ###########
    
        $_ = <TEXTS>;
        $_ = <TEXTS>    
            while /$TEXTS_IGNORE_LINE_REG/;         # ignore comments/blank lines
    
        $short_text = $_;

        &fatal("Format error in $PROBLEMTEXT_FILE: code = $code, short_text = $short_text.")
            if ( $short_text =~ $TEXTEND_REG || $short_text =~ /^\s*$/ ); 
    
        ######## GET LONG PROBLEM DESCRIPTION ##############
    
        $long_text = "";

        while (<TEXTS>) {

            next if /$TEXTS_IGNORE_LINE_REG/;       # ignore comments/blank lines

            last if /$TEXTEND_REG/;
    
            $long_text .= $_;
        }

        &fatal("Format error in $PROBLEMTEXT_FILE code = $code, short_text = $short_text, long_text = $long_text.")
            if ($long_text =~ $TEXTEND_REG || $long_text =~ /^\s*&/);
    
        chop $long_text;
    
        # Replace variables in text with the values defined in the config file

        &substitute_variables_in_text(\$long_text);
    
        # If it's an email report need to make some changes to the text

        if ($email_report_flag) {

            &html_to_text(\$long_text);
            &html_to_text(\$short_text);
                

            $long_text =~ s/ +/ /g;            # compress multiple spaces to tidy up
        }   
    
        $prob_type->{$code} = $problem_type;
        $prob_short->{$code} = $short_text;
        $prob_long->{$code} = $long_text;
    }

    &fatal("$! while closing pipe from problem text file $PROBLEMTEXT_FILE.")
        if ( ! close TEXTS );
}

#   Purpose :   Read in texts from a file into a hash which will be used in
#               composing an email report.
#   In      :   $%: pointer to a hash to place the texts into.
#   Out     :   VOID
#
sub get_email_texts {

    my $hash_ref = shift @_;

    &dprint("READING: $EMAILTEXT_FILE");

    &fatal("$! while opening email text file $EMAILTEXT_FILE")
        if ( ! open TEXTS, $EMAILTEXT_FILE );
    
    my ( $code, $text );
    
    while (<TEXTS>) {

        next if /$TEXTS_IGNORE_LINE_REG/;             # ignore comments/blank lines

        ########## GET CODE FOR TEXT ###########

        chop;

        $code = $_;

        &fatal("Format error in $EMAILTEXT_FILE: code = $code.")
            if ( $code =~ /$TEXTEND_REG/ || $code =~ /^\s*$/ || $code !~ /^[A-Z_]+\s*$/); 
               
        ############### GET TEXT ##############

        $text = "";

        while (<TEXTS>) {

            last if /$TEXTEND_REG/;

            $text .= $_;
        }
            
        &fatal("Format error in $EMAILTEXT_FILE code = $code, text = $text.")
            if ($text =~ $TEXTEND_REG || $text =~ /^\s*&/);
    
        $hash_ref->{$code} = $text;
    }

    &fatal("$! while closing pipe from email text file $EMAILTEXT_FILE.")
        if ( ! close TEXTS );
}

#   Purpose :   Read in texts from a file into a hash which will be used in
#               composing a web report.
#   In      :   $%: pointer to a hash to place the texts into.
#   Out     :   VOID
#   Comments:   get_email_texts should probably do this
#
sub get_web_texts {

    my $hash_ref = shift @_;

    &dprint("READING: $WEBTEXT_FILE");

    &fatal("$! while opening email text file $WEBTEXT_FILE")
        if ( ! open TEXTS, $WEBTEXT_FILE );
    
    my ( $code, $text );
    
    while (<TEXTS>) {

        next if /$TEXTS_IGNORE_LINE_REG/;             # ignore comments/blank lines

        ########## GET CODE FOR TEXT ###########

        chop;

        $code = $_;

        &fatal("Format error in $WEBTEXT_FILE: code = $code.")
            if ( $code =~ /$TEXTEND_REG/ || $code =~ /^\s*$/ || $code !~ /^[A-Z_]+\s*$/); 
               
        ############### GET TEXT ##############

        $text = "";

        while (<TEXTS>) {

            last if /$TEXTEND_REG/;

            $text .= $_;
        }
            
        &fatal("Format error in $WEBTEXT_FILE code = $code, text = $text.")
            if ($text =~ $TEXTEND_REG || $text =~ /^\s*&/);
    
        $hash_ref->{$code} = $text;
    }

    &fatal("$! while closing pipe from email text file $WEBTEXT_FILE.")
        if ( ! close TEXTS );
}


#   Purpose :   Convert HTML based texts into pure text.
#   In      :   $$: pointer to the text to convert.
#   Out     :   VOID
#
sub html_to_text {

    my $textref = shift @_;

    $$textref =~ s/<\/{0,1}EMPH>/*/ig;                  # various emphasis tags
    $$textref =~ s/<\/{0,1}B>/*/ig;
    $$textref =~ s/<\/{0,1}CODE>/-/ig;

    $$textref =~ s/<A HREF=".*">(.*)<\/A>/$1/ig;        # hyperlink
    
    # Remove any existing newlines in the text. Doesn't make sense to
    # interpret newlines in the text AND the HTML codes. NOTE
    # MUST do this before translating the breaks!

    $$textref =~ s/\n/ /g;

    $$textref =~ s/<BR>/\n/ig;                          # break tags
    $$textref =~ s/<P>/\n\n/ig;
}

#   Purpose :   Replace any variables which occur in the given text and are 
#               mentioned in %TEXT_VARIABLES with the value defined there.
#   In      :   $$: pointer to the text to check.
#   Out     :   VOID
#
sub substitute_variables_in_text {

    my $textref = shift @_;

    my $key;

    foreach $key (keys %TEXT_VARIABLES) {

        # Variables begin with a '?'

        $$textref =~ s/\?$key/$TEXT_VARIABLES{$key}/ig;
    }
}

#Go through all the templates. Count errors and warnings separately
#for each template, and also count total number of each. This information
# is used later in deciding on output format etc.
#DONT count informs

#   Purpose :   Count the numbers of each of the different problem types
#               in each template.
#   In      :   $@: pointer to the array of extracted templates
#           :   
#           :
#           :
#           :
#   Out     :   VOID
#
sub count_errors_and_warns {

    my $templates_ref = shift @_;
    my $problem_types = shift @_;

    my ( $total_error_ref, $total_warning_ref, $total_inform_ref ) = @_;

    my ( @problems, $this_prob, @prob_parts, $this_prob_code );

    my $hash_ref;

    foreach $hash_ref ( @$templates_ref ) {

        $hash_ref->{$F{ERRORCOUNT}} = 0; 
        $hash_ref->{$F{WARNCOUNT}} = 0;
        $hash_ref->{$F{INFORMCOUNT}} = 0;

        next 
            if ( ! exists $hash_ref->{$F{PROBLEMS}} );

        # Get the problems into an array
    
        @problems = @{ $hash_ref->{$F{PROBLEMS}} };

        # Look at each one in turn
    
        while ( @problems ) {

            $this_prob = shift @problems;
            
            # Split the current problem into its parts
            # (problem code + arguments)

            @prob_parts = split /$PROB_FS/, $this_prob;

            $this_prob_code = shift @prob_parts;

            if ($problem_types->{$this_prob_code} =~ /ERROR/i) {
                $$total_error_ref++;
                $hash_ref->{$F{ERRORCOUNT}}++;
            }
            elsif ($problem_types->{$this_prob_code} =~ /WARNING/i) {
                $$total_warning_ref++;
                $hash_ref->{$F{WARNCOUNT}}++;
            }   
            elsif ($problem_types->{$this_prob_code} =~ /INFORM/i) {
                $$total_inform_ref++;
                $hash_ref->{$F{INFORMCOUNT}}++;
            }
            else {
                fatal("Undefined error code without type found $this_prob_code");
            }
        }
    
        # There is a variable in the config file to prevent the user being 
        # flooded with errors of a type. If this occurs, it's noted here
        # as a remark.

        if ( $hash_ref->{$F{ERRORCOUNT}} > $MAX_NO_ERRORS_IN_TEMPLATE_THRESHOLD) {

            &prob($hash_ref, "MAX_NO_OF_PROBLEMS_OF_A_TYPE_EXCEEDED", "errors");
            $$total_inform_ref++;
            $hash_ref->{$F{INFORMCOUNT}}++;
        }
        
        if ( $hash_ref->{$F{WARNCOUNT}} > $MAX_NO_WARNS_IN_TEMPLATE_THRESHOLD) {

            &prob($hash_ref, "MAX_NO_OF_PROBLEMS_OF_A_TYPE_EXCEEDED", "warnings");
            $$total_inform_ref++;
            $hash_ref->{$F{INFORMCOUNT}}++;
        }
    }
}

1;
