from __future__ import annotations

import abc
import functools
import importlib.util
import logging
import os
import platform
import shutil
import subprocess
import sys
import sysconfig
import tempfile
import typing
import warnings

from collections.abc import Collection, Mapping

from ._exceptions import FailedProcessError
from ._util import check_dependency


if sys.version_info >= (3, 8):
    from typing import Protocol
elif typing.TYPE_CHECKING:
    from typing_extensions import Protocol
else:
    Protocol = abc.ABC


_logger = logging.getLogger(__name__)


class IsolatedEnv(Protocol):
    """Isolated build environment ABC."""

    @property
    @abc.abstractmethod
    def python_executable(self) -> str:
        """The Python executable of the isolated environment."""

    @abc.abstractmethod
    def make_extra_environ(self) -> Mapping[str, str] | None:
        """Generate additional env vars specific to the isolated environment."""


@functools.lru_cache(maxsize=None)
def _should_use_virtualenv() -> bool:
    import packaging.requirements

    # virtualenv might be incompatible if it was installed separately
    # from build. This verifies that virtualenv and all of its
    # dependencies are installed as specified by build.
    return importlib.util.find_spec('virtualenv') is not None and not any(
        packaging.requirements.Requirement(d[1]).name == 'virtualenv'
        for d in check_dependency('build[virtualenv]')
        if len(d) > 1
    )


def _minimum_pip_version() -> str:
    if platform.system() == 'Darwin' and int(platform.mac_ver()[0].split('.')[0]) >= 11:
        # macOS 11+ name scheme change requires 20.3. Intel macOS 11.0 can be
        # told to report 10.16 for backwards compatibility; but that also fixes
        # earlier versions of pip so this is only needed for 11+.
        is_apple_silicon_python = platform.machine() != 'x86_64'
        return '21.0.1' if is_apple_silicon_python else '20.3.0'

    # PEP-517 and manylinux1 was first implemented in 19.1
    return '19.1.0'


def _has_valid_pip(__version: str | None = None, **distargs: object) -> bool:
    """
    Given a path, see if Pip is present and return True if the version is
    sufficient for build, False if it is not. ModuleNotFoundError is thrown if
    pip is not present.
    """

    import packaging.version

    from ._compat import importlib

    name = 'pip'

    try:
        pip_distribution = next(iter(importlib.metadata.distributions(name=name, **distargs)))
    except StopIteration:
        raise ModuleNotFoundError(name) from None

    current_pip_version = packaging.version.Version(pip_distribution.version)

    return current_pip_version >= packaging.version.Version(__version or _minimum_pip_version())


@functools.lru_cache(maxsize=None)
def _valid_global_pip() -> bool | None:
    """
    This checks for a valid global pip. Returns None if pip is missing, False
    if Pip is too old, and True if it can be used.
    """

    try:
        # Version to have added the `--python` option.
        return _has_valid_pip('22.3')
    except ModuleNotFoundError:
        return None


def _subprocess(cmd: list[str]) -> None:
    """Invoke subprocess and output stdout and stderr if it fails."""
    try:
        subprocess.run(cmd, check=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError as e:
        print(e.output.decode(), end='', file=sys.stderr)
        raise


class DefaultIsolatedEnv(IsolatedEnv):
    """An isolated environment which combines venv and virtualenv with pip."""

    def __enter__(self) -> DefaultIsolatedEnv:
        try:
            self._path = tempfile.mkdtemp(prefix='build-env-')
            # use virtualenv when available (as it's faster than venv)
            if _should_use_virtualenv():
                self.log('Creating virtualenv isolated environment...')
                self._python_executable, self._scripts_dir = _create_isolated_env_virtualenv(self._path)
            else:
                self.log('Creating venv isolated environment...')

                # Call ``realpath`` to prevent spurious warning from being emitted
                # that the venv location has changed on Windows. The username is
                # DOS-encoded in the output of tempfile - the location is the same
                # but the representation of it is different, which confuses venv.
                # Ref: https://bugs.python.org/issue46171
                self._path = os.path.realpath(tempfile.mkdtemp(prefix='build-env-'))
                self._python_executable, self._scripts_dir = _create_isolated_env_venv(self._path)
        except Exception:  # cleanup folder if creation fails
            self.__exit__(*sys.exc_info())
            raise

        return self

    def __exit__(self, *args: object) -> None:
        if os.path.exists(self._path):  # in case the user already deleted skip remove
            shutil.rmtree(self._path)

    @property
    def path(self) -> str:
        """The location of the isolated build environment."""
        return self._path

    @property
    def python_executable(self) -> str:
        """The python executable of the isolated build environment."""
        return self._python_executable

    def _pip_args(self) -> list[str]:
        if _valid_global_pip():
            return [sys.executable, '-m', 'pip', '--python', self.python_executable]
        else:
            return [self.python_executable, '-Im', 'pip']

    def make_extra_environ(self) -> dict[str, str]:
        path = os.environ.get('PATH')
        return {'PATH': os.pathsep.join([self._scripts_dir, path]) if path is not None else self._scripts_dir}

    def install(self, requirements: Collection[str]) -> None:
        """
        Install packages from PEP 508 requirements in the isolated build environment.

        :param requirements: PEP 508 requirement specification to install

        :note: Passing non-PEP 508 strings will result in undefined behavior, you *should not* rely on it. It is
               merely an implementation detail, it may change any time without warning.
        """
        if not requirements:
            return

        self.log(f'Installing packages in isolated environment... ({", ".join(sorted(requirements))})')

        # pip does not honour environment markers in command line arguments
        # but it does for requirements from a file
        with tempfile.NamedTemporaryFile('w', prefix='build-reqs-', suffix='.txt', delete=False, encoding='utf-8') as req_file:
            req_file.write(os.linesep.join(requirements))
        try:
            cmd = [
                *self._pip_args(),
                'install',
                '--use-pep517',
                '--no-warn-script-location',
                '-r',
                os.path.abspath(req_file.name),
            ]
            _subprocess(cmd)
        finally:
            os.unlink(req_file.name)

    @staticmethod
    def log(message: str) -> None:
        """
        Prints message

        The default implementation uses the logging module but this function can be
        overwritten by users to have a different implementation.

        :param msg: Message to output
        """
        if sys.version_info >= (3, 8):
            _logger.log(logging.INFO, message, stacklevel=2)
        else:
            _logger.log(logging.INFO, message)


def _create_isolated_env_virtualenv(path: str) -> tuple[str, str]:
    """
    We optionally can use the virtualenv package to provision a virtual environment.

    :param path: The path where to create the isolated build environment
    :return: The Python executable and script folder
    """
    import virtualenv

    if _valid_global_pip():
        cmd = [path, '--no-seed', '--activators', '']
    else:
        cmd = [path, '--no-setuptools', '--no-wheel', '--activators', '']

    result = virtualenv.cli_run(cmd, setup_logging=False)
    executable = str(result.creator.exe)
    script_dir = str(result.creator.script_dir)
    return executable, script_dir


@functools.lru_cache(maxsize=None)
def _fs_supports_symlink() -> bool:
    """Return True if symlinks are supported"""
    # Using definition used by venv.main()
    if not sys.platform.startswith('win'):
        return True

    # Windows may support symlinks (setting in Windows 10)
    with tempfile.NamedTemporaryFile(prefix='build-symlink-') as tmp_file:
        dest = f'{tmp_file}-b'
        try:
            os.symlink(tmp_file.name, dest)
            os.unlink(dest)
        except (OSError, NotImplementedError, AttributeError):
            return False
        return True


def _create_isolated_env_venv(path: str) -> tuple[str, str]:
    """
    On Python 3 we use the venv package from the standard library.

    :param path: The path where to create the isolated build environment
    :return: The Python executable and script folder
    """
    import venv

    symlinks = _fs_supports_symlink()
    try:
        with warnings.catch_warnings():
            if sys.version_info[:3] == (3, 11, 0):
                warnings.filterwarnings('ignore', 'check_home argument is deprecated and ignored.', DeprecationWarning)
            venv.EnvBuilder(with_pip=not _valid_global_pip(), symlinks=symlinks).create(path)
    except subprocess.CalledProcessError as exc:
        raise FailedProcessError(exc, 'Failed to create venv. Maybe try installing virtualenv.') from None

    executable, script_dir, purelib = _find_executable_and_scripts(path)

    # Get the version of pip in the environment
    if not _valid_global_pip() and not _has_valid_pip(path=[purelib]):
        _subprocess([executable, '-m', 'pip', 'install', f'pip>={_minimum_pip_version()}'])

    # Avoid the setuptools from ensurepip to break the isolation
    if not _valid_global_pip():
        _subprocess([executable, '-m', 'pip', 'uninstall', 'setuptools', '-y'])

    return executable, script_dir


def _find_executable_and_scripts(path: str) -> tuple[str, str, str]:
    """
    Detect the Python executable and script folder of a virtual environment.

    :param path: The location of the virtual environment
    :return: The Python executable, script folder, and purelib folder
    """
    config_vars = sysconfig.get_config_vars().copy()  # globally cached, copy before altering it
    config_vars['base'] = path
    scheme_names = sysconfig.get_scheme_names()
    if 'venv' in scheme_names:
        # Python distributors with custom default installation scheme can set a
        # scheme that can't be used to expand the paths in a venv.
        # This can happen if build itself is not installed in a venv.
        # The distributors are encouraged to set a "venv" scheme to be used for this.
        # See https://bugs.python.org/issue45413
        # and https://github.com/pypa/virtualenv/issues/2208
        paths = sysconfig.get_paths(scheme='venv', vars=config_vars)
    elif 'posix_local' in scheme_names:
        # The Python that ships on Debian/Ubuntu varies the default scheme to
        # install to /usr/local
        # But it does not (yet) set the "venv" scheme.
        # If we're the Debian "posix_local" scheme is available, but "venv"
        # is not, we use "posix_prefix" instead which is venv-compatible there.
        paths = sysconfig.get_paths(scheme='posix_prefix', vars=config_vars)
    elif 'osx_framework_library' in scheme_names:
        # The Python that ships with the macOS developer tools varies the
        # default scheme depending on whether the ``sys.prefix`` is part of a framework.
        # But it does not (yet) set the "venv" scheme.
        # If the Apple-custom "osx_framework_library" scheme is available but "venv"
        # is not, we use "posix_prefix" instead which is venv-compatible there.
        paths = sysconfig.get_paths(scheme='posix_prefix', vars=config_vars)
    else:
        paths = sysconfig.get_paths(vars=config_vars)
    executable = os.path.join(paths['scripts'], 'python.exe' if sys.platform.startswith('win') else 'python')
    if not os.path.exists(executable):
        msg = f'Virtual environment creation failed, executable {executable} missing'
        raise RuntimeError(msg)

    return executable, paths['scripts'], paths['purelib']


__all__ = [
    'IsolatedEnv',
    'DefaultIsolatedEnv',
]
