"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;

var _opensearch_security_client = require("../../backend/opensearch_security_client");

var _tenant_resolver = require("../../multitenancy/tenant_resolver");

var _errors = require("../../errors");

var _common = require("../../../common");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   *   Copyright OpenSearch Contributors
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   Licensed under the Apache License, Version 2.0 (the "License").
                                                                                                                                                                                                                   *   You may not use this file except in compliance with the License.
                                                                                                                                                                                                                   *   A copy of the License is located at
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *       http://www.apache.org/licenses/LICENSE-2.0
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   or in the "license" file accompanying this file. This file is distributed
                                                                                                                                                                                                                   *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
                                                                                                                                                                                                                   *   express or implied. See the License for the specific language governing
                                                                                                                                                                                                                   *   permissions and limitations under the License.
                                                                                                                                                                                                                   */

class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;

    _defineProperty(this, "type", void 0);

    _defineProperty(this, "securityClient", void 0);

    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;

      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      }

      const authState = {}; // if browser request, auth logic is:
      //   1. check if request includes auth header or paramter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers

      const authHeaders = {};
      let cookie;
      let authInfo; // if this is an REST API call, suppose the request includes necessary auth header
      // see https://www.elastic.co/guide/en/opensearch-dashboards/master/using-api.html

      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additonalAuthHeader = await this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additonalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additonalAuthHeader);
          cookie = this.getCookie(request, authInfo); // set tenant from cookie if exist

          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();

          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }

          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }

        if (!cookie || !(await this.isValidCookie(cookie, request))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear(); // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.

          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          } // send to auth workflow


          return this.handleUnauthedRequest(request, response, toolkit);
        } // extend session expiration time


        if (this.config.session.keepalive) {
          cookie.expiryTime = Date.now() + this.config.session.ttl;
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } // cookie is valid
        // build auth header


        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie, request);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additonalAuthHeader = await this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additonalAuthHeader);
      } // resolve tenant if necessary


      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo); // return 401 if no tenant available

          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          }

          authState.selectedTenant = tenant; // set tenant in header

          if (this.config.multitenancy.enabled && this.config.multitenancy.enable_aggregation_view) {
            // Store all saved objects in a single kibana index.
            Object.assign(authHeaders, {
              securitytenant: _common.GLOBAL_TENANT_SYMBOL
            });
          } else {
            Object.assign(authHeaders, {
              securitytenant: tenant
            });
          } // set tenant to cookie


          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);

          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }

            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }

          throw error;
        }
      }

      if (!authInfo) {
        authInfo = await this.securityClient.authinfo(request, authHeaders);
      }

      authState.authInfo = authInfo;
      return toolkit.authenticated({
        requestHeaders: authHeaders,
        state: authState
      });
    });

    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
    this.config = config;
  }

  authNotRequired(request) {
    const pathname = request.url.pathname;

    if (!pathname) {
      return false;
    } // allow requests to ignored routes


    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    } // allow requests to routes that doesn't require authentication


    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }

    return false;
  }

  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }

    const dashboardsInfo = await this.securityClient.dashboardsinfo(request, authHeader);
    return (0, _tenant_resolver.resolveTenant)({
      request,
      username: authInfo.user_name,
      roles: authInfo.roles,
      availabeTenants: authInfo.tenants,
      config: this.config,
      cookie,
      multitenancyEnabled: dashboardsInfo.multitenancy_enabled,
      privateTenantEnabled: dashboardsInfo.private_tenant_enabled,
      defaultTenant: dashboardsInfo.default_tenant
    });
  }

  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  } // abstract functions for concrete auth types to implement


}

exports.AuthenticationType = AuthenticationType;

_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/api/core/capabilities', // FIXME: need to figureout how to bypass this API call
'/app/login']);

_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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