"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToCSV = exports.buildRequestBody = void 0;

var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));

var _json2Csv = _interopRequireDefault(require("json-2-csv"));

var _lodash = _interopRequireDefault(require("lodash"));

var _moment = _interopRequireDefault(require("moment"));

var _common = require("../../../../../src/plugins/data/common");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
var metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
}; // Get the selected columns by the user.

exports.metaData = metaData;

const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;

  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }

  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
}; // Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query


exports.getSelectedFields = getSelectedFields;

const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();

  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig); // Add time range

  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }

  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  } // Add sorting to the query


  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);

  if (report._source.sorting.length > 0) {
    const sortings = report._source.sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });

    esbSearchQuery.sorts(sortings);
  } // add selected fields to query


  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  } // Add a customizer to merge queries to generate request body


  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });

  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
}; // Fetch the data from OpenSearch


exports.buildRequestBody = buildRequestBody;

const getOpenSearchData = (arrayHits, report, params, dateFormat) => {
  let hits = [];

  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields; // get all the fields of type date and format them to excel format

      let tempKeyElement = [];

      for (let dateField of report._source.dateFields) {
        let keys;
        keys = dateField.split('.');
        const dateValue = data._source[dateField];
        const fieldDateValue = fields[dateField];
        const isDateFieldPresent = isKeyPresent(data._source, dateField);

        if (isDateFieldPresent) {
          // if its not a nested date field
          if (keys.length === 1) {
            // if conditions to determine if the date field's value is an array or a string
            if (typeof dateValue === 'string') {
              data._source[keys] = (0, _moment.default)(dateValue).format(dateFormat);
            } else if (fieldDateValue.length !== 0 && fieldDateValue instanceof Array) {
              fieldDateValue.forEach((element, index) => {
                data._source[keys][index] = (0, _moment.default)(element).format(dateFormat);
              });
            } else {
              data._source[keys] = [];
            } // else to cover cases with nested date fields

          } else {
            let keyElement = keys.shift(); // if conditions to determine if the date field's value is an array or a string

            if (typeof fieldDateValue === 'string') {
              keys.push((0, _moment.default)(fieldDateValue).format(dateFormat));
            } else if (fieldDateValue.length !== 0 && fieldDateValue instanceof Array) {
              let tempArray = [];
              fieldDateValue.forEach(index => {
                tempArray.push((0, _moment.default)(index).format(dateFormat));
              });
              keys.push(tempArray);
            } else {
              keys.push([]);
            }

            const nestedJSON = arrayToNestedJSON(keys);
            let keyLength = Object.keys(data._source); // to check if the nested field have anyother keys apart from date field

            if (tempKeyElement.includes(keyElement) || keyLength.length > 1) {
              data._source[keyElement] = { ...data._source[keyElement],
                ...nestedJSON
              };
            } else {
              data._source[keyElement] = nestedJSON;
              tempKeyElement.push(keyElement);
            }
          }
        }
      }

      delete data['fields'];

      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      } // Truncate to expected limit size


      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }

  return hits;
}; //Convert the data to Csv format


exports.getOpenSearchData = getOpenSearchData;

const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};

exports.convertToCSV = convertToCSV;

function flattenHits(hits, result = {}, prefix = '') {
  for (const [key, value] of Object.entries(hits)) {
    if (!hits.hasOwnProperty(key)) continue;

    if (value != null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, prefix + key + '.');
    } else {
      result[prefix.replace(/^_source\./, '') + key] = value;
    }
  }

  return result;
} //Return only the selected fields


function traverse(data, keys, result = {}) {
  data = flattenHits(data);
  const sourceKeys = Object.keys(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);

    if (value !== undefined) result[key] = value;else {
      Object.keys(data).filter(sourceKey => sourceKey.startsWith(key + '.')).forEach(sourceKey => result[sourceKey] = data[sourceKey]);
    }
  });
  return result;
}
/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */


function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;

    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }

  return doc;
}

function arrayToNestedJSON(arr) {
  if (arr.length === 0) {
    return null;
  } else if (arr.length === 1) {
    return arr[0];
  } else {
    const key = arr[0];
    const rest = arr.slice(1);
    return {
      [key]: arrayToNestedJSON(rest)
    };
  }
}

function isKeyPresent(data, key) {
  if (typeof data === 'object' && data !== null) {
    if (key in data) {
      return true;
    }

    for (const value of Object.values(data)) {
      if (isKeyPresent(value, key)) {
        return true;
      }
    }
  }

  return false;
}

const addDocValueFields = (report, requestBody) => {
  const docValues = [];

  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  } // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.


  requestBody = { ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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