"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTemplate = exports.validateInstance = void 0;

var _ajv = _interopRequireDefault(require("ajv"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const ajv = new _ajv.default();
const staticAsset = {
  type: 'object',
  properties: {
    path: {
      type: 'string'
    },
    annotation: {
      type: 'string',
      nullable: true
    }
  },
  required: ['path'],
  additionalProperties: false
};
const templateSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    version: {
      type: 'string'
    },
    displayName: {
      type: 'string',
      nullable: true
    },
    license: {
      type: 'string'
    },
    type: {
      type: 'string'
    },
    labels: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    author: {
      type: 'string',
      nullable: true
    },
    description: {
      type: 'string',
      nullable: true
    },
    sourceUrl: {
      type: 'string',
      nullable: true
    },
    statics: {
      type: 'object',
      properties: {
        logo: { ...staticAsset,
          nullable: true
        },
        gallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        },
        darkModeLogo: { ...staticAsset,
          nullable: true
        },
        darkModeGallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        }
      },
      additionalProperties: false,
      nullable: true
    },
    components: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          }
        },
        required: ['name', 'version']
      }
    },
    assets: {
      type: 'object',
      properties: {
        savedObjects: {
          type: 'object',
          properties: {
            name: {
              type: 'string'
            },
            version: {
              type: 'string'
            }
          },
          required: ['name', 'version'],
          nullable: true,
          additionalProperties: false
        },
        queries: {
          type: 'array',
          items: {
            type: 'object',
            properties: {
              name: {
                type: 'string'
              },
              version: {
                type: 'string'
              },
              language: {
                type: 'string'
              }
            },
            required: ['name', 'version', 'language']
          },
          nullable: true
        }
      },
      additionalProperties: false
    },
    sampleData: {
      type: 'object',
      properties: {
        path: {
          type: 'string'
        }
      },
      required: ['path'],
      additionalProperties: false,
      nullable: true
    }
  },
  required: ['name', 'version', 'license', 'type', 'components', 'assets'],
  additionalProperties: false
};
const instanceSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    templateName: {
      type: 'string'
    },
    dataSource: {
      type: 'string'
    },
    creationDate: {
      type: 'string'
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          assetType: {
            type: 'string'
          },
          assetId: {
            type: 'string'
          },
          isDefaultAsset: {
            type: 'boolean'
          },
          description: {
            type: 'string'
          }
        },
        required: ['assetType', 'assetId', 'isDefaultAsset', 'description']
      }
    }
  },
  required: ['name', 'templateName', 'dataSource', 'creationDate', 'assets']
};
const templateValidator = ajv.compile(templateSchema);
const instanceValidator = ajv.compile(instanceSchema);
/**
 * Validates an integration template against a predefined schema using the AJV library.
 * Since AJV validators use side effects for errors,
 * this is a more conventional wrapper that simplifies calling.
 *
 * @param data The data to be validated as an IntegrationTemplate.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */

const validateTemplate = data => {
  if (!templateValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(templateValidator.errors))
    };
  } // We assume an invariant that the type of an integration is connected with its component.


  if (data.components.findIndex(x => x.name === data.type) < 0) {
    return {
      ok: false,
      error: new Error(`The integration type '${data.type}' must be included as a component`)
    };
  }

  return {
    ok: true,
    value: data
  };
};
/**
 * Validates an integration instance against a predefined schema using the AJV library.
 *
 * @param data The data to be validated as an IntegrationInstance.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */


exports.validateTemplate = validateTemplate;

const validateInstance = data => {
  if (!instanceValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(instanceValidator.errors))
    };
  }

  return {
    ok: true,
    value: data
  };
};

exports.validateInstance = validateInstance;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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