"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomPanelsAdaptor = void 0;

var _uuid = require("uuid");

var _custom_panels = require("../../../common/constants/custom_panels");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class CustomPanelsAdaptor {
  constructor() {
    _defineProperty(this, "indexPanel", async function (client, panelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            operationalPanel: panelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Panel Error:' + error);
      }
    });

    _defineProperty(this, "updatePanel", async function (client, panelId, updatePanelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: panelId,
          body: {
            operationalPanel: updatePanelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Panel Error:' + error);
      }
    });

    _defineProperty(this, "getPanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Panel Error:' + error);
      }
    });

    _defineProperty(this, "viewPanelList", async function (client) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'operationalPanel',
          maxItems: 10000
        });
        return response.observabilityObjectList.filter(panel => !panel.operationalPanel.applicationId).map(panel => ({
          name: panel.operationalPanel.name,
          id: panel.objectId,
          dateCreated: panel.createdTimeMs,
          dateModified: panel.lastUpdatedTimeMs
        }));
      } catch (error) {
        throw new Error('View Panel List Error:' + error);
      }
    });

    _defineProperty(this, "deletePanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectById', {
          objectId: panelId
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel Error:' + error);
      }
    });

    _defineProperty(this, "deletePanelList", async function (client, panelIdList) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: panelIdList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel List Error:' + error);
      }
    });

    _defineProperty(this, "createNewPanel", async (client, panelName, appId) => {
      const panelBody = {
        name: panelName,
        visualizations: [],
        timeRange: {
          to: 'now',
          from: 'now-1d'
        },
        queryFilter: {
          query: '',
          language: 'ppl'
        }
      };

      if (appId) {
        panelBody.applicationId = appId;
        panelBody.timeRange = {
          to: 'now',
          from: 'now-24h'
        };
      }

      try {
        const response = await this.indexPanel(client, panelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });

    _defineProperty(this, "renamePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Rename Panel Error:' + error);
      }
    });

    _defineProperty(this, "clonePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const getPanel = await this.getPanel(client, panelId);
        const clonePanelBody = {
          name: panelName,
          visualizations: getPanel.operationalPanel.visualizations,
          timeRange: getPanel.operationalPanel.timeRange,
          queryFilter: getPanel.operationalPanel.queryFilter
        };
        const indexResponse = await this.indexPanel(client, clonePanelBody);
        const getClonedPanel = await this.getPanel(client, indexResponse.objectId);
        return {
          clonePanelId: getClonedPanel.objectId,
          dateCreated: getClonedPanel.createdTimeMs,
          dateModified: getClonedPanel.lastUpdatedTimeMs
        };
      } catch (error) {
        throw new Error('Clone Panel Error:' + error);
      }
    });

    _defineProperty(this, "addPanelFilter", async (client, panelId, query, language, to, from) => {
      const updatePanelBody = {
        timeRange: {
          to,
          from
        },
        queryFilter: {
          query,
          language
        }
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Add Panel Filter Error:' + error);
      }
    });

    _defineProperty(this, "parseSavedVisualizations", visualization => {
      return {
        id: visualization.objectId,
        name: visualization.savedVisualization.name,
        query: visualization.savedVisualization.query,
        type: visualization.savedVisualization.type,
        timeField: visualization.savedVisualization.selected_timestamp.name,
        selected_date_range: visualization.savedVisualization.selected_date_range,
        selected_fields: visualization.savedVisualization.selected_fields,
        user_configs: visualization.savedVisualization.hasOwnProperty('user_configs') ? JSON.parse(visualization.savedVisualization.user_configs) : {},
        sub_type: visualization.savedVisualization.hasOwnProperty('sub_type') ? visualization.savedVisualization.sub_type : '',
        units_of_measure: visualization.savedVisualization.hasOwnProperty('units_of_measure') ? visualization.savedVisualization.units_of_measure : '',
        ...(visualization.savedVisualization.application_id ? {
          application_id: visualization.savedVisualization.application_id
        } : {})
      };
    });

    _defineProperty(this, "getAllSavedVisualizations", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'savedVisualization'
        });
        return response.observabilityObjectList.map(visualization => this.parseSavedVisualizations(visualization));
      } catch (error) {
        throw new Error('View Saved Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "getSavedVisualizationById", async (client, savedVisualizationId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: savedVisualizationId
        });
        const visualization = response.observabilityObjectList[0];
        return this.parseSavedVisualizations(visualization);
      } catch (error) {
        throw new Error('Fetch Saved Visualizations By Id Error:' + error);
      }
    });

    _defineProperty(this, "getVisualizations", async (client, panelId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0].operationalPanel.visualizations;
      } catch (error) {
        throw new Error('Get Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "calculatOverlapArea", (bb1, bb2) => {
      const xLeft = Math.max(bb1.x1, bb2.x1);
      const yTop = Math.max(bb1.y1, bb2.y1);
      const xRight = Math.min(bb1.x2, bb2.x2);
      const yBottom = Math.min(bb1.y2, bb2.y2);
      if (xRight < xLeft || yBottom < yTop) return 0;
      return (xRight - xLeft) * (yBottom - yTop);
    });

    _defineProperty(this, "getTotalOverlapArea", panelVisualizations => {
      const newVizBox = {
        x1: 0,
        y1: 0,
        x2: 6,
        y2: 4
      };
      const currentVizBoxes = panelVisualizations.map(visualization => {
        return {
          x1: visualization.x,
          y1: visualization.y,
          x2: visualization.x + visualization.w,
          y2: visualization.y + visualization.h
        };
      });
      let isOverlapping = 0;
      currentVizBoxes.map(viz => {
        isOverlapping += this.calculatOverlapArea(viz, newVizBox);
      });
      return isOverlapping;
    });

    _defineProperty(this, "getNewVizDimensions", panelVisualizations => {
      let maxY = 0;
      let maxYH = 0; // check if we can place the new visualization at default location

      if (this.getTotalOverlapArea(panelVisualizations) === 0) {
        return {
          x: 0,
          y: 0,
          w: 6,
          h: 4
        };
      } // else place the new visualization at the bottom of the panel


      panelVisualizations.map(panelVisualization => {
        if (panelVisualization.y >= maxY) {
          maxY = panelVisualization.y;
          maxYH = panelVisualization.h;
        }
      });
      return {
        x: 0,
        y: maxY + maxYH,
        w: 6,
        h: 4
      };
    });

    _defineProperty(this, "addVisualization", async (client, panelId, savedVisualizationId, oldVisualizationId) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [];

        if (oldVisualizationId === undefined) {
          newDimensions = this.getNewVizDimensions(allPanelVisualizations);
          visualizationsList = allPanelVisualizations;
        } else {
          allPanelVisualizations.map(visualization => {
            if (visualization.id !== oldVisualizationId) {
              visualizationsList.push(visualization);
            } else {
              newDimensions = {
                x: visualization.x,
                y: visualization.y,
                w: visualization.w,
                h: visualization.h
              };
            }
          });
        }

        const newPanelVisualizations = [...visualizationsList, {
          id: 'panel_viz_' + (0, _uuid.v4)(),
          savedVisualizationId,
          ...newDimensions
        }];
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: newPanelVisualizations
        });
        return newPanelVisualizations;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });

    _defineProperty(this, "addMultipleVisualizations", async (client, panelId, savedVisualizationIds) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [...allPanelVisualizations];
        savedVisualizationIds.map(savedVisualizationId => {
          newDimensions = this.getNewVizDimensions(visualizationsList);
          visualizationsList = [...visualizationsList, {
            id: 'panel_viz_' + (0, _uuid.v4)(),
            savedVisualizationId,
            ...newDimensions
          }];
        });
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: visualizationsList
        });
        return visualizationsList;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });

    _defineProperty(this, "editVisualization", async (client, panelId, visualizationParams) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        const filteredPanelVisualizations = [];

        for (let i = 0; i < allPanelVisualizations.length; i++) {
          for (let j = 0; j < visualizationParams.length; j++) {
            if (allPanelVisualizations[i].id === visualizationParams[j].i) {
              filteredPanelVisualizations.push({ ...allPanelVisualizations[i],
                x: visualizationParams[j].x,
                y: visualizationParams[j].y,
                w: visualizationParams[j].w,
                h: visualizationParams[j].h
              });
            }
          }
        }

        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: filteredPanelVisualizations
        });
        return filteredPanelVisualizations;
      } catch (error) {
        throw new Error('Edit Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "addSamplePanels", async (client, savedVisualizationIds) => {
      try {
        const panelBody = (0, _custom_panels.createDemoPanel)(savedVisualizationIds);
        const indexResponse = await this.indexPanel(client, panelBody);
        const fetchPanel = await this.getPanel(client, indexResponse.objectId);
        const fetchResponse = {
          name: fetchPanel.operationalPanel.name,
          id: fetchPanel.objectId,
          dateCreated: fetchPanel.createdTimeMs,
          dateModified: fetchPanel.lastUpdatedTimeMs
        };
        return [fetchResponse];
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
  } // index a panel
  // update a panel
  // fetch a panel by id
  // gets list of panels stored in index
  // Delete a panel by Id
  // Delete a panel by Id
  // Create a new Panel
  // Rename an existing panel
  // Clone an existing panel
  // Add filters to an existing panel
  // parses fetched saved visualization
  // gets all saved visualizations
  // gets list of savedVisualizations by Id
  // Get All Visualizations from a Panel
  // We want to check if the new visualization being added, can be placed at { x: 0, y: 0, w: 6, h: 4 };
  // To check this we try to calculate overlap between all the current visualizations and new visualization
  // if there is no overalap (i.e Total Overlap Area is 0), we place the new viz. in default position
  // else, we add it to the bottom of the panel
  // Add Visualization in the  Panel
  // Add Multiple visualizations in a Panel
  // Edits all Visualizations in the Panel
  // Create Sample Panels


}

exports.CustomPanelsAdaptor = CustomPanelsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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