"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _DataStreamService = require("./DataStreamService");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class ManagedIndexService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getManagedIndex", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          id,
          index: _constants.INDEX.OPENDISTRO_ISM_CONFIG
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const results = await callWithRequest("search", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - getManagedIndex:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getManagedIndices", async (context, request, response) => {
      try {
        const {
          from,
          size,
          sortDirection,
          sortField,
          terms,
          indices,
          dataStreams,
          showDataStreams
        } = request.query;
        const managedIndexSorts = {
          index: "managed_index.index",
          policyId: "managed_index.policy_id"
        };
        const explainParams = {
          sortField: sortField ? managedIndexSorts[sortField] : null,
          sortOrder: sortDirection,
          queryString: (0, _helpers.getSearchString)(terms, indices, dataStreams, showDataStreams),
          from: from,
          size: size
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const [explainAllResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("ism.explainAll", explainParams), (0, _DataStreamService.getIndexToDataStreamMapping)({
          callAsCurrentUser: callWithRequest
        })]);
        const managedIndices = [];

        for (const indexName in explainAllResponse) {
          if (indexName == "total_managed_indices") continue;
          const metadata = explainAllResponse[indexName]; // If showDataStreams is not true, then skip the managed index if it belongs to a data stream.

          const parentDataStream = indexToDataStreamMapping[metadata.index] || null;
          if (!showDataStreams && parentDataStream !== null) continue;
          let policy, seqNo, primaryTerm, getResponse;

          try {
            getResponse = await callWithRequest("ism.getPolicy", {
              policyId: metadata.policy_id
            });
          } catch (err) {
            if (err.statusCode === 404 && err.body.error.reason === "Policy not found") {
              console.log("managed index with not existing policy");
            } else {
              throw err;
            }
          }

          policy = _lodash.default.get(getResponse, "policy", null);
          seqNo = _lodash.default.get(getResponse, "_seq_no", null);
          primaryTerm = _lodash.default.get(getResponse, "_primary_term", null);
          managedIndices.push({
            index: metadata.index,
            indexUuid: metadata.index_uuid,
            dataStream: parentDataStream,
            policyId: metadata.policy_id,
            policySeqNo: seqNo,
            policyPrimaryTerm: primaryTerm,
            policy: policy,
            enabled: metadata.enabled,
            managedIndexMetaData: (0, _helpers.transformManagedIndexMetaData)(metadata)
          });
        }

        let totalManagedIndices = explainAllResponse.total_managed_indices;
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              managedIndices: managedIndices,
              totalManagedIndices: totalManagedIndices
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                managedIndices: [],
                totalManagedIndices: 0
              }
            }
          });
        }

        console.error("Index Management - ManagedIndexService - getManagedIndices", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "retryManagedIndexPolicy", async (context, request, response) => {
      try {
        const {
          index,
          state = null
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: index.join(",")
        };
        if (state) params.body = {
          state
        };
        const retryResponse = await callWithRequest("ism.retry", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: retryResponse.failures,
              updatedIndices: retryResponse.updated_indices,
              // TODO: remove ternary after fixing retry API to return empty array even if no failures
              failedIndices: retryResponse.failed_indices ? retryResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              })) : []
            }
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - retryManagedIndexPolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "changePolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId,
          include,
          state
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId,
            include,
            state
          }
        };
        const changeResponse = await callWithRequest("ism.change", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: changeResponse.failures,
              updatedIndices: changeResponse.updated_indices,
              failedIndices: changeResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - changePolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "removePolicy", async (context, request, response) => {
      try {
        const {
          indices
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(",")
        };
        const addResponse = await callWithRequest("ism.remove", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - removePolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  } // TODO: Not finished, need UI page that uses this first


}

exports.default = ManagedIndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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