"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = require("lodash");

var _helpers = require("./utils/helpers");

var _adHelpers = require("./utils/adHelpers");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

const MAX_DETECTOR_COUNT = 1000;

class DestinationsService {
  constructor(esDriver) {
    _defineProperty(this, "getDetector", async (context, req, res) => {
      const {
        detectorId
      } = req.params;
      const {
        callAsCurrentUser
      } = this.esDriver.asScoped(req);

      try {
        const resp = await callAsCurrentUser('alertingAD.getDetector', {
          detectorId
        });
        const {
          anomaly_detector,
          _seq_no: seqNo,
          _primary_term: primaryTerm,
          _version: version
        } = resp;
        return res.ok({
          body: {
            ok: true,
            detector: anomaly_detector,
            version,
            seqNo,
            primaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - AnomalyDetectorService - getDetector:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getDetectors", async (context, req, res) => {
      const searchRequest = {
        query: {
          bool: {}
        },
        size: MAX_DETECTOR_COUNT
      };
      const {
        callAsCurrentUser
      } = this.esDriver.asScoped(req);

      try {
        const resp = await callAsCurrentUser('alertingAD.searchDetectors', {
          body: searchRequest
        });
        const totalDetectors = resp.hits.total.value;
        const detectors = resp.hits.hits.map(hit => {
          const {
            _source: detector,
            _id: id,
            _version: version,
            _seq_no: seqNo,
            _primary_term: primaryTerm
          } = hit;
          return {
            id,
            ...detector,
            version,
            seqNo,
            primaryTerm
          };
        });
        return res.ok({
          body: {
            ok: true,
            detectors: (0, _helpers.mapKeysDeep)(detectors, _helpers.toCamel),
            totalDetectors
          }
        });
      } catch (err) {
        console.error('Alerting - AnomalyDetectorService - searchDetectors:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getDetectorResults", async (context, req, res) => {
      try {
        const {
          startTime = 0,
          endTime = 20,
          preview = 'false'
        } = req.query;
        const {
          detectorId
        } = req.params;
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);

        if (preview == 'true') {
          const requestBody = {
            period_start: startTime,
            period_end: endTime
          };
          const previewResponse = await callAsCurrentUser('alertingAD.previewDetector', {
            detectorId,
            body: requestBody
          });
          const transformedKeys = (0, _helpers.mapKeysDeep)(previewResponse, _helpers.toCamel);
          return res.ok({
            body: {
              ok: true,
              response: {
                anomalyResult: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult),
                detector: transformedKeys.anomalyDetector
              }
            }
          });
        } else {
          //Get results
          const requestBody = {
            size: 10000,
            sort: {
              data_start_time: 'asc'
            },
            query: {
              bool: {
                filter: [{
                  term: {
                    detector_id: detectorId
                  }
                }, {
                  range: {
                    data_start_time: {
                      gte: startTime,
                      lte: endTime
                    }
                  }
                }]
              }
            }
          };
          const detectorResponse = await callAsCurrentUser('alertingAD.getDetector', {
            detectorId
          });
          const anomaliesResponse = await callAsCurrentUser('alertingAD.searchResults', {
            body: requestBody
          });
          const transformedKeys = (0, _lodash.get)(anomaliesResponse, 'hits.hits', []).map(result => (0, _helpers.mapKeysDeep)(result._source, _helpers.toCamel));
          return res.ok({
            body: {
              ok: true,
              response: {
                detector: (0, _helpers.mapKeysDeep)((0, _lodash.get)(detectorResponse, 'anomaly_detector', {}), _helpers.toCamel),
                anomalyResult: (0, _adHelpers.anomalyResultMapper)(transformedKeys)
              }
            }
          });
        }
      } catch (err) {
        console.log('Alerting - AnomalyDetectorService - getDetectorResults', err);
        return res.ok({
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.esDriver = esDriver;
  }

}

exports.default = DestinationsService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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