/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* hash.h                                                                       *
* A hashtable implemented as a macro, usable in any C struct via including the *
* UT_hash_handle as a member of the structure, and using the HASH_ADD/FIND/DEL *
* macros. The hash supports constant-time add/delete/find. The hash is also a  *
* linked-list whose elements are ordered in the sequence they were appended.   *
* The number of buckets in the hash is dynamically expanded as necessary to    *
* attempt to keep all buckets from having more than 10 elements.               *
*******************************************************************************/
#include <stdlib.h>
#include <string.h> /* memcmp,strlen */
#include "libut/ut.h"

#ifndef UTHSH_H
#define UTHSH_H 

/* A version of this could be made that is usable outside of UT, 
 * provided that the UT_hash_rescale is "undef'd" and the bucket 
 size is fixed to some value appropriate to the application. */

#define UT_HASH_INITIAL_NUM_BUCKETS 32  /* start all hashes w/32 buckets */
#define UT_HASH_RESCALE_THRESHOLD   10  /* double # buckets if a bkt exceeds */

/* memory pool names used by this module */
#define UTHASHPOOL "ut_hash_tbl"

/* The hash function is the Perl pre-v5.6 hash, known as Bernstein hash. 
 * This hash is remarkably good at distributing the English dictionary and other
 * test inputs pretty equally among the buckets. The # of buckets must be 
 * a power of two. */
#define UT_HASH(key,keylen,num_bkts,bkt)                  \
          while (keylen--)  bkt = (bkt * 33) + *key++;    \
          bkt &= (num_bkts-1);          

/* key comparison function; return 0 if keys equal */
#define HASH_KEYCMP(a,b,len) memcmp(a,b,len) 

#define HASH_FIND_IN_BKT(head,tmp,field,keyptr,keylen_in) \
    tmp=head.link_head;                     \
    while (tmp) {                           \
        if (tmp->hh.keylen == keylen_in) {     \
            if ((HASH_KEYCMP(&tmp->field,keyptr,keylen_in)) == 0) break; \
        }                                   \
        tmp=tmp->hh.bkt_next;               \
    }


#define HASH_LINEAR_FIND(head,tmp,field,keyptr,keylen_in) \
    tmp=head;            \
    while (tmp) {        \
        if (tmp->hh.keylen == keylen_in) {     \
            if ((HASH_KEYCMP(&tmp->field,keyptr,keylen_in)) == 0) break; \
        }                \
        tmp=tmp->next;   \
    }

#define HASH_FIND(head,tmp,field,keyptr,keylen_in)                   \
  tmp=head;                                                          \
  if (head) {                                                        \
   if (head->hh.tbl) {                                               \
     head->hh.tbl->key = (char*)keyptr;                              \
     head->hh.tbl->keylen = keylen_in;                               \
     head->hh.tbl->bkt = 0;                                          \
     UT_HASH(head->hh.tbl->key,head->hh.tbl->keylen,head->hh.tbl->num_buckets,head->hh.tbl->bkt); \
     HASH_FIND_IN_BKT( head->hh.tbl->buckets[ head->hh.tbl->bkt],tmp,field,keyptr,keylen_in); \
   } else {                                                          \
     if (0) UT_LOG(Debugk,"Hash pending; using linear scan.");       \
     HASH_LINEAR_FIND(head,tmp,field,keyptr,keylen_in);              \
   }                                                                 \
  }


#define HASH_ADD_TO_BKT(head,tmp,add)                     \
 head.count++;                                            \
 tmp = head.link_head;                                    \
 if (tmp) {                                               \
    tmp->hh.tbl->bkt_cnt=0;                               \
    while (tmp->hh.bkt_next) { tmp=tmp->hh.bkt_next;      \
                                tmp->hh.tbl->bkt_cnt++; } \
    tmp->hh.bkt_next=add;                                 \
    tmp->hh.hh_next=&add->hh;                             \
    if (tmp->hh.tbl->bkt_cnt >= UT_HASH_RESCALE_THRESHOLD && tmp->hh.tbl->inhibit_expansion != 1) { \
       UT_hash_rescale(tmp->hh.tbl);  }                   \
 } else {                                                 \
    head.link_head=add;                                   \
    head.hh_head=&add->hh;                                \
 }

#define HASH_ADD(head,tmp,fieldname,add,fieldlen)         \
 add->hh.bkt_next = NULL;                                 \
 add->hh.hh_next  = NULL;                                 \
 add->hh.link_next = (void**)&add->next;                  \
 add->hh.key = &add->fieldname;                           \
 add->hh.keylen = fieldlen;                               \
 add->hh.link = add;                                      \
 if (!head) {                                             \
    head = add;                                           \
    head->hh.hh_prev = NULL;                              \
    head->hh.tbl = NULL;                                  \
    head->hh.tbl = (UT_hash_table*)UT_mem_alloc(UTHASHPOOL,1);        \
    head->hh.tbl->tail = add;                             \
    head->hh.tbl->inhibit_expansion = 0;                  \
    head->hh.tbl->num_buckets = UT_HASH_INITIAL_NUM_BUCKETS;          \
    head->hh.tbl->buckets    = (UT_hash_bucket*)malloc(UT_HASH_INITIAL_NUM_BUCKETS*sizeof(struct UT_hash_bucket)); \
    if (! head->hh.tbl->buckets) { UT_LOG( Fatal, "malloc failure"); }  \
    memset(head->hh.tbl->buckets,  0, UT_HASH_INITIAL_NUM_BUCKETS*sizeof(struct UT_hash_bucket)); \
 } else {                                                 \
    LL_ADD(head->hh.tbl->tail,tmp,add);                   \
    tmp = head->hh.tbl->tail;                             \
    add->hh.hh_prev = &tmp->hh;                           \
    head->hh.tbl->tail = add;                             \
 }                                                        \
 if (head->hh.tbl) {                                      \
   add->hh.tbl = head->hh.tbl;                            \
   head->hh.tbl->key = (char*)&add->fieldname;            \
   head->hh.tbl->keylen = fieldlen;                       \
   head->hh.tbl->bkt = 0;                                 \
   UT_HASH(head->hh.tbl->key,head->hh.tbl->keylen,head->hh.tbl->num_buckets,head->hh.tbl->bkt); \
   HASH_ADD_TO_BKT( head->hh.tbl->buckets[ head->hh.tbl->bkt ], tmp, add );   \
 } else {                                                 \
   UT_LOG(Fatal,"Hashtable null in HASH_ADD ");           \
 }                                                                      


#define HASH_DEL_IN_BKT(head,tmp,delptr) tmp = head.link_head; \
 if (tmp == delptr) {                                        \
        head.link_head=tmp->hh.bkt_next;                     \
        head.hh_head=tmp->hh.hh_next;                        \
        head.count--;                                        \
} else {                                                     \
        while (tmp->hh.bkt_next && tmp->hh.bkt_next != delptr) \
             tmp=tmp->hh.bkt_next;                          \
        if (tmp->hh.bkt_next) {                             \
           tmp->hh.bkt_next=(delptr)->hh.bkt_next;          \
           tmp->hh.hh_next  =(delptr)->hh.hh_next;          \
           head.count--;                                    \
        } else {                                            \
            UT_LOG(Error,"delete in bucket failed to find item"); \
        } \
}

#define HASH_DEL(head,tmp,delptr) \
    if (delptr) { \
        if (delptr->hh.hh_prev) { \
            if (delptr == head->hh.tbl->tail) head->hh.tbl->tail =  delptr->hh.hh_prev->link; \
            *delptr->hh.hh_prev->link_next = delptr->next; \
            if (delptr->next) delptr->next->hh.hh_prev = delptr->hh.hh_prev; \
        } else { \
            if (!delptr->next) { \
                free(head->hh.tbl->buckets ); \
                UT_mem_free(UTHASHPOOL,head->hh.tbl, 1); \
            } \
            head = delptr->next; \
            if (head) head->hh.hh_prev = NULL; \
        } \
        if (head) { \
            head->hh.tbl->key = (char*)delptr->hh.key;                                              \
            head->hh.tbl->keylen = delptr->hh.keylen;                                              \
            head->hh.tbl->bkt = 0;                                                     \
            UT_HASH(head->hh.tbl->key,head->hh.tbl->keylen,head->hh.tbl->num_buckets,head->hh.tbl->bkt); \
            HASH_DEL_IN_BKT(head->hh.tbl->buckets[ head->hh.tbl->bkt ], tmp, delptr); \
        } \
    } else {    \
        UT_LOG(Debug,"attempt to delete null hash item"); \
    }

/* These four macros and HASH_DEL are the "public" interface to the UT hash. */
/* These find/add a stucture whose key field is a string or int (pass by ptr), respectively */
#define HASH_FIND_STR(head,tmp,field,findstr) HASH_FIND(head,tmp,field,findstr,strlen(findstr))
#define HASH_FIND_INT(head,tmp,field,findint) HASH_FIND(head,tmp,field,findint,sizeof(int))
#define HASH_ADD_STR(head,tmp,strfield,add) HASH_ADD(head,tmp,strfield,add,strlen(add->strfield))
#define HASH_ADD_INT(head,tmp,intfield,add) HASH_ADD(head,tmp,intfield,add,sizeof(int))

typedef struct UT_hash_bucket {
   void *link_head;
   struct UT_hash_handle *hh_head;
   unsigned count;  /* for statistics */
} UT_hash_bucket;

typedef struct UT_hash_table {
   UT_hash_bucket *buckets;
   unsigned num_buckets;
   int inhibit_expansion;
   void *tail;   /* point to tail link in the list for fast append */
   /* scratch */
   unsigned bkt;
   char *key;
   int keylen;
   unsigned bkt_cnt;
} UT_hash_table;


typedef struct UT_hash_handle {
   struct UT_hash_table *tbl;
   void *link;  /* ptr to enclosing struct, used when rebucketing */
   void **link_next; /* ptr to enclosing struct's 'next' pointer, used in delete */
   struct UT_hash_handle *hh_prev;   /* for delete, previous hh in app order */
   struct UT_hash_handle *hh_next;   /* for find, next hh in bucket order */
   void *bkt_next;                   /* for find, next link in bucket order */
   void *key;   /* ptr to enclosing struct's key, used when rebucketing */
   int keylen;  /* enclosing struct's key len, used when rebucketing */
} UT_hash_handle;

int UT_hash_rescale(UT_hash_table *);

#endif /* UTHSH_H */

