! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!==============================================================================
!
#define ESMF_FILENAME "ESMF_StateAPI.cppF90"
!
! ESMF StateAPI module
module ESMF_StateAPIMod
!
!==============================================================================
!
! This file contains the State user API methods.
!
!------------------------------------------------------------------------------
! INCLUDES
!------------------------------------------------------------------------------
#include "ESMF.h"

!------------------------------------------------------------------------------
!BOPI
! !MODULE: ESMF_StateMod - Data exchange between components
!
! !DESCRIPTION:
!
! The code in this file implements the Fortran function and subroutine
! interfaces to the {\tt State} class and associated data structures.
!
!
! !USES:
      use ESMF_UtilTypesMod
      use ESMF_LogErrMod
      use ESMF_BaseMod
      use ESMF_ContainerMod
      use ESMF_VMMod
      use ESMF_ArrayMod
      use ESMF_ArrayGetMod
      use ESMF_ArrayBundleMod
      use ESMF_FieldMod
      use ESMF_FieldGetMod
      use ESMF_FieldCreateMod
      use ESMF_FieldHaloMod
      use ESMF_FieldBundleMod
      use ESMF_RHandleMod
      use ESMF_StateTypesMod
      use ESMF_StateContainerMod
      use ESMF_StateSetMod
      use ESMF_StateVaMod
      use ESMF_StateInternalsMod
      use ESMF_StateWrMod
      use ESMF_InitMacrosMod
      use ESMF_IO_NetCDFMod
      use ESMF_IOUtilMod
      use ESMF_UtilMod
      use ESMF_UtilStringMod
      use ESMF_InfoMod

      implicit none

!------------------------------------------------------------------------------
! !PRIVATE TYPES:
      private

!------------------------------------------------------------------------------
! !PUBLIC TYPES:
      public ESMF_State ! implemented in ESMF_StateTypesMod

!------------------------------------------------------------------------------

! !PUBLIC MEMBER FUNCTIONS:

      public operator(==)
      public operator(/=)

      public ESMF_StateCreate, ESMF_StateDestroy

      public ESMF_StateDestruct ! for ESMF garbage collection

      public ESMF_StateAdd, ESMF_StateAddReplace
      public ESMF_StateGet
      public ESMF_StateIsCreated

      public ESMF_StateLog
      public ESMF_StatePrint
      public ESMF_StateSet

      public ESMF_StateSerialize, ESMF_StateDeserialize

      public ESMF_StateClassFindData

      ! These methods are broken and create Arrays in a bad configuration.
      ! We are taking them out so no one uses them. Eventually, new correct
      ! versions will be implemented.
      ! public ESMF_StateRead
      ! public ESMF_StateWrite
      ! public ESMF_StateWriteRestart
      ! public ESMF_StateReadRestart


!EOPI

!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
      character(*), parameter, private :: version = &
      '$Id$'

!==============================================================================
!
! INTERFACE BLOCKS
!
!==============================================================================

!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_StateAdd -- Add items to a State

! !INTERFACE:
  interface ESMF_StateAdd

! !PRIVATE MEMBER FUNCTIONS:
!
! module procedure ESMF_StateAddOneArray
    module procedure ESMF_StateAddOneArrayX
    module procedure ESMF_StateAddArrayList

! module procedure ESMF_StateAddOneArrayBundle
    module procedure ESMF_StateAddOneArrayBundleX
    module procedure ESMF_StateAddArrayBundleList

! module procedure ESMF_StateAddOneField
    module procedure ESMF_StateAddOneFieldX
    module procedure ESMF_StateAddFieldList

! module procedure ESMF_StateAddOneFieldBundle
    module procedure ESMF_StateAddOneFieldBundleX
    module procedure ESMF_StateAddFieldBundleList

! module procedure ESMF_StateAddOneRouteHandle
    module procedure ESMF_StateAddRouteHandleList

! module procedure ESMF_StateAddOneState
    module procedure ESMF_StateAddOneStateX
    module procedure ESMF_StateAddStateList


! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_StateAdd} functions.
!
!EOPI
  end interface

!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_StateAddRplace -- Add/replace items to a State

! !INTERFACE:
  interface ESMF_StateAddReplace

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_StateAddRpArrayList
    module procedure ESMF_StateAddRpArrayBundleList
    module procedure ESMF_StateAddRpFieldList
    module procedure ESMF_StateAddRpFieldBundleList
    module procedure ESMF_StateAddRpRoutehandleList
    module procedure ESMF_StateAddRpStateList


! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_StateAdd} functions.
!
!EOPI
  end interface

!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_StateGet -- Get items from a State

! !INTERFACE:
  interface ESMF_StateGet

! !PRIVATE MEMBER FUNCTIONS:
!
      module procedure ESMF_StateGetArray
      module procedure ESMF_StateGetArrayBundle
      module procedure ESMF_StateGetField
      module procedure ESMF_StateGetFieldBundle
      module procedure ESMF_StateGetRouteHandle
      module procedure ESMF_StateGetState
      module procedure ESMF_StateGetInfo
      module procedure ESMF_StateGetItemInfo


! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_StateGet} functions.
!
!EOPI
  end interface

!===============================================================================
! StateOperator() interfaces
!===============================================================================

! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_StateAssignment(=) - State assignment
!
! !INTERFACE:
! interface assignment(=)
! state1 = state2
!
! !ARGUMENTS:
! type(ESMF_State) :: state1
! type(ESMF_State) :: state2
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Assign state1 as an alias to the same ESMF State object in memory
! as state2. If state2 is invalid, then state1 will be equally invalid after
! the assignment.
!
! The arguments are:
! \begin{description}
! \item[state1]
! The {\tt ESMF\_State} object on the left hand side of the assignment.
! \item[state2]
! The {\tt ESMF\_State} object on the right hand side of the assignment.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_StateOperator(==) - State equality operator
!
! !INTERFACE:
  interface operator(==)
! if (state1 == state2) then ... endif
! OR
! result = (state1 == state2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_State), intent(in) :: state1
! type(ESMF_State), intent(in) :: state2
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Test whether state1 and state2 are valid aliases to the same ESMF
! State object in memory. For a more general comparison of two ESMF States,
! going beyond the simple alias test, the ESMF\_StateMatch() function (not yet
! implemented) must be used.
!
! The arguments are:
! \begin{description}
! \item[state1]
! The {\tt ESMF\_State} object on the left hand side of the equality
! operation.
! \item[state2]
! The {\tt ESMF\_State} object on the right hand side of the equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_StateEQ

  end interface
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -------------------------------
!BOP
! !IROUTINE: ESMF_StateOperator(/=) - State not equal operator
!
! !INTERFACE:
  interface operator(/=)
! if (state1 /= state2) then ... endif
! OR
! result = (state1 /= state2)
! !RETURN VALUE:
! logical :: result
!
! !ARGUMENTS:
! type(ESMF_State), intent(in) :: state1
! type(ESMF_State), intent(in) :: state2
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Test whether state1 and state2 are {\it not} valid aliases to the
! same ESMF State object in memory. For a more general comparison of two ESMF
! States, going beyond the simple alias test, the ESMF\_StateMatch() function
! (not yet implemented) must be used.
!
! The arguments are:
! \begin{description}
! \item[state1]
! The {\tt ESMF\_State} object on the left hand side of the non-equality
! operation.
! \item[state2]
! The {\tt ESMF\_State} object on the right hand side of the non-equality
! operation.
! \end{description}
!
!EOP
    module procedure ESMF_StateNE

  end interface
!------------------------------------------------------------------------------


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

contains

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateEQ()"
!BOPI
! !IROUTINE: ESMF_StateEQ - Compare two States for equality
!
! !INTERFACE:
  impure elemental function ESMF_StateEQ(state1, state2)
!
! !RETURN VALUE:
    logical :: ESMF_StateEQ

! !ARGUMENTS:
    type(ESMF_State), intent(in) :: state1
    type(ESMF_State), intent(in) :: state2

! !DESCRIPTION:
! Test if both {\tt state1} and {\tt state2} alias the same ESMF State
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ESMF_INIT_TYPE sinit1, sinit2

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ! TODO: Consider moving this logic to C++: use Base class? status?
    ! Or replicate logic for C interface also.

    ! check inputs
    sinit1 = ESMF_StateGetInit(state1)
    sinit2 = ESMF_StateGetInit(state2)

    ! TODO: this line must remain split in two for SunOS f90 8.3 127000-03
    if (sinit1 .eq. ESMF_INIT_CREATED .and. &
      sinit2 .eq. ESMF_INIT_CREATED) then
      ESMF_StateEQ = associated(state1%statep,state2%statep)
    else
      ESMF_StateEQ = ESMF_FALSE
    endif

  end function ESMF_StateEQ
!-------------------------------------------------------------------------------


!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateNE()"
!BOPI
! !IROUTINE: ESMF_StateNE - Compare two States for non-equality
!
! !INTERFACE:
  impure elemental function ESMF_StateNE(state1, state2)
!
! !RETURN VALUE:
    logical :: ESMF_StateNE

! !ARGUMENTS:
    type(ESMF_State), intent(in) :: state1
    type(ESMF_State), intent(in) :: state2

! !DESCRIPTION:
! Test if both {\tt state1} and {\tt state2} alias the same ESMF State
! object.
!
!EOPI
!-------------------------------------------------------------------------------

    ! Use the following logic, rather than "ESMF-INIT-CHECK-DEEP", to gain
    ! init checks on both args, and in the case where both are uninitialized,
    ! to distinguish equality based on uninitialized type (uncreated,
    ! deleted).

    ESMF_StateNE = .not.ESMF_StateEQ(state1, state2)

  end function ESMF_StateNE
!-------------------------------------------------------------------------------



!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_StateAdd - Add a single item to a State
!
! !INTERFACE:
! subroutine ESMF_StateAdd(state, <item>, keywordEnforcer, relaxedflag, rc)
!
! !ARGUMENTS:
! type(ESMF_State), intent(inout) :: state
! <item>, see below for supported values
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! logical, intent(in), optional :: relaxedflag
! integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Add a reference to a single <item> to an existing
! {\tt state}. The name of the <item> must be unique within
! the {\tt state}.
!
! Supported values for <item> are:
! \begin{description}
! \item type(ESMF\_Array), intent(in) :: array
! \item type(ESMF\_ArrayBundle), intent(in) :: arraybundle
! \item type(ESMF\_Field), intent(in) :: field
! \item type(ESMF\_FieldBundle), intent(in) :: fieldbundle
! \item type(ESMF\_RouteHandle), intent(in) :: routehandle
! \item type(ESMF\_State), intent(in) :: nestedState
! \end{description}
!
! The arguments are:
! \begin{description}
! \item[state]
! The {\tt ESMF\_State} to which <item>s will be added.
! \item[<item>]
! The <item> to be added. This is a reference only; when
! the {\tt state} is destroyed the <item>s contained in it will
! not be destroyed. Also, the <item> cannot be safely
! destroyed before the {\tt state} is destroyed.
! Since <item>s can be added to multiple containers, it remains
! the responsibility of the user to manage their
! destruction when they are no longer in use.
! \item[{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "add",
! where it is {\em not} an error if {\tt <item>} contains items
! with names that are already found in {\tt state}. The {\tt State}
! is left unchanged for these items. For {\tt .false.} this is treated
! as an error condition. The default setting is {\tt .false.}.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
! When replacing an existing <item>, the {\tt ESMF\_StateReplace} method
! should be used.
!EOPI
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
!------------------------------------------------------------------------------
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAdd" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAdd - Add an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddOneArray (state, array, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_Array), intent(in) :: array 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, array, addflag=.true., proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddOneArray 
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAddRp" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAddRp - Add/replace an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddRpOneArray (state, array, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_Array), intent(in) :: array 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, array, & 
! addflag=.true., replaceflag=.true., & 
! proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddRpOneArray 
 
!------------------------------------------------------------------------------ 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddX" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add an item to a State with proxyflag 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddOneArrayX (state, array, & 
 proxyflag, addflag, replaceflag, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Array), intent(in) :: array 
 logical, intent(in) :: proxyflag 
 logical, intent(in), optional :: addflag 
 logical, intent(in), optional :: replaceflag 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 type(ESMF_Array) :: temp_list(1) 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 integer :: localrc 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,array,rc) 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 temp_list(1) = array 
 
 call ESMF_StateClsAdd (state%statep, temp_list, & 
 addflag=addflag, replaceflag=replaceflag, relaxedflag=relaxedflag, & 
 proxyflag=proxyflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddOneArrayX
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAdd" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAdd - Add an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddOneArrayBundle (state, arraybundle, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_ArrayBundle), intent(in) :: arraybundle 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, arraybundle, addflag=.true., proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddOneArrayBundle 
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAddRp" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAddRp - Add/replace an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddRpOneArrayBundle (state, arraybundle, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_ArrayBundle), intent(in) :: arraybundle 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, arraybundle, & 
! addflag=.true., replaceflag=.true., & 
! proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddRpOneArrayBundle 
 
!------------------------------------------------------------------------------ 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddX" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add an item to a State with proxyflag 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddOneArrayBundleX (state, arraybundle, & 
 proxyflag, addflag, replaceflag, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_ArrayBundle), intent(in) :: arraybundle 
 logical, intent(in) :: proxyflag 
 logical, intent(in), optional :: addflag 
 logical, intent(in), optional :: replaceflag 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 type(ESMF_ArrayBundle) :: temp_list(1) 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 integer :: localrc 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayBundleGetInit,arraybundle,rc) 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 temp_list(1) = arraybundle 
 
 call ESMF_StateClsAdd (state%statep, temp_list, & 
 addflag=addflag, replaceflag=replaceflag, relaxedflag=relaxedflag, & 
 proxyflag=proxyflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddOneArrayBundleX
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAdd" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAdd - Add an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddOneField (state, field, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_Field), intent(in) :: field 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, field, addflag=.true., proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddOneField 
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAddRp" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAddRp - Add/replace an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddRpOneField (state, field, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_Field), intent(in) :: field 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, field, & 
! addflag=.true., replaceflag=.true., & 
! proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddRpOneField 
 
!------------------------------------------------------------------------------ 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddX" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add an item to a State with proxyflag 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddOneFieldX (state, field, & 
 proxyflag, addflag, replaceflag, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Field), intent(in) :: field 
 logical, intent(in) :: proxyflag 
 logical, intent(in), optional :: addflag 
 logical, intent(in), optional :: replaceflag 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 type(ESMF_Field) :: temp_list(1) 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 integer :: localrc 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 temp_list(1) = field 
 
 call ESMF_StateClsAdd (state%statep, temp_list, & 
 addflag=addflag, replaceflag=replaceflag, relaxedflag=relaxedflag, & 
 proxyflag=proxyflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddOneFieldX
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAdd" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAdd - Add an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddOneFieldBundle (state, fieldbundle, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_FieldBundle), intent(in) :: fieldbundle 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, fieldbundle, addflag=.true., proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddOneFieldBundle 
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAddRp" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAddRp - Add/replace an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddRpOneFieldBundle (state, fieldbundle, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_FieldBundle), intent(in) :: fieldbundle 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, fieldbundle, & 
! addflag=.true., replaceflag=.true., & 
! proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddRpOneFieldBundle 
 
!------------------------------------------------------------------------------ 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddX" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add an item to a State with proxyflag 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddOneFieldBundleX (state, fieldbundle, & 
 proxyflag, addflag, replaceflag, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_FieldBundle), intent(in) :: fieldbundle 
 logical, intent(in) :: proxyflag 
 logical, intent(in), optional :: addflag 
 logical, intent(in), optional :: replaceflag 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 type(ESMF_FieldBundle) :: temp_list(1) 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 integer :: localrc 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundle,rc) 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 temp_list(1) = fieldbundle 
 
 call ESMF_StateClsAdd (state%statep, temp_list, & 
 addflag=addflag, replaceflag=replaceflag, relaxedflag=relaxedflag, & 
 proxyflag=proxyflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddOneFieldBundleX
#define noattributesversion
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAdd" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAdd - Add an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddOneRouteHandle (state, routehandle, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_Routehandle), intent(in) :: routehandle 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, routehandle, addflag=.true., proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddOneRouteHandle 
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAddRp" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAddRp - Add/replace an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddRpOneRouteHandle (state, routehandle, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_Routehandle), intent(in) :: routehandle 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, routehandle, & 
! addflag=.true., replaceflag=.true., & 
! proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddRpOneRouteHandle 
 
!------------------------------------------------------------------------------ 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddX" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add an item to a State with proxyflag 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddOneRouteHandleX (state, routehandle, & 
 proxyflag, addflag, replaceflag, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Routehandle), intent(in) :: routehandle 
 logical, intent(in) :: proxyflag 
 logical, intent(in), optional :: addflag 
 logical, intent(in), optional :: replaceflag 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 type(ESMF_Routehandle) :: temp_list(1) 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 integer :: localrc 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_RoutehandleGetInit,routehandle,rc) 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 temp_list(1) = routehandle 
 
 call ESMF_StateClsAdd (state%statep, temp_list, & 
 addflag=addflag, replaceflag=replaceflag, relaxedflag=relaxedflag, & 
 proxyflag=proxyflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddOneRouteHandleX
#undef noattributesversion
#define stateversion
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAdd" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAdd - Add an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddOneState (state, nestedState, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_State), intent(in) :: nestedState 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, nestedState, addflag=.true., proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddOneState 
! #undef ESMF_METHOD 
! #define ESMF_METHOD "ESMF_StateAddRp" 
! !BOPI 
! ! !IROUTINE: ESMF_StateAddRp - Add/replace an item to a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateAdd() 
! subroutine ESMF_StateAddRpOneState (state, nestedState, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_State), intent(in) :: nestedState 
! type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! call ESMF_StateAdd (state, nestedState, & 
! addflag=.true., replaceflag=.true., & 
! proxyflag=.false., rc=rc) 
! 
! end subroutine ESMF_StateAddRpOneState 
 
!------------------------------------------------------------------------------ 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddX" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add an item to a State with proxyflag 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddOneStateX (state, nestedState, & 
 proxyflag, addflag, replaceflag, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_State), intent(in) :: nestedState 
 logical, intent(in) :: proxyflag 
 logical, intent(in), optional :: addflag 
 logical, intent(in), optional :: replaceflag 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 type(ESMF_State) :: temp_list(1) 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 integer :: localrc 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,nestedState,rc) 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 temp_list(1) = nestedState 
 
 call ESMF_StateClsAdd (state%statep, temp_list, & 
 addflag=addflag, replaceflag=replaceflag, relaxedflag=relaxedflag, & 
 proxyflag=proxyflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddOneStateX
#undef stateversion

!------------------------------------------------------------------------------
!BOP
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State
!
! !INTERFACE:
! subroutine ESMF_StateAdd(state, <itemList>, relaxedFlag, rc)
!
! !ARGUMENTS:
! type(ESMF_State), intent(inout) :: state
! <itemList>, see below for supported values
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! logical, intent(in), optional :: relaxedFlag
! integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Add a list of items to a {\tt ESMF\_State}. It is an error if any item in
! <itemlist> already matches, by name, an item already contained in {\tt state}.
!
! Supported values for <itemList> are:
! \begin{description}
! \item type(ESMF\_Array), intent(in) :: arrayList(:)
! \item type(ESMF\_ArrayBundle), intent(in) :: arraybundleList(:)
! \item type(ESMF\_Field), intent(in) :: fieldList(:)
! \item type(ESMF\_FieldBundle), intent(in) :: fieldbundleList(:)
! \item type(ESMF\_RouteHandle), intent(in) :: routehandleList(:)
! \item type(ESMF\_State), intent(in) :: nestedStateList(:)
! \end{description}
!
! The arguments are:
! \begin{description}
! \item[state]
! An {\tt ESMF\_State} to which the <itemList> will be added.
! \item[<itemList>]
! The list of items to be added.
! This is a reference only; when
! the {\tt ESMF\_State} is destroyed the <itemList> items contained within it will
! not be destroyed. Also, the items in the <itemList> cannot be safely
! destroyed before the {\tt ESMF\_State} is destroyed.
! Since <itemList> items can be added to multiple containers, it remains
! the responsibility of the user to manage their
! destruction when they are no longer in use.
! \item[{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "add",
! where it is {\em not} an error if {\tt <itemList>} contains items
! with names that are found in {\tt state}. The {\tt State}
! is left unchanged for these items. For {\tt .false.} this is treated
! as an error condition. The default setting is {\tt .false.}.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddArrayList(state, arrayList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Array), intent(in) :: arrayList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
 logical :: localrelaxed 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (arrayList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,arrayList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, arrayList, & 
 addflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddArrayList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddArrayBundleList(state, arraybundleList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_ArrayBundle), intent(in) :: arraybundleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
 logical :: localrelaxed 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (arraybundleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayBundleGetInit,arraybundleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, arraybundleList, & 
 addflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddArrayBundleList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddFieldList(state, fieldList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Field), intent(in) :: fieldList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
 logical :: localrelaxed 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (fieldList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,fieldList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, fieldList, & 
 addflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddFieldList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddFieldBundleList(state, fieldbundleList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_FieldBundle), intent(in) :: fieldbundleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
 logical :: localrelaxed 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (fieldbundleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, fieldbundleList, & 
 addflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddFieldBundleList
#define noattributesversion
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddRouteHandleList(state, routehandleList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_RouteHandle), intent(in) :: routehandleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
 logical :: localrelaxed 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (routehandleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_RouteHandleGetInit,routehandleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, routehandleList, & 
 addflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddRouteHandleList
#undef noattributesversion
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateAddStateList(state, nestedStateList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_State), intent(in) :: nestedStateList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
 logical :: localrelaxed 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (nestedStateList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,nestedStateList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, nestedStateList, & 
 addflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddStateList


!------------------------------------------------------------------------------
!BOP
! !IROUTINE: ESMF_StateAddReplace - Add or replace a list of items to a State
!
! !INTERFACE:
! subroutine ESMF_StateAddReplace(state, <itemList>, rc)
!
! !ARGUMENTS:
! type(ESMF_State), intent(inout) :: state
! <itemList>, see below for supported values
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Add or replace a list of items to an {\tt ESMF\_State}. If an item in
! <itemList> does not match any items already present in {\tt state}, it is
! added. Items with names already present in the {\tt state} replace the
! existing item.
!
! Supported values for <itemList> are:
! \begin{description}
! \item type(ESMF\_Array), intent(in) :: arrayList(:)
! \item type(ESMF\_ArrayBundle), intent(in) :: arraybundleList(:)
! \item type(ESMF\_Field), intent(in) :: fieldList(:)
! \item type(ESMF\_FieldBundle), intent(in) :: fieldbundleList(:)
! \item type(ESMF\_RouteHandle), intent(in) :: routehandleList(:)
! \item type(ESMF\_State), intent(in) :: nestedStateList(:)
! \end{description}
!
! The arguments are:
! \begin{description}
! \item[state]
! An {\tt ESMF\_State} to which the <itemList> will be added or replaced.
! \item[<itemList>]
! The list of items to be added or replaced.
! This is a reference only; when
! the {\tt ESMF\_State} is destroyed the <itemList> items contained within it will
! not be destroyed. Also, the items in the <itemList> cannot be safely
! destroyed before the {\tt ESMF\_State} is destroyed.
! Since <itemList> items can be added to multiple containers, it remains
! the responsibility of the user to manage their
! destruction when they are no longer in use.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddRpList" 
!BOPI 
! !IROUTINE: ESMF_StateAddReplace - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAddReplace() 
 subroutine ESMF_StateAddRpArrayList(state, arrayList, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Array), intent(in) :: arrayList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localcount = size (arrayList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,arrayList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, arrayList, & 
 addflag=.true., replaceflag=.true., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddRpArrayList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddRpList" 
!BOPI 
! !IROUTINE: ESMF_StateAddReplace - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAddReplace() 
 subroutine ESMF_StateAddRpArrayBundleList(state, arraybundleList, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_ArrayBundle), intent(in) :: arraybundleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localcount = size (arraybundleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayBundleGetInit,arraybundleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, arraybundleList, & 
 addflag=.true., replaceflag=.true., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddRpArrayBundleList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddRpList" 
!BOPI 
! !IROUTINE: ESMF_StateAddReplace - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAddReplace() 
 subroutine ESMF_StateAddRpFieldList(state, fieldList, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Field), intent(in) :: fieldList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localcount = size (fieldList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,fieldList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, fieldList, & 
 addflag=.true., replaceflag=.true., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddRpFieldList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddRpList" 
!BOPI 
! !IROUTINE: ESMF_StateAddReplace - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAddReplace() 
 subroutine ESMF_StateAddRpFieldBundleList(state, fieldbundleList, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_FieldBundle), intent(in) :: fieldbundleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localcount = size (fieldbundleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, fieldbundleList, & 
 addflag=.true., replaceflag=.true., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddRpFieldBundleList
#define noattributesversion
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddRpList" 
!BOPI 
! !IROUTINE: ESMF_StateAddReplace - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAddReplace() 
 subroutine ESMF_StateAddRpRouteHandleList(state, routehandleList, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_RouteHandle), intent(in) :: routehandleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localcount = size (routehandleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_RouteHandleGetInit,routehandleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, routehandleList, & 
 addflag=.true., replaceflag=.true., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddRpRouteHandleList
#undef noattributesversion
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateAddRpList" 
!BOPI 
! !IROUTINE: ESMF_StateAddReplace - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAddReplace() 
 subroutine ESMF_StateAddRpStateList(state, nestedStateList, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_State), intent(in) :: nestedStateList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lvalue1, lvalue2 
 character(ESMF_MAXSTR) :: lobject, lname 
#endif 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localcount = size (nestedStateList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,nestedStateList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, nestedStateList, & 
 addflag=.true., replaceflag=.true., rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateAddRpStateList

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateCreate"
!BOP
! !IROUTINE: ESMF_StateCreate - Create a new State

! !INTERFACE:
      function ESMF_StateCreate(keywordEnforcer, stateIntent, &
                   arrayList, arraybundleList, &
                   fieldList, fieldbundleList, &
                   nestedStateList, &
                   routehandleList, name, vm, rc)
!
! !RETURN VALUE:
      type(ESMF_State) :: ESMF_StateCreate
!
! !ARGUMENTS:
    type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      type(ESMF_StateIntent_Flag), intent(in), optional :: stateIntent
      type(ESMF_Array), intent(in), optional :: arrayList(:)
      type(ESMF_ArrayBundle), intent(in), optional :: arraybundleList(:)
      type(ESMF_Field), intent(in), optional :: fieldList(:)
      type(ESMF_FieldBundle), intent(in), optional :: fieldbundleList(:)
      type(ESMF_State), intent(in), optional :: nestedStateList(:)
      type(ESMF_RouteHandle), intent(in), optional :: routehandleList(:)
      character(len=*), intent(in), optional :: name
      type(ESMF_VM), intent(in), optional :: vm
      integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt vm} to support object creation on a
! different VM than that of the current context.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Create a new {\tt ESMF\_State}, set default characteristics for
! objects added to it, and optionally add initial objects to it.
!
! The arguments are:
! \begin{description}
! \item[{[stateIntent]}]
! The intent, e.g. Import, Export, or Internal, of this {\tt ESMF\_State}.
! Possible values are listed in Section~\ref{const:stateintent}.
! The default is {\tt ESMF\_STATEINTENT\_UNSPECIFIED}.
! \item[{[arrayList]}]
! A list (Fortran array) of {\tt ESMF\_Array}s.
! \item[{[arraybundleList]}]
! A list (Fortran array) of {\tt ESMF\_ArrayBundle}s.
! \item[{[fieldList]}]
! A list (Fortran array) of {\tt ESMF\_Field}s.
! \item[{[fieldbundleList]}]
! A list (Fortran array) of {\tt ESMF\_FieldBundle}s.
! \item[{[nestedStateList]}]
! A list (Fortran array) of {\tt ESMF\_State}s to be nested
! inside the outer {\tt ESMF\_State}.
! \item[{[routehandleList]}]
! A list (Fortran array) of {\tt ESMF\_RouteHandle}s.
! \item[{[name]}]
! Name of this {\tt ESMF\_State} object. A default name will be generated
! if none is specified.
! \item[{[vm]}]
! If present, the State object is created on the specified
! {\tt ESMF\_VM} object. The default is to create on the VM of the
! current component context.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP

      ! local vars
      type (ESMF_StateClass), pointer :: stypep
      integer :: localrc ! local error status
      integer :: memstat ! Stat from allocate/deallocate
      integer :: i
      type(ESMF_Pointer) :: vmThis
      logical :: actualFlag

      ! Initialize return code; assume failure until success is certain
      if (present(rc)) rc = ESMF_RC_NOT_IMPL

      ! Initialize the pointers to null.
      nullify(ESMF_StateCreate%statep)
      nullify(stypep)

      allocate(stypep, stat=memstat)
      if (ESMF_LogFoundAllocError(memstat, msg="State type", &
               ESMF_CONTEXT, rcToReturn=rc)) return

      ! Must make sure the local PET is associated with an actual member
      actualFlag = .true.
      if (present(vm)) then
        ESMF_INIT_CHECK_DEEP(ESMF_VMGetInit, vm, rc)
        call ESMF_VMGetThis(vm, vmThis)
        if (vmThis == ESMF_NULL_POINTER) then
          actualFlag = .false. ! local PET is not for an actual member
        endif
      endif

      if (actualFlag) then

        ! check input variables
        if (present(arrayList)) then
           do i=1,size(arrayList)
ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,arrayList(i),rc)
           enddo
        endif
        if (present(arraybundleList)) then
           do i=1,size(arraybundleList)
ESMF_INIT_CHECK_DEEP(ESMF_ArrayBundleGetInit,arraybundleList(i),rc)
           enddo
        endif
        if (present(fieldbundleList)) then
           do i=1,size(fieldbundleList)
ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundleList(i),rc)
           enddo
        endif
        if (present(fieldList)) then
           do i=1,size(fieldList)
ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,fieldList(i),rc)
           enddo
        endif
        if (present(nestedStateList)) then
           do i=1,size(nestedStateList)
ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,nestedStateList(i),rc)
           enddo
        endif
        if (present(routehandleList)) then
           do i=1,size(routehandleList)
ESMF_INIT_CHECK_DEEP(ESMF_RouteHandleGetInit,routehandleList(i),rc)
           enddo
        endif

        call ESMF_StateConstruct(stypep, &
                 statename=name, stateintent=stateintent, &
                 arrays=arrayList, arraybundles=arraybundleList, &
                 fields=fieldList, fieldbundles=fieldbundleList, &
                 states=nestedStateList, &
                 routehandles=routehandleList, vm=vm, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) then
            deallocate(stypep, stat=memstat)
            return
        endif

      endif ! - actualFlag

      ! Set return values
      ESMF_StateCreate%statep => stypep
      ESMF_INIT_SET_CREATED(ESMF_StateCreate)

      if (actualFlag) then
        ! Add reference to this object into ESMF garbage collection table
        call c_ESMC_VMAddFObject(ESMF_StateCreate, ESMF_ID_STATE%objectID)
      endif ! - actualFlag

      if (present(rc)) rc = ESMF_SUCCESS

    end function ESMF_StateCreate


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateDestroy"
!BOP
! !IROUTINE: ESMF_StateDestroy - Release resources for a State
!
! !INTERFACE:
      recursive subroutine ESMF_StateDestroy(state, keywordEnforcer, noGarbage, rc)
!
! !ARGUMENTS:
      type(ESMF_State), intent(inout) :: state
    type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      logical, intent(in), optional :: noGarbage
      integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt noGarbage}.
! The argument provides a mechanism to override the default garbage collection
! mechanism when destroying an ESMF object.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Releases resources associated with this {\tt ESMF\_State}. Actual
! objects added to {\tt ESMF\_State}s will not be destroyed, it
! remains the responsibility of the user to destroy these objects in the correct
! context.
!
! The arguments are:
! \begin{description}
! \item[state]
! Destroy contents of this {\tt ESMF\_State}.
! \item[{[noGarbage]}]
! If set to {\tt .TRUE.} the object will be fully destroyed and removed
! from the ESMF garbage collection system. Note however that under this
! condition ESMF cannot protect against accessing the destroyed object
! through dangling aliases -- a situation which may lead to hard to debug
! application crashes.
!
! It is generally recommended to leave the {\tt noGarbage} argument
! set to {\tt .FALSE.} (the default), and to take advantage of the ESMF
! garbage collection system which will prevent problems with dangling
! aliases or incorrect sequences of destroy calls. However this level of
! support requires that a small remnant of the object is kept in memory
! past the destroy call. This can lead to an unexpected increase in memory
! consumption over the course of execution in applications that use
! temporary ESMF objects. For situations where the repeated creation and
! destruction of temporary objects leads to memory issues, it is
! recommended to call with {\tt noGarbage} set to {\tt .TRUE.}, fully
! removing the entire temporary object from memory.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP

        ! Local vars
        integer :: localrc ! local error status

        ! Initialize return code; assume failure until success is certain
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        ! check input variables
        ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc)

        if (.not.associated(state%statep)) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
            msg="Uninitialized or already destroyed State: statep unassociated", &
            ESMF_CONTEXT, rcToReturn=rc)
          return
        endif

        ! Call Destruct to release resources
        call ESMF_StateDestruct(state%statep, localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if (present(noGarbage)) then
          if (noGarbage) then
            ! destroy Base object (which also removes it from garbage collection)
            call ESMF_BaseDestroy(state%statep%base, noGarbage, rc=localrc)
            if (ESMF_LogFoundError(localrc, &
              ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return
            ! remove reference to this object from ESMF garbage collection table
            call c_ESMC_VMRmFObject(state)
            ! deallocate the actual fieldbundle data structure
            deallocate(state%statep, stat=localrc)
            if (ESMF_LogFoundDeallocError(localrc, &
              msg="Deallocating State information", &
              ESMF_CONTEXT, rcToReturn=rc)) return
          endif
        endif

        ! Mark this State as invalid
        nullify(state%statep)

        ! Invalidate Destroyed State
        ESMF_INIT_SET_DELETED(state)

        ! Set return code if user specified it
        if (present(rc)) rc = ESMF_SUCCESS

        end subroutine ESMF_StateDestroy

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateGetInfo"
!BOP
! !IROUTINE: ESMF_StateGet - Get object-wide information from a State
!
! !INTERFACE:
      ! Private name; call using ESMF_StateGet()
      subroutine ESMF_StateGetInfo(state, &
            keywordEnforcer, itemSearch, itemorderflag, nestedFlag, &
            stateIntent, itemCount, itemNameList, itemTypeList, name, vm, rc)
!
! !ARGUMENTS:
      type(ESMF_State), intent(in) :: state
    type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      character (len=*), intent(in), optional :: itemSearch
      type(ESMF_ItemOrder_Flag), intent(in), optional :: itemorderflag
      logical, intent(in), optional :: nestedFlag
      type(ESMF_StateIntent_Flag), intent(out), optional :: stateIntent
      integer, intent(out), optional :: itemCount
      character (len=*), intent(out), optional :: itemNameList(:)
      type(ESMF_StateItem_Flag), intent(out), optional :: itemTypeList(:)
      character (len=*), intent(out), optional :: name
      type(ESMF_VM), intent(out), optional :: vm
      integer, intent(out), optional :: rc

!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[6.1.0] Added argument {\tt itemorderflag}.
! The new argument gives the user control over the order in which
! the items are returned.
! \item[8.8.0] Added argument {\tt vm} in order to offer information about the
! VM on which the State was created.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Returns the requested information about this {\tt ESMF\_State}.
! The optional {\tt itemSearch} argument may specify the name of
! an individual item to search for. When used in conjunction with
! the {\tt nestedFlag}, nested States will also be searched.
!
! Typically, an {\tt ESMF\_StateGet()} information request will be performed
! twice. The first time, the {\tt itemCount} argument will be used to
! query the size of arrays that are needed. Arrays can then be allocated
! to the correct size for {\tt itemNameList} and {\tt itemtypeList}
! as needed. A second call to {\tt ESMF\_StateGet()} will then fill in the
! values.
!
! The arguments are:
! \begin{description}
! \item[state]
! An {\tt ESMF\_State} object to be queried.
! \item[{[itemSearch]}]
! Query objects by name in the State. When the {\tt nestedFlag}
! option is set to .true., all nested States will also be searched
! for the specified name.
! \item[{[itemorderflag]}]
! Specifies the order of the returned items in the {\tt itemNameList}
! and {\tt itemTypeList}. The default is {\tt ESMF\_ITEMORDER\_ABC}.
! See \ref{const:itemorderflag} for a full list of options.
! \item[{[nestedFlag]}]
! When set to {\tt .false.}, returns information at the current
! State level only (default)
! When set to {\tt .true.}, additionally returns information from
! nested States
! \item[{[stateIntent]}]
! The intent, e.g. Import, Export, or Internal, of this {\tt ESMF\_State}.
! Possible values are listed in Section~\ref{const:stateintent}.
! \item[{[itemCount]}]
! Count of items in this {\tt ESMF\_State}.
! When the {\tt nestedFlag} option is
! set to {\tt .true.}, the count will include items present in nested
! States. When using {\tt itemSearch}, it will count the number of
! items matching the specified name.
! \item[{[itemNameList]}]
! Array of item names in this {\tt ESMF\_State}.
! When the {\tt nestedFlag} option is
! set to {\tt .true.}, the list will include items present in nested
! States. When using {\tt itemSearch}, it will return the names of
! items matching the specified name. {\tt itemNameList} must be at least
! {\tt itemCount} long.
! \item[{[itemTypeList]}]
! Array of possible item object types in this {\tt ESMF\_State}.
! When the {\tt nestedFlag} option is
! set to {\tt .true.}, the list will include items present in nested
! States. When using {\tt itemSearch}, it will return the types of
! items matching the specified name. Must be at least {\tt itemCount}
! long. Return values are listed in Section~\ref{const:stateitem}.
! \item[{[name]}]
! Returns the name of this {\tt ESMF\_State}.
! \item [{[vm}]
! The VM on which the State object was created.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
      integer :: ilpos
      integer :: localrc
      integer :: localitemcount
      logical :: localnestedflag
      type(ESMF_ItemOrder_Flag) :: itemorderflagArg
      type(ESMF_StateClass), pointer :: stypep

      localrc = ESMF_SUCCESS
      if (present(rc)) rc = ESMF_SUCCESS

      ! check input variables
      ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc)

      call ESMF_StateValidate(state, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      localnestedflag = .false.
      if (present (nestedFlag)) then
        localnestedflag = nestedFlag
      end if

      itemorderflagArg = ESMF_ITEMORDER_ABC ! default
      if (present(itemorderflag)) &
        itemorderflagArg = itemorderflag

      stypep => state%statep

      if (present(name)) then
        if (state%isNamedAlias) then
          name = trim(state%name)
        else
          call ESMF_GetName(stypep%base, name=name, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        endif
      endif

      if (present(stateintent)) stateintent = stypep%st

      localitemcount = infoCountWorker (stypep)
      ! infoCountWorker can set localrc -> must check on return
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      if (present(itemCount)) &
        itemCount = localitemcount

      if (present(itemNameList)) then
        ilpos = 1
        call itemNameWorker (stypep, prefix="")
      endif

      if (present(itemtypeList)) then
        ilpos = 1
        call itemTypeWorker (stypep)
      endif

      if (present(vm)) then
        call ESMF_GetVM(stypep%base, vm, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif

      ! return successfully
      if (present(rc)) rc = ESMF_SUCCESS

    contains

        recursive function infoCountWorker (sp) result (icount)
          type(ESMF_StateClass), pointer :: sp
          integer :: icount

          integer :: i1
          type(ESMF_StateItemWrap), pointer :: siwrap(:)
          integer :: sipcount
          integer :: memstat1

          icount = 0

          call ESMF_ContainerGet (sp%stateContainer, itemCount=sipcount, rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return

          siwrap => null ()
          if (present (itemSearch) .or. localnestedflag) then
            call ESMF_ContainerGet (sp%stateContainer, itemList=siwrap, rc=localrc)
            if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return
          end if

          if (.not. present (itemSearch)) then
            icount = sipcount
          else
            icount = 0
            do, i1 = 1, sipcount
              if (siwrap(i1)%si%namep == itemSearch) then
                icount = icount + 1
              end if
            end do
          end if

          if (localnestedflag) then
            do, i1 = 1, sipcount
              if (siwrap(i1)%si%otype%ot == ESMF_STATEITEM_STATE%ot) then
                icount = icount + infoCountWorker (siwrap(i1)%si%datap%spp)
              end if
            end do
          end if

          if (associated (siwrap)) &
            deallocate (siwrap, stat=memstat1)

        end function infoCountWorker

        recursive subroutine itemNameWorker (sp, prefix)
          type(ESMF_StateClass), pointer :: sp
          character(*), intent(in) :: prefix

        ! Copy as many names as will fit in the output array.

          integer :: i1
          type(ESMF_StateItemWrap) , pointer :: siwrap(:)
          type(ESMF_StateItem), pointer :: sip
          integer :: sipcount
          integer :: memstat1

          siwrap => null ()
          call ESMF_ContainerGet (sp%stateContainer, &
              itemorderflag=itemorderflagArg, &
              itemCount=sipcount, &
              itemList=siwrap, &
              rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return

          do, i1 = 1, sipcount
            if (ilpos > size (itemNameList)) then
              localrc = ESMF_RC_ARG_SIZE
              exit
            end if

            sip => siwrap(i1)%si

            if (.not. present (itemSearch)) then
              itemNameList(ilpos) = ESMF_StringConcat (prefix, sip%namep)
              ilpos = ilpos + 1
            else
              if (sip%namep == itemSearch) then
                itemNameList(ilpos) = ESMF_StringConcat (prefix, sip%namep)
                ilpos = ilpos + 1
              end if
            end if

            if (sip%otype%ot == ESMF_STATEITEM_STATE%ot &
                .and. localnestedflag) then
              call itemNameWorker (sip%datap%spp, &
                  prefix=ESMF_StringConcat (prefix, ESMF_StringConcat (trim (sip%namep), '/')))
            end if
          end do

          if (associated (siwrap)) &
            deallocate (siwrap, stat=memstat1)

        end subroutine itemNameWorker

        recursive subroutine itemTypeWorker (sp)
          type(ESMF_StateClass), pointer :: sp

        ! Copy as many type fields as will fit in the output array.

          integer :: i1
          type(ESMF_StateItemWrap), pointer :: siwrap(:)
          type(ESMF_StateItem), pointer :: sip
          integer :: sipcount
          integer :: memstat1

          siwrap => null ()
          call ESMF_ContainerGet (sp%stateContainer, &
              itemorderflag=itemorderflagArg, &
              itemCount=sipcount, &
              itemList=siwrap, &
              rc=localrc)
          if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return

          do, i1 = 1, sipcount
            if (ilpos > size (itemtypeList)) then
              localrc = ESMF_RC_ARG_SIZE
              exit
            end if

            sip => siwrap(i1)%si

            if (.not. present (itemSearch)) then
              itemtypeList(ilpos) = sip%otype
              ilpos = ilpos + 1
            else
              if (sip%namep == itemSearch) then
                itemtypeList(ilpos) = sip%otype
                ilpos = ilpos + 1
              end if
            end if

            if (sip%otype%ot == ESMF_STATEITEM_STATE%ot &
                .and. localnestedflag) then
              call itemTypeWorker (sip%datap%spp)
            end if
          end do

          if (associated (siwrap)) &
            deallocate (siwrap, stat=memstat1)

        end subroutine itemTypeWorker

    end subroutine ESMF_StateGetInfo

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateGetItemInfo"
!BOP
! !IROUTINE: ESMF_StateGet - Get information about an item in a State by item name
!
! !INTERFACE:
      ! Private name; call using ESMF_StateGet()
      subroutine ESMF_StateGetItemInfo(state, itemName, itemType, keywordEnforcer, rc)
!
! !ARGUMENTS:
      type(ESMF_State), intent(in) :: state
      character (len=*), intent(in) :: itemName
      type(ESMF_StateItem_Flag), intent(out) :: itemType
    type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
      integer, intent(out), optional :: rc

!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Returns the type for the item named
! {\tt name} in this {\tt ESMF\_State}. If no item with this name
! exists, the value {\tt ESMF\_STATEITEM\_NOTFOUND} will be returned
! and the error code will not be set to an error. Thus this routine
! can be used to safely query for the existence of items by name
! whether or not they are expected to be there. The error code will
! be set in case of other errors, for example if the {\tt ESMF\_State}
! itself is invalid.
!
! The arguments are:
! \begin{description}
! \item[state]
! {\tt ESMF\_State} to be queried.
! \item[itemName]
! Name of the item to return information about.
! \item[itemType]
! Returned item types for the item with the given name, including
! placeholder names. Options are
! listed in Section~\ref{const:stateitem}. If no item with the
! given name is found, {\tt ESMF\_STATEITEM\_NOTFOUND} will be returned
! and {\tt rc} will {\bf not} be set to an error.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!
!EOP
      integer :: localrc
      type(ESMF_Container), pointer :: scp
      type(ESMF_StateItemWrap) :: siwrap
      logical :: foundflag

        ! check input variables
        ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc)

      ! Initialize return code; assume routine not implemented
      if (present(rc)) rc = ESMF_RC_NOT_IMPL
      localrc = ESMF_RC_NOT_IMPL

      call ESMF_StateValidate(state, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

      scp => state%statep%stateContainer

      ! Find the object which matches this name
      call ESMF_ContainerGet (scp, itemname=itemName, &
          isPresent=foundflag, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

      if (foundflag) then
        call ESMF_ContainerGet (scp, itemName=itemName, &
            item=siwrap, rc=localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        itemtype = siwrap%si%otype
      else
        itemtype = ESMF_STATEITEM_NOTFOUND
      end if

      if (present(rc)) rc = ESMF_SUCCESS

    end subroutine ESMF_StateGetItemInfo

!------------------------------------------------------------------------------
!BOP
! !IROUTINE: ESMF_StateGet - Get an item from a State by item name
!
! !INTERFACE:
! subroutine ESMF_StateGet(state, itemName, <item>, rc)
!
! !ARGUMENTS:
! type(ESMF_State), intent(in) :: state
! character (len=*), intent(in) :: itemName
! <item>, see below for supported values
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! integer, intent(out), optional :: rc
!
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! \begin{sloppypar}
! Returns an <item> from an {\tt ESMF\_State} by item name.
! If the {\tt ESMF\_State} contains the <item> directly, only
! {\tt itemName} is required.
! \end{sloppypar}
!
! If the {\tt state} contains nested {\tt ESMF\_State}s,
! the {\tt itemName} argument may specify a fully qualified name
! to access the desired item with a single call. This is performed
! using the '/' character to separate the names of the intermediate
! State names leading to the desired item. (E.g.,
! {\tt itemName='state1/state12/item'}).
!
! Supported values for <item> are:
! \begin{description}
! \item type(ESMF\_Array), intent(out) :: array
! \item type(ESMF\_ArrayBundle), intent(out) :: arraybundle
! \item type(ESMF\_Field), intent(out) :: field
! \item type(ESMF\_FieldBundle), intent(out) :: fieldbundle
! \item type(ESMF\_RouteHandle), intent(out) :: routehandle
! \item type(ESMF\_State), intent(out) :: nestedState
! \end{description}
!
! The arguments are:
! \begin{description}
! \item[state]
! State to query for an <item> named {\tt itemName}.
! \item[itemName]
! Name of <item> to be returned. This name may be fully
! qualified in order to access nested State items.
! \item[<item>]
! Returned reference to the <item>.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateGet" 
!BOPI 
 
! !IROUTINE: ESMF_StateGet - Retrieve an item from a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateGet() 
 subroutine ESMF_StateGetArray(state, itemName, array, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(in) :: state 
 character (len=*), intent(in) :: itemName 
 type(ESMF_Array), intent(out) :: array 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 
 
!EOPI 
 
 type(ESMF_StateItem), pointer :: dataitem 
 logical :: exists 
 integer :: localrc 
 character(len=len(itemName)+ESMF_MAXSTR) :: errmsg 
 
 ! print *, 'ESMF_StateGet', "Array", ': getting ', trim (itemName) 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 ! Assume failure until we know we will succeed 
 if (present(rc)) rc=ESMF_RC_NOT_IMPL 
 ! TODO: do we need an empty (or invalid) item to mark failure? 
 
 exists = ESMF_StateClassFindData(state%statep, & 
 dataname=itemName, expected=.true., & 
 dataitem=dataitem, & 
 rc=localrc) 
 if (.not. exists) then 
 write(errmsg, *) "no ", "ESMF_Array", " found named: ", trim(itemName) 
 if (ESMF_LogFoundError(ESMF_RC_NOT_FOUND, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
 if (dataitem%otype .ne. ESMF_STATEITEM_Array) then 
 write(errmsg, *) trim(itemName), " found but not type ", "ESMF_Array" 
 if (ESMF_LogFoundError(ESMF_RC_ARG_INCOMP, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
#if !defined (stateversion) 
 array = dataitem%datap%ap 
#else 
 array%statep => dataitem%datap%ap 
 ! validate created state 
 ESMF_INIT_SET_CREATED(nestedState) 
#endif 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 
 end subroutine ESMF_StateGetArray
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateGet" 
!BOPI 
 
! !IROUTINE: ESMF_StateGet - Retrieve an item from a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateGet() 
 subroutine ESMF_StateGetArrayBundle(state, itemName, arraybundle, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(in) :: state 
 character (len=*), intent(in) :: itemName 
 type(ESMF_ArrayBundle), intent(out) :: arraybundle 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 
 
!EOPI 
 
 type(ESMF_StateItem), pointer :: dataitem 
 logical :: exists 
 integer :: localrc 
 character(len=len(itemName)+ESMF_MAXSTR) :: errmsg 
 
 ! print *, 'ESMF_StateGet', "ArrayBundle", ': getting ', trim (itemName) 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 ! Assume failure until we know we will succeed 
 if (present(rc)) rc=ESMF_RC_NOT_IMPL 
 ! TODO: do we need an empty (or invalid) item to mark failure? 
 
 exists = ESMF_StateClassFindData(state%statep, & 
 dataname=itemName, expected=.true., & 
 dataitem=dataitem, & 
 rc=localrc) 
 if (.not. exists) then 
 write(errmsg, *) "no ", "ESMF_ArrayBundle", " found named: ", trim(itemName) 
 if (ESMF_LogFoundError(ESMF_RC_NOT_FOUND, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
 if (dataitem%otype .ne. ESMF_STATEITEM_ArrayBundle) then 
 write(errmsg, *) trim(itemName), " found but not type ", "ESMF_ArrayBundle" 
 if (ESMF_LogFoundError(ESMF_RC_ARG_INCOMP, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
#if !defined (stateversion) 
 arraybundle = dataitem%datap%abp 
#else 
 arraybundle%statep => dataitem%datap%abp 
 ! validate created state 
 ESMF_INIT_SET_CREATED(nestedState) 
#endif 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 
 end subroutine ESMF_StateGetArrayBundle
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateGet" 
!BOPI 
 
! !IROUTINE: ESMF_StateGet - Retrieve an item from a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateGet() 
 subroutine ESMF_StateGetField(state, itemName, field, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(in) :: state 
 character (len=*), intent(in) :: itemName 
 type(ESMF_Field), intent(out) :: field 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 
 
!EOPI 
 
 type(ESMF_StateItem), pointer :: dataitem 
 logical :: exists 
 integer :: localrc 
 character(len=len(itemName)+ESMF_MAXSTR) :: errmsg 
 
 ! print *, 'ESMF_StateGet', "Field", ': getting ', trim (itemName) 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 ! Assume failure until we know we will succeed 
 if (present(rc)) rc=ESMF_RC_NOT_IMPL 
 ! TODO: do we need an empty (or invalid) item to mark failure? 
 
 exists = ESMF_StateClassFindData(state%statep, & 
 dataname=itemName, expected=.true., & 
 dataitem=dataitem, & 
 rc=localrc) 
 if (.not. exists) then 
 write(errmsg, *) "no ", "ESMF_Field", " found named: ", trim(itemName) 
 if (ESMF_LogFoundError(ESMF_RC_NOT_FOUND, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
 if (dataitem%otype .ne. ESMF_STATEITEM_Field) then 
 write(errmsg, *) trim(itemName), " found but not type ", "ESMF_Field" 
 if (ESMF_LogFoundError(ESMF_RC_ARG_INCOMP, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
#if !defined (stateversion) 
 field = dataitem%datap%fp 
#else 
 field%statep => dataitem%datap%fp 
 ! validate created state 
 ESMF_INIT_SET_CREATED(nestedState) 
#endif 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 
 end subroutine ESMF_StateGetField
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateGet" 
!BOPI 
 
! !IROUTINE: ESMF_StateGet - Retrieve an item from a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateGet() 
 subroutine ESMF_StateGetFieldBundle(state, itemName, fieldbundle, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(in) :: state 
 character (len=*), intent(in) :: itemName 
 type(ESMF_FieldBundle), intent(out) :: fieldbundle 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 
 
!EOPI 
 
 type(ESMF_StateItem), pointer :: dataitem 
 logical :: exists 
 integer :: localrc 
 character(len=len(itemName)+ESMF_MAXSTR) :: errmsg 
 
 ! print *, 'ESMF_StateGet', "FieldBundle", ': getting ', trim (itemName) 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 ! Assume failure until we know we will succeed 
 if (present(rc)) rc=ESMF_RC_NOT_IMPL 
 ! TODO: do we need an empty (or invalid) item to mark failure? 
 
 exists = ESMF_StateClassFindData(state%statep, & 
 dataname=itemName, expected=.true., & 
 dataitem=dataitem, & 
 rc=localrc) 
 if (.not. exists) then 
 write(errmsg, *) "no ", "ESMF_FieldBundle", " found named: ", trim(itemName) 
 if (ESMF_LogFoundError(ESMF_RC_NOT_FOUND, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
 if (dataitem%otype .ne. ESMF_STATEITEM_FieldBundle) then 
 write(errmsg, *) trim(itemName), " found but not type ", "ESMF_FieldBundle" 
 if (ESMF_LogFoundError(ESMF_RC_ARG_INCOMP, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
#if !defined (stateversion) 
 fieldbundle = dataitem%datap%fbp 
#else 
 fieldbundle%statep => dataitem%datap%fbp 
 ! validate created state 
 ESMF_INIT_SET_CREATED(nestedState) 
#endif 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 
 end subroutine ESMF_StateGetFieldBundle
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateGet" 
!BOPI 
 
! !IROUTINE: ESMF_StateGet - Retrieve an item from a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateGet() 
 subroutine ESMF_StateGetRouteHandle(state, itemName, routehandle, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(in) :: state 
 character (len=*), intent(in) :: itemName 
 type(ESMF_RouteHandle), intent(out) :: routehandle 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 
 
!EOPI 
 
 type(ESMF_StateItem), pointer :: dataitem 
 logical :: exists 
 integer :: localrc 
 character(len=len(itemName)+ESMF_MAXSTR) :: errmsg 
 
 ! print *, 'ESMF_StateGet', "RouteHandle", ': getting ', trim (itemName) 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 ! Assume failure until we know we will succeed 
 if (present(rc)) rc=ESMF_RC_NOT_IMPL 
 ! TODO: do we need an empty (or invalid) item to mark failure? 
 
 exists = ESMF_StateClassFindData(state%statep, & 
 dataname=itemName, expected=.true., & 
 dataitem=dataitem, & 
 rc=localrc) 
 if (.not. exists) then 
 write(errmsg, *) "no ", "ESMF_RouteHandle", " found named: ", trim(itemName) 
 if (ESMF_LogFoundError(ESMF_RC_NOT_FOUND, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
 if (dataitem%otype .ne. ESMF_STATEITEM_RouteHandle) then 
 write(errmsg, *) trim(itemName), " found but not type ", "ESMF_RouteHandle" 
 if (ESMF_LogFoundError(ESMF_RC_ARG_INCOMP, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
#if !defined (stateversion) 
 routehandle = dataitem%datap%rp 
#else 
 routehandle%statep => dataitem%datap%rp 
 ! validate created state 
 ESMF_INIT_SET_CREATED(nestedState) 
#endif 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 
 end subroutine ESMF_StateGetRouteHandle
#define stateversion
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateGet" 
!BOPI 
 
! !IROUTINE: ESMF_StateGet - Retrieve an item from a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateGet() 
 subroutine ESMF_StateGetState(state, itemName, nestedState, & 
 keywordEnforcer, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(in) :: state 
 character (len=*), intent(in) :: itemName 
 type(ESMF_State), intent(out) :: nestedState 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: rc 
 
!EOPI 
 
 type(ESMF_StateItem), pointer :: dataitem 
 logical :: exists 
 integer :: localrc 
 character(len=len(itemName)+ESMF_MAXSTR) :: errmsg 
 
 ! print *, 'ESMF_StateGet', "State", ': getting ', trim (itemName) 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 ! Assume failure until we know we will succeed 
 if (present(rc)) rc=ESMF_RC_NOT_IMPL 
 ! TODO: do we need an empty (or invalid) item to mark failure? 
 
 exists = ESMF_StateClassFindData(state%statep, & 
 dataname=itemName, expected=.true., & 
 dataitem=dataitem, & 
 rc=localrc) 
 if (.not. exists) then 
 write(errmsg, *) "no ", "ESMF_State", " found named: ", trim(itemName) 
 if (ESMF_LogFoundError(ESMF_RC_NOT_FOUND, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
 if (dataitem%otype .ne. ESMF_STATEITEM_State) then 
 write(errmsg, *) trim(itemName), " found but not type ", "ESMF_State" 
 if (ESMF_LogFoundError(ESMF_RC_ARG_INCOMP, msg=errmsg, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 
 
#if !defined (stateversion) 
 nestedState = dataitem%datap%spp 
#else 
 nestedState%statep => dataitem%datap%spp 
 ! validate created state 
 ESMF_INIT_SET_CREATED(nestedState) 
#endif 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 
 end subroutine ESMF_StateGetState
#undef stateversion

! -------------------------- ESMF-public method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateIsCreated()"
!BOP
! !IROUTINE: ESMF_StateIsCreated - Check whether an State object has been created

! !INTERFACE:
  function ESMF_StateIsCreated(state, keywordEnforcer, rc)
! !RETURN VALUE:
    logical :: ESMF_StateIsCreated
!
! !ARGUMENTS:
    type(ESMF_State), intent(in) :: state
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(out), optional :: rc

! !DESCRIPTION:
! Return {\tt .true.} if the {\tt state} has been created. Otherwise return
! {\tt .false.}. If an error occurs, i.e. {\tt rc /= ESMF\_SUCCESS} is
! returned, the return value of the function will also be {\tt .false.}.
!
! The arguments are:
! \begin{description}
! \item[state]
! {\tt ESMF\_State} queried.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
  !-----------------------------------------------------------------------------
    ESMF_StateIsCreated = .false. ! initialize
    if (present(rc)) rc = ESMF_SUCCESS
    if (ESMF_StateGetInit(state)==ESMF_INIT_CREATED) &
      ESMF_StateIsCreated = .true.
  end function
!------------------------------------------------------------------------------


! -------------------------- ESMF-public method -----------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateLog()"
!BOP
! !IROUTINE: ESMF_StateLog - Log State information

! !INTERFACE:
  subroutine ESMF_StateLog(state, keywordEnforcer, prefix, logMsgFlag, nestedFlag, deepFlag, rc)
!
! !ARGUMENTS:
    type(ESMF_State), intent(in) :: state
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    character(len=*), intent(in), optional :: prefix
    type(ESMF_LogMsg_Flag), intent(in), optional :: logMsgFlag
    logical, intent(in), optional :: nestedFlag
    logical, intent(in), optional :: deepFlag
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Write information about {\tt state} to the ESMF default Log.
!
! The arguments are:
! \begin{description}
! \item[state]
! {\tt ESMF\_State} object logged.
! \item [{[prefix]}]
! String to prefix the log message. Default is no prefix.
! \item [{[logMsgFlag]}]
! Type of log message generated. See section \ref{const:logmsgflag} for
! a list of valid message types. Default is {\tt ESMF\_LOGMSG\_INFO}.
! \item[{[nestedFlag]}]
! When set to {\tt .false.} (default), only log information about the
! current State level.
! When set to {\tt .true.}, additionally log information for each nested
! State.
! \item[{[deepFlag]}]
! When set to {\tt .false.} (default), only log top level information for
! each item contained in the State.
! When set to {\tt .true.}, additionally log information for each item.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! local return code
    type(ESMF_LogMsg_Flag) :: logMsg
    character(len=:), allocatable :: prefixStr
    logical :: nestedLog, deepLog

    ! initialize return code; assume routine not implemented
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! optional prefix
    if (present(prefix)) then
      prefixStr = trim(prefix)
    else
      prefixStr = ""
    endif

    ! optional nestedFlag and deepFlag
    nestedLog = .false. ! default
    if (present(nestedFlag)) nestedLog = nestedFlag
    deepLog = .false. ! default
    if (present(deepFlag)) deepLog = deepFlag

    ! deal with optional logMsgFlag
    logMsg = ESMF_LOGMSG_INFO ! default
    if (present(logMsgFlag)) logMsg = logMsgFlag

    call ESMF_LogWrite(ESMF_StringConcat(trim(prefixStr), &
      "--- StateLog() start -----------------"), logMsg, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call StateLog(stateR=state, prefix=prefixStr, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_LogWrite(ESMF_StringConcat(trim(prefixStr), &
      "--- StateLog() end -------------------"), logMsg, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! return successfully
    if (present(rc)) rc = ESMF_SUCCESS

  contains

    recursive subroutine StateLog(stateR, prefix, rc)
      type(ESMF_State), intent(in) :: stateR
      character(len=*), intent(in) :: prefix
      integer, intent(out) :: rc
      ! - local variables
      integer :: localrc
      character(800) :: msgString
      character(ESMF_MAXSTR) :: name, tempString
      type(ESMF_StateIntent_Flag) :: stateIntent
      integer :: itemCount, item
      character(ESMF_MAXSTR), allocatable :: itemNameList(:)
      type(ESMF_StateItem_Flag), allocatable :: itemTypeList(:)
      type(ESMF_State) :: nestedState
      type(ESMF_Field) :: field
      type(ESMF_FieldBundle) :: fieldbundle
      type(ESMF_Array) :: array
      type(ESMF_ArrayBundle) :: arraybundle
      type(ESMF_RouteHandle) :: routehandle

      localrc = ESMF_RC_NOT_IMPL
      if (.not. ESMF_StateIsCreated(stateR)) then
        call ESMF_LogWrite(ESMF_StringConcat(prefix, &
          "State object is invalid! Not created or deleted!"), &
          logMsg, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      else
        ! query
        call ESMF_StateGet(stateR, name=name, stateIntent=stateIntent, &
          itemCount=itemCount, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        ! <intent>
        if (stateIntent==ESMF_STATEINTENT_IMPORT) then
          tempString = "ESMF_STATEINTENT_IMPORT"
        else if (stateIntent==ESMF_STATEINTENT_EXPORT) then
          tempString = "ESMF_STATEINTENT_EXPORT"
        else if (stateIntent==ESMF_STATEINTENT_INTERNAL) then
          tempString = "ESMF_STATEINTENT_INTERNAL"
        else if (stateIntent==ESMF_STATEINTENT_UNSPECIFIED) then
          tempString = "ESMF_STATEINTENT_UNSPECIFIED"
        else if (stateIntent==ESMF_STATEINTENT_INVALID) then
          tempString = "ESMF_STATEINTENT_INVALID"
        else
          tempString = "Out or range STATEINTENT!!!"
        endif

        write (msgString,'(A,A,A,A,A,A,A,I4,A)') &
          prefix, "<name: ", trim(name), "> <intent: ", trim(tempString), ">", &
          " <itemCount: ", itemCount, ">"
        call ESMF_LogWrite(trim(msgString), logMsg, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        if (itemCount > 0) then
          allocate(itemNameList(itemCount))
          allocate(itemTypeList(itemCount))
          call ESMF_StateGet(stateR, itemNameList=itemNameList, &
            itemtypeList=itemtypeList, rc=localrc)
          if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

          do item=1, itemCount
            ! basic item information
            write (msgString,'(A,A,I4.4,A,A,A,A,A)') &
              prefix, "+-<item: ", item, &
              "> <itemType: ", ESMF_StateItemString(itemtypeList(item)), &
              "> <itemName: ", trim(itemNameList(item)), ">"
            call ESMF_LogWrite(trim(msgString), logMsg, rc=localrc)
            if (ESMF_LogFoundError(localrc, &
              ESMF_ERR_PASSTHRU, &
              ESMF_CONTEXT, rcToReturn=rc)) return
            ! deep item logging, possibly recursion
            if ((itemtypeList(item) == ESMF_STATEITEM_STATE) .and. &
              nestedLog) then
              ! recursion into nested state
              call ESMF_StateGet(stateR, itemName=itemNameList(item), &
                nestedState=nestedState, rc=localrc)
              if (ESMF_LogFoundError(localrc, &
                ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return
              call StateLog(stateR=nestedState, &
                prefix=ESMF_StringConcat(prefix, "! "), rc=localrc)
              if (ESMF_LogFoundError(localrc, &
                ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return
            else if (deepLog) then
              ! deep logging of all other item types
              ! TODO: activate the Log() method calls below when implemented!
              if (itemtypeList(item) == ESMF_STATEITEM_FIELD) then
                call ESMF_StateGet(stateR, itemName=itemNameList(item), &
                  field=field, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
                call ESMF_FieldLog(field, &
                  prefix=ESMF_StringConcat(prefix, "! "), &
                  logMsgFlag=logMsg, deepFlag=deepLog, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
              else if (itemtypeList(item) == ESMF_STATEITEM_FIELDBUNDLE) then
                call ESMF_StateGet(stateR, itemName=itemNameList(item), &
                  fieldbundle=fieldbundle, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
! call ESMF_FieldBundleLog(fieldbundle, prefix=prefix
! logMsgFlag=logMsg, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
              else if (itemtypeList(item) == ESMF_STATEITEM_ARRAY) then
                call ESMF_StateGet(stateR, itemName=itemNameList(item), &
                  array=array, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
                call ESMF_ArrayLog(array, &
                  prefix=ESMF_StringConcat(prefix, "! "), &
                  logMsgFlag=logMsg, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
              else if (itemtypeList(item) == ESMF_STATEITEM_ARRAYBUNDLE) then
                call ESMF_StateGet(stateR, itemName=itemNameList(item), &
                  arraybundle=arraybundle, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
                call ESMF_ArrayBundleLog(arraybundle, &
                  prefix=ESMF_StringConcat(prefix, "! "), &
                  logMsgFlag=logMsg, deepFlag=deepLog, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
              else if (itemtypeList(item) == ESMF_STATEITEM_ROUTEHANDLE) then
                call ESMF_StateGet(stateR, itemName=itemNameList(item), &
                  routehandle=routehandle, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
! call ESMF_RouteHandleLog(routehandle, &
! prefix=ESMF_StringConcat(prefix, "! "), &
! logMsgFlag=logMsg, rc=localrc)
                if (ESMF_LogFoundError(localrc, &
                  ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
              endif
            endif
          enddo

          deallocate(itemNameList)
          deallocate(itemTypeList)
        endif

      endif
    end subroutine

  end subroutine ESMF_StateLog
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StatePrint"
!BOP
! !IROUTINE: ESMF_StatePrint - Print State information
!
! !INTERFACE:
      subroutine ESMF_StatePrint(state, options, nestedFlag, rc)
!
! !ARGUMENTS:
      type(ESMF_State), intent(in) :: state
      character(len=*), intent(in), optional :: options
      logical, intent(in), optional :: nestedFlag
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Prints information about the {\tt state} to {\tt stdout}.
!
! The arguments are:
! \begin{description}
! \item[state]
! The {\tt ESMF\_State} to print.
! \item[{[options]}]
! Print options:
! " ", or "brief" - print names and types of the objects within the state (default),
! "long" - print additional information, such as proxy flags
! \item[{[nestedFlag]}]
! When set to {\tt .false.}, prints information about the current
! State level only (default),
! When set to {\tt .true.}, additionally prints information from
! nested States
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!

!EOP

       integer :: localrc
       character (len=8) :: localopts
       logical :: localnestedflag ! Print nested states flag
       logical :: longflag ! Extended output
       logical :: debugflag ! Debug level output
       type(ESMF_VM) :: localvm
       integer :: mypet
       character(8) :: mypet_string

       ! Initialize return code; assume failure until success is certain
       if (present(rc)) rc = ESMF_RC_NOT_IMPL
       localrc = ESMF_RC_NOT_IMPL

       ! check input variables
       ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc)

       ! Validate options arg

       localnestedflag = .false.
       if (present (nestedFlag)) then
         localnestedflag = nestedFlag
       end if

       localopts = "brief"
       if (present (options)) then
         if (options /= " ") &
           localopts = adjustl (options)
       end if

       debugflag = .false.
       longflag = .false.
       localopts = ESMF_UtilStringLowerCase (localopts)
       select case (localopts)
       case ("brief")
       case ("debug")
         debugflag = .true.
         longflag = .true.
       case ("long")
         longflag = .true.
       case default
         write (ESMF_UtilIOStderr,*) ESMF_METHOD, ": Illegal options argument: ", &
             trim (localopts)
         if (ESMF_LogFoundError(ESMF_RC_ARG_BAD, &
             msg=ESMF_StringConcat ("Illegal options argument: ", trim (localopts)), &
             ESMF_CONTEXT, rcToReturn=rc)) return
       end select

       if (.not.associated(state%statep)) then
         write (ESMF_UtilIOStdout,*) "Uninitialized or already destroyed State"
         if (present (rc)) rc = ESMF_SUCCESS
         return
       end if
       if (state%statep%st .eq. ESMF_STATEINTENT_INVALID) then
         write (ESMF_UtilIOStdout,*) "Uninitialized or already destroyed State"
         if (present (rc)) rc = ESMF_SUCCESS
         return
       end if

       call ESMF_VMGetCurrent(vm=localvm, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return

       call ESMF_VMGet (localvm, localPet=mypet, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return

       write (mypet_string,'(i8)') mypet
       mypet_string = adjustl (mypet_string)
       write (ESMF_UtilIOStdout,*) &
           ESMF_METHOD, ": (pet ", trim (mypet_string), "):"

       call statePrintWorker (state%statep, level=0, rc1=rc)

   contains

     recursive subroutine statePrintWorker (sp1, level, rc1)
       type(ESMF_StateClass), pointer :: sp1
       integer, intent(in) :: level
       integer, intent(out), optional :: rc1

       type(ESMF_StateItemWrap), pointer :: siwrap(:)
       integer :: ptrcnt
       type(ESMF_StateItem) , pointer :: sip

       character(len=2*level+1) :: nestr
       character(len=ESMF_MAXSTR) :: name
       character(len=ESMF_MAXSTR) :: msgbuf
       character(len=ESMF_MAXSTR*2) :: outbuf

       integer :: i1
       integer :: localrc1
       integer :: memstat1

       if (present (rc1)) rc1 = ESMF_FAILURE

       nestr = repeat("->", level)

       call ESMF_GetName (sp1%base, name=name, rc=localrc1)
       if (ESMF_LogFoundError(localrc1, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc1)) return
       write (ESMF_UtilIOStdout,*) nestr, "State name: ", trim(name)

       select case (sp1%st%state)
       case (ESMF_STATEINTENT_IMPORT%state)
         msgbuf = "Import State"
       case (ESMF_STATEINTENT_EXPORT%state)
         msgbuf = "Export State"
       case (ESMF_STATEINTENT_INTERNAL%state)
         msgbuf = "Internal State"
       case (ESMF_STATEINTENT_UNSPECIFIED%state)
         msgbuf = "Unspecified intent direction"
       case (ESMF_STATEINTENT_INVALID%state)
         msgbuf = "Uninitialized or already destroyed State"
         if (present (rc1)) rc1 = ESMF_SUCCESS
         return
       case default
         msgbuf = "error: unknown state"
         if (present (rc1)) rc1 = ESMF_SUCCESS
         return
       end select

       siwrap => null ()
       call ESMF_ContainerGet (sp1%stateContainer, &
           itemList=siwrap, rc=localrc1)
       if (ESMF_LogFoundError(localrc1, ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc1)) return
       if (associated (siwrap)) &
           deallocate (siwrap, stat=memstat1)

       siwrap => null ()
       call ESMF_ContainerGet (sp1%stateContainer, &
           itemList=siwrap, rc=localrc1)
       if (ESMF_LogFoundError(localrc1, ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc1)) return
       ptrcnt = size (siwrap)

       write (ESMF_UtilIOStdout,'(1x,4a,i0)') nestr, &
           "   status: ", trim(msgbuf), &
           ", object count: ", ptrcnt

       ! Prints Attributes associated with the State
       call ESMF_UtilIOUnitFlush (unit=ESMF_UtilIOstdout, rc=localrc1)
       if (ESMF_LogFoundError(localrc1, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc1)) return
       call c_ESMC_BasePrint(sp1%base, level, "brief", ESMF_FALSE, "", &
                             ESMF_FALSE, localrc1)
       if (ESMF_LogFoundError(localrc1, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc1)) return

       if (debugflag) then
         write (ESMF_UtilIOStdout,*) 'ContainerPrint:'
         call ESMF_ContainerPrint (sp1%stateContainer, rc=localrc1)
         if (ESMF_LogFoundError(localrc1, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc1)) return
         call c_ESMC_BasePrint(sp1%base, level, "debug", ESMF_FALSE, "", &
                               ESMF_FALSE, localrc1)
         if (ESMF_LogFoundError(localrc1, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc1)) return
       end if

       do, i1 = 1, ptrcnt
         sip => siwrap(i1)%si
         write (outbuf,'(1x,2a,i0,a)') nestr, " object: ", i1, ", "
         call ESMF_StateItemPrint (sip, &
             header=trim (outbuf), &
             prefixstr=nestr, &
             longflag=longflag, debugflag=debugflag)

         if (localnestedflag .and. sip%otype%ot == ESMF_STATEITEM_STATE%ot) then
           call statePrintWorker (sip%datap%spp, level=level+1, rc1=localrc1)
           if (ESMF_LogFoundError(localrc1, &
               ESMF_ERR_PASSTHRU, &
               ESMF_CONTEXT, rcToReturn=rc1)) then
             if (associated (siwrap)) deallocate (siwrap, stat=memstat1)
             return
           end if
         end if
       end do

       if (associated (siwrap)) &
           deallocate (siwrap, stat=memstat1)
       if (ESMF_LogFoundDeallocError(memstat1, ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc1)) return

       if (present (rc1)) rc1 = ESMF_SUCCESS

     end subroutine statePrintWorker

   end subroutine ESMF_StatePrint

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateRead"
!BOPI
! !IROUTINE: ESMF_StateRead -- Read data items from a file into a State
!
! !INTERFACE:
      subroutine ESMF_StateRead(state, fileName, rc)
!
! !ARGUMENTS:
      type(ESMF_State), intent(inout) :: state
      character (len=*), intent(in) :: fileName
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Currently limited to read in all Arrays from a NetCDF file and add them
! to a State object. Future releases will enable more items of a State
! to be read from a file of various formats.
!
! Only PET 0 reads the file; the States in other PETs remain empty.
! Currently, the data is not decomposed or distributed; each PET
! has only 1 DE and only PET 0 contains data after reading the file.
! Future versions of ESMF will support data decomposition and distribution
! upon reading a file. See Section~\ref{example:StateRdWr} for
! an example.
!
! Note that the third party NetCDF library must be installed. For more
! details, see the "ESMF Users Guide",
! "Building and Installing the ESMF, Third Party Libraries, NetCDF" and
! the website http:
!
! The arguments are:
! \begin{description}
! \item[state]
! The {\tt ESMF\_State} to add items read from file. Currently only
! Arrays are supported.
! \item[fileName]
! File to be read.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! Equals {\tt ESMF\_RC\_LIB\_NOT\_PRESENT} if the NetCDF library is
! not present.
! \end{description}
!
!EOPI
! TODO: use item flag ESMF_STATEITEM_ARRAY<BUNDLE>

        integer :: localrc

        ! Initialize return code; assume failure until success is certain
        if (present(rc)) rc = ESMF_RC_NOT_IMPL
        localrc = ESMF_RC_NOT_IMPL

        ! check input variables
        ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc)

        if (fileName == ' ') then
          if (ESMF_LogFoundError (ESMF_RC_ARG_VALUE, msg='File name required', &
              ESMF_CONTEXT, rcToReturn=rc)) return
        end if

        ! invoke C to C++ entry point
        call c_ESMC_StateRead(state, state%statep%base, trim (fileName), localrc)
        if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        if (present(rc)) rc = ESMF_SUCCESS
        end subroutine ESMF_StateRead


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateReadRestart"
!BOPI
! !IROUTINE: ESMF_StateReadRestart -- ReadRestart the internal data from a State
!
! !INTERFACE:
      function ESMF_StateReadRestart(name, rc)
!
! !RETURN VALUE:
      type(ESMF_State) :: ESMF_StateReadRestart
!
!
! !ARGUMENTS:
      character (len = *), intent(in) :: name
      integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Used to reinitialize all data associated with an
! {\tt ESMF\_State} from the last call to WriteRestart.
!
! The arguments are:
! \begin{description}
! \item[name]
! Name of {\tt ESMF\_State} to reinitialize.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!
!EOPI

!
! TODO: code goes here
!
        type (ESMF_State) :: a

! this is just to shut the compiler up
        type (ESMF_StateClass), target :: b
        a%statep => b
        nullify(a%statep)

        ESMF_StateReadRestart = a
        if (present(rc)) rc = ESMF_RC_NOT_IMPL

        end function ESMF_StateReadRestart

end module ESMF_StateAPIMod
