/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.encrypted;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.util.Objects;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.repositories.encrypted.BufferOnMarkInputStream;
import org.elasticsearch.repositories.encrypted.ChainingInputStream;
import org.elasticsearch.repositories.encrypted.CountingInputStream;
import org.elasticsearch.repositories.encrypted.PrefixInputStream;

public final class EncryptionPacketsInputStream
extends ChainingInputStream {
    private final SecretKey secretKey;
    private final int packetLength;
    private final ByteBuffer packetIv;
    private final int encryptedPacketLength;
    final InputStream source;
    long counter;
    Long markCounter;
    int markSourceOnNextPacket;

    public static long getEncryptionLength(long plaintextLength, int packetLength) {
        return plaintextLength + (plaintextLength / (long)packetLength + 1L) * 28L;
    }

    public EncryptionPacketsInputStream(InputStream source, SecretKey secretKey, int nonce, int packetLength) {
        this.source = Objects.requireNonNull(source);
        this.secretKey = Objects.requireNonNull(secretKey);
        if (packetLength <= 0 || packetLength >= 0x800000) {
            throw new IllegalArgumentException("Invalid packet length [" + packetLength + "]");
        }
        this.packetLength = packetLength;
        this.packetIv = ByteBuffer.allocate(12).order(ByteOrder.LITTLE_ENDIAN);
        this.packetIv.putInt(0, nonce);
        this.encryptedPacketLength = packetLength + 12 + 16;
        this.counter = Long.MIN_VALUE;
        this.markCounter = null;
        this.markSourceOnNextPacket = -1;
    }

    @Override
    InputStream nextComponent(InputStream currentComponentIn) throws IOException {
        if (currentComponentIn != null && ((CountingInputStream)currentComponentIn).getCount() < (long)this.encryptedPacketLength) {
            return null;
        }
        if (this.markSourceOnNextPacket != -1) {
            this.source.mark(this.markSourceOnNextPacket);
            this.markSourceOnNextPacket = -1;
        }
        InputStream encryptionInputStream = new PrefixInputStream(this.source, this.packetLength, false);
        this.packetIv.putLong(4, this.counter++);
        if (this.counter == Long.MIN_VALUE) {
            throw new IOException("Maximum packet count limit exceeded");
        }
        Cipher packetCipher = EncryptionPacketsInputStream.getPacketEncryptionCipher(this.secretKey, this.packetIv.array());
        encryptionInputStream = new CipherInputStream(encryptionInputStream, packetCipher);
        encryptionInputStream = new SequenceInputStream(new ByteArrayInputStream(this.packetIv.array()), encryptionInputStream);
        encryptionInputStream = new BufferOnMarkInputStream(encryptionInputStream, this.encryptedPacketLength);
        return new CountingInputStream(encryptionInputStream, false);
    }

    @Override
    public boolean markSupported() {
        return this.source.markSupported();
    }

    @Override
    public void mark(int readlimit) {
        if (this.markSupported()) {
            if (readlimit <= 0) {
                throw new IllegalArgumentException("Mark readlimit must be a positive integer");
            }
            super.mark(this.encryptedPacketLength);
            this.markCounter = this.counter;
            this.markSourceOnNextPacket = readlimit;
        }
    }

    @Override
    public void reset() throws IOException {
        if (!this.markSupported()) {
            throw new IOException("Mark/reset not supported");
        }
        if (this.markCounter == null) {
            throw new IOException("Mark no set");
        }
        super.reset();
        this.counter = this.markCounter;
        if (this.markSourceOnNextPacket == -1) {
            this.source.reset();
        }
    }

    @Override
    public void close() throws IOException {
        IOUtils.close((Closeable[])new Closeable[]{() -> super.close(), this.source});
    }

    private static Cipher getPacketEncryptionCipher(SecretKey secretKey, byte[] packetIv) throws IOException {
        GCMParameterSpec gcmParameterSpec = new GCMParameterSpec(128, packetIv);
        try {
            Cipher packetCipher = Cipher.getInstance("AES/GCM/NoPadding");
            packetCipher.init(1, (Key)secretKey, gcmParameterSpec);
            return packetCipher;
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new IOException(e);
        }
    }
}

