/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedKeyManager;
import javax.net.ssl.X509ExtendedTrustManager;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.env.Environment;
import org.elasticsearch.xpack.core.ssl.KeyConfig;
import org.elasticsearch.xpack.core.ssl.PEMKeyConfig;
import org.elasticsearch.xpack.core.ssl.PemUtils;
import org.elasticsearch.xpack.core.ssl.SSLConfigurationSettings;
import org.elasticsearch.xpack.core.ssl.StoreKeyConfig;
import org.elasticsearch.xpack.core.ssl.X509KeyPairSettings;

public class CertParsingUtils {
    private CertParsingUtils() {
        throw new IllegalStateException("Utility class should not be instantiated");
    }

    @SuppressForbidden(reason="we don't have the environment to resolve files from when running in a transport client")
    static Path resolvePath(String path, @Nullable Environment environment) {
        if (environment != null) {
            return environment.configFile().resolve(path);
        }
        return PathUtils.get(path, new String[0]).normalize();
    }

    static List<Path> resolvePaths(List<String> certPaths, @Nullable Environment environment) {
        return certPaths.stream().map(p -> CertParsingUtils.resolvePath(p, environment)).collect(Collectors.toList());
    }

    public static KeyStore readKeyStore(Path path, String type, char[] password) throws IOException, KeyStoreException, CertificateException, NoSuchAlgorithmException {
        try (InputStream in = Files.newInputStream(path, new OpenOption[0]);){
            KeyStore store = KeyStore.getInstance(type);
            assert (password != null);
            store.load(in, password);
            KeyStore keyStore = store;
            return keyStore;
        }
    }

    public static Certificate[] readCertificates(List<String> certPaths, @Nullable Environment environment) throws CertificateException, IOException {
        List<Path> resolvedPaths = CertParsingUtils.resolvePaths(certPaths, environment);
        return CertParsingUtils.readCertificates(resolvedPaths);
    }

    public static Certificate[] readCertificates(List<Path> certPaths) throws CertificateException, IOException {
        ArrayList<? extends Certificate> certificates = new ArrayList<Certificate>();
        CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
        for (Path path : certPaths) {
            InputStream input = Files.newInputStream(path, new OpenOption[0]);
            try {
                certificates.addAll(certFactory.generateCertificates(input));
                if (!certificates.isEmpty()) continue;
                throw new CertificateException("failed to parse any certificates from [" + path.toAbsolutePath() + "]");
            }
            finally {
                if (input == null) continue;
                input.close();
            }
        }
        return certificates.toArray(new Certificate[0]);
    }

    public static X509Certificate[] readX509Certificates(List<Path> certPaths) throws CertificateException, IOException {
        ArrayList<? extends Certificate> certificates = new ArrayList<Certificate>();
        CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
        for (Path path : certPaths) {
            InputStream input = Files.newInputStream(path, new OpenOption[0]);
            try {
                certificates.addAll(certFactory.generateCertificates(input));
            }
            finally {
                if (input == null) continue;
                input.close();
            }
        }
        return certificates.toArray(new X509Certificate[0]);
    }

    public static List<Certificate> readCertificates(InputStream input) throws CertificateException, IOException {
        CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
        Collection<? extends Certificate> certificates = certFactory.generateCertificates(input);
        return new ArrayList<Certificate>(certificates);
    }

    public static Map<Certificate, Key> readPkcs12KeyPairs(Path path, char[] password, Function<String, char[]> keyPassword) throws CertificateException, NoSuchAlgorithmException, KeyStoreException, IOException, UnrecoverableKeyException {
        return CertParsingUtils.readKeyPairsFromKeystore(path, "PKCS12", password, keyPassword);
    }

    public static Map<Certificate, Key> readKeyPairsFromKeystore(Path path, String storeType, char[] password, Function<String, char[]> keyPassword) throws IOException, KeyStoreException, CertificateException, NoSuchAlgorithmException, UnrecoverableKeyException {
        KeyStore store = CertParsingUtils.readKeyStore(path, storeType, password);
        return CertParsingUtils.readKeyPairsFromKeystore(store, keyPassword);
    }

    static Map<Certificate, Key> readKeyPairsFromKeystore(KeyStore store, Function<String, char[]> keyPassword) throws KeyStoreException, NoSuchAlgorithmException, UnrecoverableKeyException {
        Enumeration<String> enumeration = store.aliases();
        HashMap<Certificate, Key> map = new HashMap<Certificate, Key>(store.size());
        while (enumeration.hasMoreElements()) {
            String alias = enumeration.nextElement();
            if (!store.isKeyEntry(alias)) continue;
            char[] pass = keyPassword.apply(alias);
            map.put(store.getCertificate(alias), store.getKey(alias, pass));
        }
        return map;
    }

    public static KeyStore getKeyStoreFromPEM(Path certificatePath, Path keyPath, char[] keyPassword) throws IOException, CertificateException, KeyStoreException, NoSuchAlgorithmException {
        PrivateKey key = PemUtils.readPrivateKey(keyPath, () -> keyPassword);
        Certificate[] certificates = CertParsingUtils.readCertificates(Collections.singletonList(certificatePath));
        return CertParsingUtils.getKeyStore(certificates, key, keyPassword);
    }

    public static X509ExtendedKeyManager keyManager(Certificate[] certificateChain, PrivateKey privateKey, char[] password) throws NoSuchAlgorithmException, UnrecoverableKeyException, KeyStoreException, IOException, CertificateException {
        KeyStore keyStore = CertParsingUtils.getKeyStore(certificateChain, privateKey, password);
        return CertParsingUtils.keyManager(keyStore, password, KeyManagerFactory.getDefaultAlgorithm());
    }

    private static KeyStore getKeyStore(Certificate[] certificateChain, PrivateKey privateKey, char[] password) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException {
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        keyStore.load(null, null);
        keyStore.setKeyEntry("key", privateKey, password, certificateChain);
        return keyStore;
    }

    public static X509ExtendedKeyManager keyManager(KeyStore keyStore, char[] password, String algorithm) throws NoSuchAlgorithmException, UnrecoverableKeyException, KeyStoreException {
        KeyManager[] keyManagers;
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(algorithm);
        kmf.init(keyStore, password);
        for (KeyManager keyManager : keyManagers = kmf.getKeyManagers()) {
            if (!(keyManager instanceof X509ExtendedKeyManager)) continue;
            return (X509ExtendedKeyManager)keyManager;
        }
        throw new IllegalStateException("failed to find a X509ExtendedKeyManager");
    }

    public static X509ExtendedKeyManager getKeyManager(X509KeyPairSettings keyPair, Settings settings, @Nullable String trustStoreAlgorithm, Environment environment) {
        KeyConfig keyConfig;
        if (trustStoreAlgorithm == null) {
            trustStoreAlgorithm = TrustManagerFactory.getDefaultAlgorithm();
        }
        if ((keyConfig = CertParsingUtils.createKeyConfig(keyPair, settings, trustStoreAlgorithm)) == null) {
            return null;
        }
        return keyConfig.createKeyManager(environment);
    }

    static KeyConfig createKeyConfig(X509KeyPairSettings keyPair, Settings settings, String trustStoreAlgorithm) {
        String keyPath = keyPair.keyPath.get(settings).orElse(null);
        String keyStorePath = keyPair.keystorePath.get(settings).orElse(null);
        String keyStoreType = SSLConfigurationSettings.getKeyStoreType(keyPair.keystoreType, settings, keyStorePath);
        if (keyPath != null && keyStorePath != null) {
            throw new IllegalArgumentException("you cannot specify a keystore and key file");
        }
        if (keyPath != null) {
            SecureString keyPassword = keyPair.keyPassword.get(settings);
            String certPath = keyPair.certificatePath.get(settings).orElse(null);
            if (certPath == null) {
                throw new IllegalArgumentException("you must specify the certificates [" + keyPair.certificatePath.getKey() + "] to use with the key [" + keyPair.keyPath.getKey() + "]");
            }
            return new PEMKeyConfig(keyPath, keyPassword, certPath);
        }
        if (keyStorePath != null || keyStoreType.equalsIgnoreCase("pkcs11")) {
            SecureString keyStorePassword = keyPair.keystorePassword.get(settings);
            String keyStoreAlgorithm = keyPair.keystoreAlgorithm.get(settings);
            SecureString keyStoreKeyPassword = keyPair.keystoreKeyPassword.get(settings);
            if (keyStoreKeyPassword.length() == 0) {
                keyStoreKeyPassword = keyStorePassword;
            }
            return new StoreKeyConfig(keyStorePath, keyStoreType, keyStorePassword, keyStoreKeyPassword, keyStoreAlgorithm, trustStoreAlgorithm);
        }
        return null;
    }

    public static X509ExtendedTrustManager trustManager(Certificate[] certificates) throws NoSuchAlgorithmException, KeyStoreException, IOException, CertificateException {
        KeyStore store = CertParsingUtils.trustStore(certificates);
        return CertParsingUtils.trustManager(store, TrustManagerFactory.getDefaultAlgorithm());
    }

    public static KeyStore trustStore(Certificate[] certificates) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException {
        assert (certificates != null) : "Cannot create trust store with null certificates";
        KeyStore store = KeyStore.getInstance(KeyStore.getDefaultType());
        store.load(null, null);
        int counter = 0;
        for (Certificate certificate : certificates) {
            store.setCertificateEntry("cert" + counter, certificate);
            ++counter;
        }
        return store;
    }

    public static X509ExtendedTrustManager trustManager(String trustStorePath, String trustStoreType, char[] trustStorePassword, String trustStoreAlgorithm, @Nullable Environment env) throws NoSuchAlgorithmException, KeyStoreException, IOException, CertificateException {
        KeyStore trustStore = CertParsingUtils.readKeyStore(CertParsingUtils.resolvePath(trustStorePath, env), trustStoreType, trustStorePassword);
        return CertParsingUtils.trustManager(trustStore, trustStoreAlgorithm);
    }

    public static X509ExtendedTrustManager trustManager(KeyStore keyStore, String algorithm) throws NoSuchAlgorithmException, KeyStoreException {
        TrustManager[] trustManagers;
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(algorithm);
        tmf.init(keyStore);
        for (TrustManager trustManager : trustManagers = tmf.getTrustManagers()) {
            if (!(trustManager instanceof X509ExtendedTrustManager)) continue;
            return (X509ExtendedTrustManager)trustManager;
        }
        throw new IllegalStateException("failed to find a X509ExtendedTrustManager");
    }

    public static boolean isOrderedCertificateChain(List<X509Certificate> chain) {
        for (int i = 1; i < chain.size(); ++i) {
            X509Certificate cert = chain.get(i - 1);
            X509Certificate issuer = chain.get(i);
            if (cert.getIssuerX500Principal().equals(issuer.getSubjectX500Principal())) continue;
            return false;
        }
        return true;
    }
}

