use rustc::{ty, mir};

use super::{TlsKey, TlsEntry, EvalResult, EvalErrorKind, Scalar, Memory, Evaluator,
            Place, StackPopCleanup, EvalContext};

pub trait MemoryExt<'tcx> {
    fn create_tls_key(&mut self, dtor: Option<ty::Instance<'tcx>>) -> TlsKey;
    fn delete_tls_key(&mut self, key: TlsKey) -> EvalResult<'tcx>;
    fn load_tls(&mut self, key: TlsKey) -> EvalResult<'tcx, Scalar>;
    fn store_tls(&mut self, key: TlsKey, new_data: Scalar) -> EvalResult<'tcx>;
    fn fetch_tls_dtor(
        &mut self,
        key: Option<TlsKey>,
    ) -> Option<(ty::Instance<'tcx>, Scalar, TlsKey)>;
}

pub trait EvalContextExt<'tcx> {
    fn run_tls_dtors(&mut self) -> EvalResult<'tcx>;
}

impl<'a, 'mir, 'tcx: 'mir + 'a> MemoryExt<'tcx> for Memory<'a, 'mir, 'tcx, Evaluator<'tcx>> {
    fn create_tls_key(&mut self, dtor: Option<ty::Instance<'tcx>>) -> TlsKey {
        let new_key = self.data.next_thread_local;
        self.data.next_thread_local += 1;
        self.data.thread_local.insert(
            new_key,
            TlsEntry {
                data: Scalar::ptr_null(*self.tcx).into(),
                dtor,
            },
        );
        trace!("New TLS key allocated: {} with dtor {:?}", new_key, dtor);
        new_key
    }

    fn delete_tls_key(&mut self, key: TlsKey) -> EvalResult<'tcx> {
        match self.data.thread_local.remove(&key) {
            Some(_) => {
                trace!("TLS key {} removed", key);
                Ok(())
            }
            None => err!(TlsOutOfBounds),
        }
    }

    fn load_tls(&mut self, key: TlsKey) -> EvalResult<'tcx, Scalar> {
        match self.data.thread_local.get(&key) {
            Some(&TlsEntry { data, .. }) => {
                trace!("TLS key {} loaded: {:?}", key, data);
                Ok(data)
            }
            None => err!(TlsOutOfBounds),
        }
    }

    fn store_tls(&mut self, key: TlsKey, new_data: Scalar) -> EvalResult<'tcx> {
        match self.data.thread_local.get_mut(&key) {
            Some(&mut TlsEntry { ref mut data, .. }) => {
                trace!("TLS key {} stored: {:?}", key, new_data);
                *data = new_data;
                Ok(())
            }
            None => err!(TlsOutOfBounds),
        }
    }

    /// Returns a dtor, its argument and its index, if one is supposed to run
    ///
    /// An optional destructor function may be associated with each key value.
    /// At thread exit, if a key value has a non-NULL destructor pointer,
    /// and the thread has a non-NULL value associated with that key,
    /// the value of the key is set to NULL, and then the function pointed
    /// to is called with the previously associated value as its sole argument.
    /// The order of destructor calls is unspecified if more than one destructor
    /// exists for a thread when it exits.
    ///
    /// If, after all the destructors have been called for all non-NULL values
    /// with associated destructors, there are still some non-NULL values with
    /// associated destructors, then the process is repeated.
    /// If, after at least {PTHREAD_DESTRUCTOR_ITERATIONS} iterations of destructor
    /// calls for outstanding non-NULL values, there are still some non-NULL values
    /// with associated destructors, implementations may stop calling destructors,
    /// or they may continue calling destructors until no non-NULL values with
    /// associated destructors exist, even though this might result in an infinite loop.
    fn fetch_tls_dtor(
        &mut self,
        key: Option<TlsKey>,
    ) -> Option<(ty::Instance<'tcx>, Scalar, TlsKey)> {
        use std::collections::Bound::*;

        let thread_local = &mut self.data.thread_local;
        let start = match key {
            Some(key) => Excluded(key),
            None => Unbounded,
        };
        for (&key, &mut TlsEntry { ref mut data, dtor }) in
            thread_local.range_mut((start, Unbounded))
        {
            if !data.is_null() {
                if let Some(dtor) = dtor {
                    let ret = Some((dtor, *data, key));
                    *data = Scalar::ptr_null(*self.tcx);
                    return ret;
                }
            }
        }
        None
    }
}

impl<'a, 'mir, 'tcx: 'mir + 'a> EvalContextExt<'tcx> for EvalContext<'a, 'mir, 'tcx, Evaluator<'tcx>> {
    fn run_tls_dtors(&mut self) -> EvalResult<'tcx> {
        let mut dtor = self.memory.fetch_tls_dtor(None);
        // FIXME: replace loop by some structure that works with stepping
        while let Some((instance, ptr, key)) = dtor {
            trace!("Running TLS dtor {:?} on {:?}", instance, ptr);
            // TODO: Potentially, this has to support all the other possible instances?
            // See eval_fn_call in interpret/terminator/mod.rs
            let mir = self.load_mir(instance.def)?;
            let ret = Place::null(&self);
            self.push_stack_frame(
                instance,
                mir.span,
                mir,
                ret,
                StackPopCleanup::None { cleanup: true },
            )?;
            let arg_local = self.frame().mir.args_iter().next().ok_or_else(
                || EvalErrorKind::AbiViolation("TLS dtor does not take enough arguments.".to_owned()),
            )?;
            let dest = self.eval_place(&mir::Place::Local(arg_local))?;
            self.write_scalar(ptr, dest)?;

            // step until out of stackframes
            self.run()?;

            dtor = match self.memory.fetch_tls_dtor(Some(key)) {
                dtor @ Some(_) => dtor,
                None => self.memory.fetch_tls_dtor(None),
            };
        }
        // FIXME: On a windows target, call `unsafe extern "system" fn on_tls_callback`.
        Ok(())
    }
}
