class TestSQLExpressionFactory < Test::Unit::TestCase
	include TapKit

	def setup
		@model   = Model.new MODEL1
		@adapter = Adapter.new_with_name('DBI', @model)
		@factory = @adapter.expression_factory
		@entity  = @model.entity 'cd'
	end

	def test_create
		expr = @factory.create @entity
		assert_kind_of(SQLExpression, expr)
	end

	def test_expression_class
		assert_equal(DBIExpression, @factory.expression_class)
	end

	def test_expression
		expr = @factory.expression 'test'
		assert_equal('test', expr.statement)
	end

	def test_delete_statement
		qualifier = Qualifier.new_with_format "title == 'test'"
		expr      = @factory.delete_statement(qualifier, @entity)

		assert_match(/DELETE FROM/, expr.statement)
	end

	def test_insert_statement
		row  = {}
		expr = @factory.insert_statement(row, @entity)

		assert_match(/INSERT INTO/, expr.statement)
	end

	def test_select_statement
		attributes = @entity.attributes
		lock       = false
		qualifier  = Qualifier.new_with_format "title like '%'"
		fetch_spec = FetchSpec.new(@entity.name, qualifier)
		expr       = @factory.select_statement(attributes, lock, fetch_spec, @entity)

		assert_match(/SELECT/, expr.statement)
	end

	def test_update_statement
		row       = {}
		qualifier = Qualifier.new_with_format "title == 'test'"
		expr      = @factory.update_statement(row, qualifier, @entity)

		assert_match(/UPDATE/, expr.statement)
	end

	def test_aggregate_statement
		qualifier = Qualifier.format "artist like 'The Beatles'"
		agg_spec = AggregateSpec.new('cd', qualifier)
		agg_spec.add('CDs', 'artist', AggregateSpec::COUNT)
		expected = "SELECT COUNT(t0.artist) CDs FROM cd t0 WHERE t0.artist LIKE ?"

		expr = @factory.aggregate_statement(@entity, agg_spec)
		assert_equal(expected, expr.statement)
	end

end


class TestSQLExpression < Test::Unit::TestCase
	include TapKit

	def setup
		@app     = Application.new MODELS
		@model   = @app.model MODEL2

		@adapter = Adapter.new_with_name('DBI', @model)
		@factory = @adapter.expression_factory
		@entity  = @model.entity 'Book'
		@attr    = @entity.attribute 'title'
		@klass   = @factory.expression_class
		@expr    = @klass.new @entity
	end


	# bind variable

	def test_bind_variable
		value = 'test'

		# create
		bind = @expr.bind_variable(@attr, value)
		assert_equal(@attr.name, bind[SQLExpression::NAME_KEY])
		assert_equal('?', bind[SQLExpression::PLACEHOLDER_KEY])
		assert_equal(@attr, bind[SQLExpression::ATTRIBUTE_KEY])
		assert_equal(value, bind[SQLExpression::VALUE_KEY])

		# add
		@expr.add_bind_variable bind
		assert_equal([bind], @expr.bind_variables)
	end

	def test_value_for_bind_variable
		tests = [
			{:type=>'char', :value=>'string', :expected=>'string'},
			{:type=>'int', :value=>1, :expected=>1},
			{:type=>'char', :value=>nil, :expected=>nil},
			{:type=>'date', :value=>Date.new, :expected=>'0000-00-00'},
			{:type=>'time', :value=>Time.new, :expected=>'00:00:00'},
			{:type=>'timestamp', :value=>Timestamp.new, :expected=>'0000-00-00 00:00:00'},
		]

		tests.each do |test|
			@attr.class_name = test[:type]
			value = @expr.value_for_bind_variable(@attr, test[:value])
			assert_equal(value, test[:expected])
		end
	end


	# list

	def test_aliases
		expected = {'' => 't0'}
		assert_equal(expected, @expr.aliases)

		path = ['publisher', 'name']
		@expr.sql_for_attribute_path path
		expected['publisher'] = 't1'
		assert_equal(expected, @expr.aliases)
	end

	def test_table_list
		list = @expr.table_list @expr.entity
		expected = 'book t0'
		assert_equal(expected, list)

		@expr.aliases['publisher.name'] = 't1'
		list = @expr.table_list(@expr.entity, false)
		expected = 'book t0, publisher t1'
		assert_equal(expected, list)

		@expr.use_aliases = false
		list = @expr.table_list @expr.entity
		expected = 'book'
		assert_equal(expected, list)
	end

	def test_append_item
		str = 'test'
		@expr.append_item str
		assert_equal(str, @expr.list_string)
	end

	def test_add_select_list
		@attr.read_format = 'UPPER(%P)'
		@expr.add_select_list @attr
		expected = 'UPPER(t0.title)'

		assert_equal(expected, @expr.list_string)
	end

	def test_add_insert_list
		value = 'test'
		@attr.write_format = 'UPPER(%V)'
		@expr.add_insert_list(@attr, value)
		expected_list  = 'UPPER(t0.title)'
		expected_value = 'test'
		bind = @expr.bind_variables.first

		assert_equal(expected_list, @expr.list_string)
		assert_equal(expected_value, bind[SQLExpression::VALUE_KEY])
	end

	def test_add_update_list
		value = 'test'
		@expr.add_update_list(@attr, value)
		expected_list  = 't0.title = ?'
		expected_value = 'test'
		bind = @expr.bind_variables.first

		assert_equal(expected_list, @expr.list_string)
		assert_equal(expected_value, bind[SQLExpression::VALUE_KEY])
	end


	# format

	def test_format_value
		# 'title' attribute external type is string
		values   = { :nil => nil, :str => 'test',
		             :num => 100, :date => Date.new(2003,1,1) }
		expected = { :nil => 'NULL', :str => "'test'",
		             :num => "'100'", :date => "'2003-01-01'" }

		assert_equal(expected[:nil],  @expr.format_value(values[:nil], @attr))
		assert_equal(expected[:str],  @expr.format_value(values[:str], @attr))
		assert_equal(expected[:num],  @expr.format_value(values[:num], @attr))
		assert_equal(expected[:date], @expr.format_value(values[:date], @attr))
	end

	def test_format_sql_string
		value = 'test'
		sql1 = @expr.format_sql_string(value, 'UPPER(%P)')
		sql2 = @expr.format_sql_string(value, nil)

		assert_equal("UPPER(#{value})", sql1)
		assert_equal(value, sql2)
	end


	# SQL string

	def test_sql_for_attribute
		sql      = @expr.sql_for_attribute @attr
		expected = 't0.title'
		assert_equal(expected, sql)

		@expr.use_aliases = false
		sql      = @expr.sql_for_attribute @attr
		expected = 'title'
		assert_equal(expected, sql)
	end

	def test_sql_for_attribute_named
		sql      = @expr.sql_for_attribute_named 'title'
		expected = 't0.title'
		assert_equal(expected, sql)

		@expr.use_aliases = false
		sql      = @expr.sql_for_attribute_named 'title'
		expected = 'title'
		assert_equal(expected, sql)
	end

	def test_sql_for_attribute_path
		path     = ['publisher', 'name']
		sql      = @expr.sql_for_attribute_path path
		expected = 't1.name'
		assert_equal(expected, sql)

		# join relationships
		assert_equal(1, @expr.join_relationships.size)

		# check repeat register
		sql = @expr.sql_for_attribute_path path
		assert_equal(expected, sql)
	end

	def test_sql_for_string
		tests = {'test'=>"'test'", "'test'"=>"'\\\\'test\\\\''", nil=>'NULL'}

		tests.each do |str, expected|
			assert_equal(expected, @expr.sql_for_string(str))
		end
	end

	def test_sql_for_number
		num = 100
		expected = '100'

		assert_equal(expected, @expr.sql_for_number(num))
		assert_equal('NULL', @expr.sql_for_number(nil))
	end

	def test_sql_for_date
		date = Date.new(2003,1,1)
		expected = "'2003-01-01'"

		assert_equal(expected, @expr.sql_for_date(date))
		assert_equal('NULL', @expr.sql_for_date(nil))
	end

	def test_sql_for_case_insensitive_like
		value = '"value"'
		key = 'key'
		sql = @expr.sql_for_case_insensitive_like(value, key)
		expected = "UPPER(#{key}) LIKE UPPER(#{value})"

		assert_equal(expected, sql)
	end

	def test_sql_for_value
		keypath = 'publisher.name'
		value   = 'test'
		assert_equal('?', @expr.sql_for_value(keypath, value))
		assert_equal(keypath, @expr.bind_variables[0][SQLExpression::NAME_KEY])
		assert_equal(value, @expr.bind_variables[0][SQLExpression::VALUE_KEY])
	end

	def test_sql_escape_char
		assert_equal('\\', @expr.sql_escape_char)
	end


	# order

	def test_order
		order1 = SortOrdering.new('publisher_id', SortOrdering::DESC)
		order2 = SortOrdering.new('title', SortOrdering::CI_ASC)
		@expr.add_order order1
		@expr.add_order order2
		expected = 't0.publisher_id DESC, UPPER(t0.title) ASC'

		assert_equal(expected, @expr.order_by_string)
	end


	# join

	def test_assemble_join_clause
		left     = 'key'
		right    = 'value'
		semantic = Relationship::INNER_JOIN # do nothing
		clause   = @expr.assemble_join(left, right, semantic)
		expected = 'key = value'

		assert_equal(expected, clause)
	end

	def test_add_join1
		left     = 'key'
		right    = 'value'
		semantic = Relationship::INNER_JOIN
		expected_semantic = "INNER"
		expected_on       = "#{left} = #{right} AND #{left} = #{right}"
		@expr.add_join(left, right, semantic)
		@expr.add_join(left, right, semantic)

		assert_equal(expected_semantic, @expr.join_semantic)
		assert_equal(expected_on, @expr.join_on)
	end

	def test_add_join2
		semantic = Relationship::FULL_OUTER_JOIN
		expected_semantic = "FULL OUTER"
		@expr.add_join('', '', semantic)
		assert_equal(expected_semantic, @expr.join_semantic)
	end

	def test_add_join3
		semantic = Relationship::LEFT_OUTER_JOIN
		expected_semantic = "LEFT OUTER"
		@expr.add_join('', '', semantic)
		assert_equal(expected_semantic, @expr.join_semantic)
	end

	def test_add_join4
		semantic = Relationship::RIGHT_OUTER_JOIN
		expected_semantic = "RIGHT OUTER"
		@expr.add_join('', '', semantic)
		assert_equal(expected_semantic, @expr.join_semantic)
	end

	def test_join_expression
		@expr.join_expression
		assert_equal('', @expr.join_clause)

		@expr.sql_for_attribute_path ['publisher', 'name']
		@expr.join_expression
		expected_semantic = 'INNER'
		expected_on       = 't0.publisher_id = t1.publisher_id'
		assert_equal(expected_semantic, @expr.join_semantic)
		assert_equal(expected_on, @expr.join_on)
	end

	def test_join_clause
		@expr.sql_for_attribute_path ['publisher', 'name']
		@expr.join_expression
		expected = 'INNER JOIN publisher t1 ON t0.publisher_id = t1.publisher_id'
		assert_equal(expected, @expr.join_clause)
	end


	# qualifier

	def test_sql_pattern
		pattern   = '* ? % _ \* \?'
		expected  = '% _ \\% \\_ * ?'
		escape    = '+'
		expected2 = '% _ +% +_ * ?'

		assert_equal(expected, @expr.sql_pattern(pattern))
		assert_equal(expected2, @expr.sql_pattern(pattern, escape))
	end

	def test_sql_for_symbol
		value = 'test'
		tests = [ \
			{:symbol=>Qualifier::EQUAL, :value=>nil, :sql=>'IS'},
			{:symbol=>Qualifier::EQUAL, :value=>'test', :sql=>'='},
			{:symbol=>Qualifier::NOT_EQUAL, :value=>nil, :sql=>'IS NOT'},
			{:symbol=>Qualifier::NOT_EQUAL, :value=>'test', :sql=>'<>'},
			{:symbol=>Qualifier::GREATER, :value=>1, :sql=>'>'},
			{:symbol=>Qualifier::GREATER_OR_EQUAL, :value=>1, :sql=>'>='},
			{:symbol=>Qualifier::LESS, :value=>1, :sql=>'<'},
			{:symbol=>Qualifier::LESS_OR_EQUAL, :value=>1, :sql=>'<='},
			{:symbol=>Qualifier::LIKE, :value=>'test', :sql=>'LIKE'}
		]

		tests.each do |test|
			assert_equal(test[:sql], @expr.sql_for_symbol(test[:symbol], test[:value]))
		end
	end

	def test_sql_for_key_value_qualifier
		tests = [
			{:format=>"== 'test'", :op=>'=', :value=>"test"},
			{:format=>"== nil",    :op=>'IS', :value=>nil},
			{:format=>"!= 'test'", :op=>'<>', :value=>"test"},
			{:format=>"!= nil", :op=>'IS NOT', :value=>nil},
			{:format=>"> 1", :op=>'>', :value=>1},
			{:format=>">= 1", :op=>'>=', :value=>1},
			{:format=>"< 1", :op=>'<', :value=>1},
			{:format=>"<= 1", :op=>'<=', :value=>1},
			{:format=>"like '*'", :op=>'LIKE', :value=>"%"},
			{:format=>"cilike '*'", :op=>'LIKE', :value=>"%"}
		]

		tests.each do |test|
			@expr.bind_variables.clear
			qualifier    = Qualifier.format "title #{test[:format]}"
			sql          = @expr.sql_for_key_value_qualifier qualifier
			value        = @expr.bind_variables[0][SQLExpression::VALUE_KEY]

			if qualifier.symbol == Qualifier::CI_LIKE then
				expected_sql = "UPPER(t0.title) #{test[:op]} UPPER(?)"
			else
				expected_sql = "t0.title #{test[:op]} ?"
			end

			assert_equal(expected_sql, sql)
			assert_equal(test[:value], value)
		end
	end

	def test_sql_for_key_comparison_qualifier
		tests = [
			{:format=>"== publisher.name", :op=>'='},
			{:format=>"!= publisher.name", :op=>'<>'},
			{:format=>"> publisher.name", :op=>'>'},
			{:format=>">= publisher.name", :op=>'>='},
			{:format=>"< publisher.name", :op=>'<'},
			{:format=>"<= publisher.name", :op=>'<='},
			{:format=>"like publisher.name", :op=>'LIKE'},
			{:format=>"cilike publisher.name", :op=>'LIKE'}
		]

		tests.each do |test|
			@expr.bind_variables.clear
			qualifier    = Qualifier.format "title #{test[:format]}"
			sql          = @expr.sql_for_key_comparison_qualifier qualifier

			if qualifier.symbol == Qualifier::CI_LIKE then
				expected_sql = "UPPER(t0.title) #{test[:op]} UPPER(t1.name)"
			else
				expected_sql = "t0.title #{test[:op]} t1.name"
			end

			assert_equal(expected_sql, sql)
		end
	end

	def test_sql_for_conjoined_qualifiers
		qualifier = Qualifier.format "(title like '*') and (publisher_id == 1)"
		expected  = "(t0.title LIKE ? AND t0.publisher_id = ?)"
		sql       = @expr.sql_for_conjoined_qualifiers qualifier.qualifiers

		assert_equal(expected, sql)
	end

	def test_sql_for_disjoined_qualifiers
		qualifier = Qualifier.format "(title like '*') or (publisher_id == 1)"
		expected  = "(t0.title LIKE ? OR t0.publisher_id = ?)"
		sql       = @expr.sql_for_disjoined_qualifiers qualifier.qualifiers

		assert_equal(expected, sql)
	end

	def test_sql_for_negated_qualifier
		qualifier = Qualifier.format "not (title like '*')"
		expected  = "NOT (t0.title LIKE ?)"
		sql       = @expr.sql_for_negated_qualifier qualifier.qualifier

		assert_equal(expected, sql)
	end

	# keyvalue, comparison, and, or, not
	def test_sql_for_qualifier
		format    = "not (((title like '*') and (publisher.name like '*')) " +
		            "or (publisher_id == 1))"
		qualifier = Qualifier.format format
		expected  = "NOT (((t0.title LIKE ? AND t1.name LIKE ?) " +
		            "OR t0.publisher_id = ?))"
		sql       = @expr.sql_for_qualifier qualifier

		assert_equal(expected, sql)
	end


	# select

	def test_assemble_select
		# attributes, islock, qualifier, fetchorder do nothing
		attributes = []
		islock     = false
		qualifier  = nil
		fetchorder = nil
		select     = 'DISTINCT'
		column     = 't0.title'
		table      = 'book t0'
		join       = 'INNER JOIN publisher t1 ON t0.publisher_id = t1.publisher_id'
		where      = "t0.title like '%'"
		orderby    = 't0.title ASC'
		lock       = nil # unsupported
		expected   = "SELECT DISTINCT t0.title FROM book t0 INNER JOIN publisher t1 " +
			"ON t0.publisher_id = t1.publisher_id WHERE t0.title like '%' " +
			"ORDER BY t0.title ASC"

		sql = @expr.assemble_select(attributes, islock, qualifier, fetchorder, 
			select, column, table, join, where, orderby, lock)
		assert_equal(expected, sql)
	end

	def test_prepare_select
		islock     = false # do nothing
		attributes = @entity.attributes
		qualifier  = Qualifier.format "publisher.name == nil"
		sort       = SortOrdering.new('title', SortOrdering::CI_DESC)
		sorts      = [sort]
		fetchspec  = FetchSpec.new('Book', qualifier, sorts)
		fetchspec.limit = 1
		expected   = "SELECT t0.book_id, t0.title, t0.publisher_id FROM book t0 " +
			"INNER JOIN publisher t1 ON t0.publisher_id = t1.publisher_id " +
			"WHERE t1.name IS ? " +
			"ORDER BY UPPER(t0.title) DESC LIMIT 1"

		@expr.prepare_select(attributes, islock, fetchspec)
		assert_equal(expected, @expr.statement)
	end


	# aggregate

	def test_assemble_aggregate
		# attributes, qualifier, fetchorder do nothing
		attributes = []
		qualifier  = nil
		having_qualifier = nil
		fetchorder = nil
		select     = 'DISTINCT'
		column     = 't0.title, SUM(t0.title)'
		table      = 'book t0'
		join       = 'INNER JOIN publisher t1 ON t0.publisher_id = t1.publisher_id'
		where      = "t0.title like '%'"
		groupby    = 't0.title'
		having     = 'SUM(t0.title) = "hello"'
		orderby    = 't0.title ASC'
		expected   = "SELECT DISTINCT t0.title, SUM(t0.title) FROM book t0 INNER JOIN publisher t1 " +
			"ON t0.publisher_id = t1.publisher_id WHERE t0.title like '%' " +
			'GROUP BY t0.title HAVING SUM(t0.title) = "hello" ORDER BY t0.title ASC'

		sql = @expr.assemble_aggregate(qualifier, having_qualifier,
			fetchorder, select, column, table, join, where, groupby, having, orderby)
		assert_equal(expected, sql)
	end

	def test_prepare_aggregate
		qualifier  = Qualifier.format "title like '*'"
		having     = Qualifier.format "titles > 1"
		sort       = SortOrdering.new('title', SortOrdering::CI_DESC)
		sorts      = [sort]
		aggspec    = AggregateSpec.new('Book', qualifier, sorts)
		aggspec.add('titles', 'title', AggregateSpec::COUNT)
		aggspec.group('title', having)
		expected   = "SELECT t0.title title, COUNT(t0.title) titles FROM book t0 " +
			"WHERE t0.title LIKE ? GROUP BY t0.title " +
			"ORDER BY UPPER(t0.title) DESC"

		@expr.prepare_aggregate(aggspec)
		assert_equal(expected, @expr.statement)
	end


	# update

	def test_assemble_update
		# row and qualifier do nothing
		row       = nil
		qualifier = nil
		table     = 'book t0'
		update    = "t0.title = 'test'"
		where     = 't0.book_id = 1'
		expected  = "UPDATE book t0 SET t0.title = 'test' WHERE t0.book_id = 1"

		sql = @expr.assemble_update(row, qualifier, table, update, where)
		assert_equal(expected, sql)
	end

	def test_prepare_update
		row       = {'title' => 'test'}
		qualifier = Qualifier.format "book_id == 1"
		expected  = "UPDATE book SET title = ? WHERE book_id = ?"

		@expr.prepare_update(row, qualifier)
		assert_equal(expected, @expr.statement)
	end


	# delete
	def test_assemble_delete
		qualifier = nil # do nothing
		table     = 'book t0'
		where     = 't0.book_id = 1'
		expected  = 'DELETE FROM book t0 WHERE t0.book_id = 1'

		sql = @expr.assemble_delete(qualifier, table, where)
		assert_equal(expected, sql)
	end

	def test_prepare_delete
		qualifier = Qualifier.format 'book_id == 1'
		expected  = 'DELETE FROM book WHERE book_id = ?'

		@expr.prepare_delete qualifier
		assert_equal(expected, @expr.statement)
	end


	# insert

	def test_assemble_insert
		row      = nil # do nothing
		table    = 'book'
		column   = 'book_id'
		value    = '1'
		expected = 'INSERT INTO book (book_id) VALUES (1)'

		sql = @expr.assemble_insert(row, table, column, value)
		assert_equal(expected, sql)
	end

	def test_prepare_insert
		row      = {'book_id'=>1, 'title'=>'test', 'publisher_id'=>1}
		expected = 'INSERT INTO book (title, publisher_id, book_id) VALUES (?, ?, ?)'

		@expr.prepare_insert row
		assert_equal(expected, @expr.statement)
	end

end


