# * do SQL statements with and without transaction
# * do SQL statements (select, insert, update, delete)

$LOAD_PATH.unshift '../../../lib'
require 'tapkit'
require 'tapkit/access/adapters/postgresql'
require 'test/unit'
require 'test/unit/ui/console/testrunner'

MODEL = 'testadapter.yaml'

class TestPostgreSQLAdapter < Test::Unit::TestCase
	include TapKit

	def setup
		app = Application.new
		model = Model.new MODEL
		@adapter = PostgreSQLAdapter.new(model, app)
	end

	def test_using_classes
		assert(PostgreSQLExpression, PostgreSQLAdapter.expression_class)
		assert_kind_of(PostgreSQLExpressionFactory, @adapter.expression_factory)
	end

	def test_internal_type
		tests = {
			'bigint'=>Integer, 'smallint'=>Integer, 'bigserial'=>Integer,
			'bit'=>String, 'bit varying'=>String, 'boolean'=>String,
			'bytea'=>String, 'serial'=>Integer, 'bigserial'=>Integer,
			'text'=>String, 'cidr'=>String, 'inet'=>String, 'macaddr'=>String
		}

		tests.each do |type, expected|
			assert_equal(expected, PostgreSQLAdapter.internal_type(type))
		end
	end
end


class TestPostgreSQLAccessing < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new [MODEL]
		@adapter = @app.adapter(@app.model(MODEL))
		@context = PostgreSQLContext.new @adapter
		@channel = @context.create_channel
		@ec = @app.ec
	end

	def test_commit_without_transaction
		_commit(false)
	end

	def test_commit_with_transaction
		_commit(true)
	end

	def _commit( transaction, pk = 100 )
		# @app.log_options[:sql] = true

		str = "test_#{Time.now}"
		num = rand(10000)
		date = Date.today
		expr = PostgreSQLExpression.new @app.entity('TestAdapter')

		# insert
		expr.statement = "insert into TEST_ADAPTER (TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE) values (#{pk}, '#{str}', #{num}, '#{date}')"
		@context.begin_transaction
		@channel.evaluate(expr)
		@context.commit_transaction

		# update
		num += 1
		expr.statement = "update TEST_ADAPTER set TEST_NUMBER = #{num} where TEST_ID = #{pk}"
		@context.begin_transaction
		@channel.evaluate expr
		@context.commit_transaction

		# select
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE from TEST_ADAPTER where TEST_ID = #{pk}"
		@context.begin_transaction
		state = @channel.evaluate expr
		@context.commit_transaction
		row = state.fetch_all.first
		assert_equal(pk, row[0])
		assert_equal(str, row[1].rstrip)
		assert_equal(num, row[2])
		assert_equal(date.to_s, row[3].to_s)

		# delete
		expr.statement = "delete from TEST_ADAPTER where TEST_ID = #{pk}"
		@context.begin_transaction
		@channel.evaluate expr
		@context.commit_transaction
	end

=begin
	# the default implementation is autocommit = true.
	# Can we commit and rollback manually using with DBI/Pg?
	def test_rollback
		@adapter.connection['transaction'] = true
		# @app.log_options[:sql] = true

		pk = 200
		str = "test_#{Time.now}"
		num = rand(10000)
		date = Date.today
		expr = PostgreSQLExpression.new @app.entity('TestAdapter')

		# send SQL statements
		expr.statement = "insert into TEST_ADAPTER (TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE) values (#{pk}, '#{str}', #{num}, '#{date}')"
		@context.begin_transaction
		@channel.evaluate(expr)

		# check
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE from TEST_ADAPTER where TEST_ID = #{pk}"
		state = @channel.evaluate expr
		row = state.fetch_all.first
		assert_equal(pk, row[0])
		assert_equal(str, row[1].rstrip)
		assert_equal(num, row[2])
		assert_equal(date.to_s, row[3].to_s)

		# rollback
		@context.rollback_transaction

		# check again
		state = @channel.evaluate expr
		assert state.fetch_all.empty?
	end
=end

	def test_accessing_date_and_time
		date = Date.today
		time = Time.now
		timestamp = Timestamp.now

		# insert
		ec = @app.ec
		obj = ec.create 'TestAdapter'
		obj['test_date'] = date
		obj['test_time'] = time
		obj['test_timestamp'] = timestamp
		assert_nothing_thrown { ec.save }

		# fetch
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		obj = ec.fetch(fs).first
		assert_equal(date, obj['test_date'])
		assert_equal(time, obj['test_time'])
		assert_equal(timestamp, obj['test_timestamp'])

		# update
		date += 1
		time += 1
		timestamp += 1
		obj['test_date'] = date
		obj['test_time'] = time
		obj['test_timestamp'] = timestamp
		assert_nothing_thrown { ec.save }

		# fetch again
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		obj = ec.fetch(fs).first
		assert_equal(date, obj['test_date'])
		assert_equal(time, obj['test_time'])
		assert_equal(timestamp, obj['test_timestamp'])

		# delete
		ec.delete obj
		assert_nothing_thrown { ec.save }

		# fetch again again
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		objs = ec.fetch(fs)
		assert objs.empty?
	end

	def fetchspec( date, time, timestamp )
		q = Qualifier.format(
			"(test_date = %@) and (test_time = %@) and (test_timestamp = %@)",
			[date, time, timestamp])
		FetchSpec.new('TestAdapter', q)
	end
end


class TestPostgreSQLExpression < Test::Unit::TestCase
	include TapKit

	def test_building_multiple_join
		app = Application.new '../../../examples/model/employee2.yaml'
		expected = 'SELECT t0.EMPLOYEE_ID FROM ' +
		'EMPLOYEE_PROJECT t0 INNER JOIN EMPLOYEE t1 ON ' +
		't0.EMPLOYEE_ID = t1.EMPLOYEE_ID ' +
		'INNER JOIN PROJECT t2 ON t0.PROJECT_ID = t2.PROJECT_ID ' +
		'WHERE (t1.NAME LIKE ? AND t2.NAME LIKE ?)'

		q = Qualifier.format "(employee.name like '*') and (project.name like '*')"
		fs = FetchSpec.new('EmployeeProject', q)
		entity = app.entity 'EmployeeProject'
		attrs = [entity.attribute('employee_id')]
		expr = PostgreSQLExpression.new entity
		expr.prepare_select(attrs, false, fs)
		assert_equal(expected, expr.statement)
	end

	def test_using_multiple_join
		app = Application.new 'employee.yaml'
		ec = app.ec
		q = Qualifier.format("(employee.name like '*') and (project.name like '*')")
		fs = FetchSpec.new('EmployeeProject', q)
		objs = []
		assert_nothing_thrown do
			objs = ec.fetch fs
		end
		assert (objs.size >= 1)
	end
end


if __FILE__ == $0 then
	suite = Test::Unit::TestSuite.new 'TapKit TestSuite'
	suite << TestPostgreSQLAdapter.suite
	suite << TestPostgreSQLAccessing.suite
	suite << TestPostgreSQLExpression.suite
	runner = Test::Unit::UI::Console::TestRunner.new suite
	runner.start
end
