# * do SQL statements with and without transaction
# * do SQL statements (select, insert, update, delete)

$LOAD_PATH.unshift '../../../lib'
require 'tapkit'
require 'tapkit/access/adapters/openbase'
require 'test/unit'
require 'test/unit/ui/console/testrunner'

MODEL = 'testadapter.yaml'

class TestOpenBaseAdapter < Test::Unit::TestCase
	include TapKit

	def setup
		app = Application.new
		model = Model.new MODEL
		@adapter = OpenBaseAdapter.new(model, app)
	end

	def test_using_classes
		assert(OpenBaseExpression, OpenBaseAdapter.expression_class)
		assert_kind_of(OpenBaseExpressionFactory, @adapter.expression_factory)
	end
end


class TestOpenBaseChannel < Test::Unit::TestCase
	include TapKit

	def setup
		app = Application.new
		model = Model.new MODEL
		@adapter = OpenBaseAdapter.new(model, app)
		@context = OpenBaseContext.new @adapter
		@channel = @context.create_channel
	end

	def test_describe_table_names
		assert @channel.describe_table_names.empty?
	end

	def test_describe_model
		model = @channel.describe_model ['TEST_ADAPTER']

		# adapter name
		assert_equal('OpenBase', model.adapter_name)

		# connection
		assert_equal('tapkit', model.connection['database'])
		assert_equal('localhost', model.connection['host'])
		assert_equal('admin', model.connection['user'])
		assert_equal('', model.connection['password'])

		# entity
		test_adapter  = model.entity 'TestAdapter'
		test_id       = test_adapter.attribute 'test_id'
		test_string   = test_adapter.attribute 'test_string'
		test_number   = test_adapter.attribute 'test_number'
		test_date     = test_adapter.attribute 'test_date'
		test_time     = test_adapter.attribute 'test_time'
		test_datetime = test_adapter.attribute 'test_datetime'

		# class properties
		assert_equal('TEST_ADAPTER', test_adapter.external_name)
		assert test_adapter.class_property?(test_string)
		assert test_adapter.class_property?(test_number)
		assert test_adapter.class_property?(test_date)
		assert test_adapter.class_property?(test_time)
		assert test_adapter.class_property?(test_datetime)

		# relationships
		assert test_adapter.relationships.empty?		

		# primary keys
		assert test_adapter.primary_key_attribute?(test_id)

		# each attributes
		assert_equal('test_id', test_id.name)
		assert_equal('TEST_ID', test_id.column_name)
		assert_equal(false, test_id.allow_null)
		assert_equal('long', test_id.external_type)
		assert_equal('Integer', test_id.class_name)

		assert_equal('test_string', test_string.name)
		assert_equal('TEST_STRING', test_string.column_name)
		assert_equal(true, test_string.allow_null)
		assert_equal('char', test_string.external_type)
		assert_equal('String', test_string.class_name)
		assert_equal(255, test_string.width)

		assert_equal('test_number', test_number.name)
		assert_equal('TEST_NUMBER', test_number.column_name)
		assert_equal(true, test_number.allow_null)
		assert_equal('int', test_number.external_type)
		assert_equal('Integer', test_number.class_name)

		assert_equal('test_date', test_date.name)
		assert_equal('TEST_DATE', test_date.column_name)
		assert_equal(true, test_date.allow_null)
		assert_equal('date', test_date.external_type)
		assert_equal('Date', test_date.class_name)

		assert_equal('test_time', test_time.name)
		assert_equal('TEST_TIME', test_time.column_name)
		assert_equal(true, test_time.allow_null)
		assert_equal('time', test_time.external_type)
		assert_equal('Time', test_time.class_name)

		assert_equal('test_datetime', test_datetime.name)
		assert_equal('TEST_DATETIME', test_datetime.column_name)
		assert_equal(true, test_datetime.allow_null)
		assert_equal('datetime', test_datetime.external_type)
		assert_equal('Timestamp', test_datetime.class_name)
	end
end


class TestOpenBaseAccessing < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new [MODEL]
		@adapter = @app.adapter(@app.model(MODEL))
		@context = OpenBaseContext.new @adapter
		@channel = @context.create_channel
		@ec = @app.ec
	end

	def test_commit_without_transaction
		_commit(false)
	end

	def test_commit_with_transaction
		_commit(true)
	end

	def _commit( transaction, pk = 100 )
		# @app.log_options[:sql] = true

		str = "test_#{Time.now}"
		num = rand(10000)
		date = Date.today
		expr = OpenBaseExpression.new @app.entity('TestAdapter')

		# insert
		expr.statement = "insert into TEST_ADAPTER (TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE) values (#{pk}, '#{str}', #{num}, '#{date}')"
		@context.begin_transaction
		@channel.evaluate(expr)
		@context.commit_transaction

		# update
		num += 1
		expr.statement = "update TEST_ADAPTER set TEST_NUMBER = #{num} where TEST_ID = #{pk}"
		@context.begin_transaction
		@channel.evaluate expr
		@context.commit_transaction

		# select
		@channel.select_attributes(@app.entity('TestAdapter').attributes, nil, nil, nil)
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE from TEST_ADAPTER where TEST_ID = #{pk}"
		@channel.evaluate expr
		row = @channel.fetch_all.first
		assert_equal(pk, row['test_id'])
		assert_equal(str, row['test_string'])
		assert_equal(num, row['test_number'])
		assert_equal(date.to_s, row['test_date'].to_s)

		# delete
		expr.statement = "delete from TEST_ADAPTER where TEST_ID = #{pk}"
		@context.begin_transaction
		@channel.evaluate expr
		@context.commit_transaction
	end

	def test_rollback
		# @app.log_options[:sql] = true

		pk = 100
		str = "test_#{Time.now}"
		num = rand(10000)
		date = Date.today
		expr = OpenBaseExpression.new @app.entity('TestAdapter')

		# send SQL statements
		expr.statement = "insert into TEST_ADAPTER (TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE) values (#{pk}, '#{str}', #{num}, '#{date}')"
		@context.begin_transaction
		@channel.evaluate(expr)

		# check
		@channel.select_attributes(@app.entity('TestAdapter').attributes, nil, nil, nil)
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE from TEST_ADAPTER where TEST_ID = #{pk}"
		@channel.evaluate expr
		row = @channel.fetch_all.first
		assert_equal(pk, row['test_id'])
		assert_equal(str, row['test_string'])
		assert_equal(num, row['test_number'])
		assert_equal(date.to_s, row['test_date'].to_s)

		# rollback
		@context.rollback_transaction

		# check again
		result = @channel.evaluate expr
		assert_nil result.fetch
	end

	def test_data_types_select
		expr = OpenBaseExpression.new @app.entity('TestAdapter')
		@channel.select_attributes(@app.entity('TestAdapter').attributes, nil, nil, nil)
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE, TEST_TIME, TEST_DATETIME from TEST_ADAPTER where TEST_ID = 1"
		@channel.evaluate expr
		row = @channel.fetch_all.first

		assert_equal(1, row['test_id'])
		assert_equal('test_00:52:42', row['test_string'])
		assert_equal(1, row['test_number'])
		assert_equal('2004-05-06', row['test_date'].to_s)
		assert_equal('12:30:00', row['test_time'].to_s)
		assert_equal('1981-12-27 17:00:00 +0900', row['test_datetime'].to_s)
	end

	def test_data_types_insert
		# @app.log_options[:sql] = true

		pk = 200
		str = "test_#{Time.now}"
		num = rand(10000)
		date = Date.today
		expr = OpenBaseExpression.new @app.entity('TestAdapter')
		time = Time.now
		datetime = Timestamp.now

		# insert
		expr.statement = "insert into TEST_ADAPTER (TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE, TEST_TIME, TEST_DATETIME) values (#{pk}, '#{str}', #{num}, '#{date}', '#{time}', '#{datetime}')"
		@context.begin_transaction
		@channel.evaluate(expr)
		@context.commit_transaction

		# select
		@channel.select_attributes(@app.entity('TestAdapter').attributes, nil, nil, nil)
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE, TEST_TIME, TEST_DATETIME from TEST_ADAPTER where TEST_ID = #{pk}"
		@channel.evaluate expr
		row = @channel.fetch_all.first
		assert_equal(pk, row['test_id'])
		assert_equal(str, row['test_string'])
		assert_equal(num, row['test_number'])
		assert_equal(date.to_s, row['test_date'].to_s)
		time.sec = 0
		assert_equal(time.to_s, row['test_time'].to_s)

		# OpenBase calculate the timestamp as GMT timezone.
		# assert_equal(datetime.to_s, row['test_datetime'].to_s)

		# delete
		expr.statement = "delete from TEST_ADAPTER where TEST_ID = #{pk}"
		@context.begin_transaction
		@channel.evaluate expr
		@context.commit_transaction
	end

	def test_accessing_date_and_time
		date = Date.today
		time = Time.now
		timestamp = Timestamp.now

		# insert
		ec = @app.ec
		obj = ec.create 'TestAdapter'
		obj['test_date'] = date
		obj['test_time'] = time
		obj['test_datetime'] = timestamp
		assert_nothing_thrown { ec.save }

		# fetch
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		obj = ec.fetch(fs).first
		assert_equal(date, obj['test_date'])
		assert_equal(time, obj['test_time'])
		assert_equal(timestamp, obj['test_datetime'])

		# update
		date += 1
		time += 1
		timestamp += 1
		obj['test_date'] = date
		obj['test_time'] = time
		obj['test_datetime'] = timestamp
		assert_nothing_thrown { ec.save }

		# fetch again
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		obj = ec.fetch(fs).first
		assert_equal(date, obj['test_date'])
		assert_equal(time, obj['test_time'])
		assert_equal(timestamp, obj['test_datetime'])

		# delete
		ec.delete obj
		assert_nothing_thrown { ec.save }

		# fetch again again
		ec = @app.ec
		fs = fetchspec(date, time, timestamp)
		objs = ec.fetch(fs)
		assert objs.empty?
	end

	def fetchspec( date, time, timestamp )
		q = Qualifier.format(
			"(test_date = %@) and (test_time = %@) and (test_datetime = %@)",
			[date, time, timestamp])
		FetchSpec.new('TestAdapter', q)
	end
end


if __FILE__ == $0 then
	suite = Test::Unit::TestSuite.new 'TapKit TestSuite'
	suite << TestOpenBaseAdapter.suite
	suite << TestOpenBaseChannel.suite
	suite << TestOpenBaseAccessing.suite
	runner = Test::Unit::UI::Console::TestRunner.new(suite, 2)
	runner.start
end
