require 'strscan'
require 'set'

module TapKit
	class QualifierParser
		class ParseError < StandardError; end #:nodoc:

		attr_reader :qualifier

		def initialize( format, bindings )
			expanded   = _expand(format.dup, bindings)
			tokens     = _analyze expanded
			@qualifier = _parse tokens
		end

		private

		def _expand( format, bindings )
			num = 0
			replace = nil
			format.gsub!(/%%/, '%')
			format.gsub!(/(%s|%d|%f|%@|%K)/) do
				case $1
				when '%s' then replace = "'#{_escape(bindings[num])}'"
				when '%d' then replace = bindings[num].to_i
				when '%f' then replace = bindings[num].to_f
				when '%@' then replace = _convert bindings[num]
				when '%K'
					if String === bindings[num] then
						replace = bindings[num]
					else
						replace = _convert bindings[num]
					end
				end
				num += 1
				replace
			end
			format
		end

		def _escape( string )
			string.to_s.gsub(/'/) { "\\'" }
		end

		def _convert( object )
			case object
			when String    then "'#{_escape(object)}'"
			when Integer   then object.to_i
			when Float     then object.to_f
			when true      then 'true'
			when false     then 'false'
			when nil       then 'nil'
			when Date      then "(Date)'#{object}'"
			when Time      then "(Time)'#{object}'"
			when Timestamp then "(Timestamp)'#{object}'"
			else
				"(#{object.class})'#{_escape(object.to_s)}'"
			end
		end

		def _analyze( format )
			format    = "(#{format})"
			scanner   = StringScanner.new format
			qualifier = nil
			tokens    = []
			op_reg    = /\A(==|!=|>=|<=|>|<|=|like|cilike)/im

			until scanner.eos? do
				scanner.skip(/\A[\s]+/)

				if str = scanner.scan(/\A\([A-Z]([a-zA-Z0-9]*)\)'(([^'\\]|\\.)*)'/) then
					tokens << scanner[0]
				elsif str = scanner.scan(/\A\([A-Z]([a-zA-Z0-9]*)\)"(([^'\\]|\\.)*)"/) then
					tokens << scanner[0]
				elsif str = scanner.scan(/\A(\(|\))/) then
					tokens << str
				elsif str = scanner.scan(op_reg) then
					tokens << Qualifier.operator_symbol(str)
				elsif str = scanner.scan(/\A\d+\.\d+/) then
					tokens << str.to_f
				elsif str = scanner.scan(/\A\d+/) then
					tokens << str.to_i
				elsif scanner.match?(/\Atrue\W/) then
					scanner.scan(/\Atrue/)
					tokens << true
				elsif scanner.match?(/\Afalse\W/) then
					scanner.scan(/\Afalse/)
					tokens << false
				elsif scanner.match?(/\Anil\W/) then
					scanner.scan(/\Anil/)
					tokens << nil
				elsif str = scanner.scan(/\A'(([^'\\]|\\.)*)'/) then
					tokens << scanner[0]
				elsif str = scanner.scan(/\A"(([^"\\]|\\.)*)"/) then
					tokens << scanner[0]
				else
					str = scanner.scan(/\A[^\s\(\)]+/)
					tokens << str
				end
			end

			tokens
		end

		def _parse( tokens )
			op_stack  = []
			out_stack = []
			op = left = right = q = nil

			reg_and = /\Aand\Z/mi
			reg_or  = /\Aor\Z/mi
			reg_not = /\Anot\Z/mi

			tokens.each do |token|
				case token
				when '('
					op_stack << token
				when ')'
					until op_stack.last == '(' do
						op    = op_stack.pop
						right = out_stack.pop
						left  = out_stack.pop

						case op
						when Symbol
							if right =~ /\A\(([A-Z][a-zA-Z0-9]*)\)'(([^'\\]|\\.)*)'/
								value = Object.__send__($1, $2)
								q = KeyValueQualifier.new(left, op, value)
							elsif right =~ /\A\(([A-Z][a-zA-Z0-9]*)\)"(([^'\\]|\\.)*)"/
								value = Object.__send__($1, $2)
								q = KeyValueQualifier.new(left, op, value)
							elsif right =~ /\A'(([^'\\]|\\.)*)'/ then
								q = KeyValueQualifier.new(left, op, $1)
							elsif right =~ /\A"(([^"\\]|\\.)*)"/ then
								q = KeyValueQualifier.new(left, op, $1)
							elsif (Numeric === right) or (right == true) or \
								(right == false) or right.nil? then
								q = KeyValueQualifier.new(left, op, right)
							else
								q = KeyComparisonQualifier.new(left, op, right)
							end
						when reg_and
							if AndQualifier === right then
								right.qualifiers.unshift left
								q = right
							else
								q = AndQualifier.new [left, right]
							end
						when reg_or
							if OrQualifier === right then
								right.qualifiers.unshift left
								q = right
							else
								q = OrQualifier.new [left, right]
							end
						when reg_not
							q = NotQualifier.new right
						end
						out_stack << q
					end
					op_stack.pop			
				when reg_and
					op_stack << token
				when reg_or
					op_stack << token
				when reg_not
					op_stack << token
				when Symbol
					op_stack << token
				else
					out_stack << token
				end
			end

			result = out_stack.pop
			unless out_stack.empty? and op_stack.empty? then
				raise ParseError, 'parse error'
			end

			result
		end
	end


	# == Format Strings
	#
	# %s:: String
	# %d:: Integer
	# %f:: Float
	# %@:: An arbitrary Object argument.
	# %K:: Treated as a key
	# %%:: Literal % character
	#
	class Qualifier
		module ComparisonSupport
			def compare( left, right, symbol )
				__send__(symbol, left, right)
			end

			def equal?( left, right )
				left == right
			end

			def not_equal?( left, right )
				left != right
			end

			def greater?( left, right )
				left > right
			end

			def greater_or_equal?( left, right )
				left >= right
			end

			def less?( left, right )
				left < right
			end

			def less_or_equal?( left, right )
				left <= right
			end

			def like?( left, right )
				regexp = _convert_operator_to_regexp right
				if left =~ regexp then
					true
				else
					false
				end
			end

			def ci_like?( left, right )
				regexp = _convert_operator_to_regexp(right, true)
				if left =~ regexp then
					true
				else
					false
				end
			end

			private

			# * - (.*)
			# ? - .
			# option - m, i
			def _convert_operator_to_regexp( string, ci = false )
				converted = string.gsub(/(\*|\?)/) do |matched|
					case matched
					when '*' then '(.*)'
					when '?' then '.'
					end
				end

				if ci == true then
					/\A#{converted}/mi
				else
					/\A#{converted}/m
				end
			end
		end

		class UnknownKeyError < StandardError; end #:nodoc:

		extend ComparisonSupport

		EQUAL            = :'equal?'
		NOT_EQUAL        = :'not_equal?'
		CONTAIN          = :'contain?'
		GREATER          = :'greater?'
		GREATER_OR_EQUAL = :'greater_or_equal?'
		LESS             = :'less?'
		LESS_OR_EQUAL    = :'less_or_equal?'
		LIKE             = :'like?'
		CI_LIKE          = :'ci_like?'

		class << self
			R_OPERATORS = ['=', '==', '!=', '<', '<=', '>', '>=']
			OPERATORS   = ['=', '==', '!=', '<', '<=', '>', '>=', 'like', 'cilike']

			def new_with_format( format, bindings = [] )
				parser = QualifierParser.new(format, bindings)
				parser.qualifier
			end

			alias format new_with_format

			# Creates an AndQualifier with KeyValueQualifiers.
			def new_to_match_all_values( values )
				qualifiers = []
				values.each do |key, value|
					qualifier = KeyValueQualifier.new(key, EQUAL, value)
					qualifiers << qualifier
				end
				AndQualifier.new qualifiers
			end

			def new_to_match_any_value( values )
				qualifiers = []
				values.each do |key, value|
					qualifier = KeyValueQualifier.new(key, EQUAL, value)
					qualifiers << qualifier
				end
				OrQualifier.new qualifiers
			end

			def operator_symbol( string )
				case string.upcase
				when '='      then EQUAL
				when '=='     then EQUAL
				when '!='     then NOT_EQUAL
				when '>'      then GREATER
				when '>='     then GREATER_OR_EQUAL
				when '<'      then LESS
				when '<='     then LESS_OR_EQUAL
				when 'CILIKE' then CI_LIKE
				when 'LIKE'   then LIKE
				end
			end

			def operator_string( operator )
				case operator
				when EQUAL            then '='
				when NOT_EQUAL        then '!='
				when GREATER          then '>'
				when GREATER_OR_EQUAL then '>='
				when LESS             then '<'
				when LESS_OR_EQUAL    then '<='
				when LIKE             then 'like'
				when CI_LIKE          then 'cilike'
				end
			end

			def operators
				OPERATORS
			end

			def relational_operators
				R_OPERATORS
			end

			def filter( objects, qualifier )
				filtered = []
				objects.each do |object|
					if qualifier.eval? object then
						filtered << object
					end
				end
				filtered
			end
		end

		def initialize
			@binding_keys = [] # unsupported
		end

		def qualifier_keys
			set = Set.new
			add_qualifier_keys set
			set
		end

		# abstract - subclasses must override it
		def add_qualifier_keys( set ); end

		def validate( description )
			property_keys = description.all_property_keys

			qualifier_keys.each do |keypath|
				key = keypath.split('.').first
				unless property_keys.include? key then
					raise UnknownKeyError, "Qualifier found an unknown key '#{key}'"
				end
			end
		end

		alias inspect to_s
	end


	class KeyValueQualifier < Qualifier
		attr_reader :key, :value, :symbol

		def initialize( key, symbol, value )
			super()
			@key = key
			@symbol = symbol
			@value = value
		end

		def add_qualifier_keys( set )
			set << @key
		end

		def ==( other )
			bool = false
			if KeyValueQualifier === other then
				if (@key == other.key) and (@symbol == other.symbol) and \
					(@value == other.value) then
					bool = true
				end
			end

			bool
		end

		def eval?( object )
			keypath = @key.split '.'
			_compare_with_keypath(object, keypath)
		end

		private

		def _compare_with_keypath( object, keypath )
			key = keypath.shift
			dest = object[key]

			result = false

			if ((FaultingDelayEvaluationArray === dest) or (Array === dest)) \
				and (keypath.empty? == false)  then
				result = true
				dest.each do |to_many|
					if _compare_with_keypath(to_many, keypath) == false then
						result = false
					end
				end
			elsif keypath.empty? then
				result = Qualifier.compare(dest, @value, @symbol)
			else
				result = _compare_with_keypath(dest, keypath)
			end

			result
		end

		public

		def to_s
			op = Qualifier.operator_string @symbol
			if String === @value then
				value_s = "'#@value'"
			else
				value_s = @value
			end

			"(#@key #{op} #{value_s})"
		end
	end


	class KeyComparisonQualifier < Qualifier
		attr_reader :left, :symbol, :right

		def initialize( left, symbol, right )
			super()
			@left   = left
			@symbol = symbol
			@right  = right
		end

		def add_qualifier_keys( set )
			set << @left
		end

		def ==( other )
			bool = false
			if KeyComparisonQualifier === other then
				if (@left == other.left) and (@symbol == other.symbol) and \
					(@right == other.right) then
					bool = true
				end
			end

			bool
		end

		def eval?( object )
			Qualifier.compare(object[@left], object[@right], @symbol)
		end

		def to_s
			op = Qualifier.operator_string @symbol
			"(#@left #{op} #@right)"
		end
	end


	class AndQualifier < Qualifier
		attr_reader :qualifiers

		def initialize( qualifiers )
			super()
			@qualifiers = qualifiers
		end

		def each
			qualifiers.each do |qualifier|
				yield qualifier
			end
		end

		def add_qualifier_keys( set )
			@qualifiers.each do |qualifier|
				qualifier.add_qualifier_keys set
			end
		end

		def ==( other )
			bool = false
			if AndQualifier === other then
				if @qualifiers == other.qualifiers then
					bool = true
				end
			end

			bool
		end

		def eval?( object )
			@qualifiers.each do |qualifier|
				unless qualifier.eval? object then
					return false
				end
			end
			true
		end

		def to_s
			str = '('
			@qualifiers.each do |q|
				str << q.to_s
				unless @qualifiers.last == q then
					str << " AND "
				end
			end
			str << ')'
			str
		end
	end


	class OrQualifier < Qualifier
		attr_reader :qualifiers

		def initialize( qualifiers )
			super()
			@qualifiers = qualifiers
		end

		def each
			qualifiers.each do |qualifier|
				yield qualifier
			end
		end

		def add_qualifier_keys( set )
			@qualifiers.each do |qualifier|
				qualifier.add_qualifier_keys set
			end
		end

		def ==( other )
			bool = false
			if OrQualifier === other then
				if @qualifiers == other.qualifiers then
					bool = true
				end
			end

			bool
		end

		def eval?( object )
			@qualifiers.each do |qualifier|
				if qualifier.eval? object then
					return true
				end
			end
			false
		end

		def to_s
			str = '('
			@qualifiers.each do |q|
				str << q.to_s
				unless @qualifiers.last == q then
					str << " OR "
				end
			end
			str << ')'
			str
		end
	end


	class NotQualifier < Qualifier
		attr_reader :qualifier

		def initialize( qualifier )
			super()
			@qualifier = qualifier
		end

		def add_qualifier_keys( set )
			qualifier.add_qualifier_keys set
		end

		def ==( other )
			bool = false
			if NotQualifier === other then
				if @qualifier == other.qualifier then
					bool = true
				end
			end

			bool
		end

		def eval?( object )
			unless @qualifier.eval? object then
				true
			else
				false
			end
		end

		def to_s
			"(NOT #{qualifier})"
		end
	end
end


