module TapKit

	# KeyValueCoding provides methods to access instance variables of the object.
	# The methods try accessing instance variables by using accessor method.
	# If it is failure, its try accessing directly.
	#
	# If a class method "access_instance_variables?" defines in the class and
	# the method returns "true", the direct access is success. Or failure.
	module KeyValueCoding
		class UnknownKeyError < Exception #:nodoc:
		end

		# Retrieves value of the instance variable with method chain.
		def retrieve_value( key )
			object = nil
			if respond_to? key then
				object = __send__ key
			elsif respond_to? :retrieve_stored_value then
				object = retrieve_stored_value key
			elsif directly? then
				object = instance_eval "@#{key}"
			else
				begin
					object = handle_retrieve_value key
				rescue UnknownKeyError
					_raise_error(self, key)
				end
			end
			object
		end

		# Sets value for the instance variable with method chain.
		def take_value( key, value )
			writer = "#{key}="
			if respond_to? writer then
				__send__(writer, value)
			elsif respond_to? :take_stored_value then
				take_stored_value(key, value)
			elsif directly? then
				instance_eval "@#{key} = value"
			else
				begin
					handle_take_value(key, value)
				rescue UnknownKeyError
					_raise_error(self, key)
				end
			end
		end

		def directly?( object = self )
			if object.class.respond_to? 'access_instance_variables?' then
				object.class.access_instance_variables?
			else
				false
			end
		end

		# Hook method invoked from retrieve_value() when method for the key is not found.
		def handle_retrieve_value( key )
			raise UnknownKeyError
		end

		# Hook method invoked from take_value() when method for the key is not found.
		def handle_take_value( key, value )
			_raise_error(self, key)
		end

		def _raise_error( object, key ) #:nodoc:
			msg =  "This \"#{object.class}\" object does not have a method "
			msg << "\"#{key}\", nor an instance variable \"@#{key.sub(/=$/,'')}\"."
			raise UnknownKeyError, msg
		end
	end


	module KeyValueCodingAdditions
		include KeyValueCoding

		def retrieve_value_for_keypath( keypath )
			paths = keypath.split '.'
			object = self
			paths.each do |path|
				if object.respond_to? :retrieve_value then
					object = object.retrieve_value path
				elsif object.respond_to? path then
					object = object.__send__ path
				elsif directly? object then
					object = object.instance_eval "@#{path}"
				else
					_raise_error(object, path)
				end
			end
			object
		end

		# Sets value for the instance variable with method chain.
		def take_value_for_keypath( keypath, value )
			paths  = keypath.split '.'
			object = self
			paths.each_with_index do |path, index|
				writer = "#{path}="

				if (index + 1) == paths.size then
					# takes value for key
					if object.respond_to? :take_value then
						object.take_value(path, value)
					elsif object.respond_to? writer then
						object.__send__(writer, value)
					elsif directly? object then
						object.instance_eval "@#{path}=value"
					else
						begin
							handle_take_value(writer, value)
						rescue UnknownKeyError
							_raise_error(object, path)
						end
					end
				else
					# get and set value for the object
					if object.respond_to? :retrieve_value then
						object = object.retrieve_value path
					elsif object.respond_to? path then
						object = object.__send__ path
					elsif directly? object then
						object = object.instance_eval "@#{path}"
					else
						begin
							object = handle_retrieve_value path
						rescue UnknownKeyError
							_raise_error(object, path)
						end
					end
				end
			end
		end

		def retrieve_values( keys )
			values = {}
			keys.each do |key|
				values[key] = retrieve_value(key)
			end
			values
		end

		def take_values( values )
			values.each do |key, value|
				take_value(key, value)
			end
		end
	end

	module StoredKeyValueCoding
		include KeyValueCoding

		def retrieve_stored_value( key )
			unless property_keys.include?(key) then
				begin
					object = handle_retrieve_value key
				rescue UnknownKeyError
					_raise_error(self, key)
				end
			end
			_retrieve_stored_value( key )
		end

		def _retrieve_stored_value( key ) #:nodoc:
			will_read
			@state[key]
		end

		def take_stored_value( key, value )
			unless property_keys.include?(key) then
				begin
					handle_take_value(key, value)
				rescue UnknownKeyError
					_raise_error(self, key)
				end
			end
			_take_stored_value( key, value )
		end

		def _take_stored_value( key, value ) #:nodoc:
			if fault? then
				will_read
			end

			will_change
			@state[key] = value
		end

	end

end
