/**
 * Copyright (C) 2004, 2005 Gregoire Lejeune <gregoire.lejeune@free.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "xpath.h"

static VALUE each_pair( VALUE obj ) {
  return rb_funcall( obj, rb_intern("each"), 0, 0 );
}

static VALUE process_pair( VALUE pair, VALUE rbparams ) {
    VALUE key, value;
    static int count = 0;

    Check_Type(pair, T_ARRAY);

    key   = RARRAY(pair)->ptr[0];
    value = RARRAY(pair)->ptr[1];

    Check_Type(key, T_STRING);
    Check_Type(value, T_STRING);

    rb_ary_store( rbparams, count, key );
    rb_ary_store( rbparams, count + 1, value );

    count += 2;
    return Qnil;
}

/* ------------------------------------------------------------------------------- */

/**
 * register_namespaces:
 * @xpathCtx:        the pointer to an XPath context.
 * @nsList:          the list of known namespaces in 
 *            {<prefix1>=><href1>, <prefix2>=>href2>, ...} format.
 *
 * Registers namespaces from @nsList in @xpathCtx.
 *
 * Returns 0 on success and a negative value otherwise.
 */
int register_namespaces( xmlXPathContextPtr xpathCtx, VALUE nsList ) {
  xmlChar* prefix;
  xmlChar* href;
  int iNbNs;
  int iCpt;

  VALUE lxNS = rb_ary_new( );
  (void)rb_iterate( each_pair, nsList, process_pair, lxNS );
  iNbNs = FIX2INT( rb_funcall( nsList, rb_intern("size"), 0, 0 ) );

  for( iCpt = 0; iCpt <= iNbNs; iCpt += 2 ) {
    prefix = (xmlChar*)STR2CSTR( rb_ary_entry( lxNS, iCpt ) );
    href = (xmlChar*)STR2CSTR( rb_ary_entry( lxNS, iCpt+1 ) );

    if( xmlXPathRegisterNs( xpathCtx, prefix, href ) != 0 ) {
      return( -1 );
    }
  }

  return( 0 );
}

int get_xpath_nodes( xmlNodeSetPtr nodes, RbTxpath *pRbTxpath ) {
  int RCod;
  int iCpt;
  int jCpt;
  struct _xmlAttr *attr;
  VALUE rbNode;
  VALUE rbListAttr;

  pRbTxpath->rbResult = rb_ary_new( );

  RCod = pRbTxpath->iNbNodes = (nodes) ? nodes->nodeNr : 0;
  pRbTxpath->iCurrentNode = 0;

  for( iCpt = 0; iCpt < RCod; iCpt++ ) {
    rbNode = rb_hash_new( );

    rb_hash_aset( rbNode, rb_str_new2( "type" ),  INT2NUM( (int)nodes->nodeTab[iCpt]->type ) );
    rb_hash_aset( rbNode, rb_str_new2( "path" ),  rb_str_new2( (char*)xmlGetNodePath(nodes->nodeTab[iCpt]) ) );
    rb_hash_aset( rbNode, rb_str_new2( "name" ),  rb_str_new2( (char*)nodes->nodeTab[iCpt]->name ) );

    if( nodes->nodeTab[iCpt]->ns != NULL ) {
      rb_hash_aset( rbNode, rb_str_new2( "NS href" ),   rb_str_new2( (char*)nodes->nodeTab[iCpt]->ns->href ) );
      rb_hash_aset( rbNode, rb_str_new2( "NS prefix" ), rb_str_new2( (char*)nodes->nodeTab[iCpt]->ns->prefix ) );
    }

    switch( nodes->nodeTab[iCpt]->type ) {
      case XML_ELEMENT_NODE:
        rbListAttr = rb_hash_new( );

        jCpt = 0;
        if( nodes->nodeTab[iCpt]->children != NULL && nodes->nodeTab[iCpt]->children->content != NULL ) {
          rb_hash_aset( rbNode, rb_str_new2( "content" ), rb_str_new2( (char*)nodes->nodeTab[iCpt]->children->content ) );
        }

        attr = nodes->nodeTab[iCpt]->properties;
        while( attr != NULL ) {
            rb_hash_aset( rbListAttr, rb_str_new2( (char*)attr->name ), rb_str_new2( (char*)attr->children->content ) );

            jCpt++;
            attr = attr->next;
        }

        if( jCpt > 0 ) {
          rb_hash_aset( rbNode, rb_str_new2( "attrs" ), rbListAttr );
        }
        break;

      case XML_ATTRIBUTE_NODE:
        if( nodes->nodeTab[iCpt]->children != NULL ) {
          rb_hash_aset( rbNode, rb_str_new2( "content" ), rb_str_new2( (char*)nodes->nodeTab[iCpt]->children->content ) );
        }
        break;

      default:
        break;
    }

    rb_ary_push( pRbTxpath->rbResult, rbNode );
  }

  return( RCod );
}

/**
 * execute_xpath_expression:
 *
 * Parses input XML file, evaluates XPath expression and return results.
 *
 * Returns 0 on success and a negative value otherwise.
 */
int parse( const char* xml, int iXmlType, const xmlChar* xpathExpr, RbTxpath *pRbTxpath ) {
  xmlErrorPtr ptXMLError;
  xmlDocPtr doc = NULL;
  xmlXPathContextPtr xpathCtx;
  xmlXPathObjectPtr xpathObj;
  int RCod = -1;

  /* Init libxml */
  xmlInitParser();
  LIBXML_TEST_VERSION

  /** Act: Parse XML */
  xmlResetLastError( );
  ptXMLError = NULL;
  if( iXmlType == RUBY_XPATH_XMLSRC_TYPE_STR ) {
    doc = xmlParseMemory( xml, strlen( xml ) );
  } else if( iXmlType == RUBY_XPATH_XMLSRC_TYPE_FILE ) {
    doc = xmlParseFile( xml );
  }

  ptXMLError = xmlGetLastError();
  if( doc == NULL || ptXMLError != NULL ) {
    rb_fatal( "XML::XPATH: XML parsing error : %s\n", ptXMLError->message );
    return( RCod );
  }

  /* Create xpath evaluation context */
  xpathCtx = xmlXPathNewContext(doc);
  if(xpathCtx == NULL) {
	ptXMLError = xmlGetLastError();
    rb_fatal( "Error: unable to create new XPath context : %s\n", ptXMLError->message);
    xmlFreeDoc(doc);
    return( RCod );
  }

  /* Register namespaces from list (if any) */
  if( pRbTxpath->hxNS != Qnil ) {
	if( register_namespaces(xpathCtx, pRbTxpath->hxNS) < 0 ) {
      ptXMLError = xmlGetLastError();
      rb_fatal( "Error: failed to register namespaces list : %s\n", ptXMLError->message );
      xmlXPathFreeContext(xpathCtx);
      xmlFreeDoc(doc);
      return( RCod );
	}
  }

  /* Evaluate xpath expression */
  xpathObj = xmlXPathEvalExpression(xpathExpr, xpathCtx);
  if(xpathObj == NULL) {
    ptXMLError = xmlGetLastError();
    rb_fatal("Error: unable to evaluate xpath expression \"%s\" : %s\n", xpathExpr, ptXMLError->message );
    xmlXPathFreeContext(xpathCtx);
    xmlFreeDoc(doc);
    return( RCod );
  }

  /* Print results */
  RCod = get_xpath_nodes( xpathObj->nodesetval, pRbTxpath );

  /* Cleanup */
  xmlXPathFreeObject(xpathObj);
  xmlXPathFreeContext(xpathCtx);
  xmlFreeDoc(doc);

  /* Shutdown libxml */
  xmlCleanupParser();

  /*
   * this is to debug memory for regression tests
   */
  xmlMemoryDump();
  return( RCod );
}

VALUE get_node_element( VALUE self, const char *xField ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );
  VALUE xValue = Qnil;

  if( pRbTxpath->iNbNodes > 0 ) {
    VALUE rbNode = rb_ary_entry( pRbTxpath->rbResult, pRbTxpath->iCurrentNode );
    xValue  = rb_hash_aref( rbNode, rb_str_new2( xField ) );
  }

  return( xValue );
}

/**
 * vOut = object_to_string( VALUE object );
 */
VALUE object_to_string( VALUE object ) {
  VALUE vOut = Qnil;

  switch( TYPE( object ) ) {
    case T_STRING:
      {
		if( isFile( STR2CSTR( object ) ) == 0 ) {
          vOut = object;
        } else {
          long iBufferLength;
          long iCpt;
          char *xBuffer;
        
		  FILE* fStream = fopen( STR2CSTR( object ), "r" );
		  if( fStream == NULL ) {
			return( Qnil );
		  }

          fseek( fStream, 0L, 2 );
          iBufferLength = ftell( fStream );
          xBuffer = (char *)malloc( (int)iBufferLength + 1 );
          if( !xBuffer )
            rb_raise( rb_eNoMemError, "Memory allocation error" );

          xBuffer[iBufferLength] = 0;
          fseek( fStream, 0L, 0 );
          iCpt = fread( xBuffer, 1, (int)iBufferLength, fStream );

          if( iCpt != iBufferLength ) {
            free( (char *)xBuffer );
            rb_raise( rb_eSystemCallError, "Read file error" );
          }

          vOut = rb_str_new2( xBuffer );
        
		  fclose( fStream );
        }
      }
      break;

    case T_DATA:
    case T_OBJECT:
      {
        if( strcmp( getRubyObjectName( object ), "XML::Simple::Dom" ) == 0 ) {
          vOut = rb_funcall( object, rb_intern( "to_s" ), 0 );
        } else {
          rb_raise( rb_eSystemCallError, "Can't read XML from object %s", getRubyObjectName( object ) );
        }
      }
      break;

    default:
      rb_raise( rb_eArgError, "XML object #0x%x not supported", TYPE( object ) );
  }

  return( vOut );
}

/* ------------------------------------------------------------------------------- */

/** xmlData = o.xml */
VALUE ruby_xpath_xml_str_get( VALUE self ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );

  if( pRbTxpath->iXmlType == RUBY_XPATH_XMLSRC_TYPE_NULL )
    return( Qnil );
  else
    return( (VALUE)pRbTxpath->xXmlData );
}

/** o.xml = "<...>" */
VALUE ruby_xpath_xml_str_set( VALUE self, VALUE xml_doc_str ) {
  RbTxpath *pRbTxpath;

  Data_Get_Struct( self, RbTxpath, pRbTxpath );

  pRbTxpath->iXmlType = RUBY_XPATH_XMLSRC_TYPE_STR;
  pRbTxpath->xXmlData = object_to_string( xml_doc_str );

  return( xml_doc_str );
}

/** o.xmlfile="file.xml" */
VALUE ruby_xpath_xml_file_set( VALUE self, VALUE xml_doc_file ) {
  rb_warn( "XML::XSLT#xmlfile=<file> is deprecated. Please use XML::XSLT#xml=<file> !" );
  return( ruby_xpath_xml_str_set( self, xml_doc_file ) );
}

/** o.ns={ ... } */
VALUE ruby_xpath_xml_ns_set( VALUE self, VALUE ns_hash ) {
  RbTxpath *pRbTxpath;
  Check_Type( ns_hash, T_HASH );

  Data_Get_Struct( self, RbTxpath, pRbTxpath );
  pRbTxpath->hxNS = ns_hash;

  return( Qnil );
}

/* ------------------------------------------------------------------------------- */

/** o.execute( XPathExp ) */
VALUE ruby_xpath_xml_execute( VALUE self, VALUE xpathExp ) {
  int RCod = -1;
  RbTxpath *pRbTxpath;
  Check_Type( xpathExp, T_STRING );

  Data_Get_Struct( self, RbTxpath, pRbTxpath );

  RCod = parse( STR2CSTR( pRbTxpath->xXmlData ), pRbTxpath->iXmlType, BAD_CAST STR2CSTR( xpathExp ), pRbTxpath );

  return( INT2NUM( RCod ) );
}

/* a = o.to_a */
VALUE ruby_xpath_xml_to_a( VALUE self ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );

  return( pRbTxpath->rbResult );
}

/* ------------------------------------------------------------------------------- */

VALUE ruby_xpath_xml_each( VALUE self ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );
  int iCpt;

  if (rb_block_given_p()) {
    for( iCpt = 0; iCpt < pRbTxpath->iNbNodes; iCpt++ ) {
      VALUE rbNode = rb_ary_entry( pRbTxpath->rbResult, iCpt );
      rb_yield( rbNode );
    }
  }
  return( Qnil );
}

VALUE ruby_xpath_xml_first( VALUE self ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );
  int RCod = 0;

  if( pRbTxpath->iNbNodes > 0 ) {
    pRbTxpath->iCurrentNode = 0;
    RCod = 1;
  }

  return( INT2NUM( RCod ) );
}

VALUE ruby_xpath_xml_last( VALUE self ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );
  int RCod = 0;

  if( pRbTxpath->iNbNodes > 0 ) {
    pRbTxpath->iCurrentNode = pRbTxpath->iNbNodes - 1;
    RCod = pRbTxpath->iNbNodes;
  }

  return( INT2NUM( RCod ) );
}

VALUE ruby_xpath_xml_next( VALUE self ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );
  int RCod = 0;

  if( pRbTxpath->iCurrentNode + 1 < pRbTxpath->iNbNodes ) {
    pRbTxpath->iCurrentNode += 1;
    RCod = pRbTxpath->iCurrentNode + 1;
  }

  return( INT2NUM( RCod ) );
}

VALUE ruby_xpath_xml_prev( VALUE self ) {
  RbTxpath *pRbTxpath;
  Data_Get_Struct( self, RbTxpath, pRbTxpath );
  int RCod = 0;

  if( pRbTxpath->iCurrentNode - 1 >= 0 ) {
    pRbTxpath->iCurrentNode -= 1;
    RCod = pRbTxpath->iCurrentNode + 1;
  }

  return( INT2NUM( RCod ) );
}

/* ------------------------------------------------------------------------------- */

VALUE ruby_xpath_xml_name( VALUE self ) {
  return( get_node_element( self, "name" ) );
}

VALUE ruby_xpath_xml_attrs( VALUE self ) {
  return( get_node_element( self, "attrs" ) );
}

VALUE ruby_xpath_xml_type( VALUE self ) {
  return( get_node_element( self, "type" ) );
}

VALUE ruby_xpath_xml_content( VALUE self ) {
  return( get_node_element( self, "content" ) );
}

VALUE ruby_xpath_xml_path( VALUE self ) {
  return( get_node_element( self, "path" ) );
}


/* ------------------------------------------------------------------------------- */

void ruby_xpath_free( RbTxpath *pRbTxpath ) {
  if (pRbTxpath != NULL)
    free(pRbTxpath);
}

void ruby_xpath_mark( RbTxpath *pRbTxpath ) {
  if( pRbTxpath == NULL ) return;
  if( !NIL_P(pRbTxpath->xXmlData) ) rb_gc_mark( pRbTxpath->xXmlData );
  if( !NIL_P(pRbTxpath->hxNS) )     rb_gc_mark( pRbTxpath->hxNS );
  if( !NIL_P(pRbTxpath->rbResult) ) rb_gc_mark( pRbTxpath->rbResult );
}

/** o = XML::XPATH.new() */
VALUE ruby_xpath_new( VALUE class ) {
  RbTxpath *pRbTxpath;

  pRbTxpath = (RbTxpath *)malloc(sizeof(RbTxpath));
  if( pRbTxpath == NULL )
    rb_raise(rb_eNoMemError, "No memory left for XPATH struct");

  pRbTxpath->iXmlType     = RUBY_XPATH_XMLSRC_TYPE_NULL;
  pRbTxpath->xXmlData     = Qnil;
  pRbTxpath->hxNS         = Qnil;
  pRbTxpath->rbResult     = Qnil;
  pRbTxpath->iCurrentNode = 0;
  pRbTxpath->iNbNodes     = 0;

  return( Data_Wrap_Struct( class, ruby_xpath_mark, ruby_xpath_free, pRbTxpath ) );
}

VALUE mXML;
VALUE cXPATH;

void Init_xpath( void ) {
  mXML  = rb_define_module( "XML" );
  cXPATH = rb_define_class_under( mXML, "XPATH", rb_cObject );

  rb_define_const( cXPATH, "XML_ELEMENT_NODE",       INT2NUM( 1 ) );
  rb_define_const( cXPATH, "XML_ATTRIBUTE_NODE",     INT2NUM( 2 ) );
  rb_define_const( cXPATH, "XML_TEXT_NODE",          INT2NUM( 3 ) );
  rb_define_const( cXPATH, "XML_CDATA_SECTION_NODE", INT2NUM( 4 ) );
  rb_define_const( cXPATH, "XML_ENTITY_REF_NODE",    INT2NUM( 5 ) );
  rb_define_const( cXPATH, "XML_ENTITY_NODE",        INT2NUM( 6 ) );
  rb_define_const( cXPATH, "XML_PI_NODE",            INT2NUM( 7 ) );
  rb_define_const( cXPATH, "XML_COMMENT_NODE",       INT2NUM( 8 ) );
  rb_define_const( cXPATH, "XML_DOCUMENT_NODE",      INT2NUM( 9 ) );
  rb_define_const( cXPATH, "XML_DOCUMENT_TYPE_NODE", INT2NUM( 10 ) );
  rb_define_const( cXPATH, "XML_DOCUMENT_FRAG_NODE", INT2NUM( 11 ) );
  rb_define_const( cXPATH, "XML_NOTATION_NODE",      INT2NUM( 12 ) );
  rb_define_const( cXPATH, "XML_HTML_DOCUMENT_NODE", INT2NUM( 13 ) );
  rb_define_const( cXPATH, "XML_DTD_NODE",           INT2NUM( 14 ) );
  rb_define_const( cXPATH, "XML_ELEMENT_DECL",       INT2NUM( 15 ) );
  rb_define_const( cXPATH, "XML_ATTRIBUTE_DECL",     INT2NUM( 16 ) );
  rb_define_const( cXPATH, "XML_ENTITY_DECL",        INT2NUM( 17 ) );
  rb_define_const( cXPATH, "XML_NAMESPACE_DECL",     INT2NUM( 18 ) );
  rb_define_const( cXPATH, "XML_XINCLUDE_START",     INT2NUM( 19 ) );
  rb_define_const( cXPATH, "XML_XINCLUDE_END",       INT2NUM( 20 ) );
  rb_define_const( cXPATH, "XML_DOCB_DOCUMENT_NODE", INT2NUM( 21 ) );
  rb_define_const( cXPATH, "RUBY_XPATH_VERSION",     rb_str_new2( RUBY_XPATH_VERSION ) );

  rb_define_singleton_method( cXPATH, "new", ruby_xpath_new, 0 );

  rb_define_method( cXPATH, "xml",      ruby_xpath_xml_str_get,  0 );
  rb_define_method( cXPATH, "xml=",     ruby_xpath_xml_str_set,  1 );
  rb_define_method( cXPATH, "xmlfile=", ruby_xpath_xml_file_set, 1 ); // DEPRECATED
  rb_define_method( cXPATH, "ns=",      ruby_xpath_xml_ns_set,   1 );

  rb_define_method( cXPATH, "execute",  ruby_xpath_xml_execute,  1 );
  rb_define_method( cXPATH, "to_a",     ruby_xpath_xml_to_a,     0 );
  
  rb_define_method( cXPATH, "each",     ruby_xpath_xml_each,     0 );
  rb_define_method( cXPATH, "first",    ruby_xpath_xml_first,    0 );
  rb_define_method( cXPATH, "last",     ruby_xpath_xml_last,     0 );
  rb_define_method( cXPATH, "next",     ruby_xpath_xml_next,     0 );
  rb_define_method( cXPATH, "prev",     ruby_xpath_xml_prev,     0 );

  rb_define_method( cXPATH, "name",     ruby_xpath_xml_name,     0 );
  rb_define_method( cXPATH, "attrs",    ruby_xpath_xml_attrs,    0 );
  rb_define_method( cXPATH, "type",     ruby_xpath_xml_type,     0 );
  rb_define_method( cXPATH, "content",  ruby_xpath_xml_content,  0 );
  rb_define_method( cXPATH, "path",     ruby_xpath_xml_path,     0 );
}
