/*
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "OSAllocator.h"
#include "PageBlock.h"

#if OS(WINRT)

#include "windows.h"
#include <wtf/Assertions.h>

// Try to use JIT by default and fallback to non-JIT on first error
static bool qt_winrt_use_jit = true;

namespace WTF {

inline size_t getPageSize()
{
    SYSTEM_INFO info;
    GetNativeSystemInfo(&info);
    return info.dwPageSize;
}

static inline DWORD protection(bool writable, bool executable)
{
    if (writable && executable)
        qFatal("read/write executable areas are not allowed on WinRT");
    return executable ?
                (writable ? PAGE_EXECUTE_READWRITE : PAGE_EXECUTE_READ) :
                (writable ? PAGE_READWRITE : PAGE_READONLY);
}

void* OSAllocator::reserveUncommitted(size_t bytes, Usage, bool writable, bool executable)
{
    void *result = VirtualAllocFromApp(0, bytes, MEM_RESERVE, protection(writable, executable));
    return result;
}

void* OSAllocator::reserveAndCommit(size_t bytes, Usage, bool writable, bool executable, bool includesGuardPages)
{
    void *result;
    result = VirtualAllocFromApp(0, bytes, MEM_RESERVE | MEM_COMMIT,
                                 protection(writable, executable));

    if (includesGuardPages && qt_winrt_use_jit) {
        size_t guardSize = pageSize();
        DWORD oldProtect;
        if (!VirtualProtectFromApp(result, guardSize, protection(false, false), &oldProtect) ||
            !VirtualProtectFromApp(static_cast<char*>(result) + bytes - guardSize, guardSize,
                                   protection(false, false), &oldProtect)) {
            CRASH();
        }
    }
    return result;
}

void OSAllocator::commit(void *bytes, size_t size, bool writable, bool executable)
{
    void *result = VirtualAllocFromApp(bytes, size, MEM_COMMIT,
                                       protection(writable, executable));
    if (!result)
        CRASH();
}

void OSAllocator::decommit(void* address, size_t bytes)
{
    bool result = VirtualFree(address, bytes, MEM_DECOMMIT);
    if (!result)
        CRASH();
}

void OSAllocator::releaseDecommitted(void* address, size_t)
{
    bool result = VirtualFree(address, 0, MEM_RELEASE);
    if (!result)
        CRASH();
}

bool OSAllocator::canAllocateExecutableMemory()
{
    if (qt_winrt_use_jit) {
        // For WinRT we first check if code generation is enabled. If successful
        // we allow to use JIT, otherwise fallback to the interpreter
        const size_t pageSize = getPageSize();
        void *all = VirtualAllocFromApp(0, pageSize, MEM_RESERVE | MEM_COMMIT,
                                        protection(true, false));
        DWORD oldProtect;
        bool res = VirtualProtectFromApp(all, pageSize, PAGE_EXECUTE, &oldProtect);
        VirtualFree(all, 0, MEM_RELEASE);
        if (!res) {
            qt_winrt_use_jit = false;
            qWarning("Could not enable JIT, fallback to interpreter mode. "
                     "Consider setting the code-generation capability");
        }
    }
    return qt_winrt_use_jit;
}


} // namespace WTF

#endif // OS(WINRT)
