/* -*- mode: c++; indent-tabs-mode: nil -*- */
/** @file ql_xml.qpp defines XML functions */
/*
    ql_xml.qpp

    Qore XML functions

    Qore Programming Language

    Copyright 2003 - 2022 Qore Technologies, s.r.o.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "qore-xml-module.h"

#include "QC_XmlDoc.h"
#include "QoreXmlReader.h"
#include "QoreXmlRpcReader.h"
#include "ql_xml.h"
#include "MakeXmlOpts.h"

#include <libxml/xmlwriter.h>

#include <string.h>
#include <memory>

// list of libxml2 element type names
static const char* xml_element_type_names[] = {
    "XML_ELEMENT_NODE",
    "XML_ATTRIBUTE_NODE",
    "XML_TEXT_NODE",
    "XML_CDATA_SECTION_NODE",
    "XML_ENTITY_REF_NODE",
    "XML_ENTITY_NODE",
    "XML_PI_NODE",
    "XML_COMMENT_NODE",
    "XML_DOCUMENT_NODE",
    "XML_DOCUMENT_TYPE_NODE",
    "XML_DOCUMENT_FRAG_NODE",
    "XML_NOTATION_NODE",
    "XML_HTML_DOCUMENT_NODE",
    "XML_DTD_NODE",
    "XML_ELEMENT_DECL",
    "XML_ATTRIBUTE_DECL",
    "XML_ENTITY_DECL",
    "XML_NAMESPACE_DECL",
    "XML_XINCLUDE_START",
    "XML_XINCLUDE_END",
    "XML_DOCB_DOCUMENT_NODE"
};

#define XETN_SIZE (sizeof(xml_element_type_names) / sizeof(char*))

static const char* xml_node_type_names[] = {
    "XML_NODE_TYPE_NONE",
    "XML_NODE_TYPE_ELEMENT",
    "XML_NODE_TYPE_ATTRIBUTE",
    "XML_NODE_TYPE_TEXT",
    "XML_NODE_TYPE_CDATA",
    "XML_NODE_TYPE_ENTITY_REFERENCE",
    "XML_NODE_TYPE_ENTITY",
    "XML_NODE_TYPE_PROCESSING_INSTRUCTION",
    "XML_NODE_TYPE_COMMENT",
    "XML_NODE_TYPE_DOCUMENT",
    "XML_NODE_TYPE_DOCUMENT_TYPE",
    "XML_NODE_TYPE_DOCUMENT_FRAGMENT",
    "XML_NODE_TYPE_NOTATION",
    "XML_NODE_TYPE_WHITESPACE",
    "XML_NODE_TYPE_SIGNIFICANT_WHITESPACE",
    "XML_NODE_TYPE_END_ELEMENT",
    "XML_NODE_TYPE_END_ENTITY",
    "XML_NODE_TYPE_XML_DECLARATION",};

#define XNTN_SIZE (sizeof(xml_node_type_names) / sizeof(char*))

const char* get_xml_element_type_name(int t) {
   return (t > 0 && t <= (int)XETN_SIZE) ? xml_element_type_names[t - 1] : 0;
}

const char* get_xml_node_type_name(int t) {
   return (t > 0 && t <= (int)XNTN_SIZE) ? xml_node_type_names[t - 1] : 0;
}

#if 0
// does not work well, produces ugly, uninformative output
static void qore_xml_structured_error_func(ExceptionSink* xsink, xmlErrorPtr error) {
   QoreStringNode* desc = new QoreStringNode;

   if (error->line)
      desc->sprintf("line %d: ", error->line);

   if (error->int2)
      desc->sprintf("column %d: ", error->int2);

   desc->concat(error->message);
   desc->chomp();

   if (error->str1)
      desc->sprintf(", %s", error->str1);

   if (error->str2)
      desc->sprintf(", %s", error->str1);

   if (error->str3)
      desc->sprintf(", %s", error->str1);

   xsink->raiseException("PARSE-XML-EXCEPTION", desc);
}
#endif

#ifdef HAVE_XMLTEXTREADERRELAXNGSETSCHEMA
static void qore_xml_relaxng_error_func(ExceptionSink* xsink, const char* msg, ...) {
   if (*xsink)
      return;

   va_list args;
   QoreStringNode* desc = new QoreStringNode;

   while (true) {
      va_start(args, msg);
      int rc = desc->vsprintf(msg, args);
      va_end(args);
      if (!rc)
         break;
   }
   desc->chomp();

   xsink->raiseException("RELAXNG-SYNTAX-ERROR", desc);
}
#endif

#ifdef HAVE_XMLTEXTREADERSETSCHEMA
static void qore_xml_schema_error_func(QoreXmlSchemaContext* xsdctx, const char* msg, ...) {
   if (*xsdctx->xsink)
      return;

   va_list args;
   QoreStringNode* desc = new QoreStringNode;

   while (true) {
      va_start(args, msg);
      int rc = desc->vsprintf(msg, args);
      va_end(args);
      if (!rc)
         break;
   }
   desc->chomp();

   xsdctx->xsink->raiseException("XSD-SYNTAX-ERROR", desc);
}
#endif

#if defined(HAVE_XMLTEXTREADERRELAXNGSETSCHEMA) || defined(HAVE_XMLTEXTREADERSETSCHEMA)
static void qore_xml_schema_warning_func(void* ignored, const char* msg, ...) {
#ifdef DEBUG
   va_list args;
   QoreString buf;

   while (true) {
      va_start(args, msg);
      int rc = buf.vsprintf(msg, args);
      va_end(args);
      if (!rc)
         break;
   }

   printf("%s", buf.c_str());
#endif
}
#endif

#ifdef HAVE_XMLTEXTREADERSETSCHEMA
static void qore_xml_schema_valid_error_func(ExceptionSink* xsink,
                                             const char* msg, ...) {
   if (*xsink)
      return;

   va_list args;
   QoreStringNode* desc = new QoreStringNode;

   while (true) {
      va_start(args, msg);
      int rc = desc->vsprintf(msg, args);
      va_end(args);
      if (!rc)
         break;
   }
   desc->chomp();

   xsink->raiseException("XSD-ERROR", desc);
}
#endif

#if defined(HAVE_XMLTEXTREADERRELAXNGSETSCHEMA) || defined(HAVE_XMLTEXTREADERSETSCHEMA)
static void qore_xml_schema_valid_warning_func(ExceptionSink* xsink,
                                               const char* msg, ...) {
#ifdef DEBUG
   va_list args;
   QoreString buf;

   while (true) {
      va_start(args, msg);
      int rc = buf.vsprintf(msg, args);
      va_end(args);
      if (!rc)
         break;
   }

   printf("%s", buf.c_str());
#endif
}
#endif

#ifdef HAVE_XMLTEXTREADERSETSCHEMA
QoreXmlSchemaContext::QoreXmlSchemaContext(const QoreString& xsd, ExceptionSink* xsink) : Utf8StringHelper(xsd, xsink) {
    if (*xsink)
        return;

    xmlSchemaParserCtxtPtr scp = xmlSchemaNewMemParserCtxt(utf8str->c_str(), utf8str->size());
    if (!scp) {
        xsink->raiseException("XSD-SYNTAX-ERROR", "XML schema provided for validation could not be parsed");
        return;
    }

    {
        ON_BLOCK_EXIT(xmlSchemaFreeParserCtxt, scp);

        //xmlSchemaSetParserStructuredErrors(scp, (xmlStructuredErrorFunc)qore_xml_structured_error_func, xsink);
        xmlSchemaSetParserErrors(scp, (xmlSchemaValidityErrorFunc)qore_xml_schema_error_func,
                                 (xmlSchemaValidityErrorFunc)qore_xml_schema_warning_func, this);
        // set exception context
        this->xsink = xsink;
        schema = xmlSchemaParse(scp);
    }

    if (!schema) {
        if (!*xsink)
            xsink->raiseException("XSD-SYNTAX-ERROR", "XML schema provided for validation could not be parsed");
        return;
    }
}

xmlSchemaValidCtxtPtr QoreXmlSchemaContext::getPtr() {
    if (!ctx) {
        ctx = xmlSchemaNewValidCtxt(schema);
        assert(ctx);
        xmlSchemaSetValidErrors(
            ctx,
            reinterpret_cast<xmlSchemaValidityErrorFunc>(
                qore_xml_schema_valid_error_func),
            reinterpret_cast<xmlSchemaValidityErrorFunc>(
                qore_xml_schema_valid_warning_func),
            xsink);
    }
    return ctx;
}
#endif

#ifdef HAVE_XMLTEXTREADERRELAXNGSETSCHEMA
QoreXmlRelaxNGContext::QoreXmlRelaxNGContext(const char* rng, int size, ExceptionSink* xsink) : schema(0) {
   xmlRelaxNGParserCtxtPtr rcp = xmlRelaxNGNewMemParserCtxt(rng, size);
   if (!rcp)
      return;

   xmlRelaxNGSetParserErrors(rcp, (xmlRelaxNGValidityErrorFunc)qore_xml_relaxng_error_func,
                             (xmlRelaxNGValidityErrorFunc)qore_xml_schema_warning_func, xsink);

   schema = xmlRelaxNGParse(rcp);
   xmlRelaxNGFreeParserCtxt(rcp);
}
#endif

static int concat_simple_value(ExceptionSink* xsink, QoreString &str, const QoreValue n, const MakeXmlOpts &opts) {
    //printd(5, "concat_simple_value() n: %p (%s) %s\n", n, n->getTypeName(), n->getType() == NT_STRING ? ((QoreStringNode*)n)->c_str() : "unknown");
    if (n.isNullOrNothing())
        return 0;

    qore_type_t t = n.getType();
    switch (t) {
        case NT_INT: {
            str.sprintf("%lld", n.getAsBigInt());
            return 0;
        }

        case NT_FLOAT: {
            str.sprintf("%.9g", n.getAsFloat());
            return 0;
        }

        case NT_NUMBER: {
            n.get<const QoreNumberNode>()->getStringRepresentation(str);
            return 0;
        }

        case NT_BOOLEAN: {
            str.sprintf("%d", n.getAsBool());
            return 0;
        }

        case NT_DATE: {
            const DateTimeNode* date = n.get<const DateTimeNode>();
            date->format(str, opts.m_dateFormat.c_str());
            return 0;
        }

        case NT_STRING:
            break;

        default:
            // issue #3588: throw an exception for unsupported types
            xsink->raiseException("MAKE-XML-ERROR", "cannot serialize type '%s'", n.getTypeName());
            return -1;
    }

    assert(t == NT_STRING);
    if (str.concatEncode(xsink, *n.get<const QoreStringNode>(),
                        CE_XML | (opts.m_useNumericRefs ? CE_NONASCII : 0)))
        return -1;
    return 0;
}

static int concat_simple_cdata_value(QoreString &str, const QoreValue n, ExceptionSink* xsink) {
   //printd(5, "concat_simple_cdata_value() n: %p (%s) %s\n", n, n->getTypeName(), n->getType() == NT_STRING ? ((QoreStringNode*)n)->c_str() : "unknown");
   if (n.getType() == NT_STRING) {
      const QoreStringNode* qsn = n.get<const QoreStringNode>();
      if (strstr(qsn->c_str(), "]]>")) {
         xsink->raiseException("MAKE-XML-ERROR", "CDATA text contains illegal ']]>' sequence");
         return -1;
      }
      str.concat(qsn, xsink);
      return *xsink ? -1 : 0;
   }

   return concat_simple_value(xsink, str, n, MakeXmlOpts());
}

static int concat_simple_comment(QoreString &str, const QoreValue n, ExceptionSink* xsink) {
   //printd(5, "concat_simple_comment() n: %p (%s) %s\n", n, n->getTypeName(), n->getType() == NT_STRING ? ((QoreStringNode*)n)->c_str() : "unknown");
   if (n.getType() == NT_STRING) {
      const QoreStringNode* qsn = n.get<const QoreStringNode>();
      if (strstr(qsn->c_str(), "-->")) {
         xsink->raiseException("MAKE-XML-ERROR", "Comment text contains illegal '-->' sequence");
         return -1;
      }
      str.concat(qsn, xsink);
      return *xsink ? -1 : 0;
   }

   return concat_simple_value(xsink, str, n, MakeXmlOpts());
}

static int make_xml(ExceptionSink* xsink, QoreString &str, const QoreHashNode &h, int indent, const MakeXmlOpts &opts);
static QoreStringNode* make_xml_intern(ExceptionSink* xsink, const QoreStringNode* pstr, const QoreHashNode* pobj, const MakeXmlOpts &opts);

QoreStringNode* make_xml(ExceptionSink* xsink, const QoreHashNode &h, const MakeXmlOpts &opts) {
   return make_xml_intern(xsink, nullptr, &h, opts);
}

static void add_xml_element(ExceptionSink* xsink, const char* key, QoreString &str, const QoreValue n, int indent, const MakeXmlOpts &opts) {
    //QORE_TRACE("add_xml_element()");

    if (n.isNullOrNothing()) {
        str.concat('<');
        str.concat(key);
        str.concat("/>");
        return;
    }

    qore_type_t ntype = n.getType();

    if (ntype == NT_LIST) {
        const QoreListNode* l = n.get<const QoreListNode>();
        // iterate through the list
        int ls = l->size();
        if (ls) {
            for (int j = 0; j < ls; ++j) {
                QoreValue v = l->retrieveEntry(j);
                // indent all but first entry if necessary
                if (j && opts.m_formatWithWhitespaces) {
                    str.concat('\n');
                    str.addch(' ', indent);
                }

                add_xml_element(xsink, key, str, v, indent, opts);
            }
        } else {    // close node
            str.concat('<');
            str.concat(key);
            str.concat("/>");
        }
        return;
    }

    // open node
    str.concat('<');
    str.concat(key);

    if (ntype == NT_HASH) {
        const QoreHashNode* h = n.get<const QoreHashNode>();
        // inc = ignore node counter, see if special keys exists and increment counter even if they have no value
        qore_size_t inc = 0;
        int vn = 0;
        bool exists;
        QoreValue value;
        QoreValue v = h->getKeyValueExistence("^value^", exists);
        if (exists) {
            ++vn;
            if (v.isNullOrNothing()) {
                ++inc;
            } else {
                value = v;
            }
            // find all ^value*^ nodes
            QoreString val;
            while (true) {
                val.sprintf("^value%d^", vn);
                v = h->getKeyValueExistence(val.c_str(), exists);
                if (!exists) {
                    break;
                } else if (v.isNullOrNothing()) { // if the node exists but there is no value, then skip
                    ++inc;
                } else {
                    value = v;
                }
                ++vn;
            }
        }

        int cn = 0;
        v = h->getKeyValueExistence("^comment^", exists);
        if (exists) {
            ++cn;
            if (v.isNullOrNothing()) {
                ++inc;
            }
            // find all ^comment*^ nodes
            QoreString val;
            while (true) {
                val.sprintf("^comment%d^", cn);
                v = h->getKeyValueExistence(val.c_str(), exists);
                if (!exists) {
                    break;
                } else if (v.isNullOrNothing()) { // if the node exists but there is no value, then skip
                    ++inc;
                }
                ++cn;
            }
        }
        QoreValue attrib = h->getKeyValueExistence("^attributes^", exists);
        if (exists)
            ++inc;

        // add attributes for objects
        if (attrib.getType() == NT_HASH) {
            const QoreHashNode* ah = attrib.get<const QoreHashNode>();
            // add attributes to node
            ConstHashIterator hi(ah);
            while (hi.next()) {
                const char* tkey = hi.getKey();
                str.sprintf(" %s=\"", tkey);
                QoreValue v = hi.get();
                if (!v.isNullOrNothing()) {
                    if (v.getType() == NT_STRING) {
                        if (str.concatEncode(xsink, v.get<const QoreStringNode>(),
                            CE_XML | (opts.m_useNumericRefs ? CE_NONASCII : 0))) {
                            return;
                        }
                    } else { // convert to string and add
                        QoreStringValueHelper temp(v);
                        str.concat(*temp, xsink);
                    }
                }
                str.concat('\"');
            }
        }

        //printd(5, "inc: %d vn: %d\n", inc, vn);

        // if there are no more elements, close node immediately
        if (h->size() == inc) {
            str.concat("/>");
            return;
        }

        // close node
        str.concat('>');

        // print value without CRLF
        if (!value.isNullOrNothing() && h->size() == (inc + 1)) {
            if (concat_simple_value(xsink, str, value, opts)) {
                return;
            }
        } else { // add additional elements and formatting only if the additional elements exist
            if (opts.m_formatWithWhitespaces && !vn) {
                str.concat('\n');
            }
            auto innerOpts = opts;
            if (vn) {
                innerOpts.m_formatWithWhitespaces = false;
            }
            make_xml(xsink, str, *h, indent + 2, innerOpts);
            // indent closing entry
            if (opts.m_formatWithWhitespaces && !vn) {
                str.concat('\n');
                str.addch(' ', indent);
            }
        }
    } else {
        // close node
        str.concat('>');
        concat_simple_value(xsink, str, n, opts);
    }

    // close node
    str.concat("</");
    str.concat(key);
    str.concat('>');
}

static int make_xml(ExceptionSink* xsink, QoreString &str, const QoreHashNode &h, int indent, const MakeXmlOpts &opts) {
    QORE_TRACE("make_xml()");

    ConstHashIterator hi(h);
    bool done = false;
    while (hi.next()) {
        std::unique_ptr<QoreString> keyStr(hi.getKeyString());
        // convert string if needed
        if (keyStr->getEncoding() != str.getEncoding()) {
            QoreString* ns = keyStr->convertEncoding(str.getEncoding(), xsink);
            if (xsink->isEvent())
                return -1;
            keyStr.reset(ns);
        }

        const char* key = keyStr->c_str();
        if (!strcmp(key, "^attributes^"))
            continue;

        if (!strncmp(key, "^value", 6)) {
            if (concat_simple_value(xsink, str, hi.get(), opts))
                return -1;
            continue;
        }

        if (!strncmp(key, "^cdata", 6)) {
            str.concat("<![CDATA[");
            if (concat_simple_cdata_value(str, hi.get(), xsink))
                return -1;
            str.concat("]]>");
            continue;
        }

        if (!strncmp(key, "^comment", 8)) {
            if (opts.m_formatWithWhitespaces) {
                if (done) {
                str.concat('\n');
                }
                str.addch(' ', indent);
            }
            str.concat("<!--");
            if (concat_simple_comment(str, hi.get(), xsink))
                return -1;
            str.concat("-->");
            done = true;
            continue;
        }

        // make sure it's a valid XML tag element name
        if (!key || (!isalpha(key[0]) && key[0] != '_')) {
            xsink->raiseException("MAKE-XML-ERROR", "tag: \"%s\" is not a valid XML tag element name", key ? key : "");
            return -1;
        }

        // process key name - remove ^# from end of key name if present
        qore_size_t l = keyStr->strlen() - 1;
        while (isdigit(key[l]))
            l--;

        if (l != (keyStr->strlen() - 1) && key[l] == '^')
            keyStr->terminate(l);

        // indent entry
        if (opts.m_formatWithWhitespaces) {
            if (done)
                str.concat('\n');
            str.addch(' ', indent);
        }
        //printd(5, "make_xml() level %d adding member %s\n", indent / 2, node->c_str());
        add_xml_element(xsink, key, str, hi.get(), indent, opts);
        done = true;
    }

    return 0;
}

// returns top-level key name
static bool hash_ok(const QoreHashNode* h) {
   int count = 0;

   ConstHashIterator hi(h);

   while (hi.next()) {
      const char* k = hi.getKey();
      if (!k[0] || k[0] == '^')
         continue;
      if (++count > 1)
         break;
      QoreValue n = hi.get();
      if (n.getType() == NT_LIST) {
         // top level hash value of list type does not generate valid xml as one root element is mandatory
         const QoreListNode *l = n.get<const QoreListNode>();
         if (l->size() > 1) {
            return false;
         }
      }
   }
   return count == 1;
}

static QoreStringNode* make_xml_intern(ExceptionSink* xsink, const QoreStringNode* pstr, const QoreHashNode* pobj, const MakeXmlOpts &opts) {
    SimpleRefHolder<QoreStringNode> str(new QoreStringNode(opts.m_encoding));
    str->sprintf("<?xml version=\"%s\" encoding=\"%s\"?>", opts.m_docVersion.c_str(), opts.m_encoding->getCode());
    str->concat('\n'); // always separate the header with new line
    if (pstr) {
        TempEncodingHelper key(pstr, QCS_UTF8, xsink);
        if (!key) {
            return 0;
        }
        add_xml_element(xsink, key->c_str(), *(*str), pobj, 0, opts);
    } else {
        make_xml(xsink, *(*str), *pobj, 0, opts);
    }

    str->concat('\n'); // add new line after last line of xml

    //printd(5, "make_xml_intern() returning %s\n", str->c_str());

    return str.release();
}

static QoreStringNode* make_xml_intern(ExceptionSink* xsink, const QoreStringNode* pstr, const QoreHashNode* pobj, const QoreEncoding* ccs, int flags = XGF_NONE) {
    auto opts = MakeXmlOpts::createFromFlags(flags, ccs);
    return make_xml_intern(xsink, pstr, pobj, opts);
}

static int add_xmlrpc_value(QoreString* str, const QoreValue n, int indent, const QoreEncoding* ccs, int flags, ExceptionSink* xsink);

#define EMPTY_KEY_STRING "!!empty-hash-key!!"

static int add_xmlrpc_value_intern_hash(QoreString* str, const QoreHashNode* h, int indent, const QoreEncoding* ccs, int flags, ExceptionSink* xsink) {
   bool fmt = flags & XGF_ADD_FORMATTING;

   str->concat("<struct>");
   if (fmt)
      str->concat('\n');
   ConstHashIterator hi(h);
   while (hi.next()) {
      std::unique_ptr<QoreString> member(hi.getKeyString());
      // here we allow hashes with empty key names to be serialized
      if (!member->strlen())
         member.reset(new QoreString(EMPTY_KEY_STRING, ccs));

      // convert string if needed
      if (member->getEncoding() != ccs) {
         QoreString* ns = member->convertEncoding(ccs, xsink);
         if (*xsink) {
            assert(!ns);
            return -1;
         }
         //printd(5, "add_xmlrpc_value_intern_hash() converted %s->%s, \"%s\"->\"%s\"\n", member->getEncoding()->getCode(), ccs->getCode(), member->c_str(), ns->c_str());
         member.reset(ns);
      }
      //else printd(5, "add_xmlrpc_value_intern_hash() not converting %sx \"%s\"\n", member->getEncoding()->getCode(), member->c_str());
      // indent
      if (fmt)
         str->addch(' ', indent + 2);
      str->concat("<member>");
      if (fmt) {
         str->concat('\n');
         // indent
         str->addch(' ', indent + 4);
      }
      str->concat("<name>");
      if (str->concatEncode(xsink, *member.get(), CE_XML | (flags & XGF_ENCODE_MASK)))
         return -1;

      member.reset();

      str->concat("</name>");
      if (fmt)
         str->concat('\n');
      QoreValue val = hi.get();
      if (add_xmlrpc_value(str, val, indent + 4, ccs, flags, xsink))
         return -1;
      // indent
      if (fmt)
         str->addch(' ', indent + 2);
      str->concat("</member>");
      if (fmt)
         str->concat('\n');
   }
   // indent
   if (fmt)
      str->addch(' ', indent);
   str->concat("</struct>");

   return 0;
}

static int add_xmlrpc_value_intern(QoreString* str, const QoreValue n, int indent, const QoreEncoding* ccs, int flags, ExceptionSink* xsink) {
    assert(!n.isNullOrNothing());
    qore_type_t ntype = n.getType();

    bool fmt = flags & XGF_ADD_FORMATTING;

    if (ntype == NT_BOOLEAN)
        str->sprintf("<boolean>%d</boolean>", (int)n.getAsBool());

    else if (ntype == NT_INT) {
        int64 val = n.getAsBigInt();
        if (val >= -2147483647 && val <= 2147483647)
            str->sprintf("<i4>%lld</i4>", val);
        else
            str->sprintf("<string>%lld</string>", val);
    }

    else if (ntype == NT_STRING) {
        str->concat("<string>");
        if (str->concatEncode(xsink, *n.get<const QoreStringNode>(), CE_XML | (flags & XGF_ENCODE_MASK)))
            return -1;
        str->concat("</string>");
    }

    else if (ntype == NT_FLOAT)
        str->sprintf("<double>%.20g</double>", n.getAsFloat());

    else if (ntype == NT_NUMBER) {
        str->concat("<double>");
        n.get<const QoreNumberNode>()->getStringRepresentation(*str);
        str->concat("</double>");
    }

    else if (ntype == NT_DATE) {
        str->concat("<dateTime.iso8601>");
        str->concatISO8601DateTime(n.get<const DateTimeNode>());
        str->concat("</dateTime.iso8601>");
    }

    else if (ntype == NT_BINARY) {
        str->concat("<base64>");
        if (fmt) {
            str->concat('\n');
            // indent
            str->addch(' ', indent + 4);
        }
        str->concatBase64(n.get<const BinaryNode>());
        if (fmt) {
            str->concat('\n');
            // indent
            str->addch(' ', indent);
        }
        str->concat("</base64>");
    }

    else if (ntype == NT_HASH) {
        if (add_xmlrpc_value_intern_hash(str, n.get<const QoreHashNode>(), indent + 2, ccs, flags, xsink))
            return -1;
    }

    else if (ntype == NT_LIST) {
        const QoreListNode* l = n.get<const QoreListNode>();
        str->concat("<array>");
        if (fmt) {
            str->concat('\n');
            // indent
            str->addch(' ', indent + 4);
        }
        if (l->size()) {
            str->concat("<data>");
            if (fmt)
                str->concat('\n');
            for (unsigned i = 0; i < l->size(); ++i) {
                if (add_xmlrpc_value(str, l->retrieveEntry(i), indent + 6, ccs, flags, xsink))
                    return -1;
            }
            if (fmt)
                str->addch(' ', indent + 4);
            str->concat("</data>");
        }
        else
            str->concat("<data/>");
        if (fmt) {
            str->concat('\n');
            // indent
            str->addch(' ', indent + 2);
        }
        str->concat("</array>");
        //if (flags & XGF_ADD_FORMATTING) str->concat('\n');
    }
    else {
        xsink->raiseException("XMLRPC-SERIALIZATION-ERROR", "don't know how to serialize type '%s' to XML-RPC", n.getTypeName());
        return -1;
    }

    if (fmt) {
        str->concat('\n');
        // indent
        str->addch(' ' , indent);
    }

    return 0;
}

static int add_xmlrpc_value(QoreString* str, const QoreValue n, int indent, const QoreEncoding* ccs, int flags, ExceptionSink* xsink) {
   QORE_TRACE("add_xmlrpc_value()");

   bool fmt = flags & XGF_ADD_FORMATTING;

   // add value node
   // indent
   if (fmt)
      str->addch(' ', indent);

   if (!n.isNullOrNothing()) {
      str->concat("<value>");
      if (fmt) {
         str->concat('\n');
         // indent
         str->addch(' ', indent + 2);
      }

      if (add_xmlrpc_value_intern(str, n, indent, ccs, flags, xsink))
         return -1;

      // close value node
      str->concat("</value>");
   }
   else
      str->concat("<value/>");
   if (fmt)
      str->concat('\n');

   return 0;
}

QoreStringNode* make_xmlrpc_call(ExceptionSink* xsink, const QoreEncoding* ccs, int offset, const QoreListNode* params, int flags) {
    const QoreStringNode* p0 = HARD_QORE_VALUE_STRING(params, offset);

    bool fmt = XGF_ADD_FORMATTING & flags;

    QoreStringNodeHolder str(new QoreStringNode(ccs));
    str->sprintf("<?xml version=\"1.0\" encoding=\"%s\"?>%s<methodCall>%s<methodName>", ccs->getCode(), fmt ? "\n" : "", fmt ? "\n  " : "");
    if (str->concatEncode(xsink, *p0, CE_XML | (XGF_ENCODE_MASK & flags)))
        return 0;

    str->sprintf("</methodName>%s", fmt ? "\n" : "");

    // now process all arguments
    int ls = num_args(params);
    if (ls > offset) {
        str->sprintf("%s<params>%s", fmt ? "  " : "", fmt ? "\n" : "");

        for (int i = offset + 1; i < ls; ++i) {
            const QoreValue p = get_param_value(params, i);
            str->sprintf("%s<param>%s", fmt ? "    " : "", fmt ? "\n" : "");
            if (add_xmlrpc_value(*str, p, fmt ? 6 : 0, ccs, flags, xsink))
                return 0;
            str->sprintf("%s</param>%s", fmt ? "    " : "", fmt ? "\n" : "");
        }
        str->sprintf("%s</params>%s", fmt ? "  " : "", fmt ? "\n" : "");
    } else {
        str->sprintf("%s<params/>%s", fmt ? "  " : "", fmt ? "\n" : "");
    }
    str->concat("</methodCall>");

    return str.release();
}

QoreStringNode* make_xmlrpc_call_args(ExceptionSink* xsink, const QoreEncoding* ccs, int offset, const QoreListNode* params, int flags) {
    const QoreStringNode* p0 = HARD_QORE_VALUE_STRING(params, offset);

    bool fmt = XGF_ADD_FORMATTING & flags;

    QoreStringNodeHolder str(new QoreStringNode(ccs));
    str->sprintf("<?xml version=\"1.0\" encoding=\"%s\"?>%s<methodCall>%s<methodName>", ccs->getCode(), fmt ? "\n" : "", fmt ? "\n  " : "");
    if (str->concatEncode(xsink, *p0, CE_XML | (XGF_ENCODE_MASK & flags)))
        return 0;

    str->sprintf("</methodName>%s", fmt ? "\n" : "");

    const QoreValue p1 = get_param_value(params, offset + 1);
    if (p1.getType() == NT_LIST) {
        const QoreListNode* l = p1.get<const QoreListNode>();
        if (l->size()) {
            str->sprintf("%s<params>%s", fmt ? "  " : "", fmt ? "\n" : "");

            for (unsigned int i = 0 ; i < l->size(); ++i) {
                QoreValue p = l->retrieveEntry(i);
                str->sprintf("%s<param>%s", fmt ? "    " : "", fmt ? "\n" : "");
                if (add_xmlrpc_value(*str, p, fmt ? 6 : 0, ccs, flags, xsink))
                return 0;
                str->sprintf("%s</param>%s", fmt ? "    " : "", fmt ? "\n" : "");
            }
            str->sprintf("%s</params>%s", fmt ? "  " : "", fmt ? "\n" : "");
        }
    } else if (!p1.isNullOrNothing()) {
        str->sprintf("<params>%s", fmt ? "\n" : "");
        str->sprintf("%s<param>%s", fmt ? "    " : "", fmt ? "\n" : "");
        if (add_xmlrpc_value(*str, p1, fmt ? 6 : 0, ccs, 0, xsink))
            return 0;
        str->sprintf("%s</param>%s", fmt ? "    " : "", fmt ? "\n" : "");
        str->sprintf("%s</params>%s", fmt ? "  " : "", fmt ? "\n" : "");
    } else {
        str->sprintf("%s<params/>%s", fmt ? "  " : "", fmt ? "\n" : "");
    }
    str->concat("</methodCall>");

    return str.release();
}

// NOTE: the libxml2 library requires all input to be in UTF-8 encoding
static AbstractQoreNode* parse_xml_intern(int pflags, const QoreListNode* args, ExceptionSink* xsink) {
   const QoreStringNode* p0 = HARD_QORE_VALUE_STRING(args, 0);

   const QoreEncoding* ccsid = get_value_encoding_param(args, 1);

   //printd(5, "parse_xml_intern(%d, %s)\n", as_data, p0->c_str());

   // convert to UTF-8
   TempEncodingHelper str(p0, QCS_UTF8, xsink);
   if (!str)
      return 0;

   QoreXmlReader reader(*str, QORE_XML_PARSER_OPTIONS, xsink);
   if (!reader)
      return 0;

   return reader.parseXmlData(ccsid, pflags, xsink);
}

static AbstractQoreNode* make_xmlrpc_fault(ExceptionSink* xsink, const QoreEncoding* ccs, int code, const QoreStringNode* p1, int flags = 0) {
   QORE_TRACE("make_xmlrpc_fault()");

   if (!ccs) ccs = p1->getEncoding();
   //printd(5, "ccsid: %016x (%s) (%s) code: %d\n", ccsid, ccsid->getCode(), ((QoreStringNode*)p1)->c_str(), code);

   bool fmt = XGF_ADD_FORMATTING & flags;

   // for speed, the XML is created directly here
   QoreStringNodeHolder str(new QoreStringNode(ccs));
   if (fmt)
      str->sprintf("<?xml version=\"1.0\" encoding=\"%s\"?>\n<methodResponse>\n  <fault>\n    <value>\n      <struct>\n        <member>\n          <name>faultCode</name>\n          <value><int>%d</int></value>\n        </member>\n        <member>\n          <name>faultString</name>\n          <value><string>",
                ccs->getCode(), code);
   else
      str->sprintf("<?xml version=\"1.0\" encoding=\"%s\"?><methodResponse><fault><value><struct><member><name>faultCode</name><value><int>%d</int></value></member><member><name>faultString</name><value><string>",
                   ccs->getCode(), code);

   str->concatEncode(xsink, *p1, CE_XML | (XGF_ENCODE_MASK & flags));
   if (*xsink)
       return 0;

   if (fmt)
      str->concat("</string></value>\n        </member>\n      </struct>\n    </value>\n  </fault>\n</methodResponse>");
   else
      str->concat("</string></value></member></struct></value></fault></methodResponse>");

   return str.release();
}

static AbstractQoreNode* make_xmlrpc_response(ExceptionSink* xsink, const QoreEncoding* ccs, int offset, const QoreListNode* args, int flags = 0) {
   QORE_TRACE("make_xmlrpc_response()");

   unsigned na = num_args(args);
   if (na == (unsigned)offset)
      return 0;

   bool fmt = XGF_ADD_FORMATTING & flags;

   QoreStringNodeHolder str(new QoreStringNode(ccs));
   str->sprintf("<?xml version=\"1.0\" encoding=\"%s\"?>", ccs->getCode());
   if (fmt)
      str->concat("\n<methodResponse>\n  <params>\n");
   else
      str->concat("<methodResponse><params>");

   // now loop through the params
   for (int i = offset, end = na - offset; i < end; ++i) {
      const QoreValue p = get_param_value(args, i);
      if (fmt)
         str->concat("    <param>\n");
      else
         str->concat("<param>");
      if (add_xmlrpc_value(*str, p, fmt ? 6 : 0, ccs, flags, xsink))
         return 0;

      if (fmt)
         str->concat("    </param>\n");
      else
         str->concat("</param>");
   }

   if (fmt)
      str->concat("  </params>\n</methodResponse>");
   else
      str->concat("</params></methodResponse>");

   return str.release();
}

static AbstractQoreNode* make_xmlrpc_response_args(ExceptionSink* xsink, const QoreEncoding* ccs, const QoreValue arg, int flags = 0) {
    QORE_TRACE("make_xmlrpc_response_args()");

    bool fmt = XGF_ADD_FORMATTING & flags;

    QoreStringNodeHolder str(new QoreStringNode(ccs));
    str->sprintf("<?xml version=\"1.0\" encoding=\"%s\"?>", ccs->getCode());
    if (fmt)
        str->concat("\n<methodResponse>\n  <params>\n");
    else
        str->concat("<methodResponse><params>");

    if (!arg.isNullOrNothing()) {
        if (arg.getType() == NT_LIST) {
            const QoreListNode* l = arg.get<QoreListNode>();
            // now loop through the params
            for (unsigned int i = 0; i < l->size(); ++i) {
                QoreValue p = l->retrieveEntry(i);
                if (fmt)
                str->concat("    <param>\n");
                else
                str->concat("<param>");
                if (add_xmlrpc_value(*str, p, fmt ? 6 : 0, ccs, flags, xsink))
                return 0;
                if (fmt)
                str->concat("    </param>\n");
                else
                str->concat("</param>");
            }
        } else {
            if (fmt)
                str->concat("    <param>\n");
            else
                str->concat("<param>");
            if (add_xmlrpc_value(*str, arg, fmt ? 6 : 0, ccs, flags, xsink))
                return 0;

            if (fmt)
                str->concat("    </param>\n");
            else
                str->concat("</param>");
        }
    }

    if (fmt)
        str->concat("  </params>\n</methodResponse>");
    else
        str->concat("</params></methodResponse>");

    return str.release();
}

QoreStringNode* make_xmlrpc_value(ExceptionSink* xsink, QoreValue value, const QoreStringNode* encoding, int flags = 0) {
    if (value.isNullOrNothing()) {
        return 0;
    }

    const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;

    QoreStringNodeHolder rv(new QoreStringNode(qe));
    if (add_xmlrpc_value_intern(*rv, value, 0, qe, flags, xsink))
        return 0;

    return rv.release();
}

static AbstractQoreNode* qore_xml_exception(const char* ex, const char* info, ExceptionSink* xsink) {
    if (!*xsink)
        xsink->raiseException(ex, "error parsing XML string: %s", info);
    return 0;
}

static AbstractQoreNode* qore_xml_exception(const char* ex, ExceptionSink* xsink) {
    if (!*xsink)
        xsink->raiseException(ex, "error parsing XML string");
    return 0;
}

static QoreHashNode* qore_xml_hash_exception(const char* ex, const char* info, ExceptionSink* xsink, const QoreString* xml = 0) {
    if (!*xsink)
        xsink->raiseExceptionArg(ex, xml ? new QoreStringNode(*xml) : 0, "error parsing XML string: %s", info);
    return 0;
}

/*
static QoreHashNode* qore_xml_hash_exception(const char* ex, ExceptionSink* xsink, const QoreString* xml = 0) {
    if (!*xsink)
        xsink->raiseExceptionArg(ex, xml ? new QoreStringNode(*xml) : 0, "error parsing XML string");
    return 0;
}
*/

// parseXMLRPCResponse
QoreHashNode* parse_xmlrpc_response(ExceptionSink* xsink, const QoreString* msg, const QoreEncoding* ccsid, int flags) {
   //printd(5, "parse_xmlrpc_response() %s\n", msg->c_str());

   TempEncodingHelper str(msg, QCS_UTF8, xsink);
   if (!str)
      return 0;

   QoreXmlRpcReader reader(*str, QORE_XML_PARSER_OPTIONS, xsink);
   if (!reader)
      return 0;

   if (reader.read(xsink) != 1)
      return 0;

   int nt;
   // get "methodResponse" element
   if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_ELEMENT)
       return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'methodResponse' element", xsink, *str);

   if (reader.checkXmlRpcMemberName("methodResponse", xsink))
      return 0;

   // check for params or fault element
   if (reader.readXmlRpc("expecting 'params' or 'fault' element", xsink))
      return 0;

   if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_ELEMENT)
       return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'params' or 'fault' element", xsink, *str);

   const char* name = reader.constName();
   if (!name) {
      xsink->raiseExceptionArg("PARSE-XMLRPC-RESPONSE-ERROR", new QoreStringNode(*str), "missing 'params' or 'fault' element tag");
      return 0;
   }

   Qore::Xml::intern::XmlRpcValue v;
   bool fault = false;
   if (!strcmp(name, "params")) {
      int depth = reader.depth();

      // get "params" element
      if (reader.readXmlRpc("expecting 'params' element", xsink))
         return 0;

      int params_depth = reader.depth();

      // if params was not an empty element
      if (depth < params_depth) {
         if ((nt = reader.readXmlRpcNode(xsink)) == -1)
            return 0;

         if (nt != XML_READER_TYPE_END_ELEMENT) {
            if (nt != XML_READER_TYPE_ELEMENT)
               return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'param' element", xsink, *str);

            if (reader.checkXmlRpcMemberName("param", xsink))
               return 0;

            // get "value" element
            if (reader.readXmlRpc("expecting 'value' element", xsink))
               return 0;

            // if param was not an empty element
            depth = reader.depth();
            if (params_depth < depth) {
               if ((nt = reader.readXmlRpcNode(xsink)) == -1)
                  return 0;

               if (nt != XML_READER_TYPE_END_ELEMENT) {
                  if (nt != XML_READER_TYPE_ELEMENT)
                     return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'value' element", xsink, *str);

                  if (reader.checkXmlRpcMemberName("value", xsink))
                     return 0;

                  // position at next element
                  if (reader.readXmlRpc("expecting XML-RPC value element", xsink))
                     return 0;

                  // if value was not an empty element
                  if (depth < reader.depth()) {
                     if (reader.getValueData(&v, ccsid, true, xsink))
                        return 0;

                     //printd(5, "parseXMLRPCResponse() name: %s reader.nodeType: %d\n", reader.constName(), reader.nodeType());

                     if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_END_ELEMENT) {
                        //printd(5, "nt: %d name: %s\n", nt, reader.constName());
                        return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "2: expecting 'value' end element", xsink, *str);
                     }

                     if (reader.checkXmlRpcMemberName("value", xsink))
                        return 0;

                     if (reader.readSkipWhitespace("expecting 'param' end element", xsink) == -1)
                        return 0;
                  }

                  //printd(5, "parseXMLRPCResponse() (expecing param) name: %s reader.nodeType: %d (nt: %d)\n", reader.constName(), reader.nodeType(), nt);

                  if ((nt = reader.nodeType()) != XML_READER_TYPE_END_ELEMENT)
                     return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'param' end element", xsink, *str);

                  if (reader.checkXmlRpcMemberName("param", xsink))
                     return 0;
               }

               // get "params" end element
               if (reader.readXmlRpc("expecting 'params' end element", xsink))
                  return 0;
            }
            if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_END_ELEMENT)
               return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'params' end element", xsink, *str);
         }
         // get "methodResponse" end element
         if (reader.readXmlRpc("expecting 'methodResponse' end element", xsink))
            return 0;
      }
   }
   else if (!strcmp(name, "fault")) {
      fault = true;

      // get "value" element
      if (reader.readXmlRpc("expecting 'value' element", xsink))
         return 0;

      if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_ELEMENT)
         return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting fault 'value' element", xsink, *str);

      if (reader.checkXmlRpcMemberName("value", xsink))
         return 0;

      // position at next element
      if (reader.readXmlRpc("expecting XML-RPC value element", xsink))
         return 0;

      // get fault structure
      if (reader.getValueData(&v, ccsid, true, xsink))
         return 0;

      if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_END_ELEMENT)
         return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'value' end element", xsink, *str);

      if (reader.readSkipWhitespace("expecting 'fault' end element", xsink) == -1)
         return 0;

      if ((nt = reader.nodeType()) != XML_READER_TYPE_END_ELEMENT)
         return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'fault' end element", xsink, *str);

      // get "methodResponse" end element
      if (reader.readXmlRpc("expecting 'methodResponse' end element", xsink))
         return 0;
   }
   else {
      xsink->raiseException("PARSE-XMLRPC-RESPONSE-ERROR", "unexpected element '%s', expecting 'params' or 'fault'", name, *str);
      return 0;
   }

   if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_END_ELEMENT)
      return qore_xml_hash_exception("PARSE-XMLRPC-RESPONSE-ERROR", "expecting 'methodResponse' end element", xsink, *str);

   QoreHashNode* h = new QoreHashNode(autoTypeInfo);
   if (fault)
      h->setKeyValue("fault", v.getValue(), 0);
   else
      h->setKeyValue("params", v.getValue(), 0);
   return h;
}

// NOTE: the libxml2 library requires all input to be in UTF-8 encoding
// syntax: parse_xml_with_schema_intern(xml_string, xsd_string [, output encoding])
static AbstractQoreNode* parse_xml_with_schema_intern(ExceptionSink* xsink, bool with_flags, const QoreListNode* args, int flags = 0) {
// NOTE: the libxml2 library requires all input to be in UTF-8
#ifdef HAVE_XMLTEXTREADERSETSCHEMA
    const QoreStringNode* p0 = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* xsd = HARD_QORE_VALUE_STRING(args, 1);

    const QoreEncoding* ccsid;
    if (with_flags) {
        flags |= get_param_value(args, 2).getAsBigInt();
        ccsid = get_value_encoding_param(args, 3);
    }
    else
        ccsid = get_value_encoding_param(args, 2);

    printd(5, "parse_xml_with_schema_intern() xml: %s\n xsd: %s\n", p0->c_str(), xsd->c_str());

    // convert to UTF-8
    TempEncodingHelper str(p0, QCS_UTF8, xsink);
    if (!str)
        return nullptr;

    QoreXmlSchemaContext schema(*xsd, xsink);
    if (!schema) {
        assert(*xsink);
        return nullptr;
    }

    QoreXmlReader reader(*str, QORE_XML_PARSER_OPTIONS, xsink);
    if (!reader)
        return nullptr;

    int rc = reader.setSchema(schema.getSchema());
    if (rc < 0) {
        if (!*xsink)
        xsink->raiseException("XSD-VALIDATION-ERROR", "XML schema could not be validated");
        return nullptr;
    }

    return reader.parseXmlData(ccsid, flags, xsink);
#else
    xsink->raiseException("MISSING-FEATURE-ERROR", "the libxml2 version used to compile the xml module did not support the xmlTextReaderSetSchema() function, therefore XSD validation functionality is not available; for maximum portability, use the constant Option::HAVE_PARSEXMLWITHSCHEMA to check if this function is implemented before calling");
    return nullptr;
#endif
}

// NOTE: the libxml2 library requires all input to be in UTF-8 encoding
// syntax: parse_xml_with_relaxng_intern(xml_string, xsd_string [, output encoding])
static AbstractQoreNode* parse_xml_with_relaxng_intern(ExceptionSink* xsink, bool with_flags, const QoreListNode* args, int flags = 0) {
#ifdef HAVE_XMLTEXTREADERRELAXNGSETSCHEMA
   const QoreStringNode* p0 = HARD_QORE_VALUE_STRING(args, 0);
   const QoreStringNode* p1 = HARD_QORE_VALUE_STRING(args, 1);

   const QoreEncoding* ccsid;
   if (with_flags) {
      flags |= get_param_value(args, 2).getAsBigInt();
      ccsid = get_value_encoding_param(args, 3);
   }
   else
      ccsid = get_value_encoding_param(args, 2);

   printd(5, "parse_xml_with_relaxng_intern() xml: %s\n xsd: %s\n", p0->c_str(), p1->c_str());

   // convert to UTF-8
   TempEncodingHelper str(p0, QCS_UTF8, xsink);
   if (!str)
      return 0;

   TempEncodingHelper rng(p1, QCS_UTF8, xsink);
   if (!rng)
      return 0;

   QoreXmlRelaxNGContext schema(rng->c_str(), rng->strlen(), xsink);
   if (!schema) {
      if (!*xsink)
         xsink->raiseException("RELAXNG-SYNTAX-ERROR", "RelaxNG schema passed as second argument to parseXMLWithRelaxNG() could not be parsed");
      return 0;
   }

   QoreXmlReader reader(*str, QORE_XML_PARSER_OPTIONS, xsink);
   if (!reader)
      return 0;

   int rc = reader.setRelaxNG(schema.getSchema());
   if (rc < 0) {
      if (!*xsink)
         xsink->raiseException("RELAXNG-VALIDATION-ERROR", "RelaxNG schema passed as second argument to parseXMLWithRelaxNG() could not be validated");
      return 0;
   }

   return reader.parseXmlData(ccsid, flags, xsink);
#else
   xsink->raiseException("MISSING-FEATURE-ERROR", "the libxml2 version used to compile the xml module did not support the xmlTextReaderSetRelaxNG() function, therefore parseXMLWithRelaxNG() and parseXMLAsDataWithRelaxNG() are not available; for maximum portability, use the constant Option::HAVE_PARSEXMLWITHRELAXNG to check if this function is implemented before calling");
   return 0;
#endif
}

static QoreHashNode* parse_xmlrpc_call(const QoreStringNode* xml, const QoreStringNode* encoding, ExceptionSink* xsink) {
    const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;

    //printd(5, "parse_xmlrpc_call() c: %d str: %s\n", p0->c_str()[0], p0->c_str());

    TempEncodingHelper str(xml, QCS_UTF8, xsink);
    if (!str)
        return 0;

    QoreXmlRpcReader reader(*str, QORE_XML_PARSER_OPTIONS, xsink);
    if (!reader)
        return 0;

    if (reader.read(xsink) != 1)
        return 0;

    int nt;
    // get "methodCall" element
    if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_ELEMENT)
        return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting 'methodCall' element", xsink);

    if (reader.checkXmlRpcMemberName("methodCall", xsink))
        return 0;

    // get "methodName" element
    if (reader.readXmlRpc("expecting methodName element", xsink))
        return 0;

    if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_ELEMENT)
        return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting 'methodName' element", xsink);

    if (reader.checkXmlRpcMemberName("methodName", xsink))
        return 0;

    // get method name string
    if (reader.readXmlRpc("expecting method name", xsink))
        return 0;

    if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_TEXT)
        return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting method name", xsink);

    const char* method_name = reader.constValue();
    if (!method_name)
        return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting method name", xsink);

    ReferenceHolder<QoreHashNode> h(new QoreHashNode(autoTypeInfo), xsink);
    h->setKeyValue("methodName", new QoreStringNode(method_name), 0);

    // get methodName close tag
    if (reader.readXmlRpc("expecting methodName close element", xsink))
        return 0;

    if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_END_ELEMENT)
        return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting 'methodName' close element", xsink);

    // get "params" element
    if (reader.readXmlRpc("expecting params element", xsink))
        return 0;

    if ((nt = reader.readXmlRpcNode(xsink)) == -1)
        return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", xsink);

    // if the methodCall end element was not found
    if (nt != XML_READER_TYPE_END_ELEMENT) {
        if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_ELEMENT)
            return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting 'params' element", xsink);

        if (reader.checkXmlRpcMemberName("params", xsink))
            return 0;

        // get 'param' element or close params
        if (reader.readXmlRpc("expecting param element", xsink))
            return 0;

        if ((nt = reader.readXmlRpcNode(xsink)) == -1)
            return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", xsink);

        Qore::Xml::intern::XmlRpcValue v;
        if (reader.depth()) {
            if (nt != XML_READER_TYPE_END_ELEMENT) {
                if (nt != XML_READER_TYPE_ELEMENT)
                    return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting 'params' element", xsink);

                if (reader.getParams(&v, qe, xsink))
                    return 0;
            }

            // get methodCall close tag
            if (reader.readXmlRpc("expecting methodCall close tag", xsink))
                return 0;
        }

        if ((nt = reader.nodeTypeSkipWhitespace()) != XML_READER_TYPE_END_ELEMENT) {
            return (QoreHashNode*)qore_xml_exception("PARSE-XMLRPC-CALL-ERROR", "expecting 'methodCall' close element", xsink);
        }

        if (reader.checkXmlRpcMemberName("methodCall", xsink, true))
            return 0;

        h->setKeyValue("params", v.getValue(), xsink);
    }

    return h.release();
}

static QoreValue parse_xmlrpc_value(const QoreStringNode* xml, const QoreStringNode* encoding, ExceptionSink* xsink) {
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;

   TempEncodingHelper str(xml, QCS_UTF8, xsink);
   if (!str)
      return 0;

   QoreXmlRpcReader reader(*str, QORE_XML_PARSER_OPTIONS, xsink);
   if (!reader)
      return 0;

   if (reader.read(xsink) != 1)
      return 0;

   Qore::Xml::intern::XmlRpcValue v;
   if (reader.getValueData(&v, qe, false, xsink))
      return 0;

   return v.getValue();
}
# 1566 "ql_xml.qpp"
# 1571 "ql_xml.qpp"
# 1593 "ql_xml.qpp"
# 1621 "ql_xml.qpp"
# 1654 "ql_xml.qpp"
# 1672 "ql_xml.qpp"
# 1696 "ql_xml.qpp"
# 1715 "ql_xml.qpp"
# 1739 "ql_xml.qpp"
# 1765 "ql_xml.qpp"
# 1787 "ql_xml.qpp"
# 1793 "ql_xml.qpp"
# 1817 "ql_xml.qpp"
# 1823 "ql_xml.qpp"
# 1858 "ql_xml.qpp"
# 1878 "ql_xml.qpp"
# 1884 "ql_xml.qpp"
# 1908 "ql_xml.qpp"
# 1914 "ql_xml.qpp"
# 1945 "ql_xml.qpp"
# 1969 "ql_xml.qpp"
# 1995 "ql_xml.qpp"
# 2024 "ql_xml.qpp"
# 2048 "ql_xml.qpp"
# 2072 "ql_xml.qpp"
// string makeFormattedXMLFragment(hash h, *string encoding) {}
static QoreValue f_makeFormattedXMLFragment_VhNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 1805 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;

   SimpleRefHolder<QoreStringNode> str(new QoreStringNode(qe));
   MakeXmlOpts opts;
   opts.m_formatWithWhitespaces = true;
   if (make_xml(xsink, *(*str), *h, 0, opts))
      return 0;

   return str.release();
}

// nothing makeFormattedXMLFragment() {}
static QoreValue f_makeFormattedXMLFragment(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1820 "ql_xml.qpp"

    return QoreValue();
}

// string makeFormattedXMLString(string key, hash h, *string encoding) {}
static QoreValue f_makeFormattedXMLString_VsVhNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* key = HARD_QORE_VALUE_STRING(args, 0);
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 1709 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;

   return make_xml_intern(xsink, key, h, qe);
}

// string makeFormattedXMLString(hash h, *string encoding) {}
static QoreValue f_makeFormattedXMLString_VhNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 1728 "ql_xml.qpp"
   if (!hash_ok(h)) {
      xsink->raiseException("MAKE-FORMATTED-XML-STRING-PARAMETER-EXCEPTION",
                            "this variant of makeFormattedXMLString() expects a hash with a single key for the top-level XML element name without multi-list value");
      return 0;
   }

   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;
   return make_xml_intern(xsink, 0, h, qe, XGF_ADD_FORMATTING);
}

// string makeXMLFragment(hash h, *string encoding) {}
static QoreValue f_makeXMLFragment_VhNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 1777 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;

   SimpleRefHolder<QoreStringNode> str(new QoreStringNode(qe));
   if (make_xml(xsink, *(*str), *h, 0, MakeXmlOpts()))
      return 0;

   return str.release();
}

// nothing makeXMLFragment() {}
static QoreValue f_makeXMLFragment(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1790 "ql_xml.qpp"

    return QoreValue();
}

// string makeXMLString(string key, hash h, *string encoding) {}
static QoreValue f_makeXMLString_VsVhNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* key = HARD_QORE_VALUE_STRING(args, 0);
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 1667 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;
   return make_xml_intern(xsink, key, h, qe);
}

// string makeXMLString(hash h, *string encoding) {}
static QoreValue f_makeXMLString_VhNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 1685 "ql_xml.qpp"
   if (!hash_ok(h)) {
      xsink->raiseException("MAKE-XML-STRING-PARAMETER-EXCEPTION",
                            "this variant of makeXMLString() expects a hash with a single key for the top-level XML element name without multi-list value");
      return 0;
   }

   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;
   return make_xml_intern(xsink, 0, h, qe);
}

// string make_xml(string key, hash h, *int flags, *string encoding) {}
static QoreValue f_make_xml_VsVhNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* key = HARD_QORE_VALUE_STRING(args, 0);
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 1);
    int64 flags = HARD_QORE_VALUE_INT(args, 2);
    const QoreStringNode* encoding = get_param_value(args, 3).get<const QoreStringNode>();
# 1588 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;
   return make_xml_intern(xsink, key, h, qe, flags);
}

// string make_xml(hash h, *int flags, *string encoding) {}
static QoreValue f_make_xml_VhNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    int64 flags = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 1610 "ql_xml.qpp"
   if (!hash_ok(h)) {
      xsink->raiseException("MAKE-XML-STRING-PARAMETER-EXCEPTION",
                            "this variant of make_xml() expects a hash with a single key for the top-level XML element name without multi-list value");
      return 0;
   }

   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;
   return make_xml_intern(xsink, 0, h, qe, flags);
}

// string make_xml(hash h, hash opts) {}
static QoreValue f_make_xml_VhVh(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    const QoreHashNode* opts = HARD_QORE_VALUE_HASH(args, 1);
# 1638 "ql_xml.qpp"
   if (!hash_ok(h)) {
      xsink->raiseException("MAKE-XML-STRING-PARAMETER-EXCEPTION",
                            "this variant of make_xml() expects a hash with a single key for the top-level XML element name without multi-list value");
      return 0;
   }
   try {
       return make_xml_intern(xsink, nullptr, h, MakeXmlOpts::createFromHash(opts));
   } catch (const MakeXmlOpts::InvalidHash &exc) {
      xsink->raiseException("MAKE-XML-OPTS-INVALID",
                            "the opts hash passed is not valid; invalid argument: '%s'",
                            exc.what());
      return 0;
   }
    return QoreValue();
}

// string make_xml_fragment(hash h, *int flags, *string encoding) {}
static QoreValue f_make_xml_fragment_VhNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    int64 flags = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 1755 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;

   SimpleRefHolder<QoreStringNode> str(new QoreStringNode(qe));
   if (make_xml(xsink, *(*str), *h, 0, MakeXmlOpts::createFromFlags(flags, qe)))
      return 0;

   return str.release();
}

// hash parseXML(string xml, *string encoding) {}
static QoreValue f_parseXML_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1874 "ql_xml.qpp"
   return parse_xml_intern(XPF_PRESERVE_ORDER, args, xsink);
}

// nothing parseXML() {}
static QoreValue f_parseXML(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1881 "ql_xml.qpp"

    return QoreValue();
}

// hash parseXMLAsData(string xml, *string encoding) {}
static QoreValue f_parseXMLAsData_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1904 "ql_xml.qpp"
   return parse_xml_intern(XPF_NONE, args, xsink);
}

// nothing parseXMLAsData() {}
static QoreValue f_parseXMLAsData(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1911 "ql_xml.qpp"

    return QoreValue();
}

// hash parseXMLAsDataWithRelaxNG(string xml, string relaxng, *string encoding) {}
static QoreValue f_parseXMLAsDataWithRelaxNG_VsVsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2068 "ql_xml.qpp"
   return parse_xml_with_relaxng_intern(xsink, false, args);
}

// hash parseXMLAsDataWithSchema(string xml, string xsd, *string encoding) {}
static QoreValue f_parseXMLAsDataWithSchema_VsVsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1991 "ql_xml.qpp"
   return parse_xml_with_schema_intern(xsink, false, args);
}

// hash parseXMLWithRelaxNG(string xml, string relaxng, *string encoding) {}
static QoreValue f_parseXMLWithRelaxNG_VsVsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2044 "ql_xml.qpp"
   return parse_xml_with_relaxng_intern(xsink, false, args, XPF_PRESERVE_ORDER);
}

// hash parseXMLWithSchema(string xml, string xsd, *string encoding) {}
static QoreValue f_parseXMLWithSchema_VsVsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1965 "ql_xml.qpp"
   return parse_xml_with_schema_intern(xsink, false, args, XPF_PRESERVE_ORDER);
}

// hash parse_xml(string xml, *int pflags, *string encoding) {}
static QoreValue f_parse_xml_VsNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    int64 pflags = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 1843 "ql_xml.qpp"
   //printd(5, "parseXMLintern(%d, %s)\n", as_data, p0->c_str());

   // convert to UTF-8
   TempEncodingHelper str(xml, QCS_UTF8, xsink);
   if (!str)
      return 0;

   QoreXmlReader reader(*str, QORE_XML_PARSER_OPTIONS, xsink);
   if (!reader)
      return 0;

   return reader.parseXmlData(encoding ? QEM.findCreate(encoding) : QCS_DEFAULT, pflags, xsink);
}

// hash parse_xml_with_dtd(string xml, string dtd, *int pflags, *string encoding) {}
static QoreValue f_parse_xml_with_dtd_VsVsNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* dtd = HARD_QORE_VALUE_STRING(args, 1);
    int64 pflags = HARD_QORE_VALUE_INT(args, 2);
    const QoreStringNode* encoding = get_param_value(args, 3).get<const QoreStringNode>();
# 2096 "ql_xml.qpp"
    SimpleRefHolder<QoreXmlDocData> xd(new QoreXmlDocData(xml));
    if (!xd->isValid()) {
       xsink->raiseException("PARSE-XML-EXCEPTION", "error parsing XML string in parse_xml_with_dtd()");
       return QoreValue();
    }

    if (xd->validateDtd(dtd, xsink))
        return QoreValue();

    QoreXmlReader reader(xd->getDocPtr(), xsink);
    if (*xsink)
        return QoreValue();
    return reader.parseXmlData(encoding ? QEM.findCreate(encoding) : QCS_UTF8, pflags, xsink);
}

// hash parse_xml_with_relaxng(string xml, string relaxng, *int pflags, *string encoding) {}
static QoreValue f_parse_xml_with_relaxng_VsVsNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2020 "ql_xml.qpp"
   return parse_xml_with_relaxng_intern(xsink, true, args);
}

// hash parse_xml_with_schema(string xml, string xsd, *int pflags, *string encoding) {}
static QoreValue f_parse_xml_with_schema_VsVsNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 1941 "ql_xml.qpp"
   return parse_xml_with_schema_intern(xsink, true, args);
}

# 2113 "ql_xml.qpp"
# 2118 "ql_xml.qpp"
# 2147 "ql_xml.qpp"
# 2163 "ql_xml.qpp"
# 2181 "ql_xml.qpp"
# 2197 "ql_xml.qpp"
# 2214 "ql_xml.qpp"
# 2237 "ql_xml.qpp"
# 2251 "ql_xml.qpp"
# 2268 "ql_xml.qpp"
# 2282 "ql_xml.qpp"
# 2299 "ql_xml.qpp"
# 2325 "ql_xml.qpp"
# 2339 "ql_xml.qpp"
# 2354 "ql_xml.qpp"
# 2372 "ql_xml.qpp"
# 2388 "ql_xml.qpp"
# 2404 "ql_xml.qpp"
# 2421 "ql_xml.qpp"
# 2437 "ql_xml.qpp"
# 2455 "ql_xml.qpp"
# 2469 "ql_xml.qpp"
# 2485 "ql_xml.qpp"
# 2501 "ql_xml.qpp"
# 2520 "ql_xml.qpp"
# 2536 "ql_xml.qpp"
# 2542 "ql_xml.qpp"
# 2564 "ql_xml.qpp"
# 2583 "ql_xml.qpp"
# 2589 "ql_xml.qpp"
# 2612 "ql_xml.qpp"
# 2632 "ql_xml.qpp"
# 2638 "ql_xml.qpp"
// auto get_xml_value(auto n) {}
static QoreValue f_get_xml_value_VA(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue n = get_param_value(args, 0);
# 2666 "ql_xml.qpp"
    if (n.isNullOrNothing()) {
        return QoreValue();
    }
    ValueHolder rv(xsink);

    switch (n.getType()) {
        case NT_HASH: {
            const QoreHashNode* h = n.get<const QoreHashNode>();
            const QoreValue v = h->getKeyValue("^value^");
            if (v.isNullOrNothing()) {
                return QoreValue();
            }
            switch (v.getType()) {
                case NT_HASH:
                case NT_LIST:
                    return QoreValue();
                default:
                    rv = v.refSelf();
                    break;
            }
            break;
        }
        case NT_LIST:
            return QoreValue();
        default:
            // just return parameter as-is, we need inc reference
            rv = n.refSelf();
            break;
    }
    return rv.release();
}

// string makeFormattedXMLRPCCallString(string method, ...) {}
static QoreValue f_makeFormattedXMLRPCCallString_VsVV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2433 "ql_xml.qpp"
   return make_xmlrpc_call(xsink, QCS_DEFAULT, 0, args, XGF_ADD_FORMATTING);
}

// string makeFormattedXMLRPCCallStringArgs(string method, any argv) {}
static QoreValue f_makeFormattedXMLRPCCallStringArgs_VsVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2400 "ql_xml.qpp"
   return make_xmlrpc_call_args(xsink, QCS_DEFAULT, 0, args, XGF_ADD_FORMATTING);
}

// string makeFormattedXMLRPCCallStringArgsWithEncoding(string encoding, string method, any argv) {}
static QoreValue f_makeFormattedXMLRPCCallStringArgsWithEncoding_VsVsVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
# 2417 "ql_xml.qpp"
   return make_xmlrpc_call_args(xsink, QEM.findCreate(encoding), 1, args, XGF_ADD_FORMATTING);
}

// string makeFormattedXMLRPCCallStringWithEncoding(string encoding, string method, ...) {}
static QoreValue f_makeFormattedXMLRPCCallStringWithEncoding_VsVsVV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
# 2451 "ql_xml.qpp"
   return make_xmlrpc_call(xsink, QEM.findCreate(encoding), 1, args, XGF_ADD_FORMATTING);
}

// string makeFormattedXMLRPCFaultResponseString(softint code, string msg) {}
static QoreValue f_makeFormattedXMLRPCFaultResponseString_viVs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 code = HARD_QORE_VALUE_INT(args, 0);
    const QoreStringNode* msg = HARD_QORE_VALUE_STRING(args, 1);
# 2278 "ql_xml.qpp"
   return make_xmlrpc_fault(xsink, 0, code, msg, XGF_ADD_FORMATTING);
}

// string makeFormattedXMLRPCFaultResponseStringWithEncoding(string encoding, softint code, string msg) {}
static QoreValue f_makeFormattedXMLRPCFaultResponseStringWithEncoding_VsviVs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
    int64 code = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* msg = HARD_QORE_VALUE_STRING(args, 2);
# 2295 "ql_xml.qpp"
   return make_xmlrpc_fault(xsink, QEM.findCreate(encoding), code, msg, XGF_ADD_FORMATTING);
}

// string makeFormattedXMLRPCResponseString(...) {}
static QoreValue f_makeFormattedXMLRPCResponseString_VV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2465 "ql_xml.qpp"
   return make_xmlrpc_response(xsink, QCS_DEFAULT, 0, args, XGF_ADD_FORMATTING);
}

// string makeFormattedXMLRPCResponseStringWithEncoding(string encoding, ...) {}
static QoreValue f_makeFormattedXMLRPCResponseStringWithEncoding_VsVV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
# 2481 "ql_xml.qpp"
   return make_xmlrpc_response(xsink, QEM.findCreate(encoding), 0, args, XGF_ADD_FORMATTING);
}

// *string makeFormattedXMLRPCValueString(any value, *string encoding) {}
static QoreValue f_makeFormattedXMLRPCValueString_VaNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue value = get_param_value(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2497 "ql_xml.qpp"
   return make_xmlrpc_value(xsink, value, encoding, XGF_ADD_FORMATTING);
}

// string makeXMLRPCCallString(string method, ...) {}
static QoreValue f_makeXMLRPCCallString_VsVV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2159 "ql_xml.qpp"
   return make_xmlrpc_call(xsink, QCS_DEFAULT, 0, args);
}

// string makeXMLRPCCallStringArgs(string method, any args) {}
static QoreValue f_makeXMLRPCCallStringArgs_VsVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2193 "ql_xml.qpp"
   return make_xmlrpc_call_args(xsink, QCS_DEFAULT, 0, args);
}

// string makeXMLRPCCallStringArgsWithEncoding(string encoding, string method, any args) {}
static QoreValue f_makeXMLRPCCallStringArgsWithEncoding_VsVsVa(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
# 2210 "ql_xml.qpp"
   return make_xmlrpc_call_args(xsink, QEM.findCreate(encoding), 1, args);
}

// string makeXMLRPCCallStringWithEncoding(string encoding, string method, ...) {}
static QoreValue f_makeXMLRPCCallStringWithEncoding_VsVsVV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
# 2177 "ql_xml.qpp"
   return make_xmlrpc_call(xsink, QEM.findCreate(encoding), 1, args);
}

// string makeXMLRPCFaultResponseString(softint code, string msg) {}
static QoreValue f_makeXMLRPCFaultResponseString_viVs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 code = HARD_QORE_VALUE_INT(args, 0);
    const QoreStringNode* msg = HARD_QORE_VALUE_STRING(args, 1);
# 2247 "ql_xml.qpp"
   return make_xmlrpc_fault(xsink, 0, code, msg);
}

// string makeXMLRPCFaultResponseStringWithEncoding(string encoding, softint code, string msg) {}
static QoreValue f_makeXMLRPCFaultResponseStringWithEncoding_VsviVs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
    int64 code = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* msg = HARD_QORE_VALUE_STRING(args, 2);
# 2264 "ql_xml.qpp"
   return make_xmlrpc_fault(xsink, QEM.findCreate(encoding), code, msg);
}

// string makeXMLRPCResponseString(...) {}
static QoreValue f_makeXMLRPCResponseString_VV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2336 "ql_xml.qpp"
   return make_xmlrpc_response(xsink, QCS_DEFAULT, 0, args);
}

// string makeXMLRPCResponseStringWithEncoding(string encoding, ...) {}
static QoreValue f_makeXMLRPCResponseStringWithEncoding_VsVV(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* encoding = HARD_QORE_VALUE_STRING(args, 0);
# 2350 "ql_xml.qpp"
   return make_xmlrpc_response(xsink, QEM.findCreate(encoding), 1, args);
}

// *string makeXMLRPCValueString(any value, *string encoding) {}
static QoreValue f_makeXMLRPCValueString_VaNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue value = get_param_value(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2384 "ql_xml.qpp"
   return make_xmlrpc_value(xsink, value, encoding);
}

// string make_xmlrpc_call(string method, any args, *int flags, *string encoding) {}
static QoreValue f_make_xmlrpc_call_VsVaNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 flags = HARD_QORE_VALUE_INT(args, 2);
    const QoreStringNode* encoding = get_param_value(args, 3).get<const QoreStringNode>();
# 2143 "ql_xml.qpp"
   return make_xmlrpc_call_args(xsink, encoding ? QEM.findCreate(encoding) : QCS_DEFAULT, 0, args, flags);
}

// string make_xmlrpc_fault(softint code, string msg, *int flags, *string encoding) {}
static QoreValue f_make_xmlrpc_fault_viVsNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 code = HARD_QORE_VALUE_INT(args, 0);
    const QoreStringNode* msg = HARD_QORE_VALUE_STRING(args, 1);
    int64 flags = HARD_QORE_VALUE_INT(args, 2);
    const QoreStringNode* encoding = get_param_value(args, 3).get<const QoreStringNode>();
# 2233 "ql_xml.qpp"
   return make_xmlrpc_fault(xsink, encoding ? QEM.findCreate(encoding) : 0, code, msg, flags);
}

// string make_xmlrpc_response(any call_args, *int flags, *string encoding) {}
static QoreValue f_make_xmlrpc_response_VaNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue call_args = get_param_value(args, 0);
    int64 flags = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 2321 "ql_xml.qpp"
   return make_xmlrpc_response_args(xsink, encoding ? QEM.findCreate(encoding) : QCS_DEFAULT, call_args, flags);
}

// *string make_xmlrpc_value(any value, *int flags, *string encoding) {}
static QoreValue f_make_xmlrpc_value_VaNiNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue value = get_param_value(args, 0);
    int64 flags = HARD_QORE_VALUE_INT(args, 1);
    const QoreStringNode* encoding = get_param_value(args, 2).get<const QoreStringNode>();
# 2368 "ql_xml.qpp"
   return make_xmlrpc_value(xsink, value, encoding, flags);
}

// hash parseXMLRPCCall(string xml, *string encoding) {}
static QoreValue f_parseXMLRPCCall_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2579 "ql_xml.qpp"
   return parse_xmlrpc_call(xml, encoding, xsink);
}

// nothing parseXMLRPCCall() {}
static QoreValue f_parseXMLRPCCall(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2586 "ql_xml.qpp"

    return QoreValue();
}

// hash parseXMLRPCResponse(string xml, *string encoding) {}
static QoreValue f_parseXMLRPCResponse_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2627 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;
   return parse_xmlrpc_response(xsink, xml, qe);
}

// nothing parseXMLRPCResponse() {}
static QoreValue f_parseXMLRPCResponse(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2635 "ql_xml.qpp"

    return QoreValue();
}

// any parseXMLRPCValue(string xml, *string encoding) {}
static QoreValue f_parseXMLRPCValue_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2532 "ql_xml.qpp"
   return parse_xmlrpc_value(xml, encoding, xsink);
}

// nothing parseXMLRPCValue() {}
static QoreValue f_parseXMLRPCValue(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 2539 "ql_xml.qpp"

    return QoreValue();
}

// hash parse_xmlrpc_call(string xml, *string encoding) {}
static QoreValue f_parse_xmlrpc_call_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2560 "ql_xml.qpp"
   return parse_xmlrpc_call(xml, encoding, xsink);
}

// hash parse_xmlrpc_response(string xml, *string encoding) {}
static QoreValue f_parse_xmlrpc_response_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2607 "ql_xml.qpp"
   const QoreEncoding* qe = encoding ? QEM.findCreate(encoding) : QCS_DEFAULT;
   return parse_xmlrpc_response(xsink, xml, qe);
}

// any parse_xmlrpc_value(string xml, *string encoding) {}
static QoreValue f_parse_xmlrpc_value_VsNs(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* encoding = get_param_value(args, 1).get<const QoreStringNode>();
# 2516 "ql_xml.qpp"
   return parse_xmlrpc_value(xml, encoding, xsink);
}


DLLLOCAL void init_xml_functions(QoreNamespace& ns) {
    // string makeFormattedXMLFragment(hash h, *string encoding) {}
    ns.addBuiltinVariant("makeFormattedXMLFragment", (q_func_n_t)f_makeFormattedXMLFragment_VhNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "h", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing makeFormattedXMLFragment() {}
    ns.addBuiltinVariant("makeFormattedXMLFragment", (q_func_n_t)f_makeFormattedXMLFragment, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // string makeFormattedXMLString(string key, hash h, *string encoding) {}
    ns.addBuiltinVariant("makeFormattedXMLString", (q_func_n_t)f_makeFormattedXMLString_VsVhNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "key", hashTypeInfo, QORE_PARAM_NO_ARG, "h", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string makeFormattedXMLString(hash h, *string encoding) {}
    ns.addBuiltinVariant("makeFormattedXMLString", (q_func_n_t)f_makeFormattedXMLString_VhNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "h", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string makeXMLFragment(hash h, *string encoding) {}
    ns.addBuiltinVariant("makeXMLFragment", (q_func_n_t)f_makeXMLFragment_VhNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "h", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing makeXMLFragment() {}
    ns.addBuiltinVariant("makeXMLFragment", (q_func_n_t)f_makeXMLFragment, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // string makeXMLString(string key, hash h, *string encoding) {}
    ns.addBuiltinVariant("makeXMLString", (q_func_n_t)f_makeXMLString_VsVhNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "key", hashTypeInfo, QORE_PARAM_NO_ARG, "h", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string makeXMLString(hash h, *string encoding) {}
    ns.addBuiltinVariant("makeXMLString", (q_func_n_t)f_makeXMLString_VhNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "h", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string make_xml(string key, hash h, *int flags, *string encoding) {}
    ns.addBuiltinVariant("make_xml", (q_func_n_t)f_make_xml_VsVhNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "key", hashTypeInfo, QORE_PARAM_NO_ARG, "h", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "flags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string make_xml(hash h, *int flags, *string encoding) {}
    ns.addBuiltinVariant("make_xml", (q_func_n_t)f_make_xml_VhNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 3, hashTypeInfo, QORE_PARAM_NO_ARG, "h", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "flags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string make_xml(hash h, hash opts) {}
    ns.addBuiltinVariant("make_xml", (q_func_n_t)f_make_xml_VhVh, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "h", hashTypeInfo, QORE_PARAM_NO_ARG, "opts");
    // string make_xml_fragment(hash h, *int flags, *string encoding) {}
    ns.addBuiltinVariant("make_xml_fragment", (q_func_n_t)f_make_xml_fragment_VhNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 3, hashTypeInfo, QORE_PARAM_NO_ARG, "h", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "flags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parseXML(string xml, *string encoding) {}
    ns.addBuiltinVariant("parseXML", (q_func_n_t)f_parseXML_VsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing parseXML() {}
    ns.addBuiltinVariant("parseXML", (q_func_n_t)f_parseXML, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // hash parseXMLAsData(string xml, *string encoding) {}
    ns.addBuiltinVariant("parseXMLAsData", (q_func_n_t)f_parseXMLAsData_VsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing parseXMLAsData() {}
    ns.addBuiltinVariant("parseXMLAsData", (q_func_n_t)f_parseXMLAsData, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // hash parseXMLAsDataWithRelaxNG(string xml, string relaxng, *string encoding) {}
    ns.addBuiltinVariant("parseXMLAsDataWithRelaxNG", (q_func_n_t)f_parseXMLAsDataWithRelaxNG_VsVsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringTypeInfo, QORE_PARAM_NO_ARG, "relaxng", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parseXMLAsDataWithSchema(string xml, string xsd, *string encoding) {}
    ns.addBuiltinVariant("parseXMLAsDataWithSchema", (q_func_n_t)f_parseXMLAsDataWithSchema_VsVsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringTypeInfo, QORE_PARAM_NO_ARG, "xsd", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parseXMLWithRelaxNG(string xml, string relaxng, *string encoding) {}
    ns.addBuiltinVariant("parseXMLWithRelaxNG", (q_func_n_t)f_parseXMLWithRelaxNG_VsVsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringTypeInfo, QORE_PARAM_NO_ARG, "relaxng", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parseXMLWithSchema(string xml, string xsd, *string encoding) {}
    ns.addBuiltinVariant("parseXMLWithSchema", (q_func_n_t)f_parseXMLWithSchema_VsVsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringTypeInfo, QORE_PARAM_NO_ARG, "xsd", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parse_xml(string xml, *int pflags, *string encoding) {}
    ns.addBuiltinVariant("parse_xml", (q_func_n_t)f_parse_xml_VsNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "pflags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parse_xml_with_dtd(string xml, string dtd, *int pflags, *string encoding) {}
    ns.addBuiltinVariant("parse_xml_with_dtd", (q_func_n_t)f_parse_xml_with_dtd_VsVsNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringTypeInfo, QORE_PARAM_NO_ARG, "dtd", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "pflags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parse_xml_with_relaxng(string xml, string relaxng, *int pflags, *string encoding) {}
    ns.addBuiltinVariant("parse_xml_with_relaxng", (q_func_n_t)f_parse_xml_with_relaxng_VsVsNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringTypeInfo, QORE_PARAM_NO_ARG, "relaxng", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "pflags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parse_xml_with_schema(string xml, string xsd, *int pflags, *string encoding) {}
    ns.addBuiltinVariant("parse_xml_with_schema", (q_func_n_t)f_parse_xml_with_schema_VsVsNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringTypeInfo, QORE_PARAM_NO_ARG, "xsd", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "pflags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // auto get_xml_value(auto n) {}
    ns.addBuiltinVariant("get_xml_value", (q_func_n_t)f_get_xml_value_VA, QCF_CONSTANT, QDOM_DEFAULT, autoTypeInfo, 1, autoTypeInfo, QORE_PARAM_NO_ARG, "n");
    // string makeFormattedXMLRPCCallString(string method, ...) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCCallString", (q_func_n_t)f_makeFormattedXMLRPCCallString_VsVV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "method");
    // string makeFormattedXMLRPCCallStringArgs(string method, any argv) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCCallStringArgs", (q_func_n_t)f_makeFormattedXMLRPCCallStringArgs_VsVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "argv");
    // string makeFormattedXMLRPCCallStringArgsWithEncoding(string encoding, string method, any argv) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCCallStringArgsWithEncoding", (q_func_n_t)f_makeFormattedXMLRPCCallStringArgsWithEncoding_VsVsVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding", stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "argv");
    // string makeFormattedXMLRPCCallStringWithEncoding(string encoding, string method, ...) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCCallStringWithEncoding", (q_func_n_t)f_makeFormattedXMLRPCCallStringWithEncoding_VsVsVV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding", stringTypeInfo, QORE_PARAM_NO_ARG, "method");
    // string makeFormattedXMLRPCFaultResponseString(softint code, string msg) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCFaultResponseString", (q_func_n_t)f_makeFormattedXMLRPCFaultResponseString_viVs, QCF_CONSTANT|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "msg");
    // string makeFormattedXMLRPCFaultResponseStringWithEncoding(string encoding, softint code, string msg) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCFaultResponseStringWithEncoding", (q_func_n_t)f_makeFormattedXMLRPCFaultResponseStringWithEncoding_VsviVs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding", softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "msg");
    // string makeFormattedXMLRPCResponseString(...) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCResponseString", (q_func_n_t)f_makeFormattedXMLRPCResponseString_VV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo);
    // string makeFormattedXMLRPCResponseStringWithEncoding(string encoding, ...) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCResponseStringWithEncoding", (q_func_n_t)f_makeFormattedXMLRPCResponseStringWithEncoding_VsVV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // *string makeFormattedXMLRPCValueString(any value, *string encoding) {}
    ns.addBuiltinVariant("makeFormattedXMLRPCValueString", (q_func_n_t)f_makeFormattedXMLRPCValueString_VaNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringOrNothingTypeInfo, 2, anyTypeInfo, QORE_PARAM_NO_ARG, "value", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string makeXMLRPCCallString(string method, ...) {}
    ns.addBuiltinVariant("makeXMLRPCCallString", (q_func_n_t)f_makeXMLRPCCallString_VsVV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "method");
    // string makeXMLRPCCallStringArgs(string method, any args) {}
    ns.addBuiltinVariant("makeXMLRPCCallStringArgs", (q_func_n_t)f_makeXMLRPCCallStringArgs_VsVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "args");
    // string makeXMLRPCCallStringArgsWithEncoding(string encoding, string method, any args) {}
    ns.addBuiltinVariant("makeXMLRPCCallStringArgsWithEncoding", (q_func_n_t)f_makeXMLRPCCallStringArgsWithEncoding_VsVsVa, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding", stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "args");
    // string makeXMLRPCCallStringWithEncoding(string encoding, string method, ...) {}
    ns.addBuiltinVariant("makeXMLRPCCallStringWithEncoding", (q_func_n_t)f_makeXMLRPCCallStringWithEncoding_VsVsVV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding", stringTypeInfo, QORE_PARAM_NO_ARG, "method");
    // string makeXMLRPCFaultResponseString(softint code, string msg) {}
    ns.addBuiltinVariant("makeXMLRPCFaultResponseString", (q_func_n_t)f_makeXMLRPCFaultResponseString_viVs, QCF_CONSTANT|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 2, softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "msg");
    // string makeXMLRPCFaultResponseStringWithEncoding(string encoding, softint code, string msg) {}
    ns.addBuiltinVariant("makeXMLRPCFaultResponseStringWithEncoding", (q_func_n_t)f_makeXMLRPCFaultResponseStringWithEncoding_VsviVs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding", softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "msg");
    // string makeXMLRPCResponseString(...) {}
    ns.addBuiltinVariant("makeXMLRPCResponseString", (q_func_n_t)f_makeXMLRPCResponseString_VV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo);
    // string makeXMLRPCResponseStringWithEncoding(string encoding, ...) {}
    ns.addBuiltinVariant("makeXMLRPCResponseStringWithEncoding", (q_func_n_t)f_makeXMLRPCResponseStringWithEncoding_VsVV, QCF_RET_VALUE_ONLY|QCF_DEPRECATED|QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, stringTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // *string makeXMLRPCValueString(any value, *string encoding) {}
    ns.addBuiltinVariant("makeXMLRPCValueString", (q_func_n_t)f_makeXMLRPCValueString_VaNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, stringOrNothingTypeInfo, 2, anyTypeInfo, QORE_PARAM_NO_ARG, "value", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string make_xmlrpc_call(string method, any args, *int flags, *string encoding) {}
    ns.addBuiltinVariant("make_xmlrpc_call", (q_func_n_t)f_make_xmlrpc_call_VsVaNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "args", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "flags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string make_xmlrpc_fault(softint code, string msg, *int flags, *string encoding) {}
    ns.addBuiltinVariant("make_xmlrpc_fault", (q_func_n_t)f_make_xmlrpc_fault_viVsNiNs, QCF_CONSTANT, QDOM_DEFAULT, stringTypeInfo, 4, softBigIntTypeInfo, QORE_PARAM_NO_ARG, "code", stringTypeInfo, QORE_PARAM_NO_ARG, "msg", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "flags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // string make_xmlrpc_response(any call_args, *int flags, *string encoding) {}
    ns.addBuiltinVariant("make_xmlrpc_response", (q_func_n_t)f_make_xmlrpc_response_VaNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo, 3, anyTypeInfo, QORE_PARAM_NO_ARG, "call_args", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "flags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // *string make_xmlrpc_value(any value, *int flags, *string encoding) {}
    ns.addBuiltinVariant("make_xmlrpc_value", (q_func_n_t)f_make_xmlrpc_value_VaNiNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringOrNothingTypeInfo, 3, anyTypeInfo, QORE_PARAM_NO_ARG, "value", bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "flags", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parseXMLRPCCall(string xml, *string encoding) {}
    ns.addBuiltinVariant("parseXMLRPCCall", (q_func_n_t)f_parseXMLRPCCall_VsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing parseXMLRPCCall() {}
    ns.addBuiltinVariant("parseXMLRPCCall", (q_func_n_t)f_parseXMLRPCCall, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // hash parseXMLRPCResponse(string xml, *string encoding) {}
    ns.addBuiltinVariant("parseXMLRPCResponse", (q_func_n_t)f_parseXMLRPCResponse_VsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing parseXMLRPCResponse() {}
    ns.addBuiltinVariant("parseXMLRPCResponse", (q_func_n_t)f_parseXMLRPCResponse, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // any parseXMLRPCValue(string xml, *string encoding) {}
    ns.addBuiltinVariant("parseXMLRPCValue", (q_func_n_t)f_parseXMLRPCValue_VsNs, QCF_RET_VALUE_ONLY|QCF_DEPRECATED, QDOM_DEFAULT, anyTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // nothing parseXMLRPCValue() {}
    ns.addBuiltinVariant("parseXMLRPCValue", (q_func_n_t)f_parseXMLRPCValue, QCF_RUNTIME_NOOP|QCF_DEPRECATED, QDOM_DEFAULT, nothingTypeInfo);
    // hash parse_xmlrpc_call(string xml, *string encoding) {}
    ns.addBuiltinVariant("parse_xmlrpc_call", (q_func_n_t)f_parse_xmlrpc_call_VsNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // hash parse_xmlrpc_response(string xml, *string encoding) {}
    ns.addBuiltinVariant("parse_xmlrpc_response", (q_func_n_t)f_parse_xmlrpc_response_VsNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");
    // any parse_xmlrpc_value(string xml, *string encoding) {}
    ns.addBuiltinVariant("parse_xmlrpc_value", (q_func_n_t)f_parse_xmlrpc_value_VsNs, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, anyTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "xml", stringOrNothingTypeInfo, QORE_PARAM_NO_ARG, "encoding");

}

DLLLOCAL void init_xml_constants(QoreNamespace& ns) {
# 1511 "ql_xml.qpp"
# 1517 "ql_xml.qpp"
# 1520 "ql_xml.qpp"
    ns.addConstant("XGF_ADD_FORMATTING", ((int64)XGF_ADD_FORMATTING));
    ns.addConstant("XGF_NONE", ((int64)XGF_NONE));
    ns.addConstant("XGF_USE_NUMERIC_REFS", ((int64)XGF_USE_NUMERIC_REFS));
# 1530 "ql_xml.qpp"
# 1537 "ql_xml.qpp"
# 1540 "ql_xml.qpp"
# 1554 "ql_xml.qpp"
    ns.addConstant("XPF_ADD_COMMENTS", ((int64)XPF_ADD_COMMENTS));
    ns.addConstant("XPF_NONE", ((int64)XPF_NONE));
    ns.addConstant("XPF_PRESERVE_ORDER", ((int64)XPF_PRESERVE_ORDER));
    ns.addConstant("XPF_STRIP_NS_PREFIXES", ((int64)XPF_STRIP_NS_PREFIXES));
# 1566 "ql_xml.qpp"
# 1571 "ql_xml.qpp"
# 1593 "ql_xml.qpp"
# 1621 "ql_xml.qpp"
# 1654 "ql_xml.qpp"
# 1672 "ql_xml.qpp"
# 1696 "ql_xml.qpp"
# 1715 "ql_xml.qpp"
# 1739 "ql_xml.qpp"
# 1765 "ql_xml.qpp"
# 1787 "ql_xml.qpp"
# 1793 "ql_xml.qpp"
# 1817 "ql_xml.qpp"
# 1823 "ql_xml.qpp"
# 1858 "ql_xml.qpp"
# 1878 "ql_xml.qpp"
# 1884 "ql_xml.qpp"
# 1908 "ql_xml.qpp"
# 1914 "ql_xml.qpp"
# 1945 "ql_xml.qpp"
# 1969 "ql_xml.qpp"
# 1995 "ql_xml.qpp"
# 2024 "ql_xml.qpp"
# 2048 "ql_xml.qpp"
# 2072 "ql_xml.qpp"
# 2113 "ql_xml.qpp"
# 2118 "ql_xml.qpp"
# 2147 "ql_xml.qpp"
# 2163 "ql_xml.qpp"
# 2181 "ql_xml.qpp"
# 2197 "ql_xml.qpp"
# 2214 "ql_xml.qpp"
# 2237 "ql_xml.qpp"
# 2251 "ql_xml.qpp"
# 2268 "ql_xml.qpp"
# 2282 "ql_xml.qpp"
# 2299 "ql_xml.qpp"
# 2325 "ql_xml.qpp"
# 2339 "ql_xml.qpp"
# 2354 "ql_xml.qpp"
# 2372 "ql_xml.qpp"
# 2388 "ql_xml.qpp"
# 2404 "ql_xml.qpp"
# 2421 "ql_xml.qpp"
# 2437 "ql_xml.qpp"
# 2455 "ql_xml.qpp"
# 2469 "ql_xml.qpp"
# 2485 "ql_xml.qpp"
# 2501 "ql_xml.qpp"
# 2520 "ql_xml.qpp"
# 2536 "ql_xml.qpp"
# 2542 "ql_xml.qpp"
# 2564 "ql_xml.qpp"
# 2583 "ql_xml.qpp"
# 2589 "ql_xml.qpp"
# 2612 "ql_xml.qpp"
# 2632 "ql_xml.qpp"
# 2638 "ql_xml.qpp"

}
