# PyORQ, version 0.1
# Python Object-Relational binding with Queries
#
# Copyright (c) 1994 Roeland Rengelink
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

"""PyORQ v 0.1 - Copyright (c) 2004 Roeland Rengelink

PyORQ (Python Object Relational binding with Queries) implements persistence
for Python objects using a relational database for storage.

PyORQ currently provides interfaces to three backends:

postgresql_db -- PostgreSQL backend, based on pyPgSQL-2.4 by Billy G. Allie
                 (http://sourceforge.net/projects/pypgsql)
mysql_db      -- MySQL backend, based on MySQL-python-0.9.2 by Andy Dustman
                 (http://sourceforge.net/projects/mysql-python)
sqlite_db     -- SQLite backend, based on PySQLite-0.5.0 by Michael Owens,
                 Gerhard Haring (http://sourceforge.net/projects/pysqlite)

The modules for these backend can be found in package pyorq.interface

A persistent object is defined by subclassing pobject and providing a database
attribute referring to an instance of one of the database interfaces. E.g.:

db = pyorq.interface.sqlite_db.sqlite_db(database='testdb')
class MyObject(pyorq.pobject):
    database = db
    ...

All objects in a schema must refer to the same database interface instance.

Each persistent object should define at least one persistent attribute.
Persistent attributes refer to special property objects (currently: pint,
pfloat, pstr, pdate, ptime, pdatetime or pref)

class AObject(pyorq.pobject):
    database = db
    a = pyorq.pint()
    b = pyorq.pfloat()
    c = pyorq.pstr()
    d = pyorq.pdate()
    e = pyorq.ptime()
    f = pyorq.pdatetime()

defines a class with persistent attributes of type 'int', 'float', 'str',
'datetime.date', 'datetime.time' and 'datatime.datetime' respectively.  Note
that persistent attributes are statically typed. Assigning a float to an
attribute pint() is illegal.

Persistent objects can refer to other persistent objects.

class BObject(pyorq.pobject):
    database = db
    a = pyorq.pref(AObject)

defines a class with a reference to objects that are instances of
AObject. PyORQ is polymorphic, in the sense that instances of subclasses of
AObject may be assigned too. Queries (see below) understand this.

An object can be stored in the database by using commit()

a = AObject()
... # assign interesting values to the attributes of a
b = BObject()
b.a = a
b.commit()

Commit is recursive. An object with references to other objects, will commit
those objects too.

If an object has not been previously committed, the object will be inserted
into the database and an object identifier will be generated. This identifier
is stored in the attribute 'oid' of the object.

If one knows the object identifier, an object can be restored from the
database, using:

b = BObject(oid=the_identifier)

since b.a knows the identifier of a, acces of b.a will automatically retrieve
the instance of AObject that b.a refers to.

Of course, one normally does not know the identifier of an object stored in
the database. Instead, we use queries to retrieve objects from the database.

In Pyorq queries are expression on pyorq properties. For example, given:

class Person(pyorq.pobject):
      database = db
      name = pyorq.pstr()
      birth = pyorq.pdate()

a query can be something like

Person.name == 'Roeland Rengelink'
Person.birth < datetime.date(1968, 1, 1)

Note that Person.name and Person.birth refer to the persistent property
objects pstr() and pref(). These references can be used in expressions to
define queries.

Queries are iterators, so we can do something like:

for p in Person.birth < datetime.date(1968, 1, 1):
    print p.name, p.birth

Queries understand encapsulation. Given:

class A(pyorq.pobject):
    a = pyorq.pint()

class B(pyorq.pobject):
    b = pyorq.pref(A)

then (B.b.a == 3) is a valied query.

Queries understand inheritance, Given:

class C(B):pass

the query (B.b.a == 3), may return objects of type B and C. Also:

class AA(A):pass

implies that a query (B.b.a == 3) will search among objects of type A and AA,
referred to by B.b or C.b.

Queries support the following operators:

Comparisons : ==, !=, <, <=, >, >= 
Logical     : & (and), | (or), ~ (not)
Arithmetic  : +, -, *, /

Note that queries uses the bitwise operators (&,|,~) as logical
operators. Because bitwise operators have higher precedence than comparison
operators, queries usually require paranthesis:

(X.x+3 == X.y) & (X.z <= 7)

Queries should have only one 'free variable'. For example:

(X.x == Y.y)

is not a valid query, since it's not clear if we are looking for objects of
type X or objects of type Y.

You can use

from pyorq import *

to import the following objects:

pobject - from pyorq.ptype
pint, pfloat, pstr, pdate, ptime, pdatetime, pref - from pyorq.pprop
"""

# The following objects comprise the PyORQ user interface
from pyorq.ptype import pobject
from pyorq.pprop import pint, pfloat, pstr, pdate, ptime, pdatetime, pref
from pyorq.interface.db_base import DBError
