#!/usr/local/bin/python

# ========================================================================
#  Classes which define connections to devices
#
#  Brian Landers <blanders@sapient.com>
#  07.07.2001
#
#  $Id: connections.py,v 1.10 2001/10/24 18:41:58 bluecoat93 Exp $
# ========================================================================

from netdevicelib.devices import DeviceFactory
import getopt, re, sys, telnetlib, types

# ------------------------------------------------------------------------

LoginFailedException   = "Login failed. Bad username or password"
EnableFailedException  = "Enable failed. Access denied"
DisableFailedException = "Disable command failed."

# ------------------------------------------------------------------------

class Connection:
    """ Base class for all connections """
    
    def __init__( self, inDevice=None, inTimeout=10 ):
        """ Constructor """
        assert inDevice != None

        self._device      = inDevice
        self._timeout     = inTimeout
        self._isDebugging = 0;
        self._isOpen      = 0
        self._lastPrompt  = ''

    # Virtual methods -- must be overridden
    def open( self, inHost=None, inPort=23 ):
        """ Open the connection to the device """
        raise RuntimeException, "Unimplemented base class method called"

    def close( self ):
        """ Close the connection to the device """
        raise RuntimeException, "Unimplemented base class method called"

    def login( self, inUser=None, inPass=None ):
        """ Login to the device using a username and password """
        raise RuntimeException, "Unimplemented base class method called"

    def cmd( self, inCmd=None, inPrompt=None ):
        """ Run a command on the device and return the output """
        raise RuntimeException, "Unimplemented base class method called"

    # Base class methods -- may be overridden
    def _debuglog( self, inMessage="No Message" ):
        """ Write a debug message to STDERR if debugging is enabled """
        
        if self.debug():
            sys.stderr.write( "DEBUG: %s\n" % inMessage )

    def debug( self, inLevel=None ):
        """ Accessor method for the debugging flag """

        if inLevel != None:
            self._isDebugging = inLevel

        return self._isDebugging
    
    def isEnabled( self ):
        """ Returns true if the connection is in 'superuser' mode """
        return 1

    def enable( self, inPass=None ):
        """ Put the connection in 'superuser' mode """
        pass

    def disable( self ):
        """ Take the connection out of 'superuser' mode """
        pass

    def getLastPrompt( self ):
        """ Accessor method to get the last prompt we saw """
        return self._lastPrompt

    def disablePaging( self ):
        """ Helper function to disable screen paging for a connection """
        self.cmd( self._device.getCommand('disablePaging') )

    def enablePaging( self ):
        """ Helper function to enable screen paging for a connection """
        self.cmd( self._device.getCommand('enablePaging') )
  
    def getConfig( self ):
        """ Helper function to get the current config from a connection """
        if self._device._needsEnable and not self.isEnabled():
            raise RuntimeError( "You must be enabled first" )
        
        return self.cmd( self._device.getCommand('getConfig') )
    
# ------------------------------------------------------------------------

class TelnetConnection( Connection ):
    """ Encapsulates a telnet connection to a device """
    
    def __init__( self, inDevice=None ):
        """ Constructor """
        assert inDevice != None
        
        Connection.__init__( self, inDevice )
        self._conn = telnetlib.Telnet()

    def open( self, inHost=None, inPort=23 ):
        """ Open the connection to the device """
        assert inHost != None
        
        self._conn.open( inHost )
        self._isOpen = 1
        self._debuglog( "Connection open" )

    def close( self ):
        """ Close the connection to the device """
        
        if self._isOpen:
            self.enablePaging()
            self._conn.close()

        self._isOpen = 0
        
    def login( self, inUser=None, inPass=None ):
        """ Login to the device using a username and password """
        assert inUser != None
        assert inPass != None

        # Wait for the login prompt
        matches = [ self._device.getPrompt('login'),
                    self._device.getPrompt('username'),
                    self._device.getPrompt('password') ]
        
        self._debuglog( "Looking for username prompt (" +
                        self._device.getPrompt( 'username' ) + ")" )
        
        result = self._conn.expect( matches, self._timeout )

        self._debuglog( "Found username prompt, match %s" % result[0] )

        # If we didn't find a match, bail out
        if result[0] == -1:
            raise RuntimeError( "Didn't find a login prompt" )
        
        # Unless we only got a password prompt, send the username
        if result[0] != 2:
            # Send the login name
            self._debuglog( "Sending username (" + inUser + ")" )
            self._conn.write( inUser + "\n" )
            
            # Wait for password prompt
            self._debuglog( "Looking for password prompt (" + 
                            self._device.getPrompt( 'password' ) + ")" )
                  
            result = self._conn.expect( [ self._device.getPrompt('password') ],
                                        self._timeout )

            self._debuglog( "Found password prompt" )
            
        # Send password
        self._debuglog( "Sending password (" + inPass + ")" )
        self._conn.write( inPass + "\n" )

        # Wait for command prompt or another login prompt
        self._debuglog( "Looking for cmd prompt + (" +
                        self._device.getPrompt('command') + ")" )

        result = self._conn.expect( matches +
                                    [ self._device.getPrompt('command') ],
                                    self._timeout )

        self._debuglog( "Found cmd prompt" )
        
        # Bad login if we got another login prompt
        if result[0] != 3:
            raise LoginFailedException
        else:
            self._debuglog( "login succeeded" )

        # Disable paging so that commands work correctly
        self.disablePaging()

    def cmd( self, inCmd=None, inPrompt=None ):
        """ Run a command on the device and return the output """
        assert inCmd != None

        # Handle blank commands (i.e. unimplemented in the Device subclass)
        if inCmd == "":
            return ""
        
        self._debuglog( "running command (" + inCmd + ")" )
        self._conn.write( inCmd + "\n" )

        if inPrompt != None:
            if type( inPrompt ) == types.ListType:
                prompts = inPrompt
            else:
                prompts = [ inPrompt ]
        else:
            prompts = [ self._device.getPrompt('command') ]

        self._debuglog( "Looking for cmd prompt + (" + str(prompts) + ")" )

        result = self._conn.expect( prompts, self._timeout )

        # Store the last prompt we saw
        self._lastPrompt = result[1].group()

        # Remove the command itself from the output
        exp = re.compile( '^%s\s*$\n' % inCmd, re.MULTILINE )
        output = exp.sub( '', result[2], 1 )
        
        # Remove the prompt from the output and return the results
        exp = re.compile( self._device.getPrompt('command') )
        return exp.sub( '', output )

    def enable( self, inPass=None ):
        """ Put the connection in 'superuser' mode """
        assert inPass != None
        
        self._debuglog( "trying to enable with password %s" % inPass )

        # Send the enable command, expecting the password prompt
        self.cmd( self._device.getCommand('enable'),
                  self._device.getPrompt('enable') )

        # Set the password
        self.cmd( inPass )

        # Make sure we got enabled
        if not self.isEnabled():
            raise EnableFailedException
        
    def disable( self ):
        """ Take the connection out of 'superuser' mode """
        
        self.cmd( self._device.getCommand('disable') )

        # Make sure we disabled
        if self.isEnabled():
            raise DisableFailedException

    def isEnabled( self ):
        """ Returns true if the connection is in 'superuser' mode """
        
        exp = re.compile( self._device.getPrompt( 'enabledIndicator' ) )
        if exp.search( self._lastPrompt ) == None:
            return 0
        else:
            return 1
  
# ------------------------------------------------------------------------

class ConnectionFactory:
    """ Factory class for creating Connecton sub-class objects """
    
    def createConnection( self, inType=None, inClass=None ):
        """ Factory method to create Connection sub-class objects """
        assert inType  != None
        assert inClass != None

        # Create the device object
        device = DeviceFactory().createDevice( inClass )
        
        if inType == 'telnet':
            return TelnetConnection( device )
        else:
            raise RuntimeError( "Type '" + inType + "' not supported" )

# ========================================================================
#  Test driver
# ========================================================================

if __name__ == "__main__":

    # Parse our command-line arguments
    debugging = 0
    try:
        opts, args = getopt.getopt( sys.argv[1:], "d", ["debug"] )
    except getopt.GetoptError:
        print "Use -d or --debug for debugging"
        sys.exit(1)
    for o,a in opts:
        if o in ( '-d', '--debug' ):
            debugging = 1

    # Make sure they entered all the parameters we need
    if len( args ) < 5:
        print "usage: connections.py " + \
              "hostname type username password [enable] command"
        sys.exit(1)

    # Create the connection object
    conn = ConnectionFactory().createConnection( "telnet", args[1] )

    # Set the debugging flag if the user indicated debugging
    if debugging:
        conn.debug(1)

    # Open the connection and login
    conn.open( args[0] )
    conn.login( args[2], args[3] )

    # If the user provided an enable password, go into enable mode
    if len( args ) == 6:
        conn.enable( args[4] )
        cmd = args[5]
    else:
        cmd = args[4]

    # Run the command and print the output
    lines = conn.cmd( cmd )
    print lines

    # Finally, close the connection
    conn.close()
