# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['checkdigit']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'checkdigit',
    'version': '0.2.0',
    'description': 'A check digit library for data validation',
    'long_description': '.. image:: https://raw.githubusercontent.com/harens/checkdigit/master/art/logo.png\n   :alt: checkdigit logo\n   :align: center\n\n|\n\n.. image:: https://img.shields.io/github/workflow/status/harens/checkdigit/Tests?logo=github&style=flat-square\n   :alt: GitHub Tests status\n\n.. image:: https://img.shields.io/codecov/c/github/harens/checkdigit?logo=codecov&style=flat-square\n   :alt: Codecov\n\n.. image:: https://img.shields.io/pypi/dm/checkdigit?logo=python&logoColor=white&style=flat-square\n   :alt: PyPi - Downloads\n\n.. image:: https://img.shields.io/codefactor/grade/github/harens/checkdigit?logo=codefactor&style=flat-square\n   :alt: CodeFactor Grade\n\n.. image:: https://img.shields.io/lgtm/grade/python/github/harens/checkdigit?logo=lgtm&style=flat-square\n   :alt: LGTM Grade\n\n|\n\n.. image:: https://raw.githubusercontent.com/harens/checkdigit/master/art/demo.gif\n   :alt: checkdigit example gif\n   :align: center\n\n**checkdigit** is a pure Python library built for identification numbers.\nYou want to validate a credit card number, or maybe even calculate a missing digit on an ISBN code?\nWe\'ve got you covered 😎.\n\nWant to know more? Check out the `API Reference and documentation <https://checkdigit.readthedocs.io/en/latest/reference.html>`_!\n\nInstallation\n------------\n\n`MacPorts <https://ports.macports.org/port/py-checkdigit/summary>`_ 🍎\n*************************************************************************\n\n.. code-block::\n\n    sudo port install py-checkdigit\n\n`PyPi <https://pypi.org/project/checkdigit/>`_ 🐍\n**************************************************\n\n.. code-block::\n\n    pip install checkdigit\n\n✨ Features\n------------\n\n* `PEP 561 compatible <https://www.python.org/dev/peps/pep-0561>`_, with built in support for type checking.\n* Capable of calculating missing digits or validating a block of data.\n* Extensive in-code comments and docstrings to explain how it works behind the scenes. \U0001fa84\n\n✅ Supported Formats\n---------------------\n\n* Even and odd binary parity\n* Bookland\n* CRC (credit to `@sapieninja <https://github.com/sapieninja>`_)\n* EAN-13\n* GS1 (credit to `@OtherBarry <https://github.com/OtherBarry>`_)\n* ISBN-10\n* ISBN-13\n* Luhn\n* UPC-A\n\nFor each of these formats, we provide functions to validate them and calculate missing digits.\n\nDo you have any formats that you\'d like to see supported? 🤔 Feel free to raise an issue,\nor even to send a pull request!\n\n🔨 Contributing\n---------------\n\n- Issue Tracker: `<https://github.com/harens/checkdigit/issues>`_\n- Source Code: `<https://github.com/harens/checkdigit>`_\n\nAny change, big or small, that you think can help improve this project is more than welcome 🎉.\n\nAs well as this, feel free to open an issue with any new suggestions or bug reports. Every contribution is appreciated.\n\n🏗 Setup\n*********\n\nFirst, fork the project to your account. Then, run the following with your GitHub handle in place of\n:code:`INSERT_GITHUB_NAME`:\n\n.. code-block:: console\n\n    git clone https://github.com/INSERT_GITHUB_NAME/checkdigit\n    poetry install && poetry shell\n    pre-commit install\n    \n\n🏛 Project structure\n********************\n\n..\n   Credit for file structure: https://stackoverflow.com/a/38819161\n\n::\n\n    checkdigit\n    ├── scripts\n    │   ├── format.sh\n    │   └── tests.sh\n    ├── checkdigit\n    │   ├── gs1.py\n    │   ├── isbn.py\n    │   ├── luhn.py\n    │   └── etc.\n    └── tests\n\nEach new format goes into a separate file which is named accordingly. Similar formats (e.g. ISBN-10 and ISBN-13)\nshould go in the same file.\n\nBefore submitting any new changes, please run the :code:`format.sh` and :code:`tests.sh` scripts beforehand. Thank you :)\n\n📚 Building the Docs\n*********************\n\nThe documentation can be found in :code:`docs/source`.\n\nWe can use `sphinx-autobuild <https://github.com/executablebooks/sphinx-autobuild>`_ to continuously rebuild the docs when changes are made.\n\n.. code-block:: console\n\n    sphinx-autobuild docs/source docs/_build/html\n\n🎪 File structure\n*****************\n\nEach of the Python files follow the same general format:\n\n.. code-block:: python\n\n    # License + File docstring\n\n    from checkdigit._data import cleanse, convert\n\n\n    def calculate(data: str) -> str:\n        """Determines check digit.\n\n        Args:\n            data: A string of data missing a check digit\n\n        Returns:\n            str: The single missing check digit (not the whole block of data)\n        """\n        # This helps to deal with user formatting inconsistencies\n        # e.g. spaces, hyphens, etc.\n        data = cleanse(data)\n\n        # Deals with 10 or 11 being the possible check digit\n        return convert(...)\n\n\n    def validate(data: str) -> bool:\n        """Validates a block of data from the check digit.\n\n        Args:\n            data: A string representing a full block of data\n\n        Returns:\n            bool: A boolean representing whether the data is valid or not\n        """\n        data = cleanse(data)\n\n        # Remove the check digit and see if it matches\n        return calculate(data[:-1]) == data[-1]\n\n\n    def missing(data: str) -> str:\n        """Returns the missing digit from a block of data.\n\n        Args:\n            data: A string with a question mark in the place of a missing digit.\n\n        Returns:\n            A string representing the missing digit (not the whole block of data)\n        """\n        data = cleanse(data)\n\n        return ...\n\nFor similar data formats, the names can be adjusted accordingly (e.g. :code:`validate10` for ISBN-10 and :code:`validate13` for ISBN-13).\n\n📙 License\n-----------\n\nThis project is licensed under `GPL-3.0-or-later <https://github.com/harens/checkdigit/blob/master/LICENSE>`_.\n',
    'author': 'harens',
    'author_email': 'harensdeveloper@gmail.com',
    'maintainer': 'harens',
    'maintainer_email': 'harensdeveloper@gmail.com',
    'url': 'https://checkdigit.readthedocs.io',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6.1,<4.0',
}


setup(**setup_kwargs)
