<!--startmeta
custom_edit_url: "https://github.com/netdata/netdata/edit/master/src/collectors/python.d.plugin/go_expvar/README.md"
meta_yaml: "https://github.com/netdata/netdata/edit/master/src/collectors/python.d.plugin/go_expvar/metadata.yaml"
sidebar_label: "Go applications (EXPVAR)"
learn_status: "Published"
learn_rel_path: "Collecting Metrics/APM"
most_popular: False
message: "DO NOT EDIT THIS FILE DIRECTLY, IT IS GENERATED BY THE COLLECTOR'S metadata.yaml FILE"
endmeta-->

# Go applications (EXPVAR)


<img src="https://netdata.cloud/img/go.png" width="150"/>


Plugin: python.d.plugin
Module: go_expvar

<img src="https://img.shields.io/badge/maintained%20by-Netdata-%2300ab44" />

## Overview

This collector monitors Go applications that expose their metrics with the use of the `expvar` package from the Go standard library. It produces charts for Go runtime memory statistics and optionally any number of custom charts.

It connects via http to gather the metrics exposed via the `expvar` package.

This collector is supported on all platforms.

This collector supports collecting metrics from multiple instances of this integration, including remote instances.


### Default Behavior

#### Auto-Detection

This integration doesn't support auto-detection.

#### Limits

The default configuration for this integration does not impose any limits on data collection.

#### Performance Impact

The default configuration for this integration is not expected to impose a significant performance impact on the system.

## Metrics

Metrics grouped by *scope*.

The scope defines the instance that the metric belongs to. An instance is uniquely identified by a set of labels.



### Per Go applications (EXPVAR) instance

These metrics refer to the entire monitored application.

This scope has no labels.

Metrics:

| Metric | Dimensions | Unit |
|:------|:----------|:----|
| expvar.memstats.heap | alloc, inuse | KiB |
| expvar.memstats.stack | inuse | KiB |
| expvar.memstats.mspan | inuse | KiB |
| expvar.memstats.mcache | inuse | KiB |
| expvar.memstats.live_objects | live | objects |
| expvar.memstats.sys | sys | KiB |
| expvar.memstats.gc_pauses | avg | ns |



## Alerts

There are no alerts configured by default for this integration.


## Setup


### Prerequisites

#### Enable the go_expvar collector

The `go_expvar` collector is disabled by default. To enable it, use `edit-config` from the Netdata [config directory](https://github.com/netdata/netdata/blob/master/docs/netdata-agent/configuration/README.md), which is typically at `/etc/netdata`, to edit the `python.d.conf` file.

```bash
cd /etc/netdata   # Replace this path with your Netdata config directory, if different
sudo ./edit-config python.d.conf
```

Change the value of the `go_expvar` setting to `yes`. Save the file and restart the Netdata Agent with `sudo systemctl restart netdata`, or the [appropriate method](https://github.com/netdata/netdata/blob/master/docs/netdata-agent/start-stop-restart.md) for your system.


#### Sample `expvar` usage in a Go application

The `expvar` package exposes metrics over HTTP and is very easy to use.
Consider this minimal sample below:

```go
package main

import (
        _ "expvar"
        "net/http"
)

func main() {
        http.ListenAndServe("127.0.0.1:8080", nil)
}
```

When imported this way, the `expvar` package registers a HTTP handler at `/debug/vars` that
exposes Go runtime's memory statistics in JSON format. You can inspect the output by opening
the URL in your browser (or by using `wget` or `curl`).

Sample output:

```json
{
"cmdline": ["./expvar-demo-binary"],
"memstats": {"Alloc":630856,"TotalAlloc":630856,"Sys":3346432,"Lookups":27, <omitted for brevity>}
}
```

You can of course expose and monitor your own variables as well.
Here is a sample Go application that exposes a few custom variables:

```go
package main

import (
    "expvar"
    "net/http"
    "runtime"
    "time"
)

func main() {

    tick := time.NewTicker(1 * time.Second)
    num_go := expvar.NewInt("runtime.goroutines")
    counters := expvar.NewMap("counters")
    counters.Set("cnt1", new(expvar.Int))
    counters.Set("cnt2", new(expvar.Float))

    go http.ListenAndServe(":8080", nil)

    for {
        select {
        case <- tick.C:
            num_go.Set(int64(runtime.NumGoroutine()))
            counters.Add("cnt1", 1)
            counters.AddFloat("cnt2", 1.452)
        }
    }
}
```

Apart from the runtime memory stats, this application publishes two counters and the
number of currently running Goroutines and updates these stats every second.



### Configuration

#### Options

There are 2 sections:

* Global variables
* One or more JOBS that can define multiple different instances to monitor.

The following options can be defined globally: priority, penalty, autodetection_retry, update_every, but can also be defined per JOB to override the global values.

Additionally, the following collapsed table contains all the options that can be configured inside a JOB definition.

Every configuration JOB starts with a `job_name` value which will appear in the dashboard, unless a `name` parameter is specified. Each JOB can be used to monitor a different Go application.


<details open><summary>Config options</summary>



| Option | Description | Default | Required |
|:-----|:------------|:--------|:---------:|
| update_every | Sets the default data collection frequency. | 5 | no |
| priority | Controls the order of charts at the netdata dashboard. | 60000 | no |
| autodetection_retry | Sets the job re-check interval in seconds. | 0 | no |
| penalty | Indicates whether to apply penalty to update_every in case of failures. | yes | no |
| name | Job name. This value will overwrite the `job_name` value. JOBS with the same name are mutually exclusive. Only one of them will be allowed running at any time. This allows autodetection to try several alternatives and pick the one that works. |  | no |
| url | the URL and port of the expvar endpoint. Please include the whole path of the endpoint, as the expvar handler can be installed in a non-standard location. |  | yes |
| user | If the URL is password protected, this is the username to use. |  | no |
| pass | If the URL is password protected, this is the password to use. |  | no |
| collect_memstats | Enables charts for Go runtime's memory statistics. |  | no |
| extra_charts | Defines extra data/charts to monitor, please see the example below. |  | no |


</details>



#### via File

The configuration file name for this integration is `python.d/go_expvar.conf`.

The file format is YAML. Generally, the structure is:

```yaml
update_every: 1
autodetection_retry: 0

job_name:
  job_option1: some_value
  job_option2: some_other_vlaue
```
You can edit the configuration file using the [`edit-config`](https://github.com/netdata/netdata/blob/master/docs/netdata-agent/configuration/README.md#edit-a-configuration-file-using-edit-config) script from the
Netdata [config directory](https://github.com/netdata/netdata/blob/master/docs/netdata-agent/configuration/README.md#the-netdata-config-directory).

```bash
cd /etc/netdata 2>/dev/null || cd /opt/netdata/etc/netdata
sudo ./edit-config python.d/go_expvar.conf
```

##### Examples

###### Monitor a Go app1 application

The example below sets a configuration for a Go application, called `app1`. Besides the `memstats`, the application also exposes two counters and the number of currently running Goroutines and updates these stats every second.

The `go_expvar` collector can monitor these as well with the use of the `extra_charts` configuration variable.

The `extra_charts` variable is a YaML list of Netdata chart definitions.
Each chart definition has the following keys:

```
id:         Netdata chart ID
options:    a key-value mapping of chart options
lines:      a list of line definitions
```

**Note: please do not use dots in the chart or line ID field.
See [this issue](https://github.com/netdata/netdata/pull/1902#issuecomment-284494195) for explanation.**

Please see these two links to the official Netdata documentation for more information about the values:

-   [External plugins - charts](https://github.com/netdata/netdata/blob/master/src/plugins.d/README.md#chart)
-   [Chart variables](https://github.com/netdata/netdata/blob/master/src/collectors/python.d.plugin/README.md#global-variables-order-and-chart)

**Line definitions**

Each chart can define multiple lines (dimensions).
A line definition is a key-value mapping of line options.
Each line can have the following options:

```
# mandatory
expvar_key: the name of the expvar as present in the JSON output of /debug/vars endpoint
expvar_type: value type; supported are "float" or "int"
id: the id of this line/dimension in Netdata

# optional - Netdata defaults are used if these options are not defined
name: ''
algorithm: absolute
multiplier: 1
divisor: 100 if expvar_type == float, 1 if expvar_type == int
hidden: False
```

Please see the following link for more information about the options and their default values:
[External plugins - dimensions](https://github.com/netdata/netdata/blob/master/src/plugins.d/README.md#dimension)

Apart from top-level expvars, this plugin can also parse expvars stored in a multi-level map;
All dicts in the resulting JSON document are then flattened to one level.
Expvar names are joined together with '.' when flattening.

Example:

```
{
    "counters": {"cnt1": 1042, "cnt2": 1512.9839999999983},
    "runtime.goroutines": 5
}
```

In the above case, the exported variables will be available under `runtime.goroutines`,
`counters.cnt1` and `counters.cnt2` expvar_keys. If the flattening results in a key collision,
the first defined key wins and all subsequent keys with the same name are ignored.


```yaml
app1:
 name : 'app1'
 url  : 'http://127.0.0.1:8080/debug/vars'
 collect_memstats: true
 extra_charts:
   - id: "runtime_goroutines"
     options:
       name: num_goroutines
       title: "runtime: number of goroutines"
       units: goroutines
       family: runtime
       context: expvar.runtime.goroutines
       chart_type: line
     lines:
       - {expvar_key: 'runtime.goroutines', expvar_type: int, id: runtime_goroutines}
   - id: "foo_counters"
     options:
       name: counters
       title: "some random counters"
       units: awesomeness
       family: counters
       context: expvar.foo.counters
       chart_type: line
     lines:
       - {expvar_key: 'counters.cnt1', expvar_type: int, id: counters_cnt1}
       - {expvar_key: 'counters.cnt2', expvar_type: float, id: counters_cnt2}

```


## Troubleshooting

### Debug Mode


To troubleshoot issues with the `go_expvar` collector, run the `python.d.plugin` with the debug option enabled. The output
should give you clues as to why the collector isn't working.

- Navigate to the `plugins.d` directory, usually at `/usr/libexec/netdata/plugins.d/`. If that's not the case on
  your system, open `netdata.conf` and look for the `plugins` setting under `[directories]`.

  ```bash
  cd /usr/libexec/netdata/plugins.d/
  ```

- Switch to the `netdata` user.

  ```bash
  sudo -u netdata -s
  ```

- Run the `python.d.plugin` to debug the collector:

  ```bash
  ./python.d.plugin go_expvar debug trace
  ```

### Getting Logs

If you're encountering problems with the `go_expvar` collector, follow these steps to retrieve logs and identify potential issues:

- **Run the command** specific to your system (systemd, non-systemd, or Docker container).
- **Examine the output** for any warnings or error messages that might indicate issues.  These messages should provide clues about the root cause of the problem.

#### System with systemd

Use the following command to view logs generated since the last Netdata service restart:

```bash
journalctl _SYSTEMD_INVOCATION_ID="$(systemctl show --value --property=InvocationID netdata)" --namespace=netdata --grep go_expvar
```

#### System without systemd

Locate the collector log file, typically at `/var/log/netdata/collector.log`, and use `grep` to filter for collector's name:

```bash
grep go_expvar /var/log/netdata/collector.log
```

**Note**: This method shows logs from all restarts. Focus on the **latest entries** for troubleshooting current issues.

#### Docker Container

If your Netdata runs in a Docker container named "netdata" (replace if different), use this command:

```bash
docker logs netdata 2>&1 | grep go_expvar
```


