/*
-------------------------------------------------------------------------
OBJECT NAME:	stats.c

FULL NAME:	Statistics Window & callbacks

ENTRY POINTS:	ViewStats()
		SetStatsData()
		ComputeStats()

STATIC FNS:	CreateStatsWindow()
		DismissStats()
		PrintStats()
		formatLine()

DESCRIPTION:	

REFERENCES:	none

REFERENCED BY:	Menu button.

COPYRIGHT:	University Corporation for Atmospheric Research, 1997-2018
-------------------------------------------------------------------------
*/

#include "define.h"
#include "ps.h"

#include <ctype.h>

#define NO_NETCDF_2

#include <netcdf.h>

#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

static Widget StatsShell, StatsWindow, statsText;

static void
	CreateStatsWindow(),
	PrintStats(Widget w, XtPointer client, XtPointer call),
	DismissStats(Widget w, XtPointer client, XtPointer call);
static char *formatLine(char *, DATASET_INFO *);

static char *statTitle = "Variable              nPoints             Min          Max         Mean       Sigma         Var\n";

extern Widget	AppShell;


/* -------------------------------------------------------------------- */
static void DismissStats(Widget w, XtPointer client, XtPointer call)
{
  StatsWinOpen = False;

  if (StatsWindow)
    {
    XtUnmanageChild(StatsWindow);
    XtPopdown(XtParent(StatsWindow));
    }

}	/* END DISMISSSTATS */

/* -------------------------------------------------------------------- */
void ViewStats(Widget w, XtPointer client, XtPointer call)
{
  SetStatsData();

  if (StatsWindow)
    {
    XtManageChild(StatsWindow);
    XtPopup(XtParent(StatsWindow), XtGrabNone);
    StatsWinOpen = True;
    }

}	/* END VIEWSTATS */

/* -------------------------------------------------------------------- */
void SetStatsData()
{
  size_t i;

  static bool	firstTime = True;

  if (firstTime) {
    CreateStatsWindow();
    firstTime = False;
    }

  XmTextSetString(statsText, statTitle);

  for (i = 0; i < NumberDataSets; ++i)
    {
    formatLine(buffer, &dataSet[i]);
    XmTextInsert(statsText, XmTextGetLastPosition(statsText), buffer);
    }

  for (i = 0; i < NumberXYXsets; ++i)
    {
    formatLine(buffer, &xyXset[i]);
    XmTextInsert(statsText, XmTextGetLastPosition(statsText), buffer);
    }

  for (i = 0; i < NumberXYYsets; ++i)
    {
    formatLine(buffer, &xyYset[i]);
    XmTextInsert(statsText, XmTextGetLastPosition(statsText), buffer);
    }

  for (i = 0; i < 3; ++i)
    if (xyzSet[i].varInfo)
      {
      formatLine(buffer, &xyzSet[i]);
      XmTextInsert(statsText, XmTextGetLastPosition(statsText), buffer);
      }

  if (WindBarbs)
    {
    formatLine(buffer, &ui);
    XmTextInsert(statsText, XmTextGetLastPosition(statsText), buffer);

    formatLine(buffer, &vi);
    XmTextInsert(statsText, XmTextGetLastPosition(statsText), buffer);
    }

}	/* END SETSTATSDATA */

/* -------------------------------------------------------------------- */
void ComputeStats(DATASET_INFO *set)
{
  size_t i, missCnt, len = 0;
  double y, sum, sigma;
 
  /* Read in variables units from file.
   */
  buffer[0] = '\0';

  if (set->varInfo->inVarID != COMPUTED)
    {
    nc_inq_attlen(dataFile[set->fileIndex].ncid, set->varInfo->inVarID,
						"units", (size_t *)&len);
    nc_get_att_text(dataFile[set->fileIndex].ncid, set->varInfo->inVarID,
						"units", buffer);
    buffer[len] = '\0';
    while (isspace(buffer[--len]) || buffer[len] == 0) // Strip trailing spaces.
      buffer[len] = '\0';

    if (strcmp(buffer, "C") == 0 ||
        strcmp(buffer, "deg_C") == 0)
      {
      buffer[0] = 0xb0;
      buffer[1] = 'C';
      buffer[2] = '\0';
      }
    }
  else
    strcpy(buffer, "Unknown");

  set->stats.units = buffer;

  missCnt = 0;
  sum = sigma = 0.0;
 
  set->stats.min = FLT_MAX;
  set->stats.max = -FLT_MAX;
 
  for (i = 0; i < set->nPoints; ++i)
    {
    if (isMissingValue((y = set->data[i]), set->missingValue) ||
        y < set->stats.outlierMin || y > set->stats.outlierMax)
      {
      ++missCnt;
      continue;
      }
 
    set->stats.min = std::min(set->stats.min, y);
    set->stats.max = std::max(set->stats.max, y);

    sum += y;
    }
 
  if (set->nPoints == missCnt)
    {
    set->stats.mean = 0.0;
    set->stats.min = 0.0;
    set->stats.max = 0.0;
    }
  else
    {
    set->stats.nPoints = set->nPoints - missCnt;
    set->stats.mean = sum / set->stats.nPoints;
    }

  for (i = 0; i < set->nPoints; ++i)
    if (!isMissingValue((y = set->data[i]), set->missingValue) ||
        y < set->stats.outlierMin || y > set->stats.outlierMax)
      sigma += pow(y - set->stats.mean, 2.0);
 
  set->stats.variance = sigma / (set->stats.nPoints - 1);
  set->stats.sigma = (float)sqrt(sigma / (set->stats.nPoints - 1));

}   /* END COMPUTESTATS */

/* -------------------------------------------------------------------- */
static void CreateStatsWindow()
{
  Arg         args[8];
  Cardinal    n;
  Widget      drFrame, drRC, b[3];

  n = 0;
  StatsShell = XtCreatePopupShell("statsShell",
                  topLevelShellWidgetClass, AppShell, args, n);

  n = 0;
  StatsWindow = XmCreateForm(StatsShell, (char *)"statsForm", args, n);

  n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  drFrame = XmCreateFrame(StatsWindow, (char *)"buttonFrame", args, n);
  XtManageChild(drFrame);

  n = 0;
  drRC = XmCreateRowColumn(drFrame, (char *)"buttonRC", args, n);
  XtManageChild(drRC);


  n = 0;
  b[0] = XmCreatePushButton(drRC, (char *)"dismissButton", args, n);
  b[1] = XmCreatePushButton(drRC, (char *)"printButton", args, n);
  b[2] = XmCreatePushButton(drRC, (char *)"parmsButton", args, n);
  XtManageChildren(b, 3);
  XtAddCallback(b[0], XmNactivateCallback, DismissStats, StatsWindow);
  XtAddCallback(b[1], XmNactivateCallback, PrintStats, NULL);
  XtAddCallback(b[2], XmNactivateCallback, EditStatsParms, NULL);


  n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(args[n], XmNtopWidget, drFrame); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  statsText = XmCreateScrolledText(StatsWindow, (char *)"statsText", args, n);
  XtManageChild(statsText);

}	/* END CREATESTATSWINDOW */

/* -------------------------------------------------------------------- */
static void PrintStats(Widget w, XtPointer client, XtPointer call)
{
  FILE *fp;
  size_t i;
  char *p;

  if ((p = getenv("LPDEST")) != NULL)
    printf("Output being sent to %s.\n", p);

  if ((fp = popen(printerSetup.lpCommand.c_str(), "w")) == NULL)
    {
    ShowError("PrintStats: can't open pipe to 'lp'");
    return;
    }


  fprintf(fp, "%s, %s\n\n",
	mainPlot[0].title.c_str(), mainPlot[0].subTitle.c_str());
  fprintf(fp, "%s", statTitle);

  for (i = 0; i < NumberDataSets; ++i)
    {
    formatLine(buffer, &dataSet[i]);
    fprintf(fp, "%s", buffer);
    }

  for (i = 0; i < NumberXYXsets; ++i)
    {
    formatLine(buffer, &xyXset[i]);
    fprintf(fp, "%s", buffer);
    }

  for (i = 0; i < NumberXYYsets; ++i)
    {
    formatLine(buffer, &xyYset[i]);
    fprintf(fp, "%s", buffer);
    }

  for (i = 0; i < 3; ++i)
    if (xyzSet[i].varInfo)
      {
      formatLine(buffer, &xyzSet[i]);
      fprintf(fp, "%s", buffer);
      }

  if (WindBarbs)
    {
    formatLine(buffer, &ui);
    fprintf(fp, "%s", buffer);

    formatLine(buffer, &vi);
    fprintf(fp, "%s", buffer);
    }

  pclose(fp);

}	/* END PRINTSTATS */

/* -------------------------------------------------------------------- */
static char *formatLine(char buff[], DATASET_INFO *set)
{
  char	temp[64];

  memset(buff, ' ', 256);

  sprintf(temp, "%s (%s)", set->varInfo->name.c_str(), set->stats.units.c_str());
  memcpy(buff, temp, strlen(temp));

  sprintf(temp, "%5ld/%ld", (long)set->stats.nPoints, (long)set->nPoints);
  memcpy(&buff[20], temp, strlen(temp));

  sprintf(&buff[36], "%12.6e %12.6e %12.6e %12.6e %12.6e\n",
      set->stats.min,
      set->stats.max,
      set->stats.mean,
      set->stats.sigma,
      set->stats.variance);

  return(buff);

}

/* END STATS.C */
