"""
This pre-processor parses a single file containing a list of
MP_REGISTER_ROOT_POINTER(variable declaration) items.

These are used to generate a header with the required entries for
"struct _mp_state_vm_t" in py/mpstate.h
"""

import argparse
import io
import re


PATTERN = re.compile(r"MP_REGISTER_ROOT_POINTER\((.*?)\);")


def find_root_pointer_registrations(filename):
    """Find any MP_REGISTER_ROOT_POINTER definitions in the provided file.

    :param str filename: path to file to check
    :return: List[variable_declaration]
    """
    with io.open(filename, encoding="utf-8") as c_file_obj:
        return set(re.findall(PATTERN, c_file_obj.read()))


def generate_root_pointer_header(root_pointers):
    """Generate header with root pointer entries.

    :param List[variable_declaration] root_pointers: root pointer declarations
    :return: None
    """

    # Print header file for all external modules.
    print("// Automatically generated by make_root_pointers.py.")
    print()

    for item in root_pointers:
        print(item, end=";")
        print()


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("file", nargs=1, help="file with MP_REGISTER_ROOT_POINTER definitions")
    args = parser.parse_args()

    root_pointers = find_root_pointer_registrations(args.file[0])
    generate_root_pointer_header(sorted(root_pointers))


if __name__ == "__main__":
    main()
