//
// Copyright (C) 2011-14 Mark Wiebe, DyND Developers
// BSD 2-Clause License, see LICENSE.txt
//

#include <time.h>

#include <dynd/types/time_util.hpp>
#include <dynd/types/time_parser.hpp>
#include <dynd/types/date_util.hpp>
#include <dynd/types/cstruct_type.hpp>

using namespace std;
using namespace dynd;

 
int64_t time_hmst::to_ticks(int hour, int minute, int second, int tick)
{
    if (is_valid(hour, minute, second, tick)) {
        return static_cast<int64_t>(tick) +
            second * DYND_TICKS_PER_SECOND +
            minute * DYND_TICKS_PER_MINUTE +
            hour * DYND_TICKS_PER_HOUR;
    } else {
        return DYND_TIME_NA;
    }
}

std::string time_hmst::to_str(int hour, int minute, int second, int tick)
{
    string s;
    if (is_valid(hour, minute, second, tick)) {
        s.resize(2 + 1 + 2 + 1 + 2 + 1 + 7);
        s[0] = '0' + (hour / 10);
        s[1] = '0' + (hour % 10);
        s[2] = ':';
        s[3] = '0' + (minute / 10);
        s[4] = '0' + (minute % 10);
        if (second != 0 || tick != 0) {
            s[5] = ':';
            s[6] = '0' + (second / 10);
            s[7] = '0' + (second % 10);
            if (tick != 0) {
                s[8] = '.';
                int i = 9, divisor = 1000000;
                while (tick != 0) {
                    s[i] = '0' + (tick / divisor);
                    tick = tick % divisor;
                    divisor = divisor / 10;
                    ++i;
                }
                s.resize(i);
            } else {
                s.resize(8);
            }
        } else {
            s.resize(5);
        }
    }
    return s;
}

void time_hmst::set_from_ticks(int64_t ticks)
{
    if (ticks >= 0 && ticks < DYND_TICKS_PER_DAY) {
        tick = static_cast<int32_t>(ticks % DYND_TICKS_PER_SECOND);
        ticks = ticks / DYND_TICKS_PER_SECOND;
        second = static_cast<int8_t>(ticks % 60);
        ticks = ticks / 60;
        minute = static_cast<int8_t>(ticks % 60);
        hour = static_cast<int8_t>(ticks / 60);
    } else {
        set_to_na();
    }
}

void time_hmst::set_from_str(const std::string& s)
{
    if (!string_to_time(s.data(), s.data() + s.size(), *this)) {
        stringstream ss;
        ss << "Unable to parse ";
        print_escaped_utf8_string(ss, s);
        ss << " as a time";
        throw invalid_argument(ss.str());
    }
}

time_hmst time_hmst::get_current_local_time()
{
    // TODO: Could use C++11 chrono library
    struct tm tm_;
#if defined(_MSC_VER)
    __time64_t rawtime;
    _time64(&rawtime);
    if (_localtime64_s(&tm_, &rawtime) != 0) {
        throw std::runtime_error("Failed to use '_localtime64_s' to convert "
                                "to a local time");
    }
#else
    time_t rawtime;
    time(&rawtime);
    if (localtime_r(&rawtime, &tm_) == NULL) {
        throw std::runtime_error("Failed to use 'localtime_r' to convert "
                                "to a local time");
    }
#endif
    time_hmst hmst;
    hmst.hour = tm_.tm_hour;
    hmst.minute = tm_.tm_min;
    hmst.second = tm_.tm_sec;
    hmst.tick = 0;
    return hmst;
}
 
const ndt::type& time_hmst::type()
{
    static ndt::type tp = ndt::make_cstruct(
            ndt::make_type<int8_t>(), "hour",
            ndt::make_type<int8_t>(), "minute",
            ndt::make_type<int8_t>(), "second",
            ndt::make_type<int32_t>(), "tick");
    return tp;
}
