/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_auto_ber_bf.h>
#include <gnuradio/io_signature.h>

#include <gnuradio/digital/glfsr.h>

#include <stdio.h>

/*
 * Create a new instance of baz_auto_ber_bf and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_auto_ber_bf_sptr baz_make_auto_ber_bf (int degree, int sync_bits, int sync_decim)
{
	return baz_auto_ber_bf_sptr (new baz_auto_ber_bf (degree, sync_bits, sync_decim));
}

/*
 * Specify constraints on number of input and output streams.
 * This info is used to construct the input and output signatures
 * (2nd & 3rd args to gr::block's constructor).  The input and
 * output signatures are used by the runtime system to
 * check that a valid number and type of inputs and outputs
 * are connected to this block.  In this case, we accept
 * only 1 input and 1 output.
 */
static const int MIN_IN = 1;	// mininum number of input streams
static const int MAX_IN = 1;	// maximum number of input streams
static const int MIN_OUT = 1;	// minimum number of output streams
static const int MAX_OUT = 1;	// maximum number of output streams

/*
 * The private constructor
 */
baz_auto_ber_bf::baz_auto_ber_bf (int degree, int sync_bits, int sync_decim)
	: gr::sync_block ("auto_ber_bf",
		gr::io_signature::make (MIN_IN, MAX_IN, sizeof (char)),
		gr::io_signature::make (MIN_OUT, MAX_OUT, sizeof (float)))
	, d_sync_bit_length(sync_bits)
{
	d_glfsr_length = (unsigned int)((1ULL << degree)-1);
	d_glfsr_rounded_length = d_glfsr_length + 1;
	int mask = 0;
	int seed = 1;	// MAGIC
	if (mask == 0)
		mask = gr::digital::glfsr::glfsr_mask(degree);
	d_glfsr = new gr::digital::glfsr(mask, seed);
	
	int i = 0;
	uint64_t word = 0;
	int word_count = 0;
	int bit_count = 0;
	for (; i < d_glfsr_rounded_length; i++)
	{
		word <<= 1;
		if (i < d_glfsr_length)
		{
			if (d_glfsr->next_bit())
				word |= 1;
		}
		++bit_count;
		if (bit_count == sync_bits)
		{
			d_sync_list.push_back(word);
			
			if ((word_count % sync_decim) == 0)
			{
				if (d_dupe_map.find(word) != d_dupe_map.end())
				{
					d_dupe_map[word] += 1;
				}
				else if (d_sync_map.find(word) != d_sync_map.end())
				{
					//fprintf(stderr, "Already saw word %llx\n", word);
					
					d_sync_map.erase(word);
					
					if (d_dupe_map.find(word) != d_dupe_map.end())
						d_dupe_map[word] += 1;	// Shouldn't get here
					else
						d_dupe_map[word] = 1;
				}
				else
				{
					d_sync_map[word] = d_sync_list.size() - 1;
				}
			}
			
			word = 0;
			bit_count = 0;
			++word_count;
		}
	}
	
	fprintf(stderr, "Sync map count: %d\n", d_sync_map.size());
	fprintf(stderr, "Dupe map count: %d\n", d_dupe_map.size());
	
	if (bit_count > 0)
	{
		fprintf(stderr, "Residual bit count: %d\n", bit_count);
	}
}

/*
 * Our virtual destructor.
 */
baz_auto_ber_bf::~baz_auto_ber_bf ()
{
	delete d_glfsr;
}
/*
void baz_auto_ber_bf::set_exponent(float exponent)
{
  d_exponent = exponent;
}
*/
int baz_auto_ber_bf::work (int noutput_items, gr_vector_const_void_star &input_items, gr_vector_void_star &output_items)
{
	const unsigned char *in = (const unsigned char *) input_items[0];
	float *out = (float *) output_items[0];

	d_current_word <<= 1;
	if (*in++)
		d_current_word |= 1;
	
	uint64_t word = d_current_word & (1ULL << d_sync_bit_length);
	SyncMap::iterator it = d_sync_map.find(word);
	if (it != d_sync_map.end())
	{
		// Found a sync word
		
		it->second;	// Index into sync list (if there are dupes, this won't work and will have to wait for next)
	}

	return noutput_items;
}
