//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// $Id: G4AtomicShells.hh 95972 2016-03-04 19:04:49Z vnivanch $

// class description
//
// Class consists of atomic subshell binding energies for first 120 elements.
// Data taken from the references for the Band-Raman Internal Conversion
// Coefficients, BrIcc, code.
// see: http://www.nndc.bnl.gov/nndcscr/ensdf_pgm/analysis/BrIcc/
//
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....

// 24-04-98, implementation of the first version, V. Grichine
// 16-11-98, GetBindingEnergy(Z,ShellNb), M.Maire
// 11-02-05, GetNumberOfElectrons(Z,ShellNb), V.Ivanchenko
// 30-04-10, added fIndexOfShells, V.Ivanchenko
// 08-04-15, Complete re-write of the class. Same methods but
//           values modified to match EADL source data up to
//           element Z=120, L.G. Sarmiento

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....

#ifndef G4ATOMICSHELLS_EADL_H
#define G4ATOMICSHELLS_EADL_H

#include "globals.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....

class G4AtomicShells_EADL
{
public :  //with description

  static G4int    GetNumberOfShells    (G4int Z);
  static G4int    GetNumberOfElectrons (G4int Z, G4int SubshellNb);
  static G4double GetBindingEnergy     (G4int Z, G4int SubshellNb);
  static G4double GetTotalBindingEnergy(G4int Z);

private :

#ifdef G4VERBOSE
  static G4int PrintErrorZ(G4int Z, const G4String&);
  static G4int PrintErrorShell(G4int Z, G4int SubshellNb, const G4String&);
#endif

  G4AtomicShells_EADL(const G4AtomicShells_EADL&) = delete;
  const G4AtomicShells_EADL& operator=(const G4AtomicShells_EADL&) = delete;

  static const G4int    fNumberOfShells   [121];
  static const G4int    fIndexOfShells    [121];
  static const G4int    fNumberOfElectrons[2167];
  static const G4double fBindingEnergies  [2167];

};


/******************************************************************************
Below is the atomic subshell binding energy table for elements Z=1-120.
The data was taken from:

       Z  =   0 : This is just element zero for convenience so
                  the index matches the atomic number

  1 <= Z <=   4 : T.A.Carlson, Photoelectron and Auger Spectroscopy,
                  Plenum, New York, 1976, p.337.

  5 <= Z <= 110 : T. kibedi et al., Evaluation of theoretical conversion
                  coefficients using BrIcc,
                  Nuclear Instrument and methods in Physics Research A 589
                  (2008) 202-229*,**.

111 <= Z <= 120 : T. kibedi et al., Conversion coefficients for superheavy
                  elements,
                  Atomic Data and Nuclear Data Tables 98 (2012) 313-355***.

                 ---->      (All data in keV)       <----

  *: O4 binding energy for Z=65  is missing from the table.
     Value from Z=64 adopted.

 **: P4 binding energy for Z=100 is present in table but never used.

***: The following element names have been modified from the displayed in
     the table.

                                  Z=113 Ut -> Uut,
                                  Z=114 Uq ->  Fl,
                                  Z=115 Up -> Uup,
                                  Z=116 Uh ->  Lv,
                                  Z=117 Us -> Uus,
                                  Z=118 Uo -> Uuo.
     Note that this names are not the ones to be used in Geant4




// ----------------------------------------------

  0    Z =   0
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (0)   	  0.0000 


// ----------------------------------------------

  H    Z =   1
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (1)   	  0.0136 


// ----------------------------------------------

  He   Z =   2
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.0246 


// ----------------------------------------------

  Li   Z =   3
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.0580 
  2 s1/2 (1)   	  0.0054 


// ----------------------------------------------

  Be   Z =   4
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.1150 
  2 s1/2 (2)   	  0.0093 


// ----------------------------------------------

  B    Z =   5
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.1880 
  2 s1/2 (2)   	  0.0126 
  2 p1/2 (1)   	  0.0047 


// ----------------------------------------------

  C    Z =   6
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.2841 
  2 s1/2 (2)   	  0.0181 
  2 p1/2 (2)   	  0.0090 


// ----------------------------------------------

  N    Z =   7
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.4005 
  2 s1/2 (2)   	  0.0243 
  2 p3/2 (1)   	  0.0097 
  2 p1/2 (2)   	  0.0089 


// ----------------------------------------------

  O    Z =   8
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.5320 
  2 s1/2 (2)   	  0.0237 
  2 p3/2 (2)   	  0.0074 
  2 p1/2 (2)   	  0.0068 


// ----------------------------------------------

  F    Z =   9
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.6854 
  2 s1/2 (2)   	  0.0340 
  2 p3/2 (3)   	  0.0087 
  2 p1/2 (2)   	  0.0084 


// ----------------------------------------------

  Ne   Z =  10
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  0.8701 
  2 s1/2 (2)   	  0.0485 
  2 p1/2 (2)   	  0.0217 
  2 p3/2 (4)   	  0.0216 


// ----------------------------------------------

  Na   Z =  11
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  1.0721 
  2 s1/2 (2)   	  0.0633 
  2 p1/2 (2)   	  0.0311 
  2 p3/2 (4)   	  0.0310 
  3 s1/2 (1)   	  0.0007 


// ----------------------------------------------

  Mg   Z =  12
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  1.3050 
  2 s1/2 (2)   	  0.0894 
  2 p1/2 (2)   	  0.0515 
  2 p3/2 (4)   	  0.0513 
  3 s1/2 (2)   	  0.0021 


// ----------------------------------------------

  Al   Z =  13
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  1.5596 
  2 s1/2 (2)   	  0.1177 
  2 p1/2 (2)   	  0.0732 
  2 p3/2 (4)   	  0.0727 
  3 p1/2 (1)   	  0.0060 
  3 s1/2 (2)   	  0.0007 


// ----------------------------------------------

  Si   Z =  14
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  1.8389 
  2 s1/2 (2)   	  0.1487 
  2 p1/2 (2)   	  0.0995 
  2 p3/2 (4)   	  0.0989 
  3 s1/2 (2)   	  0.0076 
  3 p1/2 (2)   	  0.0030 


// ----------------------------------------------

  P    Z =  15
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  2.1455 
  2 s1/2 (2)   	  0.1893 
  2 p1/2 (2)   	  0.1362 
  2 p3/2 (4)   	  0.1353 
  3 s1/2 (2)   	  0.0162 
  3 p3/2 (1)   	  0.0101 
  3 p1/2 (2)   	  0.0096 


// ----------------------------------------------

  S    Z =  16
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  2.4720 
  2 s1/2 (2)   	  0.2292 
  2 p1/2 (2)   	  0.1654 
  2 p3/2 (4)   	  0.1642 
  3 s1/2 (2)   	  0.0158 
  3 p3/2 (2)   	  0.0082 
  3 p1/2 (2)   	  0.0078 


// ----------------------------------------------

  Cl   Z =  17
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  2.8224 
  2 s1/2 (2)   	  0.2702 
  2 p1/2 (2)   	  0.2016 
  2 p3/2 (4)   	  0.2000 
  3 s1/2 (2)   	  0.0175 
  3 p3/2 (3)   	  0.0067 
  3 p1/2 (2)   	  0.0067 


// ----------------------------------------------

  Ar   Z =  18
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  3.2029 
  2 s1/2 (2)   	  0.3260 
  2 p1/2 (2)   	  0.2506 
  2 p3/2 (4)   	  0.2485 
  3 s1/2 (2)   	  0.0292 
  3 p1/2 (2)   	  0.0159 
  3 p3/2 (4)   	  0.0158 


// ----------------------------------------------

  K    Z =  19
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  3.6074 
  2 s1/2 (2)   	  0.3771 
  2 p1/2 (2)   	  0.2963 
  2 p3/2 (4)   	  0.2936 
  3 s1/2 (2)   	  0.0339 
  3 p1/2 (2)   	  0.0181 
  3 p3/2 (4)   	  0.0178 
  4 s1/2 (1)   	  0.0001 


// ----------------------------------------------

  Ca   Z =  20
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  4.0381 
  2 s1/2 (2)   	  0.4378 
  2 p1/2 (2)   	  0.3500 
  2 p3/2 (4)   	  0.3464 
  3 s1/2 (2)   	  0.0437 
  3 p1/2 (2)   	  0.0258 
  3 p3/2 (4)   	  0.0255 
  4 s1/2 (2)   	  0.0018 


// ----------------------------------------------

  Sc   Z =  21
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  4.4928 
  2 s1/2 (2)   	  0.5004 
  2 p1/2 (2)   	  0.4067 
  2 p3/2 (4)   	  0.4022 
  3 s1/2 (2)   	  0.0538 
  3 p1/2 (2)   	  0.0338 
  3 p3/2 (4)   	  0.0315 
  3 d3/2 (1)   	  0.0066 
  4 s1/2 (2)   	  0.0017 


// ----------------------------------------------

  Ti   Z =  22
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  4.9664 
  2 s1/2 (2)   	  0.5637 
  2 p1/2 (2)   	  0.4615 
  2 p3/2 (4)   	  0.4555 
  3 s1/2 (2)   	  0.0603 
  3 p1/2 (2)   	  0.0356 
  3 p3/2 (4)   	  0.0322 
  3 d3/2 (2)   	  0.0037 
  4 s1/2 (2)   	  0.0016 


// ----------------------------------------------

  V    Z =  23
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  5.4651 
  2 s1/2 (2)   	  0.6282 
  2 p1/2 (2)   	  0.5205 
  2 p3/2 (4)   	  0.5129 
  3 s1/2 (2)   	  0.0665 
  3 p1/2 (2)   	  0.0400 
  3 p3/2 (4)   	  0.0350 
  3 d3/2 (3)   	  0.0022 
  4 s1/2 (2)   	  0.0017 


// ----------------------------------------------

  Cr   Z =  24
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  5.9892 
  2 s1/2 (2)   	  0.6946 
  2 p1/2 (2)   	  0.5837 
  2 p3/2 (4)   	  0.5745 
  3 s1/2 (2)   	  0.0741 
  3 p1/2 (2)   	  0.0459 
  3 p3/2 (4)   	  0.0399 
  3 d3/2 (4)   	  0.0029 
  3 d5/2 (1)   	  0.0022 
  4 s1/2 (1)   	  0.0010 


// ----------------------------------------------

  Mn   Z =  25
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  6.5390 
  2 s1/2 (2)   	  0.7690 
  2 p1/2 (2)   	  0.6514 
  2 p3/2 (4)   	  0.6403 
  3 s1/2 (2)   	  0.0839 
  3 p1/2 (2)   	  0.0531 
  3 p3/2 (4)   	  0.0464 
  3 d3/2 (4)   	  0.0035 
  3 d5/2 (1)   	  0.0027 
  4 s1/2 (2)   	  0.0019 


// ----------------------------------------------

  Fe   Z =  26
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  7.1130 
  2 s1/2 (2)   	  0.8461 
  2 p1/2 (2)   	  0.7211 
  2 p3/2 (4)   	  0.7081 
  3 s1/2 (2)   	  0.0929 
  3 p1/2 (2)   	  0.0581 
  3 p3/2 (4)   	  0.0520 
  3 d3/2 (4)   	  0.0039 
  3 d5/2 (2)   	  0.0031 
  4 s1/2 (2)   	  0.0021 


// ----------------------------------------------

  Co   Z =  27
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  7.7089 
  2 s1/2 (2)   	  0.9256 
  2 p1/2 (2)   	  0.7936 
  2 p3/2 (4)   	  0.7786 
  3 s1/2 (2)   	  0.1007 
  3 p1/2 (2)   	  0.0632 
  3 p3/2 (4)   	  0.0577 
  3 d5/2 (3)   	  0.0033 
  3 d3/2 (4)   	  0.0027 
  4 s1/2 (2)   	  0.0019 


// ----------------------------------------------

  Ni   Z =  28
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  8.3328 
  2 s1/2 (2)   	  1.0081 
  2 p1/2 (2)   	  0.8719 
  2 p3/2 (4)   	  0.8547 
  3 s1/2 (2)   	  0.1118 
  3 p1/2 (2)   	  0.0712 
  3 p3/2 (4)   	  0.0697 
  3 d3/2 (4)   	  0.0039 
  3 d5/2 (4)   	  0.0033 
  4 s1/2 (2)   	  0.0022 


// ----------------------------------------------

  Cu   Z =  29
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  8.9789 
  2 s1/2 (2)   	  1.0961 
  2 p1/2 (2)   	  0.9510 
  2 p3/2 (4)   	  0.9311 
  3 s1/2 (2)   	  0.1198 
  3 p1/2 (2)   	  0.0753 
  3 p3/2 (4)   	  0.0728 
  3 d3/2 (4)   	  0.0018 
  3 d5/2 (6)   	  0.0015 
  4 s1/2 (1)   	  0.0012 


// ----------------------------------------------

  Zn   Z =  30
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	  9.6586 
  2 s1/2 (2)   	  1.1936 
  2 p1/2 (2)   	  1.0428 
  2 p3/2 (4)   	  1.0197 
  3 s1/2 (2)   	  0.1359 
  3 p1/2 (2)   	  0.0886 
  3 p3/2 (4)   	  0.0856 
  3 d5/2 (6)   	  0.0080 
  3 d3/2 (4)   	  0.0079 
  4 s1/2 (2)   	  0.0013 


// ----------------------------------------------

  Ga   Z =  31
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 10.3671 
  2 s1/2 (2)   	  1.2977 
  2 p1/2 (2)   	  1.1423 
  2 p3/2 (4)   	  1.1154 
  3 s1/2 (2)   	  0.1581 
  3 p1/2 (2)   	  0.1068 
  3 p3/2 (4)   	  0.1029 
  3 d3/2 (4)   	  0.0207 
  3 d5/2 (6)   	  0.0157 
  4 s1/2 (2)   	  0.0056 
  4 p1/2 (1)   	  0.0008 


// ----------------------------------------------

  Ge   Z =  32
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 11.1031 
  2 s1/2 (2)   	  1.4143 
  2 p1/2 (2)   	  1.2478 
  2 p3/2 (4)   	  1.2167 
  3 s1/2 (2)   	  0.1800 
  3 p1/2 (2)   	  0.1279 
  3 p3/2 (4)   	  0.1208 
  3 d3/2 (4)   	  0.0292 
  3 d5/2 (6)   	  0.0285 
  4 s1/2 (2)   	  0.0090 
  4 p1/2 (2)   	  0.0023 


// ----------------------------------------------

  As   Z =  33
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 11.8667 
  2 s1/2 (2)   	  1.5265 
  2 p1/2 (2)   	  1.3586 
  2 p3/2 (4)   	  1.3231 
  3 s1/2 (2)   	  0.2035 
  3 p1/2 (2)   	  0.1464 
  3 p3/2 (4)   	  0.1405 
  3 d3/2 (4)   	  0.0417 
  3 d5/2 (6)   	  0.0409 
  4 s1/2 (2)   	  0.0125 
  4 p3/2 (1)   	  0.0025 
  4 p1/2 (2)   	  0.0025 


// ----------------------------------------------

  Se   Z =  34
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 12.6578 
  2 s1/2 (2)   	  1.6539 
  2 p1/2 (2)   	  1.4762 
  2 p3/2 (4)   	  1.4358 
  3 s1/2 (2)   	  0.2315 
  3 p1/2 (2)   	  0.1682 
  3 p3/2 (4)   	  0.1619 
  3 d3/2 (4)   	  0.0574 
  3 d5/2 (6)   	  0.0564 
  4 s1/2 (2)   	  0.0162 
  4 p3/2 (2)   	  0.0056 
  4 p1/2 (2)   	  0.0056 


// ----------------------------------------------

  Br   Z =  35
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 13.4737 
  2 s1/2 (2)   	  1.7820 
  2 p1/2 (2)   	  1.5960 
  2 p3/2 (4)   	  1.5499 
  3 s1/2 (2)   	  0.2565 
  3 p1/2 (2)   	  0.1893 
  3 p3/2 (4)   	  0.1815 
  3 d3/2 (4)   	  0.0701 
  3 d5/2 (6)   	  0.0690 
  4 s1/2 (2)   	  0.0273 
  4 p1/2 (2)   	  0.0052 
  4 p3/2 (3)   	  0.0046 


// ----------------------------------------------

  Kr   Z =  36
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 14.3256 
  2 s1/2 (2)   	  1.9210 
  2 p1/2 (2)   	  1.7272 
  2 p3/2 (4)   	  1.6749 
  3 s1/2 (2)   	  0.2921 
  3 p1/2 (2)   	  0.2221 
  3 p3/2 (4)   	  0.2144 
  3 d3/2 (4)   	  0.0950 
  3 d5/2 (6)   	  0.0938 
  4 s1/2 (2)   	  0.0275 
  4 p1/2 (2)   	  0.0147 
  4 p3/2 (4)   	  0.0140 


// ----------------------------------------------

  Rb   Z =  37
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 15.1997 
  2 s1/2 (2)   	  2.0651 
  2 p1/2 (2)   	  1.8639 
  2 p3/2 (4)   	  1.8044 
  3 s1/2 (2)   	  0.3221 
  3 p1/2 (2)   	  0.2474 
  3 p3/2 (4)   	  0.2385 
  3 d3/2 (4)   	  0.1118 
  3 d5/2 (6)   	  0.1103 
  4 s1/2 (2)   	  0.0293 
  4 p1/2 (2)   	  0.0148 
  4 p3/2 (4)   	  0.0140 
  5 s1/2 (1)   	  0.0040 


// ----------------------------------------------

  Sr   Z =  38
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 16.1046 
  2 s1/2 (2)   	  2.2163 
  2 p1/2 (2)   	  2.0068 
  2 p3/2 (4)   	  1.9396 
  3 s1/2 (2)   	  0.3575 
  3 p1/2 (2)   	  0.2798 
  3 p3/2 (4)   	  0.2691 
  3 d3/2 (4)   	  0.1350 
  3 d5/2 (6)   	  0.1331 
  4 s1/2 (2)   	  0.0377 
  4 p1/2 (2)   	  0.0207 
  4 p3/2 (4)   	  0.0195 
  5 s1/2 (2)   	  0.0050 


// ----------------------------------------------

  Y    Z =  39
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 17.0384 
  2 s1/2 (2)   	  2.3725 
  2 p1/2 (2)   	  2.1555 
  2 p3/2 (4)   	  2.0800 
  3 s1/2 (2)   	  0.3936 
  3 p1/2 (2)   	  0.3124 
  3 p3/2 (4)   	  0.3003 
  3 d3/2 (4)   	  0.1596 
  3 d5/2 (6)   	  0.1574 
  4 s1/2 (2)   	  0.0454 
  4 p1/2 (2)   	  0.0251 
  4 p3/2 (4)   	  0.0228 
  5 s1/2 (2)   	  0.0060 
  4 d3/2 (1)   	  0.0024 


// ----------------------------------------------

  Zr   Z =  40
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 17.9976 
  2 s1/2 (2)   	  2.5316 
  2 p1/2 (2)   	  2.3067 
  2 p3/2 (4)   	  2.2223 
  3 s1/2 (2)   	  0.4303 
  3 p1/2 (2)   	  0.3442 
  3 p3/2 (4)   	  0.3305 
  3 d3/2 (4)   	  0.1824 
  3 d5/2 (6)   	  0.1800 
  4 s1/2 (2)   	  0.0513 
  4 p1/2 (2)   	  0.0293 
  4 p3/2 (4)   	  0.0257 
  5 s1/2 (2)   	  0.0070 
  4 d3/2 (2)   	  0.0030 


// ----------------------------------------------

  Nb   Z =  41
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 18.9856 
  2 s1/2 (2)   	  2.6977 
  2 p1/2 (2)   	  2.4647 
  2 p3/2 (4)   	  2.3705 
  3 s1/2 (2)   	  0.4684 
  3 p1/2 (2)   	  0.3784 
  3 p3/2 (4)   	  0.3630 
  3 d3/2 (4)   	  0.2074 
  3 d5/2 (6)   	  0.2046 
  4 s1/2 (2)   	  0.0581 
  4 p1/2 (2)   	  0.0356 
  4 p3/2 (4)   	  0.0296 
  5 s1/2 (1)   	  0.0070 
  4 d3/2 (4)   	  0.0032 


// ----------------------------------------------

  Mo   Z =  42
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 19.9995 
  2 s1/2 (2)   	  2.8655 
  2 p1/2 (2)   	  2.6251 
  2 p3/2 (4)   	  2.5202 
  3 s1/2 (2)   	  0.5046 
  3 p1/2 (2)   	  0.4097 
  3 p3/2 (4)   	  0.3923 
  3 d3/2 (4)   	  0.2303 
  3 d5/2 (6)   	  0.2270 
  4 s1/2 (2)   	  0.0618 
  4 p1/2 (2)   	  0.0383 
  4 p3/2 (4)   	  0.0323 
  5 s1/2 (1)   	  0.0070 
  4 d3/2 (4)   	  0.0019 
  4 d5/2 (1)   	  0.0012 


// ----------------------------------------------

  Tc   Z =  43
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 21.0440 
  2 s1/2 (2)   	  3.0425 
  2 p1/2 (2)   	  2.7932 
  2 p3/2 (4)   	  2.6769 
  3 s1/2 (2)   	  0.5440 
  3 p1/2 (2)   	  0.4449 
  3 p3/2 (4)   	  0.4250 
  3 d3/2 (4)   	  0.2564 
  3 d5/2 (6)   	  0.2529 
  4 s1/2 (2)   	  0.0688 
  4 p1/2 (2)   	  0.0428 
  4 p3/2 (4)   	  0.0369 
  5 s1/2 (2)   	  0.0070 
  4 d3/2 (4)   	  0.0020 
  4 d5/2 (1)   	  0.0012 


// ----------------------------------------------

  Ru   Z =  44
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 22.1172 
  2 s1/2 (2)   	  3.2240 
  2 p1/2 (2)   	  2.9669 
  2 p3/2 (4)   	  2.8379 
  3 s1/2 (2)   	  0.5850 
  3 p1/2 (2)   	  0.4828 
  3 p3/2 (4)   	  0.4606 
  3 d3/2 (4)   	  0.2836 
  3 d5/2 (6)   	  0.2794 
  4 s1/2 (2)   	  0.0749 
  4 p1/2 (2)   	  0.0470 
  4 p3/2 (4)   	  0.0412 
  5 s1/2 (1)   	  0.0070 
  4 d3/2 (4)   	  0.0024 
  4 d5/2 (3)   	  0.0018 


// ----------------------------------------------

  Rh   Z =  45
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 23.2199 
  2 s1/2 (2)   	  3.4119 
  2 p1/2 (2)   	  3.1461 
  2 p3/2 (4)   	  3.0038 
  3 s1/2 (2)   	  0.6271 
  3 p1/2 (2)   	  0.5210 
  3 p3/2 (4)   	  0.4962 
  3 d3/2 (4)   	  0.3117 
  3 d5/2 (6)   	  0.3070 
  4 s1/2 (2)   	  0.0810 
  4 p1/2 (2)   	  0.0519 
  4 p3/2 (4)   	  0.0463 
  5 s1/2 (1)   	  0.0070 
  4 d3/2 (4)   	  0.0028 
  4 d5/2 (4)   	  0.0022 


// ----------------------------------------------

  Pd   Z =  46
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 24.3503 
  2 s1/2 (2)   	  3.6043 
  2 p1/2 (2)   	  3.3303 
  2 p3/2 (4)   	  3.1733 
  3 s1/2 (2)   	  0.6699 
  3 p1/2 (2)   	  0.5591 
  3 p3/2 (4)   	  0.5315 
  3 d3/2 (4)   	  0.3400 
  3 d5/2 (6)   	  0.3347 
  4 s1/2 (2)   	  0.0864 
  4 p1/2 (2)   	  0.0544 
  4 p3/2 (4)   	  0.0500 
  4 d3/2 (4)   	  0.0017 
  4 d5/2 (6)   	  0.0013 


// ----------------------------------------------

  Ag   Z =  47
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 25.5140 
  2 s1/2 (2)   	  3.8058 
  2 p1/2 (2)   	  3.5237 
  2 p3/2 (4)   	  3.3511 
  3 s1/2 (2)   	  0.7175 
  3 p1/2 (2)   	  0.6024 
  3 p3/2 (4)   	  0.5714 
  3 d3/2 (4)   	  0.3728 
  3 d5/2 (6)   	  0.3667 
  4 s1/2 (2)   	  0.0952 
  4 p1/2 (2)   	  0.0626 
  4 p3/2 (4)   	  0.0559 
  5 s1/2 (1)   	  0.0080 
  4 d3/2 (4)   	  0.0036 
  4 d5/2 (6)   	  0.0031 


// ----------------------------------------------

  Cd   Z =  48
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 26.7112 
  2 s1/2 (2)   	  4.0180 
  2 p1/2 (2)   	  3.7270 
  2 p3/2 (4)   	  3.5375 
  3 s1/2 (2)   	  0.7702 
  3 p1/2 (2)   	  0.6507 
  3 p3/2 (4)   	  0.6165 
  3 d3/2 (4)   	  0.4105 
  3 d5/2 (6)   	  0.4037 
  4 s1/2 (2)   	  0.1076 
  4 p1/2 (2)   	  0.0708 
  4 p3/2 (4)   	  0.0650 
  4 d3/2 (4)   	  0.0097 
  4 d5/2 (6)   	  0.0090 
  5 s1/2 (2)   	  0.0090 


// ----------------------------------------------

  In   Z =  49
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 27.9399 
  2 s1/2 (2)   	  4.2375 
  2 p1/2 (2)   	  3.9380 
  2 p3/2 (4)   	  3.7301 
  3 s1/2 (2)   	  0.8256 
  3 p1/2 (2)   	  0.7022 
  3 p3/2 (4)   	  0.6643 
  3 d3/2 (4)   	  0.4508 
  3 d5/2 (6)   	  0.4431 
  4 s1/2 (2)   	  0.1219 
  4 p1/2 (2)   	  0.0819 
  4 p3/2 (4)   	  0.0751 
  4 d3/2 (4)   	  0.0168 
  4 d5/2 (6)   	  0.0158 
  5 p1/2 (1)   	  0.0008 
  5 s1/2 (2)   	  0.0001 


// ----------------------------------------------

  Sn   Z =  50
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 29.2001 
  2 s1/2 (2)   	  4.4647 
  2 p1/2 (2)   	  4.1561 
  2 p3/2 (4)   	  3.9288 
  3 s1/2 (2)   	  0.8838 
  3 p1/2 (2)   	  0.7564 
  3 p3/2 (4)   	  0.7144 
  3 d3/2 (4)   	  0.4933 
  3 d5/2 (6)   	  0.4848 
  4 s1/2 (2)   	  0.1365 
  4 p1/2 (2)   	  0.0939 
  4 p3/2 (4)   	  0.0860 
  4 d3/2 (4)   	  0.0246 
  4 d5/2 (6)   	  0.0234 
  5 p1/2 (2)   	  0.0011 
  5 s1/2 (2)   	  0.0009 


// ----------------------------------------------

  Sb   Z =  51
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 30.4912 
  2 s1/2 (2)   	  4.6983 
  2 p1/2 (2)   	  4.3804 
  2 p3/2 (4)   	  4.1323 
  3 s1/2 (2)   	  0.9437 
  3 p1/2 (2)   	  0.8119 
  3 p3/2 (4)   	  0.7656 
  3 d3/2 (4)   	  0.5369 
  3 d5/2 (6)   	  0.5275 
  4 s1/2 (2)   	  0.1520 
  4 p1/2 (2)   	  0.1043 
  4 p3/2 (4)   	  0.0954 
  4 d3/2 (4)   	  0.0322 
  4 d5/2 (6)   	  0.0308 
  5 s1/2 (2)   	  0.0067 
  5 p1/2 (2)   	  0.0022 
  5 p3/2 (1)   	  0.0020 


// ----------------------------------------------

  Te   Z =  52
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 31.8138 
  2 s1/2 (2)   	  4.9392 
  2 p1/2 (2)   	  4.6120 
  2 p3/2 (4)   	  4.3414 
  3 s1/2 (2)   	  1.0060 
  3 p1/2 (2)   	  0.8697 
  3 p3/2 (4)   	  0.8187 
  3 d3/2 (4)   	  0.5825 
  3 d5/2 (6)   	  0.5721 
  4 s1/2 (2)   	  0.1683 
  4 p1/2 (2)   	  0.1168 
  4 p3/2 (4)   	  0.0969 
  4 d3/2 (4)   	  0.0408 
  4 d5/2 (6)   	  0.0392 
  5 s1/2 (2)   	  0.0116 
  5 p1/2 (2)   	  0.0026 
  5 p3/2 (2)   	  0.0020 


// ----------------------------------------------

  I    Z =  53
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 33.1694 
  2 s1/2 (2)   	  5.1881 
  2 p1/2 (2)   	  4.8521 
  2 p3/2 (4)   	  4.5571 
  3 s1/2 (2)   	  1.0721 
  3 p1/2 (2)   	  0.9305 
  3 p3/2 (4)   	  0.8746 
  3 d3/2 (4)   	  0.6313 
  3 d5/2 (6)   	  0.6194 
  4 s1/2 (2)   	  0.1864 
  4 p1/2 (2)   	  0.1301 
  4 p3/2 (4)   	  0.1190 
  4 d3/2 (4)   	  0.0507 
  4 d5/2 (6)   	  0.0489 
  5 s1/2 (2)   	  0.0136 
  5 p1/2 (2)   	  0.0038 
  5 p3/2 (3)   	  0.0029 


// ----------------------------------------------

  Xe   Z =  54
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 34.5644 
  2 s1/2 (2)   	  5.4528 
  2 p1/2 (2)   	  5.1037 
  2 p3/2 (4)   	  4.7822 
  3 s1/2 (2)   	  1.1487 
  3 p1/2 (2)   	  0.9990 
  3 p3/2 (4)   	  0.9370 
  3 d3/2 (4)   	  0.6906 
  3 d5/2 (6)   	  0.6747 
  4 s1/2 (2)   	  0.2177 
  4 p1/2 (2)   	  0.1639 
  4 p3/2 (4)   	  0.1565 
  4 d3/2 (4)   	  0.0695 
  4 d5/2 (6)   	  0.0676 
  5 s1/2 (2)   	  0.0234 
  5 p1/2 (2)   	  0.0134 
  5 p3/2 (4)   	  0.0121 


// ----------------------------------------------

  Cs   Z =  55
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 35.9846 
  2 s1/2 (2)   	  5.7143 
  2 p1/2 (2)   	  5.3594 
  2 p3/2 (4)   	  5.0119 
  3 s1/2 (2)   	  1.2171 
  3 p1/2 (2)   	  1.0650 
  3 p3/2 (4)   	  0.9976 
  3 d3/2 (4)   	  0.7395 
  3 d5/2 (6)   	  0.7255 
  4 s1/2 (2)   	  0.2308 
  4 p1/2 (2)   	  0.1723 
  4 p3/2 (4)   	  0.1616 
  4 d3/2 (4)   	  0.0788 
  4 d5/2 (6)   	  0.0765 
  5 s1/2 (2)   	  0.0227 
  5 p1/2 (2)   	  0.0131 
  5 p3/2 (4)   	  0.0114 
  6 s1/2 (1)   	  0.0040 


// ----------------------------------------------

  Ba   Z =  56
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 37.4406 
  2 s1/2 (2)   	  5.9888 
  2 p1/2 (2)   	  5.6236 
  2 p3/2 (4)   	  5.2470 
  3 s1/2 (2)   	  1.2928 
  3 p1/2 (2)   	  1.1367 
  3 p3/2 (4)   	  1.0622 
  3 d3/2 (4)   	  0.7961 
  3 d5/2 (6)   	  0.7807 
  4 s1/2 (2)   	  0.2530 
  4 p1/2 (2)   	  0.1918 
  4 p3/2 (4)   	  0.1797 
  4 d3/2 (4)   	  0.0925 
  4 d5/2 (6)   	  0.0899 
  5 s1/2 (2)   	  0.0291 
  5 p1/2 (2)   	  0.0166 
  5 p3/2 (4)   	  0.0146 
  6 s1/2 (2)   	  0.0050 


// ----------------------------------------------

  La   Z =  57
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 38.9246 
  2 s1/2 (2)   	  6.2663 
  2 p1/2 (2)   	  5.8906 
  2 p3/2 (4)   	  5.4827 
  3 s1/2 (2)   	  1.3613 
  3 p1/2 (2)   	  1.2044 
  3 p3/2 (4)   	  1.1234 
  3 d3/2 (4)   	  0.8485 
  3 d5/2 (6)   	  0.8317 
  4 s1/2 (2)   	  0.2704 
  4 p1/2 (2)   	  0.2058 
  4 p3/2 (4)   	  0.1914 
  4 d3/2 (4)   	  0.1007 
  4 d5/2 (6)   	  0.0977 
  5 s1/2 (2)   	  0.0323 
  5 p1/2 (2)   	  0.0166 
  5 p3/2 (4)   	  0.0133 
  5 d3/2 (1)   	  0.0060 
  6 s1/2 (2)   	  0.0050 


// ----------------------------------------------

  Ce   Z =  58
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 40.4430 
  2 s1/2 (2)   	  6.5488 
  2 p1/2 (2)   	  6.1642 
  2 p3/2 (4)   	  5.7234 
  3 s1/2 (2)   	  1.4346 
  3 p1/2 (2)   	  1.2728 
  3 p3/2 (4)   	  1.1854 
  3 d3/2 (4)   	  0.9013 
  3 d5/2 (6)   	  0.8833 
  4 s1/2 (2)   	  0.2896 
  4 p1/2 (2)   	  0.2233 
  4 p3/2 (4)   	  0.2072 
  4 d3/2 (4)   	  0.1136 
  4 d5/2 (6)   	  0.1076 
  5 s1/2 (2)   	  0.0378 
  5 p1/2 (2)   	  0.0218 
  5 p3/2 (4)   	  0.0188 
  5 d3/2 (1)   	  0.0060 
  6 s1/2 (2)   	  0.0050 
  4 f5/2 (1)   	  0.0001 


// ----------------------------------------------

  Pr   Z =  59
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 41.9906 
  2 s1/2 (2)   	  6.8348 
  2 p1/2 (2)   	  6.4404 
  2 p3/2 (4)   	  5.9643 
  3 s1/2 (2)   	  1.5110 
  3 p1/2 (2)   	  1.3374 
  3 p3/2 (4)   	  1.2422 
  3 d3/2 (4)   	  0.9511 
  3 d5/2 (6)   	  0.9310 
  4 s1/2 (2)   	  0.3045 
  4 p1/2 (2)   	  0.2363 
  4 p3/2 (4)   	  0.2176 
  4 d3/2 (4)   	  0.1179 
  4 d5/2 (6)   	  0.1101 
  5 s1/2 (2)   	  0.0374 
  5 p1/2 (2)   	  0.0246 
  5 p3/2 (4)   	  0.0212 
  5 d3/2 (1)   	  0.0060 
  6 s1/2 (2)   	  0.0050 
  4 f5/2 (2)   	  0.0020 


// ----------------------------------------------

  Nd   Z =  60
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 43.5689 
  2 s1/2 (2)   	  7.1260 
  2 p1/2 (2)   	  6.7215 
  2 p3/2 (4)   	  6.2079 
  3 s1/2 (2)   	  1.5753 
  3 p1/2 (2)   	  1.4028 
  3 p3/2 (4)   	  1.2974 
  3 d3/2 (4)   	  0.9999 
  3 d5/2 (6)   	  0.9777 
  4 s1/2 (2)   	  0.3152 
  4 p1/2 (2)   	  0.2433 
  4 p3/2 (4)   	  0.2246 
  4 d3/2 (4)   	  0.1234 
  4 d5/2 (6)   	  0.1135 
  5 s1/2 (2)   	  0.0375 
  5 p1/2 (2)   	  0.0236 
  5 p3/2 (4)   	  0.0198 
  6 s1/2 (2)   	  0.0050 
  4 f5/2 (4)   	  0.0015 


// ----------------------------------------------

  Pm   Z =  61
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 45.1840 
  2 s1/2 (2)   	  7.4279 
  2 p1/2 (2)   	  7.0128 
  2 p3/2 (4)   	  6.4593 
  3 s1/2 (2)   	  1.6486 
  3 p1/2 (2)   	  1.4714 
  3 p3/2 (4)   	  1.3569 
  3 d3/2 (4)   	  1.0515 
  3 d5/2 (6)   	  1.0269 
  4 s1/2 (2)   	  0.3314 
  4 p1/2 (2)   	  0.2547 
  4 p3/2 (4)   	  0.2362 
  4 d3/2 (4)   	  0.1276 
  4 d5/2 (6)   	  0.1156 
  5 s1/2 (2)   	  0.0360 
  5 p1/2 (2)   	  0.0245 
  5 p3/2 (4)   	  0.0201 
  6 s1/2 (2)   	  0.0050 
  4 f5/2 (5)   	  0.0035 


// ----------------------------------------------

  Sm   Z =  62
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 46.8342 
  2 s1/2 (2)   	  7.7368 
  2 p1/2 (2)   	  7.3118 
  2 p3/2 (4)   	  6.7162 
  3 s1/2 (2)   	  1.7228 
  3 p1/2 (2)   	  1.5407 
  3 p3/2 (4)   	  1.4198 
  3 d3/2 (4)   	  1.1060 
  3 d5/2 (6)   	  1.0802 
  4 s1/2 (2)   	  0.3457 
  4 p1/2 (2)   	  0.2656 
  4 p3/2 (4)   	  0.2474 
  4 d3/2 (4)   	  0.1375 
  4 d5/2 (6)   	  0.1233 
  5 s1/2 (2)   	  0.0374 
  5 p1/2 (2)   	  0.0236 
  5 p3/2 (4)   	  0.0189 
  4 f5/2 (6)   	  0.0055 
  6 s1/2 (2)   	  0.0050 


// ----------------------------------------------

  Eu   Z =  63
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 48.5190 
  2 s1/2 (2)   	  8.0520 
  2 p1/2 (2)   	  7.6171 
  2 p3/2 (4)   	  6.9769 
  3 s1/2 (2)   	  1.8000 
  3 p1/2 (2)   	  1.6139 
  3 p3/2 (4)   	  1.4806 
  3 d3/2 (4)   	  1.1606 
  3 d5/2 (6)   	  1.1309 
  4 s1/2 (2)   	  0.3602 
  4 p1/2 (2)   	  0.2839 
  4 p3/2 (4)   	  0.2566 
  4 d3/2 (4)   	  0.1414 
  4 d5/2 (6)   	  0.1277 
  5 s1/2 (2)   	  0.0318 
  5 p1/2 (2)   	  0.0252 
  5 p3/2 (4)   	  0.0204 
  6 s1/2 (2)   	  0.0060 
  4 f5/2 (6)   	  0.0015 
  4 f7/2 (1)   	  0.0012 


// ----------------------------------------------

  Gd   Z =  64
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 50.2391 
  2 s1/2 (2)   	  8.3756 
  2 p1/2 (2)   	  7.9303 
  2 p3/2 (4)   	  7.2428 
  3 s1/2 (2)   	  1.8808 
  3 p1/2 (2)   	  1.6883 
  3 p3/2 (4)   	  1.5440 
  3 d3/2 (4)   	  1.2172 
  3 d5/2 (6)   	  1.1852 
  4 s1/2 (2)   	  0.3758 
  4 p1/2 (2)   	  0.2885 
  4 p3/2 (4)   	  0.2709 
  4 d3/2 (4)   	  0.1495 
  4 d5/2 (6)   	  0.1345 
  5 s1/2 (2)   	  0.0361 
  5 p1/2 (2)   	  0.0243 
  5 p3/2 (4)   	  0.0183 
  5 d3/2 (1)   	  0.0060 
  6 s1/2 (2)   	  0.0060 
  4 f5/2 (6)   	  0.0020 
  4 f7/2 (1)   	  0.0013 


// ----------------------------------------------

  Tb   Z =  65
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 51.9957 
  2 s1/2 (2)   	  8.7080 
  2 p1/2 (2)   	  8.2516 
  2 p3/2 (4)   	  7.5140 
  3 s1/2 (2)   	  1.9675 
  3 p1/2 (2)   	  1.7677 
  3 p3/2 (4)   	  1.6113 
  3 d3/2 (4)   	  1.2750 
  3 d5/2 (6)   	  1.2412 
  4 s1/2 (2)   	  0.3979 
  4 p1/2 (2)   	  0.3102 
  4 p3/2 (4)   	  0.2850 
  4 d3/2 (4)   	  0.1545 
  4 d5/2 (6)   	  0.1420 
  5 s1/2 (2)   	  0.0390 
  5 p1/2 (2)   	  0.0263 
  5 p3/2 (4)   	  0.0213 
  5 d3/2 (1)   	  0.0060 
  6 s1/2 (2)   	  0.0060 
  4 f5/2 (6)   	  0.0040 
  4 f7/2 (2)   	  0.0016 


// ----------------------------------------------

  Dy   Z =  66
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 53.7885 
  2 s1/2 (2)   	  9.0458 
  2 p1/2 (2)   	  8.5806 
  2 p3/2 (4)   	  7.7901 
  3 s1/2 (2)   	  2.0468 
  3 p1/2 (2)   	  1.8418 
  3 p3/2 (4)   	  1.6756 
  3 d3/2 (4)   	  1.3325 
  3 d5/2 (6)   	  1.2949 
  4 s1/2 (2)   	  0.4163 
  4 p1/2 (2)   	  0.3318 
  4 p3/2 (4)   	  0.2929 
  4 d3/2 (4)   	  0.1614 
  4 d5/2 (6)   	  0.1494 
  5 s1/2 (2)   	  0.0629 
  5 p1/2 (2)   	  0.0282 
  5 p3/2 (4)   	  0.0229 
  6 s1/2 (2)   	  0.0060 
  4 f5/2 (6)   	  0.0055 
  4 f7/2 (4)   	  0.0033 


// ----------------------------------------------

  Ho   Z =  67
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 55.6177 
  2 s1/2 (2)   	  9.3942 
  2 p1/2 (2)   	  8.9178 
  2 p3/2 (4)   	  8.0711 
  3 s1/2 (2)   	  2.1283 
  3 p1/2 (2)   	  1.9228 
  3 p3/2 (4)   	  1.7412 
  3 d3/2 (4)   	  1.3915 
  3 d5/2 (6)   	  1.3514 
  4 s1/2 (2)   	  0.4357 
  4 p1/2 (2)   	  0.3435 
  4 p3/2 (4)   	  0.3066 
  4 d3/2 (4)   	  0.1678 
  4 d5/2 (6)   	  0.1568 
  5 s1/2 (2)   	  0.0512 
  5 p1/2 (2)   	  0.0249 
  5 p3/2 (4)   	  0.0195 
  6 s1/2 (2)   	  0.0060 
  4 f5/2 (6)   	  0.0048 
  4 f7/2 (5)   	  0.0028 


// ----------------------------------------------

  Er   Z =  68
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 57.4855 
  2 s1/2 (2)   	  9.7513 
  2 p1/2 (2)   	  9.2643 
  2 p3/2 (4)   	  8.3579 
  3 s1/2 (2)   	  2.2167 
  3 p1/2 (2)   	  2.0058 
  3 p3/2 (4)   	  1.8118 
  3 d3/2 (4)   	  1.4533 
  3 d5/2 (6)   	  1.4093 
  4 s1/2 (2)   	  0.4491 
  4 p1/2 (2)   	  0.3662 
  4 p3/2 (4)   	  0.3200 
  4 d3/2 (4)   	  0.1767 
  4 d5/2 (6)   	  0.1676 
  5 s1/2 (2)   	  0.0598 
  5 p1/2 (2)   	  0.0279 
  5 p3/2 (4)   	  0.0223 
  6 s1/2 (2)   	  0.0060 
  4 f5/2 (6)   	  0.0053 
  4 f7/2 (6)   	  0.0036 


// ----------------------------------------------

  Tm   Z =  69
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 59.3896 
  2 s1/2 (2)   	 10.1157 
  2 p1/2 (2)   	  9.6169 
  2 p3/2 (4)   	  8.6480 
  3 s1/2 (2)   	  2.3068 
  3 p1/2 (2)   	  2.0898 
  3 p3/2 (4)   	  1.8845 
  3 d3/2 (4)   	  1.5146 
  3 d5/2 (6)   	  1.4677 
  4 s1/2 (2)   	  0.4717 
  4 p1/2 (2)   	  0.3859 
  4 p3/2 (4)   	  0.3366 
  4 d3/2 (4)   	  0.1855 
  4 d5/2 (6)   	  0.1757 
  5 s1/2 (2)   	  0.0532 
  5 p1/2 (2)   	  0.0362 
  5 p3/2 (4)   	  0.0304 
  4 f5/2 (6)   	  0.0062 
  6 s1/2 (2)   	  0.0060 
  4 f7/2 (7)   	  0.0047 


// ----------------------------------------------

  Yb   Z =  70
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 61.3323 
  2 s1/2 (2)   	 10.4864 
  2 p1/2 (2)   	  9.9782 
  2 p3/2 (4)   	  8.9436 
  3 s1/2 (2)   	  2.3981 
  3 p1/2 (2)   	  2.1730 
  3 p3/2 (4)   	  1.9498 
  3 d3/2 (4)   	  1.5763 
  3 d5/2 (6)   	  1.5278 
  4 s1/2 (2)   	  0.4872 
  4 p1/2 (2)   	  0.3967 
  4 p3/2 (4)   	  0.3435 
  4 d3/2 (4)   	  0.1981 
  4 d5/2 (6)   	  0.1849 
  5 s1/2 (2)   	  0.0541 
  5 p1/2 (2)   	  0.0274 
  5 p3/2 (4)   	  0.0214 
  4 f5/2 (6)   	  0.0070 
  6 s1/2 (2)   	  0.0060 
  4 f7/2 (8)   	  0.0058 


// ----------------------------------------------

  Lu   Z =  71
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 63.3138 
  2 s1/2 (2)   	 10.8704 
  2 p1/2 (2)   	 10.3486 
  2 p3/2 (4)   	  9.2441 
  3 s1/2 (2)   	  2.4912 
  3 p1/2 (2)   	  2.2635 
  3 p3/2 (4)   	  2.0236 
  3 d3/2 (4)   	  1.6394 
  3 d5/2 (6)   	  1.5885 
  4 s1/2 (2)   	  0.5062 
  4 p1/2 (2)   	  0.4101 
  4 p3/2 (4)   	  0.3593 
  4 d3/2 (4)   	  0.2048 
  4 d5/2 (6)   	  0.1950 
  5 s1/2 (2)   	  0.0568 
  5 p1/2 (2)   	  0.0330 
  5 p3/2 (4)   	  0.0255 
  4 f5/2 (6)   	  0.0078 
  6 s1/2 (2)   	  0.0070 
  4 f7/2 (8)   	  0.0062 
  5 d3/2 (1)   	  0.0046 


// ----------------------------------------------

  Hf   Z =  72
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 65.3508 
  2 s1/2 (2)   	 11.2707 
  2 p1/2 (2)   	 10.7394 
  2 p3/2 (4)   	  9.5607 
  3 s1/2 (2)   	  2.6009 
  3 p1/2 (2)   	  2.3654 
  3 p3/2 (4)   	  2.1076 
  3 d3/2 (4)   	  1.7164 
  3 d5/2 (6)   	  1.6617 
  4 s1/2 (2)   	  0.5381 
  4 p1/2 (2)   	  0.4370 
  4 p3/2 (4)   	  0.3804 
  4 d3/2 (4)   	  0.2238 
  4 d5/2 (6)   	  0.2137 
  5 s1/2 (2)   	  0.0649 
  5 p1/2 (2)   	  0.0382 
  5 p3/2 (4)   	  0.0290 
  4 f5/2 (6)   	  0.0182 
  4 f7/2 (8)   	  0.0163 
  6 s1/2 (2)   	  0.0070 
  5 d3/2 (2)   	  0.0066 


// ----------------------------------------------

  Ta   Z =  73
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 67.4164 
  2 s1/2 (2)   	 11.6815 
  2 p1/2 (2)   	 11.1361 
  2 p3/2 (4)   	  9.8811 
  3 s1/2 (2)   	  2.7080 
  3 p1/2 (2)   	  2.4687 
  3 p3/2 (4)   	  2.1940 
  3 d3/2 (4)   	  1.7932 
  3 d5/2 (6)   	  1.7351 
  4 s1/2 (2)   	  0.5655 
  4 p1/2 (2)   	  0.4648 
  4 p3/2 (4)   	  0.4045 
  4 d3/2 (4)   	  0.2413 
  4 d5/2 (6)   	  0.2293 
  5 s1/2 (2)   	  0.0711 
  5 p1/2 (2)   	  0.0437 
  5 p3/2 (4)   	  0.0347 
  4 f5/2 (6)   	  0.0275 
  4 f7/2 (8)   	  0.0256 
  6 s1/2 (2)   	  0.0080 
  5 d3/2 (3)   	  0.0057 


// ----------------------------------------------

  W    Z =  74
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 69.5250 
  2 s1/2 (2)   	 12.0998 
  2 p1/2 (2)   	 11.5440 
  2 p3/2 (4)   	 10.2068 
  3 s1/2 (2)   	  2.8196 
  3 p1/2 (2)   	  2.5749 
  3 p3/2 (4)   	  2.2810 
  3 d3/2 (4)   	  1.8716 
  3 d5/2 (6)   	  1.8092 
  4 s1/2 (2)   	  0.5950 
  4 p1/2 (2)   	  0.4916 
  4 p3/2 (4)   	  0.4253 
  4 d3/2 (4)   	  0.2588 
  4 d5/2 (6)   	  0.2454 
  5 s1/2 (2)   	  0.0771 
  5 p1/2 (2)   	  0.0467 
  4 f5/2 (6)   	  0.0374 
  5 p3/2 (4)   	  0.0365 
  4 f7/2 (8)   	  0.0351 
  6 s1/2 (2)   	  0.0080 
  5 d3/2 (4)   	  0.0061 


// ----------------------------------------------

  Re   Z =  75
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 71.6764 
  2 s1/2 (2)   	 12.5267 
  2 p1/2 (2)   	 11.9587 
  2 p3/2 (4)   	 10.5353 
  3 s1/2 (2)   	  2.9317 
  3 p1/2 (2)   	  2.6816 
  3 p3/2 (4)   	  2.3673 
  3 d3/2 (4)   	  1.9489 
  3 d5/2 (6)   	  1.8829 
  4 s1/2 (2)   	  0.6250 
  4 p1/2 (2)   	  0.5179 
  4 p3/2 (4)   	  0.4444 
  4 d3/2 (4)   	  0.2737 
  4 d5/2 (6)   	  0.2602 
  5 s1/2 (2)   	  0.0828 
  5 p1/2 (2)   	  0.0484 
  4 f5/2 (6)   	  0.0481 
  4 f7/2 (8)   	  0.0457 
  5 p3/2 (4)   	  0.0368 
  6 s1/2 (2)   	  0.0080 
  5 d3/2 (4)   	  0.0038 
  5 d5/2 (1)   	  0.0025 


// ----------------------------------------------

  Os   Z =  76
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 73.8708 
  2 s1/2 (2)   	 12.9680 
  2 p1/2 (2)   	 12.3850 
  2 p3/2 (4)   	 10.8702 
  3 s1/2 (2)   	  3.0485 
  3 p1/2 (2)   	  2.7922 
  3 p3/2 (4)   	  2.4572 
  3 d3/2 (4)   	  2.0308 
  3 d5/2 (6)   	  1.9601 
  4 s1/2 (2)   	  0.6543 
  4 p1/2 (2)   	  0.5465 
  4 p3/2 (4)   	  0.4682 
  4 d3/2 (4)   	  0.2894 
  4 d5/2 (6)   	  0.2728 
  5 s1/2 (2)   	  0.0837 
  5 p1/2 (2)   	  0.0580 
  4 f5/2 (6)   	  0.0538 
  4 f7/2 (8)   	  0.0510 
  5 p3/2 (4)   	  0.0454 
  6 s1/2 (2)   	  0.0080 
  5 d5/2 (2)   	  0.0009 
  5 d3/2 (4)   	  0.0004 


// ----------------------------------------------

  Ir   Z =  77
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 76.1110 
  2 s1/2 (2)   	 13.4185 
  2 p1/2 (2)   	 12.8241 
  2 p3/2 (4)   	 11.2152 
  3 s1/2 (2)   	  3.1737 
  3 p1/2 (2)   	  2.9087 
  3 p3/2 (4)   	  2.5507 
  3 d3/2 (4)   	  2.1161 
  3 d5/2 (6)   	  2.0404 
  4 s1/2 (2)   	  0.6901 
  4 p1/2 (2)   	  0.5771 
  4 p3/2 (4)   	  0.4943 
  4 d3/2 (4)   	  0.3114 
  4 d5/2 (6)   	  0.2949 
  5 s1/2 (2)   	  0.0952 
  4 f5/2 (6)   	  0.0638 
  5 p1/2 (2)   	  0.0630 
  4 f7/2 (8)   	  0.0608 
  5 p3/2 (4)   	  0.0496 
  6 s1/2 (2)   	  0.0090 
  5 d3/2 (4)   	  0.0042 
  5 d5/2 (3)   	  0.0032 


// ----------------------------------------------

  Pt   Z =  78
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 78.3948 
  2 s1/2 (2)   	 13.8801 
  2 p1/2 (2)   	 13.2726 
  2 p3/2 (4)   	 11.5637 
  3 s1/2 (2)   	  3.2972 
  3 p1/2 (2)   	  3.0267 
  3 p3/2 (4)   	  2.6457 
  3 d3/2 (4)   	  2.2017 
  3 d5/2 (6)   	  2.1214 
  4 s1/2 (2)   	  0.7228 
  4 p1/2 (2)   	  0.6084 
  4 p3/2 (4)   	  0.5190 
  4 d3/2 (4)   	  0.3307 
  4 d5/2 (6)   	  0.3134 
  5 s1/2 (2)   	  0.1017 
  4 f5/2 (6)   	  0.0743 
  4 f7/2 (8)   	  0.0709 
  5 p1/2 (2)   	  0.0653 
  5 p3/2 (4)   	  0.0516 
  6 s1/2 (1)   	  0.0090 
  5 d3/2 (4)   	  0.0028 
  5 d5/2 (5)   	  0.0014 


// ----------------------------------------------

  Au   Z =  79
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 80.7249 
  2 s1/2 (2)   	 14.3528 
  2 p1/2 (2)   	 13.7336 
  2 p3/2 (4)   	 11.9187 
  3 s1/2 (2)   	  3.4249 
  3 p1/2 (2)   	  3.1478 
  3 p3/2 (4)   	  2.7430 
  3 d3/2 (4)   	  2.2911 
  3 d5/2 (6)   	  2.2057 
  4 s1/2 (2)   	  0.7588 
  4 p1/2 (2)   	  0.6437 
  4 p3/2 (4)   	  0.5454 
  4 d3/2 (4)   	  0.3520 
  4 d5/2 (6)   	  0.3339 
  5 s1/2 (2)   	  0.1078 
  4 f5/2 (6)   	  0.0873 
  4 f7/2 (8)   	  0.0837 
  5 p1/2 (2)   	  0.0717 
  5 p3/2 (4)   	  0.0569 
  6 s1/2 (1)   	  0.0090 
  5 d3/2 (4)   	  0.0033 
  5 d5/2 (6)   	  0.0018 


// ----------------------------------------------

  Hg   Z =  80
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 83.1023 
  2 s1/2 (2)   	 14.8393 
  2 p1/2 (2)   	 14.2087 
  2 p3/2 (4)   	 12.2839 
  3 s1/2 (2)   	  3.5616 
  3 p1/2 (2)   	  3.2785 
  3 p3/2 (4)   	  2.8471 
  3 d3/2 (4)   	  2.3849 
  3 d5/2 (6)   	  2.2949 
  4 s1/2 (2)   	  0.8030 
  4 p1/2 (2)   	  0.6769 
  4 p3/2 (4)   	  0.5710 
  4 d3/2 (4)   	  0.3783 
  4 d5/2 (6)   	  0.3598 
  5 s1/2 (2)   	  0.1203 
  4 f5/2 (6)   	  0.1033 
  4 f7/2 (8)   	  0.0994 
  5 p1/2 (2)   	  0.0805 
  5 p3/2 (4)   	  0.0618 
  6 s1/2 (2)   	  0.0090 
  5 d3/2 (4)   	  0.0075 
  5 d5/2 (6)   	  0.0057 


// ----------------------------------------------

  Tl   Z =  81
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 85.5304 
  2 s1/2 (2)   	 15.3467 
  2 p1/2 (2)   	 14.6979 
  2 p3/2 (4)   	 12.6575 
  3 s1/2 (2)   	  3.7041 
  3 p1/2 (2)   	  3.4157 
  3 p3/2 (4)   	  2.9566 
  3 d3/2 (4)   	  2.4851 
  3 d5/2 (6)   	  2.3893 
  4 s1/2 (2)   	  0.8455 
  4 p1/2 (2)   	  0.7213 
  4 p3/2 (4)   	  0.6090 
  4 d3/2 (4)   	  0.4066 
  4 d5/2 (6)   	  0.3862 
  5 s1/2 (2)   	  0.1363 
  4 f5/2 (6)   	  0.1230 
  4 f7/2 (8)   	  0.1187 
  5 p1/2 (2)   	  0.0996 
  5 p3/2 (4)   	  0.0745 
  5 d3/2 (4)   	  0.0153 
  5 d5/2 (6)   	  0.0131 
  6 p1/2 (1)   	  0.0060 
  6 s1/2 (2)   	  0.0008 


// ----------------------------------------------

  Pb   Z =  82
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 88.0045 
  2 s1/2 (2)   	 15.8608 
  2 p1/2 (2)   	 15.2000 
  2 p3/2 (4)   	 13.0352 
  3 s1/2 (2)   	  3.8507 
  3 p1/2 (2)   	  3.5542 
  3 p3/2 (4)   	  3.0664 
  3 d3/2 (4)   	  2.5856 
  3 d5/2 (6)   	  2.4840 
  4 s1/2 (2)   	  0.8936 
  4 p1/2 (2)   	  0.7639 
  4 p3/2 (4)   	  0.6445 
  4 d3/2 (4)   	  0.4352 
  4 d5/2 (6)   	  0.4129 
  5 s1/2 (2)   	  0.1473 
  4 f5/2 (6)   	  0.1418 
  4 f7/2 (8)   	  0.1369 
  5 p1/2 (2)   	  0.1048 
  5 p3/2 (4)   	  0.0845 
  5 d3/2 (4)   	  0.0218 
  5 d5/2 (6)   	  0.0192 
  6 s1/2 (2)   	  0.0031 
  6 p1/2 (2)   	  0.0007 


// ----------------------------------------------

  Bi   Z =  83
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 90.5259 
  2 s1/2 (2)   	 16.3875 
  2 p1/2 (2)   	 15.7111 
  2 p3/2 (4)   	 13.4186 
  3 s1/2 (2)   	  3.9991 
  3 p1/2 (2)   	  3.6963 
  3 p3/2 (4)   	  3.1769 
  3 d3/2 (4)   	  2.6876 
  3 d5/2 (6)   	  2.5796 
  4 s1/2 (2)   	  0.9382 
  4 p1/2 (2)   	  0.8053 
  4 p3/2 (4)   	  0.6789 
  4 d3/2 (4)   	  0.4636 
  4 d5/2 (6)   	  0.4400 
  4 f5/2 (6)   	  0.1623 
  5 s1/2 (2)   	  0.1593 
  4 f7/2 (8)   	  0.1572 
  5 p1/2 (2)   	  0.1168 
  5 p3/2 (4)   	  0.0929 
  5 d3/2 (4)   	  0.0265 
  5 d5/2 (6)   	  0.0244 
  6 s1/2 (2)   	  0.0075 
  6 p1/2 (2)   	  0.0012 
  6 p3/2 (1)   	  0.0002 


// ----------------------------------------------

  Po   Z =  84
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 93.0999 
  2 s1/2 (2)   	 16.9279 
  2 p1/2 (2)   	 16.2380 
  2 p3/2 (4)   	 13.8106 
  3 s1/2 (2)   	  4.1535 
  3 p1/2 (2)   	  3.8443 
  3 p3/2 (4)   	  3.2934 
  3 d3/2 (4)   	  2.7936 
  3 d5/2 (6)   	  2.6792 
  4 s1/2 (2)   	  0.9875 
  4 p1/2 (2)   	  0.8509 
  4 p3/2 (4)   	  0.7152 
  4 d3/2 (4)   	  0.4957 
  4 d5/2 (6)   	  0.4699 
  4 f5/2 (6)   	  0.1846 
  4 f7/2 (8)   	  0.1789 
  5 s1/2 (2)   	  0.1775 
  5 p1/2 (2)   	  0.1318 
  5 p3/2 (4)   	  0.1037 
  5 d3/2 (4)   	  0.0338 
  5 d5/2 (6)   	  0.0306 
  6 s1/2 (2)   	  0.0110 
  6 p1/2 (2)   	  0.0032 
  6 p3/2 (2)   	  0.0014 


// ----------------------------------------------

  At   Z =  85
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 95.7240 
  2 s1/2 (2)   	 17.4815 
  2 p1/2 (2)   	 16.7773 
  2 p3/2 (4)   	 14.2080 
  3 s1/2 (2)   	  4.3117 
  3 p1/2 (2)   	  3.9958 
  3 p3/2 (4)   	  3.4105 
  3 d3/2 (4)   	  2.9018 
  3 d5/2 (6)   	  2.7807 
  4 s1/2 (2)   	  1.0382 
  4 p1/2 (2)   	  0.8977 
  4 p3/2 (4)   	  0.7537 
  4 d3/2 (4)   	  0.5276 
  4 d5/2 (6)   	  0.5001 
  4 f5/2 (6)   	  0.2070 
  4 f7/2 (8)   	  0.2008 
  5 s1/2 (2)   	  0.1934 
  5 p1/2 (2)   	  0.1456 
  5 p3/2 (4)   	  0.1136 
  5 d3/2 (4)   	  0.0409 
  5 d5/2 (6)   	  0.0374 
  6 s1/2 (2)   	  0.0150 
  6 p1/2 (2)   	  0.0057 
  6 p3/2 (3)   	  0.0028 


// ----------------------------------------------

  Rn   Z =  86
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	 98.3972 
  2 s1/2 (2)   	 18.0487 
  2 p1/2 (2)   	 17.3297 
  2 p3/2 (4)   	 14.6114 
  3 s1/2 (2)   	  4.4743 
  3 p1/2 (2)   	  4.1515 
  3 p3/2 (4)   	  3.5305 
  3 d3/2 (4)   	  3.0123 
  3 d5/2 (6)   	  2.8842 
  4 s1/2 (2)   	  1.0905 
  4 p1/2 (2)   	  0.9462 
  4 p3/2 (4)   	  0.7912 
  4 d3/2 (4)   	  0.5604 
  4 d5/2 (6)   	  0.5311 
  4 f5/2 (6)   	  0.2301 
  4 f7/2 (8)   	  0.2236 
  5 s1/2 (2)   	  0.2096 
  5 p1/2 (2)   	  0.1595 
  5 p3/2 (4)   	  0.1239 
  5 d3/2 (4)   	  0.0480 
  5 d5/2 (6)   	  0.0442 
  6 s1/2 (2)   	  0.0187 
  6 p1/2 (2)   	  0.0076 
  6 p3/2 (4)   	  0.0041 


// ----------------------------------------------

  Fr   Z =  87
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	101.1299 
  2 s1/2 (2)   	 18.6341 
  2 p1/2 (2)   	 17.9005 
  2 p3/2 (4)   	 15.0256 
  3 s1/2 (2)   	  4.6457 
  3 p1/2 (2)   	  4.3160 
  3 p3/2 (4)   	  3.6573 
  3 d3/2 (4)   	  3.1297 
  3 d5/2 (6)   	  2.9949 
  4 s1/2 (2)   	  1.1490 
  4 p1/2 (2)   	  1.0007 
  4 p3/2 (4)   	  0.8351 
  4 d3/2 (4)   	  0.5987 
  4 d5/2 (6)   	  0.5675 
  4 f5/2 (6)   	  0.2586 
  4 f7/2 (8)   	  0.2516 
  5 s1/2 (2)   	  0.2309 
  5 p1/2 (2)   	  0.1787 
  5 p3/2 (4)   	  0.1387 
  5 d3/2 (4)   	  0.0600 
  5 d5/2 (6)   	  0.0556 
  6 s1/2 (2)   	  0.0263 
  6 p1/2 (2)   	  0.0132 
  6 p3/2 (4)   	  0.0088 
  7 s1/2 (1)   	  0.0040 


// ----------------------------------------------

  Ra   Z =  88
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	103.9162 
  2 s1/2 (2)   	 19.2367 
  2 p1/2 (2)   	 18.4843 
  2 p3/2 (4)   	 15.4444 
  3 s1/2 (2)   	  4.8220 
  3 p1/2 (2)   	  4.4850 
  3 p3/2 (4)   	  3.7866 
  3 d3/2 (4)   	  3.2484 
  3 d5/2 (6)   	  3.1049 
  4 s1/2 (2)   	  1.2084 
  4 p1/2 (2)   	  1.0576 
  4 p3/2 (4)   	  0.8791 
  4 d3/2 (4)   	  0.6359 
  4 d5/2 (6)   	  0.6027 
  4 f5/2 (6)   	  0.2879 
  4 f7/2 (8)   	  0.2804 
  5 s1/2 (2)   	  0.2544 
  5 p1/2 (2)   	  0.2004 
  5 p3/2 (4)   	  0.1528 
  5 d3/2 (4)   	  0.0694 
  5 d5/2 (6)   	  0.0638 
  6 s1/2 (2)   	  0.0355 
  6 p1/2 (2)   	  0.0192 
  6 p3/2 (4)   	  0.0137 
  7 s1/2 (2)   	  0.0050 


// ----------------------------------------------

  Ac   Z =  89
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	106.7563 
  2 s1/2 (2)   	 19.8459 
  2 p1/2 (2)   	 19.0830 
  2 p3/2 (4)   	 15.8712 
  3 s1/2 (2)   	  5.0006 
  3 p1/2 (2)   	  4.6568 
  3 p3/2 (4)   	  3.9167 
  3 d3/2 (4)   	  3.3701 
  3 d5/2 (6)   	  3.2197 
  4 s1/2 (2)   	  1.2694 
  4 p1/2 (2)   	  1.1128 
  4 p3/2 (4)   	  0.9243 
  4 d3/2 (4)   	  0.6739 
  4 d5/2 (6)   	  0.6411 
  4 f5/2 (6)   	  0.3164 
  4 f7/2 (8)   	  0.3084 
  5 s1/2 (2)   	  0.2735 
  5 p1/2 (2)   	  0.2169 
  5 p3/2 (4)   	  0.1678 
  5 d3/2 (4)   	  0.0833 
  5 d5/2 (6)   	  0.0777 
  6 s1/2 (2)   	  0.0398 
  6 p1/2 (2)   	  0.0241 
  6 p3/2 (4)   	  0.0170 
  7 s1/2 (2)   	  0.0060 
  6 d3/2 (1)   	  0.0060 


// ----------------------------------------------

  Th   Z =  90
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	109.6491 
  2 s1/2 (2)   	 20.4721 
  2 p1/2 (2)   	 19.6932 
  2 p3/2 (4)   	 16.3003 
  3 s1/2 (2)   	  5.1823 
  3 p1/2 (2)   	  4.8304 
  3 p3/2 (4)   	  4.0461 
  3 d3/2 (4)   	  3.4908 
  3 d5/2 (6)   	  3.3320 
  4 s1/2 (2)   	  1.3295 
  4 p1/2 (2)   	  1.1682 
  4 p3/2 (4)   	  0.9672 
  4 d3/2 (4)   	  0.7137 
  4 d5/2 (6)   	  0.6766 
  4 f5/2 (6)   	  0.3444 
  4 f7/2 (8)   	  0.3350 
  5 s1/2 (2)   	  0.2902 
  5 p1/2 (2)   	  0.2320 
  5 p3/2 (4)   	  0.1808 
  5 d3/2 (4)   	  0.0941 
  5 d5/2 (6)   	  0.0873 
  6 s1/2 (2)   	  0.0414 
  6 p1/2 (2)   	  0.0258 
  6 p3/2 (4)   	  0.0173 
  7 s1/2 (2)   	  0.0060 
  6 d3/2 (2)   	  0.0060 


// ----------------------------------------------

  Pa   Z =  91
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	112.5961 
  2 s1/2 (2)   	 21.1114 
  2 p1/2 (2)   	 20.3137 
  2 p3/2 (4)   	 16.7291 
  3 s1/2 (2)   	  5.3669 
  3 p1/2 (2)   	  5.0027 
  3 p3/2 (4)   	  4.1738 
  3 d3/2 (4)   	  3.6064 
  3 d5/2 (6)   	  3.4394 
  4 s1/2 (2)   	  1.3871 
  4 p1/2 (2)   	  1.2243 
  4 p3/2 (4)   	  1.0067 
  4 d3/2 (4)   	  0.7434 
  4 d5/2 (6)   	  0.7082 
  4 f5/2 (6)   	  0.3712 
  4 f7/2 (8)   	  0.3595 
  5 s1/2 (2)   	  0.3096 
  5 p1/2 (2)   	  0.2446 
  5 p3/2 (4)   	  0.1863 
  5 d3/2 (4)   	  0.0973 
  5 d5/2 (6)   	  0.0892 
  6 s1/2 (2)   	  0.0467 
  6 p1/2 (2)   	  0.0281 
  6 p3/2 (4)   	  0.0189 
  7 s1/2 (2)   	  0.0060 
  6 d3/2 (1)   	  0.0060 
  5 f5/2 (2)   	  0.0060 


// ----------------------------------------------

  U    Z =  92
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	115.6006 
  2 s1/2 (2)   	 21.7574 
  2 p1/2 (2)   	 20.9476 
  2 p3/2 (4)   	 17.1663 
  3 s1/2 (2)   	  5.5480 
  3 p1/2 (2)   	  5.1822 
  3 p3/2 (4)   	  4.3034 
  3 d3/2 (4)   	  3.7276 
  3 d5/2 (6)   	  3.5517 
  4 s1/2 (2)   	  1.4408 
  4 p1/2 (2)   	  1.2718 
  4 p3/2 (4)   	  1.0449 
  4 d3/2 (4)   	  0.7802 
  4 d5/2 (6)   	  0.7377 
  4 f5/2 (6)   	  0.3907 
  4 f7/2 (8)   	  0.3799 
  5 s1/2 (2)   	  0.3233 
  5 p1/2 (2)   	  0.2593 
  5 p3/2 (4)   	  0.1959 
  5 d3/2 (4)   	  0.1044 
  5 d5/2 (6)   	  0.0952 
  6 s1/2 (2)   	  0.0495 
  6 p1/2 (2)   	  0.0308 
  6 p3/2 (4)   	  0.0186 
  7 s1/2 (2)   	  0.0060 
  6 d3/2 (1)   	  0.0060 
  5 f5/2 (3)   	  0.0060 


// ----------------------------------------------

  Np   Z =  93
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	118.6686 
  2 s1/2 (2)   	 22.4281 
  2 p1/2 (2)   	 21.6013 
  2 p3/2 (4)   	 17.6106 
  3 s1/2 (2)   	  5.7396 
  3 p1/2 (2)   	  5.3667 
  3 p3/2 (4)   	  4.4334 
  3 d3/2 (4)   	  3.8498 
  3 d5/2 (6)   	  3.6652 
  4 s1/2 (2)   	  1.5001 
  4 p1/2 (2)   	  1.3277 
  4 p3/2 (4)   	  1.0860 
  4 d3/2 (4)   	  0.8161 
  4 d5/2 (6)   	  0.7708 
  4 f5/2 (6)   	  0.4143 
  4 f7/2 (8)   	  0.4034 
  5 s1/2 (2)   	  0.3398 
  5 p1/2 (2)   	  0.2834 
  5 p3/2 (4)   	  0.2062 
  5 d3/2 (4)   	  0.1093 
  5 d5/2 (6)   	  0.1012 
  6 s1/2 (2)   	  0.0500 
  6 p1/2 (2)   	  0.0293 
  6 p3/2 (4)   	  0.0175 
  7 s1/2 (2)   	  0.0060 
  6 d3/2 (1)   	  0.0060 
  5 f5/2 (4)   	  0.0060 


// ----------------------------------------------

  Pu   Z =  94
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	121.7911 
  2 s1/2 (2)   	 23.0978 
  2 p1/2 (2)   	 22.2668 
  2 p3/2 (4)   	 18.0570 
  3 s1/2 (2)   	  5.9329 
  3 p1/2 (2)   	  5.5412 
  3 p3/2 (4)   	  4.5566 
  3 d3/2 (4)   	  3.9726 
  3 d5/2 (6)   	  3.7781 
  4 s1/2 (2)   	  1.5593 
  4 p1/2 (2)   	  1.3774 
  4 p3/2 (4)   	  1.1209 
  4 d3/2 (4)   	  0.8489 
  4 d5/2 (6)   	  0.8015 
  4 f5/2 (6)   	  0.4374 
  4 f7/2 (8)   	  0.4252 
  5 s1/2 (2)   	  0.3519 
  5 p1/2 (2)   	  0.2825 
  5 p3/2 (4)   	  0.2153 
  5 d3/2 (4)   	  0.1160 
  5 d5/2 (6)   	  0.1052 
  6 s1/2 (2)   	  0.0486 
  6 p1/2 (2)   	  0.0306 
  6 p3/2 (4)   	  0.0184 
  7 s1/2 (2)   	  0.0060 
  5 f5/2 (6)   	  0.0060 


// ----------------------------------------------

  Am   Z =  95
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	124.9809 
  2 s1/2 (2)   	 23.8051 
  2 p1/2 (2)   	 22.9494 
  2 p3/2 (4)   	 18.5062 
  3 s1/2 (2)   	  6.1326 
  3 p1/2 (2)   	  5.7470 
  3 p3/2 (4)   	  4.7060 
  3 d3/2 (4)   	  4.0921 
  3 d5/2 (6)   	  3.8869 
  4 s1/2 (2)   	  1.6192 
  4 p1/2 (2)   	  1.4351 
  4 p3/2 (4)   	  1.1680 
  4 d3/2 (4)   	  0.8804 
  4 d5/2 (6)   	  0.8300 
  4 f5/2 (6)   	  0.4633 
  4 f7/2 (8)   	  0.4490 
  5 s1/2 (2)   	  0.3730 
  5 p1/2 (2)   	  0.3030 
  5 p3/2 (4)   	  0.2164 
  5 d3/2 (4)   	  0.1180 
  5 d5/2 (6)   	  0.1079 
  6 s1/2 (2)   	  0.0504 
  6 p1/2 (2)   	  0.0311 
  6 p3/2 (4)   	  0.0181 
  7 s1/2 (2)   	  0.0060 
  5 f7/2 (1)   	  0.0060 
  5 f5/2 (6)   	  0.0060 


// ----------------------------------------------

  Cm   Z =  96
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	128.2000 
  2 s1/2 (2)   	 24.5200 
  2 p1/2 (2)   	 23.6600 
  2 p3/2 (4)   	 18.9700 
  3 s1/2 (2)   	  6.3350 
  3 p1/2 (2)   	  5.9400 
  3 p3/2 (4)   	  4.8500 
  3 d3/2 (4)   	  4.2300 
  3 d5/2 (6)   	  4.0150 
  4 s1/2 (2)   	  1.6850 
  4 p1/2 (2)   	  1.4950 
  4 p3/2 (4)   	  1.2050 
  4 d3/2 (4)   	  0.9250 
  4 d5/2 (6)   	  0.8700 
  4 f5/2 (6)   	  0.4950 
  4 f7/2 (8)   	  0.4850 
  5 s1/2 (2)   	  0.3970 
  5 p1/2 (2)   	  0.2350 
  5 p3/2 (4)   	  0.1350 
  5 d3/2 (4)   	  0.1200 
  7 s1/2 (2)   	  0.0970 
  5 f7/2 (1)   	  0.0920 
  6 s1/2 (2)   	  0.0670 
  6 p1/2 (2)   	  0.0500 
  5 d5/2 (6)   	  0.0060 
  6 p3/2 (4)   	  0.0060 
  6 d3/2 (1)   	  0.0060 
  5 f5/2 (6)   	  0.0060 


// ----------------------------------------------

  Bk   Z =  97
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	131.6000 
  2 s1/2 (2)   	 25.2600 
  2 p1/2 (2)   	 24.3800 
  2 p3/2 (4)   	 19.4400 
  3 s1/2 (2)   	  6.5450 
  3 p1/2 (2)   	  6.1450 
  3 p3/2 (4)   	  4.9900 
  3 d3/2 (4)   	  4.3600 
  3 d5/2 (6)   	  4.1350 
  4 s1/2 (2)   	  1.7450 
  4 p1/2 (2)   	  1.5550 
  4 p3/2 (4)   	  1.2450 
  4 d3/2 (4)   	  0.9600 
  4 d5/2 (6)   	  0.9000 
  4 f5/2 (6)   	  0.5250 
  4 f7/2 (8)   	  0.5100 
  5 s1/2 (2)   	  0.4150 
  5 p1/2 (2)   	  0.3300 
  5 p3/2 (4)   	  0.2430 
  5 d3/2 (4)   	  0.1400 
  5 d5/2 (6)   	  0.1250 
  6 s1/2 (2)   	  0.0960 
  6 p1/2 (2)   	  0.0700 
  6 p3/2 (4)   	  0.0500 
  7 s1/2 (2)   	  0.0060 
  5 f7/2 (2)   	  0.0060 
  5 f5/2 (6)   	  0.0060 
  6 d3/2 (1)   	  0.0044 


// ----------------------------------------------

  Cf   Z =  98
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	135.0000 
  2 s1/2 (2)   	 26.0100 
  2 p1/2 (2)   	 25.1200 
  2 p3/2 (4)   	 19.9100 
  3 s1/2 (2)   	  6.7550 
  3 p1/2 (2)   	  6.3500 
  3 p3/2 (4)   	  5.1350 
  3 d3/2 (4)   	  4.4950 
  3 d5/2 (6)   	  4.2600 
  4 s1/2 (2)   	  1.8100 
  4 p1/2 (2)   	  1.6150 
  4 p3/2 (4)   	  1.2850 
  4 d3/2 (4)   	  0.9950 
  4 d5/2 (6)   	  0.9300 
  4 f5/2 (6)   	  0.5550 
  4 f7/2 (8)   	  0.5400 
  5 s1/2 (2)   	  0.4300 
  5 p1/2 (2)   	  0.3450 
  5 p3/2 (4)   	  0.2500 
  5 d3/2 (4)   	  0.1450 
  5 d5/2 (6)   	  0.1300 
  6 s1/2 (2)   	  0.1000 
  6 p1/2 (2)   	  0.0700 
  6 p3/2 (4)   	  0.0500 
  7 s1/2 (2)   	  0.0060 
  5 f7/2 (4)   	  0.0060 
  5 f5/2 (6)   	  0.0060 


// ----------------------------------------------

  Es   Z =  99
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	138.4000 
  2 s1/2 (2)   	 26.7800 
  2 p1/2 (2)   	 25.8800 
  2 p3/2 (4)   	 20.3900 
  3 s1/2 (2)   	  6.9750 
  3 p1/2 (2)   	  6.5600 
  3 p3/2 (4)   	  5.2800 
  3 d3/2 (4)   	  4.6300 
  3 d5/2 (6)   	  4.3850 
  4 s1/2 (2)   	  1.8750 
  4 p1/2 (2)   	  1.6750 
  4 p3/2 (4)   	  1.3250 
  4 d3/2 (4)   	  1.0300 
  4 d5/2 (6)   	  0.9650 
  4 f5/2 (6)   	  0.5850 
  4 f7/2 (8)   	  0.5700 
  5 s1/2 (2)   	  0.4450 
  5 p1/2 (2)   	  0.3600 
  5 p3/2 (4)   	  0.2600 
  5 d3/2 (4)   	  0.1550 
  5 d5/2 (6)   	  0.1350 
  6 s1/2 (2)   	  0.1000 
  6 p1/2 (2)   	  0.0700 
  6 p3/2 (4)   	  0.0500 
  7 s1/2 (2)   	  0.0060 
  5 f7/2 (5)   	  0.0060 
  5 f5/2 (6)   	  0.0060 


// ----------------------------------------------

  Fm   Z = 100
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	142.0000 
  2 s1/2 (2)   	 27.5700 
  2 p1/2 (2)   	 26.6600 
  2 p3/2 (4)   	 20.8700 
  3 s1/2 (2)   	  7.2000 
  3 p1/2 (2)   	  6.7800 
  3 p3/2 (4)   	  5.4250 
  3 d3/2 (4)   	  4.7650 
  3 d5/2 (6)   	  4.5100 
  4 s1/2 (2)   	  1.9400 
  4 p1/2 (2)   	  1.7400 
  4 p3/2 (4)   	  1.3700 
  4 d3/2 (4)   	  1.0650 
  4 d5/2 (6)   	  1.0000 
  4 f5/2 (6)   	  0.6150 
  4 f7/2 (8)   	  0.6000 
  5 s1/2 (2)   	  0.4600 
  5 p1/2 (2)   	  0.3750 
  5 p3/2 (4)   	  0.2700 
  5 d3/2 (4)   	  0.1650 
  5 d5/2 (6)   	  0.1400 
  6 s1/2 (2)   	  0.1000 
  6 p1/2 (2)   	  0.0800 
  6 p3/2 (4)   	  0.0500 
  7 s1/2 (2)   	  0.0060 
  5 f7/2 (6)   	  0.0060 
  5 f5/2 (6)   	  0.0060 


// ----------------------------------------------

  Md   Z = 101
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	145.6000 
  2 s1/2 (2)   	 28.3800 
  2 p1/2 (2)   	 27.4600 
  2 p3/2 (4)   	 21.3600 
  3 s1/2 (2)   	  7.4300 
  3 p1/2 (2)   	  7.0000 
  3 p3/2 (4)   	  5.5750 
  3 d3/2 (4)   	  4.9050 
  3 d5/2 (6)   	  4.6330 
  4 s1/2 (2)   	  2.0050 
  4 p1/2 (2)   	  1.8050 
  4 p3/2 (4)   	  1.4150 
  4 d3/2 (4)   	  1.1050 
  4 d5/2 (6)   	  1.0350 
  4 f5/2 (6)   	  0.6450 
  4 f7/2 (8)   	  0.6300 
  5 s1/2 (2)   	  0.4800 
  5 p1/2 (2)   	  0.3900 
  5 p3/2 (4)   	  0.2800 
  5 d3/2 (4)   	  0.1750 
  5 d5/2 (6)   	  0.1450 
  6 s1/2 (2)   	  0.1100 
  6 p1/2 (2)   	  0.0800 
  6 p3/2 (4)   	  0.0500 
  5 f7/2 (7)   	  0.0070 
  5 f5/2 (6)   	  0.0070 
  7 s1/2 (2)   	  0.0060 


// ----------------------------------------------

  No   Z = 102
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	149.3000 
  2 s1/2 (2)   	 29.2220 
  2 p1/2 (2)   	 28.2800 
  2 p3/2 (4)   	 21.8500 
  3 s1/2 (2)   	  7.6700 
  3 p1/2 (2)   	  7.2300 
  3 p3/2 (4)   	  5.7300 
  3 d3/2 (4)   	  5.0500 
  3 d5/2 (6)   	  4.7650 
  4 s1/2 (2)   	  2.0750 
  4 p1/2 (2)   	  1.8750 
  4 p3/2 (4)   	  1.4600 
  4 d3/2 (4)   	  1.1450 
  4 d5/2 (6)   	  1.0700 
  4 f5/2 (6)   	  0.6800 
  4 f7/2 (8)   	  0.6500 
  5 s1/2 (2)   	  0.5000 
  5 p1/2 (2)   	  0.4100 
  5 p3/2 (4)   	  0.2900 
  5 d3/2 (4)   	  0.1850 
  5 d5/2 (6)   	  0.1550 
  6 s1/2 (2)   	  0.1100 
  6 p1/2 (2)   	  0.0800 
  6 p3/2 (4)   	  0.0500 
  5 f7/2 (8)   	  0.0100 
  5 f5/2 (6)   	  0.0100 
  7 s1/2 (2)   	  0.0060 


// ----------------------------------------------

  Lr   Z = 103
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	153.0000 
  2 s1/2 (2)   	 30.0800 
  2 p1/2 (2)   	 29.1300 
  2 p3/2 (4)   	 22.3500 
  3 s1/2 (2)   	  7.9200 
  3 p1/2 (2)   	  7.4700 
  3 p3/2 (4)   	  5.8900 
  3 d3/2 (4)   	  5.2000 
  3 d5/2 (6)   	  4.9000 
  4 s1/2 (2)   	  2.1500 
  4 p1/2 (2)   	  1.9500 
  4 p3/2 (4)   	  1.5050 
  4 d3/2 (4)   	  1.1900 
  4 d5/2 (6)   	  1.1100 
  4 f5/2 (6)   	  0.7200 
  4 f7/2 (8)   	  0.6900 
  5 s1/2 (2)   	  0.5200 
  5 p1/2 (2)   	  0.4300 
  5 p3/2 (4)   	  0.3000 
  5 d3/2 (4)   	  0.2000 
  5 d5/2 (6)   	  0.1600 
  6 s1/2 (2)   	  0.1200 
  6 p1/2 (2)   	  0.0900 
  6 p3/2 (4)   	  0.0600 
  5 f7/2 (8)   	  0.0150 
  5 f5/2 (6)   	  0.0150 
  7 s1/2 (2)   	  0.0070 
  6 d3/2 (1)   	  0.0070 


// ----------------------------------------------

  Rf   Z = 104
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	156.9000 
  2 s1/2 (2)   	 30.9600 
  2 p1/2 (2)   	 30.0000 
  2 p3/2 (4)   	 22.8600 
  3 s1/2 (2)   	  8.1700 
  3 p1/2 (2)   	  7.7200 
  3 p3/2 (4)   	  6.0550 
  3 d3/2 (4)   	  5.3500 
  3 d5/2 (6)   	  5.0400 
  4 s1/2 (2)   	  2.2250 
  4 p1/2 (2)   	  2.0250 
  4 p3/2 (4)   	  1.5550 
  4 d3/2 (4)   	  1.2350 
  4 d5/2 (6)   	  1.1500 
  4 f5/2 (6)   	  0.7600 
  4 f7/2 (8)   	  0.7200 
  5 s1/2 (2)   	  0.5400 
  5 p1/2 (2)   	  0.4550 
  5 p3/2 (4)   	  0.3150 
  5 d3/2 (4)   	  0.2150 
  5 d5/2 (6)   	  0.1700 
  6 s1/2 (2)   	  0.1200 
  6 p1/2 (2)   	  0.0900 
  6 p3/2 (4)   	  0.0600 
  5 f7/2 (8)   	  0.0200 
  5 f5/2 (6)   	  0.0200 
  7 s1/2 (2)   	  0.0080 
  6 d3/2 (2)   	  0.0080 


// ----------------------------------------------

  Db   Z = 105
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	160.8000 
  2 s1/2 (2)   	 31.8700 
  2 p1/2 (2)   	 30.9000 
  2 p3/2 (4)   	 23.3800 
  3 s1/2 (2)   	  8.4300 
  3 p1/2 (2)   	  7.9800 
  3 p3/2 (4)   	  6.6250 
  3 d3/2 (4)   	  5.5050 
  3 d5/2 (6)   	  5.1850 
  4 s1/2 (2)   	  2.3050 
  4 p1/2 (2)   	  2.1050 
  4 p3/2 (4)   	  1.6050 
  4 d3/2 (4)   	  1.2800 
  4 d5/2 (6)   	  1.1900 
  4 f5/2 (6)   	  0.8000 
  4 f7/2 (8)   	  0.7500 
  5 s1/2 (2)   	  0.5600 
  5 p1/2 (2)   	  0.4800 
  5 p3/2 (4)   	  0.3300 
  5 d3/2 (4)   	  0.2300 
  5 d5/2 (6)   	  0.1800 
  6 s1/2 (2)   	  0.1300 
  6 p1/2 (2)   	  0.1000 
  6 p3/2 (4)   	  0.0600 
  5 f7/2 (8)   	  0.0250 
  5 f5/2 (6)   	  0.0250 
  6 d3/2 (3)   	  0.0090 
  7 s1/2 (2)   	  0.0080 


// ----------------------------------------------

  Sg   Z = 106
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	164.9000 
  2 s1/2 (2)   	 32.8000 
  2 p1/2 (2)   	 31.8200 
  2 p3/2 (4)   	 23.9100 
  3 s1/2 (2)   	  8.7000 
  3 p1/2 (2)   	  8.2400 
  3 p3/2 (4)   	  6.4000 
  3 d3/2 (4)   	  5.6650 
  3 d5/2 (6)   	  5.3300 
  4 s1/2 (2)   	  2.3850 
  4 p1/2 (2)   	  2.1900 
  4 p3/2 (4)   	  1.6600 
  4 d3/2 (4)   	  1.3250 
  4 d5/2 (6)   	  1.2350 
  4 f5/2 (6)   	  0.8400 
  4 f7/2 (8)   	  0.7800 
  5 s1/2 (2)   	  0.5850 
  5 p1/2 (2)   	  0.5050 
  5 p3/2 (4)   	  0.3450 
  5 d3/2 (4)   	  0.2450 
  5 d5/2 (6)   	  0.1900 
  6 s1/2 (2)   	  0.1400 
  6 p1/2 (2)   	  0.1100 
  6 p3/2 (4)   	  0.0600 
  5 f7/2 (8)   	  0.0300 
  5 f5/2 (6)   	  0.0300 
  6 d3/2 (4)   	  0.0090 
  7 s1/2 (2)   	  0.0080 


// ----------------------------------------------

  Bh   Z = 107
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	169.0000 
  2 s1/2 (2)   	 33.7600 
  2 p1/2 (2)   	 32.7700 
  2 p3/2 (4)   	 24.4400 
  3 s1/2 (2)   	  8.9800 
  3 p1/2 (2)   	  8.5100 
  3 p3/2 (4)   	  6.5800 
  3 d3/2 (4)   	  5.8300 
  3 d5/2 (6)   	  5.4750 
  4 s1/2 (2)   	  2.4700 
  4 p1/2 (2)   	  2.2750 
  4 p3/2 (4)   	  1.7150 
  4 d3/2 (4)   	  1.3700 
  4 d5/2 (6)   	  1.2800 
  4 f5/2 (6)   	  0.8800 
  4 f7/2 (8)   	  0.8200 
  5 s1/2 (2)   	  0.6100 
  5 p1/2 (2)   	  0.5300 
  5 p3/2 (4)   	  0.3600 
  5 d3/2 (4)   	  0.2600 
  5 d5/2 (6)   	  0.2000 
  6 s1/2 (2)   	  0.1400 
  6 p1/2 (2)   	  0.1100 
  6 p3/2 (4)   	  0.0700 
  5 f7/2 (8)   	  0.0400 
  5 f5/2 (6)   	  0.0400 
  6 d3/2 (4)   	  0.0100 
  6 d5/2 (1)   	  0.0100 
  7 s1/2 (2)   	  0.0090 


// ----------------------------------------------

  Hs   Z = 108
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	173.2000 
  2 s1/2 (2)   	 34.7500 
  2 p1/2 (2)   	 33.7400 
  2 p3/2 (4)   	 24.9200 
  3 s1/2 (2)   	  9.2700 
  3 p1/2 (2)   	  8.7900 
  3 p3/2 (4)   	  6.7600 
  3 d3/2 (4)   	  5.9950 
  3 d5/2 (6)   	  5.6200 
  4 s1/2 (2)   	  2.5550 
  4 p1/2 (2)   	  2.3650 
  4 p3/2 (4)   	  1.7700 
  4 d3/2 (4)   	  1.4200 
  4 d5/2 (6)   	  1.3250 
  4 f5/2 (6)   	  0.9200 
  4 f7/2 (8)   	  0.8600 
  5 s1/2 (2)   	  0.6350 
  5 p1/2 (2)   	  0.5550 
  5 p3/2 (4)   	  0.3750 
  5 d3/2 (4)   	  0.2750 
  5 d5/2 (6)   	  0.2100 
  6 s1/2 (2)   	  0.1500 
  6 p1/2 (2)   	  0.1200 
  6 p3/2 (4)   	  0.0700 
  5 f7/2 (8)   	  0.0500 
  5 f5/2 (6)   	  0.0500 
  6 d3/2 (4)   	  0.0100 
  6 d5/2 (2)   	  0.0100 
  7 s1/2 (2)   	  0.0090 


// ----------------------------------------------

  Mt   Z = 109
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	177.6000 
  2 s1/2 (2)   	 35.7700 
  2 p1/2 (2)   	 34.7400 
  2 p3/2 (4)   	 25.5200 
  3 s1/2 (2)   	  9.5700 
  3 p1/2 (2)   	  9.0800 
  3 p3/2 (4)   	  6.9400 
  3 d3/2 (4)   	  6.1600 
  3 d5/2 (6)   	  5.7700 
  4 s1/2 (2)   	  2.6450 
  4 p1/2 (2)   	  2.4600 
  4 p3/2 (4)   	  1.8250 
  4 d3/2 (4)   	  1.4700 
  4 d5/2 (6)   	  1.3700 
  4 f5/2 (6)   	  0.9600 
  4 f7/2 (8)   	  0.9000 
  5 s1/2 (2)   	  0.6600 
  5 p1/2 (2)   	  0.5800 
  5 p3/2 (4)   	  0.3900 
  5 d3/2 (4)   	  0.2900 
  5 d5/2 (6)   	  0.2200 
  6 s1/2 (2)   	  0.1600 
  6 p1/2 (2)   	  0.1200 
  6 p3/2 (4)   	  0.0700 
  5 f7/2 (8)   	  0.0600 
  5 f5/2 (6)   	  0.0600 
  7 s1/2 (2)   	  0.0100 
  6 d3/2 (4)   	  0.0100 
  6 d5/2 (3)   	  0.0100 


// ----------------------------------------------

  Ds   Z = 110
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	182.0000 
  2 s1/2 (2)   	 36.8100 
  2 p1/2 (2)   	 35.7700 
  2 p3/2 (4)   	 26.0700 
  3 s1/2 (2)   	  9.8700 
  3 p1/2 (2)   	  9.3700 
  3 p3/2 (4)   	  7.1200 
  3 d3/2 (4)   	  6.3300 
  3 d5/2 (6)   	  5.9200 
  4 s1/2 (2)   	  2.7350 
  4 p1/2 (2)   	  2.5600 
  4 p3/2 (4)   	  1.8800 
  4 d3/2 (4)   	  1.5200 
  4 d5/2 (6)   	  1.4150 
  4 f5/2 (6)   	  1.0000 
  4 f7/2 (8)   	  0.9400 
  5 s1/2 (2)   	  0.6900 
  5 p1/2 (2)   	  0.6100 
  5 p3/2 (4)   	  0.4050 
  5 d3/2 (4)   	  0.3050 
  5 d5/2 (6)   	  0.2300 
  6 s1/2 (2)   	  0.1600 
  6 p1/2 (2)   	  0.1300 
  6 p3/2 (4)   	  0.0800 
  5 f7/2 (8)   	  0.0700 
  5 f5/2 (6)   	  0.0700 
  7 s1/2 (1)   	  0.0100 
  6 d3/2 (4)   	  0.0100 
  6 d5/2 (5)   	  0.0100 


// ----------------------------------------------

  Rg   Z = 111
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	186.1360 
  2 s1/2 (2)   	 37.8810 
  2 p1/2 (2)   	 36.7990 
  2 p3/2 (4)   	 26.6480 
  3 s1/2 (2)   	 10.1870 
  3 p1/2 (2)   	  9.6870 
  3 p3/2 (4)   	  7.2240 
  3 d3/2 (4)   	  6.4470 
  3 d5/2 (6)   	  6.0250 
  4 s1/2 (2)   	  2.9260 
  4 p1/2 (2)   	  2.6730 
  4 p3/2 (4)   	  1.9780 
  4 d3/2 (4)   	  1.5970 
  4 d5/2 (6)   	  1.4830 
  4 f5/2 (6)   	  0.9604 
  4 f7/2 (8)   	  0.9267 
  5 s1/2 (2)   	  0.7283 
  5 p1/2 (2)   	  0.6450 
  5 p3/2 (4)   	  0.2810 
  5 d3/2 (4)   	  0.2510 
  5 d5/2 (6)   	  0.2445 
  6 s1/2 (2)   	  0.1267 
  6 p1/2 (2)   	  0.0901 
  5 f5/2 (6)   	  0.0671 
  5 f7/2 (8)   	  0.0614 
  6 p3/2 (4)   	  0.0498 
  7 s1/2 (1)   	  0.0094 
  6 d3/2 (4)   	  0.0094 
  6 d5/2 (6)   	  0.0066 


// ----------------------------------------------

  Cp   Z = 112
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	190.7170 
  2 s1/2 (2)   	 38.9840 
  2 p1/2 (2)   	 37.8960 
  2 p3/2 (4)   	 27.2130 
  3 s1/2 (2)   	 10.5080 
  3 p1/2 (2)   	 10.0020 
  3 p3/2 (4)   	  7.4060 
  3 d3/2 (4)   	  6.6180 
  3 d5/2 (6)   	  6.1780 
  4 s1/2 (2)   	  3.0330 
  4 p1/2 (2)   	  2.7760 
  4 p3/2 (4)   	  2.0390 
  4 d3/2 (4)   	  1.6520 
  4 d5/2 (6)   	  1.5320 
  4 f5/2 (6)   	  1.0016 
  4 f7/2 (8)   	  0.9662 
  5 s1/2 (2)   	  0.7624 
  5 p1/2 (2)   	  0.6770 
  5 p3/2 (4)   	  0.2960 
  5 d3/2 (4)   	  0.2650 
  5 d5/2 (6)   	  0.2577 
  6 s1/2 (2)   	  0.1349 
  6 p1/2 (2)   	  0.0969 
  5 f5/2 (6)   	  0.0753 
  5 f7/2 (8)   	  0.0690 
  6 p3/2 (4)   	  0.0528 
  6 d3/2 (4)   	  0.0104 
  7 s1/2 (2)   	  0.0098 
  6 d5/2 (6)   	  0.0073 


// ----------------------------------------------

  Uut  Z = 113
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	195.4100 
  2 s1/2 (2)   	 40.1210 
  2 p1/2 (2)   	 39.0340 
  2 p3/2 (4)   	 27.7880 
  3 s1/2 (2)   	 10.8420 
  3 p1/2 (2)   	 10.3310 
  3 p3/2 (4)   	  7.5930 
  3 d3/2 (4)   	  6.7930 
  3 d5/2 (6)   	  6.3350 
  4 s1/2 (2)   	  3.1460 
  4 p1/2 (2)   	  2.8860 
  4 p3/2 (4)   	  2.1070 
  4 d3/2 (4)   	  1.7120 
  4 d5/2 (6)   	  1.5870 
  4 f5/2 (6)   	  1.0464 
  4 f7/2 (8)   	  1.0092 
  5 s1/2 (2)   	  0.8006 
  5 p1/2 (2)   	  0.7140 
  5 p3/2 (4)   	  0.3150 
  5 d3/2 (4)   	  0.2820 
  5 d5/2 (6)   	  0.2740 
  6 s1/2 (2)   	  0.1464 
  6 p1/2 (2)   	  0.1069 
  5 f5/2 (6)   	  0.0866 
  5 f7/2 (8)   	  0.0798 
  6 p3/2 (4)   	  0.0586 
  6 d3/2 (4)   	  0.0138 
  7 s1/2 (2)   	  0.0116 
  6 d5/2 (6)   	  0.0101 
  7 p1/2 (1)   	  0.0049 


// ----------------------------------------------

  Fl   Z = 114
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	200.2140 
  2 s1/2 (2)   	 41.2940 
  2 p1/2 (2)   	 40.2070 
  2 p3/2 (4)   	 28.3700 
  3 s1/2 (2)   	 11.1840 
  3 p1/2 (2)   	 10.6720 
  3 p3/2 (4)   	  7.7830 
  3 d3/2 (4)   	  6.9720 
  3 d5/2 (6)   	  6.4950 
  4 s1/2 (2)   	  3.2640 
  4 p1/2 (2)   	  2.9990 
  4 p3/2 (4)   	  2.1740 
  4 d3/2 (4)   	  1.7740 
  4 d5/2 (6)   	  1.6420 
  4 f5/2 (6)   	  1.0923 
  4 f7/2 (8)   	  1.0533 
  5 s1/2 (2)   	  0.8409 
  5 p1/2 (2)   	  0.7530 
  5 p3/2 (4)   	  0.3350 
  5 d3/2 (4)   	  0.3000 
  5 d5/2 (6)   	  0.2910 
  6 s1/2 (2)   	  0.1583 
  6 p1/2 (2)   	  0.1174 
  5 f5/2 (6)   	  0.0984 
  5 f7/2 (8)   	  0.0911 
  6 p3/2 (4)   	  0.0646 
  6 d3/2 (4)   	  0.0173 
  7 s1/2 (2)   	  0.0135 
  6 d5/2 (6)   	  0.0130 
  7 p1/2 (2)   	  0.0057 


// ----------------------------------------------

  Uup  Z = 115
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	205.1360 
  2 s1/2 (2)   	 42.5010 
  2 p1/2 (2)   	 41.4200 
  2 p3/2 (4)   	 28.9590 
  3 s1/2 (2)   	 11.5410 
  3 p1/2 (2)   	 11.0220 
  3 p3/2 (4)   	  7.9790 
  3 d3/2 (4)   	  7.1550 
  3 d5/2 (6)   	  6.6580 
  4 s1/2 (2)   	  3.3860 
  4 p1/2 (2)   	  3.1180 
  4 p3/2 (4)   	  2.2450 
  4 d3/2 (4)   	  1.8380 
  4 d5/2 (6)   	  1.7010 
  4 f5/2 (6)   	  1.1410 
  4 f7/2 (8)   	  1.1000 
  5 s1/2 (2)   	  0.8831 
  5 p1/2 (2)   	  0.7940 
  5 p3/2 (4)   	  0.3570 
  5 d3/2 (4)   	  0.3200 
  5 d5/2 (6)   	  0.3100 
  6 s1/2 (2)   	  0.1726 
  6 p1/2 (2)   	  0.1301 
  5 f5/2 (6)   	  0.1124 
  5 f7/2 (8)   	  0.1045 
  6 p3/2 (4)   	  0.0724 
  6 d3/2 (4)   	  0.0227 
  6 d5/2 (6)   	  0.0177 
  7 s1/2 (2)   	  0.0169 
  7 p1/2 (2)   	  0.0078 
  7 p3/2 (1)   	  0.0038 


// ----------------------------------------------

  Lv   Z = 116
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	210.1770 
  2 s1/2 (2)   	 43.7500 
  2 p1/2 (2)   	 42.6740 
  2 p3/2 (4)   	 29.5560 
  3 s1/2 (2)   	 11.9070 
  3 p1/2 (2)   	 11.3870 
  3 p3/2 (4)   	  8.1760 
  3 d3/2 (4)   	  7.3410 
  3 d5/2 (6)   	  6.8240 
  4 s1/2 (2)   	  3.5130 
  4 p1/2 (2)   	  3.2410 
  4 p3/2 (4)   	  2.3160 
  4 d3/2 (4)   	  1.9030 
  4 d5/2 (6)   	  1.7590 
  4 f5/2 (6)   	  1.1907 
  4 f7/2 (8)   	  1.1477 
  5 s1/2 (2)   	  0.9275 
  5 p1/2 (2)   	  0.8380 
  5 p3/2 (4)   	  0.3790 
  5 d3/2 (4)   	  0.3400 
  5 d5/2 (6)   	  0.3296 
  6 s1/2 (2)   	  0.1874 
  6 p1/2 (2)   	  0.1434 
  5 f5/2 (6)   	  0.1269 
  5 f7/2 (8)   	  0.1184 
  6 p3/2 (4)   	  0.0805 
  6 d3/2 (4)   	  0.0282 
  6 d5/2 (6)   	  0.0226 
  7 s1/2 (2)   	  0.0203 
  7 p1/2 (2)   	  0.0101 
  7 p3/2 (2)   	  0.0044 


// ----------------------------------------------

  Uus  Z = 117
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	215.3370 
  2 s1/2 (2)   	 44.0310 
  2 p1/2 (2)   	 43.9750 
  2 p3/2 (4)   	 30.1580 
  3 s1/2 (2)   	 12.2830 
  3 p1/2 (2)   	 11.7620 
  3 p3/2 (4)   	  8.3760 
  3 d3/2 (4)   	  7.5300 
  3 d5/2 (6)   	  6.9910 
  4 s1/2 (2)   	  3.6430 
  4 p1/2 (2)   	  3.4690 
  4 p3/2 (4)   	  2.3900 
  4 d3/2 (4)   	  1.9700 
  4 d5/2 (6)   	  1.8190 
  4 f5/2 (6)   	  1.2413 
  4 f7/2 (8)   	  1.1962 
  5 s1/2 (2)   	  0.9733 
  5 p1/2 (2)   	  0.8820 
  5 p3/2 (4)   	  0.4020 
  5 d3/2 (4)   	  0.3610 
  5 d5/2 (6)   	  0.3496 
  6 s1/2 (2)   	  0.2026 
  6 p1/2 (2)   	  0.1571 
  5 f5/2 (6)   	  0.1417 
  5 f7/2 (8)   	  0.1327 
  6 p3/2 (4)   	  0.0886 
  6 d3/2 (4)   	  0.0337 
  6 d5/2 (6)   	  0.0275 
  7 s1/2 (2)   	  0.0237 
  7 p1/2 (2)   	  0.0124 
  7 p3/2 (3)   	  0.0052 


// ----------------------------------------------

  Uuo  Z = 118
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	220.6300 
  2 s1/2 (2)   	 46.3540 
  2 p1/2 (2)   	 45.3540 
  2 p3/2 (4)   	 30.7670 
  3 s1/2 (2)   	 12.6700 
  3 p1/2 (2)   	 12.1510 
  3 p3/2 (4)   	  8.5790 
  3 d3/2 (4)   	  7.7200 
  3 d5/2 (6)   	  7.1610 
  4 s1/2 (2)   	  3.7780 
  4 p1/2 (2)   	  3.5000 
  4 p3/2 (4)   	  2.4640 
  4 d3/2 (4)   	  2.0370 
  4 d5/2 (6)   	  1.8800 
  4 f5/2 (6)   	  1.2928 
  4 f7/2 (8)   	  1.2455 
  5 s1/2 (2)   	  1.0205 
  5 p1/2 (2)   	  0.9280 
  5 p3/2 (4)   	  0.4250 
  5 d3/2 (4)   	  0.3820 
  5 d5/2 (6)   	  0.3699 
  6 s1/2 (2)   	  0.2183 
  6 p1/2 (2)   	  0.1714 
  5 f5/2 (6)   	  0.1570 
  5 f7/2 (8)   	  0.1472 
  6 p3/2 (4)   	  0.0968 
  6 d3/2 (4)   	  0.0394 
  6 d5/2 (6)   	  0.0324 
  7 s1/2 (2)   	  0.0272 
  7 p1/2 (2)   	  0.0149 
  7 p3/2 (4)   	  0.0060 


// ----------------------------------------------

  Uue  Z = 119
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	226.0630 
  2 s1/2 (2)   	 47.7240 
  2 p1/2 (2)   	 46.7140 
  2 p3/2 (4)   	 31.3860 
  3 s1/2 (2)   	 13.0740 
  3 p1/2 (2)   	 12.5560 
  3 p3/2 (4)   	  8.7890 
  3 d3/2 (4)   	  7.9180 
  3 d5/2 (6)   	  7.3360 
  4 s1/2 (2)   	  3.9200 
  4 p1/2 (2)   	  3.6390 
  4 p3/2 (4)   	  2.5430 
  4 d3/2 (4)   	  2.1090 
  4 d5/2 (6)   	  1.9440 
  4 f5/2 (6)   	  1.3476 
  4 f7/2 (8)   	  1.2982 
  5 s1/2 (2)   	  1.0720 
  5 p1/2 (2)   	  0.9800 
  5 p3/2 (4)   	  0.4520 
  5 d3/2 (4)   	  0.4060 
  5 d5/2 (6)   	  0.3931 
  6 s1/2 (2)   	  0.2371 
  6 p1/2 (2)   	  0.1887 
  5 f5/2 (6)   	  0.1751 
  5 f7/2 (8)   	  0.1646 
  6 p3/2 (4)   	  0.1075 
  6 d3/2 (4)   	  0.0476 
  6 d5/2 (6)   	  0.0399 
  7 s1/2 (2)   	  0.0331 
  7 p1/2 (2)   	  0.0197 
  7 p3/2 (4)   	  0.0084 
  8 s1/2 (1)   	  0.0036 


// ----------------------------------------------

  Ubn  Z = 120
  ______________________
  N   L  (#)      E(keV)
  ______________________
  1 s1/2 (2)   	231.6340 
  2 s1/2 (2)   	 49.1390 
  2 p1/2 (2)   	 48.1580 
  2 p3/2 (4)   	 32.0130 
  3 s1/2 (2)   	 13.4890 
  3 p1/2 (2)   	 12.9750 
  3 p3/2 (4)   	  9.0000 
  3 d3/2 (4)   	  8.1170 
  3 d5/2 (6)   	  7.5130 
  4 s1/2 (2)   	  4.0660 
  4 p1/2 (2)   	  3.7840 
  4 p3/2 (4)   	  2.6220 
  4 d3/2 (4)   	  2.1820 
  4 d5/2 (6)   	  2.0100 
  4 f5/2 (6)   	  1.4040 
  4 f7/2 (8)   	  1.3523 
  5 s1/2 (2)   	  1.1256 
  5 p1/2 (2)   	  1.0330 
  5 p3/2 (4)   	  0.4790 
  5 d3/2 (4)   	  0.4310 
  5 d5/2 (6)   	  0.4173 
  6 s1/2 (2)   	  0.2570 
  6 p1/2 (2)   	  0.2071 
  5 f5/2 (6)   	  0.1941 
  5 f7/2 (8)   	  0.1830 
  6 p3/2 (4)   	  0.1190 
  6 d3/2 (4)   	  0.0565 
  6 d5/2 (6)   	  0.0479 
  7 s1/2 (2)   	  0.0396 
  7 p1/2 (2)   	  0.0251 
  7 p3/2 (4)   	  0.0113 
  8 s1/2 (2)   	  0.0042 
******************************************************************************/

#endif   // end of G4AtomicShells_EADL.hh